#include "pack.h"

#include <drive/backend/offers/actions/helpers.h>
#include <drive/backend/offers/context.h>
#include <drive/backend/offers/ranking/calcer.h>

#include <drive/backend/common/localization.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/events.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/logging/evlog.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/users/user.h>

#include <kernel/daemon/common/time_guard.h>

#include <rtline/protos/proto_helper.h>
#include <rtline/util/json_processing.h>
#include <rtline/util/algorithm/datetime.h>

const TString TPackOffer::DistanceThresholdPushName = "pack_offer_distance_threshold";
const TString TPackOffer::DurationThresholdPushName = "pack_offer_duration_threshold";

const TString PackSegmentName = "pack";
const TString OverrunSegmentName = "overrun";
const TString OvertimeSegmentName = "overtime";
const TString ServicingSegmentName = "servicing";

TPackOffer::TFactory::TRegistrator<TPackOffer> TPackOffer::Registrator(TPackOfferConstructor::GetTypeStatic());
TUserAction::TFactory::TRegistrator<TPackOfferConstructor> TPackOfferConstructor::Registrator(TPackOfferConstructor::GetTypeStatic());

void TPriceSchedule::AddQuantum(TInstant timestamp, TMaybe<ui32> mileage, ui32 value) {
    Quanta.emplace_back(timestamp, UseMileage ? mileage : Nothing(), value);
}

TPriceSchedule::TQuanta::const_iterator TPriceSchedule::GetNextQuantumImpl(TInstant timestamp, double mileage) const {
    auto it = Quanta.begin();
    while (it != Quanta.end() && it->ShouldBePaid(timestamp, mileage)) {
        ++it;
    }
    return it;
}

ui32 TPriceSchedule::GetPrice(TInstant timestamp, double mileage) const {
    ui32 result = 0;
    auto end = GetNextQuantumImpl(timestamp, mileage);
    for (auto it = Quanta.begin(); it != end; ++it) {
        result += it->Value;
    }
    return result;
}

bool TPriceSchedule::HasMileageConditions() const {
    return std::any_of(Quanta.cbegin(), Quanta.cend(), [](auto quantum) {
        return quantum.Mileage;
    });
}

TPriceSchedule::TOptionalQuantum TPriceSchedule::GetNextQuantum(TInstant timestamp, double mileage) const {
    auto it = GetNextQuantumImpl(timestamp, mileage);
    if (it == Quanta.end()) {
        return {};
    }
    Y_ASSERT(timestamp <= it->Timestamp);
    Y_ASSERT(!it->Mileage || mileage <= it->Mileage);
    return *it;
}

NJson::TJsonValue TPriceSchedule::GetReport(ELocalization locale, const NDrive::IServer& server) const {
    Y_UNUSED(locale);
    Y_UNUSED(server);
    NJson::TJsonValue result = NJson::JSON_ARRAY;
    for (auto&& [timestamp, mileage, value] : Quanta) {
        NJson::TJsonValue quantum;
        quantum["timestamp"] = timestamp.Seconds();
        if (mileage) {
            quantum["mileage"] = *mileage;
        }
        quantum["value"] = value;
        result.AppendValue(std::move(quantum));
    }
    return result;
}

NDrive::NProto::TPriceSchedule TPriceSchedule::Serialize() const {
    NDrive::NProto::TPriceSchedule result;
    for (auto&& [timestamp, mileage, value] : Quanta) {
        auto quantum = result.AddQuantum();
        quantum->SetTimestamp(timestamp.Seconds());
        if (mileage) {
            quantum->SetMileage(*mileage);
        }
        quantum->SetPrice(value);
    }
    result.SetUseMileage(UseMileage);
    return result;
}

bool TPriceSchedule::Deserialize(const NDrive::NProto::TPriceSchedule& proto) {
    if (proto.HasUseMileage()) {
        UseMileage = proto.GetUseMileage();
    }
    for (auto&& quantum : proto.GetQuantum()) {
        AddQuantum(
            TInstant::Seconds(quantum.GetTimestamp()),
            quantum.HasMileage() ? TMaybe<ui32>(quantum.GetMileage()) : Nothing(),
            quantum.GetPrice()
        );
    }
    return true;
}

TPackOfferState& TPackOfferState::AddPricing(const TOfferPricing& pricing) {
    if (!Pricing) {
        Pricing = pricing;
    } else {
        *Pricing = *Pricing + pricing;
    }
    return *this;
}

NJson::TJsonValue TPackOfferState::GetReport(ELocalization locale, const NDrive::IServer& server) const {
    auto localization = server.GetLocalization();
    NJson::TJsonValue result;
    result.InsertValue("distance_threshold_push_sent", DistanceThresholdPushSent);
    result.InsertValue("duration_threshold_push_sent", DurationThresholdPushSent);
    result.InsertValue("remaining_distance", DistanceRemaining);
    result.InsertValue("remaining_time", TimeRemaining.Seconds());

    NJson::InsertField(result, "current_pack_price", CurrentPackPrice);
    NJson::InsertField(result, "pack_price", PackPrice);
    NJson::InsertNonNull(result, "pack_price_round", ICommonOffer::RoundPrice(PackPrice, 100));

    //just stub (+WaitingPrice)
    NJson::InsertNonNull(result, "overtime_price", ICommonOffer::RoundPrice(OvertimePrice));
    result.InsertValue("overrun_time", Overtime.Seconds());

    NJson::InsertNonNull(result, "waiting_price", ICommonOffer::RoundPrice(WaitingPrice));

    NJson::InsertNonNull(result, "overrun_price", ICommonOffer::RoundPrice(OverrunPrice));
    result.InsertValue("overrun_distance", Overrun);

    bool exceeded = static_cast<i64>(Overrun) > 0 || Overtime.Minutes() > 0;
    result.InsertValue("exceeded", exceeded);
    result.InsertValue("show_remainders", ShowRemainders);
    if (ServicingDuration) {
        result["servicing_duration"] = ServicingDuration.Seconds();
    }
    if (ServicingMileage > 0) {
        result["servicing_mileage"] = ServicingMileage;
    }
    if (ServicingOmittedPrice) {
        result["servicing_omitted_price"] = ServicingOmittedPrice;
    }
    if (Since) {
        result["since"] = Since->Seconds();
    }
    if (Until) {
        result["until"] = Until->Seconds();
    }
    if (Upsaleable) {
        result["upsaleable"] = true;
    }
    if (UpsaleMessage && localization) {
        auto upsaleMessage = localization->GetLocalString(locale, UpsaleMessage);
        SubstGlobal(upsaleMessage, "_OvertimePriceDiscount_", localization->FormatPrice(locale, OvertimePriceDiscount));
        SubstGlobal(upsaleMessage, "_RemainingDuration_", localization->HoursFormat(TimeRemaining, locale));
        result["upsale_message"] = upsaleMessage;
    }
    if (UpsaleTitle && localization) {
        auto upsaleTitle = localization->GetLocalString(locale, UpsaleTitle);
        SubstGlobal(upsaleTitle, "_OvertimePriceDiscount_", localization->FormatPrice(locale, OvertimePriceDiscount));
        SubstGlobal(upsaleTitle, "_RemainingDuration_", localization->HoursFormat(TimeRemaining, locale));
        result["upsale_title"] = upsaleTitle;
    }

    return result;
}

TString TPackOfferState::FormDescriptionElement(const TString& value, const TString& currency, ELocalization locale, const ILocalization& localization) const {
    TString result = value;
    SubstGlobal(result, "_Mileage_", localization.DistanceFormatKm(locale, Mileage, true));
    SubstGlobal(result, "_OvertimePrice_", GetLocalizedPrice(OvertimePrice, currency, locale, localization));
    SubstGlobal(result, "_OverrunPrice_", GetLocalizedPrice(OverrunPrice, currency, locale, localization));
    SubstGlobal(result, "_WaitingPrice_", GetLocalizedPrice(WaitingPrice, currency, locale, localization));
    SubstGlobal(result, "_PackPrice_", GetLocalizedPrice(PackPrice, currency, locale, localization));
    SubstGlobal(result, "_CurrentPackPrice_", GetLocalizedPrice(CurrentPackPrice, currency, locale, localization));
    SubstGlobal(result, "_UpsaleMessage_", UpsaleMessage);
    SubstGlobal(result, "_UpsaleTitle_", UpsaleTitle);
    SubstGlobal(result, "_WaitingDuration_", localization.DaysFormat(WaitingDuration, locale));
    SubstGlobal(result, "_PackRemainingTime_", localization.DaysFormat(PackRemainingTime, locale));
    SubstGlobal(result, "_RemainingDuration_", localization.MonthsFormat(GetPackRemainingTime(), locale));
    SubstGlobal(result, "_RemainingTime_", localization.MonthsFormat(GetPackRemainingTime(), locale));
    SubstGlobal(result, "_UsedTime_", localization.DaysFormat(UsedTime, locale));
    SubstGlobal(result, "_OvertimePriceDiscount_", GetLocalizedPrice(OvertimePriceDiscount, currency, locale, localization));
    SubstGlobal(result, "_DistanceRemaining_", localization.DistanceFormatKm(locale, DistanceRemaining, true));
    SubstGlobal(result, "_Overrun_", localization.DistanceFormatKm(locale, Overrun, true));
    SubstGlobal(result, "_Overtime_", localization.DaysFormat(Overtime, locale));

    result = TBase::FormDescriptionElement(result, currency, locale, localization);
    return localization.ApplyResources(result, locale);
}

NJson::TJsonValue TPackOffer::DoBuildJsonReport(const TReportOptions& options, const ICommonOfferBuilderAction* constructor, const NDrive::IServer& server) const {
    NJson::TJsonValue packOffer = TBase::DoBuildJsonReport(options, constructor, server);
    if (!(options.Traits & NDriveSession::ReportOfferDetails)) {
        return packOffer;
    }
    packOffer.InsertValue("extension", ExtraDuration.Seconds());
    TDuration duration = Duration + ExtraDuration;
    TInstant now = Now();
    auto locale = options.Locale;
    packOffer.InsertValue("duration", duration.Seconds());
    packOffer.InsertValue("start_instant", now.Seconds());
    packOffer.InsertValue("finish_instant", (FinishInstant ? *FinishInstant : (now + duration)).Seconds());
    packOffer.InsertValue("pack_price", GetPublicDiscountedPackPrice({}, 100));
    packOffer.InsertValue("pack_price_undiscounted", GetPublicOriginalPackPrice());
    auto mileageLimit = MileageLimit + ExtraMileageLimit;
    packOffer.InsertValue("mileage_limit", mileageLimit);
    packOffer.InsertValue("overrun_price", GetPublicDiscountedPrice(GetOverrunKm()));
    packOffer.InsertValue("overtime_price", GetPublicDiscountedPrice(GetOvertimeRiding()));
    packOffer.InsertValue("rerun_price_km", GetPublicDiscountedPrice(GetOverrunKm()));
    if (PackInsurancePrice) {
        packOffer.InsertValue("pack_insurance_price", GetPublicDiscountedPrice(GetPackInsurancePriceRef()));
    }
    if (PackInsuranceOverrunPrice) {
        packOffer.InsertValue("pack_insurance_overrun_price", GetPublicDiscountedPrice(GetPackInsuranceOverrunPriceRef()));
    }
    if (PackPriceSchedule) {
        packOffer.InsertValue("pack_price_schedule", PackPriceSchedule->GetReport(locale, server));
    }
    auto packOfferBuilder = dynamic_cast<const TPackOfferConstructor*>(constructor);
    if (packOfferBuilder) {
        const auto& detailedDescription = ((GetMileageLimit() == 0) && packOfferBuilder->HasZeroMileageDetailedDescription())
            ? packOfferBuilder->GetZeroMileageDetailedDescriptionRef()
            : packOfferBuilder->GetDetailedDescription();
        packOffer.InsertValue("detailed_description", FormDescriptionElement(detailedDescription, locale, server.GetLocalization()));
    }
    if (UseMileageForSchedule) {
        packOffer.InsertValue("use_mileage_for_schedule", UseMileageForSchedule);
    }
    return packOffer;
}

TString TPackOffer::GetTypeNameStatic() {
    return TPackOfferConstructor::GetTypeStatic();
}

TString TPackOffer::GetTypeName() const {
    return GetTypeNameStatic();
}

TString TPackOffer::GetDefaultGroupName() const {
    return "Пакетный";
}

void TPackOffer::FillBillPack(TBill& bill, const TOfferPricing& pricing, TOfferStatePtr segmentState, ELocalization locale, const NDrive::IServer* server) const {
    Y_UNUSED(segmentState);
    auto localization = server ? server->GetLocalization() : nullptr;
    {
        auto& records = bill.MutableRecords();
        auto pack = pricing.GetPrices().find(PackSegmentName);
        if (pack != pricing.GetPrices().end()) {
            const TOfferSegment& segment = pack->second;

            TBillRecord record;
            record.SetCost(segment.GetOriginalPrice());
            record.SetDuration(segment.GetDuration());
            record.SetTitle(NDrive::TLocalization::FormalTariffName(GetName()));
            record.SetType(PackSegmentName);
            records.push_back(std::move(record));
        }

        auto overrun = pricing.GetPrices().find(OverrunSegmentName);
        if (overrun != pricing.GetPrices().end()) {
            const TOfferSegment& segment = overrun->second;

            TBillRecord record;
            record.SetCost(segment.GetOriginalPrice());
            if (MileageLimit > 0) {
                record.SetDetails(localization->DistanceFormatKm(locale, segment.GetDistance()));
                record.SetTitle(NDrive::TLocalization::OverrunReceiptElementTitle());
            } else {
                record.SetTitle(localization->DistanceFormatKm(locale, segment.GetDistance()));
            }
            record.SetType(OverrunSegmentName);
            records.push_back(std::move(record));
        }

        auto overtime = pricing.GetPrices().find(OvertimeSegmentName);
        if (overtime != pricing.GetPrices().end()) {
            const TOfferSegment& segment = overtime->second;

            TBillRecord record;
            record.SetCost(segment.GetOriginalPrice());
            record.SetDetails(localization->FormatDuration(locale, segment.GetDuration(), true));
            record.SetTitle(NDrive::TLocalization::OvertimeReceiptElementTitle());
            record.SetType(OvertimeSegmentName);
            records.push_back(std::move(record));
        }
    }
}

void TPackOffer::PatchSessionReport(NJson::TJsonValue& result, NDriveSession::TReportTraits traits, ELocalization locale,
                                    const NDrive::IServer& server, const TOfferSessionPatchData& patchData) const {
    TBase::PatchSessionReport(result, traits, locale, server, patchData);

    auto expectedDuration = GetTotalDuration();
    auto expectedFinish = patchData.StartInstant + expectedDuration;
    result.InsertValue("expected_duration", expectedDuration.Seconds());
    result.InsertValue("expected_finish", expectedFinish.Seconds());
}

void TPackOffer::FillBillPricing(TBill& bill, const TOfferPricing& pricing, TOfferStatePtr segmentState, ELocalization locale, const NDrive::IServer* server) const {
    auto state = dynamic_cast<const TPackOfferState*>(segmentState.Get());
    if (!state) {
        TBase::FillBillPricing(bill, pricing, segmentState, locale, server);
        return;
    }
    if (!state->GetRecalculatedOnFinish()) {
        FillBillPack(bill, pricing, segmentState, locale, server);

        TOfferPricing pricesLocal = pricing;
        pricesLocal.CleanOtherPrices({"old_state_parking", "old_state_acceptance", "old_state_reservation"});

        TBase::FillBillPricing(bill, pricesLocal, segmentState, locale, server);
    } else {
        TBase::FillBillPricing(bill, pricing, segmentState, locale, server);
        auto overrun = pricing.GetPrices().find(OverrunSegmentName);
        if (overrun != pricing.GetPrices().end()) {
            const TOfferSegment& segment = overrun->second;

            TBillRecord record;
            record.SetCost(segment.GetOriginalPrice());
            record.SetDetails(server->GetLocalization()->DistanceFormatKm(locale, segment.GetDistance()));
            record.SetTitle(NDrive::TLocalization::OverrunReceiptElementTitle());
            record.SetType(OverrunSegmentName);
            bill.MutableRecords().emplace_back(std::move(record));
        }
    }
}

void TPackOffer::FillBill(TBill& bill, const TOfferPricing& pricing, TOfferStatePtr segmentState, ELocalization locale, const NDrive::IServer* server, ui32 cashbackPercent) const {
    auto state = dynamic_cast<const TPackOfferState*>(segmentState.Get());
    if (!state || state->GetRecalculatedOnFinish()) {
        TBase::FillBill(bill, pricing, segmentState, locale, server, GetOnlyOriginalOfferCashback() ? 0 : cashbackPercent);
    } else {
        TBase::FillBill(bill, pricing, segmentState, locale, server, cashbackPercent);
    }
}

ui32 TPackOffer::GetDeposit() const {
    if (GetUseDeposit()) {
        if (!GetDepositAmountModelInfos().empty()) {
            return GetDepositAmount();
        }
        return GetPublicDiscountedPackPrice();
    } else {
        return 0;
    }
}

bool TPackOffer::GetFreeAndPricedDurations(TDuration& freeDuration, TDuration& pricedDuration, const TMap<TString, TOfferSegment>& segments) const {
    auto itReservation = segments.find("old_state_reservation");
    const TDuration reservationDuration = (itReservation != segments.end()) ? itReservation->second.GetDuration() : TDuration::Zero();
    auto itAcceptance = segments.find("old_state_acceptance");
    const TDuration acceptanceDuration = (itAcceptance != segments.end()) ? itAcceptance->second.GetDuration() : TDuration::Zero();

    if (GetIsParkingIncludeInPack()) {
        TDuration sumDuration = TDuration::Zero();
        for (auto&& i : segments) {
            sumDuration += i.second.GetDuration();
        }
        if (reservationDuration > GetFreeDuration("old_state_reservation")) {
            freeDuration = GetFreeDuration("old_state_reservation");
            pricedDuration = sumDuration - freeDuration;
            return true;
        }
        freeDuration = reservationDuration + Min(acceptanceDuration, GetFreeDuration("old_state_acceptance"));
        pricedDuration = sumDuration - freeDuration;
        return true;
    } else {
        return TBase::GetFreeAndPricedDurations(freeDuration, pricedDuration, segments);
    }
}

TDuration TPackOffer::GetFreeTime(const TDuration usedDuration, const TString& tagName) const {
    if (tagName == "old_state_parking" && !GetIsParkingIncludeInPack()) {
        return TBase::GetFreeTime(usedDuration, tagName);
    }
    if (tagName == "old_state_reservation" || tagName == "old_state_acceptance") {
        return TBase::GetFreeTime(usedDuration, tagName);
    } else {
        return TDuration::Zero();
    }
}

TMaybe<TInstant> TPackOffer::GetStart() const {
    return {};
}

TString TPackOffer::DoFormDescriptionElement(const TString& value, ELocalization locale, const ILocalization* localization) const {
    TString result = TStandartOffer::DoFormDescriptionElement(value, locale, localization);

    if (FinishInstant) {
        const TString finishInstantString = localization->FormatInstant(locale, (*FinishInstant + TDuration::Hours(3)));
        SubstGlobal(result, "_PackFinishInstant_", finishInstantString);
    }

    const TString packDuration = localization->FormatDuration(locale, Duration);
    SubstGlobal(result, "_PackDuration_", packDuration);

    SubstGlobal(result, "_PackDurationHours_", localization->HoursFormat(Duration, locale));
    SubstGlobal(result, "_PackDurationMinutes_", localization->MinutesFormat(Duration, locale));
    SubstGlobal(result, "_PackDurationDays_", localization->DaysFormat(Duration, locale));
    SubstGlobal(result, "_PackDurationWeeks_", localization->WeeksFormat(Duration, locale));
    SubstGlobal(result, "_PackDurationHoursOrDays_", Duration.Days() >= 1 ? localization->DaysFormat(Duration, locale) : localization->HoursFormat(Duration, locale));
    SubstGlobal(result, "_RecalcMinutesDuration_", localization->FormatDuration(locale, ReturningDuration));

    const TString packPrice = localization->FormatPrice(locale, GetPublicOriginalPrice(PackPrice, "", 100));
    SubstGlobal(result, "_PackPrice_", packPrice);
    const TString overrunPrice = localization->FormatPrice(locale, GetPublicOriginalPrice(GetOverrunKm()));
    SubstGlobal(result, "_OverrunPrice_", overrunPrice);
    const TString overtimePrice = localization->FormatPrice(locale, GetPublicOriginalPrice(GetOvertimeRiding()));
    SubstGlobal(result, "_OvertimePrice_", overtimePrice);

    const TString packPriceDiscount = localization->FormatPrice(locale, GetPublicDiscountedPrice(PackPrice, "", 100));
    SubstGlobal(result, "_PackPriceDiscount_", packPriceDiscount);
    const TString overrunPriceDiscount = localization->FormatPrice(locale, GetPublicDiscountedPrice(GetOverrunKm()));
    SubstGlobal(result, "_OverrunPriceDiscount_", overrunPriceDiscount);
    const TString overtimePriceDiscount = localization->FormatPrice(locale, GetPublicDiscountedPrice(GetOvertimeRiding()));
    SubstGlobal(result, "_OvertimePriceDiscount_", overtimePriceDiscount);

    const TString mileageLimit = ::ToString(MileageLimit);
    SubstGlobal(result, "_MileageLimit_", mileageLimit);
    return result;
}

TString TPackOffer::DoBuildCalculationDescription(ELocalization locale, const TFullCompiledRiding& fcr, const NDrive::IServer& server) const {
    TStringBuilder sb;
    sb << TBase::DoBuildCalculationDescription(locale, fcr, server) << Endl;
    sb << "\\subsection{Pack}" << Endl;
    sb << "\\begin{enumerate}" << Endl;
    sb << "\\item Duration = " << server.GetLocalization()->FormatDuration(locale, Duration, true) << "+" << server.GetLocalization()->FormatDuration(locale, ExtraDuration, true) << Endl;
    sb << "\\item Pack price = " << server.GetLocalization()->FormatPrice(locale, PackPrice) << Endl;
    sb << "\\item Mileage limit = " << MileageLimit << " + " << ExtraMileageLimit << Endl;
    sb << "\\item Extra mileage price = " << server.GetLocalization()->FormatPrice(locale, GetOverrunKm()) << Endl;
    sb << "\\item Extra mileage final price = " << server.GetLocalization()->FormatPrice(locale, GetPublicDiscountedPrice(GetOverrunKm())) << Endl;
    sb << "\\item Returning duration = " << server.GetLocalization()->FormatDuration(locale, ReturningDuration, true) << Endl;
    if (FinishInstant) {
        sb << "\\item Finish instant = " << server.GetLocalization()->FormatInstant(locale, *FinishInstant) << Endl;
    }
    sb << "\\item OvertimeParkingIsRiding = " << OvertimeParkingIsRiding << Endl;
    sb << "\\item IsParkingIncludeInPackFlag = " << IsParkingIncludeInPackFlag << Endl;
    sb << "\\end{enumerate}" << Endl;
    return sb;
}

TPackOffer::TRecalcByMinutesInfo TPackOffer::CheckRecalcByMinutes(const TRidingInfo& rInfo, const TInstant startPack) const {
    const bool needRecalcInterval = (rInfo.GetLastInstant() < startPack + GetReturningDuration());
    if (rInfo.OptionalDelegationType() == ECarDelegationType::P2PPassOffer) {
        return TRecalcByMinutesInfo(false, false, true, false);
    }
    if (rInfo.GetIsSwitch()) {
        return TRecalcByMinutesInfo(false, false, true, true);
    }
    if (!rInfo.GetIsFinished()) {
        return TRecalcByMinutesInfo(false, needRecalcInterval, true, false);
    }
    return TRecalcByMinutesInfo(needRecalcInterval, false, true, false);
}

bool TPackOffer::DeserializePackFromProto(const NDrive::NProto::TPackOffer& info) {
    if (info.HasDuration()) {
        Duration = TDuration::Seconds(info.GetDuration());
    } else {
        Duration = TInstant::Seconds(info.GetPackFinishInstant()) - TInstant::Seconds(info.GetPackStartInstant());
    }
    if (info.HasIsParkingIncludeInPackFlag()) {
        IsParkingIncludeInPackFlag = info.GetIsParkingIncludeInPackFlag();
    }
    PackPrice = info.GetPackPrice();
    PackPriceQuantumByMonths = info.GetPackPriceQuantumByMonths();
    MileageLimit = info.GetMileageLimit();
    SetOverrunKm(info.GetRerunPriceKM());
    ReturningDuration = TDuration::Seconds(info.GetReturningDuration());
    if (info.HasOvertimeParkingIsRiding()) {
        OvertimeParkingIsRiding = info.GetOvertimeParkingIsRiding();
    }
    if (info.HasFinishInstant()) {
        FinishInstant = TInstant::Seconds(info.GetFinishInstant());
    }
    if (info.HasDistancePushThreshold()) {
        DistancePushThreshold = info.GetDistancePushThreshold();
    }
    if (info.HasDurationPushThreshold()) {
        DurationPushThreshold = TDuration::Seconds(info.GetDurationPushThreshold());
    }
    if (info.HasExtraDuration()) {
        ExtraDuration = TDuration::Seconds(info.GetExtraDuration());
    }
    if (info.HasExtraMileageLimit()) {
        ExtraMileageLimit = info.GetExtraMileageLimit();
    }
    if (info.HasRemainingDurationPushThreshold()) {
        RemainingDurationPushThreshold = TDuration::Seconds(info.GetRemainingDurationPushThreshold());
    }
    if (info.HasRemainingDistancePushThreshold()) {
        RemainingDistancePushThreshold = info.GetRemainingDistancePushThreshold();
    }
    if (const TString& shortName = info.GetShortName()) {
        SetShortName(shortName);
    }
    if (info.HasCriticalDistanceRemainder()) {
        CriticalDistanceRemainder = info.GetCriticalDistanceRemainder();
    }
    if (info.HasCriticalDurationRemainder()) {
        CriticalDurationRemainder = TDuration::Seconds(info.GetCriticalDurationRemainder());
    }
    if (info.HasPackInsurancePrice()) {
        PackInsurancePrice = info.GetPackInsurancePrice();
    }
    if (info.HasPackInsuranceOverrunPrice()) {
        PackInsuranceOverrunPrice = info.GetPackInsuranceOverrunPrice();
    }
    if (info.HasPackPriceQuantum()) {
        PackPriceQuantum = info.GetPackPriceQuantum();
    }
    if (info.HasPackPriceQuantumPeriod()) {
        PackPriceQuantumPeriod = TDuration::Seconds(info.GetPackPriceQuantumPeriod());
    }
    if (info.HasPackPriceSchedule()) {
        PackPriceSchedule.ConstructInPlace();
        if (!PackPriceSchedule->Deserialize(info.GetPackPriceSchedule())) {
            return false;
        }
    }
    for (auto&& i : info.GetDurationModelInfo()) {
        DurationModelInfos.push_back({i.GetName(), i.GetBefore(), i.GetAfter()});
    }
    for (auto&& i : info.GetPackPriceModelInfo()) {
        PackPriceModelInfos.push_back({i.GetName(), i.GetBefore(), i.GetAfter()});
    }
    for (auto&& i : info.GetMileageLimitModelInfo()) {
        MileageLimitModelInfos.push_back({i.GetName(), i.GetBefore(), i.GetAfter()});
    }
    for (auto&& i : info.GetOverrunKmModelInfo()) {
        OverrunKmModelInfos.push_back({i.GetName(), i.GetBefore(), i.GetAfter()});
    }
    for (auto&& i : info.GetPackInsurancePriceModelInfo()) {
        PackInsurancePriceModelInfos.push_back({i.GetName(), i.GetBefore(), i.GetAfter()});
    }
    if (info.HasUseMileageForSchedule()) {
        UseMileageForSchedule = info.GetUseMileageForSchedule();
    }
    return true;
}

bool TPackOffer::DeserializeFromProto(const NDrive::NProto::TOffer& info) {
    if (!TBase::DeserializeFromProto(info)) {
        return false;
    }
    return DeserializePackFromProto(info.GetPackOffer());
}

namespace {
    // AssignProtoModelInfo helps in assigning model info to proto.
    void AssignProtoModelInfo(NDrive::NProto::TPriceModelInfo* proto, const TPriceModelInfo& value) {
        Y_ENSURE(proto);
        proto->SetName(value.Name);
        proto->SetBefore(value.Before);
        proto->SetAfter(value.After);
    }
}

void TPackOffer::SerializePackToProto(NDrive::NProto::TPackOffer& info) const {
    info.SetDuration(Duration.Seconds());
    info.SetPackPrice(PackPrice);
    info.SetPackPriceQuantumByMonths(PackPriceQuantumByMonths);
    info.SetMileageLimit(MileageLimit);
    info.SetRerunPriceKM(GetOverrunKm());
    info.SetShortName(GetShortName());
    info.SetReturningDuration(ReturningDuration.Seconds());
    info.SetOvertimeParkingIsRiding(OvertimeParkingIsRiding);
    info.SetIsParkingIncludeInPackFlag(IsParkingIncludeInPackFlag);
    if (FinishInstant) {
        info.SetFinishInstant(FinishInstant->Seconds());
    }
    if (DistancePushThreshold) {
        info.SetDistancePushThreshold(DistancePushThreshold);
    }
    if (DurationPushThreshold.Seconds()) {
        info.SetDurationPushThreshold(DurationPushThreshold.Seconds());
    }
    if (ExtraDuration) {
        info.SetExtraDuration(ExtraDuration.Seconds());
    }
    if (ExtraMileageLimit) {
        info.SetExtraMileageLimit(ExtraMileageLimit);
    }
    if (RemainingDurationPushThreshold.Seconds()) {
        info.SetRemainingDurationPushThreshold(RemainingDurationPushThreshold.Seconds());
    }
    if (RemainingDistancePushThreshold) {
        info.SetRemainingDistancePushThreshold(RemainingDistancePushThreshold);
    }
    if (CriticalDistanceRemainder) {
        info.SetCriticalDistanceRemainder(*CriticalDistanceRemainder);
    }
    if (CriticalDurationRemainder) {
        info.SetCriticalDurationRemainder(CriticalDurationRemainder->Seconds());
    }
    if (PackInsurancePrice) {
        info.SetPackInsurancePrice(*PackInsurancePrice);
    }
    if (PackInsuranceOverrunPrice) {
        info.SetPackInsuranceOverrunPrice(*PackInsuranceOverrunPrice);
    }
    if (PackPriceQuantum) {
        info.SetPackPriceQuantum(PackPriceQuantum);
    }
    if (PackPriceQuantumPeriod) {
        info.SetPackPriceQuantumPeriod(PackPriceQuantumPeriod.Seconds());
    }
    if (PackPriceSchedule) {
        auto packPriceSchedule = info.MutablePackPriceSchedule();
        if (packPriceSchedule) {
            *packPriceSchedule = PackPriceSchedule->Serialize();
        }
    }
    for (auto&& value : GetDurationModelInfos()) {
        AssignProtoModelInfo(info.AddDurationModelInfo(), value);
    }
    for (auto&& value : GetPackPriceModelInfos()) {
        AssignProtoModelInfo(info.AddPackPriceModelInfo(), value);
    }
    for (auto&& value : GetMileageLimitModelInfos()) {
        AssignProtoModelInfo(info.AddMileageLimitModelInfo(), value);
    }
    for (auto&& value : GetOverrunKmModelInfos()) {
        AssignProtoModelInfo(info.AddOverrunKmModelInfo(), value);
    }
    for (auto&& value : GetPackInsurancePriceModelInfos()) {
        AssignProtoModelInfo(info.AddPackInsurancePriceModelInfo(), value);
    }
    info.SetUseMileageForSchedule(UseMileageForSchedule);
}

NDrive::NProto::TOffer TPackOffer::SerializeToProto() const {
    auto info = TBase::SerializeToProto();
    SerializePackToProto(*info.MutablePackOffer());
    return info;
}

TOfferStatePtr TPackOffer::DoCalculate(const TVector<IEventsSession<TCarTagHistoryEvent>::TTimeEvent>& timeline, const TVector<TAtomicSharedPtr<TCarTagHistoryEvent>>& events, const TInstant& until, const TRidingInfo& ridingInfo, TOfferPricing& result) const {
    auto evlog = NDrive::GetThreadEventLogger();
    auto server = NDrive::HasServer() ? &NDrive::GetServerAs<NDrive::IServer>() : nullptr;
    if (timeline.empty() || timeline.front().GetTimeEvent() == IEventsSession<TCarTagHistoryEvent>::EEvent::CurrentFinish) {
        return nullptr;
    }
    double distanceTravelled = 0;
    TSnapshotsDiffCompilation sdc;
    if (!sdc.Fill(timeline, events)) {
        return nullptr;
    }
    auto snapshotsDiff = sdc.GetSnapshotsDiff(server);
    auto mileage = snapshotsDiff ? snapshotsDiff->OptionalMileage() : Nothing();
    if (mileage) {
        distanceTravelled = std::max(distanceTravelled, *mileage);
    }

    ui32 distanceThresholdPushSent = 0;
    ui32 durationThresholdPushSent = 0;
    for (auto&& i : timeline) {
        if (i.GetTimeEvent() != IEventsSession<TCarTagHistoryEvent>::EEvent::Tag) {
            continue;
        }

        const TEventsSnapshot* evSnapshot = (*events[i.GetEventIndex()])->GetObjectSnapshotAs<TEventsSnapshot>();
        if (evSnapshot) {
            for (auto&& push : evSnapshot->GetPushes()) {
                distanceThresholdPushSent += push.Type == DistanceThresholdPushName;
                durationThresholdPushSent += push.Type == DurationThresholdPushName;
            }
        }
    }
    TDuration paymentReserve = ridingInfo.GetSegmentDuration("old_state_reservation") - GetFreeDuration("old_state_reservation");
    TDuration paymentAcceptance = ridingInfo.GetSegmentDuration("old_state_acceptance") - GetFreeDuration("old_state_acceptance");
    TDuration paymentParking = ridingInfo.GetSegmentDuration("old_state_parking") - GetFreeDuration("old_state_parking");
    TDuration waitingPaymentDuration = paymentReserve + paymentAcceptance + paymentParking;

    bool packStarted = false;
    TDuration duration = GetTotalDuration();
    TInstant startPack;
    TInstant finishPack;
    if (!GetIsParkingIncludeInPack()) {
        packStarted = ridingInfo.HasSegment("old_state_riding");
        if (packStarted) {
            startPack = *ridingInfo.GetSegmentStart("old_state_riding");
        } else {
            startPack = ridingInfo.GetLastInstant();
        }
        if (!ridingInfo.GetInstantOnSegmentsDuration({"old_state_riding"}, duration, finishPack)) {
            WARNING_LOG << "Very strange - empty ridingInfo" << Endl;
            finishPack = startPack + duration;
        }
    } else {
        packStarted = (waitingPaymentDuration != TDuration::Zero());
        if (paymentReserve) {
            startPack = *ridingInfo.GetSegmentStart("old_state_reservation") + GetFreeDuration("old_state_reservation");
        } else if (paymentAcceptance) {
            startPack = *ridingInfo.GetSegmentStart("old_state_acceptance") + GetFreeDuration("old_state_acceptance");
        } else if (ridingInfo.HasSegment("old_state_riding")) {
            startPack = *ridingInfo.GetSegmentStart("old_state_riding");
            packStarted = true;
        } else if (ridingInfo.HasSegment("old_state_acceptance")) {
            startPack = *ridingInfo.GetSegmentStart("old_state_acceptance") + GetFreeDuration("old_state_acceptance");
        } else if (ridingInfo.HasSegment("old_state_reservation")) {
            startPack = *ridingInfo.GetSegmentStart("old_state_reservation") + GetFreeDuration("old_state_reservation");
        }
        paymentReserve = TDuration::Zero();
        paymentAcceptance = TDuration::Zero();
        paymentParking = TDuration::Zero();
        waitingPaymentDuration = TDuration::Zero();
        finishPack = startPack + duration;
    }
    if (FinishInstant && finishPack > *FinishInstant) {
        finishPack = *FinishInstant;
    }

    const TInstant borderTime = ridingInfo.GetLastInstant();
    TOfferPricing resultMinutes(result.GetOffer());
    if (!TStandartOffer::DoCalculate(timeline, events, borderTime, ridingInfo, resultMinutes)) {
        if (evlog) {
            evlog->AddEvent(NJson::TMapBuilder
                ("event", "CalculatePackOfferError")
                ("type", "StandardFullCalculationFailed")
            );
        }
        return nullptr;
    }

    auto packServicingDuration = TDuration::Zero();
    auto servicingResults = TBillingSession::CalcServicingResults(timeline, events, borderTime);
    for (auto&& servicingResult : servicingResults) {
        auto info = servicingResult.Info.Get();
        if (!info) {
            continue;
        }
        if (info->Start < finishPack) {
            packServicingDuration += Min<TInstant>(info->Finish, finishPack) - info->Start;
        }
    }
    Y_ASSERT(packServicingDuration <= duration);
    auto servicingPrice = duration ? (PackPrice * packServicingDuration.Seconds()) / duration.Seconds() : 0;
    Y_ASSERT(servicingPrice <= PackPrice);
    auto packPrice = PackPrice > servicingPrice ? PackPrice - servicingPrice : 0;

    auto state = MakeAtomicShared<TPackOfferState>();
    const TRecalcByMinutesInfo finishedForRecalc = CheckRecalcByMinutes(ridingInfo, startPack);
    if (finishedForRecalc.GetIsSwitching()) {
        state->SetNeedFinishFees(false);
    }
    if (finishedForRecalc.GetNeedRecalcNow()) {
        result = resultMinutes;
        if (finishedForRecalc.GetNeedDistancePricing() && GetPublicOriginalOverrunPrice(distanceTravelled) > 0) {
            result.AddSegmentInfo(OverrunSegmentName, GetPublicDiscountedOverrunPrice(distanceTravelled), GetPublicOriginalOverrunPrice(distanceTravelled), TDuration::Zero(), distanceTravelled, 0, GetPublicDiscountedOverrunPrice(distanceTravelled));
        }
        if (false) {
            evlog->AddEvent(NJson::TMapBuilder
                ("event", "CalculatePackOffer")
                ("recalc", NJson::ToJson(std::make_tuple(
                    finishedForRecalc.GetNeedRecalcNow(),
                    finishedForRecalc.GetNeedRecalcInFuture(),
                    finishedForRecalc.GetNeedDistancePricing(),
                    finishedForRecalc.GetIsSwitching()
                )))
                ("result", NJson::ToJson(result))
            );
        }
        state->SetRecalculatedOnFinish(true);
        return state;
    }
    if (finishedForRecalc.GetIsSwitching()) {
        packStarted = true;
    }

    const double mileageLimit = GetMileageLimit() + GetExtraMileageLimit();
    const double distanceRemaining = std::max(mileageLimit - distanceTravelled, 0.0);
    const double overrun = std::max(distanceTravelled - mileageLimit, 0.0);

    TDuration timeRemaining = finishPack - borderTime;
    TDuration overtime = borderTime - finishPack;

    state->SetMileage(distanceTravelled);
    state->SetRemainingDistance(distanceRemaining);
    state->SetRemainingTime(timeRemaining);
    state->SetPackRemainingTime(packStarted ? timeRemaining : duration);
    state->SetUsedTime(borderTime - startPack);
    state->SetOverrun(overrun);
    state->SetOverrunPrice(GetPublicDiscountedOverrunPrice(overrun));
    state->SetPackPrice(GetPublicDiscountedPackPrice(packPrice));
    state->SetDistanceThresholdPushSent(distanceThresholdPushSent);
    state->SetDurationThresholdPushSent(durationThresholdPushSent);
    state->SetServicingDuration(packServicingDuration);
    state->SetServicingMileage(sdc.GetServicingMileage());
    state->SetServicingOmittedPrice(servicingPrice);
    state->SetSince(startPack);
    state->SetUntil(finishPack);

    bool shouldReportRemainders = true;
    bool criticalDistanceRemainder = CriticalDistanceRemainder ? *CriticalDistanceRemainder > state->GetRemainingDistance() : true;
    bool criticalDurationRemainder = CriticalDurationRemainder ? *CriticalDurationRemainder > state->GetRemainingTime() : true;
    if (shouldReportRemainders) {
        state->SetShowRemainders(criticalDistanceRemainder || criticalDurationRemainder);
    }

    bool shouldReportUpsale = ShouldReportUpsale();
    bool upsaleDurationRemainder = RemainingDurationPushThreshold
        ? RemainingDurationPushThreshold > state->GetRemainingTime()
        : DurationPushThreshold < state->GetUsedTime();
    if (upsaleDurationRemainder && shouldReportUpsale) {
        state->SetOvertimePriceDiscount(GetPublicDiscountedPrice(GetOvertimeRiding()));
        state->SetUpsaleable(GetSwitchable());
        if (timeRemaining) {
            state->SetUpsaleMessage("offers.pack_offer.upsale.with_remaining_time_message");
            state->SetUpsaleTitle("offers.pack_offer.upsale.with_remaining_time_title");
        } else {
            state->SetUpsaleMessage("offers.pack_offer.upsale.no_remaining_time_message");
            state->SetUpsaleTitle("offers.pack_offer.upsale.no_remaining_time_title");
        }
    }

    double overtimePriceOriginal;
    TMaybe<TOfferPricing> standartBase;
    if (packStarted && (borderTime > finishPack)) {
        standartBase.ConstructInPlace(result.GetOffer());
        if (!TStandartOffer::DoCalculate(timeline, events, finishPack, ridingInfo, *standartBase)) {
            if (evlog) {
                evlog->AddEvent(NJson::TMapBuilder
                    ("event", "CalculatePackOfferError")
                    ("type", "StandardBaseCalculationFailed")
                );
            }
            return nullptr;
        }

        TOfferPricing overtimePricing = resultMinutes - *standartBase;
        if (GetOvertimeParking() && GetOvertimeParkingIsRiding()) {
            overtimePricing.MulPrices(1.0 * GetOvertimeRiding() / GetOvertimeParking(), { TChargableTag::Parking, TChargableTag::Reservation, TChargableTag::Acceptance });
        }
        if (!GetIsParkingIncludeInPack()) {
            overtimePricing.Remove({ TChargableTag::Parking, TChargableTag::Reservation, TChargableTag::Acceptance });
        }
        state->SetOvertime(overtime);
        state->SetOvertimePrice(overtimePricing.GetReportSumPrice());
        overtimePriceOriginal = overtimePricing.GetReportSumOriginalPrice();
        state->AddPricing(std::move(overtimePricing));
    }

    result = resultMinutes;
    if (GetIsParkingIncludeInPack()) {
        result.CleanOtherPrices({ TChargableTag::Servicing });
    } else {
        result.CleanOtherPrices({ TChargableTag::Servicing, TChargableTag::Parking, TChargableTag::Reservation, TChargableTag::Acceptance });
        state->SetWaitingDuration(waitingPaymentDuration);
        state->SetWaitingPrice(result.GetReportSumPrice());
        state->AddPricing(result);
    }

    if (packStarted) {
        auto distanceTravelledWithoutExtra = std::max(distanceTravelled - GetExtraMileageLimit(), 0.0);
        auto acknowledgedMultiplier = duration ? (1.0 * (std::min(borderTime, finishPack) - startPack).Seconds() / duration.Seconds()) : 0;
        if (GetUseMileageForSchedule() && GetMileageLimit() > 0) {
            auto mileageMultiplier = std::min(distanceTravelledWithoutExtra / GetMileageLimit(), 1.0);
            acknowledgedMultiplier = std::max(acknowledgedMultiplier, mileageMultiplier);
        }
        auto acknowledged = acknowledgedMultiplier * GetPublicDiscountedPackPrice(packPrice);
        if (PackPriceSchedule && borderTime < finishPack && (!ridingInfo.GetIsFinished() || ridingInfo.IsReplacing())) {
            auto deposit = double(0);
            auto scheduledPrice = PackPriceSchedule->GetPrice(borderTime, distanceTravelled);
            auto partialPackPrice = scheduledPrice > servicingPrice ? (scheduledPrice - servicingPrice) : 0;
            auto finalPackPrice = std::min<ui32>(packPrice, partialPackPrice);
            state->SetCurrentPackPrice(GetPublicDiscountedPrice(finalPackPrice));
            { // PackPriceScheduleByDeposit = true
                deposit = finalPackPrice;
                finalPackPrice = acknowledgedMultiplier * packPrice;
            }
            result.AddSegmentInfo(
                PackSegmentName,
                GetPublicDiscountedPrice(finalPackPrice),
                GetPublicOriginalPrice(finalPackPrice),
                /*duration=*/TDuration::Zero(),
                /*mileage=*/0,
                GetPublicDiscountedPrice(deposit),
                acknowledged
            );
        } else if (PackPriceQuantum && borderTime < finishPack && (!ridingInfo.GetIsFinished() || ridingInfo.IsReplacing())) {
            auto start = GetStart().GetOrElse(startPack);
            ui64 quanta = 0;
            if (PackPriceQuantumByMonths) {
                quanta = GetMonthsDelta(start, borderTime);
            } else {
                auto partialDuration = borderTime - start;
                auto quantumPeriod = std::max(PackPriceQuantumPeriod, TDuration::Seconds(1));
                quanta = partialDuration.Seconds() / quantumPeriod.Seconds();
            }
            auto deposit = double(0);
            auto partialPackPrice = PackPriceQuantum * (1 + quanta) - servicingPrice;
            auto finalPackPrice = std::min<ui32>(packPrice, partialPackPrice);
            state->SetCurrentPackPrice(GetPublicDiscountedPrice(finalPackPrice));
            { // PackPriceScheduleByDeposit = true
                deposit = finalPackPrice;
                finalPackPrice = acknowledgedMultiplier * packPrice;
            }
            result.AddSegmentInfo(
                PackSegmentName,
                GetPublicDiscountedPrice(finalPackPrice),
                GetPublicOriginalPrice(finalPackPrice),
                /*duration=*/TDuration::Zero(),
                /*mileage=*/0,
                GetPublicDiscountedPrice(deposit),
                acknowledged
            );
        } else {
            result.AddSegmentInfo(
                PackSegmentName,
                GetPublicDiscountedPackPrice(packPrice),
                GetPublicOriginalPackPrice(packPrice),
                TDuration::Zero(),
                0,
                0,
                acknowledged
            );
            state->SetCurrentPackPrice(GetPublicDiscountedPrice(packPrice));
        }
        if (GetPublicOriginalOverrunPrice(overrun) > 0) {
            result.AddSegmentInfo(OverrunSegmentName, GetPublicDiscountedOverrunPrice(overrun), GetPublicOriginalOverrunPrice(overrun), TDuration::Zero(), overrun, 0, GetPublicDiscountedOverrunPrice(overrun));
        }
        if (overtime && state->HasPricing()) {
            result.AddSegmentInfo(OvertimeSegmentName, state->GetOvertimePrice(), overtimePriceOriginal, overtime, 0, 0, state->GetOvertimePrice());
        }
        if (finishedForRecalc.GetNeedRecalcInFuture()) {
            result.SetBorderPrices(resultMinutes.GetReportSumPrice(), resultMinutes.GetReportSumOriginalPrice());
        }
    } else {
        state->SetCurrentPackPrice(GetDeposit());
    }
    if (false) {
        evlog->AddEvent(NJson::TMapBuilder
            ("event", "CalculatePackOffer")
            ("offer_id", GetOfferId())
            ("result", NJson::ToJson(result))
            ("standard_base", NJson::ToJson(standartBase))
            ("standard_full", NJson::ToJson(resultMinutes))
            ("servicing_duration", NJson::ToJson(packServicingDuration))
            ("servicing_mileage", NJson::ToJson(sdc.GetServicingMileage()))
            ("pack_price", packPrice)
            ("until", NJson::ToJson(until))
        );
    }
    return state;
}

TVector<TString> TPackOffer::GetDefaultShortDescription(ELocalization locale, NDriveSession::TReportTraits traits, const ILocalization& localization) const {
    TDuration freeReservationDuration = GetFreeDuration("old_state_reservation");

    TVector<TString> lines;
    if (traits & NDriveSession::ReportOfferShortDescriptionPackDetails) {
        if (MileageLimit) {
            lines.push_back(TStringBuilder() << MileageLimit << " км");
        } else {
            lines.push_back(localization.GetLocalString(locale, "tariff_short_km_price"));
        }
    }
    if (freeReservationDuration >= TDuration::Minutes(1)) {
        lines.push_back(TStringBuilder() << freeReservationDuration.Minutes() << " мин бесплатного ожидания");
    } else {
        lines.push_back(TStringBuilder() << "Нет бесплатного ожидания");
    }
    return lines;
}

bool TBasePackOfferConstructor::GetPrices(const NDrive::IServer* server, ui32& ridingPrice, ui32& parkingPrice, ui32& kmPrice) const {
    if (!!SyncStandartOffer) {
        THolder<TStandartOfferConstructor> stOffer = GetStandartOfferConstructor(server, SyncStandartOffer);
        if (!stOffer) {
            ERROR_LOG << "Incorrect offer link: " << SyncStandartOffer << Endl;
            return false;
        }
        return stOffer->GetPrices(server, ridingPrice, parkingPrice, kmPrice);
    } else {
        return TBase::GetPrices(server, ridingPrice, parkingPrice, kmPrice);
    }
}

EOfferCorrectorResult TBasePackOfferConstructor::DoBuildOffers(const TUserPermissions& permissions, TVector<IOfferReport::TPtr>& results, const TOffersBuildingContext& context, const NDrive::IServer* server, NDrive::TInfoEntitySession& session) const {
    TVector<IOfferReport::TPtr> resultsStandart;
    {
        auto resultStandartBuilding = TBase::DoBuildOffers(permissions, resultsStandart, context, server, session);
        if (resultStandartBuilding != EOfferCorrectorResult::Success) {
            return resultStandartBuilding;
        }
        if (!!SyncStandartOffer && resultsStandart.size() != 1) {
            return EOfferCorrectorResult::Problems;
        }

        if (!!SyncStandartOffer) {
            THolder<TStandartOfferConstructor> stOffer = GetStandartOfferConstructor(server, SyncStandartOffer);
            if (!stOffer) {
                ERROR_LOG << "Incorrect offer link: " << SyncStandartOffer << Endl;
                return EOfferCorrectorResult::BuildingProblems;
            }
            TVector<IOfferReport::TPtr> baseResultsStandart;
            auto resultStandartBuilding = stOffer->BuildOffersClean(permissions, baseResultsStandart, context, server, session);
            if (resultStandartBuilding != EOfferCorrectorResult::Success) {
                return resultStandartBuilding;
            }
            if (baseResultsStandart.size() != 1) {
                ERROR_LOG << "Incorrect base offers count: " << baseResultsStandart.size() << Endl;
                return EOfferCorrectorResult::Problems;
            }
            if (!resultsStandart.front() || !baseResultsStandart.front()) {
                return EOfferCorrectorResult::Problems;
            }
            if (!resultsStandart.front()->template GetOfferAs<TStandartOffer>() || !baseResultsStandart.front()->template GetOfferAs<TStandartOffer>()) {
                return EOfferCorrectorResult::Problems;
            }
            resultsStandart.front()->template GetOfferAs<TStandartOffer>()->SetPricingFrom(baseResultsStandart.front()->template GetOfferAs<TStandartOffer>());
        }
    }

    for (auto&& i : resultsStandart) {
        if (!i) {
            continue;
        }
        const TStandartOfferReport& report = *dynamic_cast<const TStandartOfferReport*>(i.Get());
        TVector<IOfferReport::TPtr> offerReports;
        if (!ConstructPackOffer(report, context, server, offerReports, session)) {
            continue;
        }
        for (auto&& offerReport : offerReports) {
            TPackOfferReport* pOfferReport = dynamic_cast<TPackOfferReport*>(offerReport.Get());
            if (!pOfferReport) {
                continue;
            }
            TPackOffer* pOffer = offerReport->GetOfferAs<TPackOffer>();
            if (!pOffer) {
                continue;
            }
            pOfferReport->SetShortDescription(GetShortDescription());
            pOffer->SetIsParkingIncludeInPackFlag(GetIsParkingIncludeInPackFlag());
            pOffer->SetUseDefaultShortDescriptions(GetUseDefaultShortDescriptions());
            pOffer->SetName(GetDescription());
            pOffer->SetShortName(GetShortName());
            pOffer->SetSubName(GetSubName());
            pOffer->SetOvertimeParkingIsRiding(GetOvertimeParkingIsRiding());
            if (!pOffer->GetDistancePushThreshold()) {
                pOffer->SetDistancePushThreshold(DistancePushThreshold);
            }
            if (!pOffer->GetDurationPushThreshold()) {
                pOffer->SetDurationPushThreshold(DurationPushThreshold);
            }
            if (!pOffer->GetRemainingDistancePushThreshold()) {
                pOffer->SetRemainingDistancePushThreshold(RemainingDistancePushThreshold);
            }
            if (!pOffer->GetRemainingDurationPushThreshold()) {
                pOffer->SetRemainingDurationPushThreshold(RemainingDurationPushThreshold);
            }
            if (!pOffer->GetPackPriceQuantum()) {
                pOffer->SetPackPriceQuantum(PackPriceQuantum);
            }
            if (!pOffer->GetPackPriceQuantumPeriod()) {
                pOffer->SetPackPriceQuantumPeriod(PackPriceQuantumPeriod);
            }
            if (!pOffer->GetReturningDuration()) {
                pOffer->SetReturningDuration(GetReturningDuration());
            }
            results.emplace_back(offerReport);
        }
    }
    return EOfferCorrectorResult::Success;
}

bool TBasePackOfferConstructor::DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonValue) {
    if (!TBase::DeserializeSpecialsFromJson(jsonValue)) {
        return false;
    }
    JREAD_INT_OPT(jsonValue, "rerun_price_km", RerunPriceKM);
    JREAD_STRING_OPT(jsonValue, "sync_standart_offer", SyncStandartOffer);
    JREAD_DURATION_OPT(jsonValue, "returning_duration", ReturningDuration);
    JREAD_UINT_OPT(jsonValue, "distance_push_threshold", DistancePushThreshold);
    JREAD_DURATION_OPT(jsonValue, "duration_push_threshold", DurationPushThreshold);
    JREAD_DURATION_OPT(jsonValue, "remaining_duration_push_threshold", RemainingDurationPushThreshold);
    JREAD_DOUBLE_OPT(jsonValue, "remaining_distance_push_threshold", RemainingDistancePushThreshold);
    JREAD_BOOL_OPT(jsonValue, "overtime_parking_is_riding", OvertimeParkingIsRiding);
    JREAD_BOOL_OPT(jsonValue, "is_parking_innclude_in_pack", IsParkingIncludeInPackFlag);
    return
        NJson::ParseField(jsonValue["pack_price_quantum"], PackPriceQuantum) &&
        NJson::ParseField(jsonValue["pack_price_quantum_period"], PackPriceQuantumPeriod);
}

NJson::TJsonValue TBasePackOfferConstructor::SerializeSpecialsToJson() const {
    NJson::TJsonValue result = TBase::SerializeSpecialsToJson();
    NJson::InsertField(result, "pack_price_quantum", PackPriceQuantum);
    NJson::InsertField(result, "pack_price_quantum_period", PackPriceQuantumPeriod);
    JWRITE(result, "rerun_price_km", RerunPriceKM);
    JWRITE(result, "sync_standart_offer", SyncStandartOffer);
    JWRITE_DURATION(result, "returning_duration", ReturningDuration);
    JWRITE(result, "distance_push_threshold", DistancePushThreshold);
    JWRITE_DURATION(result, "duration_push_threshold", DurationPushThreshold);
    JWRITE_DURATION(result, "remaining_duration_push_threshold", RemainingDurationPushThreshold);
    JWRITE(result, "remaining_distance_push_threshold", RemainingDistancePushThreshold);
    JWRITE(result, "overtime_parking_is_riding", OvertimeParkingIsRiding);
    JWRITE(result, "is_parking_innclude_in_pack", IsParkingIncludeInPackFlag);
    return result;
}

NDrive::TScheme TBasePackOfferConstructor::DoGetScheme(const NDrive::IServer* server) const {
    auto result = TBase::DoGetScheme(server);
    {
        auto gTab = result.StartTabGuard("fines");
        result.Add<TFSNumeric>("rerun_price_km", "Стоимость перепробега (копеек за км)").SetMin(0).SetDefault(800).SetDeprecated();
        result.Add<TFSDuration>("returning_duration", "Период для пересчета на поминутный").SetDefault(TDuration::Zero());
    }
    {
        auto gTab = result.StartTabGuard("prices");
        TVector<TDBAction> actions = server->GetDriveAPI()->GetRolesManager()->GetActionsDB().GetCachedObjectsVector();
        TSet<TString> offerConstructorNames;
        for (auto&& action : actions) {
            const TStandartOfferConstructor* stOffer = action.GetAs<TStandartOfferConstructor>();
            if (stOffer && stOffer->GetType() == TStandartOfferConstructor::GetTypeName()) {
                offerConstructorNames.emplace(stOffer->GetName());
            }
        }
        result.Add<TFSVariants>("sync_standart_offer", "Базовый стандартный оффер (для синхронизации)").SetVariants(offerConstructorNames).AddVariant("").SetDeprecated();
        result.Add<TFSBoolean>("overtime_parking_is_riding", "Overtime парковка по цене riding-а").SetDefault(false);
        result.Add<TFSBoolean>("is_parking_innclude_in_pack", "Парковка включена в пакет").SetDefault(true);
    }
    {
        auto gTab = result.StartTabGuard("notification");
        result.Add<TFSNumeric>("distance_push_threshold", "Порог пробега для предупреждающего пуша (км)");
        result.Add<TFSNumeric>("duration_push_threshold", "Порог длительности поездки для предупреждающего пуша (с)");
        result.Add<TFSNumeric>("remaining_duration_push_threshold", "Порог остаточного времени пакета для предупреждающего пуша (с)");
        result.Add<TFSNumeric>("remaining_distance_push_threshold", "Порог остаточного расстояния для предупреждающего пуша (км)");
    }
    return result;
}

bool TPackOfferConstructor::DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonValue) {
    if (!TBase::DeserializeSpecialsFromJson(jsonValue)) {
        return false;
    }
    JREAD_DURATION(jsonValue, "duration", Duration);
    JREAD_INSTANT_OPT(jsonValue, "finish_instant", FinishInstant);
    if (jsonValue.Has("pack_price_object")) {
        if (!TEntityPriceSettings::ReadPricingPolicy(jsonValue["pack_price_object"], PackPriceCalculatorType, PackPriceCalculatorConfig, PackPriceCalculator)) {
            return false;
        }
    } else {
        ui32 packPrice;
        JREAD_INT(jsonValue, "pack_price", packPrice);
        PackPriceCalculatorType = "constant";
        PackPriceCalculatorConfig = new TConstantPriceConfig(packPrice);
        PackPriceCalculator = PackPriceCalculatorConfig->Construct();
    }
    JREAD_INT_OPT(jsonValue, "mileage_limit", MileageLimit);
    return
        NJson::ParseField(jsonValue["zero_mileage_detailed_description"], ZeroMileageDetailedDescription);
}

NJson::TJsonValue TPackOfferConstructor::SerializeSpecialsToJson() const {
    NJson::TJsonValue result = TBase::SerializeSpecialsToJson();
    JWRITE_DURATION(result, "duration", Duration);
    JWRITE_INSTANT(result, "finish_instant", FinishInstant);
    {
        NJson::TJsonValue packPriceJson;
        packPriceJson.InsertValue("pricing_type", PackPriceCalculatorType);
        packPriceJson.InsertValue("price_details", PackPriceCalculatorConfig->SerializeToJson());
        JWRITE(result, "pack_price_object", packPriceJson);
    }
    JWRITE(result, "pack_price", PackPriceCalculator->GetBasePrice());
    JWRITE(result, "mileage_limit", MileageLimit);
    NJson::InsertField(result, "zero_mileage_detailed_description", ZeroMileageDetailedDescription);
    return result;
}

bool TPackOfferConstructor::ConstructPackOffer(const TStandartOfferReport& stOfferReport, const TOffersBuildingContext& context, const NDrive::IServer* server, TVector<IOfferReport::TPtr>& offers, NDrive::TInfoEntitySession& session) const {
    auto poc = stOfferReport.GetPriceOfferConstructor();
    if (!PackPriceCalculator) {
        session.AddErrorMessage("ConstructPackOffer", "!PackPriceCalculator");
        return false;
    }

    THolder<TPackOfferReport> offerReport = Extend(stOfferReport);
    if (!offerReport) {
        session.AddErrorMessage("ConstructPackOffer", "incorrect base class");
        return false;
    }
    TPackOffer* offer = offerReport->GetOfferAs<TPackOffer>();
    const TInstant timestamp = offer->GetTimestamp();

    if (!GetActivityInterval().Empty()) {
        TInstant finish = GetActivityInterval().GetNextCorrectionTime(timestamp);
        offer->SetDuration(finish - timestamp);
        offer->SetFinishInstant(finish);
    } else if (GetFinishInstant()) {
        offer->SetDuration(GetFinishInstant() - Now());
        offer->SetFinishInstant(GetFinishInstant());
    } else {
        offer->SetDuration(GetDuration());
        if (context.HasExtraDistance()) {
            offer->SetExtraMileageLimit(context.GetExtraDistanceUnsafe());
        }
        if (context.HasExtraDuration()) {
            offer->SetExtraDuration(context.GetExtraDurationUnsafe());
        }
    }
    if (context.HasPackOfferNameOverride()) {
        offer->SetName(context.GetPackOfferNameOverrideUnsafe());
    }
    if (poc) {
        auto packPrice = poc->CalcPackPrice(Duration, MileageLimit);
        if (!packPrice) {
            session.AddErrorMessage("PackOfferConstructor::ConstructPackOffer", "cannot CalcPackPrice");
            return false;
        }
        auto overrunPrice = poc->CalcPackOverrunPrice(Duration, MileageLimit);
        if (!overrunPrice) {
            overrunPrice = offerReport->GetRerunPriceKM().GetOrElse(GetRerunPriceKM());
        }
        offer->SetPackPrice(*packPrice);
        offer->SetOverrunKm(overrunPrice);
    } else {
        offer->SetPackPrice(PackPriceCalculator->GetBasePrice());
        offer->SetOverrunKm(offerReport->GetRerunPriceKM().GetOrElse(GetRerunPriceKM()));
    }
    offer->SetUseDeposit(GetUseDeposit());
    offer->SetDepositAmount(offer->GetPublicDiscountedPackPrice());
    offer->SetMileageLimit(MileageLimit);
    offerReport->RecalcPrices(server);
    offers.emplace_back(offerReport.Release());
    return true;
}

THolder<TPackOfferReport> TPackOfferConstructor::Extend(const TStandartOfferReport& standardOfferReport) const {
    return IOfferReport::Extend<TPackOfferReport>(standardOfferReport);
}

NDrive::TScheme TPackOfferConstructor::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);
    result.Remove("deposit_default");

    result.Add<TFSNumeric>("mileage_limit", "Ограничение по пробегу (км)").SetMin(0).SetDefault(70);
    result.Add<TFSNumeric>("duration", "Длительность пакета в секундах").SetMin(0);
    result.Add<TFSNumeric>("finish_instant", "Timestamp окончания действия пакета (GMT)").SetMin(0).SetRequired(false);
    result.Add<TFSString>("zero_mileage_detailed_description", "Detailed description when no mileage included");
    {
        NDrive::TScheme& packPriceScheme = result.Add<TFSStructure>("pack_price_object", "Политика ценообразования пакета").SetStructure<NDrive::TScheme>();
        packPriceScheme.SetDeprecated();
        packPriceScheme.Add<TFSVariants>("pricing_type", "Тип ценообразования").InitVariantsClass<IPriceCalculatorConfig>();
        packPriceScheme.Add<TFSJson>("price_details", "Настройка");
    }
    return result;
}

TPackOfferConstructor& TPackOfferConstructor::SetPackPrice(const ui32 price) {
    PackPriceCalculatorType = "constant";
    PackPriceCalculatorConfig = new TConstantPriceConfig(price);
    PackPriceCalculator = PackPriceCalculatorConfig->Construct();
    return *this;
}

ui32 TPackOffer::CalcPackPrice(const NDrive::IServer* /*server*/) const {
    return GetPackPrice();
}

void TPackOffer::ApplyPackPriceModel(const NDrive::IOfferModel& model) {
    ui32 price = 100 * model.Calc(MutableFeatures());
    PackPriceModelInfos.push_back({model.GetName(), (float)GetPackPrice(), (float)price});
    SetPackPrice(price);
}

void TPackOffer::ApplyDurationModel(const NDrive::IOfferModel& model) {
    TDuration duration = TDuration::Seconds(model.Calc(MutableFeatures()));
    DurationModelInfos.push_back({model.GetName(), (float)GetDuration().Seconds(), (float)duration.Seconds()});
    SetDuration(duration);
}

void TPackOffer::ApplyMileageLimitModel(const NDrive::IOfferModel& model) {
    ui32 mileageLimit = model.Calc(MutableFeatures());
    MileageLimitModelInfos.push_back({model.GetName(), (float)GetMileageLimit(), (float)mileageLimit});
    SetMileageLimit(mileageLimit);
}

void TPackOffer::ApplyOverrunKmModel(const NDrive::IOfferModel& model) {
    ui32 price = 100 * model.Calc(MutableFeatures());
    OverrunKmModelInfos.push_back({model.GetName(), (float)GetOverrunKm(), (float)price});
    SetOverrunKm(price);
}

void TPackOffer::ApplyPackInsurancePriceModel(const NDrive::IOfferModel& model) {
    auto original = GetPackPrice() - GetPackInsurancePriceDef(0);
    ui32 price = 100 * model.Calc(MutableFeatures());
    PackInsurancePriceModelInfos.push_back({model.GetName(), (float)GetPackInsurancePriceDef(0), (float)price});
    SetPackInsurancePrice(price);
    SetPackPrice(original + price);
}

void TPackOfferReport::DoRecalcPrices(const NDrive::IServer* server) {
    TPackOffer* pOffer = GetOfferAs<TPackOffer>();
    if (pOffer) {
        pOffer->SetPackPrice(pOffer->CalcPackPrice(server));
    }
}

void TPackOfferReport::RecalculateFeatures() {
    TBase::RecalculateFeatures();
    TPackOffer* offer = GetOfferAs<TPackOffer>();
    if (offer) {
        CalcPackOfferFeatures(*offer);
    }
}

void CalcPackOfferFeatures(TPackOffer& offer) {
    NDrive::CalcOfferPackPriceFeatures(offer.MutableFeatures(), offer.GetPackPrice(), offer.GetPublicDiscountedPackPrice());
    NDrive::CalcOfferOverrunPriceFeatures(offer.MutableFeatures(), offer.GetOverrunKm());
    NDrive::CalcOfferDurationFeatures(offer.MutableFeatures(), offer.GetDuration());
    NDrive::CalcOfferMileageLimitFeatures(offer.MutableFeatures(), offer.GetMileageLimit());
    NDrive::CalcOfferInsurancePackPriceFeatures(offer.MutableFeatures(), offer.OptionalPackInsurancePrice());
}
