#pragma once

#include "abstract.h"
#include "standart.h"

#include <library/cpp/json/writer/json_value.h>

#include <rtline/util/json_processing.h>

namespace NDrive::NProto {
    class TPriceSchedule;
}

class TPriceSchedule {
public:
    struct TQuantum {
        TInstant Timestamp;
        TMaybe<ui32> Mileage;
        ui32 Value;
        TQuantum(TInstant timestamp, TMaybe<ui32> mileage, ui32 value)
            : Timestamp(timestamp)
            , Mileage(mileage)
            , Value(value)
        {
        }

        bool ShouldBePaid(TInstant timestamp, double mileage) const {
            return Timestamp < timestamp || (Mileage && static_cast<double>(*Mileage) < mileage);
        }
    };

    using TQuanta = TVector<TQuantum>;
    using TOptionalQuantum = TMaybe<TQuantum>;

public:
    TPriceSchedule(bool useMileage = false)
        : UseMileage(useMileage)
    {}
    const TQuanta& GetQuanta() const {
        return Quanta;
    }

    void AddQuantum(TInstant timestamp, TMaybe<ui32> mileage, ui32 value);
    ui32 GetPrice(TInstant timestamp, double mileage) const;
    bool HasMileageConditions() const;
    TOptionalQuantum GetNextQuantum(TInstant timestamp, double mileage) const;

    NJson::TJsonValue GetReport(ELocalization locale, const NDrive::IServer& server) const;

    NDrive::NProto::TPriceSchedule Serialize() const;
    bool Deserialize(const NDrive::NProto::TPriceSchedule& proto);

private:
    TQuanta::const_iterator GetNextQuantumImpl(TInstant timestamp, double mileage) const;
    bool UseMileage = false;
    TQuanta Quanta;
};

class TPackOfferState: public IOfferState {
private:
    using TBase = IOfferState;

public:
    R_OPTIONAL(TOfferPricing, Pricing);
    R_OPTIONAL(TInstant, Since);
    R_OPTIONAL(TInstant, Until);
    R_FIELD(double, Mileage, 0);
    R_FIELD(double, OvertimePrice, 0);
    R_FIELD(double, OverrunPrice, 0);
    R_FIELD(double, WaitingPrice, 0);
    R_FIELD(ui32, PackPrice, 0);
    R_FIELD(ui32, CurrentPackPrice, 0);
    R_FIELD(TString, UpsaleMessage);
    R_FIELD(TString, UpsaleTitle);
    R_FIELD(bool, Upsaleable, false);
    R_FIELD(TDuration, WaitingDuration, TDuration::Zero());
    R_FIELD(bool, RecalculatedOnFinish, false);
    R_FIELD(TDuration, Overtime);
    R_FIELD(TDuration, PackRemainingTime, TDuration::Zero());
    R_FIELD(TDuration, UsedTime, TDuration::Zero());
    R_FIELD(TDuration, ServicingDuration);
    R_FIELD(double, ServicingMileage, 0);
    R_FIELD(ui32, ServicingOmittedPrice, 0);
    R_FIELD(bool, ShowRemainders, true);

    R_FIELD(ui32, OvertimePriceDiscount, 0);

public:
    TPackOfferState& AddPricing(const TOfferPricing& pricing);

    virtual NJson::TJsonValue GetReport(ELocalization locale, const NDrive::IServer& server) const override;

    virtual TString FormDescriptionElement(const TString& value, const TString& currency, ELocalization locale, const ILocalization& localization) const override;

    double GetRemainingDistance() const {
        return DistanceRemaining;
    }
    TDuration GetRemainingTime() const {
        return TimeRemaining;
    }
    bool IsDistanceThresholdPushSent() const {
        return DistanceThresholdPushSent > 0;
    }
    bool IsDurationThresholdPushSent() const {
        return DurationThresholdPushSent > 0;
    }
    void SetRemainingDistance(double value) {
        DistanceRemaining = value;
    }
    void SetOverrun(double value) {
        Overrun = value;
    }
    void SetRemainingTime(TDuration value) {
        TimeRemaining = value;
    }
    void SetDistanceThresholdPushSent(ui32 value) {
        DistanceThresholdPushSent = value;
    }
    void SetDurationThresholdPushSent(ui32 value) {
        DurationThresholdPushSent = value;
    }

private:
    double DistanceRemaining = 0;
    double Overrun = 0;
    TDuration TimeRemaining = TDuration::Zero();
    ui32 DistanceThresholdPushSent = 0;
    ui32 DurationThresholdPushSent = 0;
};

class TPackOffer: public TStandartOffer {
private:
    using TBase = TStandartOffer;

private:
    R_OPTIONAL(ui32, CriticalDistanceRemainder);
    R_OPTIONAL(TDuration, CriticalDurationRemainder);
    R_OPTIONAL(TInstant, FinishInstant);
    R_OPTIONAL(i32, PackInsurancePrice);
    R_OPTIONAL(i32, PackInsuranceOverrunPrice);
    R_OPTIONAL(TPriceSchedule, PackPriceSchedule);
    R_FIELD(TDuration, Duration, TDuration::Days(1));
    R_FIELD(TPriceModelInfos, DurationModelInfos);
    R_FIELD(TDuration, ExtraDuration, TDuration::Zero());
    R_FIELD(TDuration, PackPriceQuantumPeriod, TDuration::Zero());
    R_FIELD(bool, PackPriceQuantumByMonths, false);
    R_FIELD(ui32, PackPrice, 1000000);
    R_FIELD(TPriceModelInfos, PackPriceModelInfos);
    R_FIELD(ui32, PackPriceQuantum, 0);
    R_FIELD(ui32, MileageLimit, 70);
    R_FIELD(TPriceModelInfos, MileageLimitModelInfos);
    R_FIELD(ui32, ExtraMileageLimit, 0);
    R_FIELD(ui32, DistancePushThreshold, 0);
    R_FIELD(double, RemainingDistancePushThreshold, 0);
    R_FIELD(TDuration, DurationPushThreshold, TDuration::Zero());
    R_FIELD(TDuration, RemainingDurationPushThreshold, TDuration::Zero());
    R_FIELD(TDuration, ReturningDuration, TDuration::Zero());
    R_FIELD(bool, OvertimeParkingIsRiding, false);
    R_FIELD(bool, IsParkingIncludeInPackFlag, true);
    R_FIELD(TPriceModelInfos, OverrunKmModelInfos);
    R_FIELD(TPriceModelInfos, PackInsurancePriceModelInfos);
    R_FIELD(bool, UseMileageForSchedule, false);

public:
    static const TString DistanceThresholdPushName;
    static const TString DurationThresholdPushName;

private:
    static TFactory::TRegistrator<TPackOffer> Registrator;

protected:
    class TRecalcByMinutesInfo {
    private:
        R_FIELD(bool, NeedRecalcNow, false);
        R_FIELD(bool, NeedRecalcInFuture, false);
        R_FIELD(bool, NeedDistancePricing, false);
        R_FIELD(bool, IsSwitching, false);

    public:
        TRecalcByMinutesInfo(const bool needRecalcNow, const bool needRecalcInFuture, const bool needDistancePricing, const bool isSwitching)
            : NeedRecalcNow(needRecalcNow)
            , NeedRecalcInFuture(needRecalcInFuture)
            , NeedDistancePricing(needDistancePricing)
            , IsSwitching(isSwitching)
        {
        }
    };

public:
    virtual void PatchSessionReport(NJson::TJsonValue& result, NDriveSession::TReportTraits traits, ELocalization locale,
                                    const NDrive::IServer& server, const TOfferSessionPatchData& patchData) const override;

protected:
    virtual TString DoBuildCalculationDescription(ELocalization locale, const TFullCompiledRiding& fcr, const NDrive::IServer& server) const override;
    virtual TRecalcByMinutesInfo CheckRecalcByMinutes(const TRidingInfo& rInfo, const TInstant startPack) const;
    virtual TString DoFormDescriptionElement(const TString& value, ELocalization locale, const ILocalization* localization) const override;
    virtual void FillBillPack(TBill& bill, const TOfferPricing& pricing, TOfferStatePtr segmentState, ELocalization locale, const NDrive::IServer* server) const;

    bool DeserializePackFromProto(const NDrive::NProto::TPackOffer& info);
    void SerializePackToProto(NDrive::NProto::TPackOffer& info) const;

    virtual NJson::TJsonValue DoBuildJsonReport(const TReportOptions& options, const ICommonOfferBuilderAction* constructor, const NDrive::IServer& server) const override;

    virtual TString GetDefaultGroupName() const override;

    virtual bool GetIsParkingIncludeInPack() const {
        return IsParkingIncludeInPackFlag;
    }

    virtual bool NeedStandartMileagePricing() const override {
        return false;
    }

public:
    using TBase::TBase;

    TPackOffer() = default;
    TPackOffer(const TStandartOffer& standart)
        : TBase(IOffer::DeepCopy(standart))
    {
    }

    ui32 GetDiscountedOverrunPrice() const {
        return GetPublicDiscountedPrice(GetOverrunKm(), "");
    }
    ui32 GetDiscountedOvertimePrice() const {
        return GetPublicDiscountedPrice(GetOvertimeRiding(), "");
    }

    ui32 GetPublicDiscountedOverrunPrice(const double distance, const ui32 precision = 1) const {
        return GetPublicDiscountedPrice(distance * GetOverrunKm(), "", precision);
    }
    ui32 GetPublicOriginalOverrunPrice(const double distance, const ui32 precision = 1) const {
        return GetPublicOriginalPrice(distance * GetOverrunKm(), "", precision);
    }

    virtual ui32 GetPublicDiscountedPackPrice(TMaybe<ui32> overridenPackPrice = {}, const ui32 precision = 1) const {
        return GetPublicDiscountedPrice(overridenPackPrice.GetOrElse(PackPrice), "", precision);
    }
    virtual ui32 GetPublicOriginalPackPrice(TMaybe<ui32> overridenPackPrice = {}, const ui32 precision = 1) const {
        return GetPublicOriginalPrice(overridenPackPrice.GetOrElse(PackPrice), "", precision);
    }

    TDuration GetTotalDuration() const {
        return Duration + ExtraDuration;
    }

    virtual bool ShouldReportUpsale() const {
        return true;
    }

    void FillBill(TBill& bill, const TOfferPricing& pricing, TOfferStatePtr segmentState, ELocalization locale, const NDrive::IServer* server, ui32 cashbackPercent) const override;
    void FillBillPricing(TBill& bill, const TOfferPricing& pricing, TOfferStatePtr segmentState, ELocalization locale, const NDrive::IServer* server) const override;

    virtual ui32 GetDeposit() const override;
    virtual bool GetFreeAndPricedDurations(TDuration& freeDuration, TDuration& pricedDuration, const TMap<TString, TOfferSegment>& segments) const override;
    virtual TDuration GetFreeTime(const TDuration usedDuration, const TString& tagName) const override;
    virtual TMaybe<TInstant> GetStart() const;

    virtual TString GetTypeName() const override;
    static TString GetTypeNameStatic();

    virtual TOfferStatePtr DoCalculate(const TVector<IEventsSession<TCarTagHistoryEvent>::TTimeEvent>& timeline, const TVector<TAtomicSharedPtr<TCarTagHistoryEvent>>& events, const TInstant& until, const TRidingInfo& ridingInfo, TOfferPricing& result) const override;
    virtual TVector<TString> GetDefaultShortDescription(ELocalization locale, NDriveSession::TReportTraits traits, const ILocalization& localization) const override;
    ui32 CalcPackPrice(const NDrive::IServer* server) const;

    // ApplyPackPriceModel applies model for pack price and appends record to model info.
    void ApplyPackPriceModel(const NDrive::IOfferModel& model);
    // ApplyDurationModel applies model for duration and appends record to model info.
    void ApplyDurationModel(const NDrive::IOfferModel& model);
    // ApplyMileageLimitModel applies model for mileage limit and appends record to model info.
    void ApplyMileageLimitModel(const NDrive::IOfferModel& model);
    // ApplyOverrunKmModel applies model for overrun price and appends record to model info.
    void ApplyOverrunKmModel(const NDrive::IOfferModel& model);
    // ApplyPackInsurancePriceModel applies model for insurance pack price and appends record to model info.
    //
    // This method also modifies pack price.
    void ApplyPackInsurancePriceModel(const NDrive::IOfferModel& model);

    bool DeserializeFromProto(const NDrive::NProto::TOffer& info) override;
    NDrive::NProto::TOffer SerializeToProto() const override;
};

class TPackOfferReport: public TStandartOfferReport {
private:
    using TBase = TStandartOfferReport;

protected:
    virtual void DoRecalcPrices(const NDrive::IServer* /*server*/) override;

public:
    using TBaseOffer = TPackOffer;

    using TBase::TBase;

    TPackOfferReport(const TStandartOfferReport& report)
        : TBase(report)
    {
    }

    virtual void ApplyInternalCorrection(const TString& /*areaId*/, const TOffersBuildingContext& /*context*/, const NDrive::IServer* /*server*/) override {
    }
    virtual void ApplyFlowCorrection(const TString& /*areaId*/, const TOffersBuildingContext& /*context*/, const NDrive::IServer* /*server*/) override{};

    virtual TString PredictDestination(const TOffersBuildingContext& /*context*/, const NDrive::IServer* /*server*/, const TCommonDestinationDetector& /*baseDetector*/) const override {
        return "";
    }

    // RecalculateFeatures recalculates features for pack offer.
    virtual void RecalculateFeatures() override;
};

class TBasePackOfferConstructor
    : public TStandartOfferConstructor
{
private:
    using TBase = TStandartOfferConstructor;

private:
    R_FIELD(ui32, RerunPriceKM, 800);
    R_FIELD(TString, SyncStandartOffer);
    R_FIELD(TDuration, ReturningDuration, TDuration::Zero());
    R_FIELD(TDuration, PackPriceQuantumPeriod);
    R_FIELD(ui32, PackPriceQuantum, 0);
    R_FIELD(ui32, DistancePushThreshold, 0);
    R_FIELD(TDuration, DurationPushThreshold, TDuration::Zero());
    R_FIELD(TDuration, RemainingDurationPushThreshold, TDuration::Zero());
    R_FIELD(double, RemainingDistancePushThreshold, 0);
    R_FIELD(bool, IsParkingIncludeInPackFlag, true);
    R_FIELD(bool, OvertimeParkingIsRiding, false);

protected:
    virtual EOfferCorrectorResult DoBuildOffers(const TUserPermissions& permissions, TVector<IOfferReport::TPtr>& offers, const TOffersBuildingContext& context, const NDrive::IServer* server, NDrive::TInfoEntitySession& session) const override final;
    virtual bool ConstructPackOffer(const TStandartOfferReport& stOfferReport, const TOffersBuildingContext& context, const NDrive::IServer* server, TVector<IOfferReport::TPtr>& offers, NDrive::TInfoEntitySession& session) const = 0;
    virtual bool GetPrices(const NDrive::IServer* server, ui32& ridingPrice, ui32& parkingPrice, ui32& kmPrice) const override;

public:
    using TBase::TBase;

    virtual bool DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonValue) override;
    virtual NJson::TJsonValue SerializeSpecialsToJson() const override;

    virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override;
};

class TPackOfferConstructor: public TBasePackOfferConstructor {
private:
    using TBase = TBasePackOfferConstructor;

private:
    TString PackPriceCalculatorType = "constant";
    IPriceCalculatorConfig::TPtr PackPriceCalculatorConfig = new TConstantPriceConfig(1000000);
    IPriceCalculator::TPtr PackPriceCalculator = new TConstantPrice(1000000);

private:
    R_OPTIONAL(TString, ZeroMileageDetailedDescription);
    R_FIELD(TInstant, FinishInstant, TInstant::Zero());
    R_FIELD(TDuration, Duration, TDuration::Days(1));
    R_FIELD(ui32, MileageLimit, 70);

private:
    static TFactory::TRegistrator<TPackOfferConstructor> Registrator;

protected:
    virtual bool ConstructPackOffer(const TStandartOfferReport& stOfferReport, const TOffersBuildingContext& /*context*/, const NDrive::IServer* /*server*/, TVector<IOfferReport::TPtr>& offers, NDrive::TInfoEntitySession& session) const override;
    virtual bool DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonValue) override;
    virtual NJson::TJsonValue SerializeSpecialsToJson() const override;

    virtual THolder<TPackOfferReport> Extend(const TStandartOfferReport& standardOfferReport) const;

public:
    using TBase::TBase;

    static TString GetTypeStatic() {
        return "pack_offer_builder";
    }

    static TString GetTypeName() {
        return GetTypeStatic();
    }

    virtual TString GetType() const override {
        return GetTypeStatic();
    }

    virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override;

    TPackOfferConstructor& SetPackPrice(const ui32 price);
};

void CalcPackOfferFeatures(TPackOffer& offer);
