#include "rental_offer.h"

#include <drive/backend/car_attachments/registry/registry.h>
#include <drive/backend/cars/car_model.h>
#include <drive/backend/cars/hardware.h>
#include <drive/backend/common/localization.h>
#include <drive/backend/compiled_riding/compiled_riding.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/feedback.h>
#include <drive/backend/data/leasing/company.h>
#include <drive/backend/data/rental/timetable_builder.h>
#include <drive/backend/database/drive/private_data.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/images/database.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/sessions/manager/billing.h>

#include <drive/library/cpp/geocoder/api/client.h>
#include <drive/library/cpp/mds/client.h>
#include <drive/library/cpp/raw_text/datetime.h>

#include <library/cpp/html/escape/escape.h>
#include <library/cpp/http/misc/httpcodes.h>

#include <rtline/library/json/adapters.h>
#include <rtline/util/algorithm/datetime.h>
#include <rtline/util/types/uuid.h>

#include <util/datetime/base.h>

#include <regex>

DECLARE_FIELDS_JSON_SERIALIZER(TRentalOptionValues::TConstants);
DECLARE_FIELDS_JSON_SERIALIZER(TRentalOptionValues::TStringConstants);

const TString TRentalOffer::DefaultOfferTimezone = "Europe/Moscow";
const TString TRentalOffer::DefaultDateFormat = "%d.%m.%Y";
const TString TRentalOffer::DefaultTimeFormat = "%H:%M";
const TString TRentalOffer::ServiceModeStatus = "service";

const TString TRentalOffer::AcceptanceFinishedNotificationType = "acceptance_finished";
const TString TRentalOffer::RideFinishedNotificationType = "ride_finished";
const TString TRentalOffer::BookingConfirmedNotificationType = "booking_confirmed";
const TString TRentalOffer::BookingCancelledNotificationType = "booking_cancelled";

const TString TRentalOffer::SinceDateKeyword = "delivery_date";
const TString TRentalOffer::SinceTimeKeyword = "delivery_time";
const TString TRentalOffer::UntilDateKeyword = "return_date";
const TString TRentalOffer::UntilTimeKeyword = "return_time";
const TString TRentalOffer::CompanynameKeyword = "company_name";
const TString TRentalOffer::PickupLocationKeyword = "delivery_location";
const TString TRentalOffer::CarModelKeyword = "car_model";

namespace {
struct EmailData {
    TMaybe<TInstant> ActualSince;
    TMaybe<TInstant> ActualUntil;
    TMaybe<double> Mileage;
    TMaybe<double> MileageStart;
    TMaybe<double> MileageFinish;
    TMaybe<double> FuelLevelStart;
    TMaybe<double> FuelLevelFinish;
    TString LocationStart;
    TString LocationFinish;
    TMap<TInstant, TString> ImagesBeforeRide;
    TMap<TInstant, TString> ImagesAfterRide;
    TMaybe<TDuration> Duration;
};

const TString actualSinceDateKeyword{ "actual_since_date" };
const TString actualSinceTimeKeyword{ "actual_since_time" };
const TString actualUntilDateKeyword{ "actual_until_date" };
const TString actualUntilTimeKeyword{ "actual_until_time" };
const TString mileageKeyword{ "mileage" };
const TString startMileageKeyword{ "start_mileage" };
const TString finishMileageKeyword{ "finish_mileage" };
const TString startFuelLevelKeyword{ "start_fuel_level" };
const TString finishFuelLevelKeyword{ "finish_fuel_level" };
const TString locationStartKeyword{ "location_start" };
const TString locationFinishKeyword{ "location_finish" };
const TString durationKeyword{ "duration" };
const TString imagesBeforeRideKeyword{ "images_before_ride_keyword" };
const TString imagesAfterRideKeyword{ "images_after_ride_keyword" };
const TString companyPhoneKeyword{ "company_phone" };
const TString companyEmailKeyword{ "company_mail" };
const TString totalPaymentKeyword{ "total_payment" };
const TString totalDailyCostKeyword{ "total_daily_cost" };
const TString totalDailyCostTitle{ TStringBuilder() << "rental." << totalDailyCostKeyword << ".title" };
const TString totalDailyCostSubtitle{ TStringBuilder() << "rental." << totalDailyCostKeyword << ".subtitle" };
const TString depositKeyword{ "deposit" };
const TString firstNametKeyword{ "first_name" };
const TString lastNameKeyword{ "last_name" };
const TString clientPhoneKeyword{ "client_phone" };
const TString insuranceTypeKeyword{ "insurance_type" };
const TString insuranceCostKeyword{ "insurance_cost" };
const TString currencyKeyword{ "currency" };
const TString optionsKeyword{ "options" };
const TString overrunCostPerKmKeyword{ "overrun_cost_per_km" };
const TString limitKmPerDayKeyword{ "limit_km_per_day" };

const TDuration maximumSinceShift = TDuration::Days(365);
const TString revisionIsInvalid = "rental.book.revision_is_invalid";
const TString wrongSinceUntil = "rental.book.wrong_since_until";
const TString unitKmPerDay = "unit.shorts.km_per_day";
const TString billMileageTitle = "rental.bill.mileage.title";
const TString billDurationGreaterDay = "rental.bill.duration_format.greater_than_day";
const TString billDurationLessDay =  "rental.bill.duration_format.less_than_day";
const TString billDurationTitle =  "rental.bill.duration.title";

const TString patchOfferSourceSession = "RentalOffer::PatchOffer";
const TString patchFromJsonSourceSession = "RentalOffer::PatchFromJson";
const TString getEmailDataSourceSession = "RentalOffer::GetEmailData";
constexpr ui64 secondsInDay = 86400;
const TVector<TString> limitColumnIds = { "per_day" };
const TString optionKeyName = "{{rental.option.name}}";
const TString optionKeyCost = "{{rental.option.cost}}";
const TString optionKeyCurrency = "{{rental.option.currency}}";
const TString currencyKey = "{{rental.currency}}";

namespace NLegacyOptions {
    namespace NId {
        const TString childSeat = "child_seat";
        const TString roofRack = "roof_rack";
        const TString gps = "gps";
        const TString snowChains = "snow_chains";
        const TString entryToEcoZonesInGermany = "entry_to_eco_zones_in_germany";
    }
    namespace NOrder {
        const int childSeat = 0;
        const int roofRack = 1;
        const int gps = 2;
        const int snowChains = 3;
        const int entryToEcoZonesInGermany = 4;
    }
}

bool ApplyCommonGridMigration(const NJson::TJsonValue& value, auto& grid, auto columnIdGetter, const TString& columnDataKey, const TString& dataKey, auto dataGetter) {
    auto jColumns = value.GetArray();
    for (const auto& jColumn: jColumns) {
        auto columnId = columnIdGetter(jColumn);
        auto& columnCells = jColumn[columnDataKey].GetArray();
        if (auto columnIt = grid.find(columnId); columnIt != grid.end()) {
            for (const auto& cell: columnCells) {
                auto& tariffName = cell["name"].GetString();
                if (auto cellIt = columnIt->second.find(tariffName); cellIt != columnIt->second.end()) {
                    if (!cell.Has(dataKey) || cell[dataKey].IsNull()) {
                        dataGetter(cellIt->second) = Nothing();
                    } else {
                        dataGetter(cellIt->second) = cell[dataKey].GetUInteger();
                    }
                } else {
                    return false;
                }
            }
        } else {
            return false;
        }
    }
    return true;
}

void MoveCellData(const TString& oldRowName, const TString& newRowName, auto& grid) {
    for (auto& [id, columnData]: grid) {
        auto oldCellIt = columnData.find(oldRowName);
        if (oldCellIt != columnData.end()) {
            columnData[newRowName] = std::move(oldCellIt->second);
            columnData.erase(oldCellIt);
        }
    }
}

void RemoveCell(const TString& oldRowName, auto& grid) {
    for (auto& [id, columnData]: grid) {
        columnData.erase(oldRowName);
    }
}

void FillOptionalParameter(auto key, auto value, auto& agreementTemplate) {
    if (value) {
        SubstGlobal(agreementTemplate, key, NHtml::EscapeAttributeValue(ToString(value)));
    } else {
        SubstGlobal(agreementTemplate, key, "");
    }
}

TString FillOption(const std::string& optionTemplate, TMaybe<bool> optionValue, TString&& optionName, TMaybe<ui64> cost, const TMaybe<TString>& currencyUnit) {
    if (optionValue && *optionValue) {
        std::string filledOption = optionTemplate;
        SubstGlobal(filledOption, optionKeyName, NHtml::EscapeAttributeValue(optionName));
        TString costStr{""};
        if (cost) {
            costStr = ToString(*cost);
            SubstGlobal(filledOption, optionKeyCost, costStr);
            FillOptionalParameter(optionKeyCurrency, currencyUnit, filledOption);
        } else {
            SubstGlobal(filledOption, optionKeyCost, "");
            SubstGlobal(filledOption, optionKeyCurrency, "");
        }

        return filledOption;
    }
    return TString{};
}

void FillTitles(TString&& titleId,
                TString&& subtitleId,
                NJson::TJsonValue& instance,
                const ILocalization* localization,
                ELocalization locale) {
    instance["title"] = localization ? localization->GetLocalString(locale, titleId) : titleId;
    instance["subtitle"] = localization ? localization->GetLocalString(locale, subtitleId) : subtitleId;
}

NJson::TJsonValue MakeObjectVisible(NJson::TJsonValue&& jProperty) {
    jProperty["visible"] = true;
    return jProperty;
}
}

NDrive::TScheme TRentalOptionValues::TConstants::GetScheme(const NDrive::IServer* /*server*/) const {
    NDrive::TScheme scheme;
    scheme.Add<TFSBoolean>("default_value", "Включать ли опцию по умолчанию").SetDefault(false).SetRequired(true);
    scheme.Add<TFSString>("id", "Идентификатор опции").SetRequired(true);
    scheme.Add<TFSVariants>(TString(CalculatePolicyNameId), "Тип калькуляции")
        .InitVariants<NDedicatedFleet::ECostCalculatePolicy>()
        .SetDefault(::ToString(NDedicatedFleet::ECostCalculatePolicy::None))
        .SetRequired(true);
    scheme.Add<TFSNumeric>("order", "Порядок опции в выдаче").SetDefault(0).SetRequired(true);
    return scheme;
}

NDrive::TScheme TRentalOptionValues::TStringConstants::GetScheme(const NDrive::IServer* /*server*/) const {
    NDrive::TScheme scheme;
    scheme.Add<TFSString>("title", "Заголовок опции").SetRequired(true);
    scheme.Add<TFSString>("subtitle", "Подзаголовок опции");
    return scheme;
}

template<>
TVector<NJson::TJsonValue> TRentalOptionValuesAdapter::GetReport(ELocalization locale, const NDrive::IServer& server) const {
    TOfferVariable<bool> result;
    if (Constants.Defined()) {
        const auto& constants = Constants.GetRef();
        result.SetId(constants.GetId());
        result.SetDefaultValue(constants.GetDefaultValue());
    }

    if (StringConstants.Defined()) {
        const auto& strConstants = StringConstants.GetRef();
        result.SetTitle(strConstants.GetTitle());
        result.SetSubtitle(strConstants.GetSubtitle());
    }

    if (HasValue()) {
        result.SetValue(GetValueRef());
    }

    if (HasCost()) {
        result.SetCost(GetCostRef());
    }

    if (HasCost() && HasUnit()) {
        result.SetUnit(GetUnitRef());
    }

    return {std::move(result.GetReport(locale, server))};
}

template<> template<>
bool TRentalOptionValuesAdapter::DeserializeFromProto(const NDrive::NProto::TRentalOption& option) {
    if (option.HasConstants()) {
        auto& res = Constants.ConstructInPlace();
        const auto& proto = option.GetConstants();
        if (proto.HasDefaultValue()) {
            res.SetDefaultValue(proto.GetDefaultValue());
        }

        if (proto.HasId()) {
            res.SetId(proto.GetId());
        }

        if (proto.HasCalculatePolicy()) {
            res.SetCalculatePolicy(::FromString(proto.GetCalculatePolicy()));
        }

        if (proto.HasOrder()) {
            res.SetOrder(proto.GetOrder());
        }
    }

    if (option.HasStringConstants()) {
        auto& res = StringConstants.ConstructInPlace();
        const auto& proto = option.GetStringConstants();

        if (proto.HasTitle()) {
            res.SetTitle(proto.GetTitle());
        }

        if (proto.HasSubtitle()) {
            res.SetSubtitle(proto.GetSubtitle());
        }
    }

    if (option.HasValue()) {
        SetValue(option.GetValue());
    }

    if (option.HasCost()) {
        SetCost(option.GetCost());
    }

    return true;
}

template<> template<>
bool TRentalOptionValuesAdapter::SerializeToProto(NDrive::NProto::TRentalOption& option) const {
    if (Constants.Defined()) {
        auto& proto = *option.MutableConstants();
        proto.SetDefaultValue(Constants->GetDefaultValue());
        proto.SetId(Constants->GetId());
        proto.SetCalculatePolicy(::ToString(Constants->GetCalculatePolicy()));
        proto.SetOrder(Constants->GetOrder());
    }

    if (StringConstants.Defined()) {
        auto& proto = *option.MutableStringConstants();
        proto.SetTitle(StringConstants.GetRef().GetTitle());
        proto.SetSubtitle(StringConstants.GetRef().GetSubtitle());
    }

    if (HasValue()) {
        option.SetValue(GetValueRef());
    }

    if (HasCost()) {
        option.SetCost(GetCostRef());
    }

    return true;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TRentalOptionValuesAdapter& result) {
    return result.DeserializeDataFromJson(value);
}

template <>
NJson::TJsonValue NJson::ToJson(const TRentalOptionValuesAdapter& object) {
    return object.SerializeDataToJson();
}

template<class T, class P>
void PatchOfferFromJson(T& variables, P offer, auto fieldsChecker, const TRentalOfferBuilder& builder) {
    // legacy options
    // will be deleted after frontend update
    if (fieldsChecker("offer_options")) {
        const auto options = variables["offer_options"];

        auto childSeat = NJson::FromJson<TMaybe<bool>>(options[NLegacyOptions::NId::childSeat]);
        if (childSeat) {
            offer->AddOptionFromLegacy(*childSeat, NLegacyOptions::NId::childSeat, NLegacyOptions::NOrder::childSeat);
        }

        auto roofRack = NJson::FromJson<TMaybe<bool>>(options[NLegacyOptions::NId::roofRack]);
        if (roofRack) {
            offer->AddOptionFromLegacy(*roofRack, NLegacyOptions::NId::roofRack, NLegacyOptions::NOrder::roofRack);
        }

        auto gps = NJson::FromJson<TMaybe<bool>>(options[NLegacyOptions::NId::gps]);
        if (gps) {
            offer->AddOptionFromLegacy(*gps, NLegacyOptions::NId::gps, NLegacyOptions::NOrder::gps);
        }

        auto snowChains = NJson::FromJson<TMaybe<bool>>(options[NLegacyOptions::NId::snowChains]);
        if (snowChains) {
            offer->AddOptionFromLegacy(*snowChains, NLegacyOptions::NId::snowChains, NLegacyOptions::NOrder::snowChains);
        }

        auto entryToEcoZonesInGermany = NJson::FromJson<TMaybe<bool>>(options[NLegacyOptions::NId::entryToEcoZonesInGermany]);
        if (entryToEcoZonesInGermany) {
            offer->AddOptionFromLegacy(*entryToEcoZonesInGermany, NLegacyOptions::NId::entryToEcoZonesInGermany, NLegacyOptions::NOrder::entryToEcoZonesInGermany);
        }

        std::sort(offer->MutableOptions().begin(), offer->MutableOptions().end(), [](const TRentalOffer::TOption& lhs, const TRentalOffer::TOption& rhs) {
            return !lhs.HasConstants() || !rhs.HasConstants() || lhs.GetConstantsRef().GetOrder() < rhs.GetConstantsRef().GetOrder();
        });
    }

    // legacy insurance
    // will be deleted after frontend update
    if (auto insuranceType = NJson::FromJson<TMaybe<TString>>(variables["insurance_type"]); insuranceType) {
        auto& allowedInsurances = builder.GetAllowedInsurances();
        if (auto insuranceIt = std::find_if(allowedInsurances.begin(), allowedInsurances.end(),
            [&insuranceType](const TRentalInsurance& insurance) {
                return insurance.GetId() == *insuranceType;
                }); insuranceIt != allowedInsurances.end()) {
            offer->SetInsurance(*insuranceIt);
        }
    }

    if (fieldsChecker("insurance")) {
        const auto& jInsurance = variables["insurance"];
        if (jInsurance.Has("id")) {
            auto insuranceId = jInsurance["id"].GetString();
            auto& allowedInsurances = builder.GetAllowedInsurances();
            if (auto insuranceIt = std::find_if(allowedInsurances.begin(), allowedInsurances.end(),
                [&insuranceId](const TRentalInsurance& insurance) {
                    return insurance.GetId() == insuranceId;
                    }); insuranceIt != allowedInsurances.end()) {
                offer->SetInsurance(*insuranceIt);
                if (jInsurance.Has("cost") && !jInsurance["cost"].IsNull()) {
                    offer->OptionalInsurance()->SetCost(jInsurance["cost"].GetUInteger());
                }
            }
        }
    }

    auto deliveryLocation = NJson::FromJson<TMaybe<TGeoCoord>>(variables["delivery_location"]);
    if (deliveryLocation) {
        offer->SetDeliveryLocation(deliveryLocation);
    }

    auto deliveryLocationName = NJson::FromJson<TMaybe<TString>>(variables["delivery_location_name"]);
    if (deliveryLocationName) {
        offer->SetDeliveryLocationName(*deliveryLocationName);
    }

    auto returnLocation = NJson::FromJson<TMaybe<TGeoCoord>>(variables["return_location"]);
    auto returnLocationName = NJson::FromJson<TMaybe<TString>>(variables["return_location_name"]);
    if (returnLocation || returnLocationName) {
        if (!offer->HasReturnLocations()) {
            offer->OptionalReturnLocations().ConstructInPlace();
        }
        TRentalOfferLocation* rentalOfferLocation;
        if (offer->OptionalReturnLocations()->empty()) {
            rentalOfferLocation = &offer->MutableReturnLocations()->emplace_back();
        } else {
            rentalOfferLocation = offer->MutableReturnLocations()->begin();
        }
        if (returnLocation) {
            if (rentalOfferLocation->MutableCoords().empty()) {
                rentalOfferLocation->MutableCoords().push_back(*returnLocation);
            } else {
                rentalOfferLocation->MutableCoords()[0] = *returnLocation;
            }
        }
        if (returnLocationName) {
            rentalOfferLocation->MutableLocationName() = *returnLocationName;
        }
    } else {
        auto returnLocations = NJson::FromJson<TMaybe<TRentalOfferLocations>>(variables["return_locations"]);
        if (returnLocations) {
            offer->SetReturnLocations(returnLocations);
        }
    }

    auto comment = NJson::FromJson<TMaybe<TString>>(variables["comment"]);
    if (comment) {
        offer->SetComment(comment);
    }

    auto totalPayment = NJson::FromJson<TMaybe<ui64>>(variables["total_payment"]);
    if (totalPayment) {
        offer->SetTotalPayment(totalPayment);
    }

    auto deposit = NJson::FromJson<TMaybe<ui64>>(variables["deposit"]);
    if (deposit) {
        offer->SetDeposit(deposit);
    }

    auto status = NJson::FromJson<TMaybe<TString>>(variables["status"]);
    if (status) {
        offer->SetStatus(status);
    }

    auto limitKmPerDay = NJson::FromJson<TMaybe<ui64>>(variables["limit_km_per_day"]);
    if (limitKmPerDay) {
        offer->SetLimitKmPerDay(limitKmPerDay);
    }

    auto overrunCostPerKm = NJson::FromJson<TMaybe<ui64>>(variables["overrun_cost_per_km"]);
    if (overrunCostPerKm) {
        offer->SetOverrunCostPerKm(overrunCostPerKm);
    }

    auto currency = NJson::FromJson<TMaybe<TString>>(variables["currency"]);
    if (currency) {
        offer->SetCurrency(currency);
    }

    auto totalDailyCost = NJson::FromJson<TMaybe<ui64>>(variables[totalDailyCostKeyword]);
    if (totalDailyCost) {
        offer->SetTotalDailyCost(totalDailyCost);
    }

    if (fieldsChecker("options")) {
        auto& optionsArr = variables["options"].GetArray();
        auto& offerOptions = offer->MutableOptions();
        for (auto& jOption: optionsArr) {
            auto& id = jOption["id"].GetString();
            auto optionIt = std::find_if(offerOptions.begin(), offerOptions.end(), [&id](const TRentalOffer::TOption& option) {
                if (option.HasConstants()) {
                    return option.GetConstantsRef().GetId() == id;
                } else {
                    return false;
                }
            });
            if (optionIt != offerOptions.end()) {
                if (jOption.Has("value")) {
                    if (!jOption["value"].IsNull()) {
                        optionIt->SetValue(jOption["value"].GetBoolean());
                    } else {
                        optionIt->OptionalValue().Clear();
                    }
                }
                if (jOption.Has("cost")) {
                    if (!jOption["cost"].IsNull()) {
                        optionIt->SetCost(jOption["cost"].GetInteger());
                    } else {
                        optionIt->OptionalCost().Clear();
                    }
                }
            }
        }
    }
}

EOfferCorrectorResult CheckAndSetSinceUntil(TRentalOffer& offer, auto since, auto until, auto now) {
    if (!since || since < now + maximumSinceShift) {
        if (since) {
            offer.SetSince(*since);
        }
    } else {
        return EOfferCorrectorResult::Unimplemented;
    }

    if (until) {
        offer.SetUntil(*until);
    }

    if (since && until && since >= until) {
        return EOfferCorrectorResult::Unimplemented;
    } else if ((since && !until) || (!since && until)) {
        return EOfferCorrectorResult::Unimplemented;
    }
    return EOfferCorrectorResult::Success;
}

template <>
NJson::TJsonValue NJson::ToJson(const TRentalInsurance& object) {
    NJson::TJsonValue result;
    result["id"] = object.GetId();
    result["title"] = object.GetTitle();
    return result;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TRentalInsurance& result) {
    NJson::TryFromJson(value["title"], result.MutableTitle());
    return NJson::TryFromJson(value["id"], result.MutableId());
}

template <>
NJson::TJsonValue NJson::ToJson(const TRentalOfferStatus& object) {
    NJson::TJsonValue result;
    result["id"] = object.GetId();
    return result;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TRentalOfferStatus& result) {
    return NJson::TryFromJson(value["id"], result.MutableId());
}

template <>
NJson::TJsonValue NJson::ToJson(const TRentalOfferLocation& object) {
    NJson::TJsonValue result;
    result["id"] = object.GetId();
    result["location_name"] = object.GetLocationName();
    result["coords"] = TGeoCoord::SerializeVector(object.GetCoords());
    if (object.GetType() == ::ELocationType::Coord) {
        result["lon"] = object.GetCoords().begin()->X;
        result["lat"] = object.GetCoords().begin()->Y;
    }
    return result;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TRentalOfferLocation& result) {
    bool valid =
        NJson::TryFromJson(value["id"], result.MutableId()) &&
        NJson::TryFromJson(value["location_name"], result.MutableLocationName());
    if (value.Has("coords")) {
        valid &=
            TGeoCoord::DeserializeVector(value["coords"].GetString(), result.MutableCoords()) &&
            result.GetCoords().size() != 2;
    }
    if (value.Has("lat") && value.Has("lon")) {
        if (result.GetType() == ELocationType::Coord) {
            return valid &= result.MutableCoords().front().DeserializeLatLonFromJson(value);
        } else {
            return valid &= result.MutableCoords().emplace_back().DeserializeLatLonFromJson(value);
        }
    }
    return valid && !result.GetCoords().empty();
}

template <>
NJson::TJsonValue NJson::ToJson(const TRentalOfferCurrency& object) {
    NJson::TJsonValue result;
    NJson::FieldsToJson(result, object.GetFields());
    return result;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TRentalOfferCurrency& result) {
    return TryFieldsFromJson(value, result.GetFields());
}

template <>
NJson::TJsonValue NJson::ToJson(const ERentalOfferImageStyle& object) {
    return NJson::ToJson(NJson::Stringify(object));
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, ERentalOfferImageStyle& result) {
    return NJson::TryFromJson(value, NJson::Stringify(result));
}

NDrive::NProto::TRentalOfferLocation TRentalOfferLocation::SerializeToProto() const {
    NDrive::NProto::TRentalOfferLocation result;
    result.SetId(Id);
    result.SetLocationName(LocationName);
    for (const auto& coord : GetCoords()) {
        result.MutableCoords()->MutableCoords()->Add(coord.Serialize());
    }
    return result;
}

bool TRentalOfferLocation::DeserializeFromProto(const NDrive::NProto::TRentalOfferLocation& proto) {
    SetLocationName(proto.GetLocationName());
    SetId(proto.GetId());
    MutableCoords().clear();
    for (auto&& coord : proto.GetCoords().coords()) {
        TGeoCoord tempCoord;
        if (tempCoord.Deserialize(coord)) {
            MutableCoords().push_back(tempCoord);
        } else {
            return false;
        }
    }
    return GetCoords().size() == 1 || GetCoords().size() > 2;
}

NJson::TJsonValue TRentalEconomy::TTariffRecommendation::ReportToJson(TInstant since, TInstant until, ui64& accumulatedCost) const {
    NJson::TJsonValue jTariffRecommendation = NJson::TJsonMap();
    NJson::InsertField(jTariffRecommendation, "name", Name);
    NJson::InsertField(jTariffRecommendation, "id", Id);
    NJson::InsertField(jTariffRecommendation, "daily_cost", DailyCost);
    if (DailyCost && until >= since) {
        auto duration = until - since;
        const auto durationDays = duration.Seconds() > duration.Days() * secondsInDay ? duration.Days() + 1 : duration.Days();
        const auto cost = *DailyCost * durationDays;
        jTariffRecommendation["total_daily_cost"] = cost;
        accumulatedCost += cost;
    }
    NJson::InsertField(jTariffRecommendation, "deposit", Deposit);
    NJson::InsertField(jTariffRecommendation, "overrun_cost_per_km", OverrunCostPerKm);
    return jTariffRecommendation;
}

NDrive::NProto::TRentalOfferTariffRecomendation TRentalEconomy::TTariffRecommendation::SerializeToProto() const {
    NDrive::NProto::TRentalOfferTariffRecomendation protoObj;
    if (!Id.Empty()) {
        protoObj.SetId(*Id);
    }
    if (!Name.Empty()) {
        protoObj.SetName(*Name);
    }
    if (!DailyCost.Empty()) {
        protoObj.SetDailyCost(*DailyCost);
    }
    if (!Deposit.Empty()) {
        protoObj.SetDeposit(*Deposit);
    }
    if (!OverrunCostPerKm.Empty()) {
        protoObj.SetOverrunCostPerKm(*OverrunCostPerKm);
    }
    return protoObj;
}

bool TRentalEconomy::TTariffRecommendation::DeserializeFromProto(const NDrive::NProto::TRentalOfferTariffRecomendation& meta) {
    if (meta.HasId()) {
        Id = meta.GetId();
    }
    if (meta.HasName()) {
        Name = meta.GetName();
    }
    if (meta.HasDailyCost()) {
        DailyCost = meta.GetDailyCost();
    }
    if (meta.HasDeposit()) {
        Deposit = meta.GetDeposit();
    }
    if (meta.HasOverrunCostPerKm()) {
        OverrunCostPerKm = meta.GetOverrunCostPerKm();
    }
    return true;
}

NJson::TJsonValue TRentalEconomy::TOptionsRecommendation::ReportToJson(ui64& accumulatedCost) const {
    NJson::TJsonValue jOptionsRecommendation = NJson::TJsonArray();
    for (const auto& [id, cost]: TotalOptionCosts) {
        NJson::TJsonValue jOption;
        jOption["id"] = id;
        jOption["cost"] = cost;
        accumulatedCost += cost;
        jOptionsRecommendation.AppendValue(std::move(jOption));
    }
    return jOptionsRecommendation;
}

NDrive::NProto::TRentalOptionRecomendation TRentalEconomy::TOptionsRecommendation::TOption::SerializeToProto() const {
    NDrive::NProto::TRentalOptionRecomendation meta;
    meta.SetId(Id);
    meta.SetTotalCost(TotalCost);
    return meta;
}

bool TRentalEconomy::TOptionsRecommendation::TOption::DeserializeFromProto(
    const NDrive::NProto::TRentalOptionRecomendation& meta
) {
    if (meta.HasId() && meta.HasTotalCost()) {
        Id = meta.GetId();
        TotalCost = meta.GetTotalCost();
        return true;
    }
    return false;
}

NDrive::NProto::TInsuranceRecommendation TRentalEconomy::TInsurancesRecommendation::TInsuranceRecommendation::SerializeToProto() const {
    NDrive::NProto::TInsuranceRecommendation meta;
    meta.SetId(Id);
    meta.SetCost(Cost);
    return meta;
}

bool TRentalEconomy::TInsurancesRecommendation::TInsuranceRecommendation::DeserializeFromProto(
    const NDrive::NProto::TInsuranceRecommendation& meta
) {
    if (meta.HasId() && meta.HasCost()) {
        Id = meta.GetId();
        Cost = meta.GetCost();
        return true;
    }
    return false;
}

NJson::TJsonValue TRentalEconomy::TInsurancesRecommendation::ReportToJson() const {
    NJson::TJsonValue jInsurancesRecommendation = NJson::TJsonArray();
    for (const auto& [id, cost]: Insurances) {
        NJson::TJsonValue jRec;
        jRec["id"] = id;
        jRec["cost"] = cost;
        jInsurancesRecommendation.AppendValue(std::move(jRec));
    }
    return jInsurancesRecommendation;
}

NDrive::NProto::TRentalLimitsRecommendation TRentalEconomy::TLimitsRecommendation::SerializeToProto() const {
    NDrive::NProto::TRentalLimitsRecommendation meta;
    meta.SetMileage(Mileage);
    return meta;
}

bool TRentalEconomy::TLimitsRecommendation::DeserializeFromProto(
    const NDrive::NProto::TRentalLimitsRecommendation& meta
) {
    if (meta.HasMileage()) {
        Mileage = meta.GetMileage();
        return true;
    }
    return false;
}

NJson::TJsonValue TRentalEconomy::TLimitsRecommendation::ReportToJson() const {
    NJson::TJsonValue jLimitsRecommendation = NJson::TJsonMap();
    jLimitsRecommendation["mileage"] = Mileage;
    return jLimitsRecommendation;
}

void TRentalOffer::AddOptionFromLegacy(bool value, const TString& id, int order) {
    auto optionIt = std::find_if(Options.begin(), Options.end(), [&id](const TRentalOffer::TOption& option) {
        if (option.HasConstants()) {
            return option.GetConstantsRef().GetId() == id;
        } else {
            return false;
        }
    });
    if (optionIt == Options.end()) {
        TOption option;
        option.SetValue(value);
        option.MutableConstants().ConstructInPlace();
        option.MutableConstants()->SetId(id);
        option.MutableConstants()->SetOrder(order);
        option.MutableConstants()->SetCalculatePolicy(NDedicatedFleet::ECostCalculatePolicy::None);
        option.MutableConstants()->SetDefaultValue(false);

        option.MutableStringConstants().ConstructInPlace();
        option.MutableStringConstants()->SetTitle(TStringBuilder() << "rental.option." << id << ".title");
        option.MutableStringConstants()->SetSubtitle(TStringBuilder() << "rental.option." << id << ".subtitle");
        Options.push_back(std::move(option));
    } else {
        optionIt->SetValue(value);
    }
}

void TRentalOffer::CalculateMileageLimit() {
    if (HasLimitKmPerDay()) {
        const auto duration = Until - Since;
        const auto durationDays = duration.Seconds() > duration.Days() * secondsInDay ? duration.Days() + 1 : duration.Days();
        SetMileageLimit(*LimitKmPerDay * durationDays);
    }
}

bool TRentalOffer::DeserializeFromProto(const NDrive::NProto::TOffer& info) {
    if (!TBase::DeserializeFromProto(info)) {
        return false;
    }
    const auto& meta = info.GetRentalOffer();

    if (meta.HasChildSeat()) {
        AddOptionFromLegacy(meta.GetChildSeat(), NLegacyOptions::NId::childSeat, NLegacyOptions::NOrder::childSeat);
    }
    if (meta.HasRoofRack()) {
        AddOptionFromLegacy(meta.GetRoofRack(), NLegacyOptions::NId::roofRack, NLegacyOptions::NOrder::roofRack);
    }
    if (meta.HasGPS()) {
        AddOptionFromLegacy(meta.GetGPS(), NLegacyOptions::NId::gps, NLegacyOptions::NOrder::gps);
    }
    if (meta.HasSnowChains()) {
        AddOptionFromLegacy(meta.GetSnowChains(), NLegacyOptions::NId::snowChains, NLegacyOptions::NOrder::snowChains);
    }
    if (meta.HasEntryToEcoZonesInGermany()) {
        AddOptionFromLegacy(meta.GetEntryToEcoZonesInGermany(), NLegacyOptions::NId::entryToEcoZonesInGermany, NLegacyOptions::NOrder::entryToEcoZonesInGermany);
    }
    if (meta.HasInsuranceType()) {
        Insurance.ConstructInPlace();
        Insurance->SetId(meta.GetInsuranceType());
    }
    if (meta.HasInsuranceDetails() && !Insurance.Empty()) {
        if (meta.GetInsuranceDetails().HasCost()) {
            Insurance->SetCost(meta.GetInsuranceDetails().GetCost());
        }
        if (meta.GetInsuranceDetails().HasTitle()) {
            Insurance->SetTitle(meta.GetInsuranceDetails().GetTitle());
        }
    }
    if (meta.HasSince()) {
        Since = TInstant::MicroSeconds(meta.GetSince());
    }
    if (meta.HasUntil()) {
        Until = TInstant::MicroSeconds(meta.GetUntil());
    }
    if (meta.HasDeliveryLocation()) {
        DeliveryLocation.ConstructInPlace();
        if (!DeliveryLocation->Deserialize(meta.GetDeliveryLocation())) {
            return false;
        }
    }
    if (meta.HasDeliveryLocationName()) {
        DeliveryLocationName = meta.GetDeliveryLocationName();
    }
    {
        {
            TRentalOfferLocation location;
            if (meta.HasReturnLocation()) {
                if (!location.MutableCoords().emplace_back().Deserialize(meta.GetReturnLocation())) {
                    return false;
                }
            }
            if (meta.HasReturnLocationName()) {
                location.MutableLocationName() = meta.GetReturnLocationName();
            }
            if (!location.GetLocationName().empty() || !location.GetCoords().empty()) {
                if (!HasReturnLocations()) {
                    ReturnLocations.ConstructInPlace();
                }
                MutableReturnLocations()->push_back(location);
            }
        }
        if (!HasReturnLocations()) {
            for (auto&& el : meta.GetReturnLocations()) {
                TRentalOfferLocation location;
                if (!location.DeserializeFromProto(el)) {
                    return false;
                }
                if (!HasReturnLocations()) {
                    ReturnLocations.ConstructInPlace();
                }
                MutableReturnLocations()->push_back(location);
            }
        }
    }
    if (meta.HasComment()) {
        Comment = meta.GetComment();
    }
    if (meta.HasTotalPayment()) {
        TotalPayment = meta.GetTotalPayment();
    }
    if (meta.HasDeposit()) {
        Deposit = meta.GetDeposit();
    }
    if (meta.HasStatus()) {
        Status = meta.GetStatus();
    }
    if (meta.HasLimitKmPerDay()) {
        LimitKmPerDay = meta.GetLimitKmPerDay();
    }
    if (meta.HasOverrunCostPerKm()) {
        OverrunCostPerKm = meta.GetOverrunCostPerKm();
    }
    if (meta.HasCurrency()) {
        Currency = meta.GetCurrency();
    }
    if (meta.HasRevision()) {
        Revision = meta.GetRevision();
    }
    if (meta.HasOfferImage()) {
        OfferImage = meta.GetOfferImage();
    }
    if (meta.HasOfferSmallImage()) {
        OfferSmallImage = meta.GetOfferSmallImage();
    }
    if (meta.HasOfferImageStyle()) {
        ERentalOfferImageStyle style;
        if (!TryFromString(meta.GetOfferImageStyle(), style)) {
            return false;
        }
        OfferImageStyle = style;
    }
    RenterOfferCommonInfo = TRentalOffer::TRenterOfferCommonInfo{};
    for (const auto& [key, value]: meta.GetRenterOfferCommonInfo()) {
        (*RenterOfferCommonInfo)[key] = value;
    }
    if (meta.HasOfferTimezone()) {
        OfferTimezone = meta.GetOfferTimezone();
    }
    if (meta.HasDateFormat()) {
        DateFormat = meta.GetDateFormat();
    }
    if (meta.HasTimeFormat()) {
        TimeFormat = meta.GetTimeFormat();
    }
    if (meta.HasTariffRecommendation()) {
        TariffRecommendation = TRentalEconomy::TTariffRecommendation();
        if (!TariffRecommendation->DeserializeFromProto(meta.GetTariffRecommendation())) {
            return false;
        }
    }
    if (meta.HasTotalDailyCost()) {
        TotalDailyCost = meta.GetTotalDailyCost();
    }
    if (!meta.GetOptions().empty()) {
        Options.clear();
        for (auto&& option : meta.GetOptions()) {
            TOption rentalOption;
            if (rentalOption.DeserializeFromProto(option)) {
                Options.push_back(std::move(rentalOption));
            }
        }
    }
    if (!meta.GetOptionsRecommendation().empty()) {
        OptionsRecommendation.ConstructInPlace();
        for (auto&& option: meta.GetOptionsRecommendation()) {
            TRentalEconomy::TOptionsRecommendation::TOption recommendedOption;
            if (recommendedOption.DeserializeFromProto(option)) {
                OptionsRecommendation->MutableTotalOptionCosts().push_back(std::move(recommendedOption));
            }
        }
    }
    if (!meta.GetInsurancesRecommendation().empty()) {
        InsurancesRecommendation.ConstructInPlace();
        for (auto&& insurance: meta.GetInsurancesRecommendation()) {
            TRentalEconomy::TInsurancesRecommendation::TInsuranceRecommendation recommendedInsurance;
            if (recommendedInsurance.DeserializeFromProto(insurance)) {
                InsurancesRecommendation->MutableInsurances().push_back(std::move(recommendedInsurance));
            }
        }
    }
    if (meta.HasLimitsRecommendation()) {
        LimitsRecommendation.ConstructInPlace();
        LimitsRecommendation->DeserializeFromProto(meta.GetLimitsRecommendation());
    }
    if (meta.HasStartRentalTimeout()) {
        StartRentalTimeout = TDuration::MicroSeconds(meta.GetStartRentalTimeout());
    }
    return true;
}

NDrive::NProto::TOffer TRentalOffer::SerializeToProto() const {
    auto info = TBase::SerializeToProto();
    auto meta = info.MutableRentalOffer();

    if (!Insurance.Empty()) {
        meta->SetInsuranceType(Insurance->GetId());
        NDrive::NProto::TRentalInsuranceDetails protoInsuranceDetails;
        if (Insurance->HasCost()) {
            protoInsuranceDetails.SetCost(Insurance->GetCostRef());
        }
        if (!Insurance->GetTitle().empty()) {
            protoInsuranceDetails.SetTitle(Insurance->GetTitle());
        }
        meta->MutableInsuranceDetails()->CopyFrom(protoInsuranceDetails);
    }

    meta->SetSince(Since.MicroSeconds());
    meta->SetUntil(Until.MicroSeconds());

    if (DeliveryLocation) {
        *meta->MutableDeliveryLocation() = DeliveryLocation->Serialize();
    }
    if (DeliveryLocationName) {
        meta->SetDeliveryLocationName(*DeliveryLocationName);
    }
    if (ReturnLocations) {
        for (const auto& el : *ReturnLocations) {
            meta->AddReturnLocations()->CopyFrom(el.SerializeToProto());
        }
    }
    if (Comment) {
        meta->SetComment(*Comment);
    }
    if (TotalPayment) {
        meta->SetTotalPayment(*TotalPayment);
    }
    if (Deposit) {
        meta->SetDeposit(*Deposit);
    }
    if (Status) {
        meta->SetStatus(*Status);
    }
    if (LimitKmPerDay) {
        meta->SetLimitKmPerDay(*LimitKmPerDay);
    }
    if (OverrunCostPerKm) {
        meta->SetOverrunCostPerKm(*OverrunCostPerKm);
    }
    if (Currency) {
        meta->SetCurrency(*Currency);
    }
    meta->SetRevision(Revision);
    if (OfferImage) {
        meta->SetOfferImage(*OfferImage);
    }
    if (StartRentalTimeout) {
        meta->SetStartRentalTimeout(StartRentalTimeout->MicroSeconds());
    }
    if (OfferSmallImage) {
        meta->SetOfferSmallImage(*OfferSmallImage);
    }
    if (OfferImageStyle && *OfferImageStyle != ERentalOfferImageStyle::Default) {
        meta->SetOfferImageStyle(ToString(*OfferImageStyle));
    }
    if (RenterOfferCommonInfo) {
        for (const auto& [key, value]: *RenterOfferCommonInfo) {
            (*meta->MutableRenterOfferCommonInfo())[key] = value;
        }
    }
    if (OfferTimezone) {
        meta->SetOfferTimezone(*OfferTimezone);
    }
    if (DateFormat) {
        meta->SetDateFormat(*DateFormat);
    }
    if (TimeFormat) {
        meta->SetTimeFormat(*TimeFormat);
    }
    if (TariffRecommendation) {
        meta->MutableTariffRecommendation()->CopyFrom(TariffRecommendation->SerializeToProto());
    }
    if (TotalDailyCost) {
        meta->SetTotalDailyCost(*TotalDailyCost);
    }
    if (!Options.empty()) {
        for (const auto& option: Options) {
            NDrive::NProto::TRentalOption optionProto;
            if (option.SerializeToProto(optionProto)) {
                meta->AddOptions()->CopyFrom(optionProto);
            }
        }
    }
    if (OptionsRecommendation) {
        for (const auto& option: OptionsRecommendation->GetTotalOptionCosts()) {
            meta->AddOptionsRecommendation()->CopyFrom(option.SerializeToProto());
        }
    }
    if (InsurancesRecommendation) {
        for (const auto& insurance: InsurancesRecommendation->GetInsurances()) {
            meta->AddInsurancesRecommendation()->CopyFrom(insurance.SerializeToProto());
        }
    }
    if (LimitsRecommendation) {
        meta->MutableLimitsRecommendation()->CopyFrom(LimitsRecommendation->SerializeToProto());
    }
    return info;
}

void TRentalOffer::FillBill(TBill& bill, const TOfferPricing& /*pricing*/, TOfferStatePtr /*segmentState*/, ELocalization /*locale*/, const NDrive::IServer* /*server*/, ui32 /*cashbackPercent*/) const {
    bill.MutableRecords().clear();
}

void TRentalOffer::PatchSessionReport(NJson::TJsonValue& result, NDriveSession::TReportTraits traits, ELocalization locale,
                                      const NDrive::IServer& server, const TOfferSessionPatchData& patchData) const {
    TBase::PatchSessionReport(result, traits, locale, server, patchData);
    auto localization = server.GetLocalization();

    TString localizedCurrency{ "" };
    if (Currency) {
        localizedCurrency = " " + localization->GetLocalString(locale,  MakeUnitValueId(*Currency));
    }

    if (traits & NDriveSession::EReportTraits::ReportBillSections && result.Has("bill")) {
        NJson::TJsonValue jPatchedBillData;
        {
            NJson::TJsonValue section;
            section.AppendValue(NJson::TMapBuilder
                ("type", "section")
                ("title", localization ? localization->GetLocalString(locale, "rental.car_status.title") : "rental.car_status.title")
            );

            if (patchData.MileageStart.Defined()) {
                NJson::TJsonValue jValue;
                jValue["title"] = localization ? localization->GetLocalString(locale, "rental.mileage_start.title") : "rental.mileage_start.title";
                jValue["value"] = localization->DistanceFormatKm(locale, *patchData.MileageStart);
                section.AppendValue(jValue);
            }
            if (patchData.MileageFinish.Defined()) {
                NJson::TJsonValue jValue;
                jValue["title"] = localization ? localization->GetLocalString(locale, "rental.mileage_finish.title") : "rental.mileage_finish.title";
                jValue["value"] = localization->DistanceFormatKm(locale, *patchData.MileageFinish);
                section.AppendValue(jValue);
            }
            if (patchData.FuelLevelStart.Defined()) {
                NJson::TJsonValue jValue;
                jValue["title"] = localization ? localization->GetLocalString(locale, "rental.fuel_tank_level_start.title") : "rental.fuel_tank_level_start.title";
                jValue["value"] =
                      ToString(*patchData.FuelLevelStart)
                    + (localization ? localization->GetLocalString(locale, "rental.fuel_tank_level_start.units") : " rental.fuel_tank_level_start.units");
                section.AppendValue(jValue);
            }
            if (patchData.FuelLevelFinish.Defined()) {
                NJson::TJsonValue jValue;
                jValue["title"] = localization ? localization->GetLocalString(locale, "rental.fuel_tank_level_finish.title") : "rental.fuel_tank_level_finish.title";
                jValue["value"] =
                      ToString(*patchData.FuelLevelFinish)
                    + (localization ? localization->GetLocalString(locale, "rental.fuel_tank_level_finish.units") : " rental.fuel_tank_level_finish.units");
                section.AppendValue(jValue);
            }

            if (section.GetArray().size() > 1) {
                for (auto&& map : section.GetArray()) {
                    jPatchedBillData.AppendValue(std::move(map));
                }
            }
        }
        {
            NJson::TJsonValue section;
            section.AppendValue(NJson::TMapBuilder
                ("type", "section")
                ("title", localization ? localization->GetLocalString(locale, "rental.offer_options.title") : "rental.offer_options.title")
            );

            for (const auto& option: Options) {
                if (option.HasValue() && option.GetValueRef() && option.HasStringConstants()) {
                    NJson::TJsonValue jValue;
                    jValue["title"] = localization ? localization->GetLocalString(locale, option.GetStringConstantsRef().GetTitle()) : option.GetStringConstantsRef().GetTitle();
                    if (option.HasCost()) {
                        jValue["value"] = ToString(option.GetCostRef()) + localizedCurrency;
                    }
                    section.AppendValue(jValue);
                }
            }
            if (section.GetArray().size() > 1) {
                for (auto&& map : section.GetArray()) {
                    jPatchedBillData.AppendValue(std::move(map));
                }
            }
        }
        {
            NJson::TJsonValue section;
            section.AppendValue(NJson::TMapBuilder
                ("type", "section")
                ("title", localization ? localization->GetLocalString(locale, "rental.full_receipt.title") : "rental.full_receipt.title")
            );
            if (patchData.Mileage) {
                NJson::TJsonValue jValue;
                jValue["type"] = "mileage";
                jValue["title"] = localization ? localization->GetLocalString(locale, billMileageTitle) : billMileageTitle;
                jValue["value"] = localization->DistanceFormatKm(locale, *patchData.Mileage);
                section.AppendValue(jValue);
            }
            {
                NJson::TJsonValue jValue;
                jValue["type"] = "duration";
                jValue["title"] = localization ? localization->GetLocalString(locale, billDurationTitle) : billDurationTitle;
                jValue["value"] = GetDurationReport(patchData.Duration, locale, localization);
                section.AppendValue(jValue);
            }
            if (!Insurance.Empty()) {
                NJson::TJsonValue jValue;
                jValue["title"] = localization ? localization->GetLocalString(locale, "rental.insurance_type.title") : "rental.insurance_type.title";

                TString costValue;
                if (Insurance->HasCost()) {
                    costValue = ", " + ToString(Insurance->GetCostRef()) + localizedCurrency;
                }

                TString titleValue;
                if (Insurance->GetTitle().empty()) {
                    const auto titleId = "rental.insurance_type." + Insurance->GetId() + ".title";
                    titleValue = localization ? localization->GetLocalString(locale, titleId) : titleId;
                } else {
                    titleValue = localization ? localization->GetLocalString(locale, Insurance->GetTitle()) : Insurance->GetTitle();
                }
                jValue["value"] = titleValue + costValue;
                section.AppendValue(jValue);
            }
            if (TotalDailyCost) {
                NJson::TJsonValue jValue;
                jValue["type"] = "total_daily_cost";
                jValue["title"] = localization ? localization->GetLocalString(locale, totalDailyCostTitle) : totalDailyCostTitle;
                jValue["value"] = ToString(*TotalDailyCost) + localizedCurrency;
                section.AppendValue(jValue);
            }
            if (Deposit) {
                NJson::TJsonValue jValue;
                jValue["type"] = "deposit";
                jValue["title"] = localization ? localization->GetLocalString(locale, "rental.deposit.title") : "rental.deposit.title";
                jValue["value"] = ToString(*Deposit) + localizedCurrency;
                section.AppendValue(jValue);
            }
            if (section.GetArray().size() > 1) {
                for (auto&& map : section.GetArray()) {
                    jPatchedBillData.AppendValue(std::move(map));
                }
            }
        }
        if (TotalPayment) {
            NJson::TJsonValue jValue;
            jValue["type"] = "total";
            jValue["title"] = localization ? localization->GetLocalString(locale, "rental.total_payment.title") : "rental.total_payment.title";
            jValue["value"] = ToString(*TotalPayment) + localizedCurrency;
            jPatchedBillData.AppendValue(jValue);
        }
        result["bill"] = jPatchedBillData;
    }

    if (traits & NDriveSession::EReportTraits::ReportBillSectionsDM) {
        NJson::TJsonValue jPatchedBillData;
        if (patchData.MileageStart.Defined()) {
            NJson::TJsonValue jValue;
            jValue["type"] = "mileage_start";
            jValue["value"] = ToString(*patchData.MileageStart);
            jPatchedBillData.AppendValue(jValue);
        }
        if (patchData.MileageFinish.Defined()) {
            NJson::TJsonValue jValue;
            jValue["type"] = "mileage_finish";
            jValue["value"] = ToString(*patchData.MileageFinish);
            jPatchedBillData.AppendValue(jValue);
        }
        if (patchData.FuelLevelStart.Defined()) {
            NJson::TJsonValue jValue;
            jValue["type"] = "fuel_tank_level_start";
            jValue["value"] = ToString(*patchData.FuelLevelStart);;
            jPatchedBillData.AppendValue(jValue);
        }
        if (patchData.FuelLevelFinish.Defined()) {
            NJson::TJsonValue jValue;
            jValue["type"] = "fuel_tank_level_finish";
            jValue["value"] = ToString(*patchData.FuelLevelFinish);
            jPatchedBillData.AppendValue(jValue);
        }
        {
            NJson::TJsonValue section;
            section["type"] = "offer_options";

            for (const auto& option: Options) {
                if (option.HasValue() && option.GetValueRef() && option.HasStringConstants()) {
                    section["value"].AppendValue((localization ? localization->GetLocalString(locale, option.GetStringConstantsRef().GetTitle()) : option.GetStringConstantsRef().GetTitle()));
                }
            }

            if (section["value"].IsDefined()) {
                jPatchedBillData.AppendValue(section);
            }
        }
        if (patchData.Mileage) {
            NJson::TJsonValue jValue;
            jValue["type"] = "mileage";
            jValue["value"] = ToString(*patchData.Mileage);
            jPatchedBillData.AppendValue(jValue);
        }
        {
            NJson::TJsonValue jValue;
            jValue["type"] = "duration";
            jValue["value"] = ToString(patchData.Duration.Seconds());
            jPatchedBillData.AppendValue(jValue);
        }
        if (!Insurance.Empty()) {
            NJson::TJsonValue jValue;
            jValue["type"] = "insurance_type";
            if (Insurance->GetTitle().empty()) {
                const auto titleId = "rental.insurance_type." + Insurance->GetId() + ".title";
                jValue["value"] = localization ? localization->GetLocalString(locale, titleId) : titleId;
            } else {
                jValue["value"] = localization ? localization->GetLocalString(locale, Insurance->GetTitle()) : Insurance->GetTitle();
            }
            jPatchedBillData.AppendValue(jValue);
        }
        if (Deposit) {
            NJson::TJsonValue jValue;
            jValue["type"] = "deposit";
            TString costStr = ToString(*Deposit);
            if (Currency) {
                costStr += " " + localization->GetLocalString(locale,  MakeUnitValueId(*Currency));
            }
            jValue["value"] = costStr;
            jPatchedBillData.AppendValue(jValue);
        }
        if (TotalPayment) {
            NJson::TJsonValue jValue;
            jValue["type"] = "total";
            jValue["value"] = ToString(*TotalPayment) + localizedCurrency;
            jPatchedBillData.AppendValue(jValue);
        }
        result["bill_dm"] = jPatchedBillData;
    }

    if (DeliveryLocationName) {
        NJson::TJsonValue jValue;
        result["delivery_location"] = *DeliveryLocationName;
    }
    if (ReturnLocations) {
        if (ReturnLocations->size() == 1 && ReturnLocations->begin()->GetType() == ELocationType::Coord) {
            NJson::TJsonValue jValue;
            result["return_location"] = ReturnLocations->begin()->GetLocationName();
        };
    }

    if (!patchData.LocalEvents.empty()) {
        TMaybe<TInstant> actualSince, actualUntil;
        for (auto& event: patchData.LocalEvents) {
            if (!actualSince && EObjectHistoryAction::TagEvolve == event.GetHistoryAction() && TChargableTag::Acceptance == event.GetTagName()) {
                actualSince = event.GetInstant();
            } else if (EObjectHistoryAction::DropTagPerformer ==  event.GetHistoryAction() && TChargableTag::Reservation == event.GetTagName()) {
                actualUntil = event.GetInstant();
            }
        }
        if (actualSince) {
            result["actual_since"] = actualSince->Seconds();
        }
        if (actualUntil) {
            result["actual_until"] = actualUntil->Seconds();
        }
    }
}

TRentalEconomy::TOfferTariffRecommendationData TRentalOffer::BuildOfferTariffRecommendationData() const {
    TRentalEconomy::TOfferTariffRecommendationData offerData;
    offerData.SetSince(GetSince());
    offerData.SetUntil(GetUntil());
    offerData.SetCarId(GetObjectId());
    offerData.Options = &GetOptions();
    return offerData;
}

bool TRentalOffer::PatchOffer(
    const NJson::TJsonValue& value,
    NDrive::TEntitySession& session,
    const NDrive::IServer* server,
    bool checkRidingConflict
) {
    const auto oldOfferData = BuildOfferTariffRecommendationData();

    const auto& clientId = GetUserId();
    const auto carIdFrom = GetObjectId();
    const auto carIdTo = NJson::FromJson<TMaybe<TString>>(value["car_id"]);

    if (!carIdTo) {
        session.SetErrorInfo(patchOfferSourceSession, "offer has not carId");
        return false;
    }
    if (!TRentalOfferHolderTag::LockUser(clientId, session)) {
        return false;
    }
    if (!TRentalOfferHolderTag::LockTwoCars(carIdFrom, *carIdTo, session)) {
        return false;
    }

    const auto api = Yensured(server)->GetDriveAPI();
    auto builder = api->GetRolesManager()->GetAction(GetBehaviourConstructorId());
    if (!builder) {
        return false;
    }
    auto rentalBuilder = builder->GetAs<TRentalOfferBuilder>();
    if (!rentalBuilder) {
        return false;
    }

    if (!PatchFromJson(value, session, *rentalBuilder)) {
        return false;
    }

    auto permissions = api->GetUserPermissions(clientId, {});
    if (!permissions) {
        session.SetErrorInfo(patchOfferSourceSession, "cannot create permissions for " + clientId);
        return false;
    }
    const auto rentalEvents = TTimetableBuilder::Instance().GetTimetableEvents(*permissions, server, session);
    if (!rentalEvents) {
        return false;
    }

    TMap<TString, TMap<TString, TTimetableEventMetadata>> carsTimetable;
    if (!TTimetableBuilder::Instance().BuildTimetable<TTimetableBuilder::ETimetableType::Rental>(carsTimetable,
                                                                                                rentalEvents->first,
                                                                                                rentalEvents->second,
                                                                                                GetSince(),
                                                                                                GetUntil(),
                                                                                                *permissions,
                                                                                                server,
                                                                                                session)) {
        session.SetErrorInfo(patchOfferSourceSession, "cannot build timetable");
        return false;
    }

    if (auto carOfferIt = carsTimetable.find(carIdFrom); carsTimetable.end() != carOfferIt) {
        carOfferIt->second.erase(GetOfferId());
        if (carOfferIt->second.empty()) {
            carsTimetable.erase(carOfferIt);
        }
    }

    if (auto conflictStatus = TTimetableBuilder::Instance().HasBookingTimetableConflicts(carsTimetable,
                                                                                         carIdFrom,
                                                                                         GetSince(),
                                                                                         server,
                                                                                         session,
                                                                                         clientId,
                                                                                         checkRidingConflict); conflictStatus.first) {
        session.SetErrorInfo(patchOfferSourceSession, conflictStatus.second, NDrive::MakeError(conflictStatus.second));
        return false;
    }

    if (const auto newOfferData = BuildOfferTariffRecommendationData(); TRentalEconomy::IsNewTariffRecommendationRequired(newOfferData, oldOfferData)) {
        TariffRecommendation.Clear();
        InsurancesRecommendation.Clear();
        LimitsRecommendation.Clear();

        const auto& economy = rentalBuilder->GetEconomy();
        bool sessionErrorHappend = false;
        TariffRecommendation = economy.GetRecommendedTariff(newOfferData, server, session, sessionErrorHappend);
        if (sessionErrorHappend) {
            return false;
        }
        if (!TariffRecommendation.Empty() && TariffRecommendation->HasName()) {
            InsurancesRecommendation = economy.GetRecommendedInsurances(TariffRecommendation->GetNameRef());
            LimitsRecommendation = economy.GetRecommendedLimits(TariffRecommendation->GetNameRef());
        }
        OptionsRecommendation = economy.GetRecommendedOptions(newOfferData);
    }
    return true;
}

ui64 TRentalOffer::GetOfferRevision() const {
    return Revision;
}

TExpected<bool, NJson::TJsonValue> TRentalOffer::CheckAgreementRequirements(ELocalization /*locale*/, const TUserPermissions::TPtr /*permissions*/, const NDrive::IServer* /*server*/) const {
    return true;
}

NThreading::TFuture<TString> TRentalOffer::BuildAgreement(ELocalization locale, TAtomicSharedPtr<TCompiledRiding> compiledSession, TUserPermissions::TPtr permissions, const NDrive::IServer* server) const {
    const auto& agreement = GetAgreement();
    auto agreementTemplate = permissions->GetSetting<TString>(agreement);
    R_ENSURE(agreementTemplate, HTTP_INTERNAL_SERVER_ERROR, "setting " << agreement << " is not found");

    auto tx = server->GetDriveAPI()->BuildTx<NSQL::ReadOnly>();
    const auto usersInfo = server->GetDriveAPI()->GetUsersData()->FetchInfo(GetUserId(), tx);
    R_ENSURE(usersInfo.size() == 1, HTTP_INTERNAL_SERVER_ERROR, "wrong usersInfo.size()", tx);

    TDriveUserData user = std::move(usersInfo.begin()->second);
    NThreading::TFuture<TUserPassportData> passportDataFuture = NThreading::MakeFuture(TUserPassportData());
    if (!user.GetPassportDatasyncRevision().empty()) {
        passportDataFuture = server->GetDriveAPI()->GetPrivateDataClient().GetPassport(user, user.GetPassportDatasyncRevision());
    }

    NThreading::TFuture<TUserDrivingLicenseData> licenseDataFuture = NThreading::MakeFuture(TUserDrivingLicenseData());
    if (!user.GetDrivingLicenseDatasyncRevision().empty()) {
        licenseDataFuture = server->GetDriveAPI()->GetPrivateDataClient().GetDrivingLicense(user, user.GetDrivingLicenseDatasyncRevision());
    }

    return NThreading::WaitAll(passportDataFuture.IgnoreResult(), licenseDataFuture.IgnoreResult())
        .Apply(
            [this,
            server,
            permissions,
            compiledSession,
            locale,
            user = std::move(user),
            agreementTemplate = std::move(*agreementTemplate),
            passportDataFuture = std::move(passportDataFuture),
            licenseDataFuture = std::move(licenseDataFuture)]
            (const NThreading::TFuture<void>& /*agreementFuture*/) mutable {
                auto passportData = passportDataFuture.GetValue();
                auto licenseData = licenseDataFuture.GetValue();
                auto tx = server->GetDriveAPI()->BuildTx<NSQL::ReadOnly>();
                auto localization = server->GetLocalization();
                if (localization) {
                    agreementTemplate = localization->ApplyResources(agreementTemplate, locale);
                }

                R_ENSURE(
                    FillAgreementTemplate(agreementTemplate, server, tx, compiledSession, locale, user, passportData, licenseData),
                    {},
                    "cannot fill agreement template",
                    tx
                );
                FillSharedSessionAgreement(agreementTemplate, tx, server);

                return agreementTemplate;
        });
}

bool TRentalOffer::PatchFromJson(const NJson::TJsonValue& value, NDrive::TEntitySession& session, const TRentalOfferBuilder& builder) {
    auto fieldsChecker = [&value](const TString& keyword) {
        return value.Has(keyword);
    };
    PatchOfferFromJson(value, this, fieldsChecker, builder);
    const auto carId = NJson::FromJson<TMaybe<TString>>(value["car_id"]);
    if (carId) {
        SetObjectId(*carId);
    } else {
        session.SetErrorInfo(patchFromJsonSourceSession, "json does not have car_id");
        return false;
    }

    const auto since = NJson::FromJson<TMaybe<TInstant>>(value["since"]);
    const auto until = NJson::FromJson<TMaybe<TInstant>>(value["until"]);

    if (CheckAndSetSinceUntil(*this, since, until, TInstant::Now()) != EOfferCorrectorResult::Success) {
        session.SetErrorInfo(patchFromJsonSourceSession, wrongSinceUntil, NDrive::MakeError(wrongSinceUntil));
        return false;
    }
    auto revision = NJson::FromJson<TMaybe<ui64>>(value["revision"]);
    if (!revision) {
        session.SetErrorInfo(patchFromJsonSourceSession, "json does not have revision");
        return false;
    } else {
        if (Revision <= revision) {
            Revision += 1;
        } else {
            session.SetErrorInfo(patchFromJsonSourceSession, revisionIsInvalid, NDrive::MakeError(revisionIsInvalid));
            return false;
        }
    }
    CalculateMileageLimit();
    return true;
}

NJson::TJsonValue TRentalOffer::DoBuildJsonReport(const TReportOptions& options, const ICommonOfferBuilderAction* constructor, const NDrive::IServer& server) const {
    NJson::TJsonValue result = TBase::DoBuildJsonReport(options, constructor, server);
    if (!(options.Traits & NDriveSession::ReportOfferDetails)) {
        return result;
    }

    auto locale = options.Locale;
    auto localization = server.GetLocalization();

    ReportToJson(result, server, locale);

    auto builder = dynamic_cast<const TRentalOfferBuilder*>(constructor);

    if (builder && localization) {
        NJson::TJsonValue primary = NJson::JSON_ARRAY;
        TString unitValueId;
        TString localizedUnit;
        if (Currency) {
            unitValueId = MakeUnitValueId(*Currency);
            localizedUnit = localization ? localization->GetLocalString(locale, unitValueId) : unitValueId;
        }

        // options for frontend
        {
            NJson::TJsonValue options = NJson::JSON_ARRAY;
            for (const auto& option: Options) {
                if (option.HasConstants()) {
                    if (Currency) {
                        option.SetUnit(unitValueId);
                    }
                    auto optionReport = option.GetReport(locale, server);
                    if (!optionReport.empty()) {
                        options.AppendValue(std::move(optionReport.front()));
                    }
                }
            }
            NJson::TJsonValue jOptionsFront;
            jOptionsFront["id"] = "options_front";
            jOptionsFront["options"] = std::move(options);
            primary.AppendValue(std::move(jOptionsFront));
        }

        {
            TRentalOfferVariable<ui64> totalDailyCostVariable;
            totalDailyCostVariable.SetId(totalDailyCostKeyword);
            totalDailyCostVariable.SetValue(TotalDailyCost);
            totalDailyCostVariable.SetTitle(totalDailyCostTitle);
            totalDailyCostVariable.SetSubtitle(totalDailyCostSubtitle);
            if (Currency) {
                totalDailyCostVariable.SetUnit(unitValueId);
            }
            if (TotalDailyCost) {
                primary.AppendValue(MakeObjectVisible(totalDailyCostVariable.GetReport(locale, server)));
            } else {
                primary.AppendValue(totalDailyCostVariable.GetReport(locale, server));
            }
        }

        // options for mobile
        if (!server.GetSettings().GetValue<bool>("rental.is_old_options_format").GetOrElse(true)) {
            auto optionPattern = server.GetSettings().GetValue<TString>("rental.mobile.booking_screen.option.pattern").GetOrElse("_Title_ - _Cost_ _Currency_");
            TString optionsValue;
            NJson::TJsonValue options = NJson::JSON_ARRAY;
            for (size_t i = 0; i < Options.size(); ++i) {
                const auto& option = Options[i];
                if (option.HasConstants() && option.HasStringConstants() && option.HasValue() && option.GetValueRef()) {
                    auto& titleId = option.GetStringConstantsRef().GetTitle();
                    auto localizedTitle = localization ? localization->GetLocalString(locale, titleId) : titleId;
                    if (option.HasCost()) {
                        auto copyOptionPattern = optionPattern;
                        SubstGlobal(copyOptionPattern, "_Title_", localizedTitle);
                        SubstGlobal(copyOptionPattern, "_Cost_", ToString(option.GetCostRef()));
                        SubstGlobal(copyOptionPattern, "_Currency_", localizedUnit);

                        optionsValue += copyOptionPattern;
                    } else {
                        optionsValue += localizedTitle;
                    }
                    if (i + 1 < Options.size()) {
                        optionsValue += "\n";
                    }
                }
            }
            if (!optionsValue.empty()) {
                NJson::TJsonValue jOptionsMobile;
                jOptionsMobile["id"] = "options_mobile";
                jOptionsMobile["title"] = localization ? localization->GetLocalString(locale, "rental.offer_options.title") : "rental.offer_options.title";
                jOptionsMobile["value"] = std::move(optionsValue);
                primary.AppendValue(MakeObjectVisible(std::move(jOptionsMobile)));
            }
        }

        if (!builder->GetAllowedInsurances().empty()) {
            NJson::TJsonValue insurances = NJson::JSON_ARRAY;
            for (const auto& insurance: builder->GetAllowedInsurances()) {
                insurances.AppendValue(insurance.GetReport(locale, server));
            }
            NJson::TJsonValue insurancesObj = NJson::JSON_MAP;
            insurancesObj["id"] = "insurance_types";
            insurancesObj["types"] = insurances;

            FillTitles("rental.insurance_types.title", "rental.insurance_types.subtitle", insurancesObj, localization, locale);

            primary.AppendValue(insurancesObj);
            primary.AppendValue(MakeObjectVisible(GetInsuranceReport(locale, server, builder->GetAllowedInsurances(), unitValueId)));
        }

        if (!builder->GetAllowedDeliveryLocations().empty()) {
            NJson::TJsonValue locations = NJson::JSON_ARRAY;
            for (const auto& location: builder->GetAllowedDeliveryLocations()) {
                locations.AppendValue(location.GetReport(locale, server));
            }
            NJson::TJsonValue locationsObj = NJson::JSON_MAP;
            locationsObj["id"] = "delivery_locations";
            locationsObj["locations"] = locations;
            FillTitles("rental.delivery_locations.title", "rental.delivery_locations.subtitle", locationsObj, localization, locale);
            primary.AppendValue(locationsObj);
        }

        if (!builder->GetAllowedReturnLocations().empty()) {
            NJson::TJsonValue locations = NJson::JSON_ARRAY;
            for (const auto& location: builder->GetAllowedReturnLocations()) {
                locations.AppendValue(location.GetReport(locale, server));
            }
            NJson::TJsonValue locationsObj = NJson::JSON_MAP;
            locationsObj["id"] = "return_locations";
            locationsObj["locations"] = locations;
            FillTitles("rental.return_locations.title", "rental.return_locations.subtitle", locationsObj, localization, locale);
            primary.AppendValue(locationsObj);
        }

        if (!builder->GetAllowedCurrencies().empty()) {
            NJson::TJsonValue currencies = NJson::JSON_ARRAY;
            for (const auto& currency: builder->GetAllowedCurrencies()) {
                currencies.AppendValue(currency.GetReport(locale, server));
            }

            NJson::TJsonValue currenciesObj = NJson::JSON_MAP;
            currenciesObj["id"] = "currencies";
            currenciesObj["currencies"] = currencies;
            FillTitles("rental.currencies.title", "rental.currencies.subtitle", currenciesObj, localization, locale);
            primary.AppendValue(currenciesObj);
        }

        if (builder->GetAllowComment()) {
            auto comment = builder->GetComment();
            comment.SetValue(Comment);
            primary.AppendValue(comment.GetReport(locale, server));
        }

        if (!builder->GetAllowedStatuses().empty()) {
            NJson::TJsonValue statuses = NJson::JSON_ARRAY;
            for (const auto& status: builder->GetAllowedStatuses()) {
                statuses.AppendValue(status.GetReport(locale, server));
            }
            NJson::TJsonValue statusesObj = NJson::JSON_MAP;
            statusesObj["id"] = "statuses";
            statusesObj["statuses"] = statuses;
            FillTitles("rental.statuses.title", "rental.statuses.subtitle", statusesObj, localization, locale);
            primary.AppendValue(statusesObj);
        }

        if (builder->GetAllowTotalPayment()) {
            auto totalPayment = builder->GetTotalPayment();
            totalPayment.SetValue(TotalPayment);
            if (Currency) {
                totalPayment.SetUnit(unitValueId);
            }
            primary.AppendValue(MakeObjectVisible(totalPayment.GetReport(locale, server)));
        }

        if (builder->GetAllowDeposit()) {
            auto deposit = builder->GetDeposit();
            deposit.SetValue(Deposit);
            if (Currency) {
                deposit.SetUnit(unitValueId);
            }
            primary.AppendValue(MakeObjectVisible(deposit.GetReport(locale, server)));
        }

        if (builder->GetAllowLimitKmPerDay()) {
            auto limitKmPerDay = builder->GetLimitKmPerDay();
            limitKmPerDay.SetValue(LimitKmPerDay);
            limitKmPerDay.SetUnit(unitKmPerDay);
            primary.AppendValue(MakeObjectVisible(limitKmPerDay.GetReport(locale, server)));
        }

        if (builder->GetAllowOverrunCostPerKm()) {
            auto overrunCostPerKm = builder->GetOverrunCostPerKm();
            overrunCostPerKm.SetValue(OverrunCostPerKm);
            if (Currency) {
                overrunCostPerKm.SetUnit(MakeUnitPermKmId(*Currency));
            }
            primary.AppendValue(MakeObjectVisible(overrunCostPerKm.GetReport(locale, server)));
        }
        result.InsertValue("primary", std::move(primary));
    }
    // will be removed after frontend update
    if (server.GetSettings().GetValue<bool>("rental.is_old_options_format").GetOrElse(true)) {
        NJson::TJsonValue secondary = NJson::JSON_ARRAY;
        for (const auto& option: Options) {
            if (option.HasConstants()) {
                auto optionReport = option.GetReport(locale, server);
                if (!optionReport.empty()) {
                    secondary.AppendValue(std::move(optionReport.front()));
                }
            }
        }

        result.InsertValue("secondary", std::move(secondary));
    }
    return result;
}

void TRentalOffer::ReportToJson(
    NJson::TJsonValue& value,
    const NDrive::IServer& server,
    const ELocalization locale
) const {
    // legacy options
    // will be deleted after frontend update
    {
        NJson::TJsonValue offerOptions = NJson::TJsonMap();

        for (const auto& option: Options) {
            if (option.HasConstants()) {
                NJson::InsertNonNull(offerOptions, option.GetConstantsRef().GetId(), option.OptionalValue());
            }
        }

        value["offer_options"] = offerOptions;
    }

    {
        NJson::TJsonArray offerOptions = NJson::TJsonArray();
        for (const auto& option: Options) {
            if (option.HasConstants()) {
                NJson::TJsonValue jOption;
                {
                    jOption["id"] = option.GetConstantsRef().GetId();
                    NJson::InsertField(jOption, "value", option.OptionalValue());
                    NJson::InsertField(jOption, "cost", option.OptionalCost());
                    offerOptions.AppendValue(jOption);
                }
            }
        }

        value["options"] = offerOptions;
    }
    if (!Insurance.Empty()) {
        // legacy insurance
        // will be deleted after frontend update
        value["insurance_type"] =  Insurance->GetId();

        NJson::TJsonValue jVal;
        jVal["id"] = Insurance->GetId();
        NJson::InsertField(jVal, "cost", Insurance->OptionalCost());
        value["insurance"] = jVal;
    }
    NJson::InsertNonNull(value, "since", NJson::Seconds(Since));
    NJson::InsertNonNull(value, "until", NJson::Seconds(Until));

    NJson::InsertNonNull(value, "delivery_location", DeliveryLocation);
    NJson::InsertNonNull(value, "delivery_location_name", DeliveryLocationName);

    NJson::InsertNonNull(value, "return_locations", ReturnLocations);
    if (ReturnLocations && ReturnLocations->size() == 1 && ReturnLocations->begin()->GetType() == ELocationType::Coord) {
        NJson::InsertField(value, "return_location", *ReturnLocations->begin()->GetCoords().begin());
        NJson::InsertField(value, "return_location_name", ReturnLocations->begin()->GetLocationName());
    }

    NJson::InsertNonNull(value, "comment", Comment);

    NJson::InsertNonNull(value, "total_payment", TotalPayment);
    NJson::InsertNonNull(value, "deposit", Deposit);

    NJson::InsertNonNull(value, "status", Status);
    NJson::InsertNonNull(value, "limit_km_per_day", LimitKmPerDay);
    NJson::InsertNonNull(value, "overrun_cost_per_km", OverrunCostPerKm);
    if (Currency) {
        auto localization = server.GetLocalization();
        auto unitValueId = MakeUnitValueId(*Currency);
        value["currency"] = *Currency;
        value["currency_name"] = localization ? localization->GetLocalString(locale,  unitValueId) : unitValueId;
    }
    NJson::InsertNonNull(value, "car_id", GetObjectId());
    value["revision"] = Revision;
    NJson::InsertNonNull(value, "offer_image", OfferImage);
    NJson::InsertNonNull(value, "offer_small_image", OfferSmallImage);
    if (OfferImageStyle) {
        NJson::InsertField(value, "offer_image_style", NJson::Stringify(OfferImageStyle.GetRef()));
    }
    ui64 accumulatedCost = 0;
    NJson::TJsonValue jRecommendations;
    if (TariffRecommendation) {
        jRecommendations["tariff"] = TariffRecommendation->ReportToJson(Since, Until, accumulatedCost);
    }
    if (OptionsRecommendation) {
        jRecommendations["options"] = OptionsRecommendation->ReportToJson(accumulatedCost);
    }
    if (InsurancesRecommendation) {
        jRecommendations["insurances"] = InsurancesRecommendation->ReportToJson();
    }
    if (LimitsRecommendation) {
        jRecommendations["limits"] = LimitsRecommendation->ReportToJson();
    }
    jRecommendations["total_payment"] = accumulatedCost;
    value["recommendations"] = jRecommendations;
    NJson::InsertNonNull(value, totalDailyCostKeyword, TotalDailyCost);
}

bool TRentalOffer::FillAgreementTemplate(TString& agreementTemplate,
                                         const NDrive::IServer* server,
                                         NDrive::TEntitySession& session,
                                         TAtomicSharedPtr<TCompiledRiding> compiledSession,
                                         const ELocalization locale,
                                         const TDriveUserData& user,
                                         const TUserPassportData& passportData,
                                         const TUserDrivingLicenseData& drivingLicense) const {
    R_ENSURE(server, HTTP_INTERNAL_SERVER_ERROR, "null Server");
    auto localization = server->GetLocalization();
    if (!localization) {
        session.SetErrorInfo("TRentalOffer::FillAgreementTemplate", "localization is nullptr");
        return false;
    }

    const auto carsFetchResult = server->GetDriveDatabase().GetCarManager().FetchInfo(GetObjectId(), session);
    R_ENSURE(carsFetchResult, {}, "cannot fetch object " << GetObjectId(), session);
    const auto car = carsFetchResult.GetResultPtr(GetObjectId());
    R_ENSURE(car, HTTP_INTERNAL_SERVER_ERROR, "cannot find object " << GetObjectId(), session);

    const auto modelFetchResult = server->GetDriveDatabase().GetModelsDB().GetCachedOrFetch(car->GetModel(), session);
    R_ENSURE(modelFetchResult, {}, "cannot fetch model " << car->GetModel(), session);
    const auto modelPtr = modelFetchResult.GetResultPtr(car->GetModel());

    const auto snapshotDiff = compiledSession ? compiledSession->GetSnapshotsDiffPtr() : nullptr;
    TMaybe<i64> mileage;
    TMaybe<ui32> fuelLevel;
    if (snapshotDiff) {
        if (snapshotDiff->HasStartMileage()) {
            mileage = static_cast<i64>(std::round(snapshotDiff->GetStartMileageRef()));
        }
        auto startFuelLevel = snapshotDiff->GetStartFuelLevel();
        auto startFuelVolume = snapshotDiff->GetStartFuelVolume();
        if (startFuelLevel || startFuelVolume) {
            startFuelLevel = NDrive::GetFuelLevel(startFuelLevel, startFuelVolume, car, modelPtr);
        }
        if (startFuelLevel) {
            fuelLevel = startFuelLevel->TryConvertTo<double>();
        }
    }

    auto driveApi = server->GetDriveAPI();
    const auto& agreementDateFormat = DateFormat.GetOrElse(DefaultDateFormat);
    const auto& agreementTimeFormat = TimeFormat.GetOrElse(DefaultTimeFormat);

    // Customer
    {
        SubstGlobal(agreementTemplate, "{{rental.customer.first_name}}", NHtml::EscapeAttributeValue(user.GetFirstName()));
        SubstGlobal(agreementTemplate, "{{rental.customer.last_name}}", NHtml::EscapeAttributeValue(user.GetLastName()));
        SubstGlobal(agreementTemplate, "{{rental.customer.address}}", NHtml::EscapeAttributeValue(user.GetAddress()));
        SubstGlobal(agreementTemplate, "{{rental.customer.phone}}", NHtml::EscapeAttributeValue(user.GetPhone()));
        SubstGlobal(agreementTemplate, "{{rental.customer.email}}", NHtml::EscapeAttributeValue(user.GetEmail()));
        SubstGlobal(agreementTemplate, "{{rental.customer.passport_number}}", NHtml::EscapeAttributeValue(passportData.GetNumber()));
        SubstGlobal(agreementTemplate, "{{rental.customer.license_number}}", NHtml::EscapeAttributeValue(drivingLicense.GetNumber()));
    }

    // Car
    if (modelPtr) {
        SubstGlobal(agreementTemplate, "{{rental.car.model}}", NHtml::EscapeAttributeValue(modelPtr->GetName()));
    }
    if (car) {
        SubstGlobal(agreementTemplate, "{{rental.car.number}}", NHtml::EscapeAttributeValue(car->GetNumber()));
        SubstGlobal(agreementTemplate, "{{rental.car.vin}}", NHtml::EscapeAttributeValue(car->GetVin()));
    }

    // Currency and units
    TMaybe<TString> currencyUnit, currencyPerKmUnit;
    {
        if (Currency) {
            currencyUnit = localization->GetLocalString(locale,  MakeUnitValueId(*Currency));
            currencyPerKmUnit = localization->GetLocalString(locale, MakeUnitPermKmId(*Currency));
        }
        FillOptionalParameter(currencyKey, currencyUnit, agreementTemplate);
        FillOptionalParameter("{{rental.currency_per_km}}", currencyPerKmUnit, agreementTemplate);

        FillOptionalParameter("{{rental.km_per_day}}", localization->GetLocalString(locale,  unitKmPerDay), agreementTemplate);
    }

    // Options
    {
        // search pattern:
        // {{#loop rental.options}}
        // something
        // {{/loop rental.options}}
        std::regex rgx(R"(\{\{#loop rental.options\}\}[^\t]*\{\{/loop rental.options\}\})");
        std::smatch match;
        if (std::regex_search(agreementTemplate.c_str(), match, rgx)) {
            for (auto matchIt = match.begin(); matchIt != match.end(); ++matchIt) {
                auto optionTemplate = matchIt->str();
                constexpr auto lengthTag = 24;
                optionTemplate = optionTemplate.substr(lengthTag, optionTemplate.size() - 2 * lengthTag);

                TString optionsHtml;
                for (const auto& option: Options) {
                    if (option.HasStringConstants()) {
                        optionsHtml += FillOption(
                                            optionTemplate,
                                            option.OptionalValue(),
                                            localization->GetLocalString(locale, option.GetStringConstantsRef().GetTitle()),
                                            option.OptionalCost(),
                                            currencyUnit
                                        );
                    }
                }

                TString filledOptions = matchIt->str();
                SubstGlobal(filledOptions, optionTemplate, optionsHtml);
                SubstGlobal(agreementTemplate, matchIt->str(), filledOptions);
            }
        }
    }

    // Insurance
    if (!Insurance.Empty()) {
        TString titleId;
        if (Insurance->GetTitle().empty()) {
            titleId = "rental.insurance_type." + Insurance->GetId() + ".title";
        } else {
            titleId = Insurance->GetTitle();
        }
        const auto title = localization ? localization->GetLocalString(locale, titleId) : titleId;
        FillOptionalParameter("{{rental.insurance_type}}", title, agreementTemplate);
        TString insuranceCost{ "" };
        if (Insurance->HasCost()) {
            insuranceCost = ToString(Insurance->GetCostRef());
        }
        FillOptionalParameter("{{rental.insurance_cost}}", insuranceCost, agreementTemplate);
    }

    // Picking up
    {
        FillTimeParameter("{{rental.delivery.date}}", Since, agreementDateFormat, agreementTemplate, OfferTimezone);
        FillTimeParameter("{{rental.delivery.time}}", Since, agreementTimeFormat, agreementTemplate, OfferTimezone);
        FillOptionalParameter("{{rental.delivery.address}}", DeliveryLocationName, agreementTemplate);
    }

    // Returning back
    {
        const auto duration = std::ceil((Until - Since).Hours() / 24.);
        SubstGlobal(agreementTemplate, "{{rental.return.duration}}", ToString(duration));
        FillTimeParameter("{{rental.return.date}}", Until, agreementDateFormat, agreementTemplate, OfferTimezone);
        FillTimeParameter("{{rental.return.time}}", Until, agreementTimeFormat, agreementTemplate, OfferTimezone);
        if (HasReturnLocations() && ReturnLocations->size() == 1) {
            SubstGlobal(agreementTemplate, "{{rental.return.address}}", NHtml::EscapeAttributeValue(ReturnLocations->begin()->GetLocationName()));

        } else {
            // search pattern:
            // {{#loop rental.returns}}
            // something
            // {{/loop rental.returns}}
            std::regex rgx(R"(\{\{#loop rental.returns\}\}[^\t]*\{\{/loop rental.returns\}\})");
            std::smatch match;
            if (std::regex_search(agreementTemplate.c_str(), match, rgx)) {
                for (const auto & matchIt : match) {
                    auto optionTemplate = matchIt.str();
                    constexpr auto lengthTag = 24;
                    optionTemplate = optionTemplate.substr(lengthTag, optionTemplate.size() - 2 * lengthTag);
                    if (!ReturnLocations) {
                        session.SetErrorInfo("TRentalOffer::FillAgreementTemplate", "cannot got return locations");
                        return false;
                    }
                    TString optionsHtml;
                    const TString optionKeyName = "{{rental.return.address}}";
                    for (const auto& returnLocation : *ReturnLocations) {
                        auto returnAddress = optionTemplate;
                        SubstGlobal(returnAddress, optionKeyName, NHtml::EscapeAttributeValue(returnLocation.GetLocationName()));
                        optionsHtml += returnAddress;
                    }

                    TString filledOptions = matchIt.str();
                    SubstGlobal(filledOptions, optionTemplate, optionsHtml);
                    SubstGlobal(agreementTemplate, matchIt.str(), filledOptions);
                }
            }
        }
    }

    // Petrol
    {
        FillOptionalParameter("{{rental.petrol.tank_fullness}}", fuelLevel, agreementTemplate);
    }

    // Mileage&limit
    {
        FillOptionalParameter("{{rental.mileage_and_limit.mileage}}", mileage, agreementTemplate);
        FillOptionalParameter("{{rental.mileage_and_limit.limit_km_per_day}}", LimitKmPerDay, agreementTemplate);
        FillOptionalParameter("{{rental.mileage_and_limit.overrun_cost_per_km}}", OverrunCostPerKm, agreementTemplate);
    }

    // Payment
    {
        FillOptionalParameter("{{rental.payment.total}}", TotalPayment, agreementTemplate);
        FillOptionalParameter("{{rental.payment.deposit}}", Deposit, agreementTemplate);
        FillOptionalParameter("{{rental.payment.total_daily_cost}}", TotalDailyCost, agreementTemplate);
    }

    // Signature
    {
        auto acceptanceStarted = compiledSession ? compiledSession->OptionalAcceptanceStarted() : Nothing();
        auto acceptanceFinished = compiledSession ? compiledSession->OptionalAcceptanceFinished() : Nothing();
        if (acceptanceStarted && acceptanceFinished) {
            FillTimeParameter("{{rental.signature.date}}", *acceptanceStarted, agreementDateFormat, agreementTemplate, OfferTimezone);
        } else {
            SubstGlobal(agreementTemplate, "{{rental.signature.date}}", "");
        }
    }

    // Images
    {
        const auto& imagesDB = driveApi->GetImagesDB();
        auto optionalImages = imagesDB.RestoreValidatedImages(GetObjectId(), session);
        if (!optionalImages) {
            session.SetErrorInfo("TRentalOffer::FillAgreementTemplate", "cannot fetch images");
            return false;
        }
        const auto hostBySourceMapping = driveApi->HasMDSClient()
            ? TCommonImageData::GetHostByImageSourceMapping(driveApi->GetMDSClient())
            : TMap<TString, TString>();

        struct DamagesInfo {
            TString Url;
            TString Title;
            TString Level;
        };
        auto& images = optionalImages->GetImages();
        TVector<DamagesInfo> damages;
        for (auto&& image : images) {
            TMaybe<TString> url, title, level;
            auto needleHost = hostBySourceMapping.Value(image.GetSource(), hostBySourceMapping.Value("default", ""));
            if (needleHost) {
                auto normalizedHost = needleHost.EndsWith("/") ? needleHost : needleHost + "/";
                url = normalizedHost + image.GetPath();
            }

            bool firstDescriptionFound = false;
            for (auto&& markUp : image.GetMarkUpList()) {
                if (!firstDescriptionFound) {
                    title = TStringBuilder() << "car.damage.element." << markUp.GetElement() << ".title";
                    level = TStringBuilder() << "car.damage.level." << markUp.GetLevel() << ".title";
                    firstDescriptionFound = true;
                }
                if (markUp.IsIsTheBest()) {
                    title = TStringBuilder() << "car.damage.element." << markUp.GetElement() << ".title";
                    level = TStringBuilder() << "car.damage.level." << markUp.GetLevel() << ".title";
                    break;
                }
            }
            if (url) {
                damages.push_back({ *url, title ? *title : "", level ? *level: "" });
            }
        }

        // search pattern:
        // {{#loop rental.car.damages}}
        // something
        // {{/loop rental.car.damages}}
        std::regex rgx(R"(\{\{#loop rental.car.damages\}\}[^\t]*\{\{/loop rental.car.damages\}\})");
        std::smatch match;
        if (std::regex_search(agreementTemplate.c_str(), match, rgx)) {
            for (auto matchIt = match.begin(); matchIt != match.end(); ++matchIt) {
                auto damageTemplate = matchIt->str();
                constexpr auto lengthTag = 28;
                damageTemplate = damageTemplate.substr(lengthTag, damageTemplate.size() - 2 * lengthTag);

                TString damagesHtml;
                const TString urlKeyName = "{{rental.car.damage.url}}";
                const TString titleKeyName = "{{rental.car.damage.title}}";
                const TString levelKeyName = "{{rental.car.damage.level}}";
                for (const auto& damage: damages) {
                    auto filledDamage = damageTemplate;
                    SubstGlobal(filledDamage, urlKeyName, NHtml::EscapeAttributeValue(damage.Url));
                    SubstGlobal(filledDamage, titleKeyName, NHtml::EscapeAttributeValue(localization->GetLocalString(locale, damage.Title)));
                    SubstGlobal(filledDamage, levelKeyName, NHtml::EscapeAttributeValue(localization->GetLocalString(locale, damage.Level)));
                    damagesHtml += filledDamage;
                }

                TString filledDamages = matchIt->str();
                SubstGlobal(filledDamages, damageTemplate, damagesHtml);
                SubstGlobal(agreementTemplate, matchIt->str(), filledDamages);
            }
        }
    }

    // Any custom data
    {
        for (const auto& [key, value]:  *RenterOfferCommonInfo) {
             SubstGlobal(agreementTemplate, key, value);
        }
    }
    return true;
}

std::pair<TString, TString> TRentalOffer::GetDateTimeReport(TInstant date) const {
    const auto dateFormat = OptionalDateFormat().GetOrElse(TRentalOffer::DefaultDateFormat);
    const auto timeFormat = OptionalTimeFormat().GetOrElse(TRentalOffer::DefaultTimeFormat);
    return { TRentalOffer::GetTimeParameter(date, dateFormat, OfferTimezone),
             TRentalOffer::GetTimeParameter(date, timeFormat, OfferTimezone) };
}

NJson::TJsonValue TRentalOffer::GetInsuranceReport(
    ELocalization locale,
    const NDrive::IServer& server,
    const TRentalInsurances& insuranceTypes,
    const TString& unitValueId
) const {
    NJson::TJsonValue result;
    result["id"] = "insurance";
    if (!Insurance.Empty()) {
        if (auto insuranceTypeIt = std::find_if(insuranceTypes.begin(), insuranceTypes.end(),
                                                [this](const TRentalInsurance& insuranceType)->bool {
                                                                return insuranceType.GetId() == Insurance->GetId();
                                                            });
                                                            insuranceTypeIt != insuranceTypes.end()) {
            auto localization = server.GetLocalization();
            const auto titleId = "rental.insurance_type.title";
            result["title"] = localization ? localization->GetLocalString(locale, titleId) : titleId;
            TString value, cost;
            if (insuranceTypeIt->GetTitle().empty()) {
                const auto valueId = "rental.insurance_type." + insuranceTypeIt->GetId() + ".title";
                value = localization ? localization->GetLocalString(locale, valueId) : valueId;
            } else {
                value = localization ? localization->GetLocalString(locale, insuranceTypeIt->GetTitle()) : insuranceTypeIt->GetTitle();
            }
            if (Insurance->HasCost()) {
                auto localizedUnit = localization ? localization->GetLocalString(locale, unitValueId) : unitValueId;
                cost = ", " +  ToString(*Insurance->OptionalCost()) + " " + localizedUnit;
            }
            result["value"] = value + cost;
        }
    }
    return result;
}

void TRentalOffer::FillTimeParameter(TString&& key, TInstant value, const TString& format, TString& text, const TMaybe<TString>& OfferTimezone) {
    if (OfferTimezone) {
        SubstGlobal(text, key,
            NHtml::EscapeAttributeValue(NUtil::FormatDatetime(NUtil::ConvertTimeZone(value, NUtil::GetUTCTimeZone(), NUtil::GetTimeZone(*OfferTimezone)), format)));
    } else {
        SubstGlobal(text, key, NHtml::EscapeAttributeValue(NUtil::FormatDatetime(value, format)));
    }
}

void TRentalOffer::FillTimeParameter(const TString& key, TInstant value, const TString& format, TString& text, const TMaybe<TString>& OfferTimezone) {
   if (OfferTimezone) {
       SubstGlobal(text, key,
           NHtml::EscapeAttributeValue(NUtil::FormatDatetime(NUtil::ConvertTimeZone(value, NUtil::GetUTCTimeZone(), NUtil::GetTimeZone(*OfferTimezone)), format)));
   } else {
       SubstGlobal(text, key,  NHtml::EscapeAttributeValue(NUtil::FormatDatetime(value, format)));
   }
}

TMaybe<IOffer::TEmailOfferData> TRentalOffer::GetEmailData(const NDrive::IServer* server, NDrive::TEntitySession& session, const TString& notificationType) const {
    Y_ENSURE(server);
    auto driveApi = server->GetDriveAPI();
    Y_ENSURE(driveApi);

    TEmailOfferData emailOfferData;
    auto affiliationTagDescription = NDrivematics::TUserOrganizationAffiliationTag::GetAffiliatedCompanyTagDescription(GetUserId(), *server, session);
    bool needRideData = false;
    if (notificationType == TRentalOffer::AcceptanceFinishedNotificationType) {
        emailOfferData.EmailTagName = affiliationTagDescription->GetAfterAcceptanceEmailTagName();
        needRideData = true;
    } else if (notificationType == TRentalOffer::RideFinishedNotificationType) {
        emailOfferData.EmailTagName = affiliationTagDescription->GetAfterFinishRideEmailTagName();
        needRideData = true;
    } else if (notificationType == TRentalOffer::BookingConfirmedNotificationType) {
        emailOfferData.EmailTagName = affiliationTagDescription->GetBookingConfirmedEmailTagName();
    } else if (notificationType == TRentalOffer::BookingCancelledNotificationType) {
        emailOfferData.EmailTagName = affiliationTagDescription->GetBookingCancelledEmailTagName();
    }

    if (emailOfferData.EmailTagName.Empty()) {
        return emailOfferData;
    }

    NJson::TJsonValue jData;
    auto& emailArgs = emailOfferData.Args;
    if (needRideData) {
        TMaybe<TInstant> actualSince, actualUntil;
        TMaybe<double> mileage, mileageStart, mileageFinish, fuelLevelStart, fuelLevelFinish;
        TString locationStartStr, locationFinishStr;
        TMap<ui64, TString> imagesBeforeRide, imagesAfterRide;
        TMaybe<TDuration> duration;
        TMaybe<NDrive::TSimpleLocation> locationStart, locationFinish;
        const auto locale = ELocalization::Eng;
        auto localization = server->GetLocalization();
        const auto& offerId = GetOfferId();

        auto objectSession = server->GetDriveDatabase().GetSessionManager().GetSession(offerId, session);
        if (!objectSession) {
            session.SetErrorInfo(getEmailDataSourceSession, "objectSession is nullptr");
            return Nothing();
        }
        auto billingSession = std::dynamic_pointer_cast<const TBillingSession>(*objectSession);
        if (!billingSession) {
            session.SetErrorInfo(getEmailDataSourceSession, "billingSession is nullptr");
            return Nothing();
        }
        TBillingSession::TBillingEventsCompilation eventsCompilation;
        if (!billingSession->FillCompilation(eventsCompilation)) {
            session.SetErrorInfo(getEmailDataSourceSession, "cannot fill BillingEventsCompilation");
            return Nothing();
        }
        const auto [acceptanceStartedSnapshot, acceptanceFinishedSnapshot] = eventsCompilation.GetAcceptanceSnapshots();

        auto fullCompiledRiding = billingSession->BuildCompiledRiding(server, nullptr);
        if (!fullCompiledRiding) {
            session.SetErrorInfo(getEmailDataSourceSession, "cannot BuildCompiledRiding");
            return {};
        }

        actualSince = acceptanceStartedSnapshot.OptionalDate();
        if (actualSince) {
            actualUntil = TInstant::Now();
            duration = *actualUntil - *actualSince;
        } else {
            session.SetErrorInfo(getEmailDataSourceSession, "cannot get AcceptanceStarted");
            return Nothing();
        }
        TInstant acceptanceFinished = acceptanceFinishedSnapshot.OptionalDate().GetOrElse(TInstant::Now());

        auto optSnapshotDiff = fullCompiledRiding->OptionalSnapshotsDiff();
        if (optSnapshotDiff) {
            mileage = optSnapshotDiff->OptionalMileage();
            mileageStart = optSnapshotDiff->OptionalStartMileage();
            fuelLevelStart = optSnapshotDiff->OptionalStartFuelLevel();
            locationStart = optSnapshotDiff->OptionalStart();
            mileageFinish = optSnapshotDiff->OptionalLastMileage();
            fuelLevelFinish = optSnapshotDiff->OptionalLastFuelLevel();
            locationFinish = optSnapshotDiff->OptionalLast();
        }

        const auto& imagesDB = driveApi->GetImagesDB();
        NSQL::TQueryOptions options;
        options.SetGenericCondition("session_id", TSet<TString>{offerId});
        auto optionalImages = imagesDB.Fetch(session, options);
        if (!optionalImages) {
            session.SetErrorInfo(getEmailDataSourceSession, "cannot Get images");
            return Nothing();
        }

        auto hostByImageSourceMapping = driveApi->HasMDSClient()
        ? TCommonImageData::GetHostByImageSourceMapping(driveApi->GetMDSClient())
        : TMap<TString, TString>();
        for (auto&& image : *optionalImages) {
            if (offerId == image.GetSessionId()) {
                if (image.GetCreatedAt() <= acceptanceFinished) {
                    imagesBeforeRide[image.GetImageId()] = image.BuildUrl(hostByImageSourceMapping, image.GetPath());
                } else {
                    imagesAfterRide[image.GetImageId()] = image.BuildUrl(hostByImageSourceMapping, image.GetPath());
                }
            }
        }

        if (locationStart || locationFinish) {
            TDuration timeout = server->GetSettings().GetValue<TDuration>("tags.geocoder_timeout_for_user_mail_tag").GetOrElse(TDuration::Zero());
            auto geocoder = server->GetDriveAPI()->HasGeocoderClient() ? &server->GetDriveAPI()->GetGeocoderClient() : nullptr;
            if (geocoder) {
                auto language = enum_cast<ELanguage>(locale);

                if (locationStart) {
                    auto startLocation = geocoder->Decode(locationStart->GetCoord(), language);
                    if (startLocation.Wait(timeout)) {
                        locationStartStr = startLocation.GetValue().Title;
                    }
                }

                if (locationFinish) {
                    auto finishLocation = geocoder->Decode(locationFinish->GetCoord(), language);
                    if (finishLocation.Wait(timeout)) {
                        locationFinishStr = finishLocation.GetValue().Title;
                    }
                }
            }
        }

        {
            auto [date, time] = GetDateTimeReport(*actualSince);
            emailArgs[actualSinceDateKeyword] = date;
            emailArgs[actualSinceTimeKeyword] = time;
        }
        {
            auto [date, time] = GetDateTimeReport(*actualUntil);
            emailArgs[actualUntilDateKeyword] = date;
            emailArgs[actualUntilTimeKeyword] = time;
        }

        emailArgs[mileageKeyword] = !mileage.Empty() ? ToString(mileage.GetRef()) : "";
        emailArgs[startMileageKeyword] = !mileageStart.Empty() ? ToString(mileageStart.GetRef()) : "";
        emailArgs[finishMileageKeyword] = !mileageFinish.Empty() ? ToString(mileageFinish.GetRef()) : "";
        emailArgs[startFuelLevelKeyword] = !fuelLevelStart.Empty() ? ToString(fuelLevelStart.GetRef()) : "";
        emailArgs[finishFuelLevelKeyword] = !fuelLevelFinish.Empty() ? ToString(fuelLevelFinish.GetRef()) : "";
        emailArgs[locationStartKeyword] = locationStartStr;
        emailArgs[locationFinishKeyword] = locationFinishStr;
        emailArgs[mileageKeyword] = !mileage.Empty() ? ToString(mileage.GetRef()) : "";
        emailArgs[durationKeyword] = !duration.Empty() ? GetDurationReport(duration.GetRef(), locale, localization) : "";

        jData[imagesBeforeRideKeyword] = NJson::RangeToJson(NContainer::Values(imagesBeforeRide));
        jData[imagesAfterRideKeyword] = NJson::RangeToJson(NContainer::Values(imagesAfterRide));
    }

    const auto dateFormat = OptionalDateFormat().GetOrElse(TRentalOffer::DefaultDateFormat);
    const auto timeFormat = OptionalTimeFormat().GetOrElse(TRentalOffer::DefaultTimeFormat);
    emailArgs[SinceDateKeyword] = TRentalOffer::GetTimeParameter(GetSince(), dateFormat, OptionalOfferTimezone());
    emailArgs[SinceTimeKeyword] = TRentalOffer::GetTimeParameter(GetSince(), timeFormat, OptionalOfferTimezone());

    emailArgs[UntilDateKeyword] = TRentalOffer::GetTimeParameter(GetUntil(), dateFormat, OptionalOfferTimezone());
    emailArgs[UntilTimeKeyword] = TRentalOffer::GetTimeParameter(GetUntil(), timeFormat, OptionalOfferTimezone());
    emailArgs[PickupLocationKeyword] = OptionalDeliveryLocationName().GetOrElse("");
    emailArgs[depositKeyword] = HasDeposit() ?  ToString(GetDepositRef()) : "";
    emailArgs[totalPaymentKeyword] = HasTotalPayment() ?  ToString(GetTotalPaymentRef()) : "";

    const auto usersInfo = driveApi->GetUsersData()->FetchInfo(GetUserId(), session);
    if (usersInfo.size() != 1) {
        session.SetErrorInfo(getEmailDataSourceSession, "wrong usersInfo.size()");
        return Nothing();
    }

    const auto localization = server->GetLocalization();
    const auto locale = ELocalization::Eng;

    auto& user = usersInfo.begin()->second;
    emailArgs[firstNametKeyword] = user.GetFirstName();
    emailArgs[lastNameKeyword] = user.GetLastName();
    emailArgs[clientPhoneKeyword] = user.GetPhone();

    if (!Insurance.Empty()) {
        if (Insurance->GetTitle().empty()) {
            const auto titleId = "rental.insurance_type." + Insurance->GetId() + ".title";
            emailArgs[insuranceTypeKeyword] = localization ? localization->GetLocalString(locale, titleId) : titleId;
        } else {
            emailArgs[insuranceTypeKeyword] = localization ? localization->GetLocalString(locale, Insurance->GetTitle()) : Insurance->GetTitle();
        }
        emailArgs[insuranceCostKeyword] = Insurance->HasCost() ? ToString(Insurance->GetCostRef()) : "";
    } else {
        emailArgs[insuranceTypeKeyword] = "";
        emailArgs[insuranceCostKeyword] = "";
    }
    emailArgs[currencyKeyword] = HasCurrency() ? localization->GetLocalString(locale,  TRentalOffer::MakeUnitValueId(GetCurrencyRef())) : "";

    NJson::TJsonArray jOptions;
    for (auto& option: Options) {
        if (option.HasValue() && option.GetValueRef() && option.HasStringConstants()) {
            NJson::TJsonValue jOption;
            jOption["title"] = localization->GetLocalString(locale, option.GetStringConstantsRef().GetTitle());
            jOption["cost"] = option.HasCost() ? ToString(option.GetCostRef()) : "";
            jOptions.AppendValue(jOption);
        }
    }

    jData[optionsKeyword] = jOptions;

    emailArgs[CompanynameKeyword] = affiliationTagDescription->GetCompanyName();
    emailArgs[companyPhoneKeyword] = affiliationTagDescription->GetPhone();
    emailArgs[companyEmailKeyword] = affiliationTagDescription->GetEmail();
    const auto& companyInfo = affiliationTagDescription->GetCompanyInformation();
    for (const auto& [key, value]: companyInfo) {
        emailArgs[key] = value;
    }

    const auto carsFetchResult = driveApi->GetCarsData()->FetchInfo(GetObjectId(), session);
    if (!carsFetchResult) {
        session.SetErrorInfo(getEmailDataSourceSession, "cannot fetch car");
        return Nothing();
    }

    const auto carsInfo = carsFetchResult.GetResult();
    if (carsInfo.size() != 1) {
        session.SetErrorInfo(getEmailDataSourceSession, "wrong carsInfo.size()");
        return Nothing();
    }

    const auto& car = carsInfo.begin()->second;
    auto modelFetchResult = driveApi->GetModelsData()->GetCachedOrFetch(car.GetModel());
    if (!modelFetchResult) {
        session.SetErrorInfo(getEmailDataSourceSession, "cannot fetch model");
        return Nothing();
    }

    auto modelPtr = modelFetchResult.GetResultPtr(car.GetModel());
    if (!modelPtr) {
        session.SetErrorInfo(getEmailDataSourceSession, "cannot find modelPtr");
        return Nothing();
    }
    emailArgs[CarModelKeyword] = modelPtr->GetName();

    emailArgs[totalDailyCostKeyword] = HasTotalDailyCost() ?  ToString(GetTotalDailyCostRef()) : "";
    emailArgs[overrunCostPerKmKeyword] = HasOverrunCostPerKm() ? ToString(GetOverrunCostPerKmRef()) : "";
    emailArgs[limitKmPerDayKeyword] = HasLimitKmPerDay() ? ToString(GetLimitKmPerDayRef()) : "";

    emailOfferData.JsonData = jData;

    return emailOfferData;
}

TString TRentalOffer::GetDurationReport(TDuration totalDuration, ELocalization locale, const ILocalization *localization) const {
    TString durationReport;
    if (totalDuration >= TDuration::Days(1)) {
        durationReport = localization ? localization->GetLocalString(locale, billDurationGreaterDay) : billDurationGreaterDay;
    } else {
        durationReport = localization ? localization->GetLocalString(locale, billDurationLessDay) : billDurationLessDay;
    }

    auto durationInstant = TInstant::FromValue(totalDuration.GetValue());
    SubstGlobal(durationReport, "_Days_", ToString(durationInstant.Days()));
    durationInstant -= TDuration::Days(durationInstant.Days());
    SubstGlobal(durationReport, "_Hours_", ToString(durationInstant.Hours()));
    durationInstant -= TDuration::Hours(durationInstant.Hours());
    SubstGlobal(durationReport, "_Minutes_", ToString(durationInstant.Minutes()));
    return durationReport;
}

TString TRentalOffer::GetTimeParameter(TInstant value, const TString& format, const TMaybe<TString>& OfferTimezone) {
    if (OfferTimezone) {
        return NHtml::EscapeAttributeValue(NUtil::FormatDatetime(NUtil::ConvertTimeZone(value, NUtil::GetUTCTimeZone(), NUtil::GetTimeZone(*OfferTimezone)), format));
    }
    return NHtml::EscapeAttributeValue(NUtil::FormatDatetime(value, format));
}


TString TRentalOffer::MakeUnitValueId(const TString& Id) {
    return "unit.shorts." + Id + ".value";
}

TString TRentalOffer::MakeUnitPermKmId(const TString& Id) {
    return "unit.shorts." + Id + ".per_km";
}

NJson::TJsonValue TRentalOfferLocation::GetReport(ELocalization locale, const NDrive::IServer& server) const {
    auto localization = server.GetLocalization();
    NJson::TJsonValue result = NJson::ToJson(this);
    result["location_name"] = localization ? localization->GetLocalString(locale, LocationName) : LocationName;
    return result;
}

NJson::TJsonValue TRentalOfferCurrency::GetReport(ELocalization locale, const NDrive::IServer& server) const {
    NJson::TJsonValue result;
    auto localization = server.GetLocalization();
    const auto currencyLocalizationId = TRentalOffer::MakeUnitValueId(Id);
    result["id"] = Id;
    result["name"] = localization ? localization->GetLocalString(locale, currencyLocalizationId) : currencyLocalizationId;
    return result;
}

NJson::TJsonValue TRentalInsurance::GetReport(ELocalization locale, const NDrive::IServer& server) const {
    NJson::TJsonValue result;
    auto localization = server.GetLocalization();
    result["id"] = Id;
    TString titleId;
    if (Title.Empty()) {
        titleId = "rental.insurance_type." + Id + ".title";
        result["title"] = localization ? localization->GetLocalString(locale, titleId) : titleId;
    } else {
        result["title"] = localization ? localization->GetLocalString(locale, Title) : Title;
    }

    return result;
}

NJson::TJsonValue TRentalOfferStatus::GetReport(ELocalization locale, const NDrive::IServer& server) const {
    NJson::TJsonValue result;
    auto localization = server.GetLocalization();
    result["id"] = Id;
    const auto titleId = "rental.status." + Id + ".title";
    result["title"] = localization ? localization->GetLocalString(locale, titleId) : titleId;
    return result;
}

bool TRentalOfferBuilder::CheckDeviceTagsPerformer(const TTaggedObject&, const TUserPermissions&, bool) const {
    return true;
}

void TRentalOfferBuilder::PatchReport(
    NJson::TJsonValue& report, ELocalization locale,
    const NDrive::IServer& server, NDrive::TEntitySession& tx,
    const TUserPermissions::TPtr permissions
) const {
    auto tariffCars = Economy.CalculateTariffCars(server, tx, permissions);
    auto localization = server.GetLocalization();
    Economy.PatchReport(report,  locale, localization, tariffCars);
    auto& jActionMeta =  report["action_meta"];
    NJson::TJsonArray currencies;
    for (const auto& currency: GetAllowedCurrencies()) {
        currencies.AppendValue(currency.GetReport(locale, server));
    }
    jActionMeta["currencies"] = std::move(currencies);

    NJson::TJsonArray jOptions;
    TMultiMap<int, NJson::TJsonValue> sortedOptions;
    for (const auto& option: GetOptions()) {
        if (option.HasConstants() && option.HasStringConstants()) {
            NJson::TJsonValue jOption;
            jOption["id"] = option.GetConstantsRef().GetId();
            jOption["title"] = localization ? localization->GetLocalString(locale, option.GetStringConstantsRef().GetTitle()) : option.GetStringConstantsRef().GetTitle();
            jOption["subtitle"] = localization ? localization->GetLocalString(locale, option.GetStringConstantsRef().GetSubtitle()) : option.GetStringConstantsRef().GetSubtitle();
            sortedOptions.insert({option.GetConstantsRef().GetOrder(), std::move(jOption)});
        }
    }
    for (auto& [order, jOption]: sortedOptions) {
        jOptions.AppendValue(std::move(jOption));
    }
    jActionMeta["economy"]["options_list"] = std::move(jOptions);

    NJson::TJsonArray jInsurances;
    for (const auto& insurance: AllowedInsurances) {
        jInsurances.AppendValue(insurance.GetReport(locale, server));
    }
    jActionMeta["economy"]["insurances_list"] = std::move(jInsurances);
}

EOfferCorrectorResult TRentalOfferBuilder::DoBuildOffers(
    const TUserPermissions& permissions,
    TVector<IOfferReport::TPtr>& offers,
    const TOffersBuildingContext& context,
    const NDrive::IServer* server,
    NDrive::TInfoEntitySession& session
) const {
    Y_UNUSED(permissions);
    Y_UNUSED(session);
    if (!server) {
        return EOfferCorrectorResult::Unimplemented;
    }

    auto variables = context.GetVariables();

    auto offer = MakeAtomicShared<TRentalOffer>();

    const TMaybe<TInstant> since = NJson::FromJson<TMaybe<TInstant>>(variables["since"]);
    const TMaybe<TInstant> until = NJson::FromJson<TMaybe<TInstant>>(variables["until"]);

    if (context.HasCarId()) {
        offer->SetObjectId(context.GetCarIdRef());
    }

    offer->SetUserId(context.GetUserHistoryContextUnsafe().GetUserId());

    auto fieldsChecker = [&variables](const TString& keyword) {
        return variables.contains(keyword);
    };

    {
        for (const auto& option: Options) {
            if (option.HasConstants() && GetEconomy().IsOptionAllowed(option.GetConstantsRef().GetId())) {
                offer->MutableOptions().push_back(option);
            }
        }
    }
    // legacy options
    // will be deleted after frontend update
    {
        if (GetAllowChildSeat() && GetEconomy().IsOptionAllowed(NLegacyOptions::NId::childSeat)) {
            offer->AddOptionFromLegacy(false, NLegacyOptions::NId::childSeat, NLegacyOptions::NOrder::childSeat);
        }

        if (GetAllowRoofRack() && GetEconomy().IsOptionAllowed(NLegacyOptions::NId::roofRack)) {
            offer->AddOptionFromLegacy(false, NLegacyOptions::NId::roofRack, NLegacyOptions::NOrder::roofRack);
        }

        if (GetAllowGPS() && GetEconomy().IsOptionAllowed(NLegacyOptions::NId::gps)) {
            offer->AddOptionFromLegacy(false, NLegacyOptions::NId::gps, NLegacyOptions::NOrder::gps);
        }

        if (GetAllowSnowChains() && GetEconomy().IsOptionAllowed(NLegacyOptions::NId::snowChains)) {
            offer->AddOptionFromLegacy(false, NLegacyOptions::NId::snowChains, NLegacyOptions::NOrder::snowChains);
        }

        if (GetAllowEntryToEcoZonesInGermany() && GetEconomy().IsOptionAllowed(NLegacyOptions::NId::entryToEcoZonesInGermany)) {
            offer->AddOptionFromLegacy(false, NLegacyOptions::NId::entryToEcoZonesInGermany, NLegacyOptions::NOrder::entryToEcoZonesInGermany);
        }
    }

    PatchOfferFromJson(variables, offer, fieldsChecker, *this);

    if (auto result = SetOfferTimeInfo(*offer, variables, context.GetRequestStartTime()); EOfferCorrectorResult::Success != result) {
        session.SetErrorInfo("RentalOfferBuilder::DoBuildOffers", wrongSinceUntil, NDrive::MakeError(wrongSinceUntil));
        return result;
    }

    offer->SetTargetHolderTag(OfferHolderTag);
    offer->SetOfferImage(OfferImage);
    offer->SetOfferSmallImage(OfferSmallImage);
    offer->SetOfferImageStyle(OfferImageStyle);
    offer->SetRenterOfferCommonInfo(RenterOfferCommonInfo);
    offer->SetOfferTimezone(OfferTimezone);
    offer->SetDateFormat(DateFormat);
    offer->SetTimeFormat(TimeFormat);
    offer->SetAgreement(Agreement);

    std::sort(offer->MutableOptions().begin(), offer->MutableOptions().end(), [](const TRentalOffer::TOption& lhs, const TRentalOffer::TOption& rhs) {
        return !lhs.HasConstants() || !rhs.HasConstants() || lhs.GetConstantsRef().GetOrder() < rhs.GetConstantsRef().GetOrder();
    });

    if (since && until && context.HasCarId()) {
        bool sessionErrorHappend = false;
        if (auto recommendation = Economy.GetRecommendedTariff(context.GetCarIdRef(), *until - *since, server, session, sessionErrorHappend); recommendation) {
            offer->SetTariffRecommendation(recommendation);
            if (recommendation->HasName()) {
                offer->SetInsurancesRecommendation(GetEconomy().GetRecommendedInsurances(recommendation->GetNameRef()));
                offer->SetLimitsRecommendation(GetEconomy().GetRecommendedLimits(recommendation->GetNameRef()));
            }
        } else {
            if (sessionErrorHappend) {
                return EOfferCorrectorResult::BuildingProblems;
            }
        }
    }

    if (auto recommendation = Economy.GetRecommendedOptions(since, until, offer->GetOptions()); recommendation) {
        offer->SetOptionsRecommendation(recommendation);
    }

    offer->CalculateMileageLimit();
    offer->SetTargetHolderTag(OfferHolderTag);
    offer->SetStartRentalTimeout(StartRentalTimeout);

    auto report = MakeAtomicShared<TRentalOfferReport>(offer, nullptr);
    offers.push_back(report);
    return EOfferCorrectorResult::Success;
}

EOfferCorrectorResult TRentalOfferBuilder::DoCheckOfferConditions(
    const TOffersBuildingContext& context,
    const TUserPermissions& permissions
) const {
    Y_UNUSED(context);
    Y_UNUSED(permissions);
    return EOfferCorrectorResult::Success;
}

NDrive::TScheme TRentalOfferBuilder::DoGetScheme(const NDrive::IServer* server) const {
    Y_UNUSED(server);
    NDrive::TScheme result = TBase::DoGetScheme(server);
    result.Add<TFSBoolean>("child_seat", "Allow child seat option");
    result.Add<TFSBoolean>("roof_rack", "Allow roof rack option");
    result.Add<TFSBoolean>("gps", "Allow gps option");
    result.Add<TFSBoolean>("snow_chains", "Allow snow chains option");
    result.Add<TFSBoolean>("entry_to_eco_zones_in_germany", "Allow entry to eco zones in germany option");

    {
        NDrive::TScheme locations;
        locations.Add<TFSString>("id", "Identifier");
        locations.Add<TFSString>("location_name", "Location name");
        locations.Add<TFSNumeric>("lon", "Longitude");
        locations.Add<TFSNumeric>("lat", "Latitude");
        result.Add<TFSArray>("delivery_locations", "Delivery locations").SetElement(locations);
        locations.Add<TFSString>("coords", "Coords (split by <space>)");
        result.Add<TFSArray>("return_locations", "Return locations").SetElement(locations);
    }

    {
        NDrive::TScheme insuranceTypes;
        insuranceTypes.Add<TFSString>("id", "Identifier");
        insuranceTypes.Add<TFSString>("title", "Title").SetRequired(false);
        result.Add<TFSArray>("insurance_types", "Insurance types").SetElement(insuranceTypes);
    }

    result.Add<TFSBoolean>("comment", "Allow comment");

    {
        NDrive::TScheme statuses;
        statuses.Add<TFSString>("id", "Identifier");
        result.Add<TFSArray>("statuses", "Statuses").SetElement(statuses);
    }

    result.Add<TFSBoolean>("limit_km_per_day", "Allow limit km per day");
    result.Add<TFSBoolean>("overrun_cost_per_km", "Allow overrun cost");

    result.Add<TFSBoolean>("total_payment", "Allow total payment");
    result.Add<TFSBoolean>("deposit", "Allow deposit");

    {
        NDrive::TScheme currencies;
        currencies.Add<TFSString>("id", "Identifier");
        result.Add<TFSArray>("currencies", "Currencies").SetElement(currencies);
    }

    result.Add<TFSString>("offer_image", "Offer image");
    result.Add<TFSString>("offer_small_image", "Offer small image");
    result.Add<TFSVariants>("offer_image_style", "Offer image style").SetVariants(GetEnumAllValues<ERentalOfferImageStyle>());
    result.Add<TFSJson>("economy", "Economics data");

    NDrive::TScheme optionScheme = TRentalOffer::TOption().GetScheme(server);
    result.Add<TFSArray>(TString("options"), "Доступные опции на этапе оформления заявки").SetElement(std::move(optionScheme)).SetRequired(false);

    {
        auto gTab = result.StartTabGuard("renter_information");
        {
            result.Add<TFSString>("offer_timezone", "Offer timezone");
            result.Add<TFSString>("date_format", "Date format");
            result.Add<TFSString>("time_format", "Time format");
            result.Add<TFSString>("agreement", "Ссылка на акт приемки-передачи");
            NDrive::TScheme& renterOfferCommonInfo = result.Add<TFSArray>("renter_offer_common_info", "Информация для заполнения Rental Agreement", 100000).SetElement<NDrive::TScheme>();
            renterOfferCommonInfo.Add<TFSString>("key", "Ключ текста");
            renterOfferCommonInfo.Add<TFSString>("value", "Текст");
        }
    }
    result.Add<TFSDuration>("start_rental_timeout", "Start rental timeout");
    return result;
}

bool TRentalOfferBuilder::DeserializeSpecialsFromJson(const NJson::TJsonValue& value) {
    bool result = TBase::DeserializeSpecialsFromJson(value) && NJson::TryFieldsFromJson(value, GetFields());
    result &= NJson::ParseField(value["start_rental_timeout"], StartRentalTimeout);
    const NJson::TJsonValue::TArray* renterOfferCommonInfoArray;
    if (value["renter_offer_common_info"].GetArrayPointer(&renterOfferCommonInfoArray)) {
        for (auto&& obj : *renterOfferCommonInfoArray) {
            TString key;
            TString value;
            if (!obj["key"].GetString(&key) || !obj["value"].GetString(&value)) {
                ERROR_LOG << "Incorrect renter_offer_common_info format: " << value << Endl;
                continue;
            }
            RenterOfferCommonInfo[key] = value;
        }
    }
    if (value.Has("economy") && !Economy.DeserializeFromJson(value["economy"])) {
        return false;
    }
    return result;
}

NJson::TJsonValue TRentalOfferBuilder::SerializeSpecialsToJson() const {
    NJson::TJsonValue result = TBase::SerializeSpecialsToJson();
    NJson::FieldsToJson(result, GetFields());
    if (!RenterOfferCommonInfo.empty()) {
        NJson::TJsonValue& jsonValue = result["renter_offer_common_info"];
        for (auto&& obj : RenterOfferCommonInfo) {
            NJson::TJsonValue& local = jsonValue.AppendValue(NJson::JSON_MAP);
            local.InsertValue("key", obj.first);
            local.InsertValue("value", obj.second);
        }
    }
    result["start_rental_timeout"] = NJson::ToJson(NJson::Hr(StartRentalTimeout));
    result["economy"] = Economy.SerializeToJson();
    return result;
}

TRentalInsurances TRentalOfferBuilder::CreateDefaultInsurances() {
    TRentalInsurances result;
    TRentalInsurance insurance;
    insurance.SetId("id_gold");
    result.push_back(insurance);
    insurance.SetId("id_silver");
    result.push_back(insurance);
    insurance.SetId("id_basic");
    result.push_back(insurance);
    return result;
}

TRentalOfferVariable<TString> TRentalOfferBuilder::CreateDefaultComment() {
    TRentalOfferVariable<TString> result;
    result.SetId("comment");
    result.SetTitle("rental.comment.title");
    result.SetSubtitle("rental.comment.subtitle");
    result.SetDefaultValue("");
    return result;
}

TRentalOfferStatuses TRentalOfferBuilder::CreateDefaultStatuses() {
    TRentalOfferStatuses result;
    TRentalOfferStatus status;
    status.SetId("draft");
    result.push_back(status);
    status.SetId("confirmed");
    result.push_back(status);
    status.SetId("paid");
    result.push_back(status);
    return result;
}

TRentalOfferVariable<ui64> TRentalOfferBuilder::CreateDefaultLimitKmPerDay(){
    TRentalOfferVariable<ui64> result;
    result.SetId("limit_km_per_day");
    result.SetTitle("rental.limit_km_per_day.title");
    result.SetSubtitle("rental.limit_km_per_day.subtitle");
    result.SetDefaultValue(0);
    return result;
}

TRentalOfferVariable<ui64> TRentalOfferBuilder::CreateDefaultOverrunCostPerKm() {
    TRentalOfferVariable<ui64> result;
    result.SetId("overrun_cost_per_km");
    result.SetTitle("rental.overrun_cost_per_km.title");
    result.SetSubtitle("rental.overrun_cost_per_km.subtitle");
    result.SetDefaultValue(0);
    return result;
}

TRentalOfferVariable<ui64> TRentalOfferBuilder::CreateDefaultPayment() {
    TRentalOfferVariable<ui64> result;
    result.SetId("total_payment");
    result.SetTitle("rental.total_payment.title");
    result.SetSubtitle("rental.total_payment.subtitle");
    result.SetDefaultValue(0);
    return result;
}

TRentalOfferVariable<ui64> TRentalOfferBuilder::CreateDefaultDeposit() {
    TRentalOfferVariable<ui64> result;
    result.SetId("deposit");
    result.SetTitle("rental.deposit.title");
    result.SetSubtitle("rental.deposit.subtitle");
    result.SetDefaultValue(0);
    return result;
}

EOfferCorrectorResult TRentalOfferBuilder::SetOfferTimeInfo(TRentalOffer& offer,
                                                            TMap<TString, NJson::TJsonValue>& variables,
                                                            const TInstant requestStartTime) const {
    const auto since = NJson::FromJson<TMaybe<TInstant>>(variables["since"]);
    const auto until = NJson::FromJson<TMaybe<TInstant>>(variables["until"]);
    return CheckAndSetSinceUntil(offer, since, until, requestStartTime);
}

void TRentalEconomy::PatchReport(
    NJson::TJsonValue& report,
    ELocalization locale,
    const ILocalization* localization,
    const TMap<TString, ui64>& tariffCars)
const {
    report["action_meta"]["economy"] = SerializeToJson(locale, localization, &tariffCars);
}

bool TRentalEconomy::DeserializeFromJson(const NJson::TJsonValue& value) {
    if (value.Has("ranges")) {
        auto& jArr = value["ranges"].GetArray();
        for (const auto& jVal: jArr) {
            Ranges[jVal["end"].GetInteger()] = jVal["id"].GetString();
        }
    }

    if (value.Has("tariffs")) {
        auto& jArr = value["tariffs"].GetArray();
        TMap<TString, TTariff> newTariffs;
        for (const auto& jVal: jArr) {
            auto& name = jVal["name"].GetString();
            if (name.empty()) {
                return false;
            }
            TTariff tariff;
            if (!tariff.DeserializeFromJson(jVal)) {
                return false;
            }
            Tariffs[name] = std::move(tariff);
        }
    }

    if (value.Has("grid")) {
        if (!DeserializeGridFromJson(value["grid"])) {
            return false;
        }
    }

    if (value.Has("options")) {
        auto& jArr = value["options"].GetArray();
        for (const auto& jVal: jArr) {
            TOptionData option;
            if (!option.DeserializeFromJson(jVal)) {
                return false;
            }
            Options[jVal["id"].GetString()] = std::move(option);
        }
    }

    if (value.Has("insurances")) {
        auto& jArr = value["insurances"].GetArray();
        Insurances.reserve(jArr.size());
        for (const auto& jVal: jArr) {
            TInsuranceData insurance;
            if (!insurance.DeserializeFromJson(jVal)) {
                return false;
            }
            Insurances.push_back(std::move(insurance));
        }
    }

    if (value.Has("insurances_grid")) {
        if (!DeserializeInsurancesGridFromJson(value["insurances_grid"])) {
            return false;
        }
    }

    if (value.Has("limits_grid")) {
        if (!DeserializeLimitsGridFromJson(value["limits_grid"])) {
            return false;
        }
    }
    if (!Tariffs.empty()) {
        TMap<TString /*tariff name*/, TLimitsCellData> newLimitsColumn;
        for (const auto& [name, tariff]: Tariffs) {
            newLimitsColumn[name] = TLimitsCellData();
        }

        for (auto& limitId: limitColumnIds) {
            if (!LimitsGrid.contains(limitId)) {
                LimitsGrid[limitId] = newLimitsColumn;
            }
        }
    }

    NJson::TJsonValue jMigration;
    if (value.Has("migration") && !ApplyMigration(value["migration"])) {
        return false;
    }

    return true;
}

bool TRentalEconomy::ApplyMigration(const NJson::TJsonValue& value) {
    if (value.Has("ranges")) {
        auto& jArr = value["ranges"].GetArray();
        TMap<TString, ui64> newRanges;
        TSet<ui64> uniqueRangeEnds;
        for (const auto& jVal: jArr) {
            const auto& id = jVal["id"].GetString();
            const auto rangeEnd = jVal["end"].GetInteger();
            if (!newRanges.contains(id) && !uniqueRangeEnds.contains(rangeEnd)) {
                newRanges[id] = rangeEnd;
                uniqueRangeEnds.insert(rangeEnd);
            } else {
                return false;
            }
        }

        for (const auto &oldRange: Ranges) {
            // delete removed columns from grid
            if (auto newRangeIt = newRanges.find(oldRange.second); newRangeIt == newRanges.end()) {
                Grid.erase(oldRange.first);
            } else {
                // update modified columns
                if (newRangeIt->second != oldRange.first) {
                    Grid[newRangeIt->second] = std::move(Grid[oldRange.first]);
                    Grid.erase(oldRange.first);
                }
            }
        }

        TMap<TString, TCellData> tariffsColumn;
        for (const auto& [tariffName, _]: Tariffs) {
            tariffsColumn[tariffName].DailyCost = Nothing();
        }

        Ranges.clear();
        for (auto& [id, rangeEnd]: newRanges) {
            // insert new columns
            if (!Grid.contains(rangeEnd)) {
                Grid[rangeEnd] = tariffsColumn;
            }
            Ranges[rangeEnd] = std::move(id);
        }
    }

    if (value.Has("tariffs")) {
        auto& jArr = value["tariffs"].GetArray();
        TMap<TString, TTariff> newTariffs;
        TMap<TString, TString> newTariffsIdToNameMap;
        for (const auto& jVal: jArr) {
            auto& name = jVal["name"].GetString();
            if (name.empty()) {
                return false;
            }

            TTariff tariff;
            if (!tariff.DeserializeFromJson(jVal)) {
                return false;
            }

            if (!newTariffs.contains(name) && !newTariffsIdToNameMap.contains(tariff.Id)) {
                newTariffsIdToNameMap[tariff.Id] = name;
                newTariffs[name] = std::move(tariff);
            } else {
                return false;
            }
        }

        for (const auto& [oldName, oldTariff]: Tariffs) {
            // delete removed rows
            if (!newTariffs.contains(oldName)) {
                if (auto newTariffIdIt = newTariffsIdToNameMap.find(oldTariff.Id); newTariffIdIt == newTariffsIdToNameMap.end()) {
                    RemoveCell(oldName, Grid);
                    RemoveCell(oldName, InsurancesGrid);
                    RemoveCell(oldName, LimitsGrid);
                } else {
                    MoveCellData(oldName, newTariffIdIt->second, Grid);
                    MoveCellData(oldName, newTariffIdIt->second, InsurancesGrid);
                    MoveCellData(oldName, newTariffIdIt->second, LimitsGrid);
                }
            }
        }

        for (const auto& [newName, newTariff]: newTariffs) {
            // insert new rows
            if (!Tariffs.contains(newName) && (Grid.empty() || !Grid.begin()->second.contains(newName))) {
                for (auto& [rangeEnd, columnTariffs]: Grid) {
                    columnTariffs[newName].DailyCost = Nothing();
                }
                for (auto& [insuranceId, columnTariffs]: InsurancesGrid) {
                    columnTariffs[newName].Cost = Nothing();
                }
                for (auto& [limitId, columnLimits]: LimitsGrid) {
                    columnLimits[newName].Mileage = Nothing();
                }
            }
        }
        Tariffs = std::move(newTariffs);
    }

    if (value.Has("grid")) {
        if (!ApplyGridMigration(value["grid"])) {
            return false;
        }
    }

    if (value.Has("options")) {
        Options.clear();
        auto& jArr = value["options"].GetArray();
        for (const auto& jVal: jArr) {
            TOptionData option;
            if (!option.DeserializeFromJson(jVal)) {
                return false;
            }
            Options[jVal["id"].GetString()] = std::move(option);
        }
    }

    if (value.Has("insurances")) {
        TMap<TString /*tariff name*/, TInsuranceCellData> column;
        for (const auto& [name, data]: Tariffs) {
            column[name] = TInsuranceCellData();
        }
        auto& jArr = value["insurances"].GetArray();
        TVector<TInsuranceData> newInsurances;
        newInsurances.reserve(jArr.size());
        for (const auto& jVal: jArr) {
            TInsuranceData insurance;
            if (!insurance.DeserializeFromJson(jVal)) {
                return false;
            }
            if (auto insuranceIt = std::find_if(Insurances.begin(), Insurances.end(),
                [&insurance](const TInsuranceData& oldInsurance){
                    return insurance.GetId() == oldInsurance.GetId();
                    }); insuranceIt == Insurances.end()) {
                InsurancesGrid[insurance.GetId()] = column;
            }
            newInsurances.push_back(std::move(insurance));
        }

        for (auto& oldInsurance: Insurances) {
            if (auto insuranceIt = std::find_if(newInsurances.begin(), newInsurances.end(),
                [&oldInsurance](const TInsuranceData& insurance){
                    return insurance.GetId() == oldInsurance.GetId();
                    }); insuranceIt == newInsurances.end()) {
                InsurancesGrid.erase(oldInsurance.GetId());
            }
        }
        Insurances = std::move(newInsurances);
    }

    if (value.Has("insurances_grid")) {
        if (!ApplyInsurancesGridMigration(value["insurances_grid"])) {
            return false;
        }
    }

    if (value.Has("limits_grid")) {
        if (!ApplyLimitsGridMigration(value["limits_grid"])) {
            return false;
        }
    }

    return true;
}

NJson::TJsonValue TRentalEconomy::SerializeToJson(ELocalization locale, const ILocalization* localization, const TMap<TString, ui64>* tariffCars) const {
    NJson::TJsonValue value;
    NJson::TJsonArray ranges, tariffs;
    for (const auto& [end, id]: Ranges) {
        NJson::TJsonValue jRange;
        jRange["end"] = end;
        jRange["id"] = id;
        ranges.AppendValue(std::move(jRange));
    }
    value["ranges"] = std::move(ranges);

    for (const auto& [name, tariff]: Tariffs) {
        NJson::TJsonValue jTariff = tariff.SerializeToJson(locale, localization, tariffCars);
        jTariff["name"] = name;
        tariffs.AppendValue(std::move(jTariff));
    }
    value["tariffs"] = std::move(tariffs);

    NJson::TJsonArray optionsArr;
    for (const auto& [id, option]: Options) {
        auto jOption = option.SerializeToJson();
        jOption["id"] = id;
        optionsArr.AppendValue(std::move(jOption));
    }
    value["options"] = std::move(optionsArr);

    NJson::TJsonArray insurancesArr;
    for (const auto& insurance: Insurances) {
        insurancesArr.AppendValue(insurance.SerializeToJson());
    }
    value["insurances"] = std::move(insurancesArr);

    value["grid"] = SerializeGridToJson();
    value["insurances_grid"] = SerializeInsurancesGridToJson();
    value["limits_grid"] = SerializeLimitsGridToJson();
    return value;
}

TMaybe<TRentalEconomy::TTariffRecommendation> TRentalEconomy::GetRecommendedTariff(
    const TOfferTariffRecommendationData& offerData,
    const NDrive::IServer* server,
    NDrive::TInfoEntitySession& session,
    bool& sessionErrorHappend
) const {
    if (offerData.HasSince() && offerData.HasUntil() && offerData.HasCarId()) {
        return GetRecommendedTariff(offerData.GetCarIdRef(), *offerData.OptionalUntil() - *offerData.OptionalSince(), server, session, sessionErrorHappend);
    }
    return Nothing();
}

TMaybe<TRentalEconomy::TTariffRecommendation> TRentalEconomy::GetRecommendedTariff(
    const TString& carId, TDuration duration, const NDrive::IServer* server,
    NDrive::TInfoEntitySession& session,
    bool& sessionErrorHappend
) const {
    if (Ranges.empty()) {
        return Nothing();
    }

    auto tx =  server->GetDriveAPI()->BuildTx<NSQL::ReadOnly>();
    const auto carsFetchResult = server->GetDriveAPI()->GetCarsData()->FetchInfo(carId, tx);
    if (!carsFetchResult) {
        session.AddErrorMessage("TRentalEconomy::GetRecommendedTariff", "cannot fetch car");
        sessionErrorHappend = true;
        return Nothing();
    }
    const auto carsInfo = carsFetchResult.GetResult();
    if (carsInfo.size() != 1) {
        session.AddErrorMessage("TRentalEconomy::GetRecommendedTariff", "wrong carsInfo.size()");
        sessionErrorHappend = true;
        return Nothing();
    }
    const auto& info = carsInfo.begin()->second;

    TMaybe<TRentalEconomy::TTariffRecommendation> recommendation;
    const auto durationDays = duration.Seconds() > duration.Days() * secondsInDay ? duration.Days() + 1 : duration.Days();
    if (auto columntIt = Grid.lower_bound(durationDays); columntIt != Grid.end()) {
        auto tariffName = GetAppropriateTariff(info, columntIt->second);
        if (tariffName) {
            if (auto cellIt = columntIt->second.find(*tariffName); cellIt != columntIt->second.end()) {
                recommendation = TTariffRecommendation();
                recommendation->SetDailyCost(cellIt->second.DailyCost);
                recommendation->SetName(*tariffName);
                auto tariffIt = Tariffs.find(*tariffName);
                if (tariffIt != Tariffs.end()) {
                    recommendation->SetId(tariffIt->second.Id);
                    recommendation->SetDeposit(tariffIt->second.Deposit);
                    recommendation->SetOverrunCostPerKm(tariffIt->second.OverrunCostPerKm);
                }
            }
        }
    }

    return recommendation;
}

TMaybe<TRentalEconomy::TOptionsRecommendation> TRentalEconomy::GetRecommendedOptions(const TOfferTariffRecommendationData& offerData) const {
    if (!offerData.Options) {
        return Nothing();
    }
    return GetRecommendedOptions(offerData.OptionalSince(), offerData.OptionalUntil(), *offerData.Options);
}

TMaybe<TRentalEconomy::TOptionsRecommendation> TRentalEconomy::GetRecommendedOptions(
    TMaybe<TInstant> since,
    TMaybe<TInstant> until,
    const TVector<TRentalOptionValuesAdapter>& options
) const {
    TOptionsRecommendation recommendation;
    TMaybe<ui64> durationDays;
    if (since && until) {
        TDuration duration = *until - *since;
        durationDays = duration.Seconds() > duration.Days() * secondsInDay ? duration.Days() + 1 : duration.Days();
    }

    for (const auto& [id, option]: Options) {
        if (std::find_if(options.begin(), options.end(),
            [id = &id](const TRentalOptionValuesAdapter& value) {
                return value.HasConstants() && value.GetConstantsRef().GetId() == *id;
            }) == options.end()) {
            continue;
        }
        if (option.HasCost()) {
            TOptionsRecommendation::TOption totalOptionCost;
            totalOptionCost.Id = id;
            totalOptionCost.TotalCost = option.GetCostRef();
            if (option.GetIsCostPerDay()) {
                if (durationDays) {
                    totalOptionCost.TotalCost *= *durationDays;
                } else {
                    continue;
                }
            }

            if (option.HasMaxTotalCost()) {
                totalOptionCost.TotalCost = std::min(totalOptionCost.TotalCost, option.GetMaxTotalCostRef());
            }
            recommendation.MutableTotalOptionCosts().push_back(totalOptionCost);
        }
    }
    return recommendation;
}

TMaybe<TRentalEconomy::TInsurancesRecommendation> TRentalEconomy::GetRecommendedInsurances(const TString& tariffName) const {
    TInsurancesRecommendation recommendation;

    for (const auto& insuranceData: Insurances) {
        if (auto columnIt = InsurancesGrid.find(insuranceData.GetId()); columnIt != InsurancesGrid.end()) {
            const auto& [insuranceId, columnData] = *columnIt;

            if (auto cellDataIt = columnData.find(tariffName); cellDataIt != columnData.end() && !cellDataIt->second.Cost.Empty()) {
                TInsurancesRecommendation::TInsuranceRecommendation insurance;
                insurance.Id = insuranceId;
                insurance.Cost = *cellDataIt->second.Cost;
                recommendation.MutableInsurances().push_back(std::move(insurance));
            }
        }
    }

    if (recommendation.GetInsurances().empty()) {
        return Nothing();
    }
    return recommendation;
}

TMaybe<TRentalEconomy::TLimitsRecommendation> TRentalEconomy::GetRecommendedLimits(const TString& tariffName) const {
    TLimitsRecommendation recommendation;
    bool recommendationFound = false;
    if (!LimitsGrid.empty()) {
        if (auto limitIt = LimitsGrid.begin()->second.find(tariffName); limitIt !=  LimitsGrid.begin()->second.end()) {
            if (!limitIt->second.Mileage.Empty()) {
                recommendationFound = true;
                recommendation.Mileage = *(limitIt->second.Mileage);
            }
        }
    }

    if (!recommendationFound) {
        return Nothing();
    }
    return recommendation;
}

bool TRentalEconomy::IsNewTariffRecommendationRequired(
    const TOfferTariffRecommendationData& currentData,
    const TOfferTariffRecommendationData& oldData
) {
    if (currentData.OptionalSince() != oldData.OptionalSince()) {
        return true;
    }
    if (currentData.OptionalUntil() != oldData.OptionalUntil()) {
        return true;
    }
    if (currentData.OptionalCarId() != oldData.OptionalCarId()) {
        return true;
    }
    return false;
}

TMap<TString, ui64> TRentalEconomy::CalculateTariffCars(
    const NDrive::IServer& server,
    NDrive::TEntitySession& tx,
    const TUserPermissions::TPtr permissions
) const {
    TSet<TString> visibleCarIds;
    TMap<TString, ui64> tariffCars;
    {
        TDeviceTagsManager::TCurrentSnapshot devicesSnapshot;
        R_ENSURE(
              server.GetDriveAPI()->GetTagsManager().GetDeviceTags().GetCurrentSnapshot(devicesSnapshot)
            , HTTP_INTERNAL_SERVER_ERROR
            , "cannot get current snapshot"
            , tx);
            for (const auto& [carId, taggedObject]: devicesSnapshot) {
                if (permissions->GetVisibility(*taggedObject, NEntityTagsManager::EEntityType::Car) == TUserPermissions::EVisibility::Visible) {
                    visibleCarIds.insert(carId);
                }
            }
    }

    auto carsInfo = server.GetDriveAPI()->GetCarsData()->FetchInfo(visibleCarIds);

    for (auto& [name, tariff]: Tariffs) {
        tariffCars[tariff.Id] = 0;
    }

    for (auto& [name, tariff]: Tariffs) {
        auto& carsCount = tariffCars[tariff.Id];
        for (const auto& [id, info]: carsInfo.GetResult()) {
            if (tariff.IsTariffMatchedWithCar(info)) {
                ++carsCount;
            }
        }
    }
    return tariffCars;
}

bool TRentalEconomy::IsOptionAllowed(const TString& id) const {
    return Options.empty() || Options.contains(id);
}

TMaybe<TString> TRentalEconomy::GetAppropriateTariff(const TDriveCarInfo& info, const TMap<TString /*name*/, TCellData>& tariffCosts) const {
    TMaybe<ui64> appropriateTariffCost;
    TMaybe<TString> appropriateTariffName;

    for (const auto& [name, tariff]: Tariffs) {
        if (auto costIt = tariffCosts.find(name); costIt != tariffCosts.end()) {
            if (tariff.IsTariffMatchedWithCar(info)) {
                if (appropriateTariffCost.Empty()) {
                    appropriateTariffCost = costIt->second.DailyCost;
                    appropriateTariffName = name;
                } else if (!costIt->second.DailyCost.Empty()) {
                    if (*appropriateTariffCost < *(costIt->second.DailyCost)) {
                        appropriateTariffCost = costIt->second.DailyCost;
                        appropriateTariffName = name;
                    }
                }
            }
        }
    }

    return appropriateTariffName;
}

bool TRentalEconomy::ApplyGridMigration(const NJson::TJsonValue& value) {
    auto columnIdGetter = [](const NJson::TJsonValue& jColumn) { return jColumn["end"].GetInteger(); };
    auto dataGetter = [](TCellData& cell)->TMaybe<ui64>&{ return cell.DailyCost; };
    return ApplyCommonGridMigration(value, Grid, columnIdGetter, "tariff_costs", "daily_cost", dataGetter);
}

bool TRentalEconomy::ApplyInsurancesGridMigration(const NJson::TJsonValue& value) {
    auto columnIdGetter = [](const NJson::TJsonValue& jColumn) { return jColumn["id"].GetString(); };
    auto dataGetter = [](TInsuranceCellData& cell)->TMaybe<ui64>&{ return cell.Cost; };
    return ApplyCommonGridMigration(value, InsurancesGrid, columnIdGetter, "insurances_costs", "cost", dataGetter);
}

bool TRentalEconomy::ApplyLimitsGridMigration(const NJson::TJsonValue& value) {
    auto columnIdGetter = [](const NJson::TJsonValue& jColumn) { return jColumn["id"].GetString(); };
    auto dataGetter = [](TLimitsCellData& cell)->TMaybe<ui64>&{ return cell.Mileage; };
    return ApplyCommonGridMigration(value, LimitsGrid, columnIdGetter, "limits", "mileage", dataGetter);
}

bool TRentalEconomy::DeserializeGridFromJson(const NJson::TJsonValue& value) {
    Grid.clear();
    auto jColumns = value.GetArray();
    for (const auto& jColumn: jColumns) {
        auto rangeEnd = jColumn["end"].GetInteger();
        auto& tariffCosts = jColumn["tariff_costs"].GetArray();
        TMap<TString, TCellData> costs;
        if (Ranges.find(rangeEnd) == Ranges.end()) {
            return false;
        }
        for (const auto& cell: tariffCosts) {
            auto& tariffName = cell["name"].GetString();
            if (Tariffs.find(tariffName) == Tariffs.end()) {
                return false;
            }
            if (!cell.Has("daily_cost") || cell["daily_cost"].IsNull()) {
                costs[tariffName].DailyCost = Nothing();
            } else {
                costs[tariffName].DailyCost = cell["daily_cost"].GetUInteger();
            }
        }
        Grid[rangeEnd] = std::move(costs);
    }
    return true;
}

bool TRentalEconomy::DeserializeInsurancesGridFromJson(const NJson::TJsonValue& value) {
    InsurancesGrid.clear();
    auto jColumns = value.GetArray();
    for (const auto& jColumn: jColumns) {
        auto& insuranceId = jColumn["id"].GetString();
        auto& insurancesCosts = jColumn["insurances_costs"].GetArray();
        TMap<TString, TInsuranceCellData> costs;
        if (auto insuranceIt = std::find_if(Insurances.begin(), Insurances.end(),
                [&insuranceId](const TInsuranceData& insurance) {
                    return insurance.GetId() == insuranceId;
                }); insuranceIt == Insurances.end()) {
            return false;
        }
        for (const auto& cell: insurancesCosts) {
            auto& tariffName = cell["name"].GetString();
            if (Tariffs.find(tariffName) == Tariffs.end()) {
                return false;
            }
            if (!cell.Has("cost") || cell["cost"].IsNull()) {
                costs[tariffName].Cost = Nothing();
            } else {
                costs[tariffName].Cost = cell["cost"].GetUInteger();
            }
        }
        InsurancesGrid[insuranceId] = std::move(costs);
    }
    return true;
}

bool TRentalEconomy::DeserializeLimitsGridFromJson(const NJson::TJsonValue& value) {
    LimitsGrid.clear();
    auto jColumns = value.GetArray();
    for (const auto& jColumn: jColumns) {
        auto& limitId = jColumn["id"].GetString();
        auto& limits = jColumn["limits"].GetArray();
        TMap<TString, TLimitsCellData> columnLimits;
        for (const auto& cell: limits) {
            auto& tariffName = cell["name"].GetString();
            if (Tariffs.find(tariffName) == Tariffs.end()) {
                return false;
            }
            if (!cell.Has("mileage") || cell["mileage"].IsNull()) {
                columnLimits[tariffName].Mileage = Nothing();
            } else {
                columnLimits[tariffName].Mileage = cell["mileage"].GetUInteger();
            }
        }
        LimitsGrid[limitId] = std::move(columnLimits);
    }
    return true;
}

NJson::TJsonValue TRentalEconomy::SerializeGridToJson() const {
    NJson::TJsonValue grid;
    for (const auto& [columnEnd, rows]: Grid) {
        NJson::TJsonValue jColumn;
        if (auto rangeIt = Ranges.find(columnEnd); rangeIt != Ranges.end()) {
            jColumn["id"] = rangeIt->second;
            jColumn["end"] = columnEnd;
            NJson::TJsonArray jArr;
            for (const auto& [tariffName, rowData]: rows) {
                if (auto tariffIt = Tariffs.find(tariffName); tariffIt != Tariffs.end()) {
                    NJson::TJsonValue jCell;
                    jCell["id"] = tariffIt->second.Id;
                    jCell["name"] = tariffName;
                    NJson::InsertField(jCell, "daily_cost", rowData.DailyCost);
                    jArr.AppendValue(std::move(jCell));
                }
            }
            jColumn["tariff_costs"] = std::move(jArr);
            grid.AppendValue(std::move(jColumn));
        }
    }
    return grid;
}

NJson::TJsonValue TRentalEconomy::SerializeInsurancesGridToJson() const {
    NJson::TJsonValue insurancesGrid;
    for (const auto& insurance: Insurances) {
        if (auto columnIt = InsurancesGrid.find(insurance.GetId()); columnIt != InsurancesGrid.end()) {
            NJson::TJsonArray jArr;

            for (const auto& [tariffName, rowData]: columnIt->second) {
                if (auto tariffIt = Tariffs.find(tariffName); tariffIt != Tariffs.end()) {
                    NJson::TJsonValue jCell;
                    jCell["id"] = tariffIt->second.Id;
                    jCell["name"] = tariffName;
                    NJson::InsertField(jCell, "cost", rowData.Cost);
                    jArr.AppendValue(std::move(jCell));
                }
            }

            NJson::TJsonValue jColumn;
            jColumn["id"] = insurance.GetId();
            jColumn["insurances_costs"] = std::move(jArr);
            insurancesGrid.AppendValue(std::move(jColumn));
        }
    }
    return insurancesGrid;
}

NJson::TJsonValue TRentalEconomy::SerializeLimitsGridToJson() const {
    NJson::TJsonValue limitsGrid;
    for (const auto& [limitId, rows]: LimitsGrid) {
        NJson::TJsonValue jColumn;
        jColumn["id"] = limitId;
        NJson::TJsonArray jArr;
        for (const auto& [tariffName, rowData]: rows) {
            if (auto tariffIt = Tariffs.find(tariffName); tariffIt != Tariffs.end()) {
                NJson::TJsonValue jCell;
                jCell["id"] = tariffIt->second.Id;
                jCell["name"] = tariffName;
                NJson::InsertField(jCell, "mileage", rowData.Mileage);
                jArr.AppendValue(std::move(jCell));
            }
        }
        jColumn["limits"] = std::move(jArr);
        limitsGrid.AppendValue(std::move(jColumn));
    }
    return limitsGrid;
}

bool TRentalEconomy::TTariff::IsTariffMatchedWithCar(const TDriveCarInfo& info) const {
    if (!ModelId.empty() && !info.GetModel().empty() && ModelId != info.GetModel()) {
        return false;
    }

    const auto& specifications = info.GetSpecifications();
    for (const auto& specification: specifications.GetSpecifications()) {
        if (!CategoryId.empty() && NDriveModelSpecification::EModelSpecificationType::Category == specification.GetType()) {
            if (specification.GetValue() != CategoryId) {
                return false;
            }
        }

        if (!ExteriorTypeId.empty() && NDriveModelSpecification::EModelSpecificationType::ExteriorType == specification.GetType()) {
            if (specification.GetValue() != ExteriorTypeId) {
                return false;
            }
        }

        if (!TransmissionId.Empty() && NDriveModelSpecification::EModelSpecificationType::Transmission == specification.GetType()) {
            if (std::get<NDriveModelSpecification::ETransmissionType>(specification.GetParsedValue()) != *TransmissionId) {
                return false;
            }
        }

        if (!FuelTypeId.empty() && NDriveModelSpecification::EModelSpecificationType::FuelType == specification.GetType()) {
            if (specification.GetValue() != FuelTypeId) {
                return false;
            }
        }

        if (!AirConditioningId.empty() && NDriveModelSpecification::EModelSpecificationType::AirConditioning == specification.GetType()) {
            if (specification.GetValue() != AirConditioningId) {
                return false;
            }
        }
    }

    return true;
}

bool TRentalEconomy::TTariff::DeserializeFromJson(const NJson::TJsonValue& value) {
    Id = value["id"].GetString();
    if (Id.empty()) {
        return false;
    }
    ModelId = value["model"]["id"].GetString();
    CategoryId = value["category"]["id"].GetString();
    ExteriorTypeId = value["exterior_type"]["id"].GetString();
    auto transmission =  value["transmission"]["id"].GetString();
    for (auto&& item : GetEnumNames<NDriveModelSpecification::ETransmissionType>()) {
        if (transmission == item.second) {
            TransmissionId = item.first;
        }
    }
    FuelTypeId = value["fuel_type"]["id"].GetString();
    AirConditioningId = value["air_conditioning"]["id"].GetString();

    if (value.Has("deposit")) {
        if (!value["deposit"].IsNull()) {
            Deposit = value["deposit"].GetUInteger();
        } else {
            Deposit = Nothing();
        }
    }

    if (value.Has("overrun_cost_per_km")) {
        if (!value["overrun_cost_per_km"].IsNull()) {
            OverrunCostPerKm = value["overrun_cost_per_km"].GetUInteger();
        } else {
            OverrunCostPerKm = Nothing();
        }
    }
    return true;
}

NJson::TJsonValue TRentalEconomy::TTariff::SerializeToJson(ELocalization locale, const ILocalization* localization, const TMap<TString, ui64>* tariffCars) const {
    NJson::TJsonValue value;
    value["id"] = Id;
    if (!ModelId.empty()) {
        NJson::TJsonValue jObj;
        jObj["id"] = ModelId;
        if (localization) {
            auto server = NDrive::HasServer() ? &NDrive::GetServer() : nullptr;
            Y_ENSURE(server);
            auto driveApi = server->GetAsSafe<NDrive::IServer>().GetDriveAPI();
            auto modelFetchResult = driveApi->GetModelsData()->GetCachedOrFetch(ModelId);
            if (modelFetchResult) {
                auto modelPtr = modelFetchResult.GetResultPtr(ModelId);
                if (modelPtr) {
                    jObj["name"] = modelPtr->GetName();
                }
            }
        }
        value["model"] = jObj;
    }

    if (!CategoryId.empty()) {
        NJson::TJsonValue jObj;
        jObj["id"] = CategoryId;
        if (localization) {
            jObj["name"] = localization->GetLocalString(locale, TStringBuilder() << "model.specification." << NDriveModelSpecification::EModelSpecificationType::Category << '.' << CategoryId);
        }
        value["category"] = jObj;
    }

    if (!ExteriorTypeId.empty()) {
        NJson::TJsonValue jObj;
        jObj["id"] = ExteriorTypeId;
        if (localization) {
            jObj["name"] = localization->GetLocalString(locale, TStringBuilder() << "model.specification." << NDriveModelSpecification::EModelSpecificationType::ExteriorType << '.' << ExteriorTypeId);
        }
        value["exterior_type"] = jObj;
    }

    if (TransmissionId) {
        NJson::TJsonValue jObj;
        jObj["id"] = TStringBuilder() << *TransmissionId;
        if (localization) {
            jObj["name"] = localization->GetLocalString(locale, TStringBuilder() << "model.specification." << NDriveModelSpecification::EModelSpecificationType::Transmission << '.' << *TransmissionId);
        }
        value["transmission"] = jObj;
    }

    if (!FuelTypeId.empty()) {
        NJson::TJsonValue jObj;
        jObj["id"] = FuelTypeId;
        if (localization) {
            jObj["name"] = localization->GetLocalString(locale, TStringBuilder() << "model.specification." << NDriveModelSpecification::EModelSpecificationType::FuelType << '.' << FuelTypeId);
        }
        value["fuel_type"] = jObj;
    }

    if (!ExteriorTypeId.empty()) {
        NJson::TJsonValue jObj;
        jObj["id"] = AirConditioningId;
        if (localization) {
            jObj["name"] = localization->GetLocalString(locale, TStringBuilder() << "model.specification." << NDriveModelSpecification::EModelSpecificationType::AirConditioning << '.' << AirConditioningId);
        }
        value["air_conditioning"] = jObj;
    }

    NJson::InsertField(value, "deposit", Deposit);
    NJson::InsertField(value, "overrun_cost_per_km", OverrunCostPerKm);

    if (tariffCars) {
        if (auto tariffCarsIt = tariffCars->find(Id); tariffCarsIt !=  tariffCars->end()) {
            value["cars_count"] = tariffCarsIt->second;
        }
    }

    return value;
}

bool TRentalEconomy::TOptionData::DeserializeFromJson(const NJson::TJsonValue& value) {
    if (value.Has("cost") && !value["cost"].IsNull()) {
        Cost = value["cost"].GetUInteger();
    }
    IsCostPerDay = value["is_cost_per_day"].GetBoolean();
    if (value.Has("max_total_cost") && !value["max_total_cost"].IsNull()) {
        MaxTotalCost = value["max_total_cost"].GetUInteger();
    }
    return true;
}

NJson::TJsonValue TRentalEconomy::TOptionData::SerializeToJson() const {
    NJson::TJsonValue value;
    NJson::InsertField(value, "cost", Cost);
    value["is_cost_per_day"] = IsCostPerDay;
    NJson::InsertField(value, "max_total_cost", MaxTotalCost);
    return value;
}

bool TRentalEconomy::TInsuranceData::DeserializeFromJson(const NJson::TJsonValue& value) {
    if (value.Has("id") && !value["id"].IsNull()) {
        Id = value["id"].GetString();
        return true;
    }
    return false;
}

NJson::TJsonValue TRentalEconomy::TInsuranceData::SerializeToJson() const {
    NJson::TJsonValue value;
    value["id"] = Id;
    return value;
}

TRentalOffer::TFactory::TRegistrator<TRentalOffer> TRentalOffer::Registrator(TRentalOffer::GetTypeNameStatic());
TRentalOfferBuilder::TFactory::TRegistrator<TRentalOfferBuilder> TRentalOfferBuilder::Registrator(TRentalOfferBuilder::GetTypeName());
