#pragma once

#include "abstract.h"
#include "pack.h"
#include "types.h"

#include <drive/backend/cars/specification.h>
#include <drive/backend/data/rental/rental_offer_holder_tag.h>
#include <drive/backend/offers/offers/dedicated_fleet_common.h>

#include <rtline/util/types/field.h>


class TRentalOptionValues {
public:
    static constexpr TStringBuf CalculatePolicyNameId = "calculate_policy";
public:
    class TConstants {
    public:
        R_FIELD(bool, DefaultValue, false);
        R_FIELD(TString, Id);
        R_FIELD(NDedicatedFleet::ECostCalculatePolicy, CalculatePolicy, NDedicatedFleet::ECostCalculatePolicy::None);
        R_FIELD(int, Order, 0);

    public:
        NDrive::TScheme GetScheme(const NDrive::IServer* server) const;

    public:
        DECLARE_FIELDS(
            Field(DefaultValue, "default_value"),
            Field(Id, "id"),
            Field(CalculatePolicy, CalculatePolicyNameId),
            Field(Order, "order")
        )
    };

    class TStringConstants {
    public:
        R_FIELD(TString, Title);
        R_FIELD(TString, Subtitle);

    public:
        NDrive::TScheme GetScheme(const NDrive::IServer* server) const;

    public:
        DECLARE_FIELDS(
            Field(Title, "title"),
            Field(Subtitle, "subtitle")
        )
    };

public:
    R_OPTIONAL(bool, Value);
    R_OPTIONAL(ui64, Cost);
    R_OPTIONAL(TString, Unit, Nothing(), mutable);
};

using TRentalOptionValuesAdapter = NDedicatedFleet::TValuesAdapter<TRentalOptionValues>;

class TRentalEconomy {
public:
    struct TOfferTariffRecommendationData {
        const TVector<TRentalOptionValuesAdapter>* Options{ nullptr };
        R_OPTIONAL(TInstant, Since);
        R_OPTIONAL(TInstant, Until);
        R_OPTIONAL(TString, CarId);
    };

    struct TTariffRecommendation {
        NJson::TJsonValue ReportToJson(TInstant since, TInstant until, ui64& accumulatedCost) const;
        NDrive::NProto::TRentalOfferTariffRecomendation SerializeToProto() const;
        bool DeserializeFromProto(const NDrive::NProto::TRentalOfferTariffRecomendation& meta);

        R_OPTIONAL(TString, Name);
        R_OPTIONAL(TString, Id);
        R_OPTIONAL(ui64, DailyCost);
        R_OPTIONAL(ui64, Deposit);
        R_OPTIONAL(ui64, OverrunCostPerKm);
    };

    struct TOptionsRecommendation {
        struct TOption {
            NDrive::NProto::TRentalOptionRecomendation SerializeToProto() const;
            bool DeserializeFromProto(const NDrive::NProto::TRentalOptionRecomendation& meta);

            TString Id;
            ui64 TotalCost = 0;
        };

        NJson::TJsonValue ReportToJson(ui64& accumulatedCost) const;

        R_FIELD(TVector<TOption>, TotalOptionCosts);
    };

    struct TInsurancesRecommendation {
        struct TInsuranceRecommendation {
            NDrive::NProto::TInsuranceRecommendation SerializeToProto() const;
            bool DeserializeFromProto(const NDrive::NProto::TInsuranceRecommendation& meta);

            TString Id;
            ui64 Cost = 0;
        };

        NJson::TJsonValue ReportToJson() const;

        R_FIELD(TVector<TInsuranceRecommendation>, Insurances);
    };

    struct TLimitsRecommendation {
        NDrive::NProto::TRentalLimitsRecommendation SerializeToProto() const;
        bool DeserializeFromProto(const NDrive::NProto::TRentalLimitsRecommendation& meta);

        NJson::TJsonValue ReportToJson() const;

        ui64 Mileage = 0;
    };

public:
    void PatchReport(NJson::TJsonValue& report, ELocalization locale, const ILocalization* localization, const TMap<TString, ui64>& tariffCars) const;
    bool DeserializeFromJson(const NJson::TJsonValue& value);
    NJson::TJsonValue SerializeToJson(ELocalization locale = ELocalization::Eng, const ILocalization* localization = nullptr, const TMap<TString, ui64>* tariffCars = nullptr) const;
    TMaybe<TTariffRecommendation> GetRecommendedTariff(const TOfferTariffRecommendationData& offerData, const NDrive::IServer* server, NDrive::TInfoEntitySession& session, bool& sessionErrorHappend) const;
    TMaybe<TTariffRecommendation> GetRecommendedTariff(const TString& carId, TDuration duration, const NDrive::IServer* server, NDrive::TInfoEntitySession& session, bool& sessionErrorHappend) const;
    TMaybe<TOptionsRecommendation> GetRecommendedOptions(const TOfferTariffRecommendationData& offerData) const;
    TMaybe<TOptionsRecommendation> GetRecommendedOptions(TMaybe<TInstant> since, TMaybe<TInstant> until, const TVector<TRentalOptionValuesAdapter>& options) const;
    TMaybe<TInsurancesRecommendation> GetRecommendedInsurances(const TString& tariffName) const;
    TMaybe<TLimitsRecommendation> GetRecommendedLimits(const TString& tariffName) const;

    static bool IsNewTariffRecommendationRequired(const TOfferTariffRecommendationData& currentData, const TOfferTariffRecommendationData& oldData);
    TMap<TString, ui64> CalculateTariffCars(const NDrive::IServer& server, NDrive::TEntitySession& tx, const TUserPermissions::TPtr permissions) const;
    bool IsOptionAllowed(const TString& id) const;

private:
    struct TCellData {
        TMaybe<ui64> DailyCost;
    };

    struct TInsuranceCellData {
        TMaybe<ui64> Cost;
    };

    struct TLimitsCellData {
        TMaybe<ui64> Mileage;
    };

private:
    bool ApplyMigration(const NJson::TJsonValue& value);
    TMaybe<TString> GetAppropriateTariff(const TDriveCarInfo& info, const TMap<TString /*name*/, TCellData>& tariffCosts) const;
    bool ApplyGridMigration(const NJson::TJsonValue& value);
    bool ApplyInsurancesGridMigration(const NJson::TJsonValue& value);
    bool ApplyLimitsGridMigration(const NJson::TJsonValue& value);
    bool DeserializeGridFromJson(const NJson::TJsonValue& value);
    bool DeserializeInsurancesGridFromJson(const NJson::TJsonValue& value);
    bool DeserializeLimitsGridFromJson(const NJson::TJsonValue& value);
    NJson::TJsonValue SerializeGridToJson() const;
    NJson::TJsonValue SerializeInsurancesGridToJson() const;
    NJson::TJsonValue SerializeLimitsGridToJson() const;

private:
    struct TTariff {
        bool IsTariffMatchedWithCar(const TDriveCarInfo& info) const;

        bool DeserializeFromJson(const NJson::TJsonValue& value);
        NJson::TJsonValue SerializeToJson(ELocalization locale, const ILocalization* localization, const TMap<TString, ui64>* tariffCars) const;

        TString Id;
        TString ModelId;
        TString CategoryId;
        TString ExteriorTypeId;
        TMaybe<NDriveModelSpecification::ETransmissionType> TransmissionId;
        TString FuelTypeId;
        TString AirConditioningId;
        TMaybe<ui64> Deposit;
        TMaybe<ui64> OverrunCostPerKm;
    };

    struct TOptionData {
        bool DeserializeFromJson(const NJson::TJsonValue& value);
        NJson::TJsonValue SerializeToJson() const;

        R_OPTIONAL(ui64, Cost);
        R_FIELD(bool, IsCostPerDay, false);
        R_OPTIONAL(ui64, MaxTotalCost);
    };

    struct TInsuranceData {
        bool DeserializeFromJson(const NJson::TJsonValue& value);
        NJson::TJsonValue SerializeToJson() const;

        R_FIELD(TString, Id);
    };

private:
    TMap<ui64 /*range end*/, TString /*range id*/> Ranges;
    TMap<TString /*name*/, TTariff> Tariffs;
    TMap<ui64 /*range end*/, TMap<TString /*name*/, TCellData>> Grid;
    TMap<TString /*id*/, TOptionData> Options;
    TVector<TInsuranceData> Insurances;
    TMap<TString /*insurance type*/, TMap<TString /*tariff name*/, TInsuranceCellData>> InsurancesGrid;
    TMap<TString /*limits id*/, TMap<TString /*tariff name*/, TLimitsCellData>> LimitsGrid;
};

enum class ERentalOfferImageStyle {
    Default,
    Render,
};

struct TRentalInsurance {
public:
    R_FIELD(TString, Id);
    R_FIELD(TString, Title);

public:
    NJson::TJsonValue GetReport(ELocalization locale, const NDrive::IServer& server) const;

public:
    DECLARE_FIELDS(
        Field(Id, "id"),
        Field(Title, "title")
    )
public:
    R_OPTIONAL(ui64, Cost);
};

enum class ELocationType {
    Uninitialized = 0 /* "uninitialized" */,
    Coord = 1 /* "coord" */,
    Polyline = 2 /* "polyline" */
};

struct TRentalOfferLocation {
public:
    R_FIELD(TString, Id);
    R_FIELD(TString, LocationName);
    R_FIELD(TVector<TGeoCoord>, Coords);

public:
    NJson::TJsonValue GetReport(ELocalization locale, const NDrive::IServer& server) const;

    ELocationType GetType() const {
        if (GetCoords().size() == 1) {
            return ELocationType::Coord;
        }
        if (GetCoords().size() > 3) {
            return ELocationType::Polyline;
        }
        return ELocationType::Uninitialized;
    }

    NDrive::NProto::TRentalOfferLocation SerializeToProto() const;
    bool DeserializeFromProto(const NDrive::NProto::TRentalOfferLocation& proto);
};

using TRentalOfferLocations = TVector<TRentalOfferLocation>;

using TRentalInsurances = TVector<TRentalInsurance>;
class TRentalOfferBuilder;

class TRentalOffer: public TPackOffer {
private:
    using TBase = TPackOffer;

public:
    using TBase::TBase;
    using TRenterOfferCommonInfo = TMap<TString, TString>;
    using TOption = TRentalOptionValuesAdapter;
    using TOptions = TVector<TOption>;

    static TString GetTypeNameStatic() {
        return "rental_offer";
    }
    TString GetTypeName() const override {
        return GetTypeNameStatic();
    }
    void AddOptionFromLegacy(bool value, const TString& id, int order);
    void CalculateMileageLimit();
    bool DeserializeFromProto(const NDrive::NProto::TOffer& info) override;
    NDrive::NProto::TOffer SerializeToProto() const override;
    void FillBill(TBill& bill, const TOfferPricing& pricing, TOfferStatePtr segmentState, ELocalization locale, const NDrive::IServer* server, ui32 cashbackPercent) const override;
    void PatchSessionReport(NJson::TJsonValue& result, NDriveSession::TReportTraits traits, ELocalization locale,
                                    const NDrive::IServer& server, const TOfferSessionPatchData& patchData) const override;
    bool PatchOffer(const NJson::TJsonValue& value, NDrive::TEntitySession& session, const NDrive::IServer* server, bool checkRidingConflict = true) override;
    ui64 GetOfferRevision() const override;

    TExpected<bool, NJson::TJsonValue> CheckAgreementRequirements(ELocalization locale, const TUserPermissions::TPtr permissions, const NDrive::IServer* server) const override;
    NThreading::TFuture<TString> BuildAgreement(ELocalization locale, TAtomicSharedPtr<TCompiledRiding> compiledSession, TUserPermissions::TPtr permissions, const NDrive::IServer* server) const override;

    bool PatchFromJson(const NJson::TJsonValue& value, NDrive::TEntitySession& session, const TRentalOfferBuilder& builder);
    void ReportToJson(NJson::TJsonValue& value, const NDrive::IServer& server, const ELocalization locale) const;
    bool FillAgreementTemplate(TString& agreementTemplate,
                               const NDrive::IServer* server,
                               NDrive::TEntitySession& session,
                               TAtomicSharedPtr<TCompiledRiding> compiledSession,
                               const ELocalization locale,
                               const TDriveUserData& user,
                               const TUserPassportData& passportData,
                               const TUserDrivingLicenseData& drivingLicense) const;
    static void FillTimeParameter(TString&& key, TInstant value, const TString& format, TString& text, const TMaybe<TString>& OfferTimezone);
    static void FillTimeParameter(const TString& key, TInstant value, const TString& format, TString& text, const TMaybe<TString>& OfferTimezone);
    TMaybe<TEmailOfferData> GetEmailData(const NDrive::IServer* server, NDrive::TEntitySession& session, const TString& notificationType) const override;
    static TString GetTimeParameter(TInstant value, const TString& format, const TMaybe<TString>& OfferTimezone);
    static TString MakeUnitValueId(const TString& Id);
    static TString MakeUnitPermKmId(const TString& Id);

public:
    static const TString DefaultOfferTimezone;
    static const TString DefaultDateFormat;
    static const TString DefaultTimeFormat;
    static const TString ServiceModeStatus;

    static const TString AcceptanceFinishedNotificationType;
    static const TString RideFinishedNotificationType;
    static const TString BookingConfirmedNotificationType;
    static const TString BookingCancelledNotificationType;

    static const TString SinceDateKeyword;
    static const TString SinceTimeKeyword;
    static const TString UntilDateKeyword;
    static const TString UntilTimeKeyword;
    static const TString CompanynameKeyword;
    static const TString PickupLocationKeyword;
    static const TString CarModelKeyword;

protected:
    NJson::TJsonValue DoBuildJsonReport(const TReportOptions& options, const ICommonOfferBuilderAction* constructor, const NDrive::IServer& server) const override;

private:
    NJson::TJsonValue GetInsuranceReport(ELocalization locale, const NDrive::IServer& server, const TRentalInsurances& insuranceTypes, const TString& unitValueId) const;
    std::pair<TString, TString> GetDateTimeReport(TInstant date) const;
    TString GetDurationReport(TDuration totalDuration, ELocalization locale, const ILocalization *localization) const;
    TRentalEconomy::TOfferTariffRecommendationData BuildOfferTariffRecommendationData() const;

private:

    R_OPTIONAL(TRentalInsurance, Insurance);

    R_FIELD(TInstant, Since);
    R_FIELD(TInstant, Until);
    R_FIELD(ui64, Revision, 0);

    R_OPTIONAL(TGeoCoord, DeliveryLocation);
    R_OPTIONAL(TString, DeliveryLocationName);

    R_OPTIONAL(TRentalOfferLocations, ReturnLocations);

    R_OPTIONAL(TString, Comment);

    R_OPTIONAL(ui64, TotalPayment, 0);
    R_OPTIONAL(ui64, Deposit, 0);

    R_OPTIONAL(TString, Status);

    R_OPTIONAL(ui64, LimitKmPerDay, 100);
    R_OPTIONAL(ui64, OverrunCostPerKm, 0);
    R_OPTIONAL(TString, Currency);
    R_OPTIONAL(TString, OfferImage);
    R_OPTIONAL(TString, OfferSmallImage);
    R_OPTIONAL(ERentalOfferImageStyle, OfferImageStyle, ERentalOfferImageStyle::Default);

    R_OPTIONAL(TRenterOfferCommonInfo, RenterOfferCommonInfo);
    R_OPTIONAL(TString, OfferTimezone, DefaultOfferTimezone);
    R_OPTIONAL(TString, DateFormat, DefaultDateFormat);
    R_OPTIONAL(TString, TimeFormat, DefaultTimeFormat);
    R_OPTIONAL(TRentalEconomy::TTariffRecommendation, TariffRecommendation);
    R_OPTIONAL(ui64, TotalDailyCost);
    R_FIELD(TOptions, Options);
    R_OPTIONAL(TRentalEconomy::TOptionsRecommendation, OptionsRecommendation);
    R_OPTIONAL(TRentalEconomy::TInsurancesRecommendation, InsurancesRecommendation);
    R_OPTIONAL(TRentalEconomy::TLimitsRecommendation, LimitsRecommendation);
    R_OPTIONAL(TDuration, StartRentalTimeout);

private:
    static TFactory::TRegistrator<TRentalOffer> Registrator;
};
using TRentalOfferReport = TPackOfferReport;

struct TRentalOfferCurrency {
public:
    R_FIELD(TString, Id);

public:
    NJson::TJsonValue GetReport(ELocalization locale, const NDrive::IServer& server) const;

public:
    DECLARE_FIELDS(
        Field(Id, "id")
    )
};

using TRentalCurrencies = TVector<TRentalOfferCurrency>;

struct TRentalOfferStatus {
public:
    R_FIELD(TString, Id);

public:
    NJson::TJsonValue GetReport(ELocalization locale, const NDrive::IServer& server) const;
};

using TRentalOfferStatuses = TVector<TRentalOfferStatus>;

template <class T>
using TRentalOfferVariable = TOfferVariable<T>;

class TRentalOfferBuilder: public IOfferBuilderAction {
private:
    using TBase = IOfferBuilderAction;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "rental_offer_builder";
    }

    virtual TString GetType() const override {
        return GetTypeName();
    }

    bool CheckDeviceTagsPerformer(const TTaggedObject& taggedObject, const TUserPermissions& permissions, bool checkEmptyPerformer) const override;
    void PatchReport(NJson::TJsonValue& report, ELocalization locale, const NDrive::IServer& server, NDrive::TEntitySession& tx, const TUserPermissions::TPtr permissions) const;

protected:
    EOfferCorrectorResult DoBuildOffers(
        const TUserPermissions& permissions,
        TVector<IOfferReport::TPtr>& offers,
        const TOffersBuildingContext& context,
        const NDrive::IServer* server,
        NDrive::TInfoEntitySession& session
    ) const override;

    EOfferCorrectorResult DoCheckOfferConditions(
        const TOffersBuildingContext& context,
        const TUserPermissions& permissions
    ) const override;

    NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override;

    bool DeserializeSpecialsFromJson(const NJson::TJsonValue& value) override;
    NJson::TJsonValue SerializeSpecialsToJson() const override;

private:
    static TRentalInsurances CreateDefaultInsurances();
    static TRentalOfferVariable<TString> CreateDefaultComment();
    static TRentalOfferStatuses CreateDefaultStatuses();
    static TRentalOfferVariable<ui64> CreateDefaultLimitKmPerDay();
    static TRentalOfferVariable<ui64> CreateDefaultOverrunCostPerKm();
    static TRentalOfferVariable<ui64> CreateDefaultPayment();
    static TRentalOfferVariable<ui64> CreateDefaultDeposit();

    EOfferCorrectorResult SetOfferTimeInfo(TRentalOffer& offer,
                                           TMap<TString, NJson::TJsonValue>& variables,
                                           const TInstant requestStartTime) const;

private:
    // legacy options
    // will be deleted after frontend update
    R_FIELD(bool, AllowChildSeat, false);
    R_FIELD(bool, AllowRoofRack, false);
    R_FIELD(bool, AllowGPS, false);
    R_FIELD(bool, AllowSnowChains, false);
    R_FIELD(bool, AllowEntryToEcoZonesInGermany, false);

    R_FIELD(TRentalInsurances, AllowedInsurances, CreateDefaultInsurances());

    R_FIELD(TRentalOfferLocations, AllowedDeliveryLocations);
    R_FIELD(TRentalOfferLocations, AllowedReturnLocations);

    R_FIELD(bool, AllowComment, true);
    R_FIELD(TRentalOfferStatuses, AllowedStatuses, CreateDefaultStatuses());

    R_FIELD(bool, AllowLimitKmPerDay, true);
    R_FIELD(bool, AllowOverrunCostPerKm, true);

    R_FIELD(bool, AllowTotalPayment, true);
    R_FIELD(bool, AllowDeposit, true);
    R_FIELD(TRentalCurrencies, AllowedCurrencies);

    R_FIELD(TString, OfferImage);
    R_FIELD(TString, OfferSmallImage);
    R_FIELD(ERentalOfferImageStyle, OfferImageStyle, ERentalOfferImageStyle::Default);

    R_FIELD(TRentalOfferVariable<TString>, Comment, CreateDefaultComment());
    R_FIELD(TRentalOfferVariable<ui64>, LimitKmPerDay, CreateDefaultLimitKmPerDay());
    R_FIELD(TRentalOfferVariable<ui64>, OverrunCostPerKm, CreateDefaultOverrunCostPerKm());
    R_FIELD(TRentalOfferVariable<ui64>, TotalPayment, CreateDefaultPayment());
    R_FIELD(TRentalOfferVariable<ui64>, Deposit, CreateDefaultDeposit());
    R_FIELD(TString, OfferTimezone, "Europe/Moscow");
    R_FIELD(TString, DateFormat, TRentalOffer::DefaultDateFormat);
    R_FIELD(TString, TimeFormat, TRentalOffer::DefaultTimeFormat);
    R_FIELD(TString, Agreement, GetTypeName());
    R_FIELD(TRentalOffer::TOptions, Options);
    R_FIELD(TDuration, StartRentalTimeout, TDuration::Minutes(15));

public:
    DECLARE_FIELDS(
        // legacy options
        // will be deleted after frontend update
        Field(AllowChildSeat, "child_seat"),
        Field(AllowRoofRack, "roof_rack"),
        Field(AllowGPS, "gps"),
        Field(AllowSnowChains, "snow_chains"),
        Field(AllowEntryToEcoZonesInGermany, "entry_to_eco_zones_in_germany"),

        Field(AllowedInsurances, "insurance_types"),
        Field(AllowedDeliveryLocations, "delivery_locations"),
        Field(AllowedReturnLocations, "return_locations"),
        Field(AllowComment, "comment"),
        Field(AllowedStatuses, "statuses"),
        Field(AllowLimitKmPerDay, "limit_km_per_day"),
        Field(AllowOverrunCostPerKm, "overrun_cost_per_km"),
        Field(AllowTotalPayment, "total_payment"),
        Field(AllowDeposit, "deposit"),
        Field(AllowedCurrencies, "currencies"),
        Field(OfferImage, "offer_image"),
        Field(OfferSmallImage, "offer_small_image"),
        Field(OfferImageStyle, "offer_image_style"),
        Field(OfferTimezone, "offer_timezone"),
        Field(DateFormat, "date_format"),
        Field(TimeFormat, "time_format"),
        Field(Agreement, "agreement"),
        Field(Options, "options")
    )

private:
    R_FIELD(TString, OfferHolderTag, TRentalOfferHolderTag::Type());
    R_FIELD(TRentalOffer::TRenterOfferCommonInfo, RenterOfferCommonInfo);

private:
    R_FIELD(TRentalEconomy, Economy);

private:
    static TFactory::TRegistrator<TRentalOfferBuilder> Registrator;
};
