#include "standard_with_discount_area.h"

#include <drive/backend/areas/location.h>
#include <drive/backend/data/area_tags.h>
#include <drive/backend/data/model.h>
#include <drive/backend/models/storage.h>
#include <drive/backend/offers/ranking/calcer.h>

#include <rtline/library/geometry/coord.h>

TUserAction::TFactory::TRegistrator<TStandardWithDiscountAreaOfferBuilder> TStandardWithDiscountAreaOfferBuilder::Registrator(TStandardWithDiscountAreaOfferBuilder::GetTypeName());

EOfferCorrectorResult TStandardWithDiscountAreaOfferBuilder::DoCheckOfferConditions(const TOffersBuildingContext& context, const TUserPermissions& /*permissions*/) const {
    if (!TagsFilter) {
        return EOfferCorrectorResult::Success;
    }
    if (!context.HasCarId()) {
        return EOfferCorrectorResult::Unimplemented;
    }
    TMaybe<TTaggedObject> td = context.GetTaggedCar();
    if (!td) {
        return EOfferCorrectorResult::Problems;
    }
    if (!TagsFilter.IsMatching(td->GetTags())) {
        return EOfferCorrectorResult::Unimplemented;
    }
    return EOfferCorrectorResult::Success;
}

EOfferCorrectorResult TStandardWithDiscountAreaOfferBuilder::BuildOffers(const TUserPermissions& permissions, const TUserActions& correctors, TVector<IOfferReport::TPtr>& offers, const TOffersBuildingContext& context, const NDrive::IServer* server, NDrive::TInfoEntitySession& session) const {
    bool needOffers = context.GetSetting<bool>("offers.need_standard_with_discount_area_offers").GetOrElse(false);
    if (!needOffers) {
        return EOfferCorrectorResult::Unimplemented;
    }
    auto constructor = GetStandartOfferConstructor(server, BaseOfferBuilder);
    TVector<IOfferReport::TPtr> stOffers;
    auto buildingResult = constructor->BuildOffers(permissions, correctors, stOffers, context, server, session);
    if (buildingResult != EOfferCorrectorResult::Success) {
        return buildingResult;
    }
    if (stOffers.size() == 0) {
        session.AddErrorMessage("TStandardWithDiscountAreaOfferBuilder::BuildOffers", "built 0 standard offers");
        return EOfferCorrectorResult::Problems;
    }

    const TStandartOfferReport& stOfferReport = *dynamic_cast<TStandartOfferReport*>(stOffers[0].Get());
    const TStandartOffer& stOffer = *stOfferReport.GetOfferAs<TStandartOffer>();

    return BuildStandardWithDiscountAreaOffers(stOffer, offers, context, server, session, permissions);
}

EOfferCorrectorResult TStandardWithDiscountAreaOfferBuilder::BuildStandardWithDiscountAreaOffers(const TStandartOffer& stOffer, TVector<IOfferReport::TPtr>& offers, const TOffersBuildingContext& context, const NDrive::IServer* server, NDrive::TInfoEntitySession& session, const TUserPermissions& permissions) const {
    EOfferCorrectorResult result = CheckOfferConditions(context, permissions);
    if (result != EOfferCorrectorResult::Success) {
        return result;
    }
    TVector<TStandardWithDiscountAreaOfferBuilder::TAreaDescription> areas = GetDestinationAreas(context, *server);
    TVector<TOffersBuildingContext::TDestinationDescription> descriptions(areas.size());
    for (size_t i = 0; i < areas.size(); ++i) {
        descriptions[i].SetCoordinate(areas[i].Finish);
        descriptions[i].Prefetch(context.GetUserHistoryContextRef());
    }
    TVector<TOffersBuildingContext::TDestination> destinations;
    destinations.reserve(areas.size());
    for (auto&& description : descriptions) {
        destinations.emplace_back(description, context);
    }
    for (auto&& destination : destinations) {
        destination.Prefetch();
    }
    for (size_t i = 0; i < areas.size(); ++i) {
        auto discountedOffer = BuildStandardWithDiscountAreaOffer(stOffer, areas[i], destinations[i], server, session);
        if (!discountedOffer) {
            session.AddErrorMessage("TStandardWithDiscountAreaOfferBuilder::BuildOffers", "couldn't build standard with discounted area offer with name " + areas[i].FinishName);
            return EOfferCorrectorResult::Problems;
        }
        offers.emplace_back(discountedOffer);
    }
    NDrive::LogCreatedOffers(offers);
    return EOfferCorrectorResult::Success;
}

TAtomicSharedPtr<TStandardWithDiscountAreaOfferReport> TStandardWithDiscountAreaOfferBuilder::BuildStandardWithDiscountAreaOffer(const TStandartOffer& stOffer, const TStandardWithDiscountAreaOfferBuilder::TAreaDescription& description, const TOffersBuildingContext::TDestination& destination, const NDrive::IServer* server, NDrive::TInfoEntitySession& session) const {
    auto discountedOffer = MakeAtomicShared<TStandardWithDiscountAreaOffer>(TStandardWithDiscountAreaOffer(stOffer));
    discountedOffer->SetOfferId(ICommonOffer::CreateOfferId());
    discountedOffer->SetBehaviourConstructorId(GetName());
    discountedOffer->SetPriceConstructorId(GetName());
    discountedOffer->SetFinish(description.Finish);
    discountedOffer->SetFinishName(description.FinishName);
    discountedOffer->SetFinishArea(description.FinishArea);
    discountedOffer->SetFinishAreaId(description.FinishAreaId);
    discountedOffer->SetDiscount(description.Discount);
    {
        const auto& coordinate = destination.GetDestination().GetCoordinate();
        const NDrive::TGeoFeatures* geoFeatures = destination.GetDestination().GetGeoFeatures();
        const NDrive::TGeoFeatures* geobaseFeatures = destination.GetDestination().GetGeobaseFeatures();
        const NDrive::TUserGeoFeatures* userGeoFeatures = destination.GetDestination().GetUserGeoFeatures();
        const NDrive::TUserGeoFeatures* userGeobaseFeatures = destination.GetDestination().GetUserGeobaseFeatures();
        const NDrive::TUserDoubleGeoFeatures* userDoubleGeoFeatures = destination.GetUserDoubleGeoFeatures();
        const NDrive::TUserDoubleGeoFeatures* userDoubleGeobaseFeatures = destination.GetUserDoubleGeobaseFeatures();
        double distance = NDrive::GeoFeaturesDefaultValue;
        double duration = NDrive::GeoFeaturesDefaultValue;
        TString finishAreaId = description.FinishAreaId;
        if (auto optionalRoute = destination.GetRoute(); optionalRoute && *optionalRoute) {
            const auto& route = optionalRoute->GetRef();
            distance = route.Length;
            duration = route.Time;
        }
        double score = 2;
        ui64 geobaseId = destination.GetDestination().GetGeobaseId();
        NDrive::CalcDestinationFeatures(discountedOffer->MutableFeatures(), coordinate, geoFeatures, userGeoFeatures, distance, duration);
        NDrive::CalcDestinationFeatures(discountedOffer->MutableFeatures(), geobaseId, geobaseFeatures, userGeobaseFeatures);
        NDrive::CalcDestinationFeatures(discountedOffer->MutableFeatures(), userDoubleGeoFeatures);
        NDrive::CalcDestinationFeatures(discountedOffer->MutableFeatures(), geobaseId, userDoubleGeobaseFeatures);
        NDrive::CalcBestDestinationFeatures(discountedOffer->MutableFeatures(), score);
        NDrive::CalcFinishAreaId(discountedOffer->MutableFeatures(), finishAreaId);
    }
    if (DiscountModel) {
        auto models = server->GetModelsStorage();
        if (!models) {
            session.SetErrorInfo("TStandardWithDiscountAreaOfferBuilder::BuildStandardWithDiscountAreaOffers", "null ModelsStorage", EDriveSessionResult::InconsistencySystem);
            return nullptr;
        }
        auto model = models->GetOfferModel(DiscountModel);
        if (!model) {
            session.SetErrorInfo("TStandardWithDiscountAreaOfferBuilder::BuildStandardWithDiscountAreaOffers", TStringBuilder() << "DiscontedModel " << DiscountModel << " is missing", EDriveSessionResult::InconsistencySystem);
            return nullptr;
        }
        discountedOffer->ApplyDiscountModel(*model);
    }
    discountedOffer->RebuildDiscountedOffer();
    auto report = MakeAtomicShared<TStandardWithDiscountAreaOfferReport>(discountedOffer, nullptr);
    report->RecalculateFeatures();
    return report;
}

TMaybe<TUserOfferContext::TDestinationDescription> TStandardWithDiscountAreaOfferBuilder::GetDestination(const TOffersBuildingContext& context, const NDrive::IServer& server) const {
    TMaybe<TUserOfferContext::TDestinationDescription> destination;
    if (context.GetUserHistoryContextUnsafe().HasUserDestination()) {
        destination = context.GetUserHistoryContextUnsafe().GetUserDestinationUnsafe();
    } else if (!context.GetDestinations().empty()){
        destination = context.GetDestinations()[0].GetDestination();
    }
    if (destination) {
        if (!FinishAreaTagsFilter) {
            auto filter = NDrive::CreateFinishAreaTagsFilter(context, server, GetType());
            if (filter) {
                FinishAreaTagsFilter = *filter;
            }
        }
        auto tags = server.GetDriveAPI()->GetAreasDB()->GetTagsInPoint(destination->GetCoordinate());
        if (FinishAreaTagsFilter) {
            return FinishAreaTagsFilter.IsMatching(tags) ? destination : TMaybe<TUserOfferContext::TDestinationDescription>{};
        } else {
            bool hasAllowDropTag = !server.GetDriveAPI()->GetAreasDB()->CheckGeoTags(destination->GetCoordinate(), { NDrive::DefaultAllowDropLocationTag }, TInstant::Now()).empty();
            return hasAllowDropTag ? destination : TMaybe<TUserOfferContext::TDestinationDescription>{};
        }
    }
    return {};
}

TVector<TStandardWithDiscountAreaOfferBuilder::TAreaDescription> TStandardWithDiscountAreaOfferBuilder::GetDestinationAreas(const TOffersBuildingContext& context, const NDrive::IServer& server) const {
    TVector<TStandardWithDiscountAreaOfferBuilder::TAreaDescription> areas;
    if (TagName) {
        auto destination = GetDestination(context, server);
        if (!destination) {
            return {};
        }
        FetchAreaFromPolygonPoint(areas, destination->GetCoordinate(), server);
        if (areas.size() == 1) {
            FetchAreasFromKNearest(areas, areas[0].Finish, server, context);
        } else {
            FetchAreaFromUserPoint(areas, *destination);
        }
    } else {
        if (context.GetUserHistoryContextUnsafe().HasUserDestination()) {
            auto destination = context.GetUserHistoryContextUnsafe().GetUserDestinationUnsafe();
            FetchAreaFromUserPoint(areas, destination);
        } else {
            FetchAreasFromSuggest(areas, context);
        }
    }
    return areas;
}

NDrive::TScheme TStandardWithDiscountAreaOfferBuilder::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);

    auto models = server->GetModelsStorage();
    auto modelNames = models ? models->ListOfferModels() : TVector<TString>{};
    result.Add<TFSVariants>("discount_model", "Модель для вычисления скидки").SetVariants(modelNames);

    TVector<TDBAction> actions = server->GetDriveAPI()->GetRolesManager()->GetActionsDB().GetCachedObjectsVector();
    TSet<TString> offerConstructorNames;
    for (auto&& action : actions) {
        const TStandartOfferConstructor* stOfferConstructor = action.GetAs<TStandartOfferConstructor>();
        if (stOfferConstructor && stOfferConstructor->GetType() == TStandartOfferConstructor::GetTypeName()) {
            offerConstructorNames.emplace(stOfferConstructor->GetName());
        }
    }
    result.Add<TFSVariants>("base_offer_builder", "Базовый конструктор стандартного оффера").SetVariants(offerConstructorNames);
    auto tagNames = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTagNames({TStandardWithDiscountAreaOfferAreaTag::TypeName});
    result.Add<TFSVariants>("tag_name", "Имя тега со скидкой в полигоне").SetVariants(tagNames);
    {
        auto gTab = result.StartTabGuard("activation");
        result.Add<TFSString>("tags_filter", "Фильтр активации предложения по тегам машины в стандартном формате");
    }
    {
        auto gTab = result.StartTabGuard("zones");
        result.Add<TFSString>("finish_area_tags_filter", "Фильтр по тегам зоны завершения аренды в стандартном формате");
    }
    return result;
}

bool TStandardWithDiscountAreaOfferBuilder::DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonValue) {
    if (!TBase::DeserializeSpecialsFromJson(jsonValue)) {
        return false;
    }
    TStringBuilder sb;
    if (jsonValue.Has("finish_area_tags_filter")) {
        const NJson::TJsonValue& tagsFilter = jsonValue["finish_area_tags_filter"];
        if (!tagsFilter.IsString() || !FinishAreaTagsFilter.DeserializeFromString(tagsFilter.GetString())) {
            return false;
        }
    }
    if (jsonValue.Has("tags_filter")) {
        const NJson::TJsonValue& tagsFilter = jsonValue["tags_filter"];
        if (!tagsFilter.IsString()) {
            return false;
        }
        sb << tagsFilter.GetString();
    }
    if (!TagsFilter.DeserializeFromString(sb)) {
        return false;
    }
    return NJson::ParseField(jsonValue["discount_model"], DiscountModel) &&
        NJson::ParseField(jsonValue["base_offer_builder"], BaseOfferBuilder) &&
        NJson::ParseField(jsonValue["tag_name"], TagName);
}

NJson::TJsonValue TStandardWithDiscountAreaOfferBuilder::SerializeSpecialsToJson() const {
    NJson::TJsonValue jsonValue = TBase::SerializeSpecialsToJson();
    jsonValue.InsertValue("discount_model", DiscountModel);
    jsonValue.InsertValue("base_offer_builder", BaseOfferBuilder);
    jsonValue.InsertValue("tag_name", TagName);
    jsonValue.InsertValue("tags_filter", TagsFilter.ToString());
    jsonValue.InsertValue("finish_area_tags_filter", FinishAreaTagsFilter.ToString());
    return jsonValue;
}

TStandardWithDiscountAreaOfferBuilder::TAreaDescription::TAreaDescription(const TUserOfferContext::TDestinationDescription& destinationDescription)
    : Finish(destinationDescription.GetCoordinate())
    , FinishName(destinationDescription.GetName())
    , Discount(0)
{
    if (const auto area = destinationDescription.GetFinishArea(); area && area->HasFinishArea()) {
        auto finishArea = area->GetFinishAreaUnsafe();
        if (finishArea.size()) {
            FinishArea = finishArea;
        }
    }
}

void TStandardWithDiscountAreaOfferBuilder::FetchAreaFromUserPoint(TVector<TAreaDescription>& areas, const TUserOfferContext::TDestinationDescription& destinationDescription) const {
    areas.emplace_back(destinationDescription);
}

void TStandardWithDiscountAreaOfferBuilder::FetchAreasFromSuggest(TVector<TAreaDescription>& areas, const TOffersBuildingContext& context) const {
    for (auto&& destination : context.GetDestinations()) {
        auto destinationDescription = destination.GetDestination();
        areas.emplace_back(destinationDescription);
    }
}

void TStandardWithDiscountAreaOfferBuilder::FetchAreaFromPolygonPoint(TVector<TAreaDescription>& areas, const TGeoCoord& coord, const NDrive::IServer& server) const {
    auto areaActor = [&areas, this](const TFullAreaInfo& areaInfo) {
        auto areaDescription = GetAreaDescriptionByTag(areaInfo);
        if (areaDescription) {
            areas.emplace_back(*areaDescription);
            return false;
        }
        return true;
    };
    Yensured(server.GetDriveAPI()->GetAreasDB())->ProcessAreasInPoint(coord, areaActor);
}

void TStandardWithDiscountAreaOfferBuilder::FetchAreasFromKNearest(TVector<TAreaDescription>& areas, const TGeoCoord& coord, const NDrive::IServer& server, const TOffersBuildingContext& context) const {
    i32 amount = context.GetSetting<i32>("standard_with_discount_area_offer.number_of_k_nearest_areas").GetOrElse(0);
    double maxDistance = context.GetSetting<double>("standard_with_discount_area_offer.max_distance_to_nearest_area").GetOrElse(50000);
    if (amount == 0) {
        return;
    }
    TMultiMap<double, TAreaDescription> polygons;
    auto snapshot = Yensured(server.GetDriveAPI()->GetAreasDB())->GetSnapshot();
    if (!snapshot) {
        return;
    }
    for (auto&& areaInfo : *snapshot) {
        auto areaDescription = GetAreaDescriptionByTag(areaInfo);
        if (areaDescription) {
            double minDistance = 10;
            double distance = areaDescription->Finish.GetLengthTo(coord);
            if (distance > minDistance && distance <= maxDistance) {
                polygons.insert({distance, *areaDescription});
            }
        }
    }

    int fetchedPolygons = 0;
    for (const auto& [distance, area] : polygons) {
        if (fetchedPolygons == amount) {
            break;
        }
        areas.emplace_back(area);
        ++fetchedPolygons;
    }
}

TMaybe<TStandardWithDiscountAreaOfferBuilder::TAreaDescription> TStandardWithDiscountAreaOfferBuilder::GetAreaDescriptionByTag(const TFullAreaInfo& areaInfo) const {
    auto tags = areaInfo.GetTagsByClass<TStandardWithDiscountAreaOfferAreaTag>();
    for (auto& tag : tags) {
        auto areaTag = tag.MutableTagAs<TStandardWithDiscountAreaOfferAreaTag>();
        if (areaTag && areaTag->GetName() == TagName) {
            TAreaDescription area;
            area.Discount = areaTag->GetDiscount() * 10000;
            area.FinishArea = areaInfo.GetArea().GetCoords();
            area.FinishAreaId = areaInfo.GetArea().GetIdentifier();
            area.Finish = TGeoCoord::CalcCenter(areaInfo.GetArea().GetCoords());
            area.FinishName = areaInfo.GetArea().GetTitle();
            return area;
        }
    }
    return {};
}
