#include "standart.h"

#include "flows_control.h"

#include <drive/backend/offers/context.h>
#include <drive/backend/offers/notifications.h>
#include <drive/backend/offers/ranking/calcer.h>
#include <drive/backend/offers/ranking/model.h>

#include <drive/backend/areas/areas.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/model.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/models/storage.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/saas/api.h>
#include <drive/backend/users/user.h>

#include <drive/library/cpp/user_events_api/client.h>

#include <kernel/daemon/common/time_guard.h>

#include <library/cpp/mediator/global_notifications/system_status.h>

#include <rtline/library/geometry/coord.h>
#include <rtline/library/time_restriction/time_restriction.h>
#include <rtline/protos/proto_helper.h>
#include <rtline/util/json_processing.h>
#include <rtline/util/algorithm/ptr.h>

#include <util/generic/guid.h>
#include <util/string/builder.h>

bool TStandartOfferConstructor::BuildAdditionalDuration(TStandartOffer& offer, const TOffersBuildingContext& context, const TUserPermissions& permissions, NDrive::TInfoEntitySession& session) const {
    if (!context.HasUserHistoryContext()) {
        return false;
    }
    if (!context.GetUserHistoryContextUnsafe().GetNeedHistoryFreeTimeFees()) {
        return true;
    }
    if (!context.GetUserHistoryContextUnsafe().FetchFeesInfo()) {
        session.SetErrorInfo("StandartOfferConstructor::BuildAdditionalDuration", "cannot FetchFeesInfo");
        return false;
    }

    if (UseDeposit && (!context.GetUserHistoryContextUnsafe().GetPricedRidesCountDef(0) || context.GetSetting<bool>("offers.standart.use_deposit").GetOrElse(false))) {
        offer.SetUseDeposit(true);
        offer.SetDepositAmount(GetDepositDefault());
    } else {
        offer.SetUseDeposit(false);
    }
    for (auto&& i : context.GetUserHistoryContextUnsafe().GetAdditionalDurationDiscounts()) {
        offer.AddDiscount(i);
    }
    {
        TDiscount discount;
        discount.SetIdentifier("walking_time");
        discount.SetVisible(false);
        {
            TDiscount::TDiscountDetails d;
            const TDuration walkingDuration = context.DetermWalkingTime(permissions);
            d.SetAdditionalTime(walkingDuration.Seconds()).SetTagName("old_state_reservation");
            discount.AddDetails(d);
        }
        offer.AddDiscount(std::move(discount));
    }
    return true;
}

EOfferCorrectorResult TStandartOfferConstructor::DoCheckOfferConditions(const TOffersBuildingContext& context, const TUserPermissions& /*permissions*/) const {
    if (!context.HasCarId()) {
        return EOfferCorrectorResult::Unimplemented;
    }
    TMaybe<TTaggedObject> td = context.GetTaggedCar();
    if (!td) {
        return EOfferCorrectorResult::Problems;
    }
    if (!TagsFilter.IsMatching(td->GetTags())) {
        return EOfferCorrectorResult::Unimplemented;
    }
    return EOfferCorrectorResult::Success;
}

bool IsTaxiHighDemand(const NDrive::TTaxiSurgeCalculator::TResult& surge, double value) {
    for (auto&& element : surge.Classes) {
        if (element.Name == "econom" && element.Value >= value) {
            return true;
        }
    }
    return false;
}

EOfferCorrectorResult TStandartOfferConstructor::DoBuildOffers(const TUserPermissions& permissions, TVector<IOfferReport::TPtr>& results, const TOffersBuildingContext& context, const NDrive::IServer* server, NDrive::TInfoEntitySession& session) const {
    if (!context.HasCarId() || !context.HasUserHistoryContext()) {
        return EOfferCorrectorResult::BuildingProblems;
    }
    const auto gg = context.BuildEventGuard("price_constr");
    const ISettings& settings = server->GetSettings();
    const TString& carId = context.GetCarIdUnsafe();
    TUserActions priceActions;
    if (PriceSource != EPriceSource::InternalOnly) {
        if (PriceOfferConstructor) {
            auto action = server->GetDriveAPI()->GetRolesManager()->GetAction(PriceOfferConstructor).GetOrElse({});
            if (!action) {
                session.AddErrorMessage("StandartOfferConstructor::DoBuildOffers", "cannot find PriceOfferConstructor " + PriceOfferConstructor);
            }
            priceActions.push_back(action.Impl());
        } else {
            auto deduced = context.GetPriceConstructorsForBehaviour(permissions);
            if (PriceOfferConstructors.empty()) {
                priceActions = std::move(deduced);
            } else {
                for (auto&& action : deduced) {
                    if (!PriceOfferConstructors.contains(action->GetName())) {
                        continue;
                    }
                    priceActions.push_back(action);
                }
            }
        }
    }
    if (priceActions.empty()) {
        if (PriceSource == EPriceSource::ExternalOnly) {
            return EOfferCorrectorResult::Unimplemented;
        }
        priceActions.emplace_back(nullptr);
    }
    double taxiHighDemandSurge = context.GetSetting<double>("offers.standart_offer.taxi_high_demand_surge").GetOrElse(1.6);
    double taxiHighDemandCarSurge = context.GetSetting<double>("offers.standart_offer.taxi_high_demand_car_surge").GetOrElse(1.6);
    for (auto&& pa : priceActions) {
        auto eg = context.BuildEventGuard("price_construction");
        TFullPricesContext fpContext;
        auto pc = std::dynamic_pointer_cast<const TPriceOfferConstructor>(pa);
        if (!pc) {
            ui32 ridingPrice;
            ui32 parkingPrice;
            ui32 kmPrice;
            if (!GetPrices(server, ridingPrice, parkingPrice, kmPrice)) {
                return EOfferCorrectorResult::BuildingProblems;
            }
            fpContext.MutableRiding().SetPrice(ridingPrice);
            fpContext.MutableRiding().SetPriceModelName(PriceModel);
            fpContext.MutableParking().SetPrice(parkingPrice);
            fpContext.MutableParking().SetPriceModelName(ParkingPriceModel);
            if (UseKmPrices) {
                fpContext.MutableRiding() = fpContext.MutableParking();
                fpContext.MutableKm().SetPrice(kmPrice);
            } else {
                fpContext.MutableKm().Clear();
            }
        } else {
            if (context.GetUserHistoryContextUnsafe().GetFilterAccounts() && !context.GetUserHistoryContextUnsafe().CheckRequestAccountId(pc->GetGrouppingTags(), {"card"})) {
                continue;
            }

            fpContext = pc->BuildFullContext();
            if (UseKmPrices) {
                fpContext.MutableRiding() = fpContext.MutableParking();
            } else {
                fpContext.MutableKm().Clear();
            }
            if (!context.GetEnableAcceptanceCost()) {
                fpContext.MutableAcceptance().Clear();
            }
        }

        auto resultOffer = MakeAtomicShared<TStandartOffer>(fpContext);
        auto resultOfferReport = MakeAtomicShared<TStandartOfferReport>(resultOffer, pc);
        if (pc && pc->GetPaymentDiscretization()) {
            resultOffer->SetPaymentDiscretization(pc->GetPaymentDiscretization());
        } else {
            resultOffer->SetPaymentDiscretization(GetPaymentDiscretization());
        }
        auto cashbackPercent = CashbackPercent;
        if (!cashbackPercent) {
            auto cashbackDefaultPercentKey = "offer_builder." + GetType() + ".cashback_default_percent";
            cashbackPercent = permissions.GetSetting<ui32>(settings, cashbackDefaultPercentKey);
            auto cashbackInfo = permissions.GetSetting<TString>(settings, "offer_builder." + GetType() + ".cashback_info");
            NJson::TJsonValue cashbackInfoJson;
            if (cashbackPercent && *cashbackPercent > 0 && cashbackInfo && NJson::ReadJsonTree(*cashbackInfo, &cashbackInfoJson)) {
                resultOffer->OptionalCashbackInfo() = cashbackInfoJson;
            }
        }
        resultOffer->SetCashbackPercent(cashbackPercent.GetOrElse(0));
        resultOffer->SetOnlyOriginalOfferCashback(GetOnlyOriginalOfferCashback());

        auto increasedCashbackPercent = permissions.GetSetting<ui32>(settings, "offer_builder." + GetType() + ".cashback_increased_percent");
        resultOffer->SetIncreasedCashbackPercent(increasedCashbackPercent);
        resultOffer->SetDebtThreshold(GetMinimalDebtThreshold());
        resultOffer->SetFuelingEnabled(FuelingEnabled);
        {
            auto g = context.BuildEventGuard("additional_durations");
            if (!BuildAdditionalDuration(*resultOffer, context, permissions, session)) {
                return EOfferCorrectorResult::Problems;
            }
        }

        resultOffer->SetObjectId(carId).SetUserId(permissions.GetUserId()).SetName(GetDescription());
        resultOfferReport->SetShortDescription(GetShortDescription());
        resultOffer->SetUseDefaultShortDescriptions(GetUseDefaultShortDescriptions());
        resultOffer->SetUseRounding(permissions.GetSetting<bool>("offer.standart.rounding.enabled", false));
        resultOffer->SetShortName(GetShortName());
        resultOffer->SetSubName(GetSubName());

        bool intercity = GetGrouppingTags().contains(IntercityOfferTag);
        if (FinishAreaTagsFilter) {
            resultOffer->SetFinishAreaTagsFilter(FinishAreaTagsFilter);
        } else if (!intercity) {
            auto filter = NDrive::CreateFinishAreaTagsFilter(context, *server, GetType());
            if (filter) {
                resultOffer->SetFinishAreaTagsFilter(*filter);
            }
        }
        if (RidingAreaTagsFilter) {
            resultOffer->SetRidingAreaTagsFilter(RidingAreaTagsFilter);
        } else if (!intercity) {
            auto filter = NDrive::CreateRidingAreaTagsFilter(context, *server, GetType());
            if (filter) {
                resultOffer->SetRidingAreaTagsFilter(*filter);
            }
        }

        {
            auto g = context.BuildEventGuard("features");
            {
                const auto geobaseId = context.GetGeobaseId();
                const auto geoFeatures = context.GetGeoFeatures();
                const auto geobaseFeatures = context.GetGeobaseFeatures();
                NDrive::CalcGeoFeatures(context.MutableFeatures(), geoFeatures);
                NDrive::CalcGeoFeatures(context.MutableFeatures(), geobaseId, geobaseFeatures);
            }
            {
                const auto* userFeatures = context.GetUserFeatures();
                NDrive::CalcUserFeatures(
                    context.MutableFeatures(),
                    permissions.GetUserId(),
                    userFeatures ? *userFeatures : Default<NDrive::TUserFeatures>()
                );
            }
            {
                const auto geobaseId = context.GetGeobaseId();
                const auto userGeoFeatures = context.GetUserGeoFeatures();
                const auto userGeobaseFeatures = context.GetUserGeobaseFeatures();
                NDrive::CalcSourceFeatures(context.MutableFeatures(), userGeoFeatures);
                NDrive::CalcSourceFeatures(context.MutableFeatures(), geobaseId, userGeobaseFeatures);
            }
            if (const auto walkingDuration = context.GetWalkingDuration(GetPedestrianSpeed())) {
                NDrive::CalcUserFeatures(context.MutableFeatures(), *walkingDuration);
                resultOfferReport->SetWalkingDuration(*walkingDuration);
            }
            if (auto&& surge = context.GetTaxiSurge()) {
                NDrive::CalcOfferTaxiCarSurgeFeatures(context.MutableFeatures(), *surge);
                if (IsTaxiHighDemand(*surge, taxiHighDemandCarSurge)) {
                    auto notification = MakeHolder<NDrive::TOfferNotification>("taxi_high_demand_car");
                    context.MutableNotifications().AddNotification(std::move(notification));
                }
            }
            if (auto&& surgeAreaId = context.GetSurgeAreaId()) {
                NDrive::CalcOfferRtmrSurgeAreaId(context.MutableFeatures(), surgeAreaId);
                if (auto&& surge = context.GetRtmrAreaSurge()) {
                    NDrive::CalcOfferRtmrAreaSurge(context.MutableFeatures(), *surge);
                }
                if (auto&& surge = context.GetRtmrAreaExtendedSurge()) {
                    NDrive::CalcOfferRtmrAreaExtendedSurge(context.MutableFeatures(), *surge);
                }
            }
            if (context.HasUserHistoryContext()) {
                if (auto&& surge = context.GetUserHistoryContextRef().GetTaxiSurge()) {
                    NDrive::CalcOfferTaxiSurgeFeatures(context.MutableFeatures(), *surge);
                    if (IsTaxiHighDemand(*surge, taxiHighDemandSurge)) {
                        auto notification = MakeHolder<NDrive::TOfferNotification>("taxi_high_demand");
                        context.MutableNotifications().AddNotification(std::move(notification));
                    }
                }
                if (auto&& scoring = context.GetUserHistoryContextRef().GetAggressionScoring()) {
                    NDrive::CalcOfferAggressionScoring(context.MutableFeatures(), scoring->GetValue());
                }
                if (auto&& time = context.GetUserHistoryContextRef().GetLastPricedRideTime()) {
                    NDrive::CalcUserLastPricedRideTime(context.MutableFeatures(), *time);
                }
            }
            resultOfferReport->CalculateFeatures(context.GetFeatures());
        }
        {
            auto g = context.BuildEventGuard("model_usage");
            resultOfferReport->ApplyModels(server);
            resultOfferReport->RecalcPrices(server);
        }

        if (context.HasInsuranceType() && !context.GetForceInsuranceType()) {
            auto g = context.BuildEventGuard("check_insurance_type");
            auto insuranceTypeOption = permissions.GetOptionSafe("insurance_type");
            auto insuranceTypes = insuranceTypeOption ? insuranceTypeOption->GetValues() : TVector<TString>();
            bool insuranceTypeAvailable = std::find(insuranceTypes.begin(), insuranceTypes.end(), context.GetInsuranceTypeRef()) != insuranceTypes.end();
            if (!insuranceTypeAvailable && !context.GetIgnoreUnavailableInsuranceType()) {
                session.SetCode(HTTP_BAD_REQUEST);
                session.SetErrorInfo(
                    "StandartOfferConstructor::DoBuildOffers",
                    "forbidden insurance_type " + context.GetInsuranceTypeRef(),
                    NDrive::MakeError("forbidden_insurance_type")
                );
                return EOfferCorrectorResult::Problems;
            }
            if (!insuranceTypeAvailable) {
                return EOfferCorrectorResult::Unimplemented;
            }
        }

        if (context.HasInsuranceType()) {
            resultOffer->SetInsuranceType(context.GetInsuranceTypeRef());
        } else {
            const IUserOption* insuranceTypeOption = permissions.GetOptionSafe("insurance_type");
            if (insuranceTypeOption) {
                resultOffer->SetInsuranceType(insuranceTypeOption->GetValueOrDefault(context.GetRequestStartTime()));
            }
        }

        {
            auto g = context.BuildEventGuard("context_insurance_types");
            auto insuranceTypes = context.GetInsuranceTypes();
            if (insuranceTypes && insuranceTypes->empty()) {
                if (g) {
                    g->AddEvent("NoAvailableInsuranceTypes");
                }
                continue;
            }
            if (insuranceTypes && !insuranceTypes->contains(resultOffer->GetInsuranceType())) {
                const auto& currentInsuranceType = resultOffer->GetInsuranceType();
                const auto& newInsuranceType = *insuranceTypes->begin();
                if (g) {
                    g->AddEvent(NJson::TMapBuilder
                        ("event", "ContextOverrideInsuranceType")
                        ("from", currentInsuranceType)
                        ("to", newInsuranceType)
                    );
                }
                resultOffer->SetInsuranceType(newInsuranceType);
            }
        }

        resultOffer->SetAgreement(NDrive::GetOfferAgreement(Agreement, context, resultOffer->GetInsuranceType()));

        auto earlyInsuranceType = settings.GetValue<bool>("standart_offer.early_insurance_type").GetOrElse(true);
        auto offer = resultOffer;
        if (offer) {
            if (pc) {
                for (auto&& i : permissions.GetOptionsSafe<TExtendPriceOption>()) {
                    const auto& id = i->GetOptionId();
                    auto value = TString();
                    if (id == InsuranceTypeId) {
                        if (earlyInsuranceType) {
                            value = offer->GetInsuranceType();
                        } else {
                            continue;
                        }
                    } else {
                        value = i->GetValueOrDefault(context.GetRequestStartTime());
                    }

                    const TOptionalPriceComponent* opc = pc->GetEquilibrium().GetOptionalPriceComponent(id + ":" + value);
                    if (opc) {
                        opc->ApplyForContext(UseKmPrices, *offer);
                    }
                }
            }
        }
        auto enableInheritance = context.GetSetting<bool>("offers.standart_offer.inheritance.enabled").GetOrElse(false);
        auto previousOffer = enableInheritance ? context.GetPreviousOffer(GetName()) : nullptr;
        if (previousOffer) {
            auto g = context.BuildEventGuard("process_previous_offer");
            auto previousStandartOffer = std::dynamic_pointer_cast<TStandartOffer>(previousOffer);
            if (previousStandartOffer) {
                TStandartOffer::TInheritedProperties inheritedProperties;
                if (previousStandartOffer->HasInheritedProperties()) {
                    inheritedProperties = previousStandartOffer->GetInheritedPropertiesRef();
                } else {
                    inheritedProperties.Timestamp = previousStandartOffer->GetTimestamp();
                    inheritedProperties.RidingPrice = previousStandartOffer->GetRiding().GetPrice();
                    inheritedProperties.ParkingPrice = previousStandartOffer->GetParking().GetPrice();
                }
                auto lifetimeLimit = permissions.GetSetting<TDuration>(
                    settings,
                    "offers.standart_offer.inheritance.lifetime_limit"
                ).GetOrElse(TDuration::Minutes(5));
                auto lifetime = offer->GetTimestamp() - inheritedProperties.Timestamp;
                bool match =
                    offer->GetInsuranceType() == previousStandartOffer->GetInsuranceType();
                if (lifetime < lifetimeLimit && match) {
                    offer->SetInheritedProperties(inheritedProperties);
                    resultOfferReport->RecalcPrices(server);
                }
            } else {
                NJson::TJsonValue ev = NJson::TMapBuilder
                    ("event", "process_previous_offer_failure")
                    ("offer_id", previousOffer->GetOfferId())
                    ("offer_type", previousOffer->GetTypeName())
                ;
                NDrive::TEventLog::Log("ProcessPreviousOfferFailure", ev);
                if (g) {
                    g->AddEvent(std::move(ev));
                }
            }
        }
        results.emplace_back(resultOfferReport);
    }
    return EOfferCorrectorResult::Success;
}

bool TStandartOfferConstructor::DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonValue) {
    if (!TBase::DeserializeSpecialsFromJson(jsonValue)) {
        return false;
    }
    TStringBuilder sb;
    if (jsonValue.Has("finish_area_tags_filter")) {
        const NJson::TJsonValue& tagsFilter = jsonValue["finish_area_tags_filter"];
        if (!tagsFilter.IsString() || !FinishAreaTagsFilter.DeserializeFromString(tagsFilter.GetString())) {
            return false;
        }
    }
    if (jsonValue.Has("riding_area_tags_filter")) {
        const NJson::TJsonValue& tagsFilter = jsonValue["riding_area_tags_filter"];
        if (!tagsFilter.IsString() || !RidingAreaTagsFilter.DeserializeFromString(tagsFilter.GetString())) {
            return false;
        }
    }
    if (jsonValue.Has("tags_filter")) {
        const NJson::TJsonValue& tagsFilter = jsonValue["tags_filter"];
        if (!tagsFilter.IsString()) {
            return false;
        }
        sb << tagsFilter.GetString();
    } else if (jsonValue.Has("filter_tag_names")) {
        const NJson::TJsonValue& tagNames = jsonValue["filter_tag_names"];
        if (!tagNames.IsArray()) {
            return false;
        }
        for (auto&& i : tagNames.GetArraySafe()) {
            if (!i.IsString()) {
                return false;
            }
            if (!sb.empty()) {
                sb << ",";
            }
            sb << i.GetString();
        }
    } else if (jsonValue.Has("filter_tag_name")) {
        const NJson::TJsonValue& tagName = jsonValue["filter_tag_name"];
        if (!tagName.IsString()) {
            return false;
        }
        sb << tagName.GetString();
    } else {
        return false;
    }
    if (!TagsFilter.DeserializeFromString(sb)) {
        return false;
    }

    JREAD_BOOL_OPT(jsonValue, "use_deposit", UseDeposit);
    JREAD_BOOL_OPT(jsonValue, "use_km_prices", UseKmPrices);
    if (jsonValue.Has("use_internal_prices_only") || jsonValue.Has("use_internal_prices")) {
        bool useInternalPrices = true;
        bool useInternalPricesOnly = true;
        JREAD_BOOL_OPT(jsonValue, "use_internal_prices", useInternalPrices);
        if (!jsonValue.Has("use_internal_prices_only")) {
            useInternalPricesOnly = useInternalPrices;
        } else {
            JREAD_BOOL_OPT(jsonValue, "use_internal_prices_only", useInternalPricesOnly);
        }
        if (!useInternalPrices && !useInternalPricesOnly) {
            PriceSource = EPriceSource::ExternalOnly;
        } else if (useInternalPrices && !useInternalPricesOnly) {
            PriceSource = EPriceSource::Both;
        } else if (useInternalPrices && useInternalPricesOnly) {
            PriceSource = EPriceSource::InternalOnly;
        } else {
            return false;
        }
    } else if (!TJsonProcessor::ReadFromString("price_source", jsonValue, PriceSource)) {
        return false;
    }

    JREAD_DOUBLE_OPT(jsonValue, "pedestrian_speed", PedestrianSpeed);
    JREAD_STRING_OPT(jsonValue, "detailed_description", DetailedDescription);
    JREAD_STRING_OPT(jsonValue, "detailed_description_icon", DetailedDescriptionIcon);
    JREAD_STRING_OPT(jsonValue, "agreement", Agreement);
    JREAD_STRING_OPT(jsonValue, "price_model", PriceModel);
    JREAD_STRING_OPT(jsonValue, "parking_price_model", ParkingPriceModel);
    JREAD_STRING_OPT(jsonValue, "price_offer_constructor", PriceOfferConstructor);
    JREAD_STRING_OPT(jsonValue, "short_name", ShortName);
    JREAD_STRING_OPT(jsonValue, "subname", SubName);

    JREAD_BOOL_OPT(jsonValue, "only_original_offer_cashback", OnlyOriginalOfferCashback);
    JREAD_INT_OPT(jsonValue, "deposit_default", DepositDefault);

    JREAD_DURATION_OPT(jsonValue, "min_free_time", MinimalFreeTime);
    JREAD_DURATION_OPT(jsonValue, "max_free_time", MaximalFreeTime);
    JREAD_DURATION_OPT(jsonValue, "min_fees_time", MinimalFeesTime);
    JREAD_INT_OPT(jsonValue, "min_debt_threshold", MinimalDebtThreshold);
    JREAD_BOOL_OPT(jsonValue, "use_default_short_description", UseDefaultShortDescriptions);

    const NJson::TJsonValue& shortDescription = jsonValue["short_description"];
    if (shortDescription.IsDefined() && !shortDescription.IsArray()) {
        return false;
    }
    for (auto&& i : shortDescription.GetArray()) {
        ShortDescription.push_back(i.GetStringRobust());
    }

    if (!TSurgeActivationConfig::DeserializeFromJson(jsonValue)) {
        return false;
    }

    if (!TEntityPriceSettings::ReadPricingPolicy(jsonValue["riding_price"], RidingPriceCalculatorType, RidingPriceCalculatorConfig, RidingPriceCalculator)) {
        return false;
    }
    if (!TEntityPriceSettings::ReadPricingPolicy(jsonValue["parking_price"], ParkingPriceCalculatorType, ParkingPriceCalculatorConfig, ParkingPriceCalculator)) {
        return false;
    }
    if (jsonValue.Has("km_price")) {
        if (!TEntityPriceSettings::ReadPricingPolicy(jsonValue["km_price"], KmPriceCalculatorType, KmPriceCalculatorConfig, KmPriceCalculator)) {
            return false;
        }
    } else {
        KmPriceCalculatorType = "constant";
        KmPriceCalculatorConfig = new TConstantPriceConfig(10000);
        KmPriceCalculator = KmPriceCalculatorConfig->Construct();
    }
    return
        NJson::ParseField(jsonValue["cashback_percent"], CashbackPercent) &&
        NJson::ParseField(jsonValue["fueling_enabled"], FuelingEnabled) &&
        NJson::ParseField(jsonValue["price_offer_constructors"], PriceOfferConstructors);
}


NJson::TJsonValue TStandartOfferConstructor::SerializeSpecialsToJson() const {
    NJson::TJsonValue jsonValue = TBase::SerializeSpecialsToJson();
    NJson::TJsonValue& arrTagNames = jsonValue.InsertValue("filter_tag_names", NJson::JSON_ARRAY);
    JWRITE(jsonValue, "finish_area_tags_filter", FinishAreaTagsFilter.ToString());
    JWRITE(jsonValue, "riding_area_tags_filter", RidingAreaTagsFilter.ToString());
    for (auto&& i : TagsFilter.GetMonoTagNames()) {
        arrTagNames.AppendValue(i);
    }

    JWRITE(jsonValue, "tags_filter", TagsFilter.ToString());

    JWRITE(jsonValue, "use_deposit", UseDeposit);
    JWRITE(jsonValue, "use_km_prices", UseKmPrices);
    TJsonProcessor::WriteAsString(jsonValue, "price_source", PriceSource);
    if (PriceSource == EPriceSource::InternalOnly) {
        JWRITE(jsonValue, "use_internal_prices", true);
        JWRITE(jsonValue, "use_internal_prices_only", true);
    } else if (PriceSource == EPriceSource::ExternalOnly) {
        JWRITE(jsonValue, "use_internal_prices", false);
        JWRITE(jsonValue, "use_internal_prices_only", false);
    } else {
        JWRITE(jsonValue, "use_internal_prices", true);
        JWRITE(jsonValue, "use_internal_prices_only", false);
    }

    JWRITE(jsonValue, "pedestrian_speed", PedestrianSpeed);
    JWRITE(jsonValue, "only_original_offer_cashback", OnlyOriginalOfferCashback);
    JWRITE(jsonValue, "deposit_default", DepositDefault);

    JWRITE_DURATION(jsonValue, "min_free_time", MinimalFreeTime);
    JWRITE_DURATION(jsonValue, "max_free_time", MaximalFreeTime);
    JWRITE_DURATION(jsonValue, "min_fees_time", MinimalFeesTime);
    JWRITE(jsonValue, "min_debt_threshold", MinimalDebtThreshold);
    JWRITE(jsonValue, "detailed_description", DetailedDescription);
    JWRITE(jsonValue, "detailed_description_icon", DetailedDescriptionIcon);
    JWRITE(jsonValue, "agreement", Agreement);
    JWRITE(jsonValue, "price_model", PriceModel);
    JWRITE(jsonValue, "parking_price_model", ParkingPriceModel);
    JWRITE(jsonValue, "price_offer_constructor", PriceOfferConstructor);
    JWRITE(jsonValue, "short_name", ShortName);
    JWRITE(jsonValue, "subname", SubName);
    JWRITE(jsonValue, "use_default_short_description", UseDefaultShortDescriptions);

    NJson::InsertNonNull(jsonValue, "cashback_percent", CashbackPercent);
    NJson::InsertField(jsonValue, "fueling_enabled", FuelingEnabled);
    NJson::InsertField(jsonValue, "price_offer_constructors", PriceOfferConstructors);

    for (auto&& i : ShortDescription) {
        jsonValue["short_description"].AppendValue(i);
    }

    TSurgeActivationConfig::SerializeToJson(jsonValue);

    if (RidingPriceCalculatorConfig) {
        NJson::TJsonValue ridingPriceJson;
        ridingPriceJson.InsertValue("pricing_type", RidingPriceCalculatorType);
        ridingPriceJson.InsertValue("price_details", RidingPriceCalculatorConfig->SerializeToJson());
        JWRITE(jsonValue, "riding_price", ridingPriceJson);
    }
    if (ParkingPriceCalculatorConfig) {
        NJson::TJsonValue parkingPriceJson;
        parkingPriceJson.InsertValue("pricing_type", ParkingPriceCalculatorType);
        parkingPriceJson.InsertValue("price_details", ParkingPriceCalculatorConfig->SerializeToJson());
        JWRITE(jsonValue, "parking_price", parkingPriceJson);
    }
    if (!!KmPriceCalculatorConfig) {
        NJson::TJsonValue kmPriceJson;
        kmPriceJson.InsertValue("pricing_type", KmPriceCalculatorType);
        kmPriceJson.InsertValue("price_details", KmPriceCalculatorConfig->SerializeToJson());
        JWRITE(jsonValue, "km_price", kmPriceJson);
    }
    return jsonValue;
}

TUserAction::TFactory::TRegistrator<TStandartOfferConstructor> TStandartOfferConstructor::Registrator(TStandartOfferConstructor::GetTypeName());

NDrive::TScheme TStandartOfferConstructor::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);
    {
        auto gTab = result.StartTabGuard("billing");
        result.Add<TFSNumeric>("min_debt_threshold", "Минимальный порог для долга (копейки)").SetMin(0);
        result.Add<TFSNumeric>("deposit_default", "Депозит в случае не достаточного количества поездок (копейки)").SetMin(0);
        result.Add<TFSBoolean>("fueling_enabled", "Использовать Яндекс.Заправки").SetDefault(FuelingEnabled);
        result.Add<TFSBoolean>("use_deposit", "Использовать депозит").SetDefault(true);
        result.Add<TFSNumeric>("cashback_percent", "Процент кешбека").SetMin(0);
        result.Add<TFSBoolean>("only_original_offer_cashback", "Начислять кешбек только при удовлетворении условий тарифа").SetDefault(true);
    }
    {
        auto gTab = result.StartTabGuard("activation");
        result.Add<TFSString>("tags_filter", "Фильтр активации предложения по тегам машины в стандартном формате");
    }
    {
        auto gTab = result.StartTabGuard("zones");
        result.Add<TFSString>("finish_area_tags_filter", "Фильтр по тегам зоны завершения аренды в стандартном формате");
        result.Add<TFSString>("riding_area_tags_filter", "Фильтр по тегам зоны допустимой поездки в стандартном формате");
    }
    {
        auto gTab = result.StartTabGuard("router");
        result.Add<TFSNumeric>("pedestrian_speed", "Скорость пешехода (км/ч)").SetPrecision(2).SetDefault(4.2).SetDeprecated(true);
    }
    {
        auto gTab = result.StartTabGuard("fines");
        result.Add<TFSNumeric>("min_free_time", "Минимальное бесплатное время на фазу (секунды)").SetMin(0).SetDeprecated(true);
        result.Add<TFSNumeric>("max_free_time", "Максимальное бесплатное время на фазу (секунды)").SetMin(0).SetDeprecated(true);
        result.Add<TFSNumeric>("min_fees_time", "Минимальное время для включения штрафов(секунды)").SetMin(0).SetDeprecated(true);
    }
    {
        auto gTab = result.StartTabGuard("client_appearance");
        result.Add<TFSString>("agreement", "Ссылка на акт приемки-передачи");
        result.Add<TFSBoolean>("use_default_short_description", "Добавить короткое описание default").SetDefault(true);
        result.Add<TFSJson>("short_description", "Короткое описание");
        result.Add<TFSText>("detailed_description", "Подробное описание");
        result.Add<TFSString>("detailed_description_icon", "Иконка для кнопки 'Подробное описание'");
        result.Add<TFSString>("short_name", "Короткое имя для склейки в карточку");
        result.Add<TFSString>("subname", "Подзаголовок");
    }
    {
        auto gTab = result.StartTabGuard("surge");
        TSurgeActivationConfig::FillScheme(result);
    }
    {
        auto gTab = result.StartTabGuard("prices");
        {
            NDrive::TScheme& ridingPricePolicy = result.Add<TFSStructure>("riding_price", "Политика ценообразования в состоянии riding").SetStructure<NDrive::TScheme>();
            ridingPricePolicy.SetDeprecated();
            ridingPricePolicy.Add<TFSVariants>("pricing_type", "Тип ценообразования").InitVariantsClass<IPriceCalculatorConfig>();
            ridingPricePolicy.Add<TFSJson>("price_details", "Настройка");
        }

        {
            NDrive::TScheme& parkingPricePolicy = result.Add<TFSStructure>("parking_price", "Политика ценообразования в состоянии parking").SetStructure<NDrive::TScheme>();
            parkingPricePolicy.SetDeprecated();
            parkingPricePolicy.Add<TFSVariants>("pricing_type", "Тип ценообразования").InitVariantsClass<IPriceCalculatorConfig>();
            parkingPricePolicy.Add<TFSJson>("price_details", "Настройка");
        }

        {
            NDrive::TScheme& kmPricePolicy = result.Add<TFSStructure>("km_price", "Политика ценообразования для 1 км").SetStructure<NDrive::TScheme>();
            kmPricePolicy.SetDeprecated();
            kmPricePolicy.Add<TFSVariants>("pricing_type", "Тип ценообразования").InitVariantsClass<IPriceCalculatorConfig>();
            kmPricePolicy.Add<TFSJson>("price_details", "Настройка");
        }
        result.Add<TFSBoolean>("use_km_prices", "Использовать цену километра").SetDefault(false);
        result.Add<TFSVariants>("price_source", "Источник ценообразования").InitVariants<EPriceSource>();
        result.Add<TFSVariants>("price_offer_constructor", "Явный PriceOfferConstructor").SetVariants(TPriceOfferConstructor::GetNames(server));
        result.Add<TFSVariants>("price_offer_constructors", "Разрешенные PriceOfferConstructor'ы").SetVariants(TPriceOfferConstructor::GetNames(server)).SetMultiSelect(true);
    }
    {
        auto gTab = result.StartTabGuard("experiment");
        auto models = server->GetModelsStorage();
        auto modelNames = models ? models->ListOfferModels() : TVector<TString>{};
        result.Add<TFSVariants>("price_model", "Модель для вычисления цены").SetVariants(modelNames);
    }
    return result;
}

void TStandartOfferReport::ApplyInternalCorrection(const TString& areaId, const TOffersBuildingContext& /*context*/, const NDrive::IServer* server) {
    TStandartOffer* stOffer = GetOfferAs<TStandartOffer>();
    if (!stOffer) {
        return;
    }
    TMaybe<TArea> areaObject = server->GetDriveAPI()->GetAreasDB()->GetCustomObject(areaId);
    if (!areaObject) {
        return;
    }
    TCarLocationContext finishContext = TCarLocationContext::BuildByArea(stOffer->GetObjectId(), *areaObject, *server);
    finishContext.SetNeedInternalFees(true);
    finishContext.SetGuaranteeFees(false);
    finishContext.SetTakeFeesFromOffer(false);
    const TCarLocationFeatures finishFeatures = finishContext.GetCarAreaFeatures(true, stOffer, server);

    if (finishFeatures.HasFeeInfo() && finishFeatures.GetFeeInfoUnsafe().GetPolicy() == EDropOfferPolicy::FeesMinute) {
        const TMaybe<ui32> deltaResult = finishFeatures.GetFeeInfoUnsafe().GetFee();
        if (deltaResult) {
            stOffer->MutableRiding().AddPrice(*deltaResult, "flow_correction_" + areaId);
        }
    }
}

TFullPricesContext* TStandartOfferReport::GetFullPricesContext() {
    return GetOfferAs<TFullPricesContext>();
}

void TStandartOfferReport::DoRecalcPrices(const NDrive::IServer* server) {
    Y_UNUSED(server);
    auto standartOffer = GetOfferAs<TStandartOffer>();
    auto inheritedProperties = standartOffer ? standartOffer->GetInheritedPropertiesPtr() : nullptr;
    if (inheritedProperties) {
        standartOffer->MutableRiding().SetPrice(inheritedProperties->RidingPrice, "inherited");
        standartOffer->MutableParking().SetPrice(inheritedProperties->ParkingPrice, "inherited");
    }
}

void TStandartOfferReport::ApplyFlowCorrection(const TString& areaId, const TOffersBuildingContext& context, const NDrive::IServer* server) {
    TStandartOffer* stOffer = GetOfferAs<TStandartOffer>();
    if (!stOffer) {
        return;
    }
    const TMaybe<TGeoCoord> cStart = context.GetStartPosition();
    if (!cStart) {
        ERROR_LOG << "Cannot determinate start coord for " << stOffer->GetObjectId() << Endl;
        return;
    }
    TMaybe<TArea> areaObject = server->GetDriveAPI()->GetAreasDB()->GetCustomObject(areaId);
    if (!areaObject) {
        return;
    }
    TCarLocationContext finishContext = TCarLocationContext::BuildByArea(stOffer->GetObjectId(), *areaObject, *server);
    finishContext.SetGuaranteeFees(false);
    finishContext.SetTakeFeesFromOffer(false);
    const TCarLocationFeatures finishFeatures = finishContext.GetCarAreaFeatures(true, stOffer, server);

    TCarLocationContext startContext = TCarLocationContext::BuildByCoord(stOffer->GetObjectId(), context.OptionalOriginalRidingStart().GetOrElse(*cStart), *server);
    startContext.SetGuaranteeFees(false);
    startContext.SetTakeFeesFromOffer(false);
    const TCarLocationFeatures startFeatures = startContext.GetCarAreaFeatures(false, stOffer, server);

    if (finishFeatures.HasFeeInfo() && finishFeatures.GetFeeInfoUnsafe().GetPolicy() == EDropOfferPolicy::FeesMinute) {
        const TMaybe<ui32> deltaResult = finishFeatures.GetDeltaFees(startFeatures);
        if (deltaResult) {
            stOffer->MutableRiding().AddPrice(*deltaResult, "flow_correction_" + areaId);
        }
    }

}

TString TStandartOfferReport::PredictDestination(const TOffersBuildingContext& context, const NDrive::IServer* server, const TCommonDestinationDetector& otherDetector) const {
    const TStandartOffer* stOffer = GetOfferAs<TStandartOffer>();
    if (!stOffer) {
        return "";
    }
    TString areaId;
    TMaybe<double> resultScore = otherDetector.CalcScore(stOffer->GetFeatures(), context.MutableFlowControlContext(), server, &otherDetector);
    const auto actor = [server, &resultScore, stOffer, &context, &areaId, &otherDetector](const TDBTag& dbTag) -> bool {
        const TDestinationDetectorTag* destTag = dbTag.GetTagAs<TDestinationDetectorTag>();
        if (destTag && !destTag->GetStartZoneAttributesFilter().IsEmpty() &&
            context.GetLocationTags() && !destTag->GetStartZoneAttributesFilter().IsMatching(context.GetLocationTags())) {
            return true;
        }
        const TMaybe<double> score = dbTag.GetTagAs<TDestinationDetectorTag>()->CalcScore(stOffer->GetFeatures(), context.MutableFlowControlContext(), server, &otherDetector);
        if (!score || (resultScore && *resultScore >= *score)) {
            return true;
        }
        areaId = dbTag.GetObjectId();
        resultScore = score;
        return true;
    };
    server->GetDriveAPI()->GetAreasDB()->ProcessHardDBTags<TDestinationDetectorTag>(actor);
    return areaId;
}

void TStandartOfferReport::RecalculateFeatures() {
    TStandartOffer* offer = GetOfferAs<TStandartOffer>();
    if (!offer) {
        return;
    }
    auto& features = offer->MutableFeatures();
    NDrive::CalcPriceFeatures(features, 0.01 * offer->GetPublicOriginalRiding(), 0.01 * offer->GetEquilibrium().GetRiding());
    NDrive::CalcWaitingPriceFeatures(features, 0.01 * offer->GetPublicOriginalParking(), 0.01 * offer->GetEquilibrium().GetParking());
    NDrive::CalcOfferAcceptancePriceFeatures(features, offer->GetAcceptance().GetPrice());
    NDrive::CalcOfferNameFeatures(features, offer->GetName(), offer->GetGroupName());
    NDrive::CalcOfferCashbackPercentFeatures(features, offer->GetCashbackPercent());
    NDrive::CalcOfferIsPlusUserFeatures(features, offer->OptionalIsPlusUser().GetOrElse(false));
    NDrive::CalcOfferInsurancePriceFeatures(features, offer->OptionalInsuranceCost());
    NDrive::CalcOfferDepositAmountFeatures(features, offer->GetDepositAmount());
}

NDrive::TLocationTags Multiply(const NDrive::TLocationTags& left, const NDrive::TLocationTags& right) {
        NDrive::TLocationTags result;
        for (auto&& i : left) {
            for (auto&& j : right) {
                result.insert(i + '*' + j);
            }
        }
        return result;
}

TMaybe<TTagsFilter> NDrive::CreateRidingAreaTagsFilter(const TOffersBuildingContext& context, const NDrive::IServer& server, const TString builderType) {
    bool enableAreaTagsFilterInheritance = context.GetSetting<bool>("offer_builder." + builderType + ".area_tags_filter.inheritance.enabled").GetOrElse(false);
    if (!enableAreaTagsFilterInheritance) {
        return {};
    }
    auto inheritedTagsString = context.GetSetting<TString>("offer_builder." + builderType + ".area_tags_filter.inheritance.riding_tags");
    auto inheritedTags = inheritedTagsString ? MakeSet<TString>(StringSplitter(*inheritedTagsString).SplitBySet(", ").SkipEmpty()) : TSet<TString>{
        "allow_riding_kzn",
        "allow_riding_msc",
        "allow_riding_spb",
        "allow_riding_sochi",
    };
    auto intersection = MakeIntersection(context.GetLocationTags(), inheritedTags);
    if (!intersection.empty()) {
        const auto& originalTags = context.HasModelTag() ? TModelTag::GetAllowRidingTags(context.GetModelTagRef(), server) : NDrive::DefaultAllowRidingLocationTags;
        const auto& patchedTags = Multiply(originalTags, intersection);
        return TTagsFilter(patchedTags, false, true);
    }
    return {};
}

TMaybe<TTagsFilter> NDrive::CreateFinishAreaTagsFilter(const TOffersBuildingContext& context, const NDrive::IServer& server, const TString builderType) {
    bool enableAreaTagsFilterInheritance = context.GetSetting<bool>("offer_builder." + builderType + ".area_tags_filter.inheritance.enabled").GetOrElse(false);
    if (!enableAreaTagsFilterInheritance) {
        return {};
    }
    auto inheritedTagsString = context.GetSetting<TString>("offer_builder." + builderType + ".area_tags_filter.inheritance.finish_tags");
    auto inheritedTags = inheritedTagsString ? MakeSet<TString>(StringSplitter(*inheritedTagsString).SplitBySet(", ").SkipEmpty()) : TSet<TString>{
        "allow_drop_car_kzn",
        "allow_drop_car_msc",
        "allow_drop_car_spb",
        "allow_drop_car_sochi",
    };
    auto intersection = MakeIntersection(context.GetLocationTags(), inheritedTags);
    if (!intersection.empty()) {
        const auto& originalTags = context.HasModelTag() ? TModelTag::GetAllowDropTags(context.GetModelTagRef(), server) : NDrive::DefaultAllowDropLocationTags;
        const auto& patchedTags = Multiply(originalTags, intersection);
        return TTagsFilter(patchedTags, false, true);
    }
    return {};
}
