#pragma once

#include "abstract.h"

#include <drive/backend/offers/context.h>
#include <drive/backend/offers/discount.h>
#include <drive/backend/offers/offers/standart.h>
#include <drive/backend/offers/price/price.h>
#include <drive/backend/offers/ranking/features.h>

#include <drive/backend/models/fwd.h>
#include <drive/backend/tags/tags_filter.h>

#include <rtline/library/json/cast.h>
#include <rtline/library/time_restriction/time_restriction.h>

#include <util/datetime/base.h>

namespace NDrive {
    struct TLocation;
}

namespace NGraph {
    class TRouter;
}

class TStandartOfferReport: public IOfferReport {
private:
    using TBase = IOfferReport;

private:
    R_FIELD(TVector<TString>, ShortDescription);

protected:
    virtual TFullPricesContext* GetFullPricesContext() override final;
    virtual void DoRecalcPrices(const NDrive::IServer* /*server*/) override;

public:
    using TBaseOffer = TStandartOffer;
    using TBase::TBase;

    TStandartOfferReport(const IOfferReport& source)
        : TBase(source)
    {
    }

    virtual void ApplyFlowCorrection(const TString& areaId, const TOffersBuildingContext& /*context*/, const NDrive::IServer* server) override;
    virtual void ApplyInternalCorrection(const TString& areaId, const TOffersBuildingContext& /*context*/, const NDrive::IServer* server) override;

    virtual TString PredictDestination(const TOffersBuildingContext& context, const NDrive::IServer* server, const TCommonDestinationDetector& otherDetector) const override;

    virtual void RecalculateFeatures() override;
};

class TStandartOfferConstructor
    : public IOfferBuilderAction
    , public TSurgeActivationConfig
{
private:
    using TBase = IOfferBuilderAction;

private:
    static TFactory::TRegistrator<TStandartOfferConstructor> Registrator;
public:
    enum class EPriceSource {
        InternalOnly /* "internal_only" */,
        ExternalOnly /* "external_only" */,
        Both /* "both" */
    };
private:
    R_OPTIONAL(ui32, CashbackPercent);
    R_READONLY(bool, OnlyOriginalOfferCashback, true);
    R_READONLY(ui32, DepositDefault, 27182);
    R_READONLY(TDuration, MaximalFreeTime, TDuration::Minutes(20));
    R_READONLY(TDuration, MinimalFreeTime, TDuration::Minutes(3));
    R_READONLY(TDuration, MinimalFeesTime, TDuration::Minutes(5));
    R_READONLY(ui32, MinimalDebtThreshold, 102400);
    R_READONLY(TString, Agreement);
    R_READONLY(TString, PriceModel);
    R_READONLY(TString, ParkingPriceModel);
    R_FIELD(TString, PriceOfferConstructor);
    R_FIELD(TSet<TString>, PriceOfferConstructors);
    R_FIELD(bool, UseDefaultShortDescriptions, true);
    R_FIELD(TVector<TString>, ShortDescription);
    R_READONLY(TString, ShortName);
    R_READONLY(TString, SubName);
    R_FIELD(TString, DetailedDescription);
    R_FIELD(TString, DetailedDescriptionIcon);
    R_FIELD(bool, FuelingEnabled, true);
    R_READONLY(bool, UseDeposit, true);
    R_READONLY(bool, UseKmPrices, false);
    R_FIELD(EPriceSource, PriceSource, EPriceSource::InternalOnly);
    R_READONLY(double, PedestrianSpeed, 4.2);

    TString ParkingPriceCalculatorType = "constant";
    TString RidingPriceCalculatorType = "constant";
    TString KmPriceCalculatorType = "constant";
    IPriceCalculatorConfig::TPtr RidingPriceCalculatorConfig;
    IPriceCalculatorConfig::TPtr ParkingPriceCalculatorConfig;
    IPriceCalculatorConfig::TPtr KmPriceCalculatorConfig;
    IPriceCalculator::TPtr RidingPriceCalculator;
    IPriceCalculator::TPtr ParkingPriceCalculator;
    IPriceCalculator::TPtr KmPriceCalculator;

    TTagsFilter TagsFilter;
    TTagsFilter FinishAreaTagsFilter;
    TTagsFilter RidingAreaTagsFilter;

protected:
    bool BuildAdditionalDuration(TStandartOffer& offer, const TOffersBuildingContext& context, const TUserPermissions& permissions, NDrive::TInfoEntitySession& session) const;

    virtual EOfferCorrectorResult DoBuildOffers(const TUserPermissions& permissions, TVector<IOfferReport::TPtr>& offers, const TOffersBuildingContext& context, const NDrive::IServer* server, NDrive::TInfoEntitySession& session) const override;
    virtual EOfferCorrectorResult DoCheckOfferConditions(const TOffersBuildingContext& context, const TUserPermissions& permissions) const override;
    virtual bool DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonValue) override;
    virtual NJson::TJsonValue SerializeSpecialsToJson() const override;

public:
    TStandartOfferConstructor() = default;
    TStandartOfferConstructor(IPriceCalculatorConfig::TPtr ridingPriceCalculatorConfig, IPriceCalculatorConfig::TPtr parkingPriceCalculatorConfig, const TTagsFilter& filter)
        : RidingPriceCalculatorConfig(ridingPriceCalculatorConfig)
        , ParkingPriceCalculatorConfig(parkingPriceCalculatorConfig)
        , TagsFilter(filter)
    {
        RidingPriceCalculator = RidingPriceCalculatorConfig->Construct();
        ParkingPriceCalculator = ParkingPriceCalculatorConfig->Construct();
    }

    virtual bool GetPrices(const NDrive::IServer* /*server*/, ui32& ridingPrice, ui32& parkingPrice, ui32& kmPrice) const {
        ridingPrice = RidingPriceCalculator->GetBasePrice();
        parkingPrice = ParkingPriceCalculator->GetBasePrice();
        if (!!KmPriceCalculator) {
            kmPrice = KmPriceCalculator->GetBasePrice();
        } else {
            kmPrice = 10000;
        }
        return true;
    }

    const TTagsFilter& GetFinishAreaTagsFilter() const {
        return FinishAreaTagsFilter;
    }

    const TTagsFilter& GetRidingAreaTagsFilter() const {
        return RidingAreaTagsFilter;
    }

    const TTagsFilter& GetTagsFilter() const {
        return TagsFilter;
    }

    static TString GetTypeStatic() {
        return "standart_offer_builder";
    }

    static TString GetTypeName() {
        return GetTypeStatic();
    }

    virtual TString GetType() const override {
        return GetTypeStatic();
    }

    virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override;
};

NDrive::TLocationTags Multiply(const NDrive::TLocationTags& left, const NDrive::TLocationTags& right);

namespace NDrive {
    TMaybe<TTagsFilter> CreateRidingAreaTagsFilter(const TOffersBuildingContext& context, const NDrive::IServer& server, const TString builderType);
    TMaybe<TTagsFilter> CreateFinishAreaTagsFilter(const TOffersBuildingContext& context, const NDrive::IServer& server, const TString builderType);
}
