#include "taxi.h"

#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/roles/manager.h>

#include <rtline/library/json/adapters.h>
#include <rtline/util/algorithm/ptr.h>

namespace {
    const TMap<TString, TString> Transitions = {
        { "old_state_riding", "old_state_parking" },
        { "old_state_parking", "old_state_riding" },
    };
}

THolder<TFixPointOfferReport> TTaxiFixPointOfferConstructor::Extend(const TStandartOfferReport& standardOfferReport) const {
    auto result = IOfferReport::ExtendCustom<TFixPointOfferReport, TTaxiFixPointOffer>(standardOfferReport);
    Y_ASSERT(result);
    Y_ASSERT(result->GetOfferAs<TTaxiFixPointOffer>() != nullptr);
    return result;
}

THolder<TPackOfferReport> TTaxiPackOfferConstructor::Extend(const TStandartOfferReport& standardOfferReport) const {
    auto result = IOfferReport::ExtendCustom<TPackOfferReport, TTaxiPackOffer>(standardOfferReport);
    Y_ASSERT(result);
    Y_ASSERT(result->GetOfferAs<TTaxiPackOffer>() != nullptr);
    return result;
}

EDriveSessionResult TTaxiValidationOffer::DoCheckSession(const TUserPermissions& permissions, NDrive::TEntitySession& session, const NDrive::IServer* server, bool onPerform) const {
    if (onPerform && TagName) {
        if (!server) {
            session.SetErrorInfo("TaxiValidationOffer::DoCheckSession", "no database interface", EDriveSessionResult::InternalError);
            return EDriveSessionResult::InternalError;
        }
        const auto& database = server->GetDriveDatabase();
        auto tag = database.GetTagsManager().GetTagsMeta().CreateTag(TagName);
        if (!tag) {
            session.SetErrorInfo("TaxiValidationOffer::DoCheckSession", "cannot create tag " + TagName, EDriveSessionResult::InternalError);
            return EDriveSessionResult::InternalError;
        }
        if (!database.GetTagsManager().GetDeviceTags().AddTag(tag, permissions.GetUserId(), GetObjectId(), server, session)) {
            return EDriveSessionResult::InternalError;
        }
    }
    return EDriveSessionResult::Success;
}

void TTaxiValidationOffer::SetTagName(const TString& value) {
    TagName = value;
}

bool TTaxiValidationOffer::DeserializeFromProto(const NDrive::NProto::TOffer& info) {
    if (!TBase::DeserializeFromProto(info)) {
        return false;
    }
    TagName = info.GetTagName();
    return true;
}

NDrive::NProto::TOffer TTaxiValidationOffer::SerializeToProto() const {
    auto info = TBase::SerializeToProto();
    info.SetTagName(TagName);
    return info;
}

NJson::TJsonValue TTaxiMetaOfferReport::BuildJsonReport(ELocalization locale, NDriveSession::TReportTraits traits, const NDrive::IServer& server, const TUserPermissions& permissions) const {
    if (Secondary) {
        auto result = Secondary->BuildJsonReport(locale, traits, server, permissions);
        result["offer_id"] = GetOffer()->GetOfferId();
        return result;
    } else {
        return TBase::BuildJsonReport(locale, traits, server, permissions);
    }
}

bool TTaxiMetaOfferContructor::DeserializeSpecialsFromJson(const NJson::TJsonValue& value) {
    if (!TBase::DeserializeSpecialsFromJson(value)) {
        return false;
    }
    return
        NJson::ParseField(value["validation_tag_name"], ValidationTagName) &&
        NJson::ParseField(value["fixpoint_offer_name"], FixPointOfferName, true) &&
        NJson::ParseField(value["pack_offer_name"], PackOfferName, true);
}

NJson::TJsonValue TTaxiMetaOfferContructor::SerializeSpecialsToJson() const {
    NJson::TJsonValue result = TBase::SerializeSpecialsToJson();
    result["fixpoint_offer_name"] = FixPointOfferName;
    result["pack_offer_name"] = PackOfferName;
    result["validation_tag_name"] = NJson::ToJson(NJson::Nullable(ValidationTagName));
    return result;
}

EOfferCorrectorResult TTaxiMetaOfferContructor::DoBuildOffers(const TUserPermissions& permissions, TVector<IOfferReport::TPtr>& offers, const TOffersBuildingContext& context, const NDrive::IServer* server, NDrive::TInfoEntitySession& session) const {
    auto fixPointOfferReport = BuildOffer(FixPointOfferName, permissions, context, server, session);
    if (!fixPointOfferReport) {
        return EOfferCorrectorResult::Problems;
    }
    auto fixPointOffer = fixPointOfferReport->GetOfferAs<TFixPointOffer>();
    if (!fixPointOffer) {
        auto offer = fixPointOfferReport->GetOffer();
        session.SetErrorInfo(
            "TaxiMetaOfferContructor::DoBuildOffers",
            TStringBuilder() << "cannot cast offer " << (offer ? offer->GetOfferId() : "null") << " to FixPointOffer",
            EDriveSessionResult::InternalError
        );
        return EOfferCorrectorResult::Problems;
    }

    auto packOfferReport = BuildOffer(PackOfferName, permissions, context, server, session);
    if (!packOfferReport) {
        return EOfferCorrectorResult::Problems;
    }
    auto packOfferReportT = std::dynamic_pointer_cast<TPackOfferReport>(packOfferReport);
    auto packOffer = packOfferReport->GetOfferAs<TPackOffer>();
    if (!packOffer || !packOfferReportT) {
        auto offer = packOfferReport->GetOffer();
        session.SetErrorInfo(
            "TaxiMetaOfferContructor::DoBuildOffers",
            TStringBuilder() << "cannot cast offer " << (offer ? offer->GetOfferId() : "null") << " to FixPointOffer",
            EDriveSessionResult::InternalError
        );
        return EOfferCorrectorResult::Problems;
    }

    auto visualOfferReport = MakeAtomicShared<TPackOfferReport>(*packOfferReportT);
    visualOfferReport->SetOffer(packOffer->Clone());
    visualOfferReport->GetOffer()->SetOfferId(ICommonOffer::CreateOfferId());
    visualOfferReport->SetPriceOfferConstructor(packOfferReport->GetPriceOfferConstructor());

    TPriceContext dropoffPrice;
    dropoffPrice.SetPrice(DropoffOvertimePrice);
    TFullPricesContext dropoffFullPricesContext;
    dropoffFullPricesContext.SetParking(dropoffPrice);
    dropoffFullPricesContext.SetRiding(dropoffPrice);
    auto dropoffOffer = MakeAtomicShared<TTaxiDropoffOffer>(dropoffFullPricesContext);
    auto dropoffOfferReport = MakeAtomicShared<TStandartOfferReport>(dropoffOffer, nullptr);

    TDiscount dropoffFreeTimeDiscount;
    TDiscount::TDiscountDetails dropoffFreeTimeDiscountDetails;
    dropoffFreeTimeDiscountDetails.SetTagName("old_state_reservation");
    dropoffFreeTimeDiscountDetails.SetAdditionalTime(DropoffFreeTime);
    dropoffFreeTimeDiscount.AddDetails(dropoffFreeTimeDiscountDetails);
    dropoffOffer->AddDiscount(dropoffFreeTimeDiscount);

    dropoffOffer->SetBehaviourConstructorId(GetName());
    dropoffOffer->SetPriceConstructorId(GetName());
    dropoffOffer->SetObjectId(context.GetCarIdUnsafe());
    dropoffOffer->SetUserId(permissions.GetUserId());

    TPriceContext validationPrice;
    validationPrice.SetPrice(ValidationOvertimePrice);
    TFullPricesContext validationFullPriceContext;
    validationFullPriceContext.SetParking(validationPrice);
    validationFullPriceContext.SetRiding(validationPrice);
    auto validationOffer = MakeAtomicShared<TTaxiValidationOffer>(validationFullPriceContext);
    auto validationOfferReport = MakeAtomicShared<TStandartOfferReport>(validationOffer, nullptr);

    TDiscount validationFreeTimeDiscount;
    TDiscount::TDiscountDetails validationFreeTimeDiscountDetails;
    validationFreeTimeDiscountDetails.SetTagName("old_state_reservation");
    validationFreeTimeDiscountDetails.SetAdditionalTime(ValidationFreeTime);
    validationFreeTimeDiscount.AddDetails(validationFreeTimeDiscountDetails);
    validationOffer->AddDiscount(validationFreeTimeDiscount);
    validationOffer->SetTagName(ValidationTagName);

    auto fixPointPrice = std::min(fixPointOffer->GetPackPrice(), packOffer->GetPackPrice());
    auto packPrice = packOffer->GetPackPrice() - fixPointPrice;

    validationOffer->SetBehaviourConstructorId(GetName());
    validationOffer->SetPriceConstructorId(GetName());
    validationOffer->SetObjectId(context.GetCarIdUnsafe());
    validationOffer->SetUserId(permissions.GetUserId());

    fixPointOffer->SetNextOfferId(validationOffer->GetOfferId());
    fixPointOffer->SetPackPrice(fixPointPrice);
    fixPointOffer->SetSwitchable(true);

    validationOfferReport->SetVisible(false);
    validationOffer->SetDeadline(fixPointOffer->GetTimestamp() + TDuration::Days(1));
    validationOffer->SetNextOfferId(packOffer->GetOfferId());

    packOfferReport->SetVisible(false);
    packOffer->SetDeadline(fixPointOffer->GetTimestamp() + TDuration::Days(1));
    packOffer->SetNextOfferId(dropoffOffer->GetOfferId());
    packOffer->SetPackPrice(packPrice);
    packOffer->SetSwitchable(true);

    dropoffOfferReport->SetVisible(false);
    dropoffOffer->SetDeadline(fixPointOffer->GetTimestamp() + TDuration::Days(1));

    visualOfferReport->SetVisible(false);

    auto meta = MakeAtomicShared<TTaxiMetaOfferReport>(fixPointOfferReport, visualOfferReport);
    offers.push_back(meta);
    offers.push_back(validationOfferReport);
    offers.push_back(packOfferReport);
    offers.push_back(dropoffOfferReport);
    offers.push_back(visualOfferReport);
    return EOfferCorrectorResult::Success;
}

EOfferCorrectorResult TTaxiMetaOfferContructor::DoCheckOfferConditions(const TOffersBuildingContext& context, const TUserPermissions& permissions) const {
    Y_UNUSED(context);
    Y_UNUSED(permissions);
    return EOfferCorrectorResult::Success;
}

NDrive::TScheme TTaxiMetaOfferContructor::DoGetScheme(const NDrive::IServer* server) const {
    TSet<TString> fixPointOffers;
    TSet<TString> packOffers;
    auto actions = server->GetDriveAPI()->GetRolesManager()->GetActions();
    for (auto&& action : actions) {
        if (!action) {
            continue;
        }
        if (action->GetType() == TTaxiFixPointOfferConstructor::TypeName()) {
            fixPointOffers.insert(action->GetName());
            continue;
        }
        if (action->GetType() == TTaxiPackOfferConstructor::TypeName()) {
            packOffers.insert(action->GetName());
            continue;
        }
    }
    NDrive::TScheme result = TBase::DoGetScheme(server);
    result.Add<TFSVariants>("fixpoint_offer_name").SetVariants(fixPointOffers);
    result.Add<TFSVariants>("pack_offer_name").SetVariants(packOffers);
    result.Add<TFSVariants>("validation_tag_name").SetVariants(
        NContainer::Keys(server->GetDriveDatabase().GetTagsManager().GetTagsMeta().GetRegisteredTags())
    );
    return result;
}

IOfferReport::TPtr TTaxiMetaOfferContructor::BuildOffer(const TString& name, const TUserPermissions& permissions, const TOffersBuildingContext& context, const NDrive::IServer* server, NDrive::TInfoEntitySession& session) const {
    auto driveApi = server ? server->GetDriveAPI() : nullptr;
    auto rolesManager = driveApi ? driveApi->GetRolesManager() : nullptr;
    if (!rolesManager) {
        session.AddErrorMessage("TaxiMetaOfferContructor::BuildOffer", "roles manager is missing");
        return nullptr;
    }
    auto action = rolesManager->GetAction(name);
    if (!action) {
        session.AddErrorMessage("TaxiMetaOfferContructor::BuildOffer", TStringBuilder() << "action " << name << " is missing");
        return nullptr;
    }
    auto builder = action->GetAs<IOfferBuilderAction>();
    if (!builder) {
        session.AddErrorMessage("TaxiMetaOfferContructor::BuildOffer", TStringBuilder() << "action " << name << " is not an OfferBuilderAction");
        return nullptr;
    }
    TVector<IOfferReport::TPtr> offers;
    auto builderResult = builder->BuildOffersClean(permissions, offers, context, server, session);
    if (builderResult != EOfferCorrectorResult::Success) {
        session.AddErrorMessage("TaxiMetaOfferContructor::BuildOffer", TStringBuilder() << "action " << name << " returned " << builderResult);
        return nullptr;
    }
    if (offers.size() != 1) {
        session.AddErrorMessage("TaxiMetaOfferContructor::BuildOffer", TStringBuilder() << "action " << name << " returned " << offers.size() << " offers");
        return nullptr;
    }
    auto result = std::move(offers[0]);
    if (!result) {
        session.AddErrorMessage("TaxiMetaOfferContructor::BuildOffer", TStringBuilder() << "action " << name << " returned null OfferReport");
        return nullptr;
    }
    if (!result->GetOffer()) {
        session.AddErrorMessage("TaxiMetaOfferContructor::BuildOffer", TStringBuilder() << "action " << name << " returned null offer");
        return nullptr;
    }
    return result;
}

ICommonOffer::TFactory::TRegistrator<TTaxiFixPointOffer> TTaxiFixPointOffer::Registrator(TTaxiFixPointOffer::TypeName());
ICommonOffer::TFactory::TRegistrator<TTaxiPackOffer> TTaxiPackOffer::Registrator(TTaxiPackOffer::TypeName());
ICommonOffer::TFactory::TRegistrator<TTaxiDropoffOffer> TTaxiDropoffOffer::Registrator(TTaxiDropoffOffer::TypeName());
ICommonOffer::TFactory::TRegistrator<TTaxiValidationOffer> TTaxiValidationOffer::Registrator(TTaxiValidationOffer::TypeName());

IOfferBuilderAction::TFactory::TRegistrator<TTaxiFixPointOfferConstructor> TTaxiFixPointOfferConstructor::Registrator(TTaxiFixPointOfferConstructor::TypeName());
IOfferBuilderAction::TFactory::TRegistrator<TTaxiPackOfferConstructor> TTaxiPackOfferConstructor::Registrator(TTaxiPackOfferConstructor::TypeName());
IOfferBuilderAction::TFactory::TRegistrator<TTaxiMetaOfferContructor> TTaxiMetaOfferContructor::Registrator(TTaxiMetaOfferContructor::TypeName());
