#include "types.h"

#include <drive/backend/abstract/frontend.h>

#include <rtline/library/json/field.h>

DECLARE_FIELDS_JSON_SERIALIZER(TOfferVariable<bool>::TNamedValue);
DECLARE_FIELDS_JSON_SERIALIZER(TOfferVariable<ui64>::TNamedValue);
DECLARE_FIELDS_JSON_SERIALIZER(TOfferVariable<TString>::TNamedValue);

DECLARE_FIELDS_JSON_SERIALIZER(TOfferVariable<bool>);
DECLARE_FIELDS_JSON_SERIALIZER(TOfferVariable<ui64>);
DECLARE_FIELDS_JSON_SERIALIZER(TOfferVariable<TString>);

template <>
TString TOfferVariable<bool>::GetDefaultType() {
    return "boolean";
}

template <>
TString TOfferVariable<ui64>::GetDefaultType() {
    return "integer";
}

template <>
TString TOfferVariable<TString>::GetDefaultType() {
    return "string";
}

template <class T>
NJson::TJsonValue TOfferVariable<T>::GetReport(ELocalization locale, const NDrive::IServer& server) const {
    NJson::TJsonValue result;
    auto localization = server.GetLocalization();
    result["id"] = Id;
    result["type"] = GetType();

    result["value"] = NJson::ToJson(Value);
    result["default_value"] = NJson::ToJson(DefaultValue);
    result["cost"] = NJson::ToJson(Cost);
    result["predicted_cost"] = NJson::ToJson(PredictedCost);
    result["title"] = localization ? localization->GetLocalString(locale, Title) : Title;
    result["subtitle"] = localization ? localization->GetLocalString(locale, Subtitle) : Subtitle;
    result["unit"] = localization ? localization->GetLocalString(locale, Unit) : Unit;

    if (NamedValues) {
        NJson::TJsonValue namedValues = NJson::JSON_ARRAY;
        for (auto&& value : *NamedValues) {
            NJson::TJsonValue namedValue;
            namedValue["value"] = value.Value;
            namedValue["name"] = localization ? localization->GetLocalString(locale, value.Name) : value.Name;
            if (value.Cost) {
                namedValue["cost"] = *value.Cost;
            }
            if (value.Link) {
                namedValue["link"] = localization ? localization->GetLocalString(locale, value.Link) : value.Link;
            }
            if (value.LinkTitle) {
                namedValue["link_title"] = localization ? localization->GetLocalString(locale, value.LinkTitle) : value.LinkTitle;
            }
            if (value.Subtitle) {
                namedValue["subtitle"] = localization ? localization->GetLocalString(locale, value.Subtitle) : value.Subtitle;
            }
            if (value.Unit) {
                namedValue["unit"] = localization ? localization->GetLocalString(locale, value.Unit) : value.Unit;
            }
            namedValues.AppendValue(std::move(namedValue));
        }
        result["named_values"] = std::move(namedValues);
    }

    if (Values) {
        result["values"] = NJson::ToJson(Values);
    } else {
        result["minimal_value"] = NJson::ToJson(MinimalValue);
        result["maximal_value"] = NJson::ToJson(MaximalValue);
    }

    return result;
}

NJson::TJsonValue DummyGetReportBool(ELocalization locale, const NDrive::IServer& server) {
    return TOfferVariable<bool>().GetReport(locale, server);
}

NJson::TJsonValue DummyGetReportUi64(ELocalization locale, const NDrive::IServer& server) {
    return TOfferVariable<ui64>().GetReport(locale, server);
}

NJson::TJsonValue DummyGetReportString(ELocalization locale, const NDrive::IServer& server) {
    return TOfferVariable<TString>().GetReport(locale, server);
}
