#include <drive/backend/offers/actions/ut/library/helper.h>

#include <drive/backend/offers/actions/correctors.h>
#include <drive/backend/offers/actions/standart.h>

#include <drive/backend/offers/context.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/testing/unittest/registar.h>

Y_UNIT_TEST_SUITE(OfferCorrectorsSuite) {
    TString DiscountOfferCorrectorDescription = R"(
{
    "id": "234232",
    "action_revision": "111",
    "action_description": "fake",
    "action_id": "my_little_discount",
    "action_type": "offer_corrector_discounts",
    "enabled": true,
    "deprecated": false,
    "parent" : "",
    "action_meta": {
        "discount_timetable": {
            "time_shift": 10800,
            "segments": [{
                "start": 0,
                "value": 10
            }, {
                "start": 374400,
                "value": -10
            }]
        }
    }
}
    )";

    Y_UNIT_TEST(Discount) {
        TStandartOffer offer = BuildOffer(1000, 200, 2000);

        UNIT_ASSERT_DOUBLES_EQUAL(offer.GetDiscountMultiplier(""), 1, 0.001);

        auto d0 = NJson::ReadJsonFastTree(DiscountOfferCorrectorDescription);
        NStorage::TTableRecord d1;
        UNIT_ASSERT(d1.DeserializeFromJson(d0));

        auto action = TUserAction::BuildFromTableRow(d1);
        UNIT_ASSERT(action);
        auto corrector = dynamic_cast<TDiscountOfferCorrector*>(action.Get());
        UNIT_ASSERT(corrector);

        TOffersBuildingContext ctx;
        {
            NDrive::TEntitySession session(nullptr);
            auto copy = ICommonOffer::DeepCopy(offer);
            copy.SetTimestamp(TInstant::Seconds(1000000));
            TStandartOfferReport sor(new TStandartOffer(std::move(copy)), nullptr);
            auto result = corrector->DoApplyForOffer(&sor, {}, ctx, "fake", nullptr, session);
            UNIT_ASSERT(result == EOfferCorrectorResult::Success);
            UNIT_ASSERT_DOUBLES_EQUAL(sor.GetOfferAs<TStandartOffer>()->GetDiscountMultiplier(""), 0.9, 0.001);
        }
        {
            NDrive::TEntitySession session(nullptr);
            auto copy = ICommonOffer::DeepCopy(offer);
            copy.SetTimestamp(TInstant::Seconds(1400000));
            TStandartOfferReport sor(new TStandartOffer(std::move(copy)), nullptr);
            auto result = corrector->DoApplyForOffer(&sor, {}, ctx, "fake", nullptr, session);
            UNIT_ASSERT(result == EOfferCorrectorResult::Success);
            UNIT_ASSERT_DOUBLES_EQUAL(sor.GetOfferAs<TStandartOffer>()->GetDiscountMultiplier(""), 1.1, 0.001);
        }
    }

    TString ProbableDiscount = R"(
{
    "id": "234232",
    "action_revision": "1343",
    "action_description": "fake",
    "action_id": "my_little_discount",
    "action_type": "offer_corrector_discounts",
    "enabled": true,
    "deprecated": false,
    "parent": "",
    "action_meta": {
        "discount": 0.42,
        "probability": 0.5,
        "visible": false
    }
}
    )";

    Y_UNIT_TEST(Probability) {
        auto d0 = NJson::ReadJsonFastTree(ProbableDiscount);
        NStorage::TTableRecord d1;
        UNIT_ASSERT(d1.DeserializeFromJson(d0));

        auto action = TUserAction::BuildFromTableRow(d1);
        UNIT_ASSERT(action);
        auto corrector = dynamic_cast<TSelectiveOfferCorrector*>(action.Get());
        UNIT_ASSERT(corrector);

        TOffersBuildingContext ctx;
        NDrive::TEntitySession session(nullptr);
        {
            ctx.SetReqId("123");
            TStandartOfferReport sor(BuildOfferPtr(1000, 200, 2000), nullptr);
            auto result = corrector->DoApplyForOffer(&sor, {}, ctx, "fake", nullptr, session);
            UNIT_ASSERT(result == EOfferCorrectorResult::Success);
        }
        {
            ctx.SetReqId("1234");
            TStandartOfferReport sor(BuildOfferPtr(1000, 200, 2000), nullptr);
            auto result = corrector->DoApplyForOffer(&sor, {}, ctx, "fake", nullptr, session);
            UNIT_ASSERT(result == EOfferCorrectorResult::Unimplemented);
        }
    }

    TString VisibilityOfferCorrectorDescription = R"(
{
    "id" : 23423,
    "action_revision": "111",
    "action_description": "fake",
    "action_id": "my_little_corrector",
    "action_type": "offer_corrector_visibility",
    "enabled": true,
    "deprecated": false,
    "parent" : "",
    "action_meta": {
        "hidden": [
            "hide_me"
        ],
        "revealed": [
            "show_me"
        ]
    }
}
    )";

    Y_UNIT_TEST(Visibility) {
        auto d0 = NJson::ReadJsonFastTree(VisibilityOfferCorrectorDescription);
        NStorage::TTableRecord d1;
        UNIT_ASSERT(d1.DeserializeFromJson(d0));

        auto action = TUserAction::BuildFromTableRow(d1);
        UNIT_ASSERT(action);
        auto corrector = dynamic_cast<TVisibilityOfferCorrector*>(action.Get());
        UNIT_ASSERT(corrector);

        TOffersBuildingContext ctx;
        NDrive::TEntitySession session(nullptr);
        {
            TStandartOfferReport sor(BuildOfferPtr(1000, 200, 2000), nullptr);
            TStandartOffer& offer = *sor.GetOfferAs<TStandartOffer>();
            offer.SetBehaviourConstructorId("hide_me");
            offer.SetPriceConstructorId("hide_me");
            UNIT_ASSERT(!offer.IsHidden());
            auto result = corrector->DoApplyForOffer(&sor, {}, ctx, "fake", nullptr, session);
            UNIT_ASSERT(result == EOfferCorrectorResult::Success);
            UNIT_ASSERT(offer.IsHidden());
        }
        {
            TStandartOfferReport sor(BuildOfferPtr(1000, 200, 2000), nullptr);
            TStandartOffer& offer = *sor.GetOfferAs<TStandartOffer>();
            offer.SetBehaviourConstructorId("show_me");
            offer.SetPriceConstructorId("show_me");
            offer.SetHidden(true);
            UNIT_ASSERT(offer.IsHidden());
            auto result = corrector->DoApplyForOffer(&sor, {}, ctx, "fake", nullptr, session);
            UNIT_ASSERT(result == EOfferCorrectorResult::Success);
            UNIT_ASSERT(!offer.IsHidden());
        }
        {
            TStandartOfferReport sor(BuildOfferPtr(1000, 200, 2000), nullptr);
            TStandartOffer& offer = *sor.GetOfferAs<TStandartOffer>();
            offer.SetBehaviourConstructorId("ignore_me");
            offer.SetPriceConstructorId("ignore_me");
            UNIT_ASSERT(!offer.IsHidden());
            auto result = corrector->DoApplyForOffer(&sor, {}, ctx, "fake", nullptr, session);
            UNIT_ASSERT(result == EOfferCorrectorResult::Unimplemented);
            UNIT_ASSERT(!offer.IsHidden());
        }
    }

    TString DisableCorrectorDescription = R"(
{
    "id" : 23423,
    "action_revision": "111",
    "action_description": "fake",
    "action_id": "disable_all_models",
    "action_type": "disable_corrector",
    "enabled": true,
    "deprecated": false,
    "parent" : "",
    "action_meta": {
        "correctors": [
            "BadPricesCorrector",
            "WorsePricesCorrector"
        ]
    }
}
    )";

    Y_UNIT_TEST(DisableCorrector) {
        TUserAction::TPtr disableCorrectorAction;
        {
            auto d0 = NJson::ReadJsonFastTree(DisableCorrectorDescription);
            NStorage::TTableRecord d1;
            UNIT_ASSERT(d1.DeserializeFromJson(d0));
            disableCorrectorAction = TUserAction::BuildFromTableRow(d1);
        }
        UNIT_ASSERT(disableCorrectorAction);
        auto disableCorrector = dynamic_cast<TDisableCorrector*>(disableCorrectorAction.Get());
        UNIT_ASSERT(disableCorrector);

        TOffersBuildingContext ctx;
        NDrive::TEntitySession session(nullptr);
        {
            TStandartOfferReport sor(BuildOfferPtr(1000, 200, 2000), nullptr);
            TStandartOffer& offer = *sor.GetOfferAs<TStandartOffer>();
            UNIT_ASSERT(offer.GetDisabledCorrectors().empty());
            auto result = disableCorrector->DoApplyForOffer(&sor, {}, ctx, "fake", nullptr, session);
            UNIT_ASSERT(result == EOfferCorrectorResult::Success);
            UNIT_ASSERT_VALUES_EQUAL(offer.GetDisabledCorrectors().size(), 2);
            UNIT_ASSERT(offer.GetDisabledCorrectors().contains("BadPricesCorrector"));
            UNIT_ASSERT(offer.GetDisabledCorrectors().contains("WorsePricesCorrector"));
        }
    }
}
