#include <drive/backend/offers/actions/ut/library/helper.h>

#include <drive/backend/offers/actions/fix_point.h>
#include <drive/backend/offers/actions/pack.h>

#include <drive/backend/data/chargable.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/device_snapshot/manager.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/system/env.h>

Y_UNIT_TEST_SUITE(FixPointOfferSuite) {

    Y_UNIT_TEST(ReturningDurationFPOfferCorrect) {
        auto offer = BuildOfferPtr<TFixPointOffer>(400, 200, 102400);
        {
            TDiscount discount;
            TDiscount::TDiscountDetails details;
            details.SetAdditionalTime(10).SetTagName("old_state_reservation");
            discount.AddDetails(details);
            offer->AddDiscount(discount);
        }
        {
            TDiscount discount;
            discount.SetDiscount(0.1).SetVisible(false);
            offer->AddDiscount(discount);
        }
        {
            TDiscount discount;
            discount.SetDiscount(0.1).SetVisible(true);
            offer->AddDiscount(discount);
        }
        offer->SetParkingInPack(true);
        offer->SetReturningDuration(TDuration::Days(5000));
        offer->SetPackPrice(10000);
        offer->SetDuration(TDuration::Seconds(40000 - 30000));
        TGeoCoord c1(37.49, 55.39);
        TGeoCoord c2(37.49, 55.41);
        TGeoCoord c3(37.51, 55.41);
        TGeoCoord c4(37.51, 55.39);
        offer->SetFinishArea({c1, c2, c3, c4, c1});
        TVector<IEventsSession<TCarTagHistoryEvent>::TTimeEvent> timeline;
        TVector<TCarTagHistoryEvent> events;
        TOfferPricing pricing(offer);
        TBillingSession session;
        {
            TCarTagHistoryEventConstructor eventBase;
            eventBase.SetObjectId("object").SetTagId("1").SetHistoryUserId("user");
            {
                THolder<TChargableTag> tag(new TChargableTag("old_state_reservation"));
                tag->SetOffer(offer);
                AddEvent(session, eventBase, std::move(tag), TInstant::Seconds(30000), EObjectHistoryAction::SetTagPerformer);
            }
            AddEvent(session, eventBase, "old_state_acceptance", TInstant::Seconds(30600), EObjectHistoryAction::TagEvolve);
            AddEvent(session, eventBase, "old_state_riding", TInstant::Seconds(31200), EObjectHistoryAction::TagEvolve);
            AddEvent(session, eventBase, "old_state_parking", TInstant::Seconds(32000), EObjectHistoryAction::TagEvolve);

            THolder<THistoryDeviceSnapshot> snapshot;
            snapshot.Reset(new THistoryDeviceSnapshot);
            NDrive::TLocation location;
            location.Latitude = 55.4;
            location.Longitude = 38.5;
            location.Timestamp = Now();
            snapshot->SetLocation(location);
            THistoryDeviceSnapshot* sOriginal = snapshot.Release();
            NDrive::IObjectSnapshot::TPtr aSnapshot = sOriginal;

            AddEvent(session, eventBase, "old_state_reservation", TInstant::Seconds(35001), EObjectHistoryAction::TagEvolve, aSnapshot);
            AddEvent(session, eventBase, "old_state_reservation", TInstant::Seconds(35001), EObjectHistoryAction::DropTagPerformer, aSnapshot);
            CheckSum(session, (19336 - 1/6.0 * 200) * (1 - 0.1) * (1 - 0.1));
            TDuration freeDuration;
            TDuration pricedDuration;
            UNIT_ASSERT(session.GetFreeAndPricedDurations(freeDuration, pricedDuration));
            UNIT_ASSERT_VALUES_EQUAL(freeDuration.Seconds(), 10);
            UNIT_ASSERT_VALUES_EQUAL(pricedDuration.Seconds(), 4991);

            location.Latitude = 55.4;
            location.Longitude = 37.5;
            sOriginal->SetLocation(location);
            CheckSum(session, 10000 * (1 - 0.1) * (1 - 0.1));

            UNIT_ASSERT(session.GetFreeAndPricedDurations(freeDuration, pricedDuration));
            UNIT_ASSERT_VALUES_EQUAL(freeDuration.Seconds(), 10);
            UNIT_ASSERT_VALUES_EQUAL(pricedDuration.Seconds(), 4991);
        }
    }

    Y_UNIT_TEST(ReturningDurationFPOfferCorrectOvertime) {
        auto offer = BuildOfferPtr<TFixPointOffer>(400, 200, 102400);
        offer->SetParkingInPack(false);
        {
            TDiscount discount;
            TDiscount::TDiscountDetails details;
            details.SetAdditionalTime(10).SetTagName("old_state_reservation");
            discount.AddDetails(details);
            offer->AddDiscount(discount);
        }
        {
            TDiscount discount;
            discount.SetDiscount(0.1).SetVisible(false);
            offer->AddDiscount(discount);
        }
        {
            TDiscount discount;
            discount.SetDiscount(0.1).SetVisible(true);
            offer->AddDiscount(discount);
        }
        offer->SetFeesForIncorrectFinish(0.1);
        offer->SetReturningDuration(TDuration::Days(2000));
        offer->SetPackPrice(10000);
        offer->SetDuration(TDuration::Seconds(5000));
        TGeoCoord c1(37.49, 55.39);
        TGeoCoord c2(37.49, 55.41);
        TGeoCoord c3(37.51, 55.41);
        TGeoCoord c4(37.51, 55.39);
        offer->SetFinishArea({c1, c2, c3, c4, c1});
        TVector<IEventsSession<TCarTagHistoryEvent>::TTimeEvent> timeline;
        TVector<TCarTagHistoryEvent> events;
        TOfferPricing pricing(offer);
        TBillingSession session;
        {
            TCarTagHistoryEventConstructor eventBase;
            eventBase.SetObjectId("object").SetTagId("1").SetHistoryUserId("user");
            {
                THolder<TChargableTag> tag(new TChargableTag("old_state_reservation"));
                tag->SetOffer(offer);
                AddEvent(session, eventBase, std::move(tag), TInstant::Seconds(30000), EObjectHistoryAction::SetTagPerformer);
            }
            // 10 * 2
            AddEvent(session, eventBase, "old_state_acceptance", TInstant::Seconds(30600), EObjectHistoryAction::TagEvolve);
            // 20 * 2
            AddEvent(session, eventBase, "old_state_riding", TInstant::Seconds(31200), EObjectHistoryAction::TagEvolve);
            // 20 * 2 + [800 / 60 * 4]
            AddEvent(session, eventBase, "old_state_parking", TInstant::Seconds(32000), EObjectHistoryAction::TagEvolve);

            THolder<THistoryDeviceSnapshot> snapshot;
            snapshot.Reset(new THistoryDeviceSnapshot);
            NDrive::TLocation location;
            location.Latitude = 55.4;
            location.Longitude = 38.5;
            location.Timestamp = Now();
            snapshot->SetLocation(location);
            THistoryDeviceSnapshot* sOriginal = snapshot.Release();
            NDrive::IObjectSnapshot::TPtr aSnapshot = sOriginal;

            // 40 + [800 / 60 * 4] + [3000 / 60 * 2]
            AddEvent(session, eventBase, "old_state_riding", TInstant::Seconds(35000), EObjectHistoryAction::TagEvolve);
            // 40 + [800 / 60 * 4] + 100 + [6000 / 60 * 4]
            AddEvent(session, eventBase, "old_state_reservation", TInstant::Seconds(41000), EObjectHistoryAction::TagEvolve, aSnapshot);
            AddEvent(session, eventBase, "old_state_reservation", TInstant::Seconds(41000), EObjectHistoryAction::DropTagPerformer, aSnapshot);
            // * 1.1
            double d = 1.1 * (590 * 2.0 / 60 + 20 + 800.0 / 60 * 4 + 100 + 6000.0 / 60 * 4) * 100 * (1 - 0.1) * (1 - 0.1);
            CheckSum(session, d);
            TDuration freeDuration;
            TDuration pricedDuration;
            UNIT_ASSERT(session.GetFreeAndPricedDurations(freeDuration, pricedDuration));
            UNIT_ASSERT_VALUES_EQUAL(freeDuration.Seconds(), 10);
            UNIT_ASSERT_VALUES_EQUAL(pricedDuration.Seconds(), 10990);

            location.Latitude = 55.4;
            location.Longitude = 37.5;
            sOriginal->SetLocation(location);
            CheckSum(session, ((140 + 100 + 1800 / 60.0 * 4) * 100 - 1 / 6.0 * 200) * (1 - 0.1) * (1 - 0.1));
            UNIT_ASSERT(session.GetFreeAndPricedDurations(freeDuration, pricedDuration));
            UNIT_ASSERT_VALUES_EQUAL(freeDuration.Seconds(), 10);
            UNIT_ASSERT_VALUES_EQUAL(pricedDuration.Seconds(), 10990);
        }
    }

}
