#pragma once
#include <drive/backend/offers/actions/standart.h>
#include <drive/backend/offers/actions/dedicated_fleet.h>

#include <drive/backend/data/chargable.h>
#include <drive/backend/data/dedicated_fleet.h>
#include <drive/backend/data/dedicated_fleet_session.h>
#include <drive/backend/device_snapshot/abstract.h>
#include <drive/backend/tags/tags.h>

class TStandartOffer;

class TTagHistoryEventConstructor: public TTagHistoryEvent {
public:
    static TTagHistoryEventConstructor& BuildEvent(TTagHistoryEventConstructor& base);

    TTagHistoryEventConstructor& SetObjectId(const TString& value) {
        ObjectId = value;
        return *this;
    }

    TTagHistoryEventConstructor& SetTagId(const TString& value) {
        TagId = value;
        return *this;
    }

    TTagHistoryEventConstructor& SetData(ITag::TPtr value) {
        TagData = value;
        return *this;
    }

    ITag* operator->() {
        return TagData.Get();
    }
};

using TCarTagHistoryEventConstructor = TTagHistoryEventConstructor;
using TAccountTagHistoryEventConstructor = TTagHistoryEventConstructor;

class TTestCompilation: public IEventsSession<TTagHistoryEvent>::ICompilation {
public:
    R_READONLY(TOfferStatePtr, OfferState);
    R_READONLY(TOfferPricing, Pricing, TOfferPricing(nullptr));
    R_FIELD(TInstant, Since, TInstant::Zero());
    R_FIELD(TInstant, Until, TInstant::Zero());

private:
    using TTimeEvent = IEventsSession<TTagHistoryEvent>::TTimeEvent;

public:
    TTestCompilation(IOffer::TPtr offer)
        : Offer(offer)
    {
    }

    virtual bool Fill(const TVector<TTimeEvent>& timeline, const TVector<TAtomicSharedPtr<TTagHistoryEvent>>& events) override {
        if (!Offer) {
            return false;
        }
        OfferState = Offer->Calculate(timeline, events, Until, Pricing);
        return !!OfferState;
    }

    virtual NJson::TJsonValue GetReport(ELocalization /*locale*/, const NDrive::IServer* /*server*/, ISessionReportCustomization::TPtr /*customization*/) const override {
        return NJson::JSON_NULL;
    }

    virtual const TString& GetSessionId() const override {
        return Offer->GetOfferId();
    }

private:
    IOffer::TPtr Offer;
};

class TTestFleetOfferCompilation: public IEventsSession<TTagHistoryEvent>::ICompilation {
public:
    R_READONLY(TOfferStatePtr, OfferState);
    R_READONLY(TOfferPricing, Pricing, TOfferPricing(nullptr));
    R_FIELD(TInstant, Since, TInstant::Zero());
    R_FIELD(TInstant, Until, TInstant::Zero());

private:
    using TTimeEvent = IEventsSession<TTagHistoryEvent>::TTimeEvent;

public:
    TTestFleetOfferCompilation(TDedicatedFleetOffer::TPtr offer)
        : Offer(offer)
    {
    }

    virtual bool Fill(const TVector<TTimeEvent>& timeline, const TVector<TAtomicSharedPtr<TTagHistoryEvent>>& events) override {
        if (!Offer) {
            return false;
        }
        OfferState = Offer->Calculate(timeline, events, Until);
        return !!OfferState;
    }

    virtual NJson::TJsonValue GetReport(ELocalization /*locale*/, const NDrive::IServer* /*server*/, ISessionReportCustomization::TPtr /*customization*/) const override {
        return NJson::JSON_NULL;
    }

    virtual const TString& GetSessionId() const override {
        return Offer->GetOfferId();
    }

private:
    TDedicatedFleetOffer::TPtr Offer;
};


template <class T = TStandartOffer>
T BuildOffer(const ui32 ridingPrice, const ui32 parkingPrice, const ui32 debtThreshold) {
    TFullPricesContext context;
    context.MutableParking().SetPrice(parkingPrice);
    context.MutableRiding().SetPrice(ridingPrice);
    T offer(context);
    offer.SetDebtThreshold(debtThreshold);
    return offer;
}

template <class T = TStandartOffer>
TAtomicSharedPtr<T> BuildOfferPtr(const ui32 ridingPrice, const ui32 parkingPrice, const ui32 debtThreshold) {
    return MakeAtomicShared<T>(BuildOffer<T>(ridingPrice, parkingPrice, debtThreshold));
}

class TCommonEvents {
public:
    static TVector<TAtomicSharedPtr<TTagHistoryEventConstructor>> EventsStorage;
    static std::atomic<NDrive::TEventId> CurrentCounter;
};

template<class TSession = TBillingSession, class TSessionSelector = TBillingSessionSelector, bool Perform = true>
void AddEvent(TSession& session, TCarTagHistoryEventConstructor& eventBase, THolder<ITag>&& tag, const TInstant instant, const EObjectHistoryAction action, NDrive::IObjectSnapshot::TPtr objSnapshot = nullptr) {
    tag->SetObjectSnapshot(objSnapshot);
    if (Perform) {
        if (action != EObjectHistoryAction::DropTagPerformer) {
            tag->SetPerformer(eventBase.GetHistoryUserId());
        } else {
            tag->SetPerformer("");
        }
    }
    TCarTagHistoryEventConstructor& event = TCarTagHistoryEventConstructor::BuildEvent(eventBase);
    event.SetData(tag.Release())
        .SetHistoryInstant(instant).SetHistoryEventId(++TCommonEvents::CurrentCounter).SetHistoryAction(action);
    session.AddEvent(new TTagHistoryEvent(event), TSessionSelector::AcceptImpl(event));
}

template<class TSession = TBillingSession, class TSessionSelector = TBillingSessionSelector, class TTag = TChargableTag, bool Perform = true>
void AddEvent(TSession& session, TCarTagHistoryEventConstructor& eventBase, const TString& tagName, const TInstant instant, const EObjectHistoryAction action, NDrive::IObjectSnapshot::TPtr objSnapshot = nullptr) {
    THolder<ITag> tag(new TTag(tagName));
    if (Perform) {
        if (action != EObjectHistoryAction::DropTagPerformer) {
            tag->SetPerformer("performer");
        } else {
            tag->SetPerformer("");
        }
    }
    AddEvent<TSession, TSessionSelector, Perform>(session, eventBase, std::move(tag), instant, action, objSnapshot);
}

template<class TSession, class TSessionSelector, class TTag>
void AddAccountEvent(TSession& session, TAccountTagHistoryEventConstructor& eventBase, const TString& tagName, const TInstant instant, const EObjectHistoryAction action) {
    THolder<ITag> tag(new TTag(tagName));
    AddEvent<TSession, TSessionSelector>(session, eventBase, std::move(tag), instant, action);
}

template<class TSession, class TSessionSelector>
void AddAccountEvent(TSession& session, TAccountTagHistoryEventConstructor& eventBase, THolder<ITag>&& tag, const TInstant instant, const EObjectHistoryAction action) {
    TAccountTagHistoryEventConstructor& event = TAccountTagHistoryEventConstructor::BuildEvent(eventBase);
    event.SetData(tag.Release())
        .SetHistoryInstant(instant).SetHistoryEventId(++TCommonEvents::CurrentCounter).SetHistoryAction(action);
    session.AddEvent(new TTagHistoryEvent(event), TSessionSelector::AcceptImpl(event));
}

void CheckSum(TBillingSession& session, const ui32 sum);
void CheckBillingSum(TBillingSession& session, const ui32 sum);
