#include <drive/backend/offers/actions/ut/library/helper.h>

#include <drive/backend/offers/actions/pack.h>

#include <drive/backend/data/chargable.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/device_snapshot/manager.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/system/env.h>

Y_UNIT_TEST_SUITE(PackOfferSuite) {
    Y_UNIT_TEST(PackOffer) {
        auto offer = BuildOfferPtr<TPackOffer>(200, 100, 102400);
        offer->SetPackPrice(10000);
        offer->SetDuration(TDuration::Seconds(40000 - 30000));
        TVector<IEventsSession<TCarTagHistoryEvent>::TTimeEvent> timeline;
        TVector<TCarTagHistoryEvent> events;
        TBillingSession session;
        {
            TCarTagHistoryEventConstructor eventBase;
            eventBase.SetObjectId("object").SetTagId("1").SetHistoryUserId("user");
            {
                THolder<TChargableTag> tag(new TChargableTag("old_state_reservation"));
                tag->SetOffer(offer);
                AddEvent(session, eventBase, std::move(tag), TInstant::Seconds(30300), EObjectHistoryAction::SetTagPerformer);
            }
            AddEvent(session, eventBase, "old_state_acceptance", TInstant::Seconds(31000), EObjectHistoryAction::TagEvolve);
            AddEvent(session, eventBase, "old_state_riding", TInstant::Seconds(32000), EObjectHistoryAction::TagEvolve);
            AddEvent(session, eventBase, "old_state_parking", TInstant::Seconds(33000), EObjectHistoryAction::TagEvolve);
            AddEvent(session, eventBase, "old_state_reservation", TInstant::Seconds(35000), EObjectHistoryAction::TagEvolve);
            AddEvent(session, eventBase, "old_state_reservation", TInstant::Seconds(35000), EObjectHistoryAction::DropTagPerformer);
        }
        {
            TBillingSession::TBillingCompilation compilation;
            compilation.SetUntil(TInstant::Max());
            UNIT_ASSERT(session.FillCompilation(compilation));
            UNIT_ASSERT_C(Abs(compilation.GetReportSumPrice() - 10000) < 1e-5, TStringBuilder() << compilation.GetReportSumPrice());
        }
    }

    class TPackOfferNoParking: public TPackOffer {
    public:
        using TPackOffer::TPackOffer;

        bool GetIsParkingIncludeInPack() const override {
            return false;
        }
    };

    Y_UNIT_TEST(ReturningDurationPackOfferCorrectWithParking) {
        auto offer = BuildOfferPtr<TPackOfferNoParking>(200, 100, 102400);
        offer->SetReturningDuration(TDuration::Seconds(5000));
        offer->SetPackPrice(10000);
        offer->SetDuration(TDuration::Seconds(40000 - 30000));
        TBillingSession session;
        {
            TCarTagHistoryEventConstructor eventBase;
            eventBase.SetObjectId("object").SetTagId("1").SetHistoryUserId("user");
            {
                THolder<TChargableTag> tag(new TChargableTag("old_state_reservation"));
                tag->SetOffer(offer);
                AddEvent(session, eventBase, std::move(tag), TInstant::Seconds(30000), EObjectHistoryAction::SetTagPerformer);
            }
            AddEvent(session, eventBase, "old_state_acceptance", TInstant::Seconds(30600), EObjectHistoryAction::TagEvolve);
            CheckBillingSum(session, 1000);
            CheckSum(session, 1000);
            AddEvent(session, eventBase, "old_state_riding", TInstant::Seconds(31200), EObjectHistoryAction::TagEvolve);
            CheckBillingSum(session, 2000);
            CheckSum(session, 2000 + 10000);
            AddEvent(session, eventBase, "old_state_parking", TInstant::Seconds(32000), EObjectHistoryAction::TagEvolve);
            CheckBillingSum(session, 2000 + 800.0 / 60 * 200);
            CheckSum(session, 2000 + 10000);
            AddEvent(session, eventBase, "old_state_reservation", TInstant::Seconds(36300), EObjectHistoryAction::TagEvolve);
            AddEvent(session, eventBase, "old_state_reservation", TInstant::Seconds(36300), EObjectHistoryAction::DropTagPerformer);
        }
        CheckSum(session, 19166);
        CheckBillingSum(session, 19166);
    }

    Y_UNIT_TEST(ReturningDurationPackOfferIncorrect) {
        auto offer = BuildOfferPtr<TPackOffer>(200, 100, 102400);
        offer->SetReturningDuration(TDuration::Seconds(5000));
        offer->SetPackPrice(10000);
        offer->SetDuration(TDuration::Seconds(40000 - 30000));
        TVector<IEventsSession<TCarTagHistoryEvent>::TTimeEvent> timeline;
        TVector<TCarTagHistoryEvent> events;
        TBillingSession session;
        {
            TCarTagHistoryEventConstructor eventBase;
            eventBase.SetObjectId("object").SetTagId("1").SetHistoryUserId("user");
            {
                THolder<TChargableTag> tag(new TChargableTag("old_state_reservation"));
                tag->SetOffer(offer);
                AddEvent(session, eventBase, std::move(tag), TInstant::Seconds(30000), EObjectHistoryAction::SetTagPerformer);
            }
            AddEvent(session, eventBase, "old_state_acceptance", TInstant::Seconds(30600), EObjectHistoryAction::TagEvolve);
            AddEvent(session, eventBase, "old_state_riding", TInstant::Seconds(31200), EObjectHistoryAction::TagEvolve);
            AddEvent(session, eventBase, "old_state_parking", TInstant::Seconds(32000), EObjectHistoryAction::TagEvolve);
            AddEvent(session, eventBase, "old_state_reservation", TInstant::Seconds(34000), EObjectHistoryAction::TagEvolve);
            AddEvent(session, eventBase, "old_state_reservation", TInstant::Seconds(34000), EObjectHistoryAction::DropTagPerformer);
        }
        CheckSum(session, 7999);
    }

    Y_UNIT_TEST(ReturningDurationPackOfferCorrect) {
        auto offer = BuildOfferPtr<TPackOffer>(200, 100, 102400);
        offer->SetReturningDuration(TDuration::Seconds(5000));
        offer->SetPackPrice(10000);
        offer->SetDuration(TDuration::Seconds(40000 - 30000));
        TVector<IEventsSession<TCarTagHistoryEvent>::TTimeEvent> timeline;
        TVector<TCarTagHistoryEvent> events;
        TBillingSession session;
        {
            TCarTagHistoryEventConstructor eventBase;
            eventBase.SetObjectId("object").SetTagId("1").SetHistoryUserId("user");
            {
                THolder<TChargableTag> tag(new TChargableTag("old_state_reservation"));
                tag->SetOffer(offer);
                AddEvent(session, eventBase, std::move(tag), TInstant::Seconds(30000), EObjectHistoryAction::SetTagPerformer);
            }
            AddEvent(session, eventBase, "old_state_acceptance", TInstant::Seconds(30600), EObjectHistoryAction::TagEvolve);
            AddEvent(session, eventBase, "old_state_riding", TInstant::Seconds(31200), EObjectHistoryAction::TagEvolve);
            AddEvent(session, eventBase, "old_state_parking", TInstant::Seconds(32000), EObjectHistoryAction::TagEvolve);
            AddEvent(session, eventBase, "old_state_reservation", TInstant::Seconds(35001), EObjectHistoryAction::TagEvolve);
            AddEvent(session, eventBase, "old_state_reservation", TInstant::Seconds(35001), EObjectHistoryAction::DropTagPerformer);
        }
        CheckSum(session, 10000);
    }

    Y_UNIT_TEST(PackOfferWide) {
        auto offer = BuildOfferPtr<TPackOffer>(200, 100, 102400);
        offer->SetPackPrice(10000);
        offer->SetDuration(TDuration::Seconds(40000 - 30000));
        TVector<IEventsSession<TCarTagHistoryEvent>::TTimeEvent> timeline;
        TVector<TCarTagHistoryEvent> events;
        TBillingSession session;
        {
            TCarTagHistoryEventConstructor eventBase;
            eventBase.SetObjectId("object").SetTagId("1").SetHistoryUserId("user");
            AddEvent(session, eventBase, "old_state_reservation", TInstant::Seconds(30000), EObjectHistoryAction::SetTagPerformer);
            AddEvent(session, eventBase, "old_state_acceptance", TInstant::Seconds(30000), EObjectHistoryAction::TagEvolve);
            AddEvent(session, eventBase, "old_state_riding", TInstant::Seconds(31000), EObjectHistoryAction::TagEvolve);
            AddEvent(session, eventBase, "old_state_parking", TInstant::Seconds(32000), EObjectHistoryAction::TagEvolve);
            AddEvent(session, eventBase, "old_state_reservation", TInstant::Seconds(35000), EObjectHistoryAction::TagEvolve);
            AddEvent(session, eventBase, "old_state_reservation", TInstant::Seconds(45000), EObjectHistoryAction::DropTagPerformer);
        }
        {
            TTestCompilation compilation(offer);
            compilation.SetSince(TInstant::Zero()).SetUntil(TInstant::Max());
            UNIT_ASSERT(session.FillCompilation(compilation));
            UNIT_ASSERT_C(Abs(compilation.GetPricing().GetReportSumOriginalPrice() - 10000 - 100 * (5000.0 / 60)) < 1, TStringBuilder() << compilation.GetPricing().GetReportSumOriginalPrice());
        }
    }

    Y_UNIT_TEST(PackOfferWideNotFinished) {
        auto offer = BuildOfferPtr<TPackOffer>(200, 100, 102400);
        offer->SetPackPrice(10000);
        offer->SetDuration(TDuration::Seconds(40000 - 30000));
        TVector<IEventsSession<TCarTagHistoryEvent>::TTimeEvent> timeline;
        TVector<TCarTagHistoryEvent> events;
        TBillingSession session;
        {
            TCarTagHistoryEventConstructor eventBase;
            eventBase.SetObjectId("object").SetTagId("1").SetHistoryUserId("user");
            AddEvent(session, eventBase, "old_state_reservation", TInstant::Seconds(30000), EObjectHistoryAction::SetTagPerformer);
            AddEvent(session, eventBase, "old_state_acceptance", TInstant::Seconds(30000), EObjectHistoryAction::TagEvolve);
            AddEvent(session, eventBase, "old_state_riding", TInstant::Seconds(31000), EObjectHistoryAction::TagEvolve);
            AddEvent(session, eventBase, "old_state_parking", TInstant::Seconds(32000), EObjectHistoryAction::TagEvolve);
        }
        TInstantGuard ig(TInstant::Seconds(45000));
        session.SignalRefresh(TInstant::Seconds(45000));
        {
            TTestCompilation compilation(offer);
            compilation.SetSince(TInstant::Zero()).SetUntil(TInstant::Max());
            UNIT_ASSERT(session.FillCompilation(compilation));
            UNIT_ASSERT_C(Abs(compilation.GetPricing().GetReportSumOriginalPrice() - 10000 - 100 * (5000.0 / 60)) < 1, TStringBuilder() << compilation.GetPricing().GetReportSumOriginalPrice());
        }
    }

    Y_UNIT_TEST(PackOfferReserveNotFinished) {
        auto offer = BuildOfferPtr<TPackOffer>(200, 100, 102400);
        offer->SetPackPrice(10000);
        offer->SetDuration(TDuration::Seconds(40000 - 30000));
        TVector<IEventsSession<TCarTagHistoryEvent>::TTimeEvent> timeline;
        TVector<TCarTagHistoryEvent> events;
        TBillingSession session;
        {
            TCarTagHistoryEventConstructor eventBase;
            eventBase.SetObjectId("object").SetTagId("1").SetHistoryUserId("user");
            {
                THolder<TChargableTag> tag(new TChargableTag("old_state_reservation"));
                tag->SetOffer(offer);
                AddEvent(session, eventBase, std::move(tag), TInstant::Seconds(30000), EObjectHistoryAction::SetTagPerformer);
            }
            AddEvent(session, eventBase, "old_state_acceptance", TInstant::Seconds(33000), EObjectHistoryAction::TagEvolve);
        }
        TInstantGuard ig(TInstant::Seconds(35000));
        {
            TBillingSession::TBillingCompilation compilation;
            UNIT_ASSERT(session.FillCompilation(compilation));
            UNIT_ASSERT_C(Abs(compilation.GetReportSumPrice() - 1.0 * 10000) < 1, TStringBuilder() << compilation.GetReportSumPrice());
        }
    }

    Y_UNIT_TEST(PackOfferReserveNotFinishedWide) {
        auto offer = BuildOfferPtr<TPackOffer>(200, 100, 102400);
        offer->SetPackPrice(10000);
        offer->SetDuration(TDuration::Seconds(40000 - 30000));
        TVector<IEventsSession<TCarTagHistoryEvent>::TTimeEvent> timeline;
        TVector<TCarTagHistoryEvent> events;
        TBillingSession session;
        {
            TCarTagHistoryEventConstructor eventBase;
            eventBase.SetObjectId("object").SetTagId("1").SetHistoryUserId("user");
            {
                THolder<TChargableTag> tag(new TChargableTag("old_state_reservation"));
                tag->SetOffer(offer);
                AddEvent(session, eventBase, std::move(tag), TInstant::Seconds(30000), EObjectHistoryAction::SetTagPerformer);
            }
            AddEvent(session, eventBase, "old_state_acceptance", TInstant::Seconds(30000), EObjectHistoryAction::TagEvolve);
        }
        TInstantGuard ig(TInstant::Seconds(45000));
        session.SignalRefresh(TInstant::Seconds(45000));
        {
            TBillingSession::TBillingCompilation compilation;
            UNIT_ASSERT(session.FillCompilation(compilation));
            UNIT_ASSERT_C(Abs(compilation.GetReportSumPrice() - 10000 - (100 * (15000.0 / 60) - 100 * (10000.0 / 60))) < 1, TStringBuilder() << compilation.GetReportSumPrice());
        }
    }

    Y_UNIT_TEST(PackPriceSchedule) {
        auto now = Now();
        TPriceSchedule schedule;
        schedule.AddQuantum(now, 0, 10);
        schedule.AddQuantum(now + TDuration::Days(1), 10, 42);
        schedule.AddQuantum(now + TDuration::Days(2), 20, 43);
        schedule.AddQuantum(now + TDuration::Days(3), 30, 1);

        UNIT_ASSERT_VALUES_EQUAL(schedule.GetPrice(TInstant::Zero(), 0), 0);
        UNIT_ASSERT_VALUES_EQUAL(schedule.GetPrice(now, 0), 0);
        UNIT_ASSERT_VALUES_EQUAL(schedule.GetPrice(now + TDuration::Seconds(1), 0), 10);
        UNIT_ASSERT_VALUES_EQUAL(schedule.GetPrice(now + TDuration::Hours(48), 0), 10 + 42);
        UNIT_ASSERT_VALUES_EQUAL(schedule.GetPrice(now + TDuration::Hours(49), 0), 10 + 42 + 43);
        UNIT_ASSERT_VALUES_EQUAL(schedule.GetPrice(TInstant::Max(), 0), 10 + 42 + 43 + 1);
        UNIT_ASSERT_VALUES_EQUAL(schedule.GetPrice(TInstant::Zero(), 9), 0);
        UNIT_ASSERT_VALUES_EQUAL(schedule.GetPrice(TInstant::Zero(), 10), 0);
        UNIT_ASSERT_VALUES_EQUAL(schedule.GetPrice(TInstant::Zero(), 21), 0);
        UNIT_ASSERT_VALUES_EQUAL(schedule.GetPrice(now + TDuration::Hours(50), 12), 10 + 42 + 43);

        auto next = schedule.GetNextQuantum(now + TDuration::Hours(42), 14);
        UNIT_ASSERT(next);
        UNIT_ASSERT_VALUES_EQUAL(next->Value, 43);
        next = schedule.GetNextQuantum(now + TDuration::Hours(19), 30);
        UNIT_ASSERT(next);
        UNIT_ASSERT_VALUES_EQUAL(next->Value, 42);
        next = schedule.GetNextQuantum(now + TDuration::Hours(72), 30);
        UNIT_ASSERT(next);
        UNIT_ASSERT_VALUES_EQUAL(next->Value, 1);
        next = schedule.GetNextQuantum(now + TDuration::Hours(73), 30);
        UNIT_ASSERT(!next);
    }

    Y_UNIT_TEST(PackPriceScheduleWithMileage) {
        auto now = Now();
        TPriceSchedule schedule(true);
        schedule.AddQuantum(now, 0, 10);
        schedule.AddQuantum(now + TDuration::Days(1), 10, 42);
        schedule.AddQuantum(now + TDuration::Days(2), 20, 43);
        schedule.AddQuantum(now + TDuration::Days(3), 30, 1);

        UNIT_ASSERT_VALUES_EQUAL(schedule.GetPrice(TInstant::Zero(), 0), 0);
        UNIT_ASSERT_VALUES_EQUAL(schedule.GetPrice(now, 0), 0);
        UNIT_ASSERT_VALUES_EQUAL(schedule.GetPrice(now + TDuration::Seconds(1), 0), 10);
        UNIT_ASSERT_VALUES_EQUAL(schedule.GetPrice(now + TDuration::Hours(48), 0), 10 + 42);
        UNIT_ASSERT_VALUES_EQUAL(schedule.GetPrice(now + TDuration::Hours(49), 0), 10 + 42 + 43);
        UNIT_ASSERT_VALUES_EQUAL(schedule.GetPrice(TInstant::Max(), 0), 10 + 42 + 43 + 1);
        UNIT_ASSERT_VALUES_EQUAL(schedule.GetPrice(TInstant::Zero(), 9), 10);
        UNIT_ASSERT_VALUES_EQUAL(schedule.GetPrice(TInstant::Zero(), 10), 10);
        UNIT_ASSERT_VALUES_EQUAL(schedule.GetPrice(TInstant::Zero(), 21), 10 + 42 + 43);
        UNIT_ASSERT_VALUES_EQUAL(schedule.GetPrice(now + TDuration::Hours(50), 12), 10 + 42 + 43);

        auto next = schedule.GetNextQuantum(now + TDuration::Hours(42), 14);
        UNIT_ASSERT(next);
        UNIT_ASSERT_VALUES_EQUAL(next->Value, 43);
        next = schedule.GetNextQuantum(now + TDuration::Hours(19), 30);
        UNIT_ASSERT(next);
        UNIT_ASSERT_VALUES_EQUAL(next->Value, 1);
        next = schedule.GetNextQuantum(now + TDuration::Hours(72), 30);
        UNIT_ASSERT(next);
        UNIT_ASSERT_VALUES_EQUAL(next->Value, 1);
        next = schedule.GetNextQuantum(now + TDuration::Hours(19), 31);
        UNIT_ASSERT(!next);
    }
}
