#include <drive/backend/offers/actions/ut/library/helper.h>

#include <drive/backend/offers/actions/pack.h>
#include <drive/backend/offers/actions/standart.h>

#include <drive/backend/data/chargable.h>
#include <drive/backend/data/device_tags.h>

#include <drive/telematics/protocol/proto/sensor.pb.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/system/env.h>

Y_UNIT_TEST_SUITE(StandartOfferSuite) {
    Y_UNIT_TEST(Simple) {
        auto offer = BuildOfferPtr(200, 100, 102400);
        {
            TDiscount discount;
            discount.SetDiscount(0.5);
            offer->AddDiscount(discount);
        }
        {
            TDiscount discount;
            {
                TDiscount::TDiscountDetails detail;
                detail.SetAdditionalTime(2 * 60).SetTagName("old_state_acceptance");
                discount.AddDetails(detail);
            }
            {
                TDiscount::TDiscountDetails detail;
                detail.SetAdditionalTime(5 * 60).SetTagName("old_state_reservation");
                discount.AddDetails(detail);
            }
            offer->AddDiscount(discount);
        }
        offer->MutableAcceptance().SetPrice(4242, "testing");

        auto copy = offer->Clone();
        auto copiedStandartOffer = std::dynamic_pointer_cast<TStandartOffer>(copy);
        UNIT_ASSERT_VALUES_EQUAL(copiedStandartOffer->GetAcceptance().GetPrice(), 4242);

        TVector<IEventsSession<TCarTagHistoryEvent>::TTimeEvent> timeline;
        TVector<TCarTagHistoryEvent> events;
        TBillingSession session;
        {
            TCarTagHistoryEventConstructor eventBase;
            eventBase.SetObjectId("object").SetTagId("1").SetHistoryUserId("user");
            AddEvent(session, eventBase, "old_state_reservation", TInstant::Seconds(1000), EObjectHistoryAction::SetTagPerformer);
            AddEvent(session, eventBase, "old_state_acceptance", TInstant::Seconds(1020), EObjectHistoryAction::TagEvolve);
            AddEvent(session, eventBase, "old_state_riding", TInstant::Seconds(1040), EObjectHistoryAction::TagEvolve);
            AddEvent(session, eventBase, "old_state_parking", TInstant::Seconds(1140), EObjectHistoryAction::TagEvolve);
            AddEvent(session, eventBase, "old_state_riding", TInstant::Seconds(1240), EObjectHistoryAction::TagEvolve);
            AddEvent(session, eventBase, "old_state_reservation", TInstant::Seconds(1340), EObjectHistoryAction::TagEvolve);
            AddEvent(session, eventBase, "old_state_reservation", TInstant::Seconds(1341), EObjectHistoryAction::DropTagPerformer);
        }
        {
            TTestCompilation compilation(offer);
            compilation.SetSince(TInstant::Zero()).SetUntil(TInstant::Zero());
            UNIT_ASSERT(session.FillCompilation(compilation));
            UNIT_ASSERT(compilation.GetPricing().GetPrices().empty());
        }
        {
            TTestCompilation compilation(offer);
            compilation.SetSince(TInstant::Seconds(1000)).SetUntil(TInstant::Seconds(1030));
            UNIT_ASSERT(session.FillCompilation(compilation));
            UNIT_ASSERT(!compilation.GetPricing().GetPrices("acceptance_cost"));
        }
        {
            TTestCompilation compilation(offer);
            compilation.SetSince(TInstant::Seconds(1000)).SetUntil(TInstant::Seconds(1500));
            UNIT_ASSERT(session.FillCompilation(compilation));
            UNIT_ASSERT_VALUES_EQUAL(compilation.GetPricing().GetPrices("acceptance_cost")->GetOriginalPrice(), 4242 / 2);
            UNIT_ASSERT_VALUES_EQUAL(compilation.GetPricing().GetPrices("old_state_riding")->GetDuration().Seconds(), 200);
            UNIT_ASSERT_VALUES_EQUAL(compilation.GetPricing().GetPrices("old_state_parking")->GetDuration().Seconds(), 100);
            UNIT_ASSERT_VALUES_EQUAL(compilation.GetPricing().GetPrices("old_state_acceptance")->GetDuration().Seconds(), 20);
            UNIT_ASSERT_VALUES_EQUAL(compilation.GetPricing().GetPrices("old_state_reservation")->GetDuration().Seconds(), 21);
            UNIT_ASSERT_DOUBLES_EQUAL(compilation.GetPricing().GetReportSumOriginalPrice(), 0.5 * (200.0 / 60 * 200 + 100.0 / 60 * 100 + 4242), 1 + 1);
            UNIT_ASSERT_DOUBLES_EQUAL(compilation.GetPricing().GetReportSumOriginalPrice(), compilation.GetPricing().GetReportSumPrice(), 1);
            INFO_LOG << compilation.GetPricing().GetReport() << Endl;
            INFO_LOG << compilation.GetPricing().GetReportSumPrice() << Endl;
            INFO_LOG << compilation.GetPricing().GetReportSumOriginalPrice() << Endl;

            double sumPricePred = 0;
            for (ui32 i = 900; i < 1400; ++i) {
                TTestCompilation compilation1(offer);
                compilation1.SetSince(TInstant::Zero()).SetUntil(TInstant::Seconds(i));
                UNIT_ASSERT(session.FillCompilation(compilation1));
                UNIT_ASSERT(sumPricePred <= compilation1.GetPricing().GetReportSumPrice());
                sumPricePred = compilation1.GetPricing().GetReportSumPrice();
            }
        }
    }

    Y_UNIT_TEST(Serialization) {
        NDrive::NProto::TOffer report;
        {
            auto standartOffer = BuildOfferPtr(213, 132, 132210);
            standartOffer->SetPaymentDiscretization(123);
            {
                TDiscount discount;
                {
                    TDiscount::TDiscountDetails detail;
                    TTimeRestrictionsPool<TTimeRestriction> pool;
                    {
                        TTimeRestriction restriction;
                        restriction.SetTimeRestriction(2350, 530);
                        pool.Add(restriction).SetTimezoneShift(1);
                    }
                    {
                        TTimeRestriction restriction;
                        restriction.SetTimeRestriction(1350, 1530);
                        pool.Add(restriction).SetTimezoneShift(3);
                    }
                    detail.SetFreeTimetable(pool).SetTagName("old_state_parking");
                    discount.AddDetails(detail);
                }
                {
                    TDiscount::TDiscountDetails detail;
                    detail.SetAdditionalTime(1 * 60).SetTagName("old_state_acceptance");
                    discount.AddDetails(detail);
                }
                {
                    TDiscount::TDiscountDetails detail;
                    detail.SetAdditionalTime(2 * 60).SetTagName("old_state_reservation");
                    discount.AddDetails(detail);
                }
                standartOffer->AddDiscount(discount);
            }
            {
                TDiscount discount;
                {
                    TDiscount::TDiscountDetails detail;
                    TTimeRestrictionsPool<TTimeRestriction> pool;
                    TTimeRestriction restriction;
                    restriction.SetTimeRestriction(150, 230);
                    pool.Add(restriction).SetTimezoneShift(0);
                    detail.SetFreeTimetable(pool).SetTagName("old_state_riding");
                    discount.AddDetails(detail);
                }
                {
                    TDiscount::TDiscountDetails detail;
                    detail.SetAdditionalTime(3 * 60).SetTagName("old_state_acceptance");
                    discount.AddDetails(detail);
                }
                {
                    TDiscount::TDiscountDetails detail;
                    detail.SetAdditionalTime(4 * 60).SetTagName("old_state_reservation");
                    discount.AddDetails(detail);
                }
                standartOffer->AddDiscount(discount);
            }
            {
                TDiscount discount;
                {
                    TDiscount::TDiscountDetails detail;
                    TTimeRestrictionsPool<TTimeRestriction> pool;
                    TTimeRestriction restriction;
                    restriction.SetTimeRestriction(150, 230);
                    pool.Add(restriction).SetTimezoneShift(0);
                    detail.SetFreeTimetable(pool).SetTagName("old_state_parking");
                    discount.AddDetails(detail);
                }
                {
                    TDiscount::TDiscountDetails detail;
                    detail.SetAdditionalTime(5 * 60).SetTagName("old_state_acceptance");
                    discount.AddDetails(detail);
                }
                {
                    TDiscount::TDiscountDetails detail;
                    detail.SetAdditionalTime(6 * 60).SetTagName("old_state_reservation");
                    discount.AddDetails(detail);
                }
                standartOffer->AddDiscount(discount);
            }

            report = standartOffer->SerializeToProto();
        }
        {
            THolder<TStandartOffer> standartOffer(new TStandartOffer());
            UNIT_ASSERT(standartOffer->DeserializeFromProto(report));
            UNIT_ASSERT_VALUES_EQUAL(standartOffer->GetFreeDuration("old_state_reservation"), TDuration::Minutes(12));
            UNIT_ASSERT_VALUES_EQUAL(standartOffer->GetFreeDuration("old_state_acceptance"), TDuration::Minutes(9));
            UNIT_ASSERT_VALUES_EQUAL(standartOffer->GetPaymentDiscretization(), 123);
            UNIT_ASSERT_VALUES_EQUAL(standartOffer->GetParking().GetPrice(), 132);
            UNIT_ASSERT_VALUES_EQUAL(standartOffer->GetRiding().GetPrice(), 213);
            UNIT_ASSERT_VALUES_EQUAL(standartOffer->GetSummaryHiddenDiscount().GetDetails("old_state_parking").GetFreeTimetable()->RestrictionsCount(), 3);
            UNIT_ASSERT_VALUES_EQUAL(standartOffer->GetSummaryHiddenDiscount().GetDetails("old_state_parking").GetFreeTimetable()->GetWidestRestriction(Now(), Now() + TDuration::Days(2))->GetTimeFrom(), 2350);
            UNIT_ASSERT_VALUES_EQUAL(standartOffer->GetSummaryHiddenDiscount().GetDetails("old_state_parking").GetFreeTimetable()->GetWidestRestriction(Now(), Now() + TDuration::Days(2))->GetTimeTo(), 530);
        }
    }

    Y_UNIT_TEST(FreeParking) {
        auto offer = BuildOfferPtr(200, 100, 102400);
        {
            TDiscount discount;
            {
                TDiscount::TDiscountDetails detail;
                TTimeRestrictionsPool<TTimeRestriction> pool;
                TTimeRestriction restriction;
                restriction.SetTimeRestriction(2350, 530);
                pool.Add(restriction).SetTimezoneShift(0);
                detail.SetFreeTimetable(pool).SetTagName("old_state_parking");
                discount.AddDetails(detail);
            }
            {
                TDiscount::TDiscountDetails detail;
                detail.SetAdditionalTime(2 * 60).SetTagName("old_state_acceptance");
                discount.AddDetails(detail);
            }
            {
                TDiscount::TDiscountDetails detail;
                detail.SetAdditionalTime(5 * 60).SetTagName("old_state_reservation");
                discount.AddDetails(detail);
            }
            offer->AddDiscount(discount);
        }
        TVector<IEventsSession<TCarTagHistoryEvent>::TTimeEvent> timeline;
        TVector<TCarTagHistoryEvent> events;
        TBillingSession session;
        {
            TCarTagHistoryEventConstructor eventBase;
            eventBase.SetObjectId("object").SetTagId("1").SetHistoryUserId("user");
            AddEvent(session, eventBase, "old_state_reservation", TInstant::Seconds(17800), EObjectHistoryAction::SetTagPerformer);
            AddEvent(session, eventBase, "old_state_acceptance", TInstant::Seconds(17800), EObjectHistoryAction::TagEvolve);
            AddEvent(session, eventBase, "old_state_riding", TInstant::Seconds(17800), EObjectHistoryAction::TagEvolve);
            AddEvent(session, eventBase, "old_state_parking", TInstant::Seconds(17800), EObjectHistoryAction::TagEvolve);
            AddEvent(session, eventBase, "old_state_reservation", TInstant::Seconds(23400), EObjectHistoryAction::TagEvolve);
            AddEvent(session, eventBase, "old_state_reservation", TInstant::Seconds(23400), EObjectHistoryAction::DropTagPerformer);
        }
        {
            TTestCompilation compilation(offer);
            compilation.SetSince(TInstant::Zero()).SetUntil(TInstant::Zero());
            UNIT_ASSERT(session.FillCompilation(compilation));
            UNIT_ASSERT(compilation.GetPricing().GetPrices().empty());
        }
        {
            TTestCompilation compilation(offer);
            compilation.SetSince(TInstant::Seconds(0)).SetUntil(TInstant::Seconds(150000));
            UNIT_ASSERT(session.FillCompilation(compilation));
            UNIT_ASSERT_C(Abs(compilation.GetPricing().GetReportSumOriginalPrice() - 60 * 100) < 1e-5, TStringBuilder() << compilation.GetPricing().GetReportSumOriginalPrice());
        }
    }

    Y_UNIT_TEST(FreeTimeOnFreeTimetable) {
        auto offer = BuildOfferPtr(200, 100, 102400);
        {
            TDiscount discount;
            {
                TDiscount::TDiscountDetails detail;
                TTimeRestrictionsPool<TTimeRestriction> pool;
                TTimeRestriction restriction;
                restriction.SetTimeRestriction(2350, 530);
                pool.Add(restriction).SetTimezoneShift(0);
                detail.SetFreeTimetable(pool).SetTagName("old_state_reservation");
                discount.AddDetails(detail);
            }
            {
                TDiscount::TDiscountDetails detail;
                detail.SetAdditionalTime(2 * 60).SetTagName("old_state_acceptance");
                discount.AddDetails(detail);
            }
            offer->AddDiscount(discount);
        }
        {
            TDiscount discount;
            {
                TDiscount::TDiscountDetails detail;
                detail.SetAdditionalTime(20 * 60).SetTagName("old_state_reservation");
                discount.AddDetails(detail);
            }
            offer->AddDiscount(discount);
        }
        TVector<IEventsSession<TCarTagHistoryEvent>::TTimeEvent> timeline;
        TVector<TCarTagHistoryEvent> events;
        TBillingSession session;
        {
            TCarTagHistoryEventConstructor eventBase;
            eventBase.SetObjectId("object").SetTagId("1").SetHistoryUserId("user");
            {
                THolder<TChargableTag> tag(new TChargableTag("old_state_reservation"));
                tag->SetOffer(offer);
                AddEvent(session, eventBase, std::move(tag), TInstant::Seconds(1532649300), EObjectHistoryAction::SetTagPerformer);
            }
        }
        {
            session.SignalRefresh(TInstant::Seconds(1532649900));
            TInstantGuard g(TInstant::Seconds(1532649900));
            TBillingSession::TBillingCompilation compilation;
            UNIT_ASSERT(session.FillCompilation(compilation));
            UNIT_ASSERT_VALUES_EQUAL(compilation.GetFreeTime(), TDuration::Seconds(600));
        }
    }

    Y_UNIT_TEST(FreeTime) {
        auto offer = BuildOfferPtr(200, 100, 102400);
        {
            TDiscount discount;
            {
                TDiscount::TDiscountDetails detail;
                TTimeRestrictionsPool<TTimeRestriction> pool;
                TTimeRestriction restriction;
                restriction.SetTimeRestriction(2350, 530);
                pool.Add(restriction).SetTimezoneShift(0);
                detail.SetFreeTimetable(pool).SetTagName("old_state_reservation");
                discount.AddDetails(detail);
            }
            {
                TDiscount::TDiscountDetails detail;
                detail.SetAdditionalTime(2 * 60).SetTagName("old_state_acceptance");
                discount.AddDetails(detail);
            }
            offer->AddDiscount(discount);
        }
        {
            TDiscount discount;
            {
                TDiscount::TDiscountDetails detail;
                detail.SetAdditionalTime(20 * 60).SetTagName("old_state_reservation");
                discount.AddDetails(detail);
            }
            offer->AddDiscount(discount);
        }
        TVector<IEventsSession<TCarTagHistoryEvent>::TTimeEvent> timeline;
        TVector<TCarTagHistoryEvent> events;
        TOfferPricing pricing(offer);
        TBillingSession session;
        {
            TCarTagHistoryEventConstructor eventBase;
            eventBase.SetObjectId("object").SetTagId("1").SetHistoryUserId("user");
            {
                THolder<TChargableTag> tag(new TChargableTag("old_state_reservation"));
                tag->SetOffer(offer);
                AddEvent(session, eventBase, std::move(tag), TInstant::Seconds(1532648700), EObjectHistoryAction::SetTagPerformer);
            }
        }
        {
            session.SignalRefresh(TInstant::Seconds(1532649300));
            TInstantGuard g(TInstant::Seconds(1532649300));
            TBillingSession::TBillingCompilation compilation;
            UNIT_ASSERT(session.FillCompilation(compilation));
            UNIT_ASSERT_VALUES_EQUAL(compilation.GetFreeTime(), TDuration::Seconds(600));
        }
    }
}
