#include "context.h"

#include "manager.h"

#include <drive/backend/offers/ranking/calcer.h>

#include <drive/backend/billing/interfaces/account_description.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/data/area_tags.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/model.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/logging/events.h>
#include <drive/backend/saas/api.h>

#include <drive/library/cpp/geofeatures/client.h>
#include <drive/library/cpp/maps_router/router.h>
#include <drive/library/cpp/network/data/data.h>
#include <drive/library/cpp/threading/future.h>
#include <drive/library/cpp/threading/future_cast.h>
#include <drive/library/cpp/user_events_api/client.h>

#include <kernel/reqid/reqid.h>

#include <library/cpp/geobase/lookup.hpp>

#include <rtline/library/json/adapters.h>
#include <rtline/library/json/parse.h>
#include <rtline/protos/proto_helper.h>
#include <rtline/util/algorithm/ptr.h>

#include <util/string/cast.h>

namespace {
    const NUnistat::TIntervals IntervalsWalkingSignals = {0, 60, 120, 180, 240, 300, 360, 420, 480, 540, 600, 660, 720, 780, 840, 900, 960, 1020, 1080, 1140, 1200, 1260, 1320, 1380, 1440, 1500, 1560, 3600};
    class TSignalWalkingTime: public TUnistatSignal<double> {
    public:
        TSignalWalkingTime()
            : TUnistatSignal<double>({ "walking-time" }, IntervalsWalkingSignals) {
        }
    };

    static TSignalWalkingTime SignalWalkingTime;
}

NJson::TJsonValue TOffersBuildingContext::TDestination::SerializeToJson() const {
    return NJson::ToJson<TUserOfferContext::TSimpleDestination>(this->GetDestination());
}

void TOffersBuildingContext::TDestination::Prefetch() const {
    auto geobaseIdA = OfferContext->GetGeobaseId();
    auto geobaseIdB = Destination->GetGeobaseId();
    auto start = OfferContext->GetStartPosition();
    if (!Route.Initialized() && start) {
        Route = OfferContext->BuildRoute(*start, Destination->GetCoordinate(), "ptAuto");
    }
    if (!UserDoubleGeoFeatures.Initialized() && start) {
        UserDoubleGeoFeatures = OfferContext->FetchUserDoubleGeoFeatures(*start, Destination->GetCoordinate());
    }
    if (!UserDoubleGeobaseFeatures.Initialized() && geobaseIdA && geobaseIdB) {
        UserDoubleGeobaseFeatures = OfferContext->FetchUserDoubleGeoFeatures(geobaseIdA, geobaseIdB);
    }
}

const NGraph::TRouter::TOptionalRoute* TOffersBuildingContext::TDestination::GetRoute() const {
    auto g = OfferContext->BuildEventGuard("fetch_route");
    Prefetch();
    if (!Route.Initialized()) {
        return nullptr;
    }
    auto deadline = OfferContext->GetDeadline();
    if (Route.Wait(deadline) && Route.HasValue()) {
        return &Route.GetValue();
    } else {
        NDrive::TEventLog::Log("GetRouteError", NJson::TMapBuilder
            ("object_id", NJson::ToJson(OfferContext->OptionalCarId()))
            ("error", NThreading::GetExceptionInfo(Route))
        );
        return nullptr;
    }
}

const NDrive::TUserDoubleGeoFeatures* TOffersBuildingContext::TDestination::GetUserDoubleGeoFeatures() const {
    auto g = OfferContext->BuildEventGuard("GetUserDoubleGeoFeatures");
    Prefetch();
    if (!UserDoubleGeoFeatures.Initialized()) {
        return nullptr;
    }
    auto deadline = OfferContext->GetDeadline();
    if (UserDoubleGeoFeatures.Wait(deadline) && UserDoubleGeoFeatures.HasValue()) {
        return UserDoubleGeoFeatures.GetValue().Get();
    } else {
        NDrive::TEventLog::Log("GetUserDoubleGeoFeaturesError", NJson::TMapBuilder
            ("object_id", NJson::ToJson(OfferContext->OptionalCarId()))
            ("error", NThreading::GetExceptionInfo(UserDoubleGeoFeatures))
        );
        return nullptr;
    }
}

const NDrive::TUserDoubleGeoFeatures* TOffersBuildingContext::TDestination::GetUserDoubleGeobaseFeatures() const {
    auto g = OfferContext->BuildEventGuard("GetUserDoubleGeoFeatures");
    Prefetch();
    if (!UserDoubleGeobaseFeatures.Initialized()) {
        return nullptr;
    }
    auto deadline = OfferContext->GetDeadline();
    if (UserDoubleGeobaseFeatures.Wait(deadline) && UserDoubleGeobaseFeatures.HasValue()) {
        return UserDoubleGeobaseFeatures.GetValue().Get();
    } else {
        NDrive::TEventLog::Log("GetUserDoubleGeobaseFeaturesError", NJson::TMapBuilder
            ("object_id", NJson::ToJson(OfferContext->OptionalCarId()))
            ("error", NThreading::GetExceptionInfo(UserDoubleGeobaseFeatures))
        );
        return nullptr;
    }
}

TMaybe<NDrive::TTaxiSurgeCalculator::TResult> TOffersBuildingContext::GetTaxiSurge() const {
    auto g = BuildEventGuard("GetTaxiCarSurge");
    PrefetchTaxiSurge();
    if (!TaxiSurge.Initialized()) {
        return {};
    }
    if (TaxiSurge.Wait(GetDeadline()) && TaxiSurge.HasValue()) {
        return TaxiSurge.GetValue();
    }
    NDrive::TEventLog::Log(
        "GetTaxiCarSurgeError",
        NJson::TMapBuilder("exception", NThreading::GetExceptionInfo(TaxiSurge))
    );
    return {};
}

TMaybe<NDrive::TRtmrSurge> TOffersBuildingContext::GetRtmrAreaSurge() const {
    auto g = BuildEventGuard("GetRtmrAreaSurge");
    PrefetchRtmrAreaSurge();
    if (!RtmrAreaSurge.Initialized()) {
        return {};
    }
    if (RtmrAreaSurge.Wait(GetDeadline()) && RtmrAreaSurge.HasValue()) {
        return RtmrAreaSurge.GetValue();
    }
    NDrive::TEventLog::Log(
        "GetRtmrAreaSurgeError",
        NJson::TMapBuilder("exception", NThreading::GetExceptionInfo(RtmrAreaSurge))
    );
    return {};
}

TMaybe<NDrive::TRtmrExtendedSurge> TOffersBuildingContext::GetRtmrAreaExtendedSurge() const {
    auto g = BuildEventGuard("GetRtmrAreaExtendedSurge");
    PrefetchRtmrAreaExtendedSurge();
    if (!RtmrAreaExtendedSurge.Initialized()) {
        return {};
    }
    if (RtmrAreaExtendedSurge.Wait(GetDeadline()) && RtmrAreaExtendedSurge.HasValue()) {
        return RtmrAreaExtendedSurge.GetValue();
    }
    NDrive::TEventLog::Log(
        "GetRtmrAreaExtendedSurgeError",
        NJson::TMapBuilder("exception", NThreading::GetExceptionInfo(RtmrAreaExtendedSurge))
    );
    return {};
}

NThreading::TFuture<TMaybe<NDrive::TRtmrSurge>> TOffersBuildingContext::FetchRtmrAreaSurge() const {
    if (!GetSetting<bool>("offers.fetch_rtmr_area_surge").GetOrElse(false)) {
        return {};
    }
    auto client = Server->GetRtmrClient();
    if (!client) {
        return {};
    }
    if (auto areaId = GetSurgeAreaId()) {
        return client->GetSurge(NDrive::GetRtmrAreaKey(areaId));
    }
    return {};
}

NThreading::TFuture<TMaybe<NDrive::TRtmrExtendedSurge>> TOffersBuildingContext::FetchRtmrAreaExtendedSurge() const {
    if (!GetSetting<bool>("offers.fetch_rtmr_area_extended_surge").GetOrElse(false)) {
        return {};
    }
    auto client = Server->GetRtmrClient();
    if (!client) {
        return {};
    }
    if (auto areaId = GetSurgeAreaId()) {
        return client->GetExtendedSurge(NDrive::GetRtmrAreaKey(areaId));
    }
    return {};
}

bool TOffersBuildingContext::IsDelegation() const {
    if (!TaggedCar) {
        return false;
    }
    return !!TaggedCar->GetFirstTagByClass<IDelegationTag>();
}

TDuration TOffersBuildingContext::DetermWalkingTime(const TUserPermissions& permissions) const {
    const auto pedestianSpeedKmh = permissions.GetSetting<double>("offers.pedestrian_speed_km/h", 4.2);
    const auto minFreeTime = permissions.GetSetting<TDuration>("offers.min_free_time_duration", TDuration::Minutes(3));
    const auto maxFreeTime = permissions.GetSetting<TDuration>("offers.max_free_time_duration", TDuration::Minutes(20));
    const auto result = Min(maxFreeTime, Max(GetWalkingDuration(pedestianSpeedKmh).GetOrElse(maxFreeTime), minFreeTime));
    SignalWalkingTime.Signal(result.Seconds());
    return result;
}

bool TOffersBuildingContext::IsMapsRouterEnabled() const {
    return GetSetting<bool>("maps_router.enabled").GetOrElse(false);
}

bool TOffersBuildingContext::IsPedestrianRouterEnabled() const {
    return GetSetting<bool>("pedestrian_router.enabled").GetOrElse(false);
}

NThreading::TFuture<NGraph::TRouter::TOptionalRoute> TOffersBuildingContext::BuildRoute(
    const TGeoCoord& from,
    const TGeoCoord& to,
    const TString& graphPermissions
) const {
    bool isPedestrian = graphPermissions == "ptPedestrian";
    bool useStubRouter = GetSetting<bool>("stub_router.enabled").GetOrElse(false);
    if (useStubRouter || UseHaversine) {
        auto haversine = from.GetLengthTo(to);
        auto speed = isPedestrian ? 1 : 10;
        NGraph::TBaseRouter::TRoute route;
        route.Length = haversine;
        route.Time = haversine / speed;
        return NThreading::MakeFuture<NGraph::TRouter::TOptionalRoute>(std::move(route));
    }
    if (!isPedestrian && IsMapsRouterEnabled()) {
        auto router = Server->GetMapsRouter();
        Y_ENSURE(router);
        bool avoidTolls = GetSetting<bool>("maps_router.avoid_tolls").GetOrElse(false);
        return router->GetSummary(from, to, avoidTolls, GetReqId());
    }
    if (isPedestrian && IsPedestrianRouterEnabled()) {
        auto router = Server->GetPedestrianRouter();
        Y_ENSURE(router);
        return router->GetSummary(from, to, false, GetReqId());
    }
    NGraph::TRouter::TRoutingOptions options;
    options.Permissions = graphPermissions;
    options.AskRoute = false;
    options.ExtraParams = "component=Graph";

    auto apiName = GetSetting<TString>("router_api_offer_builder-" + graphPermissions);
    if (!apiName) {
        apiName = GetSetting<TString>("router_api_offer_builder");
    }
    if (!apiName) {
        apiName = "drive_router";
    }
    auto api = Server->GetRTLineAPI(*apiName);
    if (!api) {
        return NThreading::TExceptionFuture() << "missing router";
    }
    NGraph::TRouter::TOptions routerOptions;
    routerOptions.Timeout = api->GetSearchClient().GetSendConfig().GetGlobalTimeout();
    auto router = MakeHolder<NGraph::TRouter>(api->GetSearchClient(), routerOptions);
    return router->GetRouteAsync(from, to, Nothing(), options);
}

TOptionalGeobaseId TOffersBuildingContext::FetchGeobaseId() const {
    auto geobase = Server->GetGeobase();
    auto position = GetStartPosition();
    if (geobase && position) {
        return geobase->GetRegionIdByLocation(position->Y, position->X);
    } else {
        return {};
    }
}

const TString GEO_FEATURES_CLIENT_NAME = "geo_features_client_name";

TOffersBuildingContext::TGeoFeaturesFuture TOffersBuildingContext::FetchGeoFeatures() const {
    if (GetSetting<bool>("surge.fallback.enabled").GetOrElse(false)) {
        auto features = MakeMaybe<NDrive::TGeoFeatures>();
        features->RtSurgePrediction1 = GetSetting<double>("surge.fallback.value").GetOrElse(1.0);
        return NThreading::MakeFuture(std::move(features));
    }
    auto geoFeaturesClientName = GetSetting<TString>(GEO_FEATURES_CLIENT_NAME).GetOrElse("");
    auto geoFeaturesClient = Server->GetGeoFeaturesClient(geoFeaturesClientName);
    auto position = GetStartPosition();
    if (geoFeaturesClient && position && GetNeedGeoFeatures()) {
        return geoFeaturesClient->Get(position->Y, position->X);
    } else {
        return {};
    }
}

TOffersBuildingContext::TGeoFeaturesFuture TOffersBuildingContext::FetchGeobaseFeatures() const {
    auto geoFeaturesClientName = GetSetting<TString>(GEO_FEATURES_CLIENT_NAME).GetOrElse("");
    auto geoFeaturesClient = Server->GetGeoFeaturesClient(geoFeaturesClientName);
    auto geobaseId = GetGeobaseId();
    if (geoFeaturesClient && geobaseId && GetNeedGeoFeatures()) {
        return geoFeaturesClient->Get(geobaseId);
    } else {
        return {};
    }
}

TOffersBuildingContext::TUserGeoFeaturesFuture TOffersBuildingContext::FetchUserGeoFeatures() const {
    auto userEventsApi = GetFeaturesClient();
    auto userHistoryContext = UserHistoryContext.Get();
    auto position = GetStartPosition();
    if (userEventsApi && userHistoryContext && position && GetNeedGeoFeatures()) {
        return userEventsApi->GetUserGeoFeatures(userHistoryContext->GetUserId(), *position);
    } else {
        return {};
    }
}

TOffersBuildingContext::TUserGeoFeaturesFuture TOffersBuildingContext::FetchUserGeobaseFeatures() const {
    auto userEventsApi = GetFeaturesClient();
    auto userHistoryContext = UserHistoryContext.Get();
    auto geobaseId = GetGeobaseId();
    if (userEventsApi && userHistoryContext && geobaseId && GetNeedGeoFeatures()) {
        return userEventsApi->GetUserGeoFeatures(userHistoryContext->GetUserId(), geobaseId);
    } else {
        return {};
    }
}

TOffersBuildingContext::TUserDoubleGeoFeaturesFuture TOffersBuildingContext::FetchUserDoubleGeoFeatures(const TGeoCoord& from, const TGeoCoord& to) const {
    auto userEventsApi = GetFeaturesClient();
    auto userHistoryContext = UserHistoryContext.Get();
    if (userEventsApi && userHistoryContext && GetNeedGeoFeatures()) {
        return userEventsApi->GetUserDoubleGeoFeatures(userHistoryContext->GetUserId(), from, to);
    } else {
        return {};
    }
}

TOffersBuildingContext::TUserDoubleGeoFeaturesFuture TOffersBuildingContext::FetchUserDoubleGeoFeatures(TGeobaseId from, TGeobaseId to) const {
    auto userEventsApi = GetFeaturesClient();
    auto userHistoryContext = UserHistoryContext.Get();
    if (userEventsApi && userHistoryContext && GetNeedGeoFeatures()) {
        return userEventsApi->GetUserDoubleGeoFeatures(userHistoryContext->GetUserId(), from, to);
    } else {
        return {};
    }
}

TOffersBuildingContext::TUserDestinationSuggestFuture TOffersBuildingContext::FetchUserDestinationSuggest() const {
    auto avoidTolls = GetSetting<bool>("maps_router.avoid_tolls").GetOrElse(false);
    auto reqid = GetReqId();
    auto router = Server->GetMapsRouter();
    auto start = GetStartPosition();
    auto userEventsApi = GetFeaturesClient();
    auto userHistoryContext = UserHistoryContext.Get();
    if (userHistoryContext) {
        auto suggest = userHistoryContext->GetUserDestinationSuggestFuture();
        if (!suggest.Initialized()) {
            return {};
        }
        auto geobaseIdA = GetGeobaseId();
        auto needGeoFeatures = NeedGeoFeatures;
        auto needRouteFeatures = NeedRouteFeatures;
        auto userId = userHistoryContext->GetUserId();
        auto modifier = [
            reqid = std::move(reqid),
            router = std::move(router),
            start = std::move(start),
            userId = std::move(userId),
            avoidTolls,
            geobaseIdA,
            needGeoFeatures,
            needRouteFeatures,
            userEventsApi
        ](const TUserOfferContext::TUserDestinationSuggestFuture& s) {
            NGraph::TRouter::TRoutingOptions routingOptions;
            routingOptions.AskRoute = false;

            TUserDestinationSuggestEx result;
            for (auto&& i : s.GetValue().Elements) {
                TUserDestinationSuggestEx::TElement element = i;
                TGeoCoord finish = { element.Longitude, element.Latitude };
                if (router && start && needRouteFeatures) {
                    element.Route = router->GetSummary(*start, finish, avoidTolls, reqid + '-' + ToString(element.Latitude));
                }
                if (userEventsApi && start && needGeoFeatures) {
                    element.UserDoubleGeoFeatures = userEventsApi->GetUserDoubleGeoFeatures(userId, *start, finish);
                }
                auto geobaseIdB = element.GeobaseId;
                if (userEventsApi && geobaseIdA && geobaseIdB) {
                    element.UserDoubleGeobaseFeatures = userEventsApi->GetUserDoubleGeoFeatures(userId, geobaseIdA, *geobaseIdB);
                }
                result.Elements.push_back(std::move(element));
            }
            return result;
        };
        return suggest.Apply(std::move(modifier));
    } else {
        return {};
    }
}

NThreading::TFuture<NDrive::TTaxiSurgeCalculator::TResult> TOffersBuildingContext::FetchTaxiSurge() const {
    auto location = GetStartPosition();
    if (!location || !NeedTaxiSurge) {
        return {};
    }
    auto client = Server->GetTaxiSurgeCalculator();
    if (!client) {
        return {};
    }
    return client->GetMapSurge(*location);
}

TOffersBuildingContext::TOffersBuildingContext(const NDrive::IServer* server)
    : Server(server)
    , ReqId(ReqIdGenerate("FAKE"))
{
    if (Server) {
        const auto& settings = Server->GetSettings();
        NeedRouteFeatures = settings.GetValue<bool>("offer.need_route_features").GetOrElse(NeedRouteFeatures);
    }
}

TOffersBuildingContext::TOffersBuildingContext(TUserOfferContext&& uoc)
    : TOffersBuildingContext(uoc.GetServer())
{
    UserHistoryContext = std::move(uoc);
    NeedTaxiSurge = GetSetting<bool>("offer.need_taxi_surge").GetOrElse(NeedTaxiSurge);
}

bool TOffersBuildingContext::Parse(IReplyContext::TPtr context, const TUserPermissions& permissions, const NJson::TJsonValue& requestData, NDrive::TInfoEntitySession& session) {
    const TCgiParameters& cgi = context->GetCgiParameters();
    {
        const TString& borderStr = cgi.Get("border");
        TDuration border;
        if (!!borderStr && TryFromString<TDuration>(borderStr, border)) {
            Border = border;
        }
    }
    {
        TString carId;
        if (cgi.Has("car_id")) {
            carId = cgi.Get("car_id");
        } else if (cgi.Has("car_number")) {
            TString carNumber = cgi.Get("car_number");
            auto unescapeNumber = CGIUnescapeRet(carNumber);
            while (unescapeNumber != carNumber) {
                carNumber = unescapeNumber;
                unescapeNumber = CGIUnescapeRet(carNumber);
            }
            if (!!carNumber) {
                auto gCar = Server->GetDriveAPI()->GetCarNumbers()->GetCachedOrFetch(NContainer::Scalar(carNumber));
                if (gCar.GetResultPtr(carNumber)) {
                    carId = gCar.GetResultPtr(carNumber)->GetId();
                }
            }
        }
        if (!GetUuid(carId).IsEmpty()) {
            InitializeCar(carId);
        } else if (carId) {
            session.SetErrorInfo("car_id", "not_uuid", EDriveSessionResult::IncorrectRequest);
            return false;
        }
    }
    if (cgi.Has("session_id")) {
        BillingSessionId = cgi.Get("session_id");
    }
    if (context) {
        ReqId = ToString(NUtil::GetReqId(context->GetRequestData(), cgi));
        NeedOfferedHintsSeparatelyFlag = IsTrue(cgi.Get("with_hints"));
    }
    if (cgi.Has("delegation_payload")) {
        auto delegationPayload = cgi.Get("delegation_payload");
        bool isQRDelegation = delegationPayload.StartsWith("QR");
        TString tagId = isQRDelegation ? delegationPayload.substr(2) : delegationPayload;
        if (GetUuid(tagId).IsEmpty()) {
            session.SetErrorInfo("delegation_tag", "not_uuid", EDriveSessionResult::IncorrectRequest);
            return false;
        }
        bool delegationParsed = isQRDelegation ? ParseQRCodeDelegationPayload(tagId, permissions, session) : ParsePersonalizedDelegationPayload(tagId, permissions, session);
        if (!delegationParsed) {
            return false;
        }
    }
    if (cgi.Has("replacing_car") && IsTrue(cgi.Get("replacing_car"))) {
        ReplacingCar = true;
    }
    auto offersStorage = Server ? Server->GetOffersStorage() : nullptr;
    auto previousOfferId = NJson::FromJson<TMaybe<TString>>(requestData["previous_offer_id"]);
    if (previousOfferId && offersStorage) {
        PreviousOfferId = *previousOfferId;
        PreviousOffer = offersStorage->RestoreOffer(PreviousOfferId, permissions.GetUserId(), session);
        PreviousOffers[PreviousOfferId] = PreviousOffer;
    }
    auto previousOfferIds = NJson::FromJson<TMaybe<TVector<TString>>>(requestData["previous_offer_ids"]).GetOrElse({});
    for (auto&& previousOfferId : previousOfferIds) {
        PreviousOffers[previousOfferId] = offersStorage->RestoreOffer(previousOfferId, permissions.GetUserId(), session);
    }
    if (!NJson::ParseField(requestData["marker"], Marker)) {
        return false;
    }
    if (!NJson::ParseField(requestData["variables"], NJson::Dictionary(Variables))) {
        return false;
    }
    return true;
}

TOffersBuildingContext::~TOffersBuildingContext() {
}

bool TOffersBuildingContext::ParseQRCodeDelegationPayload(const TString& payload, const TUserPermissions& permissions, NDrive::TInfoEntitySession& session) {
    const auto& deviceTagManager = Server->GetDriveAPI()->GetTagsManager().GetDeviceTags();
    auto tx = deviceTagManager.BuildSession(true);
    auto optionalTag = deviceTagManager.RestoreTag(payload, tx);
    if (!optionalTag) {
        session.SetErrorInfo("ParseQRCodeDelegationPayload", tx.GetMessages());
        return false;
    }
    DelegationCarTag = std::move(*optionalTag);
    if (!DelegationCarTag) {
        session.SetErrorInfo("ParseQRCodeDelegationPayload", "bad tag_id " + payload, NDrive::MakeError("bad_or_expired_qr_code"));
        return false;
    }
    auto delegationCarTagImpl = DelegationCarTag->GetTagAs<TP2PDelegationTag>();
    if (!delegationCarTagImpl || delegationCarTagImpl->GetP2PUserId()) {
        session.SetErrorInfo("delegation", "not_p2p_tag", EDriveSessionResult::IncorrectRequest);
        return false;
    }
    CarId = DelegationCarTag->GetObjectId();
    auto delegationAbility = Server->GetDriveAPI()->GetP2PAbility(permissions, DelegationCarTag->GetObjectId(), tx, delegationCarTagImpl->IsPreserveOffer());
    if (delegationAbility != TDriveAPI::EDelegationAbility::Possible) {
        auto verdict = ToString(delegationAbility);
        session.SetErrorInfo("delegation_impossible", verdict, EDriveSessionResult::IncorrectRequest);
        session.SetLocalizedTitleKey("delegation.p2p.errors.delegatee.title." + verdict);
        session.SetLocalizedMessageKey("delegation.p2p.errors.delegatee.message." + verdict);
        return false;
    }
    return true;
}

bool TOffersBuildingContext::ParsePersonalizedDelegationPayload(const TString& payload, const TUserPermissions& permissions, NDrive::TInfoEntitySession& session) {
    const auto& userTagManager = Server->GetDriveAPI()->GetTagsManager().GetUserTags();
    auto tx = userTagManager.BuildSession(true);
    auto optionalTag = userTagManager.RestoreTag(payload, tx);
    if (!optionalTag) {
        session.SetErrorInfo("ParsePersonalizedDelegationPayload", tx.GetMessages());
        return false;
    }
    IncomingDelegationTag = std::move(*optionalTag);
    auto tagImpl = IncomingDelegationTag->GetTagAs<TIncomingDelegationUserTag>();
    if (!tagImpl || IncomingDelegationTag->GetObjectId() != permissions.GetUserId()) {
        session.SetErrorInfo("delegation", "tag_type_mismatch", EDriveSessionResult::IncorrectRequest);
        return false;
    }
    auto objectId = tagImpl->GetObjectId();
    CarId = objectId;
    TVector<TTaggedDevice> taggedDevices;
    if (!Server->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetObjectsFromCache({objectId}, {TP2PDelegationTag::TypeName}, taggedDevices, TInstant::Zero())) {
        session.SetErrorInfo("delegation", "restore_tags_failed", EDriveSessionResult::InternalError);
        return false;
    }
    if (taggedDevices.size() != 1 || taggedDevices.front().GetTagsByClass<TP2PDelegationTag>().size() != 1) {
        session.SetErrorInfo("delegation", "p2p_tags_count_mismatch", EDriveSessionResult::InternalError);
        return false;
    }
    DelegationCarTag = taggedDevices.front().GetFirstTagByClass<TP2PDelegationTag>();
    auto delegationCarTagImpl = DelegationCarTag->GetTagAs<TP2PDelegationTag>();
    if (!delegationCarTagImpl) {
        session.SetErrorInfo("delegation", "p2p_tag_type_inconsistent", EDriveSessionResult::InternalError);
        return false;
    }
    return true;
}

TInstant TOffersBuildingContext::GetDeadline() const {
    return UserHistoryContext ? UserHistoryContext->GetDeadline() : TInstant::Max();
}

const TString& TOffersBuildingContext::GetExternalUserId() const {
    if (UserHistoryContext) {
        return UserHistoryContext->GetExternalUserId();
    } else {
        return Default<TString>();
    }
}

const TString& TOffersBuildingContext::GetInsuranceTypeRef() const {
    return Yensured(UserHistoryContext)->GetInsuranceTypeRef();
}

TMaybe<bool> TOffersBuildingContext::GetIsPlusUser() const {
    if (UserHistoryContext && UserHistoryContext->GetUserPermissions()) {
        return UserHistoryContext->GetUserPermissions()->GetUserFeatures().GetIsPlusUser();
    } else {
        return {};
    }
}

const TString& TOffersBuildingContext::GetOrigin() const {
    if (UserHistoryContext) {
        return UserHistoryContext->GetOrigin();
    } else {
        return Default<TString>();
    }
}

TGeobaseId TOffersBuildingContext::GetGeobaseId() const {
    PrefetchGeobaseId();
    return GeobaseId.GetOrElse(0);
}

TMaybe<TSet<TString>> TOffersBuildingContext::GetComplementaryInsuranceTypes() const {
    if (UserHistoryContext && !UserHistoryContext->GetComplementaryInsuranceTypes().empty()) {
        return MakeSet(UserHistoryContext->GetComplementaryInsuranceTypes());
    }
    return {};
}

TMaybe<TSet<TString>> TOffersBuildingContext::GetInsuranceTypes() const {
    TMaybe<TSet<TString>> result;
    if (ModelTag && Server) {
        auto modelInsuranceTypes = TModelTag::GetInsuranceTypes(*ModelTag, *Server);
        result = std::move(modelInsuranceTypes);
    }
    if (UserHistoryContext) {
        auto accountInsuranceTypes = UserHistoryContext->GetAvailableInsuranceTypes();
        result = MakeIntersection(std::move(result), std::move(accountInsuranceTypes));
    }
    return result;
}

TUserActions TOffersBuildingContext::GetPriceConstructorsForBehaviour(const TUserPermissions& permissions) const {
    TUserActions result;
    if (PricesForObject) {
        result = *PricesForObject;
    } else {
        for (auto&& i : permissions.GetActionsActual()) {
            const TPriceOfferConstructor* poc = i.GetAs<TPriceOfferConstructor>();
            if (!poc) {
                continue;
            }
            TMaybe<TTaggedObject> td = GetTaggedCar();
            if (!!td) {
                if (!poc->GetObjectTagsFilter().IsEmpty() && !poc->GetObjectTagsFilter().IsMatching(*td)) {
                    continue;
                }
            }
            if (UseLocationTags) {
                if (!poc->GetObjectLocationTagsFilter().IsEmpty() && !poc->GetObjectLocationTagsFilter().IsMatching(GetLocationTags())) {
                    continue;
                }
            }
            result.emplace_back(i.Impl());
        }
        PricesForObject = result;
    }

    return result;
}

TMaybe<TGeoCoord> TOffersBuildingContext::GetStartPosition() const {
    if (HasCarFuturesStart()) {
        return GetCarFuturesStartUnsafe();
    }
    if (!HasCarId()) {
        return {};
    }
    NDrive::TLocation location;
    auto snapshot = Server->GetSnapshotsManager().GetSnapshot(GetCarIdUnsafe());
    if (!snapshot.GetLocation(location)) {
        return {};
    }
    return location.GetCoord();
}

TUserPermissions::TConstPtr TOffersBuildingContext::GetUserPermissions() const {
    return UserHistoryContext ? UserHistoryContext->GetUserPermissions() : nullptr;
}

bool TOffersBuildingContext::HasInsuranceType() const {
    return UserHistoryContext ? UserHistoryContext->HasInsuranceType() : false;
}

void TOffersBuildingContext::SetInsuranceType(const TString& value) {
    Yensured(UserHistoryContext)->SetInsuranceType(value);
}

void TOffersBuildingContext::InitializeCar(const TString& carId) {
    CarId = carId;
    auto expectedTag = TModelTag::Get(carId, *Server);
    if (expectedTag) {
        ModelTag = expectedTag.ExtractValue();
    }
}

void TOffersBuildingContext::Prefetch() {
    if (HasCarId()) {
        NDrive::CalcDeviceFeatures(Features, *Server, *CarId);
        NDrive::CalcRequestFeatures(Features, ModelingNow());
    }
    if (UserHistoryContext) {
        if (UserHistoryContext->HasUserPosition()) {
            NDrive::CalcUserFeatures(Features, UserHistoryContext->GetUserPositionUnsafe());
        }
        UserHistoryContext->Prefetch();
        if (Destinations.empty()) {
            if (UserHistoryContext->HasUserDestination()) {
                Destinations.emplace_back(UserHistoryContext->GetUserDestinationUnsafe(), *this);
            } else {
                for (auto&& i : UserHistoryContext->GetHintedDestinations()) {
                    Destinations.emplace_back(i, *this);
                }
            }
            for (auto&& i : Destinations) {
                i.Prefetch();
            }
        }
    }
    PrefetchGeobaseId();
    PrefetchGeoFeatures();
    PrefetchUserGeoFeatures();
    PrefetchUserDestinationSuggest();
    PrefetchWalking();
    PrefetchGeobaseFeatures();
    PrefetchUserGeobaseFeatures();
    PrefetchTaxiSurge();
    PrefetchRtmrAreaSurge();
    PrefetchRtmrAreaExtendedSurge();
}

void TOffersBuildingContext::PrefetchTaxiSurge() const {
    if (!TaxiSurge.Initialized()) {
        TaxiSurge = FetchTaxiSurge();
    }
}

TMaybe<EOfferCorrectorResult> TOffersBuildingContext::GetGeoConditionsCheckResult(TStringBuf cacheId) const {
    if (UserHistoryContext) {
        return UserHistoryContext->GetGeoConditionsCheckResult(cacheId);
    } else {
        return {};
    }
}

void TOffersBuildingContext::ClearGeoConditionsCheckResults() const {
    if (UserHistoryContext) {
        UserHistoryContext->ClearGeoConditionsCheckResults();
    }
}

void TOffersBuildingContext::SetGeoConditionsCheckResult(const TString& cacheId, EOfferCorrectorResult value) const {
    if (UserHistoryContext) {
        UserHistoryContext->SetGeoConditionsCheckResult(cacheId, value);
    }
}

void TOffersBuildingContext::ClearPricesForObject() {
    PricesForObject.Clear();
}

const NDrive::TLocationTags& TOffersBuildingContext::GetLocationTags() const {
    auto g = BuildEventGuard("get_location_tags");
    if (!HasLocationTags() && HasCarId()) {
        auto snapshot = Server->GetSnapshotsManager().GetSnapshot(GetCarIdUnsafe());
        SetLocationTags(snapshot.GetTagsInPoint());
        SetLocationAreaIds(snapshot.GetAreaIds());
    }
    if (HasLocationTags()) {
        return GetLocationTagsUnsafe();
    } else {
        return Default<NDrive::TLocationTags>();
    }
}

const NDrive::TLocationAreaIds& TOffersBuildingContext::GetLocationAreaIds() const {
    auto g = BuildEventGuard("get_location_area_ids");
    if (!HasLocationAreaIds() && HasCarId()) {
        auto snapshot = Server->GetSnapshotsManager().GetSnapshot(GetCarIdUnsafe());
        SetLocationTags(snapshot.GetTagsInPoint());
        SetLocationAreaIds(snapshot.GetAreaIds());
    }
    if (HasLocationAreaIds()) {
        return GetLocationAreaIdsUnsafe();
    } else {
        return Default<NDrive::TLocationAreaIds>();
    }
}

TString TOffersBuildingContext::GetSurgeAreaId() const {
    auto g = BuildEventGuard("GetSurgeAreaId");
    const auto modelTag = GetSetting<TString>("offers.rtmr_area_surge_model").GetOrElse("rtmr_surge");
    if (!HasSurgeAreaId()) {
        const auto action = [this, modelTag](const TFullAreaInfo& areaInfo) {
            if (areaInfo.GetArea().GetTags().contains("rtmr_surge") && areaInfo.GetArea().GetTags().contains(modelTag)) {
                SetSurgeAreaId(areaInfo.GetId());
                return false;
            }
            return true;
        };
        if (Server->GetDriveAPI()->GetAreasDB()->ProcessAreaIds(GetLocationAreaIds(), action)) {
            SetSurgeAreaId("");
        }
    }
    return OptionalSurgeAreaId().GetOrElse("");
}

NDrive::TLocationTags TOffersBuildingContext::GetUserLocationTags() const {
    if (HasUserHistoryContext() && GetUserHistoryContextUnsafe().HasUserPosition()) {
        return Server->GetDriveAPI()->GetTagsInPoint(GetUserHistoryContextUnsafe().GetUserPositionUnsafe());
    } else {
        return Server->GetDriveAPI()->GetTagsInPoint({37., 55});
    }
}

TMaybe<TTaggedObject> TOffersBuildingContext::GetTaggedCar() const {
    if (!!TaggedCar) {
        return TaggedCar;
    } else if (!!CarId) {
        TaggedCar = Server->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetObject(*CarId);
        return TaggedCar;
    } else {
        return {};
    }
}

TAtomicSharedPtr<TBillingSession> TOffersBuildingContext::GetBillingSession() const {
    if (BillingSession) {
        return BillingSession;
    } else if (BillingSessionId) {
        auto sessionBuilder = Server->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing");
        if (!sessionBuilder)
            return {};
        auto session = sessionBuilder->GetSession(BillingSessionId);
        if (!session)
            return {};
        auto billingSession = std::dynamic_pointer_cast<TBillingSession>(session);
        if (!billingSession)
            return {};
        BillingSession = billingSession;
        return BillingSession;
    } else {
        return {};
    }
}

void TOffersBuildingContext::AddError(const TString& name, const TString& errorId) const {
    if (!errorId) {
        return;
    }
    OfferConstructionProblems[name] = errorId;
    if (!OfferConstructionProblemFirst) {
        OfferConstructionProblemFirst = errorId;
    }
}

const NDrive::TGeoFeatures* TOffersBuildingContext::GetGeoFeatures() const {
    auto g = BuildEventGuard("GetGeoFeatures");
    PrefetchGeoFeatures();
    if (!GeoFeatures.Initialized()) {
        return nullptr;
    }
    auto deadline = GetDeadline();
    if (GeoFeatures.Wait(deadline) && GeoFeatures.HasValue()) {
        return GeoFeatures.GetValue().Get();
    } else {
        NDrive::TEventLog::Log("GetGeoFeaturesError", NJson::TMapBuilder
            ("object_id", NJson::ToJson(OptionalCarId()))
            ("error", NThreading::GetExceptionInfo(GeoFeatures))
        );
        return nullptr;
    }
}

const NDrive::TGeoFeatures* TOffersBuildingContext::GetGeobaseFeatures() const {
    auto g = BuildEventGuard("GetGeobaseFeatures");
    PrefetchGeobaseFeatures();
    if (!GeobaseFeatures.Initialized()) {
        return nullptr;
    }
    auto deadline = GetDeadline();
    if (GeobaseFeatures.Wait(deadline) && GeobaseFeatures.HasValue()) {
        return GeobaseFeatures.GetValue().Get();
    } else {
        NDrive::TEventLog::Log("GetGeobaseFeaturesError", NJson::TMapBuilder
            ("object_id", NJson::ToJson(OptionalCarId()))
            ("error", NThreading::GetExceptionInfo(GeobaseFeatures))
        );
        return nullptr;
    }
}

const NDrive::TUserFeatures* TOffersBuildingContext::GetUserFeatures() const {
    return UserHistoryContext ? UserHistoryContext->GetUserFeatures() : nullptr;
}

const NDrive::TUserGeoFeatures* TOffersBuildingContext::GetUserGeoFeatures() const {
    auto g = BuildEventGuard("GetUserGeoFeatures");
    PrefetchUserGeoFeatures();
    if (!UserGeoFeatures.Initialized()) {
        return nullptr;
    }
    auto deadline = GetDeadline();
    if (UserGeoFeatures.Wait(deadline) && UserGeoFeatures.HasValue()) {
        return UserGeoFeatures.GetValue().Get();
    } else {
        NDrive::TEventLog::Log("GetUserGeoFeaturesError", NJson::TMapBuilder
            ("object_id", NJson::ToJson(OptionalCarId()))
            ("error", NThreading::GetExceptionInfo(UserGeoFeatures))
        );
        return nullptr;
    }
}

const NDrive::TUserGeoFeatures* TOffersBuildingContext::GetUserGeobaseFeatures() const {
    auto g = BuildEventGuard("GetUserGeobaseFeatures");
    PrefetchUserGeobaseFeatures();
    if (!UserGeobaseFeatures.Initialized()) {
        return nullptr;
    }
    auto deadline = GetDeadline();
    if (UserGeobaseFeatures.Wait(deadline) && UserGeobaseFeatures.HasValue()) {
        return UserGeobaseFeatures.GetValue().Get();
    } else {
        NDrive::TEventLog::Log("GetUserGeobaseFeaturesError", NJson::TMapBuilder
            ("object_id", NJson::ToJson(OptionalCarId()))
            ("error", NThreading::GetExceptionInfo(UserGeobaseFeatures))
        );
        return nullptr;
    }
}

const TOffersBuildingContext::TUserDestinationSuggestEx* TOffersBuildingContext::GetUserDestinationSuggest() const {
    return MutableUserDestinationSuggest();
}

const NDrive::TUserEventsApi* TOffersBuildingContext::GetFeaturesClient() const {
    if (!GetSetting<bool>("use_features_client").GetOrElse(false)) {
        return Server->GetUserEventsApi();
    }
    return Server->GetFeaturesClient();
}

TOffersBuildingContext::TUserDestinationSuggestEx* TOffersBuildingContext::MutableUserDestinationSuggest() const {
    auto g = BuildEventGuard("GetUserDestinationSuggest");
    PrefetchUserDestinationSuggest();
    if (!UserDestinationSuggest.Initialized()) {
        return nullptr;
    }
    auto deadline = GetDeadline();
    if (UserDestinationSuggest.Wait(deadline) && UserDestinationSuggest.HasValue()) {
        return &const_cast<TUserDestinationSuggestEx&>(UserDestinationSuggest.GetValue());
    } else {
        NDrive::TEventLog::Log("GetUserDestinationSuggest", NJson::TMapBuilder
            ("object_id", NJson::ToJson(OptionalCarId()))
            ("error", NThreading::GetExceptionInfo(UserDestinationSuggest))
        );
        return nullptr;
    }
}

IOffer::TPtr TOffersBuildingContext::GetPreviousOffer(TStringBuf behaviourConstructorId, bool waitStage) const {
    auto deadline = GetDeadline();
    for (auto&& [id, previousOffer] : PreviousOffers) {
        if (!previousOffer.Initialized()) {
            continue;
        }
        if (waitStage && !previousOffer.HasValue() && !previousOffer.HasException()) {
            auto g = BuildEventGuard("wait_previous_offer:" + id);
            bool timeouted = !previousOffer.Wait(deadline);
            Y_UNUSED(timeouted);
        }
        if (previousOffer.HasValue()) {
            auto offer = std::dynamic_pointer_cast<IOffer>(previousOffer.GetValue());
            if (offer && offer->GetBehaviourConstructorId() == behaviourConstructorId && offer->GetObjectId() == CarId) {
                return offer;
            }
        } else if (waitStage) {
            NDrive::TEventLog::Log("GetPreviousOfferError", NJson::TMapBuilder
                ("offer_id", id)
                ("error", NThreading::GetExceptionInfo(previousOffer))
            );
        }
    }
    if (!waitStage) {
        return GetPreviousOffer(behaviourConstructorId, /*waitStage=*/true);
    }
    return nullptr;
}

TDuration TOffersBuildingContext::CorrectWalkingDuration(const double originalTime, const double pedestrianSpeedKmh) {
    const double durationSeconds = originalTime * 5 / pedestrianSpeedKmh;
    const TDuration resultExact = TDuration::Seconds(durationSeconds);
    return TDuration::Minutes(resultExact.Minutes() + ((resultExact.Seconds() % 60) ? 1 : 0));
}

TMaybe<TDuration> TOffersBuildingContext::GetWalkingDuration(const double pedestrianSpeedKmh) const {
    if (OverridenWalkingDuration) {
        return *OverridenWalkingDuration;
    }
    if (WalkingDistance) {
        return TDuration::Seconds(*WalkingDistance / (pedestrianSpeedKmh / 3.6));
    }
    auto g = BuildEventGuard("GetWalkingDuration");
    PrefetchWalking();
    if (!RouteWalking.Initialized()) {
        return {};
    }
    auto deadline = GetDeadline();
    if (RouteWalking.Wait(deadline) && RouteWalking.HasValue() && std::abs(pedestrianSpeedKmh) > 0.001) {
        const auto& route = RouteWalking.GetValue();
        if (!route) {
            return {};
        }
        return CorrectWalkingDuration(route->Time, pedestrianSpeedKmh);
    } else {
        NDrive::TEventLog::Log("GetWalkingDurationError", NJson::TMapBuilder
            ("object_id", NJson::ToJson(OptionalCarId()))
            ("pedestrian_speed", pedestrianSpeedKmh)
            ("error", NThreading::GetExceptionInfo(RouteWalking))
        );
        return {};
    }
}

TMaybe<TAdditionalPricedFeatureAction> TOffersBuildingContext::GetInsuranceDescription() const {
    if (UserHistoryContext && !InsuranceDescription) {
        auto permissions = UserHistoryContext->GetUserPermissions();
        if (!permissions) {
            return {};
        }
        auto selectedCharge = UserHistoryContext->GetSelectedCharge();
        NDrive::NBilling::IBillingAccount::TConstPtr selectedAccount;
        for (auto&& account : UserHistoryContext->GetUserAccounts()) {
            if (account && account->GetUniqueName() == selectedCharge) {
                selectedAccount = account;
                break;
            }
        }
        if (!selectedAccount) {
            return {};
        }
        TTagsFilter tagsFilter;
        if (!tagsFilter.DeserializeFromString(selectedAccount->GetInsuranceDescriptionFilter())) {
            return {};
        }
        for (auto&& action : permissions->GetActionsActual()) {
            auto pricedAction = action.GetAs<TAdditionalPricedFeatureAction>();
            if (!pricedAction) {
                continue;
            }
            if (tagsFilter.IsMatching(pricedAction->GetGrouppingTags())) {
                InsuranceDescription = *pricedAction;
                break;
            }
        }
    }
    return InsuranceDescription;
}

template <>
NJson::TJsonValue NJson::ToJson(const TOffersBuildingContext::TUserDestinationSuggestEx::TElement& object) {
    NJson::TJsonValue result = NJson::ToJson<TUserOfferContext::TUserDestinationSuggestEx::TElement>(object);
    result["route"] = NJson::ToJson(object.Route);
    result["user_double_geo_features"] = NJson::ToJson(object.UserDoubleGeoFeatures);
    result["user_double_geobase_features"] = NJson::ToJson(object.UserDoubleGeobaseFeatures);
    return result;
}
