#pragma once

#include "abstract.h"
#include "discount.h"
#include "notifications.h"
#include "user_context.h"

#include <drive/backend/offers/actions/abstract.h>
#include <drive/backend/offers/ranking/features.h>

#include <drive/backend/abstract/base.h>
#include <drive/backend/actions/abstract/action.h>
#include <drive/backend/areas/location.h>
#include <drive/backend/billing/interfaces/account.h>
#include <drive/backend/billing/interfaces/entities.h>
#include <drive/backend/data/area_tags.h>
#include <drive/backend/data/chargable.h>

#include <drive/library/cpp/geofeatures/features.h>
#include <drive/library/cpp/rtmr/client.h>
#include <drive/library/cpp/taxi/request.h>
#include <drive/library/cpp/user_events_api/user_features.h>
#include <drive/telematics/server/location/location.h>

#include <library/cpp/charset/ci_string.h>

#include <rtline/api/graph/router/router.h>
#include <rtline/library/geometry/coord.h>
#include <rtline/util/types/accessor.h>

#include <util/generic/map.h>

class TOffersBuildingContext {
private:
    const NDrive::IServer* Server = nullptr;

    mutable NThreading::TFuture<NGraph::TRouter::TOptionalRoute> RouteWalking;
    bool NeedOfferedHintsSeparatelyFlag = false;
    bool NeedOfferedHintsOnlyFlag = false;
    mutable TDestinationDetectorTag::TContext FlowControlContext;
    R_OPTIONAL(TGeoCoord, OriginalRidingStart, {}, mutable);
    R_OPTIONAL(TUserOfferContext, UserHistoryContext, {}, mutable);
    R_OPTIONAL(TConstDBTag, DelegationCarTag);
    R_OPTIONAL(TConstDBTag, IncomingDelegationTag);
    R_OPTIONAL(TConstDBTag, ModelTag);

private:
    bool ParseQRCodeDelegationPayload(const TString& payload, const TUserPermissions& permissions, NDrive::TInfoEntitySession& session);
    bool ParsePersonalizedDelegationPayload(const TString& payload, const TUserPermissions& permissions, NDrive::TInfoEntitySession& session);

    bool IsMapsRouterEnabled() const;
    bool IsPedestrianRouterEnabled() const;

public:
    bool Parse(IReplyContext::TPtr context, const TUserPermissions& permissions, const NJson::TJsonValue& requestData, NDrive::TInfoEntitySession& session);

    bool NeedOfferedHintsOnly() const {
        return NeedOfferedHintsOnlyFlag;
    }

public:
    using TDestinationDescription = TUserOfferContext::TDestinationDescription;
    class TDestination {
    private:
        const TDestinationDescription* Destination;
        const TOffersBuildingContext* OfferContext;

        mutable NThreading::TFuture<NGraph::TRouter::TOptionalRoute> Route;
        mutable NThreading::TFuture<NDrive::TOptionalUserDoubleGeoFeatures> UserDoubleGeoFeatures;
        mutable NThreading::TFuture<NDrive::TOptionalUserDoubleGeoFeatures> UserDoubleGeobaseFeatures;

    public:
        const TDestinationDescription& GetDestination() const {
            return *Destination;
        }

        const TOffersBuildingContext& GetOfferContext() const {
            return *OfferContext;
        }

        NJson::TJsonValue SerializeToJson() const;

        void Prefetch() const;

        const NGraph::TRouter::TOptionalRoute* GetRoute() const;
        const NDrive::TUserDoubleGeoFeatures* GetUserDoubleGeoFeatures() const;
        const NDrive::TUserDoubleGeoFeatures* GetUserDoubleGeobaseFeatures() const;

        TDestination(const TDestinationDescription& description, const TOffersBuildingContext& context)
            : Destination(&description)
            , OfferContext(&context)
        {
        }
    };
    using TDestinations = TVector<TDestination>;

    class TUserDestinationSuggestEx {
    public:
        struct TElement: public TUserOfferContext::TUserDestinationSuggestEx::TElement {
        public:
            using TBase = TUserOfferContext::TUserDestinationSuggestEx::TElement;

        public:
            TElement(const TBase& base)
                : TBase(base)
            {
            }

        public:
            NThreading::TFuture<NGraph::TRouter::TOptionalRoute> Route;
            NThreading::TFuture<NDrive::TOptionalUserDoubleGeoFeatures> UserDoubleGeoFeatures;
            NThreading::TFuture<NDrive::TOptionalUserDoubleGeoFeatures> UserDoubleGeobaseFeatures;
        };
        using TElements = TVector<TElement>;

    public:
        TElements Elements;
    };

    using TOfferConstructionProblems = TMap<TString, TString>;
    using TOfferCountFuture = NThreading::TFuture<ui64>;
    using TOfferFuture = NThreading::TFuture<ICommonOffer::TPtr>;
    using TOfferFutures = TMap<TString, TOfferFuture>;
    using TVariables = TMap<TString, NJson::TJsonValue>;

public:
    bool IsDelegation() const;
    TDuration DetermWalkingTime(const TUserPermissions& permissions) const;

    TDestinationDetectorTag::TContext& MutableFlowControlContext() const {
        return FlowControlContext;
    }

    bool NeedOfferedHintsSeparately() const {
        return NeedOfferedHintsSeparatelyFlag;
    }

    void SetNeedOfferedHintsSeparately() {
        NeedOfferedHintsSeparatelyFlag = true;
    }

    void SetNeedOfferedHintsOnly() {
        NeedOfferedHintsOnlyFlag = true;
    }

public:
    R_READONLY(TInstant, RequestStartTime, ModelingNow());
    R_READONLY(TVector<TDestination>, Destinations);
    R_OPTIONAL(TDuration, Border);
    R_OPTIONAL(TString, CarId);
    R_FIELD(NDrive::TOfferFeatures, Features, {}, mutable);
    R_FIELD(TString, PreviousOfferId);
    R_FIELD(TOfferCountFuture, PreviousOffersCount);
    R_FIELD(TOfferFutures, PreviousOffers);
    R_FIELD(TOfferFuture, PreviousOffer);
    R_FIELD(TString, BillingSessionId);
    R_FIELD(TString, Marker);
    R_FIELD(TString, ReqId);
    R_FIELD(bool, EnableAcceptanceCost, true);
    R_FIELD(bool, ForceInsuranceType, false);
    R_FIELD(bool, IgnoreUnavailableInsuranceType, false);
    R_FIELD(bool, NeedGeoFeatures, true);
    R_FIELD(bool, NeedRouteFeatures, false);
    R_FIELD(bool, NeedTaxiSurge, false);
    R_FIELD(bool, Switching, false);
    R_FIELD(bool, VisibilityRequired, true);
    R_FIELD(bool, UseHaversine, false);
    R_FIELD(bool, UseLocationTags, true);
    R_FIELD(bool, ReplacingCar, false);
    R_FIELD(bool, Prolonging, false);
    R_FIELD(TSet<TString>, SurgeTypes, {}, mutable);
    R_FIELD(NDrive::TOfferNotifications, Notifications, {}, mutable);
    R_OPTIONAL(NDrive::TLocationTags, LocationTags, {}, mutable);
    R_OPTIONAL(NDrive::TLocationAreaIds, LocationAreaIds, {}, mutable);

    R_READONLY(TOfferConstructionProblems, OfferConstructionProblems, {}, mutable);
    R_READONLY(TString, OfferConstructionProblemFirst, {}, mutable);
    R_OPTIONAL(double, CarWaitingDuration);
    R_OPTIONAL(TGeoCoord, CarFuturesStart);
    R_OPTIONAL(TVector<TGeoCoord>, AvailableStartArea);

    R_OPTIONAL(double, ExtraDistance);
    R_OPTIONAL(TDuration, ExtraDuration);
    R_OPTIONAL(TInstant, LongTermSince);
    R_OPTIONAL(TString, PackOfferNameOverride);
    R_OPTIONAL(TDuration, OverridenWalkingDuration);
    R_OPTIONAL(double, WalkingDistance);

    R_FIELD(TVariables, Variables);
    R_OPTIONAL(TString, SurgeAreaId, {}, mutable);
    R_FIELD(bool, ForceNonGroupOffer, false);
    R_FIELD(bool, BuildingFromGroupOffer, false);

public:
    using TGeoObjectsFuture = NThreading::TFuture<NGraph::TRouter::TGeoObjectInfos>;
    using TGeoFeaturesFuture = NThreading::TFuture<NDrive::TOptionalGeoFeatures>;
    using TUserGeoFeaturesFuture = NThreading::TFuture<NDrive::TOptionalUserGeoFeatures>;
    using TUserDoubleGeoFeaturesFuture = NThreading::TFuture<NDrive::TOptionalUserDoubleGeoFeatures>;
    using TUserDestinationSuggestFuture = NThreading::TFuture<TUserDestinationSuggestEx>;

private:
    mutable TOptionalGeobaseId GeobaseId;
    mutable TGeoFeaturesFuture GeoFeatures;
    mutable TGeoFeaturesFuture GeobaseFeatures;
    mutable TUserGeoFeaturesFuture UserGeoFeatures;
    mutable TUserGeoFeaturesFuture UserGeobaseFeatures;
    mutable TUserDestinationSuggestFuture UserDestinationSuggest;
    mutable TMaybe<TTaggedObject> TaggedCar;
    mutable TAtomicSharedPtr<TBillingSession> BillingSession;
    mutable TMaybe<TUserActions> PricesForObject;
    mutable NThreading::TFuture<NDrive::TTaxiSurgeCalculator::TResult> TaxiSurge;
    mutable NThreading::TFuture<TMaybe<NDrive::TRtmrSurge>> RtmrAreaSurge;
    mutable NThreading::TFuture<TMaybe<NDrive::TRtmrExtendedSurge>> RtmrAreaExtendedSurge;
    mutable TMaybe<TAdditionalPricedFeatureAction> InsuranceDescription;

public:
    NThreading::TFuture<NGraph::TRouter::TOptionalRoute> BuildRoute(
        const TGeoCoord& from,
        const TGeoCoord& to,
        const TString& graphPermissions
    ) const;

    const NDrive::IServer* GetServer() const {
        return Server;
    }

    void PrefetchGeobaseId() const {
        if (!GeobaseId) {
            GeobaseId = FetchGeobaseId();
        }
    }

    void PrefetchGeoFeatures() const {
        if (!GeoFeatures.Initialized()) {
            GeoFeatures = FetchGeoFeatures();
        }
    }

    void PrefetchGeobaseFeatures() const {
        if (!GeobaseFeatures.Initialized()) {
            GeobaseFeatures = FetchGeobaseFeatures();
        }
    }

    void PrefetchUserGeoFeatures() const {
        if (!UserGeoFeatures.Initialized()) {
            UserGeoFeatures = FetchUserGeoFeatures();
        }
    }

    void PrefetchUserGeobaseFeatures() const {
        if (!UserGeobaseFeatures.Initialized()) {
            UserGeobaseFeatures = FetchUserGeobaseFeatures();
        }
    }

    void PrefetchUserDestinationSuggest() const {
        if (!UserDestinationSuggest.Initialized()) {
            UserDestinationSuggest = FetchUserDestinationSuggest();
        }
    }

    void PrefetchRtmrAreaSurge() const {
        if (!RtmrAreaSurge.Initialized()) {
            RtmrAreaSurge = FetchRtmrAreaSurge();
        }
    }

    void PrefetchRtmrAreaExtendedSurge() const {
        if (!RtmrAreaExtendedSurge.Initialized()) {
            RtmrAreaExtendedSurge = FetchRtmrAreaExtendedSurge();
        }
    }

    void PrefetchWalking() const {
        if (!OverridenWalkingDuration && !RouteWalking.Initialized() && !!GetStartPosition() && HasUserHistoryContext() && UserHistoryContext->HasUserPosition()) {
            RouteWalking = BuildRoute(UserHistoryContext->GetUserPositionUnsafe(), *GetStartPosition(), "ptPedestrian");
        }
    }

    void PrefetchTaxiSurge() const;

    TOptionalGeobaseId FetchGeobaseId() const;
    TGeoFeaturesFuture FetchGeoFeatures() const;
    TGeoFeaturesFuture FetchGeobaseFeatures() const;
    TUserGeoFeaturesFuture FetchUserGeoFeatures() const;
    TUserGeoFeaturesFuture FetchUserGeobaseFeatures() const;
    TUserDoubleGeoFeaturesFuture FetchUserDoubleGeoFeatures(const TGeoCoord& from, const TGeoCoord& to) const;
    TUserDoubleGeoFeaturesFuture FetchUserDoubleGeoFeatures(TGeobaseId from, TGeobaseId to) const;
    TUserDestinationSuggestFuture FetchUserDestinationSuggest() const;
    NThreading::TFuture<NDrive::TTaxiSurgeCalculator::TResult> FetchTaxiSurge() const;
    NThreading::TFuture<TMaybe<NDrive::TRtmrSurge>> FetchRtmrAreaSurge() const;
    NThreading::TFuture<TMaybe<NDrive::TRtmrExtendedSurge>> FetchRtmrAreaExtendedSurge() const;

public:
    TOffersBuildingContext(const NDrive::IServer* server = nullptr);
    TOffersBuildingContext(TUserOfferContext&& uoc);
    TOffersBuildingContext(const TOffersBuildingContext& context) = default;
    ~TOffersBuildingContext();

    auto BuildEventGuard(const TString& id) const {
        return NDrive::BuildEventGuard(id);
    }

    template <class T>
    TMaybe<T> GetSetting(const TString& key) const {
        if (UserHistoryContext) {
            return UserHistoryContext->GetSetting<T>(key);
        } else if (Server) {
            return Server->GetSettings().GetValue<T>(key);
        } else {
            return {};
        }
    }

    void AddError(const TString& name, const TString& errorId) const;

    TInstant GetDeadline() const;
    const TString& GetExternalUserId() const;
    const TString& GetInsuranceTypeRef() const;
    TMaybe<bool> GetIsPlusUser() const;
    const TString& GetOrigin() const;
    TGeobaseId GetGeobaseId() const;
    TMaybe<TSet<TString>> GetComplementaryInsuranceTypes() const;
    TMaybe<TSet<TString>> GetInsuranceTypes() const;
    TMaybe<TGeoCoord> GetStartPosition() const;
    TUserActions GetPriceConstructorsForBehaviour(const TUserPermissions& permissions) const;
    TUserPermissions::TConstPtr GetUserPermissions() const;

    bool HasInsuranceType() const;
    void SetInsuranceType(const TString& value);

    void InitializeCar(const TString& carId);
    void Prefetch();

    TMaybe<EOfferCorrectorResult> GetGeoConditionsCheckResult(TStringBuf cacheId) const;
    void ClearGeoConditionsCheckResults() const;
    void SetGeoConditionsCheckResult(const TString& cacheId, EOfferCorrectorResult value) const;

    void ClearPricesForObject();

    const NDrive::TLocationTags& GetLocationTags() const;
    const NDrive::TLocationAreaIds& GetLocationAreaIds() const;
    TString GetSurgeAreaId() const;
    NDrive::TLocationTags GetUserLocationTags() const;
    TMaybe<TTaggedObject> GetTaggedCar() const;
    TAtomicSharedPtr<TBillingSession> GetBillingSession() const;

    const NDrive::TGeoFeatures* GetGeoFeatures() const;
    const NDrive::TGeoFeatures* GetGeobaseFeatures() const;
    const NDrive::TUserFeatures* GetUserFeatures() const;
    const NDrive::TUserGeoFeatures* GetUserGeoFeatures() const;
    const NDrive::TUserGeoFeatures* GetUserGeobaseFeatures() const;
    const TUserDestinationSuggestEx* GetUserDestinationSuggest() const;
    IOffer::TPtr GetPreviousOffer(TStringBuf behaviourConstructorId, bool waitStage = false) const;
    TMaybe<TDuration> GetWalkingDuration(const double pedestrianSpeedKmh) const;

    TUserDestinationSuggestEx* MutableUserDestinationSuggest() const;

    TMaybe<NDrive::TTaxiSurgeCalculator::TResult> GetTaxiSurge() const;
    TMaybe<NDrive::TRtmrSurge> GetRtmrAreaSurge() const;
    TMaybe<NDrive::TRtmrExtendedSurge> GetRtmrAreaExtendedSurge() const;
    // GetInsuranceDescription returns action that contains description about available insurance.
    TMaybe<TAdditionalPricedFeatureAction> GetInsuranceDescription() const;

public:
    static TDuration CorrectWalkingDuration(const double originalTime, const double pedestrianSpeedKmh);

private:
    const NDrive::TUserEventsApi* GetFeaturesClient() const;
};
