#include "discount.h"

#include <drive/backend/offers/actions/correctors.h>

#include <drive/backend/compiled_riding/compiled_riding.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/offers/actions/standard_with_discount_area.h>
#include <drive/backend/proto/offer.pb.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/common/localization.h>

TDiscount TDiscount::operator+(const TDiscount& discount) const {
    TDiscount result = *this;
    result.Discount = Min(1.0, Discount + discount.Discount);
    for (auto&& i : result.DiscountDetails) {
        auto it = discount.DiscountDetails.find(i.first);
        if (it != discount.DiscountDetails.end()) {
            i.second = i.second + it->second;
        }
    }
    for (auto&& i : discount.DiscountDetails) {
        auto it = result.DiscountDetails.find(i.first);
        if (it == result.DiscountDetails.end()) {
            result.DiscountDetails.emplace(i.first, i.second);
        }
    }
    return result;
}

const TDiscount::TDiscountDetails& TDiscount::GetDetails(const TString& tag) const {
    auto it = DiscountDetails.find(tag);
    if (it == DiscountDetails.end()) {
        return Default<TDiscountDetails>();
    } else {
        return it->second;
    }
}

TDiscount& TDiscount::AddDetails(const TDiscountDetails& detail) {
    DiscountDetails.emplace(detail.GetTagName(), detail);
    return *this;
}

TDiscount& TDiscount::SetDetails(const TMap<TString, TDiscountDetails>& details) {
    DiscountDetails = details;
    return *this;
}

NJson::TJsonValue TDiscount::GetPublicReport(ELocalization locale, const NDrive::IServer& server, const bool negativeTimes) const {
    NJson::TJsonValue result;
    NJson::TJsonValue& details = result.InsertValue("details", NJson::JSON_ARRAY);
    for (auto&& i : DiscountDetails) {
        NJson::TJsonValue detailsJson = i.second.GetPublicReport(negativeTimes);
        if (i.second.GetFreeTimetable() && i.second.GetTagsInPoint().empty()) {
            const TTimeRestriction* widestRestriction = i.second.GetFreeTimetable()->GetWidestRestriction(Now(), Now() + TDuration::Days(2));
            if (widestRestriction) {
                detailsJson.InsertValue("widest_free_period", widestRestriction->SerializeToJson());
            }
        }
        details.InsertValue(i.first, detailsJson);
    }

    result.InsertValue("discount", Discount);
    result.InsertValue("id", Identifier);
    TMaybe<TDBAction> actionDiscount;
    if (!!Identifier) {
        actionDiscount = server.GetDriveAPI()->GetRolesManager()->GetActionsDB().GetObject(Identifier);
    }
    const TDiscountOfferCorrector* corrector = !!actionDiscount ? actionDiscount->GetAs<TDiscountOfferCorrector>() : nullptr;

    if (corrector) {
        NJson::InsertNonNull(result, "title", server.GetLocalization()->ApplyResources(corrector->GetDescription(), locale));
        NJson::InsertNonNull(result, "description", server.GetLocalization()->ApplyResources(corrector->GetDiscountDescription(), locale));
        NJson::InsertNonNull(result, "icon", corrector->GetIcon());
        NJson::InsertNonNull(result, "small_icon", corrector->GetSmallIcon());
        NJson::InsertField(result, "visible", Visible);
    } else {
        const TStandardWithDiscountAreaOfferBuilder* builder = !!actionDiscount ? actionDiscount->GetAs<TStandardWithDiscountAreaOfferBuilder>() : nullptr;
        if (builder) {
            NJson::InsertNonNull(result, "title", server.GetLocalization()->GetLocalString(locale, "standard_with_discount_area." + Identifier + ".discount.title"));
            NJson::InsertNonNull(result, "description", server.GetLocalization()->GetLocalString(locale, "standard_with_discount_area." + Identifier + ".discount.description"));
            NJson::InsertNonNull(result, "icon", server.GetLocalization()->GetLocalString(locale, "standard_with_discount_area." + Identifier + ".discount.icon"));
            NJson::InsertNonNull(result, "small_icon", server.GetLocalization()->GetLocalString(locale, "standard_with_discount_area." + Identifier + ".discount.small_icon"));
            NJson::InsertField(result, "visible", Visible);
        }
    }
    if (PromoCode) {
        result.InsertValue("promo_code", PromoCode);
    }
    return result;
}

bool TDiscount::DeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    NJson::TJsonValue::TArray jsonArray;
    if (jsonInfo.Has("details")) {
        if (!jsonInfo["details"].GetArray(&jsonArray)) {
            return false;
        } else {
            for (auto&& i : jsonArray) {
                TDiscount::TDiscountDetails dd;
                if (!dd.DeserializeFromJson(i)) {
                    return false;
                }
                DiscountDetails.emplace(dd.GetTagName(), dd);
            }
        }
    }

    JREAD_DOUBLE(jsonInfo, "discount", Discount);
    JREAD_STRING(jsonInfo, "id", Identifier);
    JREAD_BOOL_OPT(jsonInfo, "visible", Visible);
    JREAD_BOOL_OPT(jsonInfo, "promo_code", PromoCode);
    return true;
}

void TDiscount::FillBillRecord(ELocalization locale, const NDrive::IServer& server, TBillRecord& record) const {
    record.SetId(GetIdentifier());
    record.SetDetails(::ToString(GetDiscount() * 100) + "%");
    bool storeDiscountInfo = server.GetSettings().GetValue<bool>("bill.store_discount_info").GetOrElse(true);
    if (GetIdentifier() && storeDiscountInfo) {
        TMaybe<TDBAction> actionDiscount;
        if (!!Identifier) {
            actionDiscount = server.GetDriveAPI()->GetRolesManager()->GetActionsDB().GetObject(Identifier);
        }
        const TDiscountOfferCorrector* corrector = !!actionDiscount ? actionDiscount->GetAs<TDiscountOfferCorrector>() : nullptr;
        if (corrector) {
            record.SetIcon(corrector->GetSmallIcon() ? corrector->GetSmallIcon() : corrector->GetIcon());
            record.SetTitle(server.GetLocalization()->ApplyResources(corrector->GetDescription(), locale));
        } else {
            const TStandardWithDiscountAreaOfferBuilder* builder = !!actionDiscount ? actionDiscount->GetAs<TStandardWithDiscountAreaOfferBuilder>() : nullptr;
            if (builder) {
                record.SetTitle(server.GetLocalization()->GetLocalString(locale, "standard_with_discount_area." + Identifier + ".discount.title"));
                record.SetIcon(server.GetLocalization()->GetLocalString(locale, "standard_with_discount_area." + Identifier + ".discount.small_icon"));
            }
        }
    }
}

bool TDiscount::DeserializeFromProto(const NDrive::NProto::TDiscount& info) {
    Discount = info.GetDiscount();
    Identifier = info.GetIdentifier();
    Visible = info.GetVisible();
    for (auto&& i : info.GetDetails()) {
        TDiscount::TDiscountDetails detail;
        if (!detail.DeserializeFromProto(i)) {
            return false;
        }
        DiscountDetails.emplace(i.GetTagName(), detail);
    }
    if (info.HasIsPromoCode()) {
        PromoCode = info.GetIsPromoCode();
    }
    return true;
}

NDrive::NProto::TDiscount TDiscount::SerializeToProto() const {
    NDrive::NProto::TDiscount result;
    result.SetDiscount(Discount);
    result.SetIdentifier(Identifier);
    result.SetVisible(Visible);
    for (auto&& i : DiscountDetails) {
        *result.AddDetails() = i.second.SerializeToProto();
    }
    result.SetIsPromoCode(PromoCode);
    return result;
}

TString IOfferWithDiscounts::DoBuildCommonCalculationDescription(ELocalization locale, const TFullCompiledRiding& fcr, const NDrive::IServer& server) const {
    TStringBuilder sb;
    if (HasDiscountsWithVisibility(true)) {
        sb << "\\subsection{Public discounts}" << Endl;
        sb << "\\begin{enumerate}" << Endl;
        for (auto&& i : Discounts) {
            if (i.GetVisible()) {
                sb << i.BuildCalculationDescription(fcr, server) << Endl;
            }
        }
        sb << "\\end{enumerate}" << Endl;
    }

    if (HasDiscountsWithVisibility(false)) {
        sb << "\\subsection{Hidden discounts}" << Endl;
        sb << "\\begin{enumerate}" << Endl;
        for (auto&& i : Discounts) {
            if (!i.GetVisible()) {
                sb << i.BuildCalculationDescription(fcr, server) << Endl;
            }
        }
        sb << "\\end{enumerate}" << Endl;
    }

    const TVector<TString> phases = TChargableTag::GetTagNames(server.GetDriveAPI()->GetTagsManager().GetTagsMeta());
    sb << "\\subsection{Phases}" << Endl;
    if (phases.size()) {
        sb << "\\begin{enumerate}" << Endl;
    }
    for (auto&& t : phases) {
        TDuration d;
        if (!fcr.GetPhaseDuration(t, d)) {
            sb << "\\item \\verb|" << t << "| FAILED" << Endl;
            continue;
        }
        if (d == TDuration::Zero()) {
            continue;
        }
        sb << "\\item \\verb|" << t << "|" << Endl;
        sb << "\\begin{enumerate}" << Endl;
        sb << "\\item " << "Duration: " << server.GetLocalization()->FormatDuration(locale, d, true) << Endl;
        sb << "\\item " << "Free: " << server.GetLocalization()->FormatDuration(locale, GetFreeDuration(t), true) << Endl;
        sb << "\\end{enumerate}" << Endl;
    }
    if (phases.size()) {
        sb << "\\end{enumerate}" << Endl;
    }

    if (fcr.HasSnapshotsDiff() && fcr.GetSnapshotsDiffUnsafe().HasMileage()) {
        sb << "\\subsection{Mileage " << server.GetLocalization()->DistanceFormatKm(locale, fcr.GetSnapshotsDiffUnsafe().GetMileageUnsafe()) << "}" << Endl;
    }
    return sb;
}

NJson::TJsonValue IOfferWithDiscounts::DoBuildJsonReport(const TReportOptions& options, const ICommonOfferBuilderAction* constructor, const NDrive::IServer& server) const {
    NJson::TJsonValue report = TBase::DoBuildJsonReport(options, constructor, server);
    auto locale = options.Locale;
    if (!(options.Traits & NDriveSession::ReportOfferDetails)) {
        return report;
    }
    if (Discounts.size()) {
        auto summaryDiscount = GetSummaryVisibleDiscount();
        if (summaryDiscount.GetVisible()) {
            NJson::TJsonValue& discounts = report.InsertValue("discounts", NJson::JSON_NULL);
            for (auto&& i : Discounts) {
                if (i.GetVisible()) {
                    discounts.AppendValue(i.GetPublicReport(locale, server, true));
                }
            }
            report.InsertValue("summary_discount", summaryDiscount.GetPublicReport(locale, server, false));
        }
    }
    report.InsertValue("cashback", IOfferWithCashback::BuildReport(server));
    return report;
}

NDrive::NProto::TOffer IOfferWithDiscounts::SerializeToProto() const {
    auto info = TBase::SerializeToProto();
    for (auto&& i : Discounts) {
        *info.MutableDiscountsInfo()->AddDiscounts() = i.SerializeToProto();
    }
    IOfferWithCashback::ToProto(*info.MutableCashback());
    return info;
}

bool IOfferWithDiscounts::DeserializeFromProto(const NDrive::NProto::TOffer& info) {
    Discounts.clear();
    SumHiddenDiscount = 0;
    SumVisibleDiscount = 0;
    if (!TBase::DeserializeFromProto(info)) {
        return false;
    }
    for (auto&& i : info.GetDiscountsInfo().GetDiscounts()) {
        TDiscount discount;
        if (!discount.DeserializeFromProto(i)) {
            return false;
        } else {
            AddDiscount(discount);
        }
    }
    if (!IOfferWithCashback::FromProto(info.GetCashback())) {
        return false;
    }
    return true;
}

TDiscount IOfferWithDiscounts::GetSummaryDiscount(const bool isHidden) const {
    TDiscount result;
    bool isFirst = true;
    for (auto&& i : Discounts) {
        if (i.GetVisible() == isHidden) {
            continue;
        }
        if (isFirst) {
            isFirst = false;
            result = i;
        } else {
            result = result + i;
        }
    }
    return result;
}

TDiscount IOfferWithDiscounts::GetSummaryVisibleDiscount() const {
    return GetSummaryDiscount(false);
}

TDiscount IOfferWithDiscounts::GetSummaryHiddenDiscount() const {
    return GetSummaryDiscount(true);
}

void IOfferWithDiscounts::AddDiscount(const TDiscount& discount) {
    if (!discount.GetVisible()) {
        SumHiddenDiscount += discount.GetDiscount();
    } else {
        SumVisibleDiscount += discount.GetDiscount();
    }
    Discounts.push_back(discount);
}

TDiscount::TDiscountDetails TDiscount::TDiscountDetails::operator+(const TDiscountDetails& d) const {
    TDiscountDetails result = *this;
    result.Discount = Min(1.0, Discount + d.Discount);
    result.AdditionalTime = AdditionalTime + d.AdditionalTime;
    if (!!d.GetFreeTimetable()) {
        if (!result.GetFreeTimetable()) {
            result.FreeTimetable = *d.GetFreeTimetable();
        } else {
            result.FreeTimetable.Merge(*d.GetFreeTimetable());
        }
    }
    return result;
}

bool TDiscount::TDiscountDetails::DeserializeFromProto(const NDrive::NProto::TDiscountDetails& info) {
    AdditionalTime = info.GetAdditionalTime();
    Discount = info.GetDiscount();
    TagName = info.GetTagName();
    for (auto&& i : info.GetTagsInPoint()) {
        TagsInPoint.emplace(i);
    }
    if (info.HasFreeTimetable()) {
        FreeTimetable.DeserializeFromProto(info.GetFreeTimetable());
    }
    return true;
}

NDrive::NProto::TDiscountDetails TDiscount::TDiscountDetails::SerializeToProto() const {
    NDrive::NProto::TDiscountDetails result;
    result.SetAdditionalTime(AdditionalTime);
    result.SetDiscount(Discount);
    result.SetTagName(TagName);
    for (auto&& i : TagsInPoint) {
        result.AddTagsInPoint(i);
    }
    if (!FreeTimetable.Empty()) {
        *result.MutableFreeTimetable() = FreeTimetable.SerializeToProto();
    }
    return result;
}

NJson::TJsonValue TDiscount::TDiscountDetails::GetPublicReport(const bool negativeTimes) const {
    NJson::TJsonValue result(NJson::JSON_MAP);
    JWRITE(result, "tag_name", TagName);
    JWRITE_DEF(result, "discount", Discount, 0);
    if (negativeTimes) {
        JWRITE(result, "additional_duration", AdditionalTime);
    } else {
        JWRITE(result, "additional_duration", Max(0, AdditionalTime));
    }

    if (!!GetFreeTimetable() && TagsInPoint.empty()) {
        result.InsertValue("free_timetable", FreeTimetable.SerializeToJson());
        JWRITE(result, "is_actual", FreeTimetable.IsActualNow(ModelingNow()));
    }

    return result;
}

NJson::TJsonValue TDiscount::TDiscountDetails::SerializeToJson() const {
    NJson::TJsonValue result(NJson::JSON_MAP);
    JWRITE(result, "tag_name", TagName);
    JWRITE_DEF(result, "discount", Discount, 0);
    JWRITE_DEF(result, "additional_duration", AdditionalTime, 0);
    if (TagsInPoint.size()) {
        NJson::TJsonValue& tags = result.InsertValue("tags_in_point", NJson::JSON_ARRAY);
        for (auto&& i : TagsInPoint) {
            tags.AppendValue(i);
        }
    }

    if (!!GetFreeTimetable()) {
        result.InsertValue("free_timetable", FreeTimetable.SerializeToJson());
    }

    return result;
}

bool TDiscount::TDiscountDetails::DeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!jsonInfo.IsMap()) {
        return false;
    }

    JREAD_STRING(jsonInfo, "tag_name", TagName);
    JREAD_DOUBLE_OPT(jsonInfo, "discount", Discount);
    JREAD_INT_OPT(jsonInfo, "additional_duration", AdditionalTime);
    if (jsonInfo.Has("tags_in_point")) {
        NJson::TJsonValue::TArray arr;
        if (!jsonInfo["tags_in_point"].GetArray(&arr)) {
            return false;
        }
        for (auto&& i : arr) {
            if (!i.IsString()) {
                return false;
            }
            TagsInPoint.emplace(i.GetString());
        }
    }

    if (jsonInfo.Has("free_timetable") && !!jsonInfo["free_timetable"].GetStringRobust()) {
        if (!FreeTimetable.DeserializeFromJson(jsonInfo["free_timetable"])) {
            return false;
        }
    }

    return true;
}

template<>
bool NJson::TryFromJson(const NJson::TJsonValue& jsonInfo, TCashbackInfo& cashback) {
    return
        NJson::ParseField(jsonInfo, "value", cashback.OptionalValue()) &&
        NJson::ParseField(jsonInfo, "cashback_percent", cashback.OptionalCashbackPercent()) &&
        NJson::ParseField(jsonInfo, "distance", cashback.MutableMinimalDistance(), false) &&
        NJson::ParseField(jsonInfo, "price", cashback.MutableMinimalPrice(), false) &&
        NJson::ParseField(jsonInfo, "payload", cashback.OptionalPayload()) &&
        NJson::ParseField(jsonInfo, "stages", cashback.MutableStages()) &&
        NJson::ParseField(jsonInfo, "id", cashback.MutableId());
}

template<>
NJson::TJsonValue NJson::ToJson(const TCashbackInfo& cashback) {
    NJson::TJsonValue result;
    NJson::InsertNonNull(result, "value", cashback.OptionalValue());
    NJson::InsertNonNull(result, "cashback_percent", cashback.OptionalCashbackPercent());
    NJson::InsertField(result, "distance", cashback.GetMinimalDistance());
    NJson::InsertField(result, "price", cashback.GetMinimalPrice());
    NJson::InsertNonNull(result, "payload", cashback.OptionalPayload());
    NJson::InsertField(result, "stages", cashback.GetStages());
    NJson::InsertField(result, "id", cashback.GetId());
    return result;
}

bool TCashbackInfo::DeserializeFromProto(const NDrive::NProto::TCashbackDetails& info) {
    if (info.HasValue()) {
        Value = info.GetValue();
    }
    if (info.HasCashbackPercent()) {
        CashbackPercent = info.GetCashbackPercent();
    }
    MinimalDistance = info.GetMinimalDistance();
    MinimalPrice = info.GetMinimalPrice();
    if (info.HasPayload()) {
        TString payloadStr = info.GetPayload();
        Payload.ConstructInPlace();
        if (!NJson::ReadJsonTree(payloadStr, Payload.Get())) {
            return false;
        }
    }
    for (auto&& i : info.GetStages()) {
        Stages.emplace(i);
    }
    Id = info.GetId();
    return true;
}

NDrive::NProto::TCashbackDetails TCashbackInfo::SerializeToProto() const {
    NDrive::NProto::TCashbackDetails result;
    if (Value) {
        result.SetValue(*Value);
    }
    if (CashbackPercent) {
        result.SetCashbackPercent(*CashbackPercent);
    }
    if (Payload) {
        result.SetPayload(Payload->GetStringRobust());
    }
    result.SetMinimalDistance(MinimalDistance);
    result.SetMinimalPrice(MinimalPrice);
    for (auto&& i : Stages) {
        *result.AddStages() = i;
    }
    result.SetId(Id);
    return result;
}

void TCashbackInfo::AddCashback(TOfferPricing& pricing) const {
    auto lp = pricing;
    if (Stages) {
        lp.CleanOtherPrices(Stages);
    }
    if ((i64)lp.GetReportSumPrice() < (i64)MinimalPrice || (i64)lp.GetDistance() < (i64)MinimalDistance) {
        return;
    }
    if ((i64)lp.GetBillingSumPrice() < (i64)MinimalPrice) {
        pricing.SetBorderCashback(pricing.GetCashback());
    }
    if (Value) {
        pricing.AddCashback(Id, *Value, Payload);
    }
    if (CashbackPercent) {
        i64 value = (i64)lp.GetReportSumPrice() * (*CashbackPercent)/10000.;
        if (value) {
            pricing.AddCashback(Id, value, Payload);
        }
    }
}

void TCashbackInfo::AddScheme(NDrive::TScheme& result, const NDrive::IServer* /*server*/) {
    result.Add<TFSNumeric>("value", "Количество баллов").SetMin(1);
    result.Add<TFSJson>("payload");
    result.Add<TFSNumeric>("distance", "Необходимый пробег").SetDefault(0);
    result.Add<TFSNumeric>("price", "Минимальная стоимость").SetDefault(0);
    result.Add<TFSArray>("stages", "Учитывать стадии").SetElement<TFSString>();
    result.Add<TFSNumeric>("cashback_percent", "Процент кэшбека");
}

void IOfferWithCashback::AddCashback(const TCashbackInfo& cashback) {
    Cashbacks.push_back(cashback);
}

void IOfferWithCashback::ToProto(NDrive::NProto::TCashbacksInfo& info) const {
    for (auto&& i : Cashbacks) {
        *info.AddCashbacks() = i.SerializeToProto();
    }
}

bool IOfferWithCashback::FromProto(const NDrive::NProto::TCashbacksInfo& info) {
    Cashbacks.clear();
    for (auto&& i : info.GetCashbacks()) {
        TCashbackInfo cashback;
        if (!cashback.DeserializeFromProto(i)) {
            return false;
        } else {
            AddCashback(cashback);
        }
    }
    return true;
}

void IOfferWithCashback::FillBill(TBill& bill, const TOfferPricing& pricing, ELocalization locale, const NDrive::IServer* server) const {
    if (pricing.GetCashback() > 0) {
        TBillRecord billRecord;
        billRecord.SetCost(pricing.GetCashback())
            .SetType(TBillRecord::CashbackType)
            .SetIcon(server->GetSettings().GetValueDef<TString>("plus_coin_icon", ""));
        auto localization = server ? server->GetLocalization() : nullptr;
        const TString specTitle = server->GetSettings().GetValueDef<TString>("offers.localization.special." + TBillRecord::CashbackType + ".title", NDrive::TLocalization::CashbackBillingHeader());
        billRecord.SetTitle(localization ? localization->ApplyResources(specTitle, locale) : specTitle);
        const TString specDescription = server->GetSettings().GetValueDef<TString>("offers.localization.special." + TBillRecord::CashbackType + ".description", "");
        if (specDescription) {
            billRecord.SetDetails(localization ? localization->ApplyResources(specDescription, locale) : specDescription);
        }
        bill.MutableRecords().emplace_back(std::move(billRecord));
    }
}

NJson::TJsonValue IOfferWithCashback::BuildReport(const NDrive::IServer& /*server*/) const {
    NJson::TJsonValue result;
    NJson::InsertField(result, "parts", Cashbacks);
    return result;
}

void IOfferWithCashback::Calculate(TOfferPricing& result) const {
    for (auto&& info : Cashbacks) {
        info.AddCashback(result);
    }
}

void IOfferWithDiscounts::FillBillDiscounts(TBill& bill, const TOfferPricing& pricing, ELocalization locale, const NDrive::IServer* server) const {
    for (auto&& i : Discounts) {
        if (Abs(i.GetDiscount()) < 1e-5 || Abs(i.GetDiscount() * pricing.GetReportSumOriginalPrice()) < 1e-5) {
            continue;
        }
        if (!i.GetVisible()) {
            continue;
        }
        TBillRecord billRecord;

        i.FillBillRecord(locale, *server, billRecord);

        billRecord.SetCost((int)(-i.GetDiscount() * pricing.GetReportSumOriginalPrice()));
        billRecord.SetType(TBillRecord::DiscountType);
        bill.MutableRecords().emplace_back(std::move(billRecord));
    }
}

void IOfferWithDiscounts::FillBill(TBill& bill, const TOfferPricing& pricing, TOfferStatePtr segmentState, ELocalization locale, const NDrive::IServer* server, ui32 cashbackPercent) const {
    FillBillPricing(bill, pricing, segmentState, locale, server);
    FillBillDiscounts(bill, pricing, locale, server);
    IOffer::FillBill(bill, pricing, segmentState, locale, server, cashbackPercent);
    IOfferWithCashback::FillBill(bill, pricing, locale, server);
}

TOfferStatePtr IOfferWithDiscounts::Calculate(const TVector<IEventsSession<TCarTagHistoryEvent>::TTimeEvent>& timeline, const TVector<TAtomicSharedPtr<TCarTagHistoryEvent>>& events, const TInstant& until, TOfferPricing& result) const {
    auto state = IOffer::Calculate(timeline, events, until, result);
    IOfferWithCashback::Calculate(result);
    return state;
}
