#pragma once

#include "abstract.h"

#include <library/cpp/json/writer/json_value.h>
#include <library/cpp/yconf/conf.h>

#include <rtline/library/time_restriction/time_restriction.h>
#include <rtline/util/types/accessor.h>

#include <util/datetime/base.h>
#include <util/string/cast.h>

namespace NDrive::NProto {
    class TDiscount;
    class TDiscountDetails;
    class TCashbackDetails;
    class TCashbacksInfo;
}

class TBillRecord;

class TDiscount {
    R_FIELD(double, Discount, 0);
    R_FIELD(TString, Identifier);
    R_FIELD(bool, Visible, false);
    R_FIELD(bool, PromoCode, false);
public:
    class TDiscountDetails {
        R_FIELD(i32, AdditionalTime, 0);
        R_FIELD(double, Discount, 0);
        R_FIELD(TString, TagName);
        R_FIELD(TSet<TString>, TagsInPoint);

    private:
        TTimeRestrictionsPool<TTimeRestriction> FreeTimetable;

    public:
        TDiscountDetails operator+(const TDiscountDetails& d) const;

        NJson::TJsonValue GetPublicReport(const bool negativeTimes) const;

        NJson::TJsonValue SerializeToJson() const;
        bool DeserializeFromJson(const NJson::TJsonValue& jsonInfo);

        bool DeserializeFromProto(const NDrive::NProto::TDiscountDetails& info);
        NDrive::NProto::TDiscountDetails SerializeToProto() const;

        TDiscountDetails& SetFreeTimetable(const TTimeRestrictionsPool<TTimeRestriction>& timetable) {
            FreeTimetable = timetable;
            return *this;
        }

        const TTimeRestrictionsPool<TTimeRestriction>* GetFreeTimetable() const {
            if (!FreeTimetable.RestrictionsCount()) {
                return nullptr;
            }
            return &FreeTimetable;
        }
    };

private:
    TMap<TString, TDiscountDetails> DiscountDetails;

public:
    TString BuildCalculationDescription(const TFullCompiledRiding& /*fcr*/, const NDrive::IServer& /*server*/) const {
        TStringBuilder sb;
        sb << "\\item \\verb|" << GetIdentifier() << "|;common\\_discount=" << Discount << ";promo=" << IsPromoCode() << Endl;
        if (DiscountDetails.size()) {
            sb << "\\begin{enumerate}" << Endl;
            for (auto&& i : DiscountDetails) {
                sb << "\\item \\verb|" << i.second.GetTagName() << "|;discount=" << i.second.GetDiscount() << ";additional\\_time=" << i.second.GetAdditionalTime() << Endl;
            }
            sb << "\\end{enumerate}" << Endl;
        }
        return sb;
    }

    TDiscount operator+ (const TDiscount& discount) const;

    double GetDiscount(const TString& tag) const {
        if (!!tag) {
            return GetDetails(tag).GetDiscount() + Discount;
        } else {
            return Discount;
        }
    }

    const TDiscountDetails& GetDetails(const TString& tag) const;
    TDiscount& AddDetails(const TDiscountDetails& detail);
    TDiscount& SetDetails(const TMap<TString, TDiscountDetails>& details);

    void FillBillRecord(ELocalization locale, const NDrive::IServer& server, TBillRecord& record) const;
    NJson::TJsonValue GetPublicReport(ELocalization locale, const NDrive::IServer& server, const bool negativeTimes) const;

    bool DeserializeFromJson(const NJson::TJsonValue& jsonInfo);

    NDrive::NProto::TDiscount SerializeToProto() const;
    bool DeserializeFromProto(const NDrive::NProto::TDiscount& info);
};

enum class ESessionState {
    Parking = 1 << 0 /* "old_state_parking" */,
    Riding = 1 << 1 /* "old_state_riding" */,
    Reservation = 1 << 2 /* "old_state_reservation" */,
    Acceptance = 1 << 3 /* "old_state_acceptance" */,
};

class TCashbackInfo {
    R_OPTIONAL(i32, Value);
    R_OPTIONAL(i32, CashbackPercent);
    R_OPTIONAL(NJson::TJsonValue, Payload);
    R_FIELD(ui32, MinimalDistance, 0);
    R_FIELD(ui32, MinimalPrice, 0);
    R_FIELD(TSet<TString>, Stages);
    R_FIELD(TString, Id);

public:
    void AddCashback(TOfferPricing& pricing) const;
    bool DeserializeFromProto(const NDrive::NProto::TCashbackDetails& info);
    NDrive::NProto::TCashbackDetails SerializeToProto() const;
    static void AddScheme(NDrive::TScheme& scheme, const NDrive::IServer* server);
};

class IOfferWithCashback {
protected:
    TVector<TCashbackInfo> Cashbacks;

public:
    void AddCashback(const TCashbackInfo& cashback);

    NJson::TJsonValue BuildReport(const NDrive::IServer& server) const;

    void FillBill(TBill& bill, const TOfferPricing& pricing, ELocalization locale, const NDrive::IServer* server) const;
    void Calculate(TOfferPricing& pricing) const;

    void ToProto(NDrive::NProto::TCashbacksInfo& info) const;
    bool FromProto(const NDrive::NProto::TCashbacksInfo& info);
};

class IOfferWithDiscounts: public IOffer, public IOfferWithCashback {
private:
    using TBase = IOffer;

private:
    double SumHiddenDiscount = 0;
    R_READONLY(double, SumVisibleDiscount, 0);
    TDiscount GetSummaryDiscount(const bool isHidden) const;
protected:
    TVector<TDiscount> Discounts;

protected:
    virtual void FillBillPricing(TBill& bill, const TOfferPricing& pricing, TOfferStatePtr segmentState, ELocalization locale, const NDrive::IServer* server) const = 0;
    void FillBillDiscounts(TBill& bill, const TOfferPricing& pricing, ELocalization locale, const NDrive::IServer* server) const;
    virtual TDuration GetFreeDuration(const TString& tag, const bool publicOnly = false) const = 0;

    TString DoBuildCommonCalculationDescription(ELocalization locale, const TFullCompiledRiding& fcr, const NDrive::IServer& server) const override;

    bool HasDiscountsWithVisibility(const bool visibility) const {
        for (auto&& i : Discounts) {
            if (i.GetVisible() == visibility) {
                return true;
            }
        }
        return false;
    }

    virtual ui32 GetDiscountedPrice(const ui32 price) const override {
        return GetPublicOriginalPrice(price, "");
    }

    ui32 GetPublicDiscountedPrice(const ui32 price, const ESessionState state, const ui32 precision = 1) const {
        return GetPublicDiscountedPrice(price, ::ToString(state), precision);
    }

    ui32 GetPublicOriginalPrice(const ui32 price, const ESessionState state, const ui32 precision = 1) const {
        return GetPublicOriginalPrice(price, ::ToString(state), precision);
    }

    ui32 GetPublicOriginalPrice(double price, const TString& tag = {}, ui32 precision = 1) const {
        return RoundPrice(ApplyDiscount(price, false, tag), precision);
    }

    virtual NJson::TJsonValue DoBuildJsonReport(const TReportOptions& options, const ICommonOfferBuilderAction* constructor, const NDrive::IServer& server) const override;

public:
    using TBase::TBase;

    virtual void FillBill(TBill& bill, const TOfferPricing& pricing, TOfferStatePtr segmentState, ELocalization locale, const NDrive::IServer* server, ui32 cashbackPercent) const override;

    TOfferStatePtr Calculate(const TVector<IEventsSession<TCarTagHistoryEvent>::TTimeEvent>& timeline, const TVector<TAtomicSharedPtr<TCarTagHistoryEvent>>& events, const TInstant& until, TOfferPricing& result) const override;

    ui32 GetPublicDiscountedPrice(double price, const TString& tag = {}, ui32 precision = 1) const {
        return RoundPrice(ApplyDiscount(price, tag), precision);
    }

    double GetDiscountMultiplier(const TString& tag) const {
        return (1 - GetSummaryVisibleDiscount().GetDiscount(tag)) * (1 - GetSummaryHiddenDiscount().GetDiscount(tag));
    }

    double ApplyDiscount(const double price, const TString& tag) const {
        return GetDiscountMultiplier(tag) * price;
    }

    double ApplyDiscount(const double price, const bool visibility, const TString& tag) const {
        if (visibility) {
            return (1 - GetSummaryVisibleDiscount().GetDiscount(tag)) * price;
        } else {
            return (1 - GetSummaryHiddenDiscount().GetDiscount(tag)) * price;
        }
    }

    double GetDiscountMultiplier(const ESessionState tag) const {
        return GetDiscountMultiplier(::ToString(tag));
    }

    double ApplyDiscount(const double price, const ESessionState tag) const {
        return ApplyDiscount(price, ::ToString(tag));
    }

    double ApplyDiscount(const double price, const bool visibility, const ESessionState tag) const {
        return ApplyDiscount(price, visibility, ::ToString(tag));
    }

    TDiscount GetSummaryVisibleDiscount() const;
    TDiscount GetSummaryHiddenDiscount() const;

    void AddDiscount(const TDiscount& discount);

    NDrive::NProto::TOffer SerializeToProto() const override;
    bool DeserializeFromProto(const NDrive::NProto::TOffer& info) override;
};
