#include "manager.h"

#include <drive/backend/logging/events.h>
#include <drive/backend/proto/offer.pb.h>
#include <drive/backend/saas/api.h>

#include <drive/library/cpp/compression/simple.h>
#include <drive/library/cpp/threading/future.h>

#include <rtline/protos/proto_helper.h>
#include <rtline/util/algorithm/ptr.h>

namespace {
    TNamedSignalSimple SignalOfferStoreProblems("offer-store-problems");
    TNamedSignalSimple SignalOfferStoreSuccess("offer-store-success");
}

class TDBOffersStorage::TExternalSessionGuard {
private:
    const bool ReadOnly;
    NDrive::TEntitySession* Session;
    NDrive::TEntitySession LocalSession;

public:
    TExternalSessionGuard(NDrive::TInfoEntitySession& extSession, const TDatabaseSessionConstructor& sConstructor, const bool readOnly)
        : ReadOnly(readOnly)
    {
        Session = dynamic_cast<NDrive::TEntitySession*>(&extSession);
        if (!Session) {
            LocalSession = sConstructor.BuildSession(ReadOnly);
            Session = &LocalSession;
        }
    }

    ~TExternalSessionGuard() noexcept(false) {
    }

    bool Commit() {
        if (LocalSession) {
            return LocalSession.Commit();
        }
        return true;
    }

    NDrive::TEntitySession& GetSession() const {
        return *Yensured(Session);
    }

    NStorage::ITransaction::TPtr GetTransaction() const {
        return GetSession().GetTransaction();
    }
};

NThreading::TFuture<ui64> TDBOffersStorage::GetOffersCount(const TString& objectId, const TString& userId, NDrive::TInfoEntitySession& session) {
    auto sessionGuard = BuildSessionGuard(session, true);
    auto transaction = sessionGuard.GetTransaction();
    auto query = TStringBuilder()
        << "SELECT constructor_id FROM " << GetTableName()
        << " WHERE object_id = " << transaction->Quote(objectId)
        << " AND user_id = " << transaction->Quote(userId);
    TRecordsSet records;
    auto queryResult = transaction->Exec(query, &records);
    if (!queryResult || !queryResult->IsSucceed()) {
        return {};
    }
    TMap<TString, ui64> counts;
    for (auto&& record : records) {
        const TString& constructorId = record.Get("constructor_id");
        if (!constructorId) {
            continue;
        }
        counts[constructorId] += 1;
    }
    ui64 result = 0;
    for (auto&&[constructorId, count] : counts) {
        result = std::max(result, count);
    }
    return NThreading::MakeFuture(result);
}

TDBOffersStorage::TDBOffersStorage(const NDrive::IServer& server, const TString& storageName)
    : IAutoActualization(GetTableName())
    , Server(server)
    , StorageName(storageName)
    , Database(Server.GetDatabase(storageName))
    , Table(Database->GetTable(GetTableName()))
{
    CHECK_WITH_LOG(Table) << GetTableName();
    SetPeriod(TDuration::Minutes(1));
}

NThreading::TFuture<IOffersStorage::TOffers> TDBOffersStorage::StoreOffers(const TOffers& offers, NDrive::TInfoEntitySession& session) {
    if (offers.empty()) {
        return NThreading::MakeFuture(offers);
    }

    auto sessionGuard = BuildSessionGuard(session, false);
    TRecordsSet records;
    for (auto&& offerReport : offers) {
        auto offer = offerReport->GetOffer();
        if (!offer || !offer->NeedStore(Server)) {
            continue;
        }
        auto data = NDrive::Compress(offer->SerializeToProto().SerializeAsString());
        NStorage::TTableRecord record;
        record.Set("data", *data);
        record.Set("deadline", offer->GetDeadline().Seconds());
        record.Set("id", offer->GetOfferId());
        record.Set("constructor_id", offer->GetPriceConstructorId());
        if (auto vehicleOffer = offerReport->GetOfferPtrAs<IOffer>(); vehicleOffer && vehicleOffer->GetObjectId()) {
            record.Set("object_id", vehicleOffer->GetObjectId());
        }
        record.Set("user_id", offer->GetUserId());
        records.AddRow(std::move(record));
    }

    auto insertion = Table->AddRows(records, sessionGuard.GetTransaction());
    if (!insertion || !insertion->IsSucceed()) {
        session.SetErrorInfo("DBStoreOffers", "AddRows failure", EDriveSessionResult::TransactionProblem);
        return {};
    }
    if (!sessionGuard.Commit()) {
        session.MergeErrorMessages(sessionGuard.GetSession().GetMessages(), "DBStoreOffers");
        return {};
    }
    return NThreading::MakeFuture(offers);
}

NThreading::TFuture<bool> TDBOffersStorage::RemoveOffer(const TString& id, const TString& userId, NDrive::TInfoEntitySession& session) {
    auto sessionGuard = BuildSessionGuard(session, false);

    auto condition = static_cast<NStorage::TTableRecord>(NStorage::TRecordBuilder("id", id)("user_id", userId));
    auto reading = Table->RemoveRow(condition, sessionGuard.GetTransaction());
    if (!reading || !reading->IsSucceed()) {
        session.SetErrorInfo("OffersManager::RemoveOffer", "RemoveRow failure", EDriveSessionResult::TransactionProblem);
        return {};
    }
    if (!sessionGuard.Commit()) {
        session.MergeErrorMessages(sessionGuard.GetSession().GetMessages(), "DBStoreOffers");
        return {};
    }
    return NThreading::MakeFuture(true);
}

NThreading::TFuture<IOffersStorage::TOffer> TDBOffersStorage::RestoreOffer(const TString& id, const TString& userId, NDrive::TInfoEntitySession& session) {
    auto sessionGuard = BuildSessionGuard(session, true);

    TRecordsSet records;
    NStorage::TTableRecord condition = NStorage::TRecordBuilder("id", id);
    if (userId) {
        condition.Set("user_id", userId);
    }
    auto reading = Table->GetRows(condition, records, sessionGuard.GetTransaction());
    if (!reading || !reading->IsSucceed()) {
        session.SetErrorInfo("OffersManager::RestoreOffer", "GetRows failure", EDriveSessionResult::TransactionProblem);
        return {};
    }
    if (records.empty()) {
        session.SetError(NDrive::MakeError("offer_not_found"));
        session.SetErrorInfo("OffersManager::RestoreOffer", "offer_not_found", EDriveSessionResult::OfferNotFound);
        return {};
    }
    if (records.size() > 1) {
        session.SetErrorInfo("OffersManager::RestoreOffer", "encountered " + ToString(records.size()) + " offers", EDriveSessionResult::InconsistencyOffer);
        return {};
    }

    const NStorage::TTableRecord& record = records.GetRecords()[0];
    const TString& data = record.Get("data");
    auto decompressed = NDrive::Decompress(data);
    if (!decompressed) {
        session.SetErrorInfo("DBRestoreOffer", TStringBuilder() << "cannot_decompress: " << decompressed.GetError().AsStrBuf(), EDriveSessionResult::OfferCannotRead);
        return {};
    }
    auto offer = ICommonOffer::ConstructFromStringProto(*decompressed);
    if (!offer) {
        session.SetErrorInfo("DBRestoreOffer", "offer_cannot_read", EDriveSessionResult::OfferCannotRead);
        return {};
    }
    if (ModelingNow() > offer->GetDeadline()) {
        session.SetError(NDrive::MakeError("offer_expired"));
        session.SetErrorInfo("DBRestoreOffer", "offer_expired", EDriveSessionResult::OfferExpired);
        return {};
    }
    if (userId && offer->GetUserId() != userId) {
        session.SetErrorInfo("DBOffersStorage::RestoreOffer", "offer user_id mismatch: " + offer->GetUserId() + "/" + userId, EDriveSessionResult::InconsistencyOffer);
        return {};
    }

    return NThreading::MakeFuture(std::move(offer));
}

NThreading::TFuture<IOffersStorage::TOffer> TDBOffersStorage::RestoreOffer(const TString& id, NDrive::TInfoEntitySession& session) {
    return RestoreOffer(id, TString{}, session);
}

bool TDBOffersStorage::Refresh() {
    NStorage::ITransaction::TPtr transaction = Database->CreateTransaction(true);
    auto now = ModelingNow();
    auto condition = TStringBuilder() << "deadline < " << now.Seconds();
    auto removal = Checked(Table)->RemoveRow(condition, transaction);
    if (!removal || !removal->IsSucceed()) {
        ERROR_LOG << "cannot remove old offers:" << transaction->GetErrors().GetStringReport() << Endl;
        return false;
    }
    if (!transaction->Commit()) {
        ERROR_LOG << "cannot commit transaction: " << transaction->GetErrors().GetStringReport() << Endl;
        return false;
    }
    INFO_LOG << "removed " << removal->GetAffectedRows() << " dead offers" << Endl;
    return true;
}

TDBOffersStorage::TExternalSessionGuard TDBOffersStorage::BuildSessionGuard(NDrive::TInfoEntitySession& extSession, const bool readOnly) const {
    TDatabaseSessionConstructor dsc(Server.GetDatabase(StorageName));
    return TExternalSessionGuard(extSession, dsc, readOnly);
}

static NRTLine::TAction SerializeOfferToSaasAction(const ICommonOffer::TPtr offer) {
    NRTLine::TAction result;
    NRTLine::TDocument& d = result.AddDocument();
    d.SetUrl(offer->GetUserId() + "/states/" + offer->GetOfferId());
    d.SetMimeType("text/html");
    d.SetDeadline(std::max(Now(), offer->GetDeadline()) + TDuration::Minutes(30));
    d.AddProperty("data_proto", Base64Encode(offer->SerializeToProto().SerializeAsString()));
    d.AddProperty("constructor_id", offer->GetBehaviourConstructorId());
    d.AddProperty("visual_constructor_id", offer->GetBehaviourConstructorId());
    d.AddProperty("price_constructor_id", offer->GetPriceConstructorId());
    d.AddProperty("user_id", offer->GetUserId());
    if (offer->GetOfferId()) {
        d.AddSpecialKey("offer_id", offer->GetOfferId());
    }
    if (auto vehicleOffer = std::dynamic_pointer_cast<IOffer>(offer); vehicleOffer && vehicleOffer->GetObjectId()) {
        d.AddSpecialKey("object_id", vehicleOffer->GetObjectId());
    }
    if (offer->GetUserId()) {
        d.AddSpecialKey("user_id", offer->GetUserId());
    }
    return result;
}

NThreading::TFuture<IOffersStorage::TOffers> TRTYOfferStorage::StoreOffers(const TOffers& offers, NDrive::TInfoEntitySession& /*session*/) {
    const TRTLineAPI* config = Server.GetRTLineAPI(StorageName);
    TVector<NThreading::TFuture<IOfferReport::TPtr>> result;
    for (auto&& i : offers) {
        if (!i) {
            continue;
        }
        if (!!i && !Yensured(i->GetOffer())->NeedStore(Server)) {
            result.push_back(NThreading::MakeFuture(i));
            continue;
        }
        auto future = config->GetIndexingClient()->Send(SerializeOfferToSaasAction(i->GetOffer()), NRTLine::TSendParams().SetRealtime());
        future.Subscribe([i](const auto& f) {
            if (f.HasValue() && f.GetValue().IsSucceeded()) {
                SignalOfferStoreSuccess.Signal(1);
            } else {
                SignalOfferStoreProblems.Signal(1);
                NJson::TJsonValue data;
                data["offer_id"] = (i && i->GetOffer()) ? i->GetOffer()->GetOfferId() : "unknown_offer_id";
                if (f.HasValue()) {
                    data["code"] = f.GetValue().GetHttpCode();
                    data["message"] = f.GetValue().GetMessage();
                } else {
                    data["exception"] = NThreading::GetExceptionInfo(f);
                }
                NDrive::TEventLog::Log("StoreOfferFailure", data);
            }
        });
        auto asyncOffer = future.Apply([i](const NThreading::TFuture<NRTLine::TSendResult>& f) {
            Y_ENSURE(i);
            Y_ENSURE(f.GetValue().IsSucceeded(), "cannot index offer " << i->GetOffer()->GetOfferId());
            return i;
        });
        result.push_back(asyncOffer);
    }

    return NThreading::Merge(std::move(result), /*ignoreExceptions=*/true);
}

NThreading::TFuture<ui64> TRTYOfferStorage::GetOffersCount(const TString& objectId, const TString& userId, NDrive::TInfoEntitySession& /*session*/) {
    const TRTLineAPI* config = Server.GetRTLineAPI(StorageName);
    NRTLine::TQuery query;
    query.AddExtraParam("key_name", "user_id");
    query.AddProperty("constructor_id");
    query.AddProperty("object_id");
    query.SetText(userId);
    auto asyncReply = config->GetSearchClient().SendAsyncQueryF(query);
    auto result = asyncReply.Apply([objectId](const NThreading::TFuture<NRTLine::TSearchReply>& asyncReply) {
        const NRTLine::TSearchReply& reply = asyncReply.GetValue();
        Y_ENSURE(reply.IsSucceeded(), reply.GetCode() << ' ' << reply.GetReqId());
        TMap<TString, ui64> counts;
        reply.ScanDocs([&counts, &objectId](const NMetaProtocol::TDocument& document) {
            TString currentConstructorId;
            TString currentObjectId;
            for (auto&& i : document.GetArchiveInfo().GetGtaRelatedAttribute()) {
                const TString& key = i.GetKey();
                const TString& value = i.GetValue();
                if (key == "constructor_id") {
                    currentConstructorId = value;
                    continue;
                }
                if (key == "object_id") {
                    currentObjectId = value;
                    continue;
                }
            }
            if (currentObjectId == objectId) {
                counts[currentConstructorId] += 1;
            }
        });
        ui64 result = 0;
        for (auto&&[constructorId, count] : counts) {
            result = std::max(result, count);
        }
        return result;
    });
    return result;
}

NThreading::TFuture<IOffersStorage::TOffer> TRTYOfferStorage::RestoreOffer(const TString& id, const TString& userId, NDrive::TInfoEntitySession& /*session*/) {
    const TRTLineAPI* config = Server.GetRTLineAPI(StorageName);
    if (!config) {
        return NThreading::TExceptionFuture() << "cannot find API " << StorageName;
    }

    NRTLine::TQuery query;
    query.SetText(userId + "/states/" + id);
    query.AddExtraParam("meta_search", "first_found");
    auto reply = config->GetSearchClient().SendAsyncQueryF(query);
    return RestoreOffer(reply, userId);
}

NThreading::TFuture<IOffersStorage::TOffer> TRTYOfferStorage::RestoreOffer(const TString& id, NDrive::TInfoEntitySession& /*session*/) {
    const TRTLineAPI* config = Server.GetRTLineAPI(StorageName);
    if (!config) {
        return NThreading::TExceptionFuture() << "cannot find API " << StorageName;
    }

    NRTLine::TQuery query;
    query.SetText(id);
    query.AddExtraParam("meta_search", "first_found");
    query.AddExtraParam("key_name", "offer_id");
    auto reply = config->GetSearchClient().SendAsyncQueryF(query);
    return RestoreOffer(reply, TString{});
}

NThreading::TFuture<IOffersStorage::TOffer> TRTYOfferStorage::RestoreOffer(const NThreading::TFuture<NRTLine::TSearchReply>& reply, const TString& userId) const {
    auto offer = reply.Apply([userId](const NThreading::TFuture<NRTLine::TSearchReply>& r) {
        NDrive::TInfoEntitySession session;
        const auto& reply = r.GetValue();
        if (!reply.IsSucceeded()) {
            session.SetErrorInfo("RestoreOffer", TStringBuilder() << reply.GetCode() << ' ' << reply.GetReqId());
            session.Check();
        }
        const auto& report = reply.GetReport();
        const auto groupingSize = report.GroupingSize();
        const auto groupSize = groupingSize ? report.GetGrouping(0).GroupSize() : 0;
        const auto documentSize = groupSize ? report.GetGrouping(0).GetGroup(0).DocumentSize() : 0;
        if (groupingSize != 1 || groupSize != 1 || documentSize != 1) {
            session.SetError(NDrive::MakeError("offer_not_found"));
            session.SetErrorInfo(
                "RestoreOffer",
                TStringBuilder() << "incorrect response: " << groupingSize << ':' << groupSize << ':' << documentSize,
                EDriveSessionResult::OfferNotFound
            );
            session.Check();
        }

        TReadSearchProtoHelper helper(reply.GetReport().GetGrouping(0).GetGroup(0).GetDocument(0));
        TString value;
        if (!helper.GetProperty("data_proto", value)) {
            session.SetErrorInfo("RestoreOffer", "field data_proto is missing", EDriveSessionResult::OfferCannotRead);
            session.Check();
        }
        auto offer = ICommonOffer::ConstructFromBase64Proto(value);
        if (!offer) {
            session.SetErrorInfo("RestoreOffer", "cannot construct offer", EDriveSessionResult::OfferCannotRead);
            session.Check();
        }
        if (offer->GetDeadline() < Now()) {
            session.SetError(NDrive::MakeError("offer_expired"));
            session.SetErrorInfo("RestoreOffer", "offer expired: " + offer->GetDeadline().ToString(), EDriveSessionResult::OfferExpired);
            session.Check();
        }
        if (userId && offer->GetUserId() != userId) {
            session.SetErrorInfo("RestoreOffer", "offer user_id mismatch", EDriveSessionResult::InconsistencyOffer);
            session.Check();
        }
        return offer;
    });
    return offer;
}

NThreading::TFuture<bool> TRTYOfferStorage::RemoveOffer(const TString& id, const TString& userId, NDrive::TInfoEntitySession& /*session*/) {
    const TRTLineAPI* config = Server.GetRTLineAPI(StorageName);
    NRTLine::TAction action;
    action.SetActionType(NRTLine::TAction::atDelete);
    action.GetDocument().SetUrl(userId + "/states/" + id);
    NRTLine::TSendParams sParams;
    sParams.SetInstantReply().SetRealtime();
    if (!config->GetIndexingClient()) {
        return NThreading::TExceptionFuture() << "no indexing client";
    }
    return config->GetIndexingClient()->Send(action, sParams).Apply([](const NThreading::TFuture<NRTLine::TSendResult>& sr) {
        return sr.GetValue().IsSucceeded();
    });
}

bool IOffersStorage::StoreOffers(const TOffers& offers, TOffers* storedOffers, NDrive::TInfoEntitySession* session) {
    NDrive::TInfoEntitySession localSession;
    if (!session) {
        session = &localSession;
    }
    auto result = StoreOffers(offers, *session);
    if (!result.Initialized()) {
        return false;
    }
    result.Wait();
    if (result.HasValue()) {
        if (storedOffers) {
            *storedOffers = result.ExtractValue();
        }
        return true;
    } else {
        if (session) {
            session->SetErrorInfo("IOffersStorage::StoreOffers", NThreading::GetExceptionMessage(result), EDriveSessionResult::InternalError);
        }
        return false;
    }
}

NThreading::TFuture<IOffersStorage::TOffers> TFakeOfferStorage::StoreOffers(const TOffers& /*offers*/, NDrive::TInfoEntitySession& /*session*/) {
    return NThreading::TExceptionFuture<yexception>() << "unimplemented";
}

NThreading::TFuture<ui64> TFakeOfferStorage::GetOffersCount(const TString& /*objectId*/, const TString& /*userId*/, NDrive::TInfoEntitySession& /*session*/) {
    ui64 result = 0;
    return NThreading::MakeFuture(result);
}
