#pragma once

#include "abstract.h"

#include <drive/backend/offers/actions/offer_price.h>

#include <drive/backend/database/transaction/tx.h>

class IOffersStorage {
public:
    using TOffer = ICommonOffer::TPtr;
    using TOffers = TVector<IOfferReport::TPtr>;

public:
    virtual ~IOffersStorage() = default;

    bool StoreOffers(const TOffers& offers, TOffers* storedOffers = nullptr, NDrive::TInfoEntitySession* session = nullptr);

    virtual NThreading::TFuture<ui64> GetOffersCount(const TString& objectId, const TString& userId, NDrive::TInfoEntitySession& session) = 0;
    virtual NThreading::TFuture<bool> RemoveOffer(const TString& id, const TString& userId, NDrive::TInfoEntitySession& session) = 0;
    virtual NThreading::TFuture<TOffer> RestoreOffer(const TString& id, const TString& userId, NDrive::TInfoEntitySession& session) = 0;
    virtual NThreading::TFuture<TOffer> RestoreOffer(const TString& id, NDrive::TInfoEntitySession& session) = 0;
    virtual NThreading::TFuture<TOffers> StoreOffers(const TOffers& offers, NDrive::TInfoEntitySession& session) = 0;
};

class TFakeOfferStorage: public IOffersStorage {
public:
    virtual NThreading::TFuture<bool> RemoveOffer(const TString& /*id*/, const TString& /*userId*/, NDrive::TInfoEntitySession& /*session*/) override {
        return NThreading::MakeFuture(true);
    }

    virtual NThreading::TFuture<TOffer> RestoreOffer(const TString& /*id*/, const TString& /*userId*/, NDrive::TInfoEntitySession& /*session*/) override {
        return NThreading::MakeFuture<TOffer>(nullptr);
    }

    virtual NThreading::TFuture<TOffer> RestoreOffer(const TString& /*id*/, NDrive::TInfoEntitySession& /*session*/) override {
        return NThreading::MakeFuture<TOffer>(nullptr);
    }

    virtual NThreading::TFuture<ui64> GetOffersCount(const TString& /*objectId*/, const TString& /*userId*/, NDrive::TInfoEntitySession& /*session*/) override;
    virtual NThreading::TFuture<TOffers> StoreOffers(const TOffers& /*offers*/, NDrive::TInfoEntitySession& session) override;
};

class TRTYOfferStorage: public IOffersStorage {
private:
    const NDrive::IServer& Server;
    const TString StorageName;

public:
    TRTYOfferStorage(const NDrive::IServer& server, const TString& storageName)
        : Server(server)
        , StorageName(storageName)
    {
        CHECK_WITH_LOG(server.GetRTLineAPI(StorageName));
    }

    virtual NThreading::TFuture<ui64> GetOffersCount(const TString& objectId, const TString& userId, NDrive::TInfoEntitySession& session) override;
    virtual NThreading::TFuture<bool> RemoveOffer(const TString& id, const TString& userId, NDrive::TInfoEntitySession& session) override;
    virtual NThreading::TFuture<TOffer> RestoreOffer(const TString& id, const TString& userId, NDrive::TInfoEntitySession& session) override;
    virtual NThreading::TFuture<TOffer> RestoreOffer(const TString& id, NDrive::TInfoEntitySession& session) override;
    virtual NThreading::TFuture<TOffers> StoreOffers(const TOffers& offers, NDrive::TInfoEntitySession& session) override;

private:
    NThreading::TFuture<TOffer> RestoreOffer(const NThreading::TFuture<NRTLine::TSearchReply>& reply, const TString& userId) const;
};

class TDBOffersStorage: public IOffersStorage, public IAutoActualization {
private:
    const NDrive::IServer& Server;
    const TString StorageName;

protected:
    virtual bool Refresh() override;

protected:
    class TExternalSessionGuard;

protected:
    TExternalSessionGuard BuildSessionGuard(NDrive::TInfoEntitySession& extSession, const bool readOnly) const;

public:
    static TString GetTableName() {
        return "drive_offers";
    }

    TDBOffersStorage(const NDrive::IServer& server, const TString& storageName);
    ~TDBOffersStorage() = default;

    virtual NThreading::TFuture<ui64> GetOffersCount(const TString& objectId, const TString& userId, NDrive::TInfoEntitySession& session) override;
    virtual NThreading::TFuture<bool> RemoveOffer(const TString& id, const TString& userId, NDrive::TInfoEntitySession& session) override;
    virtual NThreading::TFuture<TOffer> RestoreOffer(const TString& id, const TString& userId, NDrive::TInfoEntitySession& session) override;
    virtual NThreading::TFuture<TOffer> RestoreOffer(const TString& id, NDrive::TInfoEntitySession& session) override;
    virtual NThreading::TFuture<TOffers> StoreOffers(const TOffers& offers, NDrive::TInfoEntitySession& session) override;

private:
    NStorage::IDatabase::TPtr Database;
    NStorage::ITableAccessor::TPtr Table;
};
