#include "notifications.h"

#include "context.h"

namespace NDrive {

    TOfferNotification::TOfferNotification(const TString& name)
        : Name(name)
    {
    }

    const TString& TOfferNotification::GetName() const {
        return Name;
    }

    bool TOfferNotification::IsEnabled(const TOffersBuildingContext& ctx) const {
        if (!ctx.GetSetting<bool>("notification." + Name + ".enabled").GetOrElse(false)) {
            return false;
        }
        TString modelsStringFilter = ctx.GetSetting<TString>("notification." + Name + ".cars_filter").GetOrElse({});
        if (!modelsStringFilter) {
            return true;
        }
        auto taggedCar = ctx.GetTaggedCar();
        if (taggedCar) {
            auto tagsFilter = TTagsFilter();
            if (!tagsFilter.DeserializeFromString(modelsStringFilter)) {
                return true;
            }
            if(tagsFilter.IsMatching(*taggedCar)) {
                return true;
            }
        }
        return false;
    }

    TSet<TString> TOfferNotification::GetExcludes(const TOffersBuildingContext& ctx) const {
        TSet<TString> excludes;
        auto setting = ctx.GetSetting<TString>("notification." + Name + ".excludes").GetOrElse("");
        StringSplitter(setting).Split(',').SkipEmpty().AddTo(&excludes);
        return excludes;
    }

    NJson::TJsonValue TOfferNotification::GetReport(const TOffersBuildingContext& ctx) const {
        const ILocalization* localization = ctx.GetServer()->GetLocalization();
        auto locale = ctx.HasUserHistoryContext() ? ctx.GetUserHistoryContextRef().GetLocale() : DefaultLocale;
        auto notification = NJson::ToJson(NJson::JsonString(
            ctx.GetSetting<TString>("notification." + Name).GetOrElse("{}")
        ));
        notification["message"] = GetMessage(notification);
        localization->ApplyResourcesForJson(notification, locale);
        return notification;
    }

    TString TOfferNotification::GetMessage(const NJson::TJsonValue& cfg) const {
        return cfg["message"].GetString();
    }

    TSurgeOfferNotification::TSurgeOfferNotification(const TString& surgeType)
        : Name("surge." + surgeType + ".notification")
    {
    }

    const TString& TSurgeOfferNotification::GetName() const {
        return Name;
    }

    bool TSurgeOfferNotification::IsEnabled(const TOffersBuildingContext& ctx) const {
        Y_UNUSED(ctx);
        return true;
    }

    TSet<TString> TSurgeOfferNotification::GetExcludes(const TOffersBuildingContext& ctx) const {
        Y_UNUSED(ctx);
        return {};
    }

    NJson::TJsonValue TSurgeOfferNotification::GetReport(const TOffersBuildingContext& ctx) const {
        const ILocalization* localization = ctx.GetServer()->GetLocalization();
        auto locale = ctx.HasUserHistoryContext() ? ctx.GetUserHistoryContextRef().GetLocale() : DefaultLocale;
        auto notification = NJson::ToJson(NJson::JsonString(
            ctx.GetSetting<TString>(Name).GetOrElse("{}")
        ));
        localization->ApplyResourcesForJson(notification, locale);
        return notification;
    }

    TPriceModelOfferNotification::TPriceModelOfferNotification(const TString& name, float value)
        : TBase(name)
        , Value(value)
    {
    }

    TString TPriceModelOfferNotification::GetMessage(const NJson::TJsonValue& cfg) const {
        TString message = TBase::GetMessage(cfg);
        SubstGlobal(message, "<features.percent>", Sprintf("%d", int(Value * 100)));
        return message;
    }

    void TOfferNotifications::AddNotification(THolder<IOfferNotification>&& notification) {
        auto name = notification->GetName();
        Notifications[name] = TAtomicSharedPtr<IOfferNotification>(notification.Release());
    }

    NJson::TJsonValue TOfferNotifications::GetReport(const TOffersBuildingContext& ctx) const {
        TSet<TString> allExcludes;
        for (auto&& [name, notification] : Notifications) {
            if (!notification->IsEnabled(ctx)) {
                continue;
            }
            auto excludes = notification->GetExcludes(ctx);
            allExcludes.insert(excludes.begin(), excludes.end());
        }
        NJson::TJsonValue result = NJson::JSON_ARRAY;
        for (auto&& [name, notification] : Notifications) {
            if (!notification->IsEnabled(ctx)) {
                continue;
            }
            if (allExcludes.contains(name)) {
                continue;
            }
            auto report = notification->GetReport(ctx);
            report["type"] = name;
            result.AppendValue(report);
        }
        return result;
    }

}
