#pragma once

#include <library/cpp/json/json_value.h>
#include <util/generic/map.h>
#include <util/generic/string.h>
#include <util/generic/ptr.h>

class TOffersBuildingContext;

namespace NDrive {

    class IOfferNotification {
    public:
        virtual ~IOfferNotification() = default;

        // GetName should return name of notification that shoud be sent to client.
        //
        // You can not send two notifiations with same name, so notification will be overwritten.
        virtual const TString& GetName() const = 0;

        // IsEnabled should return flag that this notification can be shown to user.
        virtual bool IsEnabled(const TOffersBuildingContext& ctx) const = 0;

        // GetExcludes returns set of notifications that should be excluded from result.
        virtual TSet<TString> GetExcludes(const TOffersBuildingContext& ctx) const = 0;

        // GetReport should build user client json report for notification.
        virtual NJson::TJsonValue GetReport(const TOffersBuildingContext& ctx) const = 0;
    };

    class TOfferNotification : public IOfferNotification {
    private:
        TString Name;

    public:
        TOfferNotification(const TString& name);

        const TString& GetName() const override;

        bool IsEnabled(const TOffersBuildingContext& ctx) const override;

        TSet<TString> GetExcludes(const TOffersBuildingContext& ctx) const override;

        NJson::TJsonValue GetReport(const TOffersBuildingContext& ctx) const override;

        virtual TString GetMessage(const NJson::TJsonValue& cfg) const;
    };

    class TSurgeOfferNotification : public IOfferNotification {
    private:
        TString Name;

    public:
        TSurgeOfferNotification(const TString& surgeType);

        const TString& GetName() const override;

        bool IsEnabled(const TOffersBuildingContext& ctx) const override;

        TSet<TString> GetExcludes(const TOffersBuildingContext& ctx) const override;

        NJson::TJsonValue GetReport(const TOffersBuildingContext& ctx) const override;
    };

    class TPriceModelOfferNotification : public TOfferNotification {
        using TBase = TOfferNotification;

    private:
        float Value;

    public:
        TPriceModelOfferNotification(const TString& name, float value); 

        TString GetMessage(const NJson::TJsonValue& cfg) const override;
    };

    class TOfferNotifications {
    private:
        TMap<TString, TAtomicSharedPtr<IOfferNotification>> Notifications;

    public:
        TOfferNotifications() = default;
        TOfferNotifications(const TOfferNotifications& other) = default;
        TOfferNotifications(TOfferNotifications&& other) = default;
        TOfferNotifications& operator=(const TOfferNotifications& other) = default;
        TOfferNotifications& operator=(TOfferNotifications&& other) = default;

        ~TOfferNotifications() = default;

        void AddNotification(THolder<IOfferNotification>&& notification);

        NJson::TJsonValue GetReport(const TOffersBuildingContext& ctx) const;
    };

}
