#include "abstract.h"
#include "library/cpp/json/writer/json_value.h"

#include <drive/backend/offers/context.h>
#include <drive/backend/offers/actions/abstract.h>
#include <drive/backend/offers/offers/standart.h>

#include <drive/backend/abstract/base.h>
#include <drive/backend/areas/areas.h>
#include <drive/backend/areas/drop_object_features.h>
#include <drive/backend/billing/manager.h>
#include <drive/backend/data/area_tags.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/models/storage.h>
#include <drive/backend/proto/offer.pb.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/roles/roles.h>
#include <drive/backend/sessions/manager/billing.h>

#include <drive/library/cpp/tex_builder/tex_builder.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/regex/pcre/regexp.h>

#include <rtline/api/action.h>
#include <rtline/library/json/adapters.h>
#include <rtline/library/unistat/cache.h>
#include <rtline/util/algorithm/iterator.h>

#include <util/generic/xrange.h>

NJson::TJsonValue TOfferVisual::GetReport() const {
    return SerializeToJson();
}

NDrive::TScheme TOfferVisual::GetScheme() {
    NDrive::TScheme point;
    point.Add<TFSString>("color", "Color in #XXXXXX format");
    point.Add<TFSNumeric>("X", "X coordinate").SetMax(1).SetPrecision(3);
    point.Add<TFSNumeric>("Y", "Y coordinate").SetMax(1).SetPrecision(3);

    NDrive::TScheme gradient;
    gradient.Add<TFSVariants>("style", "Gradient style").SetVariants({ "linear", "radial" });
    gradient.Add<TFSArray>("points", "Gradient points").SetElement(point);

    NDrive::TScheme result;
    result.Add<TFSString>("top_color", "Top color of the gradient").SetRequired(false);
    result.Add<TFSString>("bottom_color", "Bottom color of the gradient").SetRequired(false);
    result.Add<TFSString>("offer_visual_type", "Класс визуализации");
    result.Add<TFSArray>("gradients", "Gradients").SetElement(gradient);
    return result;
}

bool TOfferVisual::ReadFromSettings(const ISettings& settings, const TString& path) {
    TMaybe<TString> topColor = settings.GetValue<TString>(path + ".top_color");
    if (topColor) {
        TopColor = *topColor;
    }
    TMaybe<TString> bottomColor = settings.GetValue<TString>(path + ".bottom_color");
    if (bottomColor) {
        BottomColor = *bottomColor;
    }
    return bottomColor || topColor;
}

bool TOfferVisual::DeserializeFromJson(const NJson::TJsonValue& value) {
    JREAD_STRING_OPT(value, "offer_visual_type", OfferVisualType);
    JREAD_STRING_OPT(value, "top_color", TopColor);
    JREAD_STRING_OPT(value, "bottom_color", BottomColor);
    const auto& gradients = value["gradients"];
    if (gradients.IsDefined() && !gradients.IsArray()) {
        return false;
    }
    for (auto&& g : gradients.GetArray()) {
        TGradient gradient;
        const auto& style = g["style"];
        if (style.IsDefined() && !style.IsString()) {
            return false;
        }
        gradient.Style = style.GetString();
        const auto& points = g["points"];
        if (points.IsDefined() && !points.IsArray()) {
            return false;
        }
        for (auto&& p : points.GetArray()) {
            TPoint point;
            point.Color = p["color"].GetString();
            point.X = p["X"].GetDouble();
            point.Y = p["Y"].GetDouble();
            gradient.Points.push_back(std::move(point));
        }
        Gradients.push_back(std::move(gradient));
    }
    return true;
}

NJson::TJsonValue TOfferVisual::SerializeToJson() const {
    NJson::TJsonValue result;
    result["offer_visual_type"] = OfferVisualType;
    result["top_color"] = TopColor;
    result["bottom_color"] = BottomColor;
    for (auto&& gradient : Gradients) {
        NJson::TJsonValue g;
        if (gradient.Style) {
            g["style"] = gradient.Style;
        }
        for (auto&& point : gradient.Points) {
            NJson::TJsonValue p;
            if (point.Color) {
                p["color"] = point.Color;
            }
            p["X"] = point.X;
            p["Y"] = point.Y;
            g["points"].AppendValue(std::move(p));
        }
        result["gradients"].AppendValue(std::move(g));
    }
    return result;
}


IOffer::TOfferSessionPatchData::TOfferSessionPatchData(
      TDuration duration
    , TInstant startInstant
    , const TVector<TCompiledLocalEvent>& localEvents
    , const TSnapshotsDiff* optSnapshotDiff
    ) :
      Duration(duration)
    , StartInstant(startInstant)
    , LocalEvents(localEvents)
{
    if (optSnapshotDiff) {
        if (optSnapshotDiff->HasMileage()) {
            Mileage = optSnapshotDiff->OptionalMileage();
        }
        if (optSnapshotDiff->HasStartMileage()) {
            MileageStart = optSnapshotDiff->OptionalStartMileage();
        }
        if (optSnapshotDiff->HasLastMileage()) {
            MileageFinish = optSnapshotDiff->OptionalLastMileage();
        }
        if (optSnapshotDiff->HasStartFuelLevel()) {
            FuelLevelStart = optSnapshotDiff->OptionalStartFuelLevel();
        }
        if (optSnapshotDiff->HasLastFuelLevel()) {
            FuelLevelFinish = optSnapshotDiff->OptionalLastFuelLevel();
        }
    }
}

NDrive::NProto::TOfferAreaInfo IOffer::TAreaInfo::SerializeToProto() const {
    NDrive::NProto::TOfferAreaInfo result;
    result.SetAreaId(AreaId);
    result.SetFee(Fee);
    return result;
}

bool IOffer::TAreaInfo::DeserializeFromProto(const NDrive::NProto::TOfferAreaInfo& proto) {
    AreaId = proto.GetAreaId();
    Fee = proto.GetFee();
    return true;
}

bool IOffer::DeserializeFromProto(const NDrive::NProto::TOffer& info) {
    if (!TBase::DeserializeFromProto(info)) {
        return false;
    }
    if (info.HasOriginalRidingStart()) {
        TGeoCoord c;
        if (c.Deserialize(info.GetOriginalRidingStart())) {
            OriginalRidingStart = c;
        }
    }
    ObjectId = info.GetObjectId();
    ObjectModel = info.GetObjectModel();
    Marker = info.GetMarker();
    SharedSessionId = info.GetSharedSessionId();
    Switchable = info.GetSwitchable();
    Switcher = info.GetSwitcher();
    FromScanner = info.GetFromScanner();
    Origin = info.GetOrigin();
    ParentId = info.GetParentId();
    NextOfferId = info.GetNextOfferId();
    Transferable = info.GetTransferable();
    TransferredFrom = info.GetTransferredFrom();
    if (info.HasTransferType()) {
        TransferType = static_cast<ECarDelegationType>(info.GetTransferType());
    }
    if (info.AvailableStartAreaSize()) {
        AvailableStartArea = TVector<TGeoCoord>();
        for (auto&& i : info.GetAvailableStartArea()) {
            TGeoCoord gc;
            if (!gc.Deserialize(i)) {
                return false;
            }
            AvailableStartArea->emplace_back(std::move(gc));
        }
    }
    for (auto&& i : info.GetAreaInfos()) {
        TAreaInfo aInfo;
        if (!aInfo.DeserializeFromProto(i)) {
            return false;
        }
        AreaInfos.emplace_back(std::move(aInfo));
    }
    if (info.HasCarWaitingDuration()) {
        CarWaitingDuration = TDuration::Seconds(info.GetCarWaitingDuration());
    }
    return true;
}

NDrive::NProto::TOffer IOffer::SerializeToProto() const {
    NDrive::NProto::TOffer result = TBase::SerializeToProto();
    if (HasOriginalRidingStart()) {
        *result.MutableOriginalRidingStart() = GetOriginalRidingStartUnsafe().Serialize();
    }
    result.SetObjectId(ObjectId);
    result.SetObjectModel(ObjectModel);
    result.SetMarker(Marker);
    result.SetSwitchable(Switchable);
    result.SetSwitcher(Switcher);
    result.SetFromScanner(FromScanner);
    result.SetParentId(ParentId);
    result.SetNextOfferId(NextOfferId);
    result.SetTransferable(Transferable);
    result.SetTransferredFrom(TransferredFrom);
    if (Origin) {
        result.SetOrigin(Origin);
    }
    if (SharedSessionId) {
        result.SetSharedSessionId(SharedSessionId);
    }
    if (TransferType) {
        result.SetTransferType(static_cast<ui32>(*TransferType));
    }
    for (auto&& i : GetAreaInfos()) {
        *result.AddAreaInfos() = i.SerializeToProto();
    }
    if (AvailableStartArea) {
        for (auto&& i : *AvailableStartArea) {
            *result.AddAvailableStartArea() = i.SerializeLP();
        }
    }
    if (CarWaitingDuration) {
        result.SetCarWaitingDuration(CarWaitingDuration->Seconds());
    }
    return result;
}

NJson::TJsonValue IOffer::DoBuildJsonReport(const TReportOptions& options, const ICommonOfferBuilderAction* constructor, const NDrive::IServer& server) const {
    const auto locale = options.Locale;
    const auto traits = options.Traits;
    NJson::TJsonValue result = TBase::DoBuildJsonReport(options, constructor, server);
    if (!(options.Traits & NDriveSession::ReportOfferDetails)) {
        return result;
    }
    result.InsertValue("from_scanner", FromScanner);
    result.InsertValue("parent_id", ParentId);
    if (traits & NDriveSession::EReportTraits::ReportSwitchable) {
        result.InsertValue("switchable", GetReportedSwitchable());
    }
    if (CarWaitingDuration) {
        result.InsertValue("car_waiting_duration", CarWaitingDuration->Seconds());
        result.InsertValue("car_waiting_duration_hr", server.GetLocalization()->FormatDuration(locale, *CarWaitingDuration));
    }
    if (ObjectModel) {
        result.InsertValue("object_model", ObjectModel);
    }
    if (Origin) {
        result.InsertValue("origin", Origin);
    }
    if (Marker) {
        result.InsertValue("marker", Marker);
    }
    return result;
}

TOfferStatePtr IOffer::Calculate(const TVector<IEventsSession<TCarTagHistoryEvent>::TTimeEvent>& timeline, const TVector<TAtomicSharedPtr<TCarTagHistoryEvent>>& events, const TInstant& until, TOfferPricing& result) const {
    TRidingInfo ridingInfo(timeline, events, until);
    TOfferStatePtr resultState = DoCalculate(timeline, events, until, ridingInfo, result);
    if (timeline.size() && timeline.back().GetTimeEvent() == IEventsSession<TCarTagHistoryEvent>::EEvent::Tag && (!resultState || resultState->GetNeedFinishFees())) {
        const TChargableTag* chargableTag = events[timeline.back().GetEventIndex()]->GetTagAs<TChargableTag>();
        if (chargableTag && !ridingInfo.GetIsSwitch()) {
            ui32 fee = chargableTag->GetFinishOfferFee();
            if (fee) {
                if (chargableTag->GetFinishOfferFeePolicy() == EDropOfferPolicy::FeesFix) {
                    auto discountedFee = GetDiscountedPrice(fee);
                    result.AddSegmentInfo("fee_drop_zone_fix", discountedFee, fee);
                } else if (result.GetBillingSumOriginalPrice() + 1 <= fee) {
                    fee = fee - result.GetBillingSumOriginalPrice();
                    auto discountedFee = GetDiscountedPrice(fee);
                    result.AddSegmentInfo("fee_drop_zone_max", discountedFee, fee);
                }
            }
        }
    }
    return resultState;
}

EDriveSessionResult IOffer::CheckSession(const TUserPermissions& permissions, NDrive::TEntitySession& session, const NDrive::IServer* server, bool onPerform) const {
    if (permissions.GetUserId() != GetUserId()) {
        session.SetErrorInfo("inconsistency_user_id", "Incorrect user id for performing with offer", EDriveSessionResult::InconsistencyOffer);
        return EDriveSessionResult::InconsistencyOffer;
    }
    TUserPermissions::TConstPtr actualPermissions = permissions.Self();
    if (GetSharedSessionId()) {
        auto optionalSharedSession = server->GetDriveDatabase().GetSessionManager().GetSession(GetSharedSessionId(), session);
        if (!optionalSharedSession) {
            session.AddErrorMessage("Offer::CheckSession", "cannot get SharedSession");
            return EDriveSessionResult::InternalError;
        }
        auto sharedSession = *optionalSharedSession;
        if (!sharedSession) {
            session.SetErrorInfo("Offer::CheckSession", "null SharedSession");
            return EDriveSessionResult::InternalError;
        }
        if (sharedSession->GetClosed()) {
            session.SetErrorInfo("offer_check", "shared_session_is_closed", EDriveSessionResult::InconsistencyOffer);
            return EDriveSessionResult::InconsistencyOffer;
        }
        actualPermissions = server->GetDriveAPI()->GetUserPermissions(sharedSession->GetUserId());
    }
    return DoCheckSession(*actualPermissions, session, server, onPerform);
}

void IOffer::FillBill(TBill& bill, const TOfferPricing& pricing, TOfferStatePtr /*segmentState*/, ELocalization locale, const NDrive::IServer* server, ui32 cashbackPercent) const {
    for (auto&& i : pricing.GetPrices()) {
        if ((int)(i.second.GetOriginalPrice()) < 1e-5) {
            continue;
        }
        if (i.first != "fee_drop_zone_fix" && i.first != "fee_drop_zone_max") {
            continue;
        }
        TBillRecord billRecord;
        billRecord.SetCost(i.second.GetPrice()).SetType(i.first);
        const TString specTitle = server->GetSettings().GetValueDef<TString>("offers.localization.special." + i.first + ".title", "Сурдж-зона");
        if (!!specTitle) {
            billRecord.SetTitle(FormDescriptionElement(specTitle, locale, server->GetLocalization()));
        }
        const TString specDescription = server->GetSettings().GetValueDef<TString>("offers.localization.special." + i.first + ".description", "");
        if (!!specDescription) {
            billRecord.SetDetails(FormDescriptionElement(specDescription, locale, server->GetLocalization()));
        }
        bill.MutableRecords().emplace_back(std::move(billRecord));
    }
    bill.SetCashbackPercent(GetSelectedCharge() == "yandex_account" ? 0 : cashbackPercent);
}

void IOffer::PatchSessionReport(NJson::TJsonValue& /*result*/, NDriveSession::TReportTraits /*traits*/, ELocalization /*locale*/,
                                const NDrive::IServer& /*server*/, const TOfferSessionPatchData& /*patchData*/) const {
}

TRidingInfo::TRidingInfo(const TVector<IEventsSession<TCarTagHistoryEvent>::TTimeEvent>& timeline, const TVector<TAtomicSharedPtr<TCarTagHistoryEvent>>& events, const TInstant until) {
    if (timeline.empty()) {
        return;
    }
    if (timeline.front().GetEventInstant() > until) {
        return;
    }
    if (timeline.back().GetEventInstant() <= until) {
        IsFinished = timeline.back().GetTimeEvent() == IEventsSession<TCarTagHistoryEvent>::EEvent::Tag;
        if (IsFinished) {
            const TChargableTag* cTag = events[timeline.back().GetEventIndex()]->GetTagAs<TChargableTag>();
            Replacing = cTag && cTag->IsReplacing();
            IsSwitch = cTag && cTag->GetIsSwitching();
            if (cTag && cTag->HasDelegationType()) {
                DelegationType = cTag->GetDelegationTypeRef();
            }
            const THistoryDeviceSnapshot* dSnapshot = (*events[timeline.back().GetEventIndex()])->GetObjectSnapshotAs<THistoryDeviceSnapshot>();
            NDrive::TLocation location;
            if (dSnapshot && dSnapshot->GetHistoryLocation(location)) {
                FinishCoord = location.GetCoord();
            }
        }
    }
    TString currentSegment;
    TInstant currentInstant;
    for (auto&& i : timeline) {
        if (i.GetEventInstant() > until) {
            LastInstant = until;
            if (!!currentSegment) {
                DurationsBySegments[currentSegment] += LastInstant - currentInstant;
            }
            return;
        } else {
            LastInstant = i.GetEventInstant();
        }
        if (i.GetTimeEvent() == IEventsSession<TCarTagHistoryEvent>::EEvent::CurrentFinish) {
            LastInstant = i.GetEventInstant();
            if (currentSegment) {
                DurationsBySegments[currentSegment] += i.GetEventInstant() - currentInstant;
            }
        } else {
            auto& e = *events[i.GetEventIndex()];
            Events.emplace_back(i.GetEventInstant(), e->GetName());
            if (!StartsBySegments.contains(e->GetName())) {
                StartsBySegments.emplace(e->GetName(), i.GetEventInstant());
            }
            if (currentSegment) {
                DurationsBySegments[currentSegment] += i.GetEventInstant() - currentInstant;
            }
            currentSegment = e->GetName();
            currentInstant = i.GetEventInstant();
        }

    }
}

TString IOffer::BuildCalculationDescription(ELocalization locale, const TFullCompiledRiding& fcr, const NDrive::IServer& server) const {
    TStringBuilder sb;
    sb << "\\section{Итоговый счет}" << Endl;
    if (fcr.HasBill() && fcr.GetBillUnsafe().GetRecords().size()) {
        sb << "\\begin{enumerate}" << Endl;
        for (auto&& i : fcr.GetBillUnsafe().GetRecords()) {
            sb << "\\item " << i.GetTitle() << " : " << i.GetCost() << (i.GetDetails() ? " (" + NTexBuilder::TTextStyle::Quote(i.GetDetails()) + ")" : "") << Endl;
        }
        sb << "\\end{enumerate}" << Endl;
    } else {
        sb << "UNDEFINED" << Endl;
    }

    sb << "\\section{Информация по поездке}" << Endl;
    sb << DoBuildCommonCalculationDescription(locale, fcr, server) << Endl;

    sb << "\\section{Детали тарифа}" << Endl;
    sb << DoBuildCommonPricesDescription(fcr, server) << Endl;

    sb << "\\section{Поминутный расчет}" << Endl;
    sb << "(фактически тарификация посекундная)" << Endl;
    sb << DoBuildCalculationDescription(locale, fcr, server) << Endl;
    return sb;
}

IOffer::TPtr IOffer::Clone() const {
    return ConstructFromProto<IOffer>(SerializeToProto());
}

void IOffer::InitAreaInfos(const NDrive::IServer* server, const TOffersBuildingContext& context) {
    if (!context.GetStartPosition()) {
        return;
    }
    TCarLocationContext clc = TCarLocationContext::BuildByCoord(GetObjectId(), context.OptionalOriginalRidingStart().GetOrElse(*context.GetStartPosition()), *server);
    clc.SetGuaranteeFees(false);
    clc.SetTakeFeesFromOffer(false);
    TCarLocationFeatures clfStart = clc.GetCarAreaFeatures(false, this, server);
    if (context.IsDelegation()) {
        clfStart.SetAllowDrop(EDropAbility::Allow);
    }

    const auto actor = [this, server, &clfStart](const TDBTag& tag) -> bool {
        TMaybe<TOfferDropPolicy> dropPolicy = tag.GetTagAs<TDropAreaFeatures>()->BuildFeeBySession(this, server, tag.GetObjectId(), false);
        if (dropPolicy && dropPolicy->GetPolicy() == EDropOfferPolicy::FeesMinute) {
            return true;
        }
        TCarLocationFeatures clfFinish(dropPolicy);
        const TMaybe<ui32> deltaFees = clfFinish.GetDeltaFees(clfStart);
        if (deltaFees) {
            dropPolicy->SetFee(*deltaFees);
            AreaInfos.emplace_back(*dropPolicy, tag.GetObjectId());
        }
        return true;
    };
    server->GetDriveAPI()->GetAreasDB()->ProcessHardDBTags<TDropAreaFeatures>(actor);
}

IOffer::TAreaInfo::TAreaInfo(const TOfferDropPolicy& dropPolicy, const TString& areaId)
    : AreaId(areaId)
    , Fee(dropPolicy.GetFee())
{
}

namespace {
    const NUnistat::TIntervals FeeIntervals = MakeVector<double>(xrange(0, 100000, 5000));
}

TOfferPricing TOfferPricing::operator+(const TOfferPricing& item) const {
    TOfferPricing result(Offer);
    result.Prices = Prices;
    for (auto&& i : item.Prices) {
        auto it = result.Prices.find(i.first);
        if (it == result.Prices.end()) {
            result.Prices.emplace(i.first, i.second);
        } else {
            it->second = it->second + i.second;
        }
    }
    result.RefreshSumPrices();
    return result;
}

TOfferPricing TOfferPricing::operator-(const TOfferPricing& item) const {
    TOfferPricing result(Offer);
    result.Prices = Prices;
    for (auto&& i : item.Prices) {
        auto it = result.Prices.find(i.first);
        if (it == result.Prices.end()) {
            result.Prices.emplace(i.first, i.second);
        } else {
            it->second = it->second - i.second;
        }
    }
    result.RefreshSumPrices();
    return result;
}

void TOfferPricing::AddSegmentInfo(
        const TString& currentSegmentName,
        double segmentPrice, double originalSegmentPrice,
        TDuration duration,
        double distance,
        double deposit,
        double revenue,
        double cashback, const TMaybe<NJson::TJsonValue>& payload)
{
    auto it = Prices.find(currentSegmentName);
    TOfferSegment segment(segmentPrice, originalSegmentPrice, duration, distance, deposit, revenue, cashback, payload);
    if (it == Prices.end()) {
        Prices.emplace(currentSegmentName, segment);
    } else {
        it->second.Add(segment);
    }

    RefreshSumPrices();
}

void TOfferPricing::AddCashback(const TString& currentSegmentName, double cashback, const TMaybe<NJson::TJsonValue>& payload) {
    AddSegmentInfo(currentSegmentName, 0, 0, TDuration::Zero(), 0, 0, 0, cashback, payload);
}

bool TOfferPricing::Compare(const TOfferPricing& item, double precision) const {
    if (Abs(BillingSumPrice - item.BillingSumPrice) > precision) {
        return false;
    }
    if (Abs(BillingSumOriginalPrice - item.BillingSumOriginalPrice) > precision) {
        return false;
    }
    if (Abs(ReportSumPrice - item.ReportSumPrice) > precision) {
        return false;
    }
    if (Abs(ReportSumOriginalPrice - item.ReportSumOriginalPrice) > precision) {
        return false;
    }
    if (Abs(Deposit - item.Deposit) > precision) {
        return false;
    }
    if (Abs(Revenue - item.Revenue) > precision) {
        return false;
    }
    if (Abs(BillingRevenue - item.BillingRevenue) > precision) {
        return false;
    }
    if (Abs(Cashback - item.Cashback) > precision) {
        return false;
    }
    if (Prices.size() != item.Prices.size()) {
        return false;
    }
    for (auto&& i : Prices) {
        auto it = item.Prices.find(i.first);
        if (it == item.Prices.end()) {
            return false;
        }
        if (!it->second.Compare(i.second, precision)) {
            return false;
        }
    }
    return true;
}

NJson::TJsonValue TOfferPricing::GetReport() const {
    NJson::TJsonValue report;
    report.InsertValue("sum_price", ReportSumPrice);
    report.InsertValue("sum_price_original", ReportSumOriginalPrice);
    report.InsertValue("bsum_price", BillingSumPrice);
    report.InsertValue("bsum_price_original", BillingSumOriginalPrice);
    report.InsertValue("deposit", Deposit);
    report.InsertValue("revenue", Revenue);
    report.InsertValue("brevenue", BillingRevenue);
    report.InsertValue("cashback", Cashback);
    report.InsertValue("bashback", BillingCashback);
    auto& segments = report.InsertValue("segments", NJson::JSON_MAP);
    for (auto&& i : Prices) {
        segments.InsertValue(i.first, i.second.GetJsonReport());
    }
    return report;
}

NJson::TJsonValue TOfferPricing::GetPublicReport(const ILocalization& localization, ELocalization locale, const TString& currency) const {
    auto getTitle = [offer=Offer, &localization, locale](const TString& name) {
        TString title = localization.GetLocalString(locale, "offer_pricing.segment." + name + ".title", "");
        if (!offer) {
            return title;
        }
        return localization.GetLocalString(locale, "offer_pricing.segment." + name + ".title." + offer->GetTypeName(), title);
    };
    auto getRecord = [&localization, locale, &currency](const TString& name, const TString& title, double price) {
        NJson::TJsonValue record;
        record.InsertValue("type", name);
        record.InsertValue("title", title);
        record.InsertValue("price_hr", localization.FormatPrice(locale, TStandartOffer::RoundPrice(price), {currency}));
        record.InsertValue("price", TStandartOffer::RoundPrice(price));
        return record;
    };
    NJson::TJsonValue report;
    auto& records = report.InsertValue("records", NJson::JSON_ARRAY);
    for (auto&& [name, segment] : Prices) {
        if (auto&& price = segment.GetPrice()) {
            auto title = getTitle(name);
            if (!title) {
                continue;
            }
            SubstGlobal(title, "_Duration_", localization.FormatDuration(locale, segment.GetDuration(), /*withSeconds=*/true, /*allowEmpty=*/true, /*withMilliseconds=*/false));
            SubstGlobal(title, "_Distance_", localization.DistanceFormatKm(locale, segment.GetDistance()));
            if (Offer) {
                SubstGlobal(title, "_OfferName_", Offer->GetName());
            }
            records.AppendValue(getRecord(name, title, price));
        }
    }
    if (auto title = getTitle("total"); title && ReportSumPrice) {
        records.AppendValue(getRecord("total", title, ReportSumPrice));
    }
    return report;
}

void TOfferPricing::CleanOtherPrices(const TSet<TString>& excludeStages) {
    for (auto&& i : Prices) {
        if (excludeStages.contains(i.first)) {
            continue;
        }
        i.second.SetPrice(0);
        i.second.SetOriginalPrice(0);
        i.second.SetDeposit(0);
        i.second.SetRevenue(0);
        i.second.SetCashback(0);
    }
    RefreshSumPrices();
}

void TOfferPricing::MulPrices(double kff, const TSet<TString>& stages) {
    for (auto&& i : Prices) {
        if (stages.empty() || stages.contains(i.first)) {
            i.second.ChangePriceKff(kff);
        }
    }
    RefreshSumPrices();
}

void TOfferPricing::RefreshSumPrices() {
    ReportSumPrice = 0;
    ReportSumOriginalPrice = 0;
    BillingSumPrice = 0;
    BillingSumOriginalPrice = 0;
    Deposit = 0;
    Revenue = 0;
    Cashback = 0;
    if (Offer) {
        Deposit = Offer->GetDeposit();
    }
    for (auto&& i : Prices) {
        Deposit = Max<double>(Deposit, i.second.GetDeposit());
        ReportSumPrice += (i64)i.second.GetPrice();
        ReportSumOriginalPrice += (i64)i.second.GetOriginalPrice();
        Revenue += (i64)i.second.GetRevenue();
        Cashback += (i64)i.second.GetCashback();
    }
    BillingSumPrice = Min(ReportSumPrice, BorderSumPrice);
    BillingSumOriginalPrice = Min(ReportSumOriginalPrice, BorderSumOriginalPrice);
    BillingRevenue = Min(Revenue, BillingSumPrice);
    BillingCashback = Min(Cashback, BorderCashback);
    Y_ASSERT(ReportSumPrice >= 0);
    Y_ASSERT(ReportSumOriginalPrice >= 0);
    Y_ASSERT(BillingSumPrice >= 0);
    Y_ASSERT(BillingSumOriginalPrice >= 0);
    Y_ASSERT(Deposit >= 0);
    Y_ASSERT(Revenue >= 0);
    Y_ASSERT(Cashback >= 0);
}

void TOfferPricing::Remove(const TSet<TString>& cleanStages) {
    for (auto&& i : cleanStages) {
        Prices.erase(i);
    }
    RefreshSumPrices();
}

void TOfferPricing::SetBorderPrices(double borderSumPrice, double borderSumOriginalPrice) {
    BorderSumPrice = borderSumPrice;
    BorderSumOriginalPrice = borderSumOriginalPrice;
    RefreshSumPrices();
}

bool IOffer::GetServiceFuelingPossibility(const NDrive::IServer* server) const {
    auto serviceFuelingTariffTags = SplitString(server->GetSettings().GetValueDef<TString>("service.fueling.tariff_tags", ""), ",");

    auto action = server->GetDriveAPI()->GetRolesManager()->GetAction(GetBehaviourConstructorId());
    auto offerBuilder = action ? action->GetAs<IOfferBuilderAction>() : nullptr;
    if (offerBuilder) {
        const auto& tariffTags = offerBuilder->GetGrouppingTags();
        for (auto&& tag : serviceFuelingTariffTags) {
            if (tariffTags.contains(tag)) {
                return true;
            }
        }
    }

    return false;
}

template <>
NJson::TJsonValue NJson::ToJson(const IOffer& object) {
    NJson::TJsonValue result;
    result["object_id"] = object.GetObjectId();
    result["offer_id"] = object.GetOfferId();
    result["user_id"] = object.GetUserId();
    return result;
}

template <>
NJson::TJsonValue NJson::ToJson(const TOfferPricing& object) {
    return object.GetReport();
}
