#pragma once

#include "base.h"

#include <drive/backend/offers/price/price.h>

#include <drive/backend/abstract/base.h>
#include <drive/backend/actions/abstract/action.h>
#include <drive/backend/database/history/session.h>
#include <drive/backend/sessions/common/report.h>
#include <drive/backend/surge/surge_snapshot.h>

#include <library/cpp/json/writer/json_value.h>

#include <rtline/util/json_processing.h>

#include <util/generic/ymath.h>
#include <util/string/subst.h>

class TBill;
class TUserPermissions;
class TFullCompiledRiding;
class TCompiledLocalEvent;

enum class ECarDelegationType: ui32;

namespace NDrive {
    class TEntitySession;
    class IOfferModel;
    class IServer;
    struct TOfferFeatures;

    class ISchedule {
    public:
        using TPtr = TAtomicSharedPtr<ISchedule>;

        virtual ~ISchedule() = default;

        // Contains checks that time is in schedule.
        virtual bool Contains(TInstant time) const = 0;

        // GetNext returns next available time that strictly greater.
        //
        // Returns Zero() if there is no available times.
        virtual TInstant GetNext(TInstant time) const = 0;
    };
};

namespace NDrive::NProto {
    class TOffer;
    class TOfferAreaInfo;
    class TOfferVisual;
}

class TRidingInfo {
private:
    class TLocalEvent {
        R_FIELD(TInstant, Start, TInstant::Zero());
        R_FIELD(TString, Name);

    public:
        TLocalEvent(const TInstant start, const TString& name)
            : Start(start)
            , Name(name)
        {
        }
    };

private:
    TMap<TString, TDuration> DurationsBySegments;
    TMap<TString, TInstant> StartsBySegments;
    TVector<TLocalEvent> Events;

public:
    R_READONLY(TInstant, LastInstant, TInstant::Zero());
    R_READONLY(bool, IsFinished, false);
    R_READONLY(bool, IsSwitch, false);
    R_READONLY(bool, Replacing, false);
    R_OPTIONAL(TGeoCoord, FinishCoord);
    R_OPTIONAL(ECarDelegationType, DelegationType);

public:
    bool IsEmpty() const {
        return Events.empty();
    }

    bool HasSegment(const TString& segmentName) const {
        return StartsBySegments.contains(segmentName);
    }

    TMaybe<TInstant> GetSegmentStart(const TString& segmentName) const {
        auto it = StartsBySegments.find(segmentName);
        if (it == StartsBySegments.end()) {
            return {};
        }
        return it->second;
    }
    TDuration GetSegmentDuration(const TString& segmentName) const {
        auto it = DurationsBySegments.find(segmentName);
        if (it == DurationsBySegments.end()) {
            return TDuration::Zero();
        }
        return it->second;
    }

    bool GetInstantOnSegmentsDuration(const TSet<TString>& segmentNames, const TDuration duration, TInstant& result) const {
        if (Events.empty()) {
            return false;
        }
        TDuration durationReady = TDuration::Zero();
        TInstant current = Events.front().GetStart();
        TString currentSegment = Events.front().GetName();
        for (auto&& i : Events) {
            if (segmentNames.contains(currentSegment)) {
                durationReady += i.GetStart() - current;
            }
            if (durationReady > duration) {
                result = (i.GetStart() - durationReady) + duration;
                return true;
            }
            current = i.GetStart();
            currentSegment = i.GetName();
        }
        if (segmentNames.contains(currentSegment)) {
            durationReady += LastInstant - current;
        }
        result = (LastInstant - durationReady) + duration;
        return true;
    }

    TRidingInfo(const TVector<IEventsSession<TCarTagHistoryEvent>::TTimeEvent>& timeline, const TVector<TAtomicSharedPtr<TCarTagHistoryEvent>>& events, const TInstant until);
};

class TOfferSegment {
    R_FIELD(double, Price, 0);
    R_FIELD(double, OriginalPrice, 0);
    R_FIELD(double, Revenue, 0);
    R_READONLY(double, Distance, 0);
    R_READONLY(TDuration, Duration, TDuration::Zero());
    R_FIELD(double, Deposit, 0);
    R_FIELD(double, Cashback, 0);
    R_OPTIONAL(NJson::TJsonValue, Payload);

public:
    void ChangePriceKff(const double kf) {
        Price = Price * kf;
        OriginalPrice = OriginalPrice * kf;
        Revenue = Revenue * kf;
        Cashback = Cashback * kf;
    }

    bool Compare(const TOfferSegment& item, const double precision) const {
        if (Abs(Price - item.Price) > precision) {
            return false;
        }
        if (Abs(OriginalPrice - item.OriginalPrice) > precision) {
            return false;
        }
        if (Abs(Distance - item.Distance) > precision) {
            return false;
        }
        if (Duration != item.Duration) {
            return false;
        }
        if (Abs(Deposit - item.Deposit) > precision) {
            return false;
        }
        if (Abs(Revenue - item.Revenue) > precision) {
            return false;
        }
        if (Abs(Cashback - item.Cashback) > precision) {
            return false;
        }
        return true;
    }

    TOfferSegment(const double price, const double originalPrice, const TDuration duration, double distance, double deposit, double revenue, double cashback, const TMaybe<NJson::TJsonValue>& payload)
        : Price(price)
        , OriginalPrice(originalPrice)
        , Revenue(revenue)
        , Distance(distance)
        , Duration(duration)
        , Deposit(deposit)
        , Cashback(cashback)
        , Payload(payload)
    {
        Y_ASSERT(price >= 0);
        Y_ASSERT(originalPrice >= 0);
        Y_ASSERT(revenue >= 0);
        Y_ASSERT(distance >= 0);
        Y_ASSERT(deposit >= 0);
        Y_ASSERT(cashback >= 0);
    }

    void Add(const TOfferSegment& item) {
        Price += item.Price;
        OriginalPrice += item.OriginalPrice;
        Revenue += item.Revenue;
        Distance += item.Distance;
        Duration += item.Duration;
        Deposit = Max<double>(Deposit, item.Deposit);
        Cashback += item.Cashback;
        if (item.Payload) {
            Y_ASSERT(!Payload || *item.Payload == *Payload);
            Payload = item.Payload;
        }
    }

    TOfferSegment operator+(const TOfferSegment& item) const {
        if (item.Payload) {
            Y_ASSERT(!Payload || *item.Payload == *Payload);
        }
        return TOfferSegment(
            Price + item.Price,
            OriginalPrice + item.OriginalPrice,
            Duration + item.Duration,
            Distance + item.Distance,
            Max<double>(Deposit, item.Deposit),
            Revenue + item.Revenue,
            Cashback + item.Cashback,
            item.Payload ? item.Payload : Payload
        );
    }

    TOfferSegment operator-(const TOfferSegment& item) const {
        if (item.Payload) {
            Y_ASSERT(!Payload || *item.Payload == *Payload);
        }
        return TOfferSegment(
            Price - item.Price,
            OriginalPrice - item.OriginalPrice,
            Duration - item.Duration,
            std::max(Distance - item.Distance, 0.0),
            Deposit,
            Revenue - item.Revenue,
            Cashback - item.Cashback,
            Payload
        );
    }

    NJson::TJsonValue GetJsonReport() const {
        NJson::TJsonValue report(NJson::JSON_MAP);
        report.InsertValue("price", (int)Price);
        report.InsertValue("original_price", (int)OriginalPrice);
        report.InsertValue("revenue", (int)Revenue);
        if (Distance > 0) {
            report.InsertValue("distance", Distance);
        }
        if (Duration) {
            report.InsertValue("duration", Duration.Seconds());
        }
        report.InsertValue("deposit", (int)Deposit);
        if (Cashback) {
            report.InsertValue("cashback", (int)Cashback);
        }
        if (Payload) {
            report.InsertValue("payload", *Payload);
        }
        return report;
    }
};

class TOfferVisual {
public:
    struct TPoint {
        TString Color;
        float X = 0;
        float Y = 0;
    };
    struct TGradient {
        TVector<TPoint> Points;
        TString Style;
    };

private:
    R_FIELD(TString, TopColor);
    R_FIELD(TString, BottomColor);
    R_FIELD(TVector<TGradient>, Gradients, {});
    R_FIELD(TString, OfferVisualType);

public:
    static NDrive::TScheme GetScheme();

public:
    NJson::TJsonValue GetReport() const;

    bool ReadFromSettings(const ISettings& settings, const TString& path);
    bool DeserializeFromJson(const NJson::TJsonValue& value);

    NJson::TJsonValue SerializeToJson() const;
};

class TOfferPricing;
class TOffersBuildingContext;
class TSnapshotsDiff;

namespace NRTLine {
    class TAction;
}
class TOfferDropPolicy;
using TCarTagHistoryEvent = TObjectEvent<TConstDBTag>;

class IOffer: public ICommonOffer {
public:
    class TAreaInfo {
    private:
        R_FIELD(TString, AreaId);
        R_FIELD(ui32, Fee, 0);
    public:
        TAreaInfo() = default;
        TAreaInfo(const TOfferDropPolicy& dropPolicy, const TString& areaId);

        NDrive::NProto::TOfferAreaInfo SerializeToProto() const;
        bool DeserializeFromProto(const NDrive::NProto::TOfferAreaInfo& proto);
    };

    struct TOfferSessionPatchData {
    public:
        TOfferSessionPatchData(
              TDuration duration
            , TInstant startInstant
            , const TVector<TCompiledLocalEvent>& localEvents
            , const TSnapshotsDiff* optSnapshotDiff);

    public:
        TDuration Duration;
        TInstant StartInstant;
        const TVector<TCompiledLocalEvent>& LocalEvents;
        TMaybe<double> Mileage;
        TMaybe<double> MileageStart;
        TMaybe<double> MileageFinish;
        TMaybe<ui8> FuelLevelStart;
        TMaybe<ui8> FuelLevelFinish;
    };

    struct TEmailOfferData {
        TMap<TString, TString> Args;
        NJson::TJsonValue JsonData;
        TString EmailTagName;
    };

    enum class EEvolutionType {
        Evolve,
        Switch,
    };
    using TBase = ICommonOffer;

public:
    R_FIELD(TString, ObjectId);
    R_FIELD(TString, ObjectModel);
    R_FIELD(TString, Marker);
    R_FIELD(TString, Origin);
    R_FIELD(TString, SharedSessionId);
    R_FIELD(bool, FromScanner, false);
    R_OPTIONAL(TDuration, CarWaitingDuration);
    R_OPTIONAL(TVector<TGeoCoord>, AvailableStartArea);
    R_FIELD(TVector<TAreaInfo>, AreaInfos);
    R_OPTIONAL(TGeoCoord, OriginalRidingStart);
    R_FIELD(bool, Switchable, false);
    R_FIELD(bool, Switcher, false);
    R_FIELD(TString, ParentId);
    R_FIELD(TString, NextOfferId);
    R_FIELD(bool, Transferable, true);
    R_FIELD(TString, TransferredFrom);
    R_OPTIONAL(ECarDelegationType, TransferType);

public:
    bool DeserializeFromProto(const NDrive::NProto::TOffer& info) override;
    NDrive::NProto::TOffer SerializeToProto() const override;

private:
    virtual EDriveSessionResult DoCheckSession(const TUserPermissions& permissions, NDrive::TEntitySession& session, const NDrive::IServer* server, bool onPerform) const {
        Y_UNUSED(permissions);
        Y_UNUSED(session);
        Y_UNUSED(server);
        Y_UNUSED(onPerform);
        return EDriveSessionResult::Success;
    }

protected:
    virtual NJson::TJsonValue DoBuildJsonReport(const TReportOptions& options, const ICommonOfferBuilderAction* constructor, const NDrive::IServer& server) const override;

    virtual TOfferStatePtr DoCalculate(const TVector<IEventsSession<TCarTagHistoryEvent>::TTimeEvent>& /*timeline*/, const TVector<TAtomicSharedPtr<TCarTagHistoryEvent>>& /*events*/, const TInstant& /*until*/, const TRidingInfo& /*ridingInfo*/, TOfferPricing& /*result*/) const {
        return nullptr;
    }

    virtual TString DoBuildCalculationDescription(ELocalization /*locale*/, const TFullCompiledRiding& /*fcr*/, const NDrive::IServer& /*server*/) const {
        return "Not implemented";
    };

    virtual TString DoBuildCommonCalculationDescription(ELocalization /*locale*/, const TFullCompiledRiding& /*fcr*/, const NDrive::IServer& /*server*/) const {
        return "Not implemented";
    };

    virtual TString DoBuildCommonPricesDescription(const TFullCompiledRiding& /*fcr*/, const NDrive::IServer& /*server*/) const {
        return "Not implemented";
    };

public:
    using TPtr = TAtomicSharedPtr<IOffer>;
    using TConstPtr = TAtomicSharedPtr<const IOffer>;

public:
    auto GetConstructionId() const {
        return std::tie(
            GetObjectId(),
            GetBehaviourConstructorId(),
            GetName()
        );
    }

    virtual bool CheckNeedFeesOnFinish(const NDrive::IServer* /*server*/) const {
        return true;
    }
    TString BuildCalculationDescription(ELocalization locale, const TFullCompiledRiding& fcr, const NDrive::IServer& server) const;

    TMaybe<TAreaInfo> GetOfferAreaInfo(const TString& areaId) const {
        for (auto&& i : AreaInfos) {
            if (i.GetAreaId() == areaId) {
                return i;
            }
        }
        return {};
    }

    virtual bool GetServiceFuelingPossibility(const NDrive::IServer* server) const;

    IOffer::TPtr Clone() const;

    virtual void InitAreaInfos(const NDrive::IServer* server, const TOffersBuildingContext& context);

    virtual TDuration CalcChargableDuration(const TInstant predInstant, const TInstant nextInstant, const TString& /*tagName*/, const TSet<TString>& /*tagsInPoint*/) const {
        return nextInstant - predInstant;
    }

    virtual TOfferStatePtr Calculate(const TVector<IEventsSession<TCarTagHistoryEvent>::TTimeEvent>& timeline, const TVector<TAtomicSharedPtr<TCarTagHistoryEvent>>& events, const TInstant& until, TOfferPricing& result) const;
    virtual EDriveSessionResult CheckSession(const TUserPermissions& permissions, NDrive::TEntitySession& session, const NDrive::IServer* server, bool onPerform) const final;
    virtual void FillBill(TBill& bill, const TOfferPricing& pricing, TOfferStatePtr /*segmentState*/, ELocalization locale, const NDrive::IServer* server, ui32 cashbackPercent) const;
    virtual void PatchSessionReport(NJson::TJsonValue& result, NDriveSession::TReportTraits traits, ELocalization locale,
                                    const NDrive::IServer& server, const TOfferSessionPatchData& patchData) const;

    virtual double CalculateOriginalPrice(const TDuration stateDuration, const TString& tagName) const {
        Y_UNUSED(stateDuration);
        Y_UNUSED(tagName);
        return 0;
    }

    virtual TDuration GetFreeTime(const TDuration usedDuration, const TString& tagName) const {
        Y_UNUSED(usedDuration);
        Y_UNUSED(tagName);
        return TDuration::Zero();
    }

    virtual TString GetPushReport(ELocalization /*locale*/, const IServerBase* /*server*/) const {
        return "";
    }

    virtual bool GetFreeAndPricedDurations(TDuration& freeDuration, TDuration& pricedDuration, const TMap<TString, TOfferSegment>& segments) const {
        Y_UNUSED(freeDuration);
        Y_UNUSED(pricedDuration);
        Y_UNUSED(segments);
        return false;
    }

    virtual bool GetReportedSwitchable() const {
        return Switchable;
    }

    virtual bool HasFueling() const {
        return true;
    }

    virtual TMaybe<TEmailOfferData> GetEmailData(const NDrive::IServer* /*server*/, NDrive::TEntitySession& /*session*/, const TString& /*notificationType*/) const {
        return TEmailOfferData();
    }
};

class TOfferPricing {
public:
    using TPrices = TMap<TString, TOfferSegment>;

public:
    R_FIELD(TPrices, Prices);
    R_READONLY(TAtomicSharedPtr<IOffer>, Offer, nullptr);
    R_READONLY(double, ReportSumPrice, 0);
    R_READONLY(double, ReportSumOriginalPrice, 0);
    R_READONLY(double, BillingSumPrice, 0);
    R_READONLY(double, BillingSumOriginalPrice, 0);
    R_READONLY(double, Deposit, 0);
    R_READONLY(double, Revenue, 0);
    R_READONLY(double, BillingRevenue, 0);
    R_READONLY(double, Cashback, 0);
    R_READONLY(double, BillingCashback, 0);
    R_FIELD(double, BorderCashback, Max<double>());

private:
    double BorderSumPrice = Max<double>();
    double BorderSumOriginalPrice = Max<double>();

public:
    void CleanPrices() {
        CleanOtherPrices({});
    }

    const TOfferSegment* GetPrices(const TString& segmentName) const {
        auto it = Prices.find(segmentName);
        if (it == Prices.end()) {
            return nullptr;
        }
        return &it->second;
    }

    i32 GetDistance() const {
        i32 result = 0;
        for (auto&& [_, segment] : Prices) {
            result += segment.GetDistance();
        }
        return result;
    }

    TOfferPricing(IOffer::TPtr offer)
        : Offer(offer)
    {
    }

    TOfferPricing operator+(const TOfferPricing& item) const;
    TOfferPricing operator-(const TOfferPricing& item) const;

    bool Compare(const TOfferPricing& item, double precision = 1e-5) const;

    NJson::TJsonValue GetReport() const;

    NJson::TJsonValue GetPublicReport(const ILocalization& localization, ELocalization locale, const TString& currency) const;

    void AddSegmentInfo(const TString& currentSegmentName, double segmentPrice, double originalSegmentPrice, TDuration duration = TDuration::Zero(), double distance = 0, double deposit = 0, double revenue = 0, double cashback = 0, const TMaybe<NJson::TJsonValue>& payload = {});
    void AddCashback(const TString& currentSegmentName, double cashback, const TMaybe<NJson::TJsonValue>& payload);
    void MulPrices(double kff, const TSet<TString>& stages = {});
    void SetBorderPrices(double borderSumPrice, double borderSumOriginalPrice);
    void CleanOtherPrices(const TSet<TString>& excludeStages);
    void Remove(const TSet<TString>& cleanStages);

private:
    void RefreshSumPrices();
};

