#include "additional_service.h"

#include <drive/backend/data/chargable.h>
#include <drive/backend/data/delivery.h>
#include <drive/backend/offers/actions/additional_service.h>
#include <drive/backend/proto/offer.pb.h>
#include <library/cpp/json/writer/json_value.h>

TString TAdditionalServiceOfferState::FormDescriptionElement(const TString& value, const TString& currency, ELocalization locale, const ILocalization& localization) const {
    auto result = IOfferState::FormDescriptionElement(value, currency, locale, localization);
    SubstGlobal(result, "_DeliveryLocation_", DeliveryLocationName);
    TDuration timeShift = TDuration::Hours(3);
    if (ScheduledAt) {
        SubstGlobal(result, "_ScheduledAt_", localization.FormatInstant(locale, *ScheduledAt + timeShift));
        SubstGlobal(result, "_ScheduledAtDate_", localization.FormatMonthDay(locale, *ScheduledAt + timeShift));
    }
    return result;
}

NJson::TJsonValue TAdditionalServiceOfferState::GetReport(ELocalization locale, const NDrive::IServer& server) const {
    NJson::TJsonValue report = NJson::JSON_MAP;
    auto localization = server.GetLocalization();
    if (!Stage) {
        return report;
    }
    NJson::InsertField(report, "stage", *Stage);
    if (!localization) {
        return report;
    }
    auto formStageElement = [
        this,
        stage=*Stage,
        locale,
        localization
    ](const TString& text) -> TString {
        auto localized = localization->GetLocalString(locale, text + "." + stage, "");
        return FormDescriptionElement(localized, "", locale, *localization);
    };
    auto formElement = [
        this,
        stage=*Stage,
        locale,
        localization
    ](const TString& text) -> TString {
        auto localized = localization->GetLocalString(locale, text + "." + stage, localization->GetLocalString(locale, text, text));
        return FormDescriptionElement(localized, "", locale, *localization);
    };
    if (auto stageText = formStageElement(TitleKey)) {	
        report["title"] = stageText;	
    }
    if (auto stageText = formStageElement(SubtitleKey)) {	
        report["subtitle"] = stageText;	
    }
    if (SamePlace) {
        if (auto stageText = formStageElement(SubtitleKey + ".same_location")) {	
            report["subtitle"] = stageText;	
        }
    }
    if (auto stageText = formStageElement(ButtonKey)) {	
        report["order_button_text"] = stageText;	
    }
    bool hasAvailableTimes = false;
    if (AvailableSchedule) {
        NJson::TJsonValue availableTimes = NJson::JSON_ARRAY;
        TInstant time = AvailableSchedule->GetNext(Now());
        int limit = server.GetSettings().GetValue<int>("offers.additional_service_offer.daily_schedule_limit").GetOrElse(365);
        for (int i = 0; i < limit && time != TInstant::Zero(); ++i) {
            NJson::TJsonValue availableTime;
            availableTime.InsertValue("ts", time.Seconds());
            availableTimes.AppendValue(std::move(availableTime));
            time = AvailableSchedule->GetNext(time);
            hasAvailableTimes = true;
        }
        report.InsertValue("available_times", std::move(availableTimes));
    }
    if (Stage && *Stage == TAdditionalServiceOfferHolderTag::Started && !AvailableDeliveryLocation) {
        report["order_button_text"] = NJson::JSON_NULL;
    }
    NJson::TJsonValue orderFields = NJson::JSON_ARRAY;
    if (hasAvailableTimes) {
        NJson::TJsonValue scheduled;
        scheduled["type"] = "scheduled_at";
        scheduled["title"] = formElement(ScheduledAtTitleKey);
        scheduled["subtitle"] = formElement(ScheduledAtSubtitleKey);
        orderFields.AppendValue(std::move(scheduled));
    }
    if (AvailableDeliveryLocation) {
        NJson::TJsonValue delivery;
        delivery["type"] = "delivery_location";
        delivery["title"] = formElement(DeliveryLocationTitleKey);
        delivery["same_location"] = formElement(SameLocationTitleKey);
        orderFields.AppendValue(std::move(delivery));
    }
    {
        NJson::TJsonValue details;
        details["type"] = "details";
        details["title"] = formElement(DetailsTitleKey);
        details["subtitle"] = formElement(DetailsSubtitleKey);
        orderFields.AppendValue(std::move(details));
    }
    report.InsertValue("order_fields", std::move(orderFields));
    return report;
}

bool TAdditionalServiceOffer::DeserializeFromProto(const NDrive::NProto::TOffer& offerProto) {
    if (!TBase::DeserializeFromProto(offerProto)) {
        return false;
    }
    auto &proto = offerProto.GetAdditionalServiceOffer();
    Price = proto.GetPrice();
    ScheduledAt = TInstant::MicroSeconds(proto.GetScheduledAt());
    SessionId = proto.GetSessionId();
    if (proto.HasDeliveryLocation()) {
        DeliveryLocation.ConstructInPlace();
        if (!DeliveryLocation->Deserialize(proto.GetDeliveryLocation())) {
            return false;
        }
    }
    DeliveryLocationName = proto.GetDeliveryLocationName();
    SameLocation = proto.GetSameLocation();
    return true;
}

NDrive::NProto::TOffer TAdditionalServiceOffer::SerializeToProto() const {
    auto offerProto = TBase::SerializeToProto();
    auto &proto = *offerProto.MutableAdditionalServiceOffer();
    proto.SetPrice(GetPrice());
    proto.SetScheduledAt(GetScheduledAt().MicroSeconds());
    proto.SetSessionId(GetSessionId());
    if (DeliveryLocation) {
        *(proto.MutableDeliveryLocation()) = DeliveryLocation->Serialize();
    }
    if (DeliveryLocationName) {
        proto.SetDeliveryLocationName(DeliveryLocationName);
    }
    proto.SetSameLocation(SameLocation);
    return offerProto;
}

NJson::TJsonValue TAdditionalServiceOffer::DoBuildJsonReport(const TReportOptions& options, const ICommonOfferBuilderAction* constructor, const NDrive::IServer& server) const {
    auto report = TBase::DoBuildJsonReport(options, constructor, server);
    if (!(options.Traits & NDriveSession::ReportOfferDetails)) {
        return report;
    }
    report.InsertValue("price", GetPrice());
    report.InsertValue("scheduled_at", GetScheduledAt().Seconds());
    auto locale = options.Locale;
    auto localization = server.GetLocalization();
    auto getLocalString = [this, locale, &localization](const TString& text) {
        return localization ? DoFormDescriptionElement(localization->GetLocalString(locale, text), locale, localization) : text;
    };
    if (auto serviceOfferBuilder = dynamic_cast<const TAdditionalServiceOfferBuilder*>(constructor)) {
        report.InsertValue("title", getLocalString(serviceOfferBuilder->GetTitle()));
        report.InsertValue("subtitle", getLocalString(serviceOfferBuilder->GetSubtitle()));
        report.InsertValue("order_button_text", getLocalString(serviceOfferBuilder->GetOrderButtonText()));
        report.InsertValue("icon", serviceOfferBuilder->GetIcon());
    }
    
    NJson::InsertNonNull(report, "delivery_location", DeliveryLocation);
    NJson::InsertNonNull(report, "delivery_location_name", DeliveryLocationName);
    NJson::InsertNonNull(report, "delivery_area", DeliveryArea);
    NJson::InsertField(report, "same_location", SameLocation);
    NJson::InsertNonNull(report, "parent_session_id", SessionId);
    return report;
}

TString TAdditionalServiceOffer::DoFormDescriptionElement(const TString& value, ELocalization locale, const ILocalization* localization) const {
    auto result = TBase::DoFormDescriptionElement(value, locale, localization);
    SubstGlobal(result, "_OfferPrice_", GetLocalizedPrice(GetPrice(), locale, *localization));
    SubstGlobal(result, "_DeliveryLocation_", DeliveryLocationName);
    TDuration timeShift = TDuration::Hours(3);
    if (ScheduledAt) {
        SubstGlobal(result, "_ScheduledAt_", localization->FormatInstant(locale, ScheduledAt + timeShift));
        SubstGlobal(result, "_ScheduledAtDate_", localization->FormatMonthDay(locale, ScheduledAt + timeShift));
    }
    return result;
}

TOfferStatePtr TAdditionalServiceOffer::Calculate(const TVector<IEventsSession<TCarTagHistoryEvent>::TTimeEvent>& timeline, const TVector<TAtomicSharedPtr<TCarTagHistoryEvent>>& events, const TInstant& until) const {
    if (timeline.empty()) {
        return nullptr;
    }
    i32 lastTagEventIndex = -1;
    for (auto&& element : timeline) {
        if (element.GetTimeEvent() != IEventsSession<TCarTagHistoryEvent>::EEvent::Tag) {
            continue;
        }
        if (element.GetEventInstant() > until) {
            break;
        }
        lastTagEventIndex = element.GetEventIndex();
    }
    if (lastTagEventIndex < 0) {
        return nullptr;
    }
    const auto& event = events[lastTagEventIndex];
    const auto* tag = event->GetTagAs<TAdditionalServiceOfferHolderTag>();
    if (!tag) {
        return nullptr;
    }
    auto state = MakeHolder<TAdditionalServiceOfferState>();
    state->SetDeliveryLocationName(GetDeliveryLocationName());
    state->SetSamePlace(GetSameLocation());
    state->SetScheduledAt(GetScheduledAt());

    auto server = NDrive::HasServer() ? &NDrive::GetServerAs<NDrive::IServer>() : nullptr;
    const auto actionBuilder = server ? server->GetDriveAPI()->GetRolesManager()->GetAction(GetBehaviourConstructorId()) : Nothing();
    const TAdditionalServiceOfferBuilder* constructor = nullptr;
    if (actionBuilder && actionBuilder->GetAs<TAdditionalServiceOfferBuilder>()) {
        constructor = actionBuilder->GetAs<TAdditionalServiceOfferBuilder>();
    }
    auto&& tagName = tag->GetName();
    if (constructor) {
        state->SetTitleKey(constructor->GetTitle());
        state->SetSubtitleKey(constructor->GetSubtitle());
        state->SetButtonKey(constructor->GetOrderButtonText());
        state->SetScheduledAtTitleKey(constructor->GetScheduledAtTitle());
        state->SetScheduledAtSubtitleKey(constructor->GetScheduledAtSubtitle());
        state->SetDeliveryLocationTitleKey(constructor->GetDeliveryLocationTitle());
        state->SetSameLocationTitleKey(constructor->GetSameLocationTitle());
        state->SetDetailsTitleKey(constructor->GetDetailsTitle());
        state->SetDetailsSubtitleKey(constructor->GetDetailsSubtitle());
        if (tagName == TAdditionalServiceOfferHolderTag::Planned) {
            auto billingSession = server ? GetBillingSession(*server) : nullptr;
            auto availableSchedule = constructor->GetAvailableSchedule(billingSession ? billingSession->GetCurrentOffer() : nullptr);
            state->SetAvailableSchedule(availableSchedule);
            state->SetAvailableDeliveryLocation(true);
        } else if (tagName == TAdditionalServiceOfferHolderTag::Starting) {
            state->SetAvailableDeliveryLocation(true);
        } else if (tagName == TAdditionalServiceOfferHolderTag::Started) {
            if (server) {
                const auto& deviceTagManager = server->GetDriveDatabase().GetTagsManager().GetDeviceTags();
                auto billingSession = GetBillingSession(*server);
                auto tx = deviceTagManager.BuildTx<NSQL::ReadOnly>();
                auto deviceTags = GetSessionDeviceTags(tx, billingSession, *server);
                Y_ENSURE(deviceTags);
                for (auto&& tag : *deviceTags) {
                    if (auto deliveryTag = tag.GetTagAs<TCarDeliveryTag>()) {
                        state->SetAvailableDeliveryLocation(!deliveryTag->GetPerformer());
                    }
                }
            }
        }
    }

    state->SetStage(tagName);

    return state.Release();
}

TAtomicSharedPtr<TBillingSession> TAdditionalServiceOffer::GetBillingSession(const NDrive::IServer& server) const {
    auto builder = server.GetDriveDatabase().GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing");
    if (!builder){
        return {};
    }
    return std::dynamic_pointer_cast<TBillingSession>(builder->GetSession(GetSessionId()));
}

TOptionalDBTags TAdditionalServiceOffer::GetSessionDeviceTags(NDrive::TEntitySession& tx, TAtomicSharedPtr<TBillingSession> session, const NDrive::IServer& server) const {
    if (!session) {
        return TDBTags{};
    }
    auto lastEvent = session->GetLastEvent();
    if (!lastEvent) {
        return TDBTags{};
    }
    auto chargable = lastEvent->GetTagAs<TChargableTag>();
    if (!chargable) {
        return TDBTags{};
    }
    auto subtagIds = chargable->GetSubtagIds();
    const auto& deviceTagManager = server.GetDriveDatabase().GetTagsManager().GetDeviceTags();
    auto optionalObject = deviceTagManager.RestoreObject(session->GetObjectId(), tx);
    if (!optionalObject) {
        return {};
    }
    auto deviceTags = optionalObject->GetTags();
    auto it = std::remove_if(deviceTags.begin(), deviceTags.end(), [&subtagIds](const TDBTag& tag) {
        return !subtagIds.contains(tag.GetTagId());
    });
    deviceTags.erase(it, deviceTags.end());
    return deviceTags;
}

TAdditionalServiceOffer::TFactory::TRegistrator<TAdditionalServiceOffer> TAdditionalServiceOffer::Registrator(TAdditionalServiceOffer::GetTypeNameStatic());
