#include "base.h"
#include <drive/backend/billing/manager.h>
#include <drive/backend/offers/actions/abstract.h>
#include <drive/backend/proto/offer.pb.h>

TSet<TString> ICommonOffer::NonCorporateMethods({"card", "mobile_payment", "yandex_account"});

void ICommonOffer::OnBooking(const TString& deviceId, const TString& mobilePaymethodId) {
    if (deviceId) {
        SetDeviceIdAccept(deviceId);
    }
    if (mobilePaymethodId) {
        SetMobilePaymethodId(mobilePaymethodId);
    }
}

bool ICommonOffer::PatchOffer(const NJson::TJsonValue& /*value*/, NDrive::TEntitySession& /*session*/, const NDrive::IServer* /*server*/, bool /*checkRidingConflict*/) {
    return false;
}

ui64 ICommonOffer::GetOfferRevision() const {
    return 0;
}

bool ICommonOffer::DeserializeFromProto(const NDrive::NProto::TOffer &info) {
    OfferId = info.GetOfferId();

    Timestamp = TInstant::Seconds(info.GetTimestamp());
    Deadline = TInstant::Seconds(info.GetDeadline());

    Name = info.GetName();
    GroupName = info.GetGroupName();
    ShortName = info.GetShortName();
    SubName = info.GetSubName();

    if (info.HasBehaviourConstructorId()) {
        BehaviourConstructorId = info.GetBehaviourConstructorId();
        PriceConstructorId = info.GetPriceConstructorId();
    } else {
        BehaviourConstructorId = info.GetConstructorId();
        PriceConstructorId = info.GetConstructorId();
    }
    if (info.HasLocale() && !TryFromString(info.GetLocale(), Locale)) {
        return false;
    }

    UserId = info.GetUserId();
    ExternalUserId = info.GetExternalUserId();

    Correctors = MakeVector(info.GetCorrector());
    DisabledCorrectors = MakeSet(info.GetDisabledCorrector());

    if (info.HasIsPlusUser()) {
        IsPlusUser = info.GetIsPlusUser();
    }
    ChargableAccounts.clear();
    for (auto&& i : info.GetChargableAccounts()) {
        ChargableAccounts.emplace_back(i);
    }
    SelectedCharge = info.GetSelectedCharge();
    SelectedCreditCard = info.GetSelectedCreditCard();
    MobilePaymethodId = info.GetMobilePaymethodId();
    YandexAccountBalance = info.GetYandexAccountBalance();
    PaymentDiscretization = info.GetPaymentDiscretization();

    CashbackPercent = info.GetCashbackPercent();
    NJson::TJsonValue cashbackInfo;
    if (info.HasCashbackInfo() && NJson::ReadJsonTree(TStringBuf(info.GetCashbackInfo()), &cashbackInfo)) {
        CashbackInfo = cashbackInfo;
    }
    if (info.HasIncreasedCashbackPercent()) {
        IncreasedCashbackPercent = info.GetIncreasedCashbackPercent();
    }
    HiddenCashback = info.GetHiddenCashback();
    OnlyOriginalOfferCashback = info.GetOnlyOriginalOfferCashback();

    if (info.HasOfferValidUntil()) {
        OfferValidUntil = TInstant::Seconds(info.GetOfferValidUntil());
    }
    if (info.HasOfferValidSince()) {
        OfferValidSince = TInstant::Seconds(info.GetOfferValidSince());
    }

    Bookable = info.GetBookable();

    TargetHolderTag = info.GetTargetHolderTag();
    for (auto&& i : info.GetTargetUserTag()) {
        TargetUserTags.insert(i);
    }

    DeviceIdAccept = info.GetDeviceIdAccept();

    for (auto&& i : info.GetProfitableTags()) {
        ProfitableTags.emplace(i);
    }
    for (auto&& i : info.GetAlreadyUsedProfitableTags()) {
        AlreadyUsedProfitableTags.emplace(i);
    }

    return true;
}

NDrive::NProto::TOffer ICommonOffer::SerializeToProto() const {
    NDrive::NProto::TOffer info;
    info.SetOfferId(OfferId);

    info.SetTimestamp(Timestamp.Seconds());
    info.SetDeadline(Deadline.Seconds());

    info.SetName(Name);
    info.SetGroupName(GroupName);
    info.SetShortName(ShortName);
    info.SetSubName(SubName);

    info.SetConstructorId(BehaviourConstructorId);
    info.SetBehaviourConstructorId(BehaviourConstructorId);
    info.SetPriceConstructorId(PriceConstructorId);

    info.SetUserId(UserId);
    if (ExternalUserId) {
        info.SetExternalUserId(ExternalUserId);
    }
    info.SetLocale(ToString(Locale));

    for (auto&& i : Correctors) {
        info.AddCorrector(i);
    }
    for (auto&& i : DisabledCorrectors) {
        info.AddDisabledCorrector(i);
    }

    if (IsPlusUser) {
        info.SetIsPlusUser(*IsPlusUser);
    }
    for (auto&& i : ChargableAccounts) {
        info.AddChargableAccounts(i);
    }
    info.SetSelectedCharge(SelectedCharge);
    info.SetSelectedCreditCard(SelectedCreditCard);
    info.SetMobilePaymethodId(MobilePaymethodId);
    info.SetYandexAccountBalance(YandexAccountBalance);
    info.SetPaymentDiscretization(PaymentDiscretization);

    info.SetCashbackPercent(CashbackPercent);
    if (CashbackInfo) {
        info.SetCashbackInfo(CashbackInfo->GetStringRobust());
    }
    if (IncreasedCashbackPercent) {
        info.SetIncreasedCashbackPercent(*IncreasedCashbackPercent);
    }
    info.SetHiddenCashback(HiddenCashback);
    info.SetOnlyOriginalOfferCashback(OnlyOriginalOfferCashback);

    if (HasOfferValidUntil()) {
        info.SetOfferValidUntil(GetOfferValidUntilRef().Seconds());
    }
    if (HasOfferValidSince()) {
        info.SetOfferValidSince(GetOfferValidSinceRef().Seconds());
    }

    info.SetBookable(Bookable);

    info.SetTargetHolderTag(TargetHolderTag);
    for (auto&& i : TargetUserTags) {
        info.AddTargetUserTag(i);
    }

    info.SetDeviceIdAccept(DeviceIdAccept);

    for (auto&& i : ProfitableTags) {
        info.AddProfitableTags(i);
    }
    for (auto&& i : AlreadyUsedProfitableTags) {
        info.AddAlreadyUsedProfitableTags(i);
    }

    info.SetInstanceType(GetTypeName());
    return info;
}

NJson::TJsonValue ICommonOffer::BuildJsonReport(const TReportOptions& options, const NDrive::IServer& server) const {
    const auto actionBuilder = server.GetDriveAPI()->GetRolesManager()->GetAction(BehaviourConstructorId);
    return DoBuildJsonReport(options, actionBuilder ? actionBuilder->GetAs<ICommonOfferBuilderAction>() : nullptr, server);
}

NJson::TJsonValue ICommonOffer::DoBuildJsonReport(const TReportOptions& options, const ICommonOfferBuilderAction* constructor, const NDrive::IServer& server) const {
    auto locale = options.Locale;
    auto localization = server.GetLocalization();
    NJson::TJsonValue result(NJson::JSON_MAP);
    result.InsertValue("type", GetReportedTypeName());
    result.InsertValue("name", GetName());
    if (ShortName) {
        result.InsertValue("short_name", FormDescriptionElement(ShortName, locale, server.GetLocalization()));
    }

    bool isCorpSession = false;
    if (options.Traits & NDriveSession::ReportOfferWallets) {
        NJson::TJsonValue& accountsJson = result.InsertValue("wallets", NJson::JSON_ARRAY);
        for (auto&& account : ChargableAccounts) {
            NJson::TJsonValue accData;
            accData["id"] = account;
            if (account == SelectedCharge) {
                accData["selected"] = true;
                isCorpSession = !ICommonOffer::NonCorporateMethods.contains(account);
            }
            accountsJson.AppendValue(accData);
        }
    }

    if (!(options.Traits & NDriveSession::ReportOfferDetails)) {
        return result;
    }

    NJson::TJsonValue& localizations = result["localizations"].SetType(NJson::JSON_MAP);
    if (constructor) {
        for (auto&& i : constructor->GetLocalizations()) {
            localizations.InsertValue(i.first, FormDescriptionElement(i.second, locale, localization));
        }
        if (!!constructor->GetVisual()) {
            result.InsertValue("visual", constructor->GetVisual()->GetReport());
        }
    }
    result.InsertValue("constructor_id", BehaviourConstructorId);
    result.InsertValue("behaviour_constructor_id", BehaviourConstructorId);
    result.InsertValue("price_constructor_id", PriceConstructorId);
    result.InsertValue("group_name", GetGroupName() ? GetGroupName() : GetDefaultGroupName());
    result.InsertValue("offer_id", GetOfferId());
    result.InsertValue("deadline", GetDeadline().Seconds());
    auto effectiveCashbackPercent = GetEffectiveCashbackPercent(server);
    if (effectiveCashbackPercent && !HiddenCashback) {
        result.InsertValue("cashback_percent", NJson::ToJson(effectiveCashbackPercent));
        result.InsertValue("only_original_offer_cashback", OnlyOriginalOfferCashback);
        NJson::TJsonValue& cashbackInfo = result["cashback_info"].SetType(NJson::JSON_MAP);
        if (IncreasedCashbackPercent && effectiveCashbackPercent >= *IncreasedCashbackPercent) {
            cashbackInfo.InsertValue("is_increased", true);
        }
    }
    if (DeviceIdAccept.size() > 4) {
        result.InsertValue("device_id", DeviceIdAccept.substr(0, 4) + "****" + DeviceIdAccept.substr(DeviceIdAccept.size() - 4));
    } else {
        result.InsertValue("device_id", DeviceIdAccept);
    }
    if (HasOfferValidUntil()) {
        result.InsertValue("offer_until", GetOfferValidUntilRef().Seconds());
    }
    if (HasOfferValidSince()) {
        result.InsertValue("offer_since", GetOfferValidSinceRef().Seconds());
    }
    if (ShortName) {
        result.InsertValue("short_name", FormDescriptionElement(ShortName, locale, localization));
    }
    if (SubName) {
        result.InsertValue("subname", FormDescriptionElement(SubName, locale, localization));
    }
    if (IsPlusUser) {
        result.InsertValue("is_plus_user", *IsPlusUser);
    }

    if (options.Traits & NDriveSession::ReportOfferWallets) {
        if (SelectedCreditCard) {
            result.InsertValue("credit_card", SelectedCreditCard);
        }
        result.InsertValue("is_corp_session", isCorpSession);
    }
    if (SelectedCreditCard) {
        result.InsertValue("credit_card", SelectedCreditCard);
    }
    result.InsertValue("is_corp_session", isCorpSession);

    return result;
}

TString ICommonOffer::DoFormDescriptionElement(const TString& value, ELocalization locale, const ILocalization* localization) const {
    auto result = value;
    SubstGlobal(result, "_DepositAmount_", GetLocalizedPrice(GetDeposit(), locale, *localization));
    SubstGlobal(result, "_OfferGroupName_", GetGroupName());
    SubstGlobal(result, "_OfferName_", GetName());
    TDuration timeShift = TDuration::Hours(3);
    if (HasOfferValidUntil()) {
        SubstGlobal(result, "_OfferUntil_", localization->FormatInstantWithYear(locale, GetOfferValidUntilRef() + timeShift));
    }
    if (HasOfferValidSince()) {
        SubstGlobal(result, "_OfferSince_", localization->FormatInstantWithYear(locale, GetOfferValidSinceRef() + timeShift));
    }
    SubstGlobal(result, "_OfferCurrency_", localization->GetLocalString(locale, "units.short." + GetCurrency()));
    return result;
}


TMaybe<ui32> ICommonOffer::GetEffectiveCashbackPercent(const NDrive::IServer& server) const {
    TSet<TString> supportedCashbackChargeTypes = { "card", "mobile_payment" };
    if (GetCashbackPercent() && supportedCashbackChargeTypes.contains(GetSelectedCharge()) && server.GetDriveAPI()->HasBillingManager() && server.GetDriveAPI()->GetBillingManager().GetAccountsManager().GetBonuses(GetUserId()) == 0) {
        return GetCashbackPercent();
    } else {
        return {};
    }
}

ui32 ICommonOffer::RoundPrice(const double price, const ui32 precision) {
    Y_ASSERT(price >= 0);
    Y_ASSERT(precision > 0);
    auto validated = std::max<double>(price, 0);
    auto roundedDown = std::floor(validated / precision);
    auto roundedUp = std::ceil(validated / precision);
    if (validated - roundedDown < 0.001) {
        return static_cast<ui32>(roundedDown) * precision;
    } else {
        return static_cast<ui32>(roundedUp) * precision;
    }
}

ui32 ICommonOffer::RoundCashback(const double price, const ui32 precision) {
    Y_ASSERT(price >= 0);
    Y_ASSERT(precision > 0);
    return std::round(std::max<double>(price, 0) / precision) * precision;
}

template <>
NJson::TJsonValue NJson::ToJson(const ICommonOffer& object) {
    NJson::TJsonValue result;
    result["offer_id"] = object.GetOfferId();
    result["user_id"] = object.GetUserId();
    return result;
}
