#pragma once

#include <drive/backend/database/drive_api.h>
#include <drive/backend/roles/manager.h>
#include <rtline/util/types/accessor.h>
#include <rtline/util/algorithm/ptr.h>

#include <util/datetime/base.h>
#include <util/generic/string.h>
#include <util/generic/guid.h>
#include <util/generic/set.h>
#include <util/generic/vector.h>
#include <rtline/api/action.h>
#include <rtline/util/instant_model.h>
#include <drive/backend/abstract/localization.h>
#include <library/cpp/object_factory/object_factory.h>

#include <drive/backend/offers/price/price.h>

#include <drive/backend/proto/offer.pb.h>
#include <drive/backend/abstract/base.h>
#include <drive/backend/actions/abstract/action.h>
#include <drive/backend/database/history/session.h>
#include <drive/backend/sessions/common/report.h>
#include <drive/backend/surge/surge_snapshot.h>

#include <library/cpp/json/writer/json_value.h>

#include <rtline/util/json_processing.h>
#include <rtline/util/types/uuid.h>

#include <util/generic/ymath.h>
#include <util/string/subst.h>

namespace NDrive::NProto {
    class TOffer;
}

class ICommonOfferBuilderAction;

constexpr ui32 DefaultPaymentDiscretization = 31415;

class ICommonOffer {
public:
    struct TReportOptions {
        TString Agreement;
        ELocalization Locale = DefaultLocale;
        NDriveSession::TReportTraits Traits = NDriveSession::ReportAll;
        TMap<TString, TDuration> DurationByTags;

        TReportOptions() = default;
        TReportOptions(ELocalization locale, NDriveSession::TReportTraits traits)
            : Locale(locale)
            , Traits(traits)
        {
        }
    };

    R_FIELD(TString, OfferId, CreateOfferId());

    // Expiration
    R_FIELD(TInstant, Timestamp, ModelingNow());
    R_FIELD(TInstant, Deadline, TInstant::Zero());

    // Name
    R_FIELD(TString, Name);
    R_FIELD(TString, GroupName);
    R_FIELD(TString, ShortName);
    R_FIELD(TString, SubName);

    // Constructor
    R_FIELD(TString, BehaviourConstructorId);
    R_FIELD(TString, PriceConstructorId);

    // User
    R_FIELD(TString, UserId);
    R_FIELD(TString, ExternalUserId);
    R_FIELD(ELocalization, Locale, DefaultLocale);

    // Correctors
    R_READONLY(TVector<TString>, Correctors);
    R_FIELD(TSet<TString>, DisabledCorrectors);

    // Payments
    R_FIELD(TString, Currency, "rub"); // Not in proto
    R_OPTIONAL(bool, IsPlusUser);
    R_FIELD(TVector<TString>, ChargableAccounts);
    R_FIELD(TString, SelectedCharge);
    R_FIELD(TString, SelectedCreditCard);
    R_FIELD(TString, MobilePaymethodId);
    R_FIELD(i64, YandexAccountBalance, 0);
    R_FIELD(ui32, PaymentDiscretization, DefaultPaymentDiscretization);

    // Cashback info
    R_FIELD(ui32, CashbackPercent, 0);
    R_OPTIONAL(NJson::TJsonValue, CashbackInfo);
    // IncreasedCashbackPercent contains minimal value of increased cashback percent.
    R_OPTIONAL(ui32, IncreasedCashbackPercent);
    R_FIELD(bool, HiddenCashback, false);
    R_FIELD(bool, OnlyOriginalOfferCashback, true);

    R_OPTIONAL(TInstant, OfferValidUntil);
    R_OPTIONAL(TInstant, OfferValidSince);

    R_FIELD(bool, Hidden, false); // Not in proto
    R_FIELD(bool, Bookable, true);

    R_FIELD(TString, TargetHolderTag);
    R_FIELD(TSet<TString>, TargetUserTags);

    R_FIELD(TString, DeviceIdAccept);

    R_FIELD(TSet<TString>, ProfitableTags);
    R_FIELD(TSet<TString>, AlreadyUsedProfitableTags);

public:
    using TPtr = TAtomicSharedPtr<ICommonOffer>;
    using TConstPtr = TAtomicSharedPtr<const ICommonOffer>;
    using TFactory = NObjectFactory::TParametrizedObjectFactory<ICommonOffer, TString>;

public:
    ICommonOffer() = default;
    virtual ~ICommonOffer() = default;

    // Name
    virtual TString GetTypeName() const = 0;
    virtual TString GetReportedTypeName() const {
        return GetTypeName();
    }
    virtual TString GetDefaultGroupName() const {
        return "";
    }
    TString GetFullName() const {
        return (GetGroupName() ? (GetGroupName() + ".") : "") + GetName();
    }

    // DeserializeFromProto deserializes offer from proto.
    virtual bool DeserializeFromProto(const NDrive::NProto::TOffer& info);
    // SerializeToProto serializes offer to proto.
    virtual NDrive::NProto::TOffer SerializeToProto() const;

    NJson::TJsonValue BuildJsonReport(const TReportOptions& options, const NDrive::IServer& server) const;

    TString FormDescriptionElement(const TString& value, ELocalization locale, const ILocalization* localization) const {
        auto valueAfterLocalization = localization->ApplyResources(value, locale);
        auto result = DoFormDescriptionElement(valueAfterLocalization, locale, localization);
        return localization->ApplyResources(result, locale);
    }

    virtual ui32 GetDeposit() const {
        return 0;
    }

    virtual ui32 GetDiscountedPrice(const ui32 price) const {
        return price;
    }

    TString GetLocalizedPrice(ui32 price, ELocalization locale, const ILocalization& localization) const {
        return localization.FormatPrice(
            locale,
            RoundPrice(price, 100),
            {"units.short." + GetCurrency()}
        );
    }

    virtual void OnBooking(const TString& deviceId, const TString& mobilePaymethodId = "");

    TMaybe<ui32> GetEffectiveCashbackPercent(const NDrive::IServer& server) const;

    void AddCorrector(const TString& name) {
        Correctors.push_back(name);
    }

    virtual bool PatchOffer(const NJson::TJsonValue& value, NDrive::TEntitySession& session, const NDrive::IServer* server, bool checkRidingConflict = true);
    virtual ui64 GetOfferRevision() const;

    virtual TExpected<bool, NJson::TJsonValue> CheckAgreementRequirements(ELocalization /*locale*/, const TUserPermissions::TPtr /*permissions*/, const NDrive::IServer* /*server*/) const {
        return true;
    }

    virtual NThreading::TFuture<TString> BuildAgreement(ELocalization /*locale*/, TAtomicSharedPtr<TCompiledRiding> /*compiledSession*/, TUserPermissions::TPtr /*permissions*/, const NDrive::IServer* /*server*/) const {
        return NThreading::MakeFuture(TString{});
    }

protected:
    virtual NJson::TJsonValue DoBuildJsonReport(const TReportOptions& options, const ICommonOfferBuilderAction* constructor, const NDrive::IServer& server) const;
    virtual TString DoFormDescriptionElement(const TString& value, ELocalization locale, const ILocalization* localization) const;

public:
    static ui32 RoundPrice(const double price, const ui32 precision = 1);
    static ui32 RoundCashback(const double price, const ui32 precision = 1);

    template<class T = ICommonOffer>
    static TAtomicSharedPtr<T> ConstructFromProto(const NDrive::NProto::TOffer& offerInfo) {
        auto result = TAtomicSharedPtr<ICommonOffer>(TFactory::Construct(offerInfo.GetInstanceType()));
        if (!result || !result->DeserializeFromProto(offerInfo)) {
            return nullptr;
        }
        return std::dynamic_pointer_cast<T>(result);
    }

    template<class T = ICommonOffer>
    static TAtomicSharedPtr<T> ConstructFromBase64Proto(const TString& data) {
        return ConstructFromStringProto<T>(Base64Decode(data));
    }

    template<class T = ICommonOffer>
    static TAtomicSharedPtr<T> ConstructFromStringProto(const TString& data) {
        NDrive::NProto::TOffer offerProto;
        if (!offerProto.ParseFromString(data)) {
            return nullptr;
        }
        return ConstructFromProto<T>(offerProto);
    }

    template <class T>
    static T DeepCopy(const T& offer) {
        T result;
        Y_ENSURE_BT(result.DeserializeFromProto(offer.SerializeToProto()));
        return result;
    }

    static TString CreateOfferId() {
        return NUtil::CreateUUID();
    }

    static TSet<TString> NonCorporateMethods;

    virtual bool NeedStore(const NDrive::IServer& server) const {
        return server.GetSettings().GetValueDef<bool>("offers.store", true);
    }
};

class IOfferState {
private:
    R_FIELD(bool, NeedFinishFees, true);

public:
    virtual ~IOfferState() = default;

    virtual NJson::TJsonValue GetReport(ELocalization locale, const NDrive::IServer& server) const = 0;

    virtual TString FormDescriptionElement(const TString& value, const TString& currency, ELocalization locale, const ILocalization& localization) const {
        Y_UNUSED(currency);
        Y_UNUSED(locale);
        return localization.ApplyResources(value, locale);
    }

    TString GetLocalizedPrice(ui32 price, const TString& currency, ELocalization locale, const ILocalization& localization) const {
        return localization.FormatPrice(locale, ICommonOffer::RoundPrice(price, 100), {"units.short." + currency});
    }
};

using TOfferStatePtr = TAtomicSharedPtr<IOfferState>;
