#include "dedicated_fleet.h"
#include "library/cpp/json/writer/json_value.h"

#include <drive/backend/offers/actions/dedicated_fleet.h>
#include <drive/backend/data/dedicated_fleet.h>
#include <drive/backend/device_snapshot/snapshots/snapshot.h>
#include <drive/backend/device_snapshot/manager.h>

#include <drive/backend/proto/offer.pb.h>

namespace {
    static EStatePriority GetStatePriority(const TString& state) {
        if (state == TSpecialFleetTag::Hold) {
            return EStatePriority::Hold;
        }

        if (state == TSpecialFleetTag::Preparing) {
            return EStatePriority::Preparing;
        }

        if (state == TSpecialFleetTag::Prepared) {
            return EStatePriority::Prepared;
        }

        if (state == TSpecialFleetTag::Delivery) {
            return EStatePriority::Delivery;
        }

        if (state == TSpecialFleetTag::Actual) {
            return EStatePriority::Actual;
        }

        return EStatePriority::Deferred;
    }
}

NJson::TJsonValue TDedicatedFleetOfferState::GetReport(ELocalization /*locale*/, const NDrive::IServer& server) const {
    NJson::TJsonValue report = NJson::JSON_MAP;
    auto localization = server.GetLocalization();
    if (!Stage) {
        return report;
    }

    NJson::InsertField(report, "stage", ::ToString(*Stage));
    if (!localization) {
        return report;
    }

    if (CancellationCost) {
        NJson::InsertField(report, "cancellation_cost", *CancellationCost);
    }

    if (DurationPrice) {
        NJson::InsertField(report, "duration_price", *DurationPrice);
    }

    if (SummaryDuration) {
        NJson::InsertField(report, "summary_duration", SummaryDuration->Seconds());
    }

    return report;
}

bool TDedicatedFleetUnitOffer::DeserializeFromProto(const NDrive::NProto::TOffer& offerProto) {
    if (!TBase::DeserializeFromProto(offerProto)) {
        return false;
    }

    auto& proto = offerProto.GetDedicatedFleetUnitOffer();
    for (const auto& protoOpt : proto.GetOption()) {
        TOption opt;
        opt.DeserializeFromProto(protoOpt);
        UnitOptions.emplace_back(std::move(opt));
    }

    CarId = proto.GetCarId();
    DeltaDurationCost = proto.GetDeltaDurationCost();

    for (const auto& protoTag : proto.GetServiceTagsToPerform()) {
        ServiceTagsToPerform.emplace(protoTag);
    }

    for (const auto& protoTag : proto.GetServiceTagsToCheck()) {
        ServiceTagsToCheck.emplace(protoTag);
    }

    return true;
}

NDrive::NProto::TOffer TDedicatedFleetUnitOffer::SerializeToProto() const {
    auto offerProto = TBase::SerializeToProto();
    auto& proto = *offerProto.MutableDedicatedFleetUnitOffer();

    for (const auto& opt : UnitOptions) {
        auto& protoOption = *proto.AddOption();
        opt.SerializeToProto(protoOption);
    }

    proto.SetCarId(GetCarId());
    proto.SetDeltaDurationCost(GetDeltaDurationCost());
    for (auto&& tagName : GetServiceTagsToPerform()) {
        *proto.AddServiceTagsToPerform() = tagName;
    }

    for (auto&& tagName : GetServiceTagsToCheck()) {
        *proto.AddServiceTagsToCheck() = tagName;
    }

    return offerProto;
}

NJson::TJsonValue TDedicatedFleetUnitOffer::DoBuildJsonReport(const TReportOptions& options, const ICommonOfferBuilderAction* constructor, const NDrive::IServer& server) const {
    auto report = TBase::DoBuildJsonReport(options, constructor, server);

    if (!(options.Traits & NDriveSession::ReportOfferDetails)) {
        return report;
    }

    auto locale = options.Locale;
    report.InsertValue("car_id", GetCarId());

    {
        NJson::TJsonValue options;
        for(auto&& opt : GetUnitOptions()) {
            for (const auto& report : opt.GetReport(locale, server)) {
                options.AppendValue(report);
            }
        }

        report.InsertValue("secondary", options);
    }

    return report;
}

TString TDedicatedFleetUnitOffer::DoFormDescriptionElement(const TString& value, ELocalization locale, const ILocalization* localization) const {
    auto result = TBase::DoFormDescriptionElement(value, locale, localization);
    return result;
}

bool TDedicatedFleetOffer::DeserializeFromProto(const NDrive::NProto::TOffer& offerProto) {
    if (!TBase::DeserializeFromProto(offerProto)) {
        return false;
    }

    auto &proto = offerProto.GetDedicatedFleetOffer();
    TotalCost = proto.GetTotalCost();
    AccountId = proto.GetAccountId();
    CancellationCostOnPreparing = proto.GetCancellationCostOnPreparing();
    NeedDeferCommunication = proto.GetNeedDeferCommunication();
    DeferThreshold = TDuration::MicroSeconds(proto.GetDeferThreshold());

    TimeIntervalValues.DeserializeFromProto(proto);
    FleetInfoValues.DeserializeFromProto(proto);

    for (auto&& unitProto : proto.GetUnitsOffers()) {
        auto unitOffer = MakeAtomicShared<TDedicatedFleetUnitOffer>();
        if (!unitOffer->DeserializeFromProto(unitProto)) {
            return false;
        }
        auto report = MakeAtomicShared<TDedicatedFleetOfferReport>(unitOffer, nullptr);
        UnitsOffers.emplace(unitOffer->GetOfferId(), report);
    }

    if (proto.HasBaseUnitOffer()) {
        auto unitOffer = MakeAtomicShared<TDedicatedFleetUnitOffer>();
        unitOffer->DeserializeFromProto(proto.GetBaseUnitOffer());
        auto report = MakeAtomicShared<TDedicatedFleetOfferReport>(unitOffer, nullptr);
        BaseUnitOffer = report;
    }

    return true;
}

NDrive::NProto::TOffer TDedicatedFleetOffer::SerializeToProto() const {
    auto offerProto = TBase::SerializeToProto();
    auto &proto = *offerProto.MutableDedicatedFleetOffer();
    proto.SetTotalCost(GetTotalCost());
    proto.SetAccountId(GetAccountId());
    proto.SetCancellationCostOnPreparing(GetCancellationCostOnPreparing());
    proto.SetNeedDeferCommunication(GetNeedDeferCommunication());
    proto.SetDeferThreshold(GetDeferThreshold().MicroSeconds());

    TimeIntervalValues.SerializeToProto(proto);
    FleetInfoValues.SerializeToProto(proto);

    for (auto&& r : GetUnitsOffers()) {
        *proto.AddUnitsOffers() = r.second->GetOfferAs<TDedicatedFleetUnitOffer>()->SerializeToProto();
    }

    if (BaseUnitOffer) {
        *proto.MutableBaseUnitOffer() = BaseUnitOffer->GetOfferAs<TDedicatedFleetUnitOffer>()->SerializeToProto();
    }

    return offerProto;
}

NJson::TJsonValue TDedicatedFleetOffer::DoBuildJsonReport(const TReportOptions& options, const ICommonOfferBuilderAction* constructor, const NDrive::IServer& server) const {
    auto report = TBase::DoBuildJsonReport(options, constructor, server);

    if (!options.Traits & NDriveSession::ReportOfferDetails) {
        return report;
    }

    auto locale = options.Locale;

    report.InsertValue("total_cost", GetTotalCost());
    report.InsertValue("account_id", GetAccountId());

    //TODO looks like there is no need in updating offer from builder configuration

    NJson::TJsonArray primaryArray;
    for (const auto& report : TimeIntervalValues.GetReport(locale, server)) {
        primaryArray.AppendValue(report);
    }

    for (const auto& report : FleetInfoValues.GetReport(locale, server)) {
        primaryArray.AppendValue(report);
    }

    report.InsertValue("primary", primaryArray);

    if (BaseUnitOffer) {
        if (auto offerReport = BaseUnitOffer->GetOfferAs<TDedicatedFleetUnitOffer>()){
            report.InsertValue("unit_builder", offerReport->DoBuildJsonReport(options, constructor, server));
        }
    }

    auto& offers = report.InsertValue("units_offers", NJson::JSON_ARRAY);
    for (auto&& report : GetUnitsOffers()) {
        if (auto uReport = report.second->GetOfferAs<TDedicatedFleetUnitOffer>()){
            auto jsonReport = uReport->DoBuildJsonReport(options, constructor, server);
            offers.AppendValue(std::move(jsonReport));
        }
    }

    return report;
}

TString TDedicatedFleetOffer::DoFormDescriptionElement(const TString& value, ELocalization locale, const ILocalization* localization) const {
    auto result = TBase::DoFormDescriptionElement(value, locale, localization);
    SubstGlobal(result, "_OfferPrice_", GetLocalizedPrice(GetTotalCost(), locale, *localization));
    return result;
}

TOfferStatePtr TDedicatedFleetOffer::Calculate(const TVector<IEventsSession<TAccountTagHistoryEvent>::TTimeEvent>& timeline, const TVector<TAtomicSharedPtr<TAccountTagHistoryEvent>>& events, const TInstant& until) const {
    TimelineOfferState accountState;
    TStates unitsStates;

    auto updateDuration = [&events, &timeline, &until] (ui64 index, TimelineOfferState& offerState) {
        if (!offerState.PrevEventId) {
            return;
        }

        auto& event = *events[timeline[*offerState.PrevEventId].GetEventIndex()];
        if (event.GetHistoryAction() == EObjectHistoryAction::Remove) {
            offerState.TagRemoved = true;
            return;
        }

        TInstant predInstant = timeline[*offerState.PrevEventId].GetEventInstant();
        TInstant currentInstant = (timeline[index].GetEventInstant() <= until) ? timeline[index].GetEventInstant() : until;
        TDuration duration = currentInstant - predInstant;
        offerState.Durations[offerState.CurrentTagName] += duration;
    };

    for (ui32 i = 0; i < timeline.size(); ++i) {
        if (i && timeline[i - 1].GetEventInstant() > until) {
            break;
        }

        if (timeline[i].GetTimeEvent() == IEventsSession<TCarTagHistoryEvent>::EEvent::CurrentFinish) {
            updateDuration(i, accountState);
            for (auto& [_, tagStates] : unitsStates) {
                for (auto& [_, tagState] : tagStates) {
                    if (!tagState.TagRemoved) {
                        updateDuration(i, tagState);
                    }
                }
            }
            break;
        }

        auto& event = (*events[timeline[i].GetEventIndex()]);
        auto accountTagImpl = std::dynamic_pointer_cast<const TDedicatedFleetOfferHolderTag>(event.GetData());
        if (accountTagImpl && accountState.CurrentTagName == TDedicatedFleetOfferHolderTag::Type() && accountState.PrevEventId) {
            updateDuration(i, accountState);
        }

        auto fleetTag = std::dynamic_pointer_cast<const TSpecialFleetTag>(event.GetData());
        if (fleetTag) {
            auto& uState = unitsStates[fleetTag->GetUnitOfferId()][event.GetTagId()];
            if (uState.CurrentTagName && uState.CurrentTagName == TSpecialFleetTag::Actual || uState.PrevEventId) {
                updateDuration(i, uState);
            }
        }

        if (event->GetName() == TDedicatedFleetOfferHolderTag::Type()) {
            accountState.CurrentTagName = (*events[timeline[i].GetEventIndex()])->GetName();
            accountState.PrevEventId = i;
        }

        if (event->GetName() == TSpecialFleetTag::Actual) {
            auto fleetTag = std::dynamic_pointer_cast<const TSpecialFleetTag>(event.GetData());
            if (fleetTag) {
                auto& uState = unitsStates[fleetTag->GetUnitOfferId()][event.GetTagId()];
                uState.CurrentTagName = (*events[timeline[i].GetEventIndex()])->GetName();
                uState.PrevEventId = i;
                uState.CurrentTagId = event.GetTagId();
            }
        }
    }

    auto state = MakeAtomicShared<TDedicatedFleetOfferState>();
    {
        TMaybe<TDuration> duration;
        for (const auto& [_, d] : accountState.Durations) {
            duration = duration ? *duration + d : d;
        }

        if (duration) {
            state->SetOfferTagDuration(duration);
        }
    }

    TMaybe<TDuration> duration;
    double durationPrice = 0;
    EStatePriority stage = EStatePriority::Deferred;
    for (auto& [_, tagStates] : unitsStates) {
        for (auto& [_, tagState] : tagStates) {
            for (const auto& [stateName, d] : tagState.Durations) {
                stage = std::max(stage, GetStatePriority(stateName));
                if (stateName == TSpecialFleetTag::Actual) {
                    duration = duration ? *duration + d : d;
                    const double segmentOriginalPrice = CalculateOriginalCost(d, stateName, BaseUnitOffer);
                    durationPrice += segmentOriginalPrice;
                }
            }
        }
    }

    if (duration) {
        state->SetSummaryDuration(duration);
        state->SetDurationPrice(durationPrice);
    }

    state->SetStage(stage);
    state->SetCancellationCost(CalculateCancellationCost(stage));

    return state;
}

double TDedicatedFleetOffer::CalculateOriginalCost(const TDuration stateDuration, const TString& tagName, const IOfferReport::TPtr offerReport) const {
    if (tagName != TSpecialFleetTag::Actual) {
        return 0;
    }

    static_assert(TDuration::Seconds(20) - TDuration::Seconds(200) == TDuration::Zero(), "autoclamp to zero");
    // TODO: there is could be some free duration in the offer
    //const auto& duration = stateDuration - GetFreeDurationChecked(tagName);
    auto unitOffer = offerReport->GetOfferAs<TDedicatedFleetUnitOffer>();
    if (!unitOffer) {
        return 0;
    }

    const auto& price = unitOffer->GetDeltaDurationCost();
    return  1.0 / 60.0 * stateDuration.Seconds() * price;
}

double TDedicatedFleetOffer::CalculateCancellationCost(const EStatePriority& stage) const {
    double cancellationCost = 0;
    switch (stage) {
        case EStatePriority::Preparing:
        case EStatePriority::Prepared:
        case EStatePriority::Delivery:
            cancellationCost = CancellationCostOnPreparing;
            break;

        case EStatePriority::Actual: {
            //TODO calculate
            break;
        }

        case EStatePriority::Deferred:
        case EStatePriority::Hold:
            break;
    }

    return cancellationCost;
}

TDedicatedFleetOffer::TFactory::TRegistrator<TDedicatedFleetOffer> TDedicatedFleetOffer::Registrator(TDedicatedFleetOffer::GetTypeNameStatic());
TDedicatedFleetUnitOffer::TFactory::TRegistrator<TDedicatedFleetUnitOffer> TDedicatedFleetUnitOffer::Registrator(TDedicatedFleetUnitOffer::GetTypeNameStatic());
