#pragma once

#include "base.h"
#include "dedicated_fleet_common.h"

#include <drive/backend/offers/actions/offer_price.h>
#include <drive/backend/offers/actions/types.h>

#include <rtline/library/json/field.h>


namespace NDrive::NProto {
    class TDedicatedFleetUnitOffer;
    class TDedicatedFleetOffer;
}

enum class EStatePriority {
    Deferred = 0 /* deferred */,
    Hold = 1 /* "hold" */,
    Preparing = 2 /* "preparing" */,
    Prepared = 3 /* "prepared" */,
    Delivery = 4 /* "delivery" */,
    Actual = 5 /* "actual" */
};

class TDedicatedFleetOfferState : public IOfferState {
private:
    R_OPTIONAL(TDuration, SummaryDuration);
    R_OPTIONAL(double, CancellationCost);
    R_OPTIONAL(double, DurationPrice, 0);
    R_OPTIONAL(EStatePriority, Stage, EStatePriority::Deferred);

    R_OPTIONAL(TDuration, OfferTagDuration);    //For testing propose

public:
    NJson::TJsonValue GetReport(ELocalization locale, const NDrive::IServer& server) const override;
};

class TDedicatedFleetOffer : public ICommonOffer {
private:
    struct TimelineOfferState {
        TString CurrentTagId;
        TString CurrentTagName;
        TMaybe<ui64> PrevEventId;
        TMap<TString, TDuration> Durations;
        //TODO support unit offer change history
        //TVector<TDedicatedFleetUnitOffer> offers;
        bool TagRemoved = false;
    };

    using TBase = ICommonOffer;
    using TTagsStates = TMap<TString, TimelineOfferState>;
    using TStates = TMap<TString, TTagsStates>;

public:
    using TPtr = TAtomicSharedPtr<TDedicatedFleetOffer>;
    using TConstPtr = TAtomicSharedPtr<const TDedicatedFleetOffer>;
    using TUnitsOffersMap = TMap<TString, IOfferReport::TPtr>;
    using TTimeIntervalValues = NDedicatedFleet::TTimeIntervalValuesAdapter;
    using TFleetInfoValues = NDedicatedFleet::TCommonFleetValuesAdapter;

public:
    static TString GetTypeNameStatic() {
        return "dedicated_fleet_offer";
    }

    TString GetTypeName() const override {
        return GetTypeNameStatic();
    }

public:
    bool DeserializeFromProto(const NDrive::NProto::TOffer& info) override;
    NDrive::NProto::TOffer SerializeToProto() const override;

protected:
    NJson::TJsonValue DoBuildJsonReport(const TReportOptions& options, const ICommonOfferBuilderAction* constructor, const NDrive::IServer& server) const override;
    TString DoFormDescriptionElement(const TString& value, ELocalization locale, const ILocalization* localization) const override;

public:
    TOfferStatePtr Calculate(const TVector<IEventsSession<TAccountTagHistoryEvent>::TTimeEvent>& timeline, const TVector<TAtomicSharedPtr<TAccountTagHistoryEvent>>& events, const TInstant& until) const;
    double CalculateOriginalCost(const TDuration stateDuration, const TString& tagName, const IOfferReport::TPtr offer) const;
    double CalculateCancellationCost(const EStatePriority& stage) const;

private:
    R_FIELD(double, TotalCost, 0);
    R_FIELD(double, CancellationCostOnPreparing, 0);
    R_FIELD(ui64, AccountId, 0);
    R_FIELD(TTimeIntervalValues, TimeIntervalValues);
    R_FIELD(TFleetInfoValues, FleetInfoValues);
    R_FIELD(IOfferReport::TPtr, BaseUnitOffer);
    R_FIELD(TUnitsOffersMap, UnitsOffers);
    R_FIELD(bool, NeedDeferCommunication, false);
    R_FIELD(TDuration, DeferThreshold);

private:
    static TFactory::TRegistrator<TDedicatedFleetOffer> Registrator;
};

class TDedicatedFleetUnitOffer : public ICommonOffer {
private:
    using TBase = ICommonOffer;

public:
    using TPtr = TAtomicSharedPtr<TDedicatedFleetUnitOffer>;
    using TConstPtr = TAtomicSharedPtr<const TDedicatedFleetUnitOffer>;
    using TOption = NDedicatedFleet::TOptionValuesAdapter;
    using TOptions = TVector<TOption>;

public:
    static TString GetTypeNameStatic() {
        return "dedicated_fleet_unit_offer";
    }

    TString GetTypeName() const override {
        return GetTypeNameStatic();
    }

public:
    bool DeserializeFromProto(const NDrive::NProto::TOffer& info) override;
    NDrive::NProto::TOffer SerializeToProto() const override;

protected:
    NJson::TJsonValue DoBuildJsonReport(const TReportOptions& options, const ICommonOfferBuilderAction* constructor, const NDrive::IServer& server) const override;
    TString DoFormDescriptionElement(const TString& value, ELocalization locale, const ILocalization* localization) const override;

public:
    R_FIELD(TString, CarId);
    R_FIELD(TOptions, UnitOptions);
    R_FIELD(TSet<TString>, ServiceTagsToPerform);
    R_FIELD(TSet<TString>, ServiceTagsToCheck);
    R_OPTIONAL(TVector<TGeoCoord>, DeliveryArea);
    R_OPTIONAL(TGeoCoord, DeliveryLocation);
    R_FIELD(TString, DeliveryLocationName);
    R_FIELD(TInstant, DeliveryTime);
    R_FIELD(ui64, DeltaDurationCost, 0);

    R_FIELD(bool, ForceUpdate, false);

    friend TDedicatedFleetOffer;

private:
    static TFactory::TRegistrator<TDedicatedFleetUnitOffer> Registrator;
};


class TDedicatedFleetUnitOfferReport: public IOfferReport {
private:
    using TBase = IOfferReport;

protected:
    TFullPricesContext* GetFullPricesContext() override final {return nullptr;}
    void DoRecalcPrices(const NDrive::IServer* /*server*/) override {}

public:
    using TBaseOffer = TDedicatedFleetUnitOffer;
    using TBase::TBase;

    TDedicatedFleetUnitOfferReport(const IOfferReport& source)
        : TBase(source)
    {
    }

    void ApplyFlowCorrection(const TString& /*areaId*/, const TOffersBuildingContext& /*context*/, const NDrive::IServer* /*server*/) override {}
    void ApplyInternalCorrection(const TString& /*areaId*/, const TOffersBuildingContext& /*context*/, const NDrive::IServer* /*server*/) override {}
    TString PredictDestination(const TOffersBuildingContext& /*context*/, const NDrive::IServer* /*server*/, const TCommonDestinationDetector& /*otherDetector*/) const override { return ""; }
    void RecalculateFeatures() override {}
};


class TDedicatedFleetOfferReport: public IOfferReport {
private:
    using TBase = IOfferReport;

protected:
    TFullPricesContext* GetFullPricesContext() override final {return nullptr;}
    void DoRecalcPrices(const NDrive::IServer* /*server*/) override {}

public:
    using TBaseOffer = TDedicatedFleetOffer;
    using TBase::TBase;

    TDedicatedFleetOfferReport(const IOfferReport& source)
        : TBase(source)
    {
    }

    void ApplyFlowCorrection(const TString& /*areaId*/, const TOffersBuildingContext& /*context*/, const NDrive::IServer* /*server*/) override {}
    void ApplyInternalCorrection(const TString& /*areaId*/, const TOffersBuildingContext& /*context*/, const NDrive::IServer* /*server*/) override {}
    TString PredictDestination(const TOffersBuildingContext& /*context*/, const NDrive::IServer* /*server*/, const TCommonDestinationDetector& /*otherDetector*/) const override { return ""; }
    void RecalculateFeatures() override {}
};
