#pragma once

#include "base.h"

#include <drive/backend/offers/actions/types.h>

#include <rtline/library/json/field.h>


namespace NDedicatedFleet {

    enum class ECostCalculatePolicy : ui32 {
        None = 0 /* "none" */,
        Mileage = 1 /* "mileage" */,
        Duration = 2 /* "duration" */,
    };

    template <class TValues>
    class TValuesAdapter : public TValues {
    public:
        using TConstants = typename TValues::TConstants;
        using TStringConstants = typename TValues::TStringConstants;

    public:
        NDrive::TScheme GetScheme(const NDrive::IServer* server) const {
            NDrive::TScheme scheme;
            scheme.Add<TFSStructure>("constants", "Константы").SetStructure(TConstants().GetScheme(server));
            scheme.Add<TFSStructure>("str_constants", "Строковые константы").SetStructure(TStringConstants().GetScheme(server));
            return scheme;
        }

        TVector<NJson::TJsonValue> GetReport(ELocalization locale, const NDrive::IServer& server) const;

        template<class Proto>
        bool DeserializeFromProto(const Proto& offer);

        template<class Proto>
        bool SerializeToProto(Proto& offer) const;

        bool DeserializeDataFromJson(const NJson::TJsonValue& value) {
            return NJson::TryFieldsFromJson(value, GetFields());
        }

        NJson::TJsonValue SerializeDataToJson() const {
            NJson::TJsonValue result;
            NJson::FieldsToJson(result, GetFields());
            return result;
        }

    public:
        R_OPTIONAL(TConstants, Constants);
        R_OPTIONAL(TStringConstants, StringConstants);

    public:
        DECLARE_FIELDS(
            Field(Constants, "constants"),
            Field(StringConstants, "str_constants")
        )
    };

    class TTimeIntervalValues {
    public:
        class TConstants {
        public:
            R_FIELD(TInstant, AvailableSince);
            R_FIELD(TInstant, AvailableUntil);
            R_FIELD(TDuration, PrepareDuration, TDuration::Hours(36));
            R_FIELD(TDuration, MinDuration, TDuration::Days(28));
            R_FIELD(TDuration, MaxDuration, TDuration::Days(365));
            R_FIELD(TDuration, FarthestStart, TDuration::Days(184));
            R_FIELD(TInstant, DefaultAvailableSince);
            R_FIELD(TInstant, DefaultAvailableUntil);
            R_FIELD(TDuration, DefaultDuration);

        public:
            NDrive::TScheme GetScheme(const NDrive::IServer* server) const;

        public:
            DECLARE_FIELDS(
                Field(AvailableSince, "available_since"),
                Field(AvailableUntil, "available_util"),
                Field(PrepareDuration, "prepare_duration"),
                Field(MinDuration, "min_duration"),
                Field(MaxDuration, "max_duration"),
                Field(FarthestStart, "farthest_start"),
                Field(DefaultAvailableSince, "default_available_since"),
                Field(DefaultAvailableUntil, "default_available_until"),
                Field(DefaultDuration, "default_duration")
            )
        };

        class TStringConstants {
        public:
            R_FIELD(TString, StartTitle, "dedicated.fleet.start.title");
            R_FIELD(TString, EndTitle, "dedicated.fleet.end.title");
            R_FIELD(TString, DurationTitle, "dedicated.fleet.duration.title");
            R_FIELD(TString, MinimalDurationText, "dedicated.fleet.minimal.duration.text");

        public:
            NDrive::TScheme GetScheme(const NDrive::IServer* server) const;

        public:
            DECLARE_FIELDS(
                Field(StartTitle, "start_title"),
                Field(EndTitle, "end_title"),
                Field(DurationTitle, "duration_title"),
                Field(MinimalDurationText, "minimal_duration_text")
            )
        };

    public:
        R_FIELD(TInstant, CreateTime);
        R_OPTIONAL(TInstant, BeginTime);
        R_OPTIONAL(TInstant, EndTime);
        R_OPTIONAL(TDuration, Duration);

        R_FIELD(TInstant, ClosestBeginInstant);
        R_FIELD(TInstant, FarthestBeginInstant);
        R_FIELD(TInstant, ClosestEndInstant);
        R_FIELD(TInstant, FarthestEndInstant);
    };

    class TCommonFleetValues {
    public:
        class TConstants {
        public:
            R_FIELD(ui64, MaxSize, 99999);
            R_FIELD(ui64, MinSize, 1);
            R_FIELD(ui64, DefaultSize, 1);

        public:
            NDrive::TScheme GetScheme(const NDrive::IServer* server) const;

        public:
            DECLARE_FIELDS(
                Field(MaxSize, "max_size"),
                Field(MinSize, "min_size"),
                Field(DefaultSize, "default_size")
            );
        };

        class TStringConstants {
        public:
            R_FIELD(TString, PlaceholderText, "dedicated.fleet.size.placeholder");
            R_FIELD(TString, AvailableText, "dedicated.fleet.available.text");

        public:
            NDrive::TScheme GetScheme(const NDrive::IServer* server) const;

        public:
            DECLARE_FIELDS(
                Field(PlaceholderText, "placeholder_text"),
                Field(AvailableText, "available_text")
            )
        };

    public:
        R_OPTIONAL(ui64, Size);

        R_OPTIONAL(ui64, MinSize, 0);
        R_OPTIONAL(ui64, MaxSize, 0);
        R_OPTIONAL(ui64, DefaultSize, 0);
    };

    class TOptionValues {
    public:
        static constexpr TStringBuf CalculatePolicyNameId = "calculate_policy";
        static constexpr TStringBuf GroupNameId = "group";

    public:
        class TConstants {
        public:
            R_FIELD(bool, DefaultValue, false);
            R_FIELD(TString, Id);
            R_FIELD(ui64, Cost, 0);
            R_FIELD(ECostCalculatePolicy, CalculatePolicy, ECostCalculatePolicy::None);

        public:
            NDrive::TScheme GetScheme(const NDrive::IServer* server) const;

        public:
            DECLARE_FIELDS(
                Field(DefaultValue, "default_value"),
                Field(Id, "id"),
                Field(Cost, "cost"),
                Field(CalculatePolicy, CalculatePolicyNameId)
            )
        };

        class TStringConstants {
        public:
            R_FIELD(TString, Title);
            R_FIELD(TString, Subtitle);
            R_FIELD(TString, Units);

        public:
            NDrive::TScheme GetScheme(const NDrive::IServer* server) const;

        public:
            DECLARE_FIELDS(
                Field(Title, "title"),
                Field(Subtitle, "subtitle"),
                Field(Units, "unit")
            )
        };

    public:
        R_OPTIONAL(bool, Value);
    };

    using TTimeIntervalValuesAdapter = TValuesAdapter<TTimeIntervalValues>;
    using TCommonFleetValuesAdapter = TValuesAdapter<TCommonFleetValues>;
    using TOptionValuesAdapter = TValuesAdapter<TOptionValues>;
}
