#include "standard_with_discount_area.h"

#include <drive/backend/device_snapshot/snapshots/snapshot.h>
#include <drive/backend/offers/actions/standard_with_discount_area.h>
#include <drive/backend/proto/offer.pb.h>

const TString TStandardWithDiscountAreaOffer::TypeName = "standard_with_discount_area";

TStandardWithDiscountAreaOffer::TFactory::TRegistrator<TStandardWithDiscountAreaOffer> TStandardWithDiscountAreaOffer::Registrator(TStandardWithDiscountAreaOffer::TypeName);

bool IsCurrentSession(const TVector<IEventsSession<TCarTagHistoryEvent>::TTimeEvent>& timeline) {
    if (timeline.empty() || timeline.back().GetTimeEvent() != IEventsSession<TCarTagHistoryEvent>::EEvent::Tag) {
        return true;
    }
    return false;
}

bool TStandardWithDiscountAreaOffer::IsInFinishPolygon(const TVector<IEventsSession<TCarTagHistoryEvent>::TTimeEvent>& timeline, const TVector<TAtomicSharedPtr<TCarTagHistoryEvent>>& events, const TInstant& until) const {
    if (!NDrive::HasServer()) {
        return false;
    }
    const auto server = &NDrive::GetServerAs<NDrive::IServer>();

    if (!IsCurrentSession(timeline)) {
        if (timeline.back().GetEventInstant() < until) {
            const THistoryDeviceSnapshot* snapshot = (*events[timeline.back().GetEventIndex()])->GetObjectSnapshotAs<THistoryDeviceSnapshot>();
            if (snapshot && IsInFinishArea(*snapshot, server, FinishArea, GetTypeName()).GetOrElse(false)) {
                return true;
            }
        }
    } else {
        auto snapshot = server->GetSnapshotsManager().GetSnapshot(GetObjectId());
        if (IsInFinishArea(snapshot, server, FinishArea, GetTypeName()).GetOrElse(false)) {
            return true;
        }
    }
    return false;
}

TOfferStatePtr TStandardWithDiscountAreaOffer::Calculate(const TVector<IEventsSession<TCarTagHistoryEvent>::TTimeEvent>& timeline, const TVector<TAtomicSharedPtr<TCarTagHistoryEvent>>& events, const TInstant& until, TOfferPricing& result) const {
    bool isCurrentSession = IsCurrentSession(timeline);
    bool isInFinishPolygon = IsInFinishPolygon(timeline, events, until);
    bool isDiscounted;

    TOfferStatePtr currentState;
    if (isCurrentSession || isInFinishPolygon) {
        isDiscounted = true;
        currentState = DiscountedOffer.Calculate(timeline, events, until, result);
    } else {
        isDiscounted = false;
        currentState = TBase::Calculate(timeline, events, until, result);
    }
    auto standartState = std::dynamic_pointer_cast<TStandartOfferState>(currentState);
    TAtomicSharedPtr<TStandardWithDiscountAreaOfferState> discountedState;
    discountedState = MakeAtomicShared<TStandardWithDiscountAreaOfferState>(*Yensured(standartState));
    discountedState->SetIsDiscountedState(isDiscounted);
    discountedState->SetInFinishPolygon(isInFinishPolygon);

    return discountedState;
}

bool TStandardWithDiscountAreaOffer::DeserializeFromProto(const NDrive::NProto::TOffer& info) {
    if (!TBase::DeserializeFromProto(info)) {
        return false;
    }
    NDrive::NProto::TStandardWithDiscountAreaOffer offerInfo = info.GetStandardWithDiscountAreaOffer();
    if (!Finish.Deserialize(offerInfo.GetFinish())) {
        return false;
    }
    for (auto&& i : offerInfo.GetFinishArea()) {
        TGeoCoord c;
        if (!c.Deserialize(i)) {
            return false;
        }
        FinishArea.emplace_back(std::move(c));
    }
    FinishName = offerInfo.GetFinishName();
    Discount = offerInfo.GetDiscount();
    RebuildDiscountedOffer();
    return true;
}

NDrive::NProto::TOffer TStandardWithDiscountAreaOffer::SerializeToProto() const {
    auto info = TBase::SerializeToProto();
    NDrive::NProto::TStandardWithDiscountAreaOffer& offerInfo = *info.MutableStandardWithDiscountAreaOffer();
    *offerInfo.MutableFinish() = Finish.Serialize();
    for (auto&& c : FinishArea) {
        *offerInfo.AddFinishArea() = c.Serialize();
    }
    offerInfo.SetFinishName(FinishName);
    offerInfo.SetDiscount(Discount);
    return info;
}

void TStandardWithDiscountAreaOffer::RebuildDiscountedOffer() {
    DiscountedOffer.DeepCopyFrom(*this);
    TDiscount discount;
    discount.SetVisible(true);
    discount.SetIdentifier(GetBehaviourConstructorId());
    discount.SetDiscount(Discount / 10000.);
    DiscountedOffer.AddDiscount(discount);
}

NJson::TJsonValue TStandardWithDiscountAreaOffer::DoBuildJsonReport(const TReportOptions& options, const ICommonOfferBuilderAction* constructor, const NDrive::IServer& server) const {
    Y_UNUSED(constructor);
    NJson::TJsonValue report = DiscountedOffer.BuildJsonReport(options, server);
    if (!(options.Traits & NDriveSession::ReportOfferDetails)) {
        return report;
    }
    report.InsertValue("discount", Discount);
    if (FinishName) {
        report.InsertValue("finish_name", FinishName);
    }
    if (!FinishArea.empty()) {
        report.InsertValue("finish_area", TGeoCoord::SerializeVector(FinishArea));
        report.InsertValue("finish", Finish.ToString());
    }
    if (FinishAreaId) {
        report.InsertValue("finish_area_id", FinishAreaId);
    } else if (auto needId = server.GetSettings().GetValue<bool>("standard_with_discount_area_offer.set_finish_area_id_for_router_zone").GetOrElse(true); needId && !FinishArea.empty()) {
        report.InsertValue("finish_area_id", Finish.ToString());
    }
    report.InsertValue("type", GetReportedTypeName());
    { // TODO: Add support for customization.
        NJson::TJsonValue style = NJson::JSON_MAP;
        if (auto bgColor = server.GetSettings().GetValue<TString>("standard_with_discount_area_offer.bg_color")) {
            style["bg_color"] = *bgColor;
        }
        if (auto borderColor = server.GetSettings().GetValue<TString>("standard_with_discount_area_offer.border_color")) {
            style["border_color"] = *borderColor;
        }
        if (auto borderWidth = server.GetSettings().GetValue<i32>("standard_with_discount_area_offer.border_width")) {
            style["border_width"] = *borderWidth;
        }
        report.InsertValue("finish_area_style", std::move(style));
    }
    return report;
}

void TStandardWithDiscountAreaOffer::ApplyDiscountModel(const NDrive::IOfferModel& model) {
    double discount = model.Calc(MutableFeatures());
    discount = 10000 * std::max(0., std::min(1., discount));
    SetDiscount(discount);
}


void TStandardWithDiscountAreaOffer::FillBill(TBill& bill, const TOfferPricing& pricing, TOfferStatePtr segmentState, ELocalization locale, const NDrive::IServer* server, ui32 cashbackPercent) const {
    auto state = std::dynamic_pointer_cast<TStandardWithDiscountAreaOfferState>(segmentState);
    if (state && state->GetIsDiscountedState()) {
        DiscountedOffer.FillBill(bill, pricing, segmentState, locale, server, cashbackPercent);
    } else {
        TBase::FillBill(bill, pricing, segmentState, locale, server, cashbackPercent);
    }
}
