#include "standard_with_discount_area.h"

#include <drive/backend/offers/actions/ut/library/helper.h>

#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/proto/offer.pb.h>
#include <drive/backend/ut/library/helper.h>

#include <library/cpp/testing/unittest/registar.h>

TAtomicSharedPtr<TStandardWithDiscountAreaOffer> CreateOffer() {
    TFullPricesContext context;
    context.MutableParking().SetPrice(100);
    context.MutableRiding().SetPrice(200);
    auto standartOffer(context);
    auto testOffer = MakeAtomicShared<TStandardWithDiscountAreaOffer>(standartOffer);
    testOffer->SetDiscount(5000);
    testOffer->SetFinishName("TestArea");
    testOffer->SetFinish(TGeoCoord(37.50, 55.40));
    TGeoCoord c1(37.49, 55.39);
    TGeoCoord c2(37.49, 55.41);
    TGeoCoord c3(37.51, 55.41);
    TGeoCoord c4(37.51, 55.39);
    testOffer->SetFinishArea({c1, c2, c3, c4, c1});
    testOffer->RebuildDiscountedOffer();
    return testOffer;
}

Y_UNIT_TEST_SUITE(TStandardWithDiscountAreaOffer) {
    Y_UNIT_TEST(SerializeDeserialize) {
        NDrive::NProto::TOffer report;
        {
            auto testOffer = MakeAtomicShared<TStandardWithDiscountAreaOffer>();

            testOffer->SetDiscount(50);
            testOffer->SetFinishName("TestArea");
            testOffer->SetFinish(TGeoCoord(37.49, 55.39));
            report = testOffer->SerializeToProto();
        }
        {
            THolder<TStandardWithDiscountAreaOffer> testOffer(new TStandardWithDiscountAreaOffer());
            UNIT_ASSERT(testOffer->DeserializeFromProto(report));
            UNIT_ASSERT_VALUES_EQUAL(testOffer->GetDiscount(), 50);
            UNIT_ASSERT_VALUES_EQUAL(testOffer->GetFinish(), TGeoCoord(37.49, 55.39));
        }
    }

    Y_UNIT_TEST(Calculate) {
        double priceWithoutDiscount;
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        {
            auto offer = CreateOffer();
            TVector<IEventsSession<TCarTagHistoryEvent>::TTimeEvent> timeline;
            TVector<TCarTagHistoryEvent> events;
            TBillingSession session;
            {
                TCarTagHistoryEventConstructor eventBase;
                eventBase.SetObjectId("object").SetTagId("1").SetHistoryUserId("user");
                AddEvent(session, eventBase, "old_state_reservation", TInstant::Seconds(1000), EObjectHistoryAction::SetTagPerformer);
                AddEvent(session, eventBase, "old_state_acceptance", TInstant::Seconds(1020), EObjectHistoryAction::TagEvolve);
                AddEvent(session, eventBase, "old_state_riding", TInstant::Seconds(10000), EObjectHistoryAction::TagEvolve);
                AddEvent(session, eventBase, "old_state_parking", TInstant::Seconds(10500), EObjectHistoryAction::TagEvolve);
                AddEvent(session, eventBase, "old_state_reservation", TInstant::Seconds(10600), EObjectHistoryAction::TagEvolve);
                AddEvent(session, eventBase, "old_state_reservation", TInstant::Seconds(106001), EObjectHistoryAction::DropTagPerformer);
            }
            {
                TTestCompilation compilation(offer);
                compilation.SetSince(TInstant::Seconds(1000)).SetUntil(TInstant::Seconds(110000));
                UNIT_ASSERT(session.FillCompilation(compilation));
                UNIT_ASSERT_DOUBLES_EQUAL(compilation.GetPricing().GetReportSumOriginalPrice(), 175833, 1);
                priceWithoutDiscount = compilation.GetPricing().GetReportSumOriginalPrice();
            }
        }
        {
            auto offer = CreateOffer();
            TVector<IEventsSession<TCarTagHistoryEvent>::TTimeEvent> timeline;
            TVector<TCarTagHistoryEvent> events;
            TBillingSession session;
            {
                THolder<THistoryDeviceSnapshot> snapshot;
                snapshot.Reset(new THistoryDeviceSnapshot);
                NDrive::TLocation location;
                location.Latitude = 55.40;
                location.Longitude = 37.50;
                location.Timestamp = Now();
                snapshot->SetLocation(location);
                THistoryDeviceSnapshot* sOriginal = snapshot.Release();
                NDrive::IObjectSnapshot::TPtr aSnapshot = sOriginal;

                TCarTagHistoryEventConstructor eventBase;
                eventBase.SetObjectId("object").SetTagId("1").SetHistoryUserId("user");
                AddEvent(session, eventBase, "old_state_reservation", TInstant::Seconds(1000), EObjectHistoryAction::SetTagPerformer);
                AddEvent(session, eventBase, "old_state_acceptance", TInstant::Seconds(1020), EObjectHistoryAction::TagEvolve);
                AddEvent(session, eventBase, "old_state_riding", TInstant::Seconds(10000), EObjectHistoryAction::TagEvolve);
                AddEvent(session, eventBase, "old_state_parking", TInstant::Seconds(10500), EObjectHistoryAction::TagEvolve, aSnapshot);
                AddEvent(session, eventBase, "old_state_reservation", TInstant::Seconds(10600), EObjectHistoryAction::TagEvolve, aSnapshot);
                AddEvent(session, eventBase, "old_state_reservation", TInstant::Seconds(106001), EObjectHistoryAction::DropTagPerformer, aSnapshot);
            }
            {
                UNIT_ASSERT(NDrive::HasServer());
                TTestCompilation compilation(offer);
                compilation.SetSince(TInstant::Seconds(1000)).SetUntil(TInstant::Seconds(110000));
                UNIT_ASSERT(session.FillCompilation(compilation));
                const auto state = std::dynamic_pointer_cast<TStandardWithDiscountAreaOfferState>(compilation.GetOfferState());
                UNIT_ASSERT(state);
                UNIT_ASSERT(state->GetInFinishPolygon());
                UNIT_ASSERT_VALUES_UNEQUAL(compilation.GetPricing().GetReportSumPrice(), priceWithoutDiscount);
                UNIT_ASSERT_DOUBLES_EQUAL(compilation.GetPricing().GetReportSumPrice(), priceWithoutDiscount * 0.5, 1);
                UNIT_ASSERT_DOUBLES_EQUAL(compilation.GetPricing().GetReportSumOriginalPrice(), priceWithoutDiscount, 1);
            }
        }
    }

    Y_UNIT_TEST(CheckForDeepCopyFrom) {
        TFullPricesContext context;
        context.MutableParking().SetPrice(100);
        context.MutableRiding().SetPrice(200);
        context.MutableAcceptance().SetPrice(99);
        context.MutableKm().SetPrice(999);
        context.SetOverrunKm(10);
        TStandartOffer baseOffer(context);
        baseOffer.SetDebtThreshold(125);
        baseOffer.SetUseDeposit(false);
        TStandartOffer copiedOffer;

        copiedOffer.DeepCopyFrom(baseOffer);
        UNIT_ASSERT_VALUES_EQUAL(copiedOffer.GetDebtThreshold(), 125);
        UNIT_ASSERT(!copiedOffer.GetUseDeposit());
        UNIT_ASSERT_VALUES_EQUAL(copiedOffer.GetOverrunKm(), 10);
        UNIT_ASSERT_VALUES_EQUAL(copiedOffer.GetParking().GetPrice(), 100);
        UNIT_ASSERT_VALUES_EQUAL(copiedOffer.GetRiding().GetPrice(), 200);
        UNIT_ASSERT_VALUES_EQUAL(copiedOffer.GetAcceptance().GetPrice(), 99);
        UNIT_ASSERT_VALUES_EQUAL(copiedOffer.GetKm().GetPrice(), 999);
    }
}
