#include "standart.h"

#include <drive/backend/offers/context.h>
#include <drive/backend/offers/actions/standart.h>
#include <drive/backend/offers/ranking/calcer.h>
#include <drive/backend/offers/ranking/model.h>

#include <drive/backend/billing/manager.h>
#include "drive/backend/billing/accounts/limited.h"
#include "drive/backend/cars/car.h"
#include "drive/backend/cars/car_model.h"
#include <drive/backend/common/localization.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/models/storage.h>
#include <drive/backend/proto/offer.pb.h>
#include <drive/backend/saas/api.h>
#include <drive/backend/users/user.h>

#include <drive/backend/sessions/manager/billing.h>

#include <drive/library/cpp/user_events_api/client.h>

#include <kernel/daemon/common/time_guard.h>

#include <library/cpp/mediator/global_notifications/system_status.h>

#include <rtline/library/geometry/coord.h>
#include <rtline/library/time_restriction/time_restriction.h>
#include <rtline/protos/proto_helper.h>
#include <rtline/util/json_processing.h>
#include <rtline/util/algorithm/ptr.h>

#include <util/generic/guid.h>
#include <util/string/builder.h>

TStandartOffer::TFactory::TRegistrator<TStandartOffer> TStandartOffer::Registrator(TStandartOffer::GetTypeNameStatic());

const NUnistat::TIntervals TStandartOffer::PriceIntervals = {0, 100, 200, 300, 350, 400, 450, 500, 550, 600, 650, 700, 750, 800, 850, 900, 950, 1000, 1100, 1200, 1300, 1400, 1500, 1600, 1700, 1800, 1900, 2000, 3000, 4000, 5000, 6000, 7000, 8000, 9000, 10000};
const NUnistat::TIntervals TStandartOffer::PriceDeltaIntervals = MakeVector<double>(xrange(0, 700, 50));
const TString AcceptanceSegmentName = "acceptance_cost";
const TString MileageSegmentName = "mileage";
const TString ServicingSegmentName = "servicing";

void TStandartOffer::SetPricingFrom(const TStandartOffer* base) {
    if (!base) {
        return;
    }
    CopyFromFullContext(*base);
}

ui32 TStandartOffer::GetPublicDiscountedAcceptance(ui32 precision) const {
    return GetPublicDiscountedPrice(GetAcceptance().GetPrice(), AcceptanceSegmentName, precision);
}

ui32 TStandartOffer::GetPublicDiscountedParking(const ui32 precision) const {
    if (GetParking().GetPriceModeling()) {
        return RoundPrice(ApplyDiscount(GetParking().GetPrice(), true, ESessionState::Parking), precision);
    } else {
        return GetPublicDiscountedPrice(GetParking().GetPrice(), ESessionState::Parking, precision);
    }
}
ui32 TStandartOffer::GetPublicDiscountedRiding(const ui32 precision) const {
    if (GetRiding().GetPriceModeling()) {
        return RoundPrice(ApplyDiscount(GetRiding().GetPrice(), true, ESessionState::Riding), precision);
    } else {
        return GetPublicDiscountedPrice(GetRiding().GetPrice(), ESessionState::Riding, precision);
    }
}
ui32 TStandartOffer::GetPublicDiscountedKm(const ui32 precision) const {
    if (GetKm().GetPriceModeling()) {
        return RoundPrice(ApplyDiscount(GetKm().GetPrice(), true, ESessionState::Riding), precision);
    } else {
        return GetPublicDiscountedPrice(GetKm().GetPrice(), ESessionState::Riding, precision);
    }
}

ui32 TStandartOffer::GetPublicOriginalParking(const ui32 precision) const {
    if (GetParking().GetPriceModeling()) {
        return RoundPrice(GetParking().GetPrice(), precision);
    } else {
        return GetPublicOriginalPrice(GetParking().GetPrice(), ESessionState::Parking, precision);
    }
}
ui32 TStandartOffer::GetPublicOriginalRiding(const ui32 precision) const {
    if (GetRiding().GetPriceModeling()) {
        return RoundPrice(GetRiding().GetPrice(), precision);
    } else {
        return GetPublicOriginalPrice(GetRiding().GetPrice(), ESessionState::Riding, precision);
    }
}
ui32 TStandartOffer::GetPublicOriginalKm(const ui32 precision) const {
    if (GetKm().GetPriceModeling()) {
        return RoundPrice(GetKm().GetPrice(), precision);
    } else {
        return GetPublicOriginalPrice(GetKm().GetPrice(), ESessionState::Riding, precision);
    }
}

TStandartOffer& TStandartOffer::SetFinishAreaTagsFilter(const TTagsFilter& filter) {
    if (filter.IsEmpty()) {
        FinishAreaTagsFilter.Reset(nullptr);
    } else {
        FinishAreaTagsFilter.Reset(new TTagsFilter(filter));
    }
    return *this;
}

const TTagsFilter& TStandartOffer::GetFinishAreaTagsFilter() const {
    if (!FinishAreaTagsFilter) {
        return Default<TTagsFilter>();
    } else {
        return *FinishAreaTagsFilter;
    }
}

TStandartOffer& TStandartOffer::SetRidingAreaTagsFilter(const TTagsFilter& filter) {
    if (filter.IsEmpty()) {
        RidingAreaTagsFilter.Reset(nullptr);
    } else {
        RidingAreaTagsFilter.Reset(new TTagsFilter(filter));
    }
    return *this;
}

const TTagsFilter& TStandartOffer::GetRidingAreaTagsFilter() const {
    if (!RidingAreaTagsFilter) {
        return Default<TTagsFilter>();
    } else {
        return *RidingAreaTagsFilter;
    }
}

TMaybe<ui32> TStandartOffer::GetReportedAcceptancePrice() const {
    auto value = GetPublicDiscountedAcceptance();
    if (value) {
        return value;
    } else {
        return {};
    }
}

TString TStandartOffer::GetPushReport(ELocalization locale, const IServerBase* server) const {
    TInstant freeTill = GetTimestamp() + GetFreeDuration(TChargableTag::Reservation);
    TString ridingStr;
    TString waitingStr;
    if (server) {
        ridingStr = server->GetLocalization()->FormatPrice(locale, GetPublicDiscountedRiding(), {"units.short." + GetCurrency(), "/", "units.short.minutes"});
        waitingStr = server->GetLocalization()->FormatPrice(locale, GetPublicDiscountedParking(), {"units.short." + GetCurrency(), "/", "units.short.minutes"});
    } else {
        ridingStr = TFakeLocalization().FormatPrice(locale, GetPublicDiscountedRiding(), {});
        waitingStr = TFakeLocalization().FormatPrice(locale, GetPublicDiscountedParking(), {});
    }
    return "Бесплатное ожидание до " + freeTill.FormatLocalTime("%H:%M") + ". В пути: " + ridingStr + ". Ожидание: " + waitingStr + ".";
}

bool TStandartOffer::GetFreeAndPricedDurations(TDuration& freeDuration, TDuration& pricedDuration, const TMap<TString, TOfferSegment>& segments) const {
    freeDuration = TDuration::Zero();
    pricedDuration = TDuration::Zero();

    for (auto&& i : segments) {
        const TDuration full = i.second.GetDuration();
        const TDuration free = Min<TDuration>(full, GetFreeDuration(i.first));
        freeDuration += free;
        pricedDuration += (full - free);
    }
    return true;
}

bool IsFreeDurationForced() {
    // avoided useless memoization coz NDrive::GetServer is a quite fast singletone impl
    return NDrive::HasServer()
        && NDrive::GetServer().GetSettings()
            .GetValue<bool>("offers.force_free_duration")
            .GetOrElse(false);
}

TDuration TStandartOffer::GetFreeDurationChecked(const TString& tagName) const {
    if (tagName == TChargableTag::Reservation || tagName == TChargableTag::Acceptance || IsFreeDurationForced()) {
        return GetFreeDuration(tagName);
    }
    return TDuration::Zero();
}

double TStandartOffer::CalculateOriginalPrice(const TDuration stateDuration, const TString& tagName) const {
    if (tagName == TChargableTag::Servicing) {
        return 0;
    };

    static_assert(TDuration::Seconds(20) - TDuration::Seconds(200) == TDuration::Zero(), "autoclamp to zero");
    const auto& duration = stateDuration - GetFreeDurationChecked(tagName);

    const auto& price = tagName == TChargableTag::Riding
        ? GetPublicOriginalRiding()
        : GetPublicOriginalParking();

    return 1.0 / 60.0 * duration.Seconds() * price;
}

TString TStandartOffer::DoBuildCommonPricesDescription(const TFullCompiledRiding& /*fcr*/, const NDrive::IServer& /*server*/) const {
    TStringBuilder sb;
    sb << "\\subsection{Standart offer}" << Endl;
    sb << "\\begin{enumerate}" << Endl;
    sb << "\\item Has riding price model: " << GetRiding().GetPriceModeling() << Endl;
    sb << "\\item Has parking parking model: " << GetParking().GetPriceModeling() << Endl;
    sb << "\\item Original parking price: " << GetParking().GetPrice() << Endl;
    sb << "\\item Original riding price: " << GetRiding().GetPrice() << Endl;
    sb << "\\item Public original parking price: " << GetPublicOriginalParking() << Endl;
    sb << "\\item Public original riding price: " << GetPublicOriginalRiding() << Endl;
    sb << "\\item Final parking price: " << GetPublicDiscountedParking() << Endl;
    sb << "\\item Final riding price: " << GetPublicDiscountedRiding() << Endl;
    sb << "\\end{enumerate}" << Endl;
    return sb;
}

TString TStandartOffer::DoBuildCalculationDescription(ELocalization locale, const TFullCompiledRiding& fcr, const NDrive::IServer& server) const {
    Y_UNUSED(locale);
    TStringBuilder sb;

    const TVector<TString> phases = TChargableTag::GetTagNames(server.GetDriveAPI()->GetTagsManager().GetTagsMeta());

    sb << "\\subsection{Phase durations}" << Endl;
    sb << "\\begin{enumerate}" << Endl;
    double publicFinalPrice = 0;
    for (auto&& t : phases) {
        TDuration d;
        if (!fcr.GetPhaseDuration(t, d)) {
            sb << "\\item \\verb|" << t << "| FAILED" << Endl;
            continue;
        }
        if (d == TDuration::Zero()) {
            continue;
        }
        sb << "\\item \\verb|" << t << "|" << Endl;
        sb << "\\begin{enumerate}" << Endl;
        const double pricedDuration = 1.0 / 60.0 * (d - GetFreeDuration(t)).Seconds();
        if (t == TChargableTag::Riding) {
            sb << "\\item " << "Original price: " << pricedDuration * GetRiding().GetPrice() << Endl;
            sb << "\\item " << "Public original price: " << pricedDuration * GetPublicOriginalRiding() << Endl;
            sb << "\\item " << "Public final price: " << pricedDuration * GetPublicDiscountedRiding() << Endl;
            publicFinalPrice += pricedDuration * GetPublicDiscountedRiding();
        } else {
            sb << "\\item " << "Original price: " << pricedDuration * GetParking().GetPrice() << Endl;
            sb << "\\item " << "Public original price: " << pricedDuration * GetPublicOriginalParking() << Endl;
            sb << "\\item " << "Public final price: " << pricedDuration * GetPublicDiscountedParking() << Endl;
            publicFinalPrice += pricedDuration * GetPublicDiscountedParking();
        }
        sb << "\\end{enumerate}" << Endl;
    }
    sb << "\\item TOTAL: " << publicFinalPrice << " / " <<  Endl;
    sb << "\\end{enumerate}" << Endl;

    return sb;
}

bool NeedToTrickPrices() {
    // avoided useless memoization coz NDrive::GetServer is a quite fast singletone impl
    return NDrive::HasServer()
        && NDrive::GetServer().GetSettings()
            .GetValue<bool>("offers.trick_prices_according_to_free_duration")
            .GetOrElse(false);
}

NJson::TJsonValue TStandartOffer::DoBuildJsonReport(const TReportOptions& options, const ICommonOfferBuilderAction* constructor, const NDrive::IServer& server) const {
    NJson::TJsonValue report = TBase::DoBuildJsonReport(options, constructor, server);
    const auto locale = options.Locale;
    const auto traits = options.Traits;
    {
        NJson::TJsonValue& prices = report.InsertValue("prices", NJson::JSON_MAP);
        prices.InsertValue("insurance_type", InsuranceType);
        if (options.Traits & NDriveSession::ReportStandartOfferPrices) {
            const auto& freeParkingDuration = GetFreeDurationChecked(TChargableTag::Parking);
            if (freeParkingDuration) {
                prices.InsertValue("parking_free_time", freeParkingDuration.Seconds());
            }
            const auto& isFreeParking = freeParkingDuration >
                options.DurationByTags.Value(TChargableTag::Parking, TDuration::Zero());

            NJson::InsertNonNull(prices, "acceptance_cost", GetReportedAcceptancePrice());
            NJson::InsertNonNull(prices, "insurance_price", InsuranceCost);
            prices.InsertValue("parking", isFreeParking && NeedToTrickPrices()
                ? 0 // highlight free parking period in UI
                : GetPublicOriginalParking());
            prices.InsertValue("riding", GetPublicOriginalRiding());
            prices.InsertValue("km", GetPublicOriginalKm());
            prices.InsertValue("km_discounted", GetPublicDiscountedKm());
            prices.InsertValue("parking_discounted", GetPublicDiscountedParking());
            prices.InsertValue("riding_discounted", GetPublicDiscountedRiding());
            prices.InsertValue("discount", GetSummaryVisibleDiscount().GetPublicReport(locale, server, false));
            prices.InsertValue("free_reservation", GetFreeDuration(TChargableTag::Reservation).Seconds());
            prices.InsertValue("use_deposit", GetUseDeposit() && GetDeposit());
            prices.InsertValue("deposit", GetDeposit());

            //by desing we round acceptance price for user report like 100,13 -> 101. But real price will be 100,13.
            TString ridingHr = server.GetLocalization()->FormatPrice(options.Locale, GetPublicDiscountedRiding(), { "units.short." + GetCurrency(), "/", "units.short.minutes" });
            auto acceptancePrice = GetPublicDiscountedAcceptance();
            if (acceptancePrice) {
                TString acceptanceHr = server.GetLocalization()->FormatPrice(options.Locale, ICommonOffer::RoundPrice(acceptancePrice, 100),  { "units.short." + GetCurrency() });
                prices.InsertValue("price_hr", acceptanceHr + " + " + ridingHr);
                prices.InsertValue("acceptance_hr", acceptanceHr);
            } else {
                prices.InsertValue("price_hr", ridingHr);
            }
            prices.InsertValue("riding_hr", ridingHr);
            prices.InsertValue("parking_hr", server.GetLocalization()->FormatPrice(options.Locale, GetPublicDiscountedParking(), {  "units.short." + GetCurrency(), "/", "units.short.minutes" }));
            prices.InsertValue("mileage_hr", server.GetLocalization()->FormatPrice(options.Locale, GetPublicDiscountedKm(), { "units.short." + GetCurrency(), "/", "units.short.km" }));

            prices["original"].InsertValue("riding_hr", server.GetLocalization()->FormatPrice(options.Locale, GetPublicOriginalRiding(), { "units.short." + GetCurrency(), "/", "units.short.minutes" }));
            prices["original"].InsertValue("parking_hr", server.GetLocalization()->FormatPrice(options.Locale, GetPublicOriginalParking(), { "units.short." + GetCurrency(), "/", "units.short.minutes" }));
            prices["original"].InsertValue("mileage_hr", server.GetLocalization()->FormatPrice(options.Locale, GetPublicOriginalKm(), { "units.short." + GetCurrency(), "/", "units.short.km" }));
        }
    }
    if (!(options.Traits & NDriveSession::ReportOfferDetails)) {
        return report;
    }
    const auto& agreement = options.Agreement ? options.Agreement : Agreement;
    if (agreement) {
        report.InsertValue("agreement", agreement);
        report["localizations"].InsertValue("custom_act", agreement);
    }
    {
        NJson::TJsonValue& debt = report.InsertValue("debt", NJson::JSON_MAP);
        debt.InsertValue("threshold", DebtThreshold);
    }
    NJson::TJsonValue& shortDescription = report.InsertValue("short_description", NJson::JSON_ARRAY);
    if (GetUseDefaultShortDescriptions()) {
        for (auto&& i : GetDefaultShortDescription(locale, traits, *server.GetLocalization())) {
            shortDescription.AppendValue(FormDescriptionElement(i, locale, server.GetLocalization()));
        }
    }
    const TStandartOfferConstructor* stConstructor = dynamic_cast<const TStandartOfferConstructor*>(constructor);
    if (stConstructor) {
        report.InsertValue("detailed_description", FormDescriptionElement(stConstructor->GetDetailedDescription(), locale, server.GetLocalization()));
        if (stConstructor->GetDetailedDescriptionIcon()) {
            report.InsertValue("finger_icon", stConstructor->GetDetailedDescriptionIcon());
        }
        for (auto&& i : stConstructor->GetShortDescription()) {
            shortDescription.AppendValue(FormDescriptionElement(i, locale, server.GetLocalization()));
        }
    }
    return report;
}

EDriveSessionResult TStandartOffer::DoCheckSession(const TUserPermissions& permissions, NDrive::TEntitySession& session, const NDrive::IServer* server, bool onPerform) const {
    if (!server->GetDriveAPI()->HasBillingManager()) {
        return EDriveSessionResult::Success;
    }
    auto& billingManager = server->GetDriveAPI()->GetBillingManager();
    auto payments = billingManager.GetActiveUserPayments(permissions.GetUserId(), session);
    if (!payments) {
        return EDriveSessionResult::TransactionProblem;
    }

    ui32 debt = 0;
    for (auto&& pay : *payments) {
        const TBillingTask& bTask = pay.GetBillingTask();
        auto depositRequired = permissions.GetSetting<bool>("billing.deposit_is_required").GetOrElse(false);
        auto depositFailSum = permissions.GetSetting<ui32>("billing.session_debt_threshold_deposit").GetOrElse(0);
        auto depositFailed = bTask.GetDeposit() > pay.GetSnapshot().GetHeldSum() + depositFailSum;

        if (bTask.GetId() == GetOfferId() && depositFailed) {
            if (bTask.IsDebt()) {
                session.SetErrorInfo("offer_check", "deposit_fails", EDriveSessionResult::DepositFails);
                return EDriveSessionResult::DepositFails;
            }
            if (depositRequired) {
                session.SetErrorInfo("offer_check", "required_deposit_is_not_held", NDrive::MakeError("required_deposit_is_not_held"));
                session.OptionalResult() = EDriveSessionResult::RequiredDepositIsNotHeld;
                return EDriveSessionResult::RequiredDepositIsNotHeld;
            }
        }
        if (!bTask.IsDebt()) {
            continue;
        }
        debt += pay.GetDebt();
    }
    auto debtThreshold = permissions.GetSetting<ui32>("billing.session_debt_threshold_override").GetOrElse(DebtThreshold);
    auto debtOnPerformThreshold = permissions.GetSetting<ui32>("billing.debt_threshold_on_perform").GetOrElse(0);
    if (debt > debtThreshold || (onPerform && debt > debtOnPerformThreshold)) {
        session.SetError(NDrive::MakeError("payment_required"));
        session.SetErrorInfo("StandartOffer::DoCheckSession", TStringBuilder() << "debt is " << debt, EDriveSessionResult::PaymentRequired);
        return EDriveSessionResult::PaymentRequired;
    }
    return EDriveSessionResult::Success;
}

TVector<TString> TStandartOffer::GetDefaultShortDescription(ELocalization locale, NDriveSession::TReportTraits traits, const ILocalization& localization) const {
    const TDuration freeReservationDuration = GetFreeDuration(TChargableTag::Reservation);

    TVector<TString> lines;
    if (traits & NDriveSession::ReportOfferShortDescriptionStandartParkingPrice) {
        lines.push_back(TStringBuilder() << "Ожидание — " << localization.FormatPrice(locale, GetPublicDiscountedParking()) << " ₽/мин");
    }
    if (freeReservationDuration >= TDuration::Minutes(1)) {
        lines.push_back(TStringBuilder() << freeReservationDuration.Minutes() << " мин бесплатного ожидания");
    } else {
        lines.push_back(TStringBuilder() << "Нет бесплатного ожидания");
    }
    return lines;
}

TString TStandartOffer::DoFormDescriptionElement(const TString& value, ELocalization locale, const ILocalization* localization) const {
    TString result = TBase::DoFormDescriptionElement(value, locale, localization);
    SubstGlobal(result, "_OfferName_", GetName());
    const TString parkingPrice = localization->FormatPrice(locale, GetParking().GetPrice());
    SubstGlobal(result, "_ParkingPrice_", parkingPrice);
    const TString ridingPrice = localization->FormatPrice(locale, GetRiding().GetPrice());
    SubstGlobal(result, "_RidingPrice_", ridingPrice);
    const TString kmPrice = localization->FormatPrice(locale, GetKm().GetPrice());
    SubstGlobal(result, "_KmPrice_", kmPrice);

    const TString acceptanceCostDiscount = localization->FormatPrice(locale, GetPublicDiscountedPrice(GetAcceptance().GetPrice(), AcceptanceSegmentName));
    SubstGlobal(result, "_AcceptanceCostDiscount_", acceptanceCostDiscount);
    const TString parkingPriceDiscount = localization->FormatPrice(locale, GetPublicDiscountedParking());
    SubstGlobal(result, "_ParkingPriceDiscount_", parkingPriceDiscount);
    const TString ridingPriceDiscount = localization->FormatPrice(locale, GetPublicDiscountedRiding());
    SubstGlobal(result, "_RidingPriceDiscount_", ridingPriceDiscount);

    {
        const TDuration freeTime = GetFreeDuration(TChargableTag::Reservation, false);
        SubstGlobal(result, "_FreeDurationReservation_", localization->FormatDuration(locale, freeTime));
        SubstGlobal(result, "_FreeDuration_", localization->FormatDuration(locale, freeTime));
        SubstGlobal(result, "_FreeDurationFull_", localization->FormatFreeWaitTime(locale, freeTime));
        SubstGlobal(result, "_FreeFullDurationReservationSeconds_", ::ToString(freeTime.Seconds()));
    }

    {
        const TDuration freeTime = GetFreeDuration(TChargableTag::Acceptance, true);
        SubstGlobal(result, "_FreeDurationAcceptance_", localization->FormatDuration(locale, freeTime));
    }

    {
        const TDuration freeTime = GetFreeDuration(TChargableTag::Parking, true);
        SubstGlobal(result, "_FreeDurationParking_", localization->FormatDuration(locale, freeTime));
    }

    return result;
}

void TStandartOffer::FillBillPricing(TBill& bill, const TOfferPricing& pricing, TOfferStatePtr /*segmentState*/, ELocalization locale, const NDrive::IServer* server) const {
    auto localization = server ? server->GetLocalization() : nullptr;
    for (auto&& i : pricing.GetPrices()) {
        if ((int)(i.second.GetOriginalPrice()) < 1e-5) {
            continue;
        }
        TBillRecord billRecord;
        billRecord.SetCost(i.second.GetOriginalPrice()).SetType(i.first).SetDuration(i.second.GetDuration()).SetDistance(i.second.GetDistance());
        auto description = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(i.first);
        if (description) {
            billRecord.SetTitle(description->GetDisplayName());
            billRecord.SetDetails(localization->FormatDuration(locale, i.second.GetDuration(), true));
            bill.MutableRecords().emplace_back(std::move(billRecord));
            continue;
        }
    }
    auto acceptance = pricing.GetPrices().find(AcceptanceSegmentName);
    if (acceptance != pricing.GetPrices().end()) {
        const TOfferSegment& segment = acceptance->second;
        auto titleKey = "session." + acceptance->first + ".bill.title";
        auto title = localization ? localization->GetLocalString(locale, titleKey) : titleKey;
        auto price = segment.GetOriginalPrice();
        if (std::abs(price) > 0.001) {
            TBillRecord record;
            record.SetCost(price);
            record.SetTitle(title);
            record.SetType(AcceptanceSegmentName);
            bill.MutableRecords().emplace_back(std::move(record));
        }
    }
    auto mileage = pricing.GetPrices().find(MileageSegmentName);
    if (mileage != pricing.GetPrices().end()) {
        const TOfferSegment& segment = mileage->second;
        TBillRecord record;
        record.SetCost(segment.GetOriginalPrice());
        record.SetDetails(localization->DistanceFormatKm(locale, segment.GetDistance()));
        record.SetTitle(NDrive::TLocalization::MileageReceiptElementTitle());
        record.SetType(MileageSegmentName);
        bill.MutableRecords().emplace_back(std::move(record));
    }
    auto servicing = pricing.GetPrices().find(ServicingSegmentName);
    if (servicing != pricing.GetPrices().end()) {
        const TOfferSegment& segment = servicing->second;
        auto titleKey = "session." + servicing->first + ".bill.title";
        auto title = localization ? localization->GetLocalString(locale, titleKey) : titleKey;

        TBillRecord record;
        record.SetCost(segment.GetOriginalPrice());
        record.SetDetails(localization->FormatDuration(locale, segment.GetDuration()));
        record.SetDuration(segment.GetDuration());
        record.SetTitle(title);
        record.SetType(ServicingSegmentName);
        bill.MutableRecords().push_back(std::move(record));
    }
}

void TStandartOffer::PatchSessionReport(NJson::TJsonValue& result, NDriveSession::TReportTraits traits, ELocalization locale,
                                    const NDrive::IServer& server, const TOfferSessionPatchData& patchData) const {
    TBase::PatchSessionReport(result, traits, locale, server, patchData);

    result.InsertValue(InsuranceTypeId, GetInsuranceType());
}

TDuration TStandartOffer::CalcChargableDuration(const TInstant predInstant, const TInstant nextInstant, const TString& tagName, const TSet<TString>& tagsInPoint) const {
    if (tagsInPoint.contains("no_price")) {
        return TDuration::Zero();
    }
    TTimeRestrictionsPool<TTimeRestriction> restrictionsPool;
    for (auto&& i : Discounts) {
        const TDiscount::TDiscountDetails& details = i.GetDetails(tagName);
        bool active = true;
        for (auto&& tagInPoint : details.GetTagsInPoint()) {
            const bool reverse = tagInPoint.StartsWith("!");
            if (reverse) {
                if (tagsInPoint.contains(tagInPoint.substr(1))) {
                    active = false;
                    break;
                }
            } else {
                if (!tagsInPoint.contains(tagInPoint)) {
                    active = false;
                    break;
                }
            }
        }
        if (!active) {
            continue;
        }
        const TTimeRestrictionsPool<TTimeRestriction>* restriction = details.GetFreeTimetable();
        if (restriction) {
            restrictionsPool.Merge(*restriction);
        }
    }
    if (!restrictionsPool.Empty()) {
        const TDuration d = restrictionsPool.GetCrossSize(predInstant, nextInstant);
        return (nextInstant - predInstant) - d;
    } else {
        return nextInstant - predInstant;
    }
}

TOfferStatePtr TStandartOffer::DoCalculate(const TVector<IEventsSession<TCarTagHistoryEvent>::TTimeEvent>& timeline, const TVector<TAtomicSharedPtr<TCarTagHistoryEvent>>& events, const TInstant& until, const TRidingInfo& ridingInfo, TOfferPricing& result) const {
    TString currentTagName;
    TMap<TString, std::pair<TDuration, TDuration>> durations;
    TMaybe<double> minMileage;
    TMaybe<double> maxMileage;
    for (ui32 i = 0; i < timeline.size(); ++i) {
        if (i && timeline[i - 1].GetEventInstant() > until) {
            break;
        }
        if (currentTagName && currentTagName != TChargableTag::Servicing) {
            TInstant predInstant = timeline[i - 1].GetEventInstant();
            TInstant currentInstant = (timeline[i].GetEventInstant() <= until) ? timeline[i].GetEventInstant() : until;
            TDuration duration = currentInstant - predInstant;
            TSet<TString> tagsInPoint;
            {
                const TCarTagHistoryEvent& ev = *events[timeline[i - 1].GetEventIndex()];
                const THistoryDeviceSnapshot* snapshotCurrent = ev->GetObjectSnapshotAs<THistoryDeviceSnapshot>();
                NDrive::TLocation location;
                if (snapshotCurrent && snapshotCurrent->GetHistoryLocation(location) && location.IsRealtime()) {
                    tagsInPoint = MakeSet(snapshotCurrent->GetLocationTagsArray());
                } else {
                    tagsInPoint.emplace("__undefined");
                }
            }
            durations[currentTagName].first += CalcChargableDuration(predInstant, currentInstant, currentTagName, tagsInPoint);
            durations[currentTagName].second += duration;
        }
        if (timeline[i].GetTimeEvent() != IEventsSession<TCarTagHistoryEvent>::EEvent::CurrentFinish) {
            if (!!(*events[timeline[i].GetEventIndex()])->GetName()) {
                currentTagName = (*events[timeline[i].GetEventIndex()])->GetName();
            }
            const TCarTagHistoryEvent& ev = *events[timeline[i].GetEventIndex()];
            const THistoryDeviceSnapshot* snapshotCurrent = ev->GetObjectSnapshotAs<THistoryDeviceSnapshot>();
            double evMileage;
            bool ignoreOldSnapshots = true;
            if (NDrive::HasServer()) {
                ignoreOldSnapshots = NDrive::GetServer().GetSettings().GetValue<bool>("offers.ignore_old_snapshots").GetOrElse(true);
            }
            if (snapshotCurrent) {
                TInstant since = ev.GetHistoryTimestamp() - TDuration::Hours(24);
                bool hasMileage = ignoreOldSnapshots ? snapshotCurrent->GetMileage(evMileage, since) : snapshotCurrent->GetMileage(evMileage);
                if (hasMileage) {
                    if (!minMileage || *minMileage > evMileage) {
                        minMileage = evMileage;
                    }
                    if (!maxMileage || *maxMileage < evMileage) {
                        maxMileage = evMileage;
                    }
                }
            }
        } else if (NDrive::HasServer()) {
            TRTDeviceSnapshot snapshot = NDrive::GetServerAs<NDrive::IServer>().GetSnapshotsManager().GetSnapshot(GetObjectId());
            double mileageCurrent;
            if (snapshot.GetMileage(mileageCurrent)) {
                if (!minMileage || *minMileage > mileageCurrent) {
                    minMileage = mileageCurrent;
                }
                if (!maxMileage || *maxMileage < mileageCurrent) {
                    maxMileage = mileageCurrent;
                }
            }
        }
    }
    auto servicingResults = TBillingSession::CalcServicingResults(timeline, events, until);
    auto totalServicingMileage = double(0);
    for (auto&& servicingResult : servicingResults) {
        Y_ASSERT(servicingResult.Finish);
        auto duration = servicingResult.Finish - servicingResult.Start;
        if (servicingResult.Info) {
            auto servicingDuration = servicingResult.Info->Finish - servicingResult.Info->Start;
            totalServicingMileage += servicingResult.Info->Mileage;
            if (servicingDuration) {
                durations[TChargableTag::Servicing].first += CalcChargableDuration(servicingResult.Info->Start, servicingResult.Info->Finish, TChargableTag::Servicing, servicingResult.LocationTags);
                durations[TChargableTag::Servicing].second += servicingDuration;
            }

            auto idleDuration = duration - servicingDuration;
            if (idleDuration) {
                durations[TChargableTag::Parking].first += CalcChargableDuration(servicingResult.Start, servicingResult.Info->Start, TChargableTag::Parking, servicingResult.LocationTags);
                durations[TChargableTag::Parking].first += CalcChargableDuration(servicingResult.Info->Finish, servicingResult.Finish, TChargableTag::Parking, servicingResult.LocationTags);
                durations[TChargableTag::Parking].second += idleDuration;
            }
        } else if (duration) {
            durations[TChargableTag::Parking].first += CalcChargableDuration(servicingResult.Start, servicingResult.Finish, TChargableTag::Parking, servicingResult.LocationTags);
            durations[TChargableTag::Parking].second += duration;
        }
    }

    auto ridingStart = ridingInfo.GetSegmentStart(TChargableTag::Riding);
    auto state = MakeAtomicShared<TStandartOfferState>();
    if (ridingStart && GetAcceptance()) {
        double originalPrice = GetPublicOriginalPrice(GetAcceptance().GetPrice(), AcceptanceSegmentName);
        double discountedPrice = ApplyDiscount(originalPrice, true, AcceptanceSegmentName);
        result.AddSegmentInfo(AcceptanceSegmentName, discountedPrice, originalPrice, TDuration::Zero(), 0, 0, discountedPrice);
        state->SetAcceptanceCost(discountedPrice);
    }
    TDuration duration;
    double durationPrice = 0;
    for (auto&& i : durations) {
        const double segmentOriginalPrice = CalculateOriginalPrice(i.second.first, i.first);
        const double segmentPrice = ApplyDiscount(segmentOriginalPrice, true, i.first);
        result.AddSegmentInfo(i.first, segmentPrice, segmentOriginalPrice, i.second.second, 0, 0, segmentPrice);
        durationPrice += segmentPrice;
        duration += i.second.second;
    }
    if (durations.size()) {
        state->SetDuration(duration);
        state->SetDurationPrice(durationPrice);
    }
    if (NeedStandartMileagePricing()) {
        double deltaKm = (maxMileage && minMileage) ? (*maxMileage - *minMileage) : 0;
        if (deltaKm > totalServicingMileage) {
            deltaKm -= totalServicingMileage;
        }
        if (deltaKm * GetPublicOriginalKm()) {
            const double mileageOriginalPrice = GetPublicOriginalKm() * deltaKm;
            const double mileagePrice = ApplyDiscount(mileageOriginalPrice, true, ESessionState::Riding);
            result.AddSegmentInfo(MileageSegmentName, mileagePrice, mileageOriginalPrice, TDuration::Zero(), deltaKm, 0, mileagePrice);

            state->SetMileage(deltaKm);
            state->SetMileagePrice(mileagePrice);
        }
    }
    return state;
}

TDuration TStandartOffer::GetFreeDuration(const TString& tag, const bool publicOnly) const {
    i32 Additional = 0;
    for (auto&& i : Discounts) {
        if (!publicOnly || i.GetVisible()) {
            Additional += i.GetDetails(tag).GetAdditionalTime();
        }
    }
    if (Additional < 0) {
        return TDuration::Zero();
    } else {
        return TDuration::Seconds(Additional);
    }
}

NJson::TJsonValue TStandartOfferState::GetReport(ELocalization locale, const NDrive::IServer& server) const {
    Y_UNUSED(locale);
    Y_UNUSED(server);
    NJson::TJsonValue result = NJson::JSON_MAP;
    if (AcceptanceCost) {
        result.InsertValue("acceptance_cost", TStandartOffer::RoundPrice(*AcceptanceCost));
    }
    if (MileagePrice) {
        JWRITE_DEF(result, "mileage_price", TStandartOffer::RoundPrice(*MileagePrice), 0);
    }
    if (Mileage) {
        result.InsertValue("mileage", *Mileage);
    }
    if (DurationPrice) {
        JWRITE_DEF(result, "duration_price", TStandartOffer::RoundPrice(*DurationPrice), 0);
    }
    if (Duration) {
        result.InsertValue("duration", Duration->Seconds());
    }
    return result;
}

bool TStandartOffer::DeserializeStandartFromProto(const NDrive::NProto::TStandartOffer& info) {
    if (info.HasFullPricesContext()) {
        if (!TFullPricesContext::DeserializePricesContextFromProto(info.GetFullPricesContext())) {
            return false;
        }
    } else {
        MutableParking().SetPriceModelName(info.GetParkingPriceModeling() ? "undefined" : "");
        MutableParking().SetPrice(info.GetPriceParking());
        MutableRiding().SetPriceModelName(info.GetRidingPriceModeling() ? "undefined" : "");
        MutableRiding().SetPrice(info.GetPriceRiding());
        MutableKm().SetPriceModelName(info.GetKmPriceModeling() ? "undefined" : "");
        MutableKm().SetPrice(info.GetPriceKm());
        for (auto&& i : info.GetPriceModelInfo()) {
            MutableRiding().MutablePriceModelInfos().push_back({i.GetName(), i.GetBefore(), i.GetAfter()});
        }
        for (auto&& i : info.GetParkingPriceModelInfo()) {
            MutableParking().MutablePriceModelInfos().push_back({i.GetName(), i.GetBefore(), i.GetAfter()});
        }
    }
    if (!DeserializeFeatures(info.GetFeatures())) {
        return false;
    }
    DebtThreshold = info.GetDebtThreshold();
    DepositAmount = info.GetDepositAmount();
    UseDeposit = info.GetUseDeposit();
    UseDefaultShortDescriptions = info.GetUseDefaultShortDescriptions();
    UseRounding = info.GetUseRounding();
    if (info.HasFuelingEnabled()) {
        FuelingEnabled = info.GetFuelingEnabled();
    }
    if (info.HasInsuranceCost()) {
        InsuranceCost = info.GetInsuranceCost();
    }
    if (info.HasInsuranceType()) {
        InsuranceType = info.GetInsuranceType();
    }
    if (info.HasAgreement()) {
        Agreement = info.GetAgreement();
    }
    if (info.HasFinishAreaTagsFilter() && !!info.GetFinishAreaTagsFilter()) {
        FinishAreaTagsFilter.Reset(new TTagsFilter);
        if (!FinishAreaTagsFilter->DeserializeFromString(info.GetFinishAreaTagsFilter())) {
            return false;
        }
    }
    if (info.HasRidingAreaTagsFilter() && !!info.GetRidingAreaTagsFilter()) {
        RidingAreaTagsFilter.Reset(new TTagsFilter);
        if (!RidingAreaTagsFilter->DeserializeFromString(info.GetRidingAreaTagsFilter())) {
            return false;
        }
    }
    for (auto&& i : info.GetCashbackPercentModelInfo()) {
        CashbackPercentModelInfos.push_back({i.GetName(), i.GetBefore(), i.GetAfter()});
    }
    if (info.HasInheritedTimestamp() || info.HasInheritedRidingPrice() || info.HasInheritedParkingPrice()) {
        InheritedProperties.ConstructInPlace();
    }
    if (info.HasInheritedTimestamp()) {
        InheritedProperties->Timestamp = TInstant::Seconds(info.GetInheritedTimestamp());
    }
    if (info.HasInheritedRidingPrice()) {
        InheritedProperties->RidingPrice = info.GetInheritedRidingPrice();
    }
    if (info.HasInheritedParkingPrice()) {
        InheritedProperties->ParkingPrice = info.GetInheritedParkingPrice();
    }
    for (auto&& i : info.GetDepositAmountModelInfo()) {
        DepositAmountModelInfos.push_back({i.GetName(), i.GetBefore(), i.GetAfter()});
    }
    for (auto&& i : info.GetInsuranceCostModelInfo()) {
        InsuranceCostModelInfos.push_back({i.GetName(), i.GetBefore(), i.GetAfter()});
    }
    return true;
}

bool TStandartOffer::DeserializeFromProto(const NDrive::NProto::TOffer& info) {
    if (!TBase::DeserializeFromProto(info)) {
        return false;
    }
    return DeserializeStandartFromProto(info.GetStandartOffer());
}

namespace {
    // AssignProtoModelInfo helps in assigning model info to proto.
    void AssignProtoModelInfo(NDrive::NProto::TPriceModelInfo* proto, const TPriceModelInfo& value) {
        Y_ENSURE(proto);
        proto->SetName(value.Name);
        proto->SetBefore(value.Before);
        proto->SetAfter(value.After);
    }
}

void TStandartOffer::SerializeStandartToProto(NDrive::NProto::TStandartOffer& info) const {
    *info.MutableFeatures() = SerializeFeatures();
    *info.MutableFullPricesContext() = TFullPricesContext::SerializePricesContextToProto(/*serializeFeatures=*/false);

    info.SetParkingPriceModeling(GetParking().GetPriceModeling());
    info.SetRidingPriceModeling(GetRiding().GetPriceModeling());
    info.SetKmPriceModeling(GetKm().GetPriceModeling());
    info.SetPriceParking(GetParking().GetPrice());
    info.SetPriceRiding(GetRiding().GetPrice());
    info.SetPriceKm(GetKm().GetPrice());
    info.SetDebtThreshold(DebtThreshold);
    info.SetDepositAmount(DepositAmount);
    info.SetFuelingEnabled(FuelingEnabled);
    info.SetUseDeposit(UseDeposit);
    info.SetUseDefaultShortDescriptions(UseDefaultShortDescriptions);
    info.SetUseRounding(UseRounding);
    info.SetInsuranceType(InsuranceType);
    info.SetAgreement(Agreement);
    if (InsuranceCost) {
        info.SetInsuranceCost(*InsuranceCost);
    }
    if (!!FinishAreaTagsFilter && !FinishAreaTagsFilter->IsEmpty()) {
        info.SetFinishAreaTagsFilter(FinishAreaTagsFilter->ToString());
    }
    if (!!RidingAreaTagsFilter && !RidingAreaTagsFilter->IsEmpty()) {
        info.SetRidingAreaTagsFilter(RidingAreaTagsFilter->ToString());
    }
    for (auto&& value : GetRiding().GetPriceModelInfos()) {
        auto* priceModelInfo = info.AddPriceModelInfo();
        priceModelInfo->SetName(value.Name);
        priceModelInfo->SetBefore(value.Before);
        priceModelInfo->SetAfter(value.After);
    }
    if (!GetRiding().GetPriceModelInfos().empty()) {
        info.SetPriceModel(GetRiding().GetPriceModelInfos().back().Name);
    }
    for (auto&& value : GetParking().GetPriceModelInfos()) {
        auto* priceModelInfo = info.AddParkingPriceModelInfo();
        priceModelInfo->SetName(value.Name);
        priceModelInfo->SetBefore(value.Before);
        priceModelInfo->SetAfter(value.After);
    }
    if (!GetParking().GetPriceModelInfos().empty()) {
        info.SetParkingPriceModel(GetParking().GetPriceModelInfos().back().Name);
    }
    for (auto&& value : GetCashbackPercentModelInfos()) {
        AssignProtoModelInfo(info.AddCashbackPercentModelInfo(), value);
    }
    if (InheritedProperties) {
        info.SetInheritedTimestamp(InheritedProperties->Timestamp.Seconds());
        info.SetInheritedRidingPrice(InheritedProperties->RidingPrice);
        info.SetInheritedParkingPrice(InheritedProperties->ParkingPrice);
    }
    for (auto&& value : GetDepositAmountModelInfos()) {
        AssignProtoModelInfo(info.AddDepositAmountModelInfo(), value);
    }
    for (auto&& value : GetInsuranceCostModelInfos()) {
        AssignProtoModelInfo(info.AddInsuranceCostModelInfo(), value);
    }
}

NDrive::NProto::TOffer TStandartOffer::SerializeToProto() const {
    auto info = TBase::SerializeToProto();
    SerializeStandartToProto(*info.MutableStandartOffer());
    return info;
}

void TStandartOffer::ApplyCashbackPercentModel(const NDrive::IOfferModel& model) {
    ui32 cashbackPercent = model.Calc(MutableFeatures());
    CashbackPercentModelInfos.push_back({model.GetName(), (float)GetCashbackPercent(), (float)cashbackPercent});
    SetCashbackPercent(cashbackPercent);
}

void TStandartOffer::ApplyDepositAmountModel(const NDrive::IOfferModel& model) {
    ui32 depositAmount = model.Calc(MutableFeatures());
    DepositAmountModelInfos.push_back({model.GetName(), (float)GetDepositAmount(), (float)depositAmount});
    SetUseDeposit(depositAmount > 0);
    SetDepositAmount(depositAmount);
}

void TStandartOffer::ApplyInsuranceCostModel(const NDrive::IOfferModel& model) {
    auto original = GetRiding().GetPrice() - GetInsuranceCostDef(0);
    i32 insuranceCost = 100 * model.Calc(MutableFeatures());
    InsuranceCostModelInfos.push_back({model.GetName(), (float)GetInsuranceCostDef(0), (float)insuranceCost});
    SetInsuranceCost(insuranceCost);
    MutableRiding().SetPrice(original + insuranceCost);
}

TExpected<bool, NJson::TJsonValue> TStandartOffer::CheckAgreementRequirements(ELocalization locale, const TUserPermissions::TPtr permissions, const NDrive::IServer* server) const {
    auto agreementFuture = BuildAgreement(locale, nullptr, permissions, server);
    agreementFuture.Wait();
    if (agreementFuture.HasException()) {
        return MakeUnexpected<NJson::TJsonValue>(NThreading::GetExceptionInfo(agreementFuture));
    }

    return true;
}

NThreading::TFuture<TString> TStandartOffer::BuildAgreement(ELocalization locale, TAtomicSharedPtr<TCompiledRiding> /*compiledSession*/, TUserPermissions::TPtr permissions, const NDrive::IServer* server) const {
    return NThreading::MakeFuture().Apply([this, locale = locale, server = server, permissions = permissions](const NThreading::TFuture<void>& /*agreementFuture*/) -> TString {
        auto agreement = GetAgreement();
        if (!agreement) {
            auto action = server->GetDriveAPI()->GetUserPermissionManager().GetAction(GetBehaviourConstructorId());
            auto builder = action ? action->GetAs<IOfferBuilderAction>() : nullptr;
            auto customAct = builder ? builder->GetLocalizations().FindPtr("custom_act") : nullptr;
            auto taxiAct = builder ? builder->GetLocalizations().FindPtr("taxi_act") : nullptr;

            if (customAct) {
                agreement = *customAct;
            } else if (taxiAct) {
                agreement = *taxiAct;
            }
        }

        R_ENSURE(agreement, HTTP_INTERNAL_SERVER_ERROR, "cannot get agreement");

        auto agreementSettingKey = TStringBuilder() << "user_app.settings." << agreement;
        auto agreementTemplate = permissions->GetSetting<TString>(agreementSettingKey);
        if (!agreementTemplate) {
            auto agreementSettingSecondaryKey = TStringBuilder() << "user_app.settings.custom_act." << agreement;
            agreementTemplate = permissions->GetSetting<TString>(agreementSettingSecondaryKey);
        }

        R_ENSURE(agreementTemplate, HTTP_INTERNAL_SERVER_ERROR, "cannot get agreement");

        auto localization = server->GetLocalization();
        if (localization) {
            *agreementTemplate = localization->ApplyResources(*agreementTemplate, locale);
        }

        auto tx = server->GetDriveAPI()->BuildTx<NSQL::ReadOnly>();
        FillSharedSessionAgreement(*agreementTemplate, tx, server);
        FillAccountDataAgreement(*agreementTemplate, locale, tx, server);
        FillObjectDataAgreement(*agreementTemplate, tx, server);

        return *agreementTemplate;
    });
}

void TStandartOffer::FillAccountDataAgreement(TString& agreementTemplate, const ELocalization locale, NDrive::TEntitySession& session, const NDrive::IServer* server) const {
    auto&& settings = server->GetSettings();
    if (GetSelectedCharge() && server->GetDriveAPI()->HasBillingManager()) {
        auto&& accountsManager = server->GetDriveAPI()->GetBillingManager().GetAccountsManager();
        auto userAccounts = accountsManager.GetUserAccounts(GetUserId());
        NDrive::NBilling::IBillingAccount::TPtr billingAccount;
        for (auto&& account : userAccounts) {
            if (account && account->GetUniqueName() == GetSelectedCharge() && account->GetType() == NDrive::NBilling::EAccount::Wallet) {
                billingAccount = account;
                break;
            }
        }

        if (billingAccount) {
            auto companyAccount = NDrive::NBilling::IBillingAccount::GetRootAccount(billingAccount);
            auto&& record = companyAccount->GetRecord();
            R_ENSURE(record, HTTP_INTERNAL_SERVER_ERROR, "cannot build organization name " << billingAccount->GetId(), session);
            TString companyStr = companyAccount->GetCompanyName();
            bool isRootAccount = companyAccount == billingAccount;
            if (!companyStr && isRootAccount) {
                companyStr = billingAccount->GetName();
            }

            if (auto limitedRecord = companyAccount->GetRecordAs<NDrive::NBilling::TLimitedAccountRecord>(); limitedRecord && limitedRecord->HasBalanceInfo()) {
                companyStr += TStringBuilder() << " " << server->GetLocalization()->GetLocalString(locale, "agreement.inn");
                SubstGlobal(companyStr, "!INN!", record->GetExternalId());
            }

            if (!companyStr && settings.GetValue<bool>("agreement.organization_check.enabled").GetOrElse(false)) {
                session.SetError(NDrive::MakeError("agreement.INCOMPLETE_ACCOUNT_DATA"));
                R_ENSURE(companyStr, HTTP_INTERNAL_SERVER_ERROR, "cannot build organization name " << billingAccount->GetId(), session);
            }

            SubstGlobal(agreementTemplate, "!COMPANY!", companyStr);
        }
    }

    auto&& userManager = server->GetDriveAPI()->GetUserManager();
    auto&& user = userManager.RestoreUser(GetUserId(), session);
    R_ENSURE(user, HTTP_INTERNAL_SERVER_ERROR, "cannot get user info " << GetSelectedCharge(), session);
    bool isUserNameComplete = user->GetFirstName() && user->GetLastName();
    if (!isUserNameComplete && settings.GetValue<bool>("agreement.user_check.enabled").GetOrElse(false)) {
        session.SetError(NDrive::MakeError("agreement.INCOMPLETE_USER_DATA"));
        R_ENSURE(isUserNameComplete, HTTP_INTERNAL_SERVER_ERROR, "cannot build user name " << GetUserId(), session);
    }

    SubstGlobal(agreementTemplate, "!NAME!", user->GetFullName());
}

void TStandartOffer::FillObjectDataAgreement(TString& agreementTemplate, NDrive::TEntitySession& session, const NDrive::IServer* server) const {
    auto&& carsManager = server->GetDriveAPI()->GetCarManager();
    auto&& modelsManager = server->GetDriveAPI()->GetModelsDB();

    auto object = carsManager.GetObject(GetObjectId());
    R_ENSURE(object, HTTP_INTERNAL_SERVER_ERROR, "cannot get car " << GetObjectId(), session);
    R_ENSURE(object->GetModel(), HTTP_INTERNAL_SERVER_ERROR, "empty model code " << GetObjectId(), session);
    auto matchedModels = modelsManager.FetchInfo(object->GetModel(), session);
    auto modelInfo = matchedModels.GetResultPtr(object->GetModel());
    R_ENSURE(modelInfo, HTTP_INTERNAL_SERVER_ERROR, "error matching models " << object->GetModel() << " " << GetObjectId(), session);

    auto&& modelStr = modelInfo->GetName();
    R_ENSURE(modelStr, HTTP_INTERNAL_SERVER_ERROR, "empty model name " << GetObjectId(), session);
    auto&& numberStr = object->GetNumber();
    R_ENSURE(!numberStr.Empty(), HTTP_INTERNAL_SERVER_ERROR, "empty number " << GetObjectId(), session);

    SubstGlobal(agreementTemplate, "!MODEL!", modelStr);
    SubstGlobal(agreementTemplate, "!NUMBER!", numberStr);
}

void TStandartOffer::FillSharedSessionAgreement(TString& agreementTemplate, NDrive::TEntitySession& session, const NDrive::IServer* server) const {
    if (const auto& sharedSessionId = GetSharedSessionId()) {
        auto optionalSharedSession = server->GetDriveAPI()->GetSessionManager().GetSession(sharedSessionId, session);
        R_ENSURE(optionalSharedSession, {}, "cannot GetSession " << sharedSessionId, session);
        auto sharedSession = *optionalSharedSession;
        if (sharedSession) {
            auto owner = server->GetDriveAPI()->GetUserManager().GetCachedObject(sharedSession->GetUserId());
            R_ENSURE(owner, HTTP_INTERNAL_SERVER_ERROR, "cannot get owner " << sharedSession->GetUserId(), session);
            SubstGlobal(agreementTemplate, "!OWNER!", owner->GetFullName());
        }
    }
}
