#pragma once

#include <drive/backend/offers/abstract.h>
#include <drive/backend/offers/discount.h>
#include <drive/backend/offers/price/model.h>

#include <drive/backend/proto/offer.pb.h>
#include <drive/backend/tags/tags_filter.h>

#include <rtline/library/json/cast.h>
#include <rtline/library/time_restriction/time_restriction.h>

#include <util/datetime/base.h>

namespace NDrive::NProto {
    class TOffer;
    class TStandartOffer;
}

inline const TString CorpAgreementId = "act_corp";
inline const TString DefaultAgreementId = "act_default";

inline const TString InsuranceTypeId = "insurance_type";
inline const TString FullInsuranceType = "full";
inline const TString StandartInsuranceType = "standart";

inline const TString IntercityOfferTag = "intercity";

class TStandartOfferState: public IOfferState {
private:
    R_OPTIONAL(double, AcceptanceCost);
    R_OPTIONAL(double, Mileage);
    R_OPTIONAL(double, MileagePrice);
    R_OPTIONAL(TDuration, Duration);
    R_OPTIONAL(double, DurationPrice);

public:
    NJson::TJsonValue GetReport(ELocalization locale, const NDrive::IServer& server) const override;
};

class TStandartOffer
    : public IOfferWithDiscounts
    , public TFullPricesContext
{
private:
    using TBase = IOfferWithDiscounts;

public:
    struct TInheritedProperties {
        TInstant Timestamp;
        ui32 RidingPrice = 0;
        ui32 ParkingPrice = 0;
    };

public:
    R_OPTIONAL(TInheritedProperties, InheritedProperties);
    R_OPTIONAL(i32, InsuranceCost);
    R_FIELD(TString, Agreement);
    R_FIELD(TString, InsuranceType, StandartInsuranceType);
    R_FIELD(ui32, DebtThreshold, 0);
    R_FIELD(ui32, DepositAmount, 0);
    R_FIELD(TPriceModelInfos, DepositAmountModelInfos);
    R_FIELD(bool, FuelingEnabled, true);
    R_FIELD(bool, UseDeposit, true);
    R_FIELD(bool, UseDefaultShortDescriptions, true);
    R_FIELD(bool, UseRounding, false);
    R_FIELD(TPriceModelInfos, CashbackPercentModelInfos);
    R_FIELD(TPriceModelInfos, InsuranceCostModelInfos);

    THolder<TTagsFilter> FinishAreaTagsFilter;
    THolder<TTagsFilter> RidingAreaTagsFilter;

private:
    static TFactory::TRegistrator<TStandartOffer> Registrator;

    TDuration GetFreeDurationChecked(const TString& tagName) const;

protected:
    virtual bool NeedStandartMileagePricing() const {
        return true;
    }

protected:
    TString DoBuildCalculationDescription(ELocalization locale, const TFullCompiledRiding& fcr, const NDrive::IServer& server) const override;
    TString DoBuildCommonPricesDescription(const TFullCompiledRiding& fcr, const NDrive::IServer& server) const override;

    bool DeserializeStandartFromProto(const NDrive::NProto::TStandartOffer& info);

    void SerializeStandartToProto(NDrive::NProto::TStandartOffer& info) const;

    virtual EDriveSessionResult DoCheckSession(const TUserPermissions& permissions, NDrive::TEntitySession& session, const NDrive::IServer* server, bool onPerform) const override;
    virtual TString DoFormDescriptionElement(const TString& value, ELocalization locale, const ILocalization* localization) const override;
    virtual TOfferStatePtr DoCalculate(const TVector<IEventsSession<TCarTagHistoryEvent>::TTimeEvent>& timeline, const TVector<TAtomicSharedPtr<TCarTagHistoryEvent>>& events, const TInstant& until, const TRidingInfo& ridingInfo, TOfferPricing& result) const override;
    NJson::TJsonValue DoBuildJsonReport(const TReportOptions& options, const ICommonOfferBuilderAction* constructor, const NDrive::IServer& server) const override;

protected:
    static const NUnistat::TIntervals PriceIntervals;
    static const NUnistat::TIntervals PriceDeltaIntervals;

public:
    TStandartOffer() = default;
    TStandartOffer(TStandartOffer&& offer) = default;
    TStandartOffer(const TFullPricesContext& fpContext)
        : TFullPricesContext(fpContext)
    {
    }

    ui32 GetPublicDiscountedAcceptance(ui32 precision = 1) const;
    ui32 GetPublicDiscountedParking(const ui32 precision = 1) const;
    ui32 GetPublicDiscountedRiding(const ui32 precision = 1) const;
    ui32 GetPublicDiscountedKm(const ui32 precision = 1) const;

    ui32 GetPublicOriginalParking(const ui32 precision = 1) const;
    ui32 GetPublicOriginalRiding(const ui32 precision = 1) const;
    ui32 GetPublicOriginalKm(const ui32 precision = 1) const;

    const TTagsFilter& GetFinishAreaTagsFilter() const;
    TStandartOffer& SetFinishAreaTagsFilter(const TTagsFilter& filter);

    const TTagsFilter& GetRidingAreaTagsFilter() const;
    TStandartOffer& SetRidingAreaTagsFilter(const TTagsFilter& filter);

    void SetPricingFrom(const TStandartOffer* base);

    virtual TVector<TString> GetDefaultShortDescription(ELocalization locale, NDriveSession::TReportTraits traits, const ILocalization& localization) const;
    virtual TMaybe<ui32> GetReportedAcceptancePrice() const;
    virtual TString GetPushReport(ELocalization locale, const IServerBase* server) const override;
    virtual ui32 GetDeposit() const override {
        return DepositAmount;
    }

    virtual TDuration GetFreeDuration(const TString& tag, const bool publicOnly = false) const override;
    virtual bool GetFreeAndPricedDurations(TDuration& freeDuration, TDuration& pricedDuration, const TMap<TString, TOfferSegment>& segments) const override;

    virtual void FillBillPricing(TBill& bill, const TOfferPricing& pricing, TOfferStatePtr /*segmentState*/, ELocalization locale, const NDrive::IServer* server) const override;
    virtual void PatchSessionReport(NJson::TJsonValue& result, NDriveSession::TReportTraits traits, ELocalization locale,
                                    const NDrive::IServer& server, const TOfferSessionPatchData& patchData) const override;

    virtual TDuration CalcChargableDuration(const TInstant predInstant, const TInstant nextInstant, const TString& tagName, const TSet<TString>& tagsInPoint) const override;
    virtual TDuration GetFreeTime(const TDuration usedDuration, const TString& tagName) const override {
        return GetFreeDuration(tagName) - usedDuration;
    }

    virtual double CalculateOriginalPrice(const TDuration stateDuration, const TString& tagName) const override;

    virtual TString GetTypeName() const override {
        return GetTypeNameStatic();
    }

    virtual TString GetDefaultGroupName() const override {
        return {};
    }

    virtual bool HasFueling() const override {
        return FuelingEnabled;
    }

    static TString GetTypeNameStatic() {
        return "standart_offer";
    }

    TExpected<bool, NJson::TJsonValue> CheckAgreementRequirements(ELocalization locale, const TUserPermissions::TPtr permissions, const NDrive::IServer* server) const override;
    NThreading::TFuture<TString> BuildAgreement(ELocalization locale, TAtomicSharedPtr<TCompiledRiding> compiledSession, TUserPermissions::TPtr permissions, const NDrive::IServer* server) const override;
    void FillAccountDataAgreement(TString& agreementTemplate, const ELocalization locale, NDrive::TEntitySession& session, const NDrive::IServer* server) const;
    void FillObjectDataAgreement(TString& agreementTemplate, NDrive::TEntitySession& session, const NDrive::IServer* server) const;
    void FillSharedSessionAgreement(TString& agreementTemplate, NDrive::TEntitySession& session, const NDrive::IServer* server) const;

    // ApplyCashbackPercentModel applies model for cashback percent and appends record to model info.
    void ApplyCashbackPercentModel(const NDrive::IOfferModel& model);
    // ApplyDepositAmountModel applies model for deposit amount and appends record to model info.
    void ApplyDepositAmountModel(const NDrive::IOfferModel& model);
    // ApplyInsuranceCostModel applies model for insurance cost and appends record to model info.
    void ApplyInsuranceCostModel(const NDrive::IOfferModel& model);

    void DeepCopyFrom(const TStandartOffer& offer) {
        Y_ENSURE_BT(DeserializeFromProto(offer.SerializeToProto()));
    }

    bool DeserializeFromProto(const NDrive::NProto::TOffer& info) override;
    NDrive::NProto::TOffer SerializeToProto() const override;
};
