#include "price.h"

#include <drive/backend/proto/offer.pb.h>

#include <rtline/util/types/interval.h>

#include <library/cpp/mediator/global_notifications/system_status.h>

#include <util/stream/file.h>

#include <cmath>

bool TConstantPriceConfig::DeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    JREAD_INT(jsonInfo, "price", PriceForMinute);
    return true;
}

NJson::TJsonValue TConstantPriceConfig::SerializeToJson() const {
    NJson::TJsonValue result(NJson::JSON_MAP);
    JWRITE(result, "price", PriceForMinute);
    return result;
}

void TConstantPriceConfig::Init(const TYandexConfig::Section* section) {
    PriceForMinute = section->GetDirectives().Value("PriceForMinute", PriceForMinute);
}

void TConstantPriceConfig::ToString(IOutputStream& os) const {
    os << "PriceForMinute: " << PriceForMinute << Endl;
}

IPriceCalculator::TPtr TConstantPriceConfig::Construct() const {
    return new TConstantPrice(PriceForMinute);
}

TConstantPriceConfig::TFactory::TRegistrator<TConstantPriceConfig> TConstantPriceConfig::Registrator("constant");

i32 TPriceByTimeConfig::GetBasePrice(const TInstant instantStart) const {
    constexpr ui32 secondsInWeek = 7 * 24 * 60 * 60;
    const ui32 secondInWeekStart = (instantStart.Seconds() - 4 * 24 * 60 * 60 + TimeShift) % secondsInWeek;
    const auto pred = [](const ui32 l, const TPriceInterval& r) ->bool {
        return l < r.GetSecondsStart();
    };
    auto it = std::upper_bound(PriceBySegments.begin(), PriceBySegments.end(), secondInWeekStart, pred);
    if (it != PriceBySegments.begin()) {
        --it;
    }
    return it->GetPrice();
}

i32 TPriceByTimeConfig::CalcPrice(const TInstant instantStart, const TInstant instantFinish) const {
    double result = 0;
    Y_ENSURE_BT(instantStart <= instantFinish);
    constexpr ui32 secondsInWeek = 7 * 24 * 60 * 60;
    const ui32 secondInWeekStart = (instantStart.Seconds() + 3 * 24 * 60 * 60) % secondsInWeek;
    const ui32 secondInWeekFinish = secondInWeekStart + (instantFinish.Seconds() - instantStart.Seconds());
    const auto pred = [](const ui32 l, const TPriceInterval& r) ->bool {
        return l < r.GetSecondsStart();
    };
    auto it = std::upper_bound(PriceBySegments.begin(), PriceBySegments.end(), secondInWeekStart, pred);
    if (it != PriceBySegments.begin()) {
        --it;
    }

    TInterval<ui32> fullInt(secondInWeekStart, secondInWeekFinish);
    if (!ChangeOnMoving) {
        return fullInt.GetLength() / 60.0 * it->GetPrice();
    }

    ui32 currentShiftSeconds = 0;
    while (currentShiftSeconds + it->GetSecondsStart() < secondInWeekFinish) {
        const ui32 secondsStartInterval = currentShiftSeconds + it->GetSecondsStart();
        const ui32 currentPrice = it->GetPrice();
        auto itNext = it + 1;
        if (itNext == PriceBySegments.end()) {
            it = PriceBySegments.begin();
            currentShiftSeconds += secondsInWeek;
        } else {
            it = itNext;
        }
        const ui32 secondsFinishInterval = currentShiftSeconds + it->GetSecondsStart();
        TInterval<ui32> priceInt(secondsStartInterval, secondsFinishInterval);
        TInterval<ui32> crossInterval;
        if (fullInt.Intersection(priceInt, crossInterval)) {
            result += crossInterval.GetLength() / 60.0 * currentPrice;
        }
    }

    return result;
}

bool TPriceByTimeConfig::DeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    JREAD_INT(jsonInfo, "time_shift", TimeShift);
    if (!jsonInfo["segments"].IsArray()) {
        return false;
    }
    for (auto&& i : jsonInfo["segments"].GetArraySafe()) {
        TPriceInterval interval(0, 0);
        if (!interval.DeserializeFromJson(i)) {
            return false;
        }
        PriceBySegments.emplace_back(interval);
    }
    return true;
}

NJson::TJsonValue TPriceByTimeConfig::SerializeToJson() const {
    NJson::TJsonValue result(NJson::JSON_MAP);
    JWRITE(result, "time_shift", TimeShift);
    NJson::TJsonValue& segments = result.InsertValue("segments", NJson::JSON_ARRAY);
    for (auto&& i : PriceBySegments) {
        segments.AppendValue(i.SerializeToJson());
    }
    return result;
}

void TPriceByTimeConfig::InitFromFile(const TFsPath& path) {
    AssertCorrectConfig(path.Exists(), "Incorrect path for table with prices: %s", PathForTable.data());
    TFileInput fi(path);
    TString line;
    ui32 idx = 0;
    if (SecondsInSegment) {
        AssertCorrectConfig(24 * 60 * 60 % SecondsInSegment == 0, "Incorrect segment size: %d seconds", SecondsInSegment);
    }
    i32 predTime = -1;
    while (fi.ReadLine(line)) {
        ui32 tsInDay = (idx++) * SecondsInSegment;
        TStringBuf r;
        if (SecondsInSegment == 0) {
            TStringBuf sb(line.data(), line.size());
            TStringBuf l;
            sb.Split(',', l, r);
            TStringBuf h;
            TStringBuf m;
            l.Split(':', h, m);
            ui32 hours;
            ui32 minutes;
            AssertCorrectConfig(TryFromString<ui32>(h, hours), "cannot parse hours in prices file line (%s)", h.data());
            AssertCorrectConfig(TryFromString<ui32>(m, minutes), "cannot parse minutes in prices file line (%s)", m.data());
            AssertCorrectConfig(hours < 24, "Incorrect hours");
            AssertCorrectConfig(minutes < 60, "Incorrect minutes");
            tsInDay = (hours * 60 + minutes) * 60;
        } else {
            r = TStringBuf(line.data(), line.size());
        }
        AssertCorrectConfig((i32)predTime < (i32)tsInDay, "Incorrect prices table");
        AssertCorrectConfig(tsInDay < 7 * 24 * 60 * 60, "Incorrect prices table (too many seconds)");
        predTime = tsInDay;
        if (idx == 1) {
            AssertCorrectConfig(tsInDay == 0, "incorrect time of day for first line: %d", tsInDay);
        }
        TVector<double> values;
        AssertCorrectConfig(TryParseStringToVector(r, values, ',', false), "Cannot parse line %d in table '%s'", idx, PathForTable.data());
        AssertCorrectConfig(values.size() == 7, "Incorrect days of week count in table: '%s' (line %d)", PathForTable.data(), idx);
        for (ui32 i = 0; i < values.size(); ++i) {
            PriceBySegments.push_back(TPriceInterval(tsInDay + i * (24 * 60 * 60), round(values[i] * 100)));
        }
    }

    std::sort(PriceBySegments.begin(), PriceBySegments.end());
    for (auto&& i : PriceBySegments) {
        DEBUG_LOG << i.GetSecondsStart() << ":" << i.GetPrice() << Endl;
    }
}

void TPriceByTimeConfig::Init(const TYandexConfig::Section* section) {
    TimeShift = section->GetDirectives().Value("TimeShift", TimeShift);
    PathForTable = section->GetDirectives().Value("PathForTable", PathForTable);
    SecondsInSegment = section->GetDirectives().Value("SecondsInSegment", SecondsInSegment);
    TFsPath path(PathForTable);
    InitFromFile(path);
}

void TPriceByTimeConfig::ToString(IOutputStream& os) const {
    os << "PathForTable: " << PathForTable << Endl;
    os << "SecondsInSegment: " << SecondsInSegment << Endl;
    os << "TimeShift: " << TimeShift << Endl;
}

IPriceCalculator::TPtr TPriceByTimeConfig::Construct() const {
    return new TPriceByTime(*this);
}

TPriceByTimeConfig::TFactory::TRegistrator<TPriceByTimeConfig> TPriceByTimeConfig::Registrator("by_time");

TSet<TString> IPriceCalculatorConfig::GetTypes() {
    TSet<TString> keys;
    TFactory::GetRegisteredKeys(keys);
    return keys;
}

bool TPriceReduceAreaIncreaseLinearScheme::CheckScheme() const {
    if (!TLinearScheme::CheckScheme()) {
        return false;
    }
    if (GetPoints().front().GetValue() > 1) {
        return false;
    }
    if (GetPoints().back().GetValue() <= 0) {
        return false;
    }
    for (ui32 i = 1; i < GetPoints().size(); ++i) {
        const double alpha =  - (GetPoints()[i].GetValue() - GetPoints()[i - 1].GetValue()) / (GetPoints()[i].GetArg() - GetPoints()[i - 1].GetArg());
        if (alpha < 0) {
            return false;
        }
        if (alpha * GetPoints()[i].GetArg() > GetPoints()[i].GetValue()) {
            return false;
        }
    }
    return true;
}

const TLinearScheme::TBorderPoliciesSet TLinearScheme::FullBorderPoliciesSet = Max<ui32>();

NDrive::NProto::TLinearSchemePoint TLinearScheme::TKVPoint::SerializeToProto() const {
    NDrive::NProto::TLinearSchemePoint result;
    result.SetArg(Arg);
    result.SetValue(Value);
    return result;
}

bool TLinearScheme::TKVPoint::DeserializeFromProto(const NDrive::NProto::TLinearSchemePoint& proto) {
    Arg = proto.GetArg();
    Value = proto.GetValue();
    return true;
}

TMaybe<double> TLinearScheme::GetValue(const double arg) const {
    if (Points.empty()) {
        return {};
    }
    if (arg < Points.front().GetArg()) {
        if (BorderPolicy & (ui32)EBorderPolicy::ProvideLeft) {
            return Points.front().GetValue();
        } else {
            return {};
        }
    }
    if (arg > Points.back().GetArg()) {
        if (BorderPolicy & (ui32)EBorderPolicy::ProvideRight) {
            return Points.back().GetValue();
        } else {
            return {};
        }
    }
    for (ui32 i = 1; i < Points.size(); ++i) {
        if (arg <= Points[i].GetArg()) {
            if (Points[i].GetArg() == Points[i - 1].GetArg()) {
                return Points[i].GetValue();
            } else if (Approximation == EApproximation::Linear) {
                const double alpha = (arg - Points[i - 1].GetArg()) / (Points[i].GetArg() - Points[i - 1].GetArg());
                return Points[i].GetValue() * alpha + Points[i - 1].GetValue() * (1 - alpha);
            } else {
                return Points[i - 1].GetValue();
            }
        }
    }
    ERROR_LOG << "Incorrect interpolation for " << arg << Endl;
    return {};
}

NJson::TJsonValue TLinearScheme::SerializeToJson() const {
    NJson::TJsonValue result(NJson::JSON_MAP);
    TJsonProcessor::WriteAsString(result, "approximation", Approximation);
    TJsonProcessor::Write<TBorderPoliciesSet>(result, "borders_policy", BorderPolicy, FullBorderPoliciesSet);
    NJson::TJsonValue& pointsJson = result.InsertValue("points", NJson::JSON_ARRAY);
    for (auto&& i : Points) {
        pointsJson.AppendValue(i.SerializeToJson());
    }
    return result;
}

bool TLinearScheme::DeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    const NJson::TJsonValue::TArray* arr;
    if (!jsonInfo["points"].GetArrayPointer(&arr)) {
        return false;
    }

    if (!TJsonProcessor::Read(jsonInfo, "borders_policy", BorderPolicy)) {
        return false;
    }
    if (!TJsonProcessor::ReadFromString("approximation", jsonInfo, Approximation)) {
        return false;
    }

    for (auto&& p : *arr) {
        TKVPoint kvp;
        if (!kvp.DeserializeFromJson(p)) {
            return false;
        }
        Points.emplace_back(std::move(kvp));
    }
    return CheckScheme();
}

NDrive::NProto::TLinearScheme TLinearScheme::SerializeToProto() const {
    NDrive::NProto::TLinearScheme result;
    result.SetBorderPolicy(BorderPolicy);
    result.SetApproximation(::ToString(Approximation));
    for (auto&& p : Points) {
        *result.AddPoints() = p.SerializeToProto();
    }
    return result;
}

bool TLinearScheme::DeserializeFromProto(const NDrive::NProto::TLinearScheme& proto) {
    SetBorderPolicy(proto.GetBorderPolicy());
    if (proto.HasApproximation() && !TryFromString<EApproximation>(proto.GetApproximation(), Approximation)) {
        return false;
    }
    for (auto&& i : proto.GetPoints()) {
        TKVPoint p;
        if (!p.DeserializeFromProto(i)) {
            return false;
        }
        Points.emplace_back(std::move(p));
    }
    return CheckScheme();
}

bool TLinearScheme::CheckScheme() const {
    if (Points.empty()) {
        return false;
    }
    for (ui32 i = 1; i < Points.size(); ++i) {
        if (Points[i].GetArg() <= Points[i - 1].GetArg()) {
            return false;
        }
    }
    return true;
}

NDrive::TScheme TLinearScheme::GetScheme() {
    NDrive::TScheme result;
    result.Add<TFSArray>("points", "Набор точек").SetElement(TKVPoint::GetScheme());
    result.Add<TFSVariants>("approximation", "Тип аппроксимации").InitVariants<EApproximation>();
    return result;
}
