#pragma once

#include <drive/backend/abstract/frontend.h>

#include <drive/library/cpp/scheme/scheme.h>

#include <rtline/util/json_processing.h>
#include <rtline/util/types/accessor.h>

#include <util/datetime/base.h>
#include <util/generic/ptr.h>

namespace NDrive::NProto {
    class TLinearScheme;
    class TLinearSchemePoint;
}

enum class ELimitViolationPolicy {
    ChangeValueAsLimit,
    IgnoreValue,
    Error
};

enum class ELimitedValueOrigination {
    ChangeValueAsLimit,
    ValueIgnored,
    IncorrectValue,
    Success
};

template <class T>
class TLimitedFraction {
private:
    R_FIELD(double, Fraction, 0);
    R_FIELD(T, Min, T());
    R_FIELD(T, Max, T());
    R_FIELD(ELimitViolationPolicy, LimitViolationMinPolicy, ELimitViolationPolicy::ChangeValueAsLimit);
    R_FIELD(ELimitViolationPolicy, LimitViolationMaxPolicy, ELimitViolationPolicy::ChangeValueAsLimit);

private:
    ELimitedValueOrigination GetLimitedValueOrigination(const ELimitViolationPolicy policy) const {
        switch (policy) {
            case ELimitViolationPolicy::ChangeValueAsLimit:
                return ELimitedValueOrigination::ChangeValueAsLimit;
            case ELimitViolationPolicy::IgnoreValue:
                return ELimitedValueOrigination::ValueIgnored;
            case ELimitViolationPolicy::Error:
                return ELimitedValueOrigination::IncorrectValue;
        }
    }

public:
    ELimitedValueOrigination Calc(const T value, T& result) const {
        const T resultValue = value * Fraction;
        if (resultValue < Min) {
            if (LimitViolationMinPolicy == ELimitViolationPolicy::ChangeValueAsLimit) {
                result = Min;
            }
            return GetLimitedValueOrigination(LimitViolationMinPolicy);
        } else if (resultValue > Max) {
            if (LimitViolationMaxPolicy == ELimitViolationPolicy::ChangeValueAsLimit) {
                result = Max;
            }
            return GetLimitedValueOrigination(LimitViolationMaxPolicy);
        }
        result = resultValue;
        return ELimitedValueOrigination::Success;
    }

    TMaybe<T> Calc(const T value) const {
        const T resultValue = value * Fraction;
        if (resultValue < Min) {
            if (LimitViolationMinPolicy == ELimitViolationPolicy::ChangeValueAsLimit) {
                return Min;
            }
            return {};
        } else if (resultValue > Max) {
            if (LimitViolationMaxPolicy == ELimitViolationPolicy::ChangeValueAsLimit) {
                return Max;
            }
            return {};
        }
        return resultValue;
    }

    template <class TMaybePolicy>
    ELimitedValueOrigination Calc(const T value, TMaybe<T, TMaybePolicy>& result) const {
        T resultValue;
        const ELimitedValueOrigination returnValue = Calc(value, result);
        switch (returnValue) {
            case ELimitedValueOrigination::ChangeValueAsLimit:
            case ELimitedValueOrigination::Success:
                result = resultValue;
                break;
            case ELimitedValueOrigination::IncorrectValue:
            case ELimitedValueOrigination::ValueIgnored:
                break;
        }
        return returnValue;
    }

    TLimitedFraction() = default;
    TLimitedFraction(const double frq, const T min, const T max)
        : Fraction(frq)
        , Min(min)
        , Max(max)
    {
    }


    bool Deserialize(const TString& limitsInfo);

    static void AddToScheme(NDrive::TScheme& result, const TString& prefixExt) {
        const TString prefix = (!prefixExt || prefixExt.EndsWith("_")) ? prefixExt : (prefixExt + "_");
        result.Add<TFSNumeric>(prefix + "fraction").SetMin(0);
        result.AddDetect<T>(prefix + "min");
        result.AddDetect<T>(prefix + "max");
        result.Add<TFSVariants>(prefix + "min_violation_policy").InitVariants<ELimitViolationPolicy>().SetDefault(::ToString(ELimitViolationPolicy::ChangeValueAsLimit));
        result.Add<TFSVariants>(prefix + "max_violation_policy").InitVariants<ELimitViolationPolicy>().SetDefault(::ToString(ELimitViolationPolicy::ChangeValueAsLimit));
    }

    static NDrive::TScheme GetScheme() {
        NDrive::TScheme result;
        AddToScheme(result, "");
        return result;
    }

    bool DeserializeFromJson(const NJson::TJsonValue& jsonInfo, const TString& prefixExt = "") {
        const TString prefix = (!prefixExt || prefixExt.EndsWith("_")) ? prefixExt : (prefixExt + "_");
        if (
            TJsonProcessor::Read(jsonInfo, prefix + "fraction", Fraction) &&
            TJsonProcessor::Read(jsonInfo, prefix + "min", Min) &&
            TJsonProcessor::Read(jsonInfo, prefix + "max", Max) &&
            TJsonProcessor::ReadFromString(prefix + "min_violation_policy", jsonInfo, LimitViolationMinPolicy) &&
            TJsonProcessor::ReadFromString(prefix + "max_violation_policy", jsonInfo, LimitViolationMaxPolicy)
        ) {
            return Min <= Max;
        }
        return false;
    }

    void SerializeToJson(NJson::TJsonValue& result, const TString& prefixExt = "") const {
        const TString prefix = (!prefixExt || prefixExt.EndsWith("_")) ? prefixExt : (prefixExt + "_");
        TJsonProcessor::Write(result, prefix + "fraction", Fraction);
        TJsonProcessor::Write(result, prefix + "min", Min);
        TJsonProcessor::Write(result, prefix + "max", Max);
        TJsonProcessor::WriteAsString(result, prefix + "min_violation_policy", LimitViolationMinPolicy);
        TJsonProcessor::WriteAsString(result, prefix + "max_violation_policy", LimitViolationMaxPolicy);
    }

    NJson::TJsonValue SerializeToJson(const TString& prefixExt = "") const {
        NJson::TJsonValue result(NJson::JSON_MAP);
        SerializeToJson(result, prefixExt);
        return result;
    }
};

class ILinearScheme {
public:
    virtual ~ILinearScheme() = default;
    virtual TMaybe<double> GetValue(const double arg) const = 0;
    virtual NJson::TJsonValue SerializeToJson() const = 0;
    virtual bool DeserializeFromJson(const NJson::TJsonValue& jsonInfo) = 0;
    virtual TMaybe<double> GetMaxValue() const = 0;
    virtual TMaybe<double> GetMinValue() const = 0;
    virtual NDrive::NProto::TLinearScheme SerializeToProto() const = 0;
};

template <class TValue, class THeaderRow, class THeaderCol = THeaderRow>
class TNamedValuesTable {
private:
    TVector<THeaderRow> Rows;
    TVector<THeaderCol> Cols;
    TVector<TMaybe<TValue>> Values;

public:
    bool IsInitialized() const {
        return Rows.size() && Cols.size();
    }

    TMaybe<TValue> GetValue(const THeaderRow& hRow, const THeaderCol& hCol) const {
        TMaybe<ui32> idxRow;
        for (ui32 i = 0; i < Rows.size(); ++i) {
            if (hRow == Rows[i]) {
                idxRow = i;
                break;
            }
        }

        TMaybe<ui32> idxCol;
        for (ui32 i = 0; i < Cols.size(); ++i) {
            if (hCol == Cols[i]) {
                idxCol = i;
                break;
            }
        }
        if (!idxCol || !idxRow) {
            return {};
        }
        const ui32 idxValue = *idxRow * Cols.size() + *idxCol;
        if (idxValue >= Values.size()) {
            return {};
        }
        return Values[idxValue];
    }

    static TFSStructure& AddToScheme(NDrive::TScheme& scheme, const TString& name, const TString& description = "", const ui32 orderIdx = Max<ui32>()) {
        NDrive::TScheme structure;
        structure.Add<TFSString>("rows", "Заголовки строк");
        structure.Add<TFSString>("cols", "Заголовки столбцов");
        structure.Add<TFSString>("values", "Значения ('-' -- не заполнен)");
        TFSStructure& result = scheme.Add<TFSStructure>(name, description, orderIdx);
        result.SetVisualType(TFSStructure::EVisualType::Table).SetStructure(std::move(structure));
        return result;
    }

    bool DeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
        if (!TJsonProcessor::ReadContainer(jsonInfo, "rows", Rows, true)) {
            return false;
        }
        if (!TJsonProcessor::ReadContainer(jsonInfo, "cols", Cols, true)) {
            return false;
        }
        if (!TJsonProcessor::ReadContainerMaybe(jsonInfo, "values", Values, true)) {
            return false;
        }
        return true;
    }

    NJson::TJsonValue SerializeToJson() const {
        NJson::TJsonValue result = NJson::JSON_MAP;
        TJsonProcessor::WriteContainerString(result, "rows", Rows);
        TJsonProcessor::WriteContainerString(result, "cols", Cols);
        TJsonProcessor::WriteContainerStringMaybe(result, "values", Values);
        return result;
    }
};

class TLinearScheme: public ILinearScheme {
public:
    enum class EBorderPolicy: ui32 {
        ProvideLeft = 1,
        ProvideRight = 1 << 1
    };
    using TBorderPoliciesSet = ui32;

public:
    static const TBorderPoliciesSet FullBorderPoliciesSet;

    enum class EApproximation {
        Linear /* "linear" */,
        ConstLeft /* "const_left" */
    };

    class TKVPoint {
    private:
        R_FIELD(double, Arg, 0);
        R_FIELD(double, Value, 0);

    public:
        TKVPoint() = default;
        TKVPoint(const double x, const double y)
            : Arg(x)
            , Value(y)
        {
        }

        NJson::TJsonValue SerializeToJson() const {
            NJson::TJsonValue result;
            result.InsertValue("x", Arg);
            result.InsertValue("y", Value);
            return result;
        }

        bool DeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
            return TJsonProcessor::Read(jsonInfo, "x", Arg) && TJsonProcessor::Read(jsonInfo, "y", Value);
        }

        NDrive::NProto::TLinearSchemePoint SerializeToProto() const;
        bool DeserializeFromProto(const NDrive::NProto::TLinearSchemePoint& proto);

        static NDrive::TScheme GetScheme() {
            NDrive::TScheme result;
            result.Add<TFSNumeric>("x");
            result.Add<TFSNumeric>("y");
            return result;
        }
    };

private:
    R_FIELD(TBorderPoliciesSet, BorderPolicy, FullBorderPoliciesSet);
    R_FIELD(TVector<TKVPoint>, Points);
    R_FIELD(EApproximation, Approximation, EApproximation::Linear);

public:
    TLinearScheme() = default;
    TLinearScheme(const double v) {
        Points.emplace_back(TKVPoint(0, v));
    }

    virtual TMaybe<double> GetValue(const double arg) const override;

    virtual TMaybe<double> GetMinValue() const override {
        TMaybe<double> result;
        for (auto&& i : Points) {
            if (!result || *result > i.GetValue()) {
                result = i.GetValue();
            }
        }
        return result;
    }

    virtual TMaybe<double> GetMaxValue() const override {
        TMaybe<double> result;
        for (auto&& i : Points) {
            if (!result || *result < i.GetValue()) {
                result = i.GetValue();
            }
        }
        return result;
    }

    virtual NJson::TJsonValue SerializeToJson() const override;
    virtual bool DeserializeFromJson(const NJson::TJsonValue& jsonInfo) override;

    NDrive::NProto::TLinearScheme SerializeToProto() const override;
    virtual bool DeserializeFromProto(const NDrive::NProto::TLinearScheme& proto);

    virtual bool CheckScheme() const;

    static NDrive::TScheme GetScheme();
};

class TPriceReduceAreaIncreaseLinearScheme: public TLinearScheme {
public:
    virtual bool CheckScheme() const override;
};

class IPriceCalculator {
public:
    using TPtr = TAtomicSharedPtr<IPriceCalculator>;

public:
    virtual ~IPriceCalculator() = default;

    virtual ui32 GetBasePrice() const = 0;
};

class IPriceCalculatorConfig {
public:
    using TFactory = NObjectFactory::TParametrizedObjectFactory<IPriceCalculatorConfig, TString>;
    using TPtr = TAtomicSharedPtr<IPriceCalculatorConfig>;

public:
    virtual ~IPriceCalculatorConfig() = default;

    virtual void Init(const TYandexConfig::Section* section) = 0;
    virtual void ToString(IOutputStream& os) const = 0;

    virtual bool DeserializeFromJson(const NJson::TJsonValue& jsonInfo) = 0;
    virtual NJson::TJsonValue SerializeToJson() const = 0;

    virtual IPriceCalculator::TPtr Construct() const = 0;

    static TSet<TString> GetTypes();
};

class TConstantPrice: public IPriceCalculator {
private:
    ui32 PriceForMinute;

public:
    TConstantPrice(const ui32 priceForMinute)
        : PriceForMinute(priceForMinute)
    {
    }

    virtual ui32 GetBasePrice() const override {
        return PriceForMinute;
    }
};

class TConstantPriceConfig: public IPriceCalculatorConfig {
private:
    ui32 PriceForMinute = 400;

private:
    static TFactory::TRegistrator<TConstantPriceConfig> Registrator;

public:
    TConstantPriceConfig() = default;
    TConstantPriceConfig(const ui32 price)
        : PriceForMinute(price)
    {
    }

    virtual bool DeserializeFromJson(const NJson::TJsonValue& jsonInfo) override;
    virtual NJson::TJsonValue SerializeToJson() const override;

    virtual void Init(const TYandexConfig::Section* section) override;
    virtual void ToString(IOutputStream& os) const override;

    virtual IPriceCalculator::TPtr Construct() const override;
};

class TPriceByTimeConfig: public IPriceCalculatorConfig {
private:
    ui32 SecondsInSegment = 30 * 60;
    bool ChangeOnMoving = false;

private:
    class TPriceInterval {
    private:
        ui32 SecondsStart;
        i32 Price;

    public:
        bool DeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
            JREAD_INT(jsonInfo, "start", SecondsStart);
            JREAD_INT(jsonInfo, "value", Price);
            return true;
        }

        NJson::TJsonValue SerializeToJson() const {
            NJson::TJsonValue result(NJson::JSON_MAP);
            JWRITE(result, "start", SecondsStart);
            JWRITE(result, "value", Price);
            return result;
        }

        bool operator<(const TPriceInterval& item) const {
            return SecondsStart < item.SecondsStart;
        }

        TPriceInterval(const ui32 secondsStart, const i32 price)
            : SecondsStart(secondsStart)
            , Price(price)
        {
        }

        i32 GetPrice() const {
            return Price;
        }

        ui32 GetSecondsStart() const {
            return SecondsStart;
        }
    };

private:
    TVector<TPriceInterval> PriceBySegments;
    TString PathForTable;
    i32 TimeShift = 3 * 60 * 60;

private:
    static TFactory::TRegistrator<TPriceByTimeConfig> Registrator;

public:
    TPriceByTimeConfig& SetSecondsInSegment(const ui32 secondsInSegment) {
        SecondsInSegment = secondsInSegment;
        return *this;
    }

    TPriceByTimeConfig& SetChangeOnMoving(const bool value) {
        ChangeOnMoving = value;
        return *this;
    }

    virtual bool DeserializeFromJson(const NJson::TJsonValue& jsonInfo) override;
    virtual NJson::TJsonValue SerializeToJson() const override;

    void InitFromFile(const TFsPath& path);
    virtual void Init(const TYandexConfig::Section* section) override;
    virtual void ToString(IOutputStream& os) const override;

    virtual IPriceCalculator::TPtr Construct() const override;

    i32 GetBasePrice(const TInstant instantStart) const;
    i32 CalcPrice(const TInstant instantStart, const TInstant instantFinish) const;
};

class TPriceByTime: public IPriceCalculator {
private:
    const TPriceByTimeConfig Config;

public:
    TPriceByTime(const TPriceByTimeConfig& config)
        : Config(config)
    {
    }

    virtual ui32 GetBasePrice() const override {
        return Config.GetBasePrice(Now());
    }
};
