#include "price.h"

#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>

#include <util/stream/file.h>

Y_UNIT_TEST_SUITE(PriceCalculationSuite) {

    Y_UNIT_TEST(Simple) {
        TPriceByTimeConfig config;
        {
            TFileOutput fi("./prices");
            TInstant timeInfo = TInstant::Zero();
            for (ui32 i = 0; i < 48; ++i) {
                fi << timeInfo.Hours() << ":" << timeInfo.Minutes() % 60 << ",";
                for (ui32 dw = 0; dw < 7; ++dw) {
                    fi << dw << "." << Sprintf("%0.2d", i) << ",";
                }
                fi << Endl;
                timeInfo += TDuration::Minutes(30);
            }
        }
        config.SetSecondsInSegment(0);
        config.SetChangeOnMoving(true);
        config.InitFromFile("./prices");
        {
            ui32 result = config.CalcPrice(TInstant::Seconds(234332322), TInstant::Seconds(234332323));
            UNIT_ASSERT_VALUES_EQUAL(result, 10);
        }
        {
            ui32 result = config.CalcPrice(TInstant::Minutes(15), TInstant::Minutes(45));
            UNIT_ASSERT_VALUES_EQUAL(result, 9015);
        }
        {
            ui32 result = config.CalcPrice(TInstant::Minutes(25), TInstant::Minutes(30));
            UNIT_ASSERT_VALUES_EQUAL(result, 1500);
        }
        {
            ui32 result = config.CalcPrice(TInstant::Minutes(30), TInstant::Minutes(35));
            UNIT_ASSERT_VALUES_EQUAL(result, 1505);
        }
        {
            ui32 result = config.CalcPrice(TInstant::Minutes(25), TInstant::Minutes(36));
            UNIT_ASSERT_VALUES_EQUAL(result, 3306);
        }
        {
            for (ui32 i = 0; i < 7; ++i) {
                ui32 result = config.CalcPrice(TInstant::Days(4 + i), TInstant::Days(4 + i) + TDuration::Days(1));
                UNIT_ASSERT_VALUES_EQUAL(result, 0.5 * (i * 100 * 2 + 47) * (24 * 60));
            }
        }
        {
            ui32 price = 0;
            for (ui32 i = 0; i < 7; ++i) {
                price += 0.5 * (i * 2 * 100 + 47) * (24 * 60);
            }
            ui32 result = config.CalcPrice(TInstant::Seconds(234332322), TInstant::Seconds(234332322) + TDuration::Days(7));
            UNIT_ASSERT_VALUES_EQUAL(result, price);
        }
    }

    Y_UNIT_TEST(SimpleFix) {
        TPriceByTimeConfig config;
        {
            TFileOutput fi("./prices");
            TInstant timeInfo = TInstant::Zero();
            for (ui32 i = 0; i < 48; ++i) {
                fi << timeInfo.Hours() << ":" << timeInfo.Minutes() % 60 << ",";
                for (ui32 dw = 0; dw < 7; ++dw) {
                    fi << dw << "." << Sprintf("%0.2d", i) << ",";
                }
                fi << Endl;
                timeInfo += TDuration::Minutes(30);
            }
        }
        config.SetSecondsInSegment(0);
        config.SetChangeOnMoving(false);
        config.InitFromFile("./prices");
        {
            ui32 result = config.CalcPrice(TInstant::Seconds(234332322), TInstant::Seconds(234332323));
            UNIT_ASSERT_VALUES_EQUAL(result, 10);
        }
        {
            ui32 result = config.CalcPrice(TInstant::Minutes(15), TInstant::Minutes(45));
            UNIT_ASSERT_VALUES_EQUAL(result, 9000);
        }
        {
            ui32 result = config.CalcPrice(TInstant::Minutes(25), TInstant::Minutes(30));
            UNIT_ASSERT_VALUES_EQUAL(result, 1500);
        }
        {
            ui32 result = config.CalcPrice(TInstant::Minutes(30), TInstant::Minutes(35));
            UNIT_ASSERT_VALUES_EQUAL(result, 1505);
        }
        {
            ui32 result = config.CalcPrice(TInstant::Minutes(25), TInstant::Minutes(36));
            UNIT_ASSERT_VALUES_EQUAL(result, 3300);
        }
        {
            for (ui32 i = 0; i < 7; ++i) {
                ui32 result = config.CalcPrice(TInstant::Days(4 + i), TInstant::Days(4 + i) + TDuration::Days(1));
                UNIT_ASSERT_VALUES_EQUAL(result, i * 100 * (24 * 60));
            }
        }
        {
            ui32 price = ((TInstant::Seconds(234332322).Days() + 3) % 7) * 100 + (TInstant::Seconds(234332322).Minutes() % (24 * 60)) / 30;
            price = price * 7 * (24 * 60);
            ui32 result = config.CalcPrice(TInstant::Seconds(234332322), TInstant::Seconds(234332322) + TDuration::Days(7));
            UNIT_ASSERT_VALUES_EQUAL(result, price);
        }
    }
}
