#include "model.h"

#include <drive/backend/offers/ranking/calcer.h>

#include <drive/backend/models/storage.h>
#include <drive/backend/proto/offer.pb.h>

NJson::TJsonValue TEntityPriceSettings::SerializeToJson() const {
    NJson::TJsonValue result(NJson::JSON_MAP);
    TJsonProcessor::Write(result, "model_name", ModelName);
    NJson::TJsonValue& jsonPricing = result.InsertValue("pricing", NJson::JSON_MAP);
    jsonPricing.InsertValue("pricing_type", PriceCalculatorType);
    jsonPricing.InsertValue("price_details", PriceCalculatorConfig->SerializeToJson());

    TPriceReduceAreaIncreaseLinearScheme ReduceByVolume;
    ReduceByVolume.MutablePoints().emplace_back();
    ReduceByVolume.MutablePoints().back().SetValue(1);
    result.InsertValue("reducing", ReduceByVolume.SerializeToJson());
    return result;
}

bool TEntityPriceSettings::DeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    TJsonProcessor::Read(jsonInfo, "model_name", ModelName);
    if (!ReadPricingPolicy(jsonInfo["pricing"], PriceCalculatorType, PriceCalculatorConfig, PriceCalculator)) {
        return false;
    }
    return true;
}

NDrive::TScheme TEntityPriceSettings::GetScheme(const NDrive::IServer* server) {
    NDrive::TScheme result;
    {
        NDrive::TScheme& pricePolicy = result.Add<TFSStructure>("pricing", "Политика ценообразования").SetStructure<NDrive::TScheme>();
        pricePolicy.Add<TFSVariants>("pricing_type", "Тип ценообразования").InitVariantsClass<IPriceCalculatorConfig>();
        pricePolicy.Add<TFSJson>("price_details", "Настройка");
    }
    {
        const NDrive::TModelsStorage* models = server->GetModelsStorage();
        auto modelNames = models ? models->ListOfferModels() : TVector<TString>{};
        result.Add<TFSVariants>("model_name", "Модель для вычисления цены").SetVariants(modelNames);
    }
    return result;
}

TPriceContext TEntityPriceSettings::BuildPriceContext() const {
    TPriceContext result;
    result.SetPrice(GetCurrentPrice());
    result.SetPriceModelName(ModelName);
    return result;
}

bool TEntityPriceSettings::ReadPricingPolicy(const NJson::TJsonValue& priceJson, TString& typeName, IPriceCalculatorConfig::TPtr& config, IPriceCalculator::TPtr& calculator) {
    if (!priceJson.IsMap()) {
        WARNING_LOG << "Incorrect TStandartOfferConstructor configuration in action" << Endl;
        return false;
    }
    JREAD_STRING_OPT(priceJson, "pricing_type", typeName);

    config = IPriceCalculatorConfig::TFactory::Construct(typeName);
    if (!config) {
        WARNING_LOG << "Incorrect RidingPriceCalculator.Type : '" << typeName << "' field in configuration" << Endl;
        return false;
    }

    if (!priceJson["price_details"].IsMap()) {
        WARNING_LOG << "Incorrect 'price_details' object in json" << Endl;
        return false;
    }

    if (!config->DeserializeFromJson(priceJson["price_details"])) {
        WARNING_LOG << "Incorrect 'price_details' object content in json" << Endl;
        return false;
    }
    calculator = config->Construct();
    return true;
}

void TEntityPriceSettings::SetConstantPrice(const ui32 price) {
    PriceCalculatorConfig = new TConstantPriceConfig(price);
    PriceCalculator = PriceCalculatorConfig->Construct();
    PriceCalculatorType = "constant";
    ModelName = "";
}

TEntityPriceSettings TEntityPriceSettings::Zero() {
    TEntityPriceSettings result;
    result.SetConstantPrice(0);
    return result;
}

TEquilibriumPriceCalculator::TEquilibriumPriceCalculator() {
    if (NDrive::HasServer()) {
        Server = &NDrive::GetServerAs<NDrive::IServer>();
    } else {
        Server = nullptr;
    }
}

bool TEquilibriumPriceCalculator::DeserializeFromJson(const NJson::TJsonValue& jsonValue) {
    const NJson::TJsonValue::TArray* arr = nullptr;
    if (jsonValue["optional_components"].GetArrayPointer(&arr)) {
        TSet<TString> ids;
        for (auto&& i : *arr) {
            TOptionalPriceComponent opc;
            if (!opc.DeserializeFromJson(i)) {
                return false;
            }
            if (!ids.emplace(opc.GetId()).second) {
                return false;
            }
            OptionalComponents.emplace_back(std::move(opc));
        }
    }
    return true;
}

NJson::TJsonValue TEquilibriumPriceCalculator::SerializeToJson() const {
    NJson::TJsonValue result(NJson::JSON_MAP);
    NJson::TJsonValue& optComponents = result.InsertValue("optional_components", NJson::JSON_ARRAY);
    for (auto&& i : OptionalComponents) {
        optComponents.AppendValue(i.SerializeToJson());
    }

    return result;
}

bool TPriceContext::DeserializeFromProto(const NDrive::NProto::TPriceContext& proto) {
    Price = proto.GetPrice();
    PriceModelName = proto.GetPriceModelName();
    for (auto&& i : proto.GetPriceModelInfos()) {
        PriceModelInfos.push_back({i.GetName(), i.GetBefore(), i.GetAfter()});
    }
    return true;
}

NDrive::NProto::TPriceContext TPriceContext::SerializeToProto() const {
    NDrive::NProto::TPriceContext result;
    result.SetPrice(Price);
    result.SetPriceModelName(PriceModelName);
    for (auto&& value : PriceModelInfos) {
        auto priceModelInfo = result.AddPriceModelInfos();
        priceModelInfo->SetName(value.Name);
        priceModelInfo->SetBefore(value.Before);
        priceModelInfo->SetAfter(value.After);
    }
    return result;
}

bool TPricesContext::DeserializeContextFromProto(const NDrive::NProto::TMarketPricesContext& proto) {
    if (proto.HasAcceptance() && !Acceptance.DeserializeFromProto(proto.GetAcceptance())) {
        return false;
    }
    if (!Parking.DeserializeFromProto(proto.GetParking())) {
        return false;
    }
    if (!Riding.DeserializeFromProto(proto.GetRiding())) {
        return false;
    }
    if (!Km.DeserializeFromProto(proto.GetKm())) {
        return false;
    }
    if (proto.HasOverrunKm()) {
        OverrunKm = proto.GetOverrunKm();
    }
    if (proto.HasOvertimeRiding()) {
        OvertimeRiding = proto.GetOvertimeRiding();
    }
    if (proto.HasOvertimeParking()) {
        OvertimeParking = proto.GetOvertimeParking();
    }
    for (auto&& i : proto.GetPaymentFeatures()) {
        PaymentFeatures.emplace(i);
    }
    return true;
}

NDrive::NProto::TMarketPricesContext TPricesContext::SerializeContextToProto() const {
    NDrive::NProto::TMarketPricesContext result;
    *result.MutableAcceptance() = Acceptance.SerializeToProto();
    *result.MutableParking() = Parking.SerializeToProto();
    *result.MutableRiding() = Riding.SerializeToProto();
    *result.MutableKm() = Km.SerializeToProto();
    if (OverrunKm) {
        result.SetOverrunKm(*OverrunKm);
    }
    if (OvertimeRiding) {
        result.SetOvertimeRiding(*OvertimeRiding);
    }
    if (OvertimeParking) {
        result.SetOvertimeParking(*OvertimeParking);
    }
    for (auto&& i : PaymentFeatures) {
        result.AddPaymentFeatures(i);
    }
    return result;
}

bool TEquilibriumPricesContext::DeserializePricesContextFromProto(const NDrive::NProto::TEquilibriumPricesContext& proto) {
    Parking = proto.GetParking();
    Riding = proto.GetRiding();
    Km = proto.GetKm();
    EquilibriumUtilization = TDuration::Seconds(proto.GetEquilibriumUtilization());
    return true;
}

NDrive::NProto::TEquilibriumPricesContext TEquilibriumPricesContext::SerializePricesContextToProto() const {
    NDrive::NProto::TEquilibriumPricesContext result;
    result.SetParking(Parking);
    result.SetRiding(Riding);
    result.SetKm(Km);
    result.SetEquilibriumUtilization(EquilibriumUtilization.Seconds());
    return result;
}

ui32 TEquilibriumPricesContext::CalcEquilibriumPackPrice(const TDuration d, const ui32 distance) const {
    if (d < EquilibriumUtilization) {
        return (d.Seconds() * Parking) / 60.0 + distance * Km;
    } else {
        return EquilibriumUtilization.Minutes() * Parking + distance * Km;
    }
}

bool TMarketPriceCalculator::DeserializeFromJson(const NJson::TJsonValue& jsonValue) {
    const auto& acceptance = jsonValue["acceptance"];
    if (acceptance.IsDefined() && !AcceptanceCost.DeserializeFromJson(acceptance)) {
        return false;
    }
    if (!ParkingPrice.DeserializeFromJson(jsonValue["parking"])) {
        return false;
    }
    if (!RidingPrice.DeserializeFromJson(jsonValue["riding"])) {
        return false;
    }
    if (!KmPrice.DeserializeFromJson(jsonValue["km"])) {
        return false;
    }
    return
        NJson::ParseField(jsonValue["pack_matrix"], PackPriceMatrix) &&
        NJson::ParseField(jsonValue["pack_overrun_matrix"], PackOverrunPriceMatrix) &&
        NJson::ParseField(jsonValue["pack_duration"], NJson::Dictionary(PackDurationPrice)) &&
        NJson::ParseField(jsonValue["pack_mileage"], NJson::Dictionary(PackMileagePrice));
}

NJson::TJsonValue TMarketPriceCalculator::SerializeToJson() const {
    NJson::TJsonValue result(NJson::JSON_MAP);
    result.InsertValue("acceptance", AcceptanceCost.SerializeToJson());
    result.InsertValue("parking", ParkingPrice.SerializeToJson());
    result.InsertValue("riding", RidingPrice.SerializeToJson());
    result.InsertValue("km", KmPrice.SerializeToJson());
    NJson::InsertField(result, "pack_duration", NJson::Dictionary(PackDurationPrice));
    NJson::InsertField(result, "pack_mileage", NJson::Dictionary(PackMileagePrice));
    NJson::InsertField(result, "pack_matrix", PackPriceMatrix);
    NJson::InsertField(result, "pack_overrun_matrix", PackOverrunPriceMatrix);
    return result;
}

NDrive::TScheme TMarketPriceCalculator::GetScheme(const NDrive::IServer* server) {
    NDrive::TScheme result;
    result.Add<TFSStructure>("acceptance").SetStructure(TEntityPriceSettings::GetScheme(server));
    result.Add<TFSStructure>("parking").SetStructure(TEntityPriceSettings::GetScheme(server));
    result.Add<TFSStructure>("riding").SetStructure(TEntityPriceSettings::GetScheme(server));
    result.Add<TFSStructure>("km").SetStructure(TEntityPriceSettings::GetScheme(server));
    result.Add<TFSJson>("pack_duration").SetDeprecated(true);
    result.Add<TFSJson>("pack_mileage").SetDeprecated(true);
    result.Add<TFSJson>("pack_matrix");
    result.Add<TFSJson>("pack_overrun_matrix");
    return result;
}

void TOptionalPriceComponent::ApplyForContext(const bool useKm, TPricesContext& pContext) const {
    if (useKm) {
        if (MinuteParkingVolume) {
            pContext.MutableParking().AddPrice(MinuteParkingVolume, Id);
            pContext.MutableRiding().AddPrice(MinuteParkingVolume, Id);
        }
        if (KmVolume) {
            pContext.MutableKm().AddPrice(KmVolume, Id);
        }
    } else {
        if (MinuteParkingVolume) {
            pContext.MutableParking().AddPrice(MinuteParkingVolume, Id);
        }
        if (MinuteRidingVolume) {
            pContext.MutableRiding().AddPrice(MinuteRidingVolume, Id);
        }
    }
    pContext.MutablePaymentFeatures().emplace(GetId());
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TOptionalPriceComponent& result) {
    return result.DeserializeFromJson(value);
}

template <>
NJson::TJsonValue NJson::ToJson(const TOptionalPriceComponent& object) {
    return object.SerializeToJson();
}
