#pragma once

#include "price.h"

#include <drive/library/cpp/scheme/scheme.h>

namespace NDrive::NProto {
    class TEquilibriumPricesContext;
    class TMarketPricesContext;
    class TPriceContext;
}

class TPriceContext;

class TEntityPriceSettings {
private:
    TString PriceCalculatorType = "constant";
    IPriceCalculatorConfig::TPtr PriceCalculatorConfig = MakeAtomicShared<TConstantPriceConfig>(static_cast<ui32>(1000));
    IPriceCalculator::TPtr PriceCalculator = MakeAtomicShared<TConstantPrice>(static_cast<ui32>(1000));
    R_READONLY(TString, ModelName);

public:
    static NDrive::TScheme GetScheme(const NDrive::IServer* server);
    static bool ReadPricingPolicy(const NJson::TJsonValue& priceJson, TString& typeName, IPriceCalculatorConfig::TPtr& config, IPriceCalculator::TPtr& calculator);
    static TEntityPriceSettings Zero();

public:
    void SetConstantPrice(const ui32 price);

    ui32 GetCurrentPrice() const {
        return PriceCalculator->GetBasePrice();
    }

    NJson::TJsonValue SerializeToJson() const;
    bool DeserializeFromJson(const NJson::TJsonValue& jsonInfo);

    TPriceContext BuildPriceContext() const;
};

class IModelPriceCalculator {
public:
    virtual ui32 CalcMinutePriceRiding() const = 0;
    virtual ui32 CalcMinutePriceParking() const = 0;
    virtual ui32 CalcKmPrice() const = 0;
};

class TPricesContext;

class TOptionalPriceComponent {
private:
    R_FIELD(TString, Id);
    R_FIELD(bool, DefaultActive, false);
    R_FIELD(ui32, MinuteParkingVolume, 0);
    R_FIELD(ui32, MinuteRidingVolume, 0);
    R_FIELD(ui32, KmVolume, 0);

public:
    bool Compare(const TOptionalPriceComponent& ext) const {
        return DefaultActive != ext.DefaultActive || MinuteRidingVolume != ext.MinuteParkingVolume ||
               MinuteRidingVolume != ext.MinuteRidingVolume || KmVolume != ext.KmVolume;
    }

    TString GetDiff(const TOptionalPriceComponent& ext) const {
        TStringBuilder sb;
        sb << "da: " << DefaultActive << " -> " << ext.DefaultActive << "\t"
           << "r: " << MinuteRidingVolume << " -> " << ext.MinuteParkingVolume << "\t"
           << "p: " << MinuteParkingVolume << " -> " << ext.MinuteParkingVolume << "\t"
           << "km: " << KmVolume << " -> " << ext.KmVolume;
        return sb;
    }

    TString GetStrInfo() const {
        TStringBuilder sb;
        sb << "da: " << DefaultActive << "\t"
           << "r: " << MinuteRidingVolume << "\t"
           << "p: " << MinuteParkingVolume << "\t"
           << "km: " << KmVolume;
        return sb;
    }

    void ApplyForContext(const bool useKm, TPricesContext& pContext) const;

    NJson::TJsonValue SerializeToJson() const {
        NJson::TJsonValue result(NJson::JSON_MAP);
        result.InsertValue("id", Id);
        TJsonProcessor::Write(result, "minute_parking_volume", MinuteParkingVolume);
        TJsonProcessor::Write(result, "minute_riding_volume", MinuteRidingVolume);
        TJsonProcessor::Write(result, "km_volume", KmVolume);
        TJsonProcessor::Write(result, "default_active", DefaultActive);
        return result;
    }

    bool DeserializeFromJson(const NJson::TJsonValue& info) {
        return
            TJsonProcessor::Read(info, "id", Id) &&
            TJsonProcessor::Read(info, "minute_parking_volume", MinuteParkingVolume, true) &&
            TJsonProcessor::Read(info, "minute_riding_volume", MinuteRidingVolume, true) &&
            TJsonProcessor::Read(info, "km_volume", KmVolume, true) &&
            TJsonProcessor::Read(info, "default_active", DefaultActive)
            ;
    }

    static NDrive::TScheme GetScheme(const NDrive::IServer* /*server*/) {
        NDrive::TScheme result;
        result.Add<TFSString>("id", "Идентификатор");
        result.Add<TFSNumeric>("minute_parking_volume").SetMin(0);
        result.Add<TFSNumeric>("minute_riding_volume").SetMin(0);
        result.Add<TFSNumeric>("km_volume").SetMin(0);
        result.Add<TFSBoolean>("default_active").SetDefault(false);
        return result;
    }
};

class TEquilibriumPriceCalculator {
private:
    TVector<TOptionalPriceComponent> OptionalComponents;
    const NDrive::IServer* Server = nullptr;

public:
    const TVector<TOptionalPriceComponent>& GetOptionalComponents() const {
        return OptionalComponents;
    }

    const TOptionalPriceComponent* GetOptionalPriceComponent(const TString& id) const {
        for (auto&& i : OptionalComponents) {
            if (i.GetId() == id) {
                return &i;
            }
        }
        return nullptr;
    }

    TOptionalPriceComponent* GetOptionalPriceComponent(const TString& id) {
        for (auto&& i : OptionalComponents) {
            if (i.GetId() == id) {
                return &i;
            }
        }
        return nullptr;
    }

    TEquilibriumPriceCalculator();
    bool DeserializeFromJson(const NJson::TJsonValue& jsonValue);
    NJson::TJsonValue SerializeToJson() const;

    static NDrive::TScheme GetScheme(const NDrive::IServer* server) {
        NDrive::TScheme result;
        result.Add<TFSArray>("optional_components", "Опциональные компоненты цены").SetElement(TOptionalPriceComponent::GetScheme(server));
        return result;
    }
};

struct TPriceModelInfo {
    TString Name;
    float Before = 0;
    float After = 0;
};
using TPriceModelInfos = TVector<TPriceModelInfo>;

class TPriceContext {
private:
    R_FIELD(ui32, Price, 0);
    R_FIELD(TPriceModelInfos, PriceModelInfos);
    R_FIELD(TString, PriceModelName);

public:
    explicit operator bool() const {
        return Price != 0;
    }

    bool HasCorrection(const TString& id) const {
        for (auto&& i : PriceModelInfos) {
            if (i.Name == id) {
                return true;
            }
        }
        return false;
    }

    void SetPrice(const ui32 price, const TString& modelId) {
        PriceModelInfos.push_back({modelId, (float)Price, (float)price});
        Price = price;
    }

    void AddPrice(const ui32 delta, const TString& modelId) {
        PriceModelInfos.push_back({modelId, (float)Price, (float)(Price + delta)});
        Price += delta;
    }

    void Clear() {
        Price = 0;
        PriceModelInfos.clear();
        PriceModelName.clear();
    }

    bool DeserializeFromProto(const NDrive::NProto::TPriceContext& proto);
    NDrive::NProto::TPriceContext SerializeToProto() const;

    bool GetPriceModeling() const {
        return !!PriceModelName;
    }
};

class TPricesContext {
private:
    R_FIELD(TPriceContext, Acceptance);
    R_FIELD(TPriceContext, Parking);
    R_FIELD(TPriceContext, Riding);
    R_FIELD(TPriceContext, Km);
    R_OPTIONAL(ui32, OverrunKm);
    R_OPTIONAL(ui32, OvertimeRiding);
    R_OPTIONAL(ui32, OvertimeParking);
    R_FIELD(TSet<TString>, PaymentFeatures);

protected:
    bool DeserializeContextFromProto(const NDrive::NProto::TMarketPricesContext& proto);
    NDrive::NProto::TMarketPricesContext SerializeContextToProto() const;

public:
    void PositiveCorrectRidingPrice(const i32 price, const TString& modelId) {
        if (price < 0) {
            return;
        }
        Riding.SetPrice(Max<ui32>(Riding.GetPrice(), price), modelId);
    }

    void NegativeCorrectRidingPrice(const i32 price, const TString& modelId) {
        if (price < 0) {
            return;
        }
        Riding.SetPrice(Min<ui32>(Riding.GetPrice(), price), modelId);
    }

    void KffCorrectRidingPrice(const double kff, const TString& modelId) {
        if (kff > 0) {
            PositiveCorrectRidingPrice(Riding.GetPrice() * (1 + kff), modelId);
        } else {
            NegativeCorrectRidingPrice(Riding.GetPrice() * (1 + kff), modelId);
        }
    }

    ui32 CalcMarketPackPrice(const TDuration d, const double distance, const double leasingDayPrice) const {
        const double l1 = (d.Seconds() * Parking.GetPrice()) / 60.0;
        if (l1 > leasingDayPrice) {
            if (d < TDuration::Days(1)) {
                return leasingDayPrice + distance * Km.GetPrice();
            } else {
                return 1.0 * d.Seconds() * leasingDayPrice / TDuration::Days(1).Seconds() + distance * Km.GetPrice();
            }
        } else {
            return l1 + distance * Km.GetPrice();
        }
    }

    ui32 CalcMarketFixPrice(const TDuration d, const double distance) const {
        return (d.Seconds() * Riding.GetPrice()) / 60.0 + distance * Km.GetPrice();
    }

    ui32 GetOverrunKm() const {
        return OverrunKm.GetOrElse(Km.GetPrice());
    }

    ui32 GetOvertimeRiding() const {
        return OvertimeRiding.GetOrElse(Riding.GetPrice());
    }

    ui32 GetOvertimeParking() const {
        return OvertimeParking.GetOrElse(Parking.GetPrice());
    }
};

class TEquilibriumPricesContext {
private:
    R_FIELD(double, Parking, 10000);
    R_FIELD(double, Riding, 10000);
    R_FIELD(double, Km, 10000);
    R_FIELD(TDuration, EquilibriumUtilization, TDuration::Hours(8));

public:
    bool DeserializePricesContextFromProto(const NDrive::NProto::TEquilibriumPricesContext& proto);
    NDrive::NProto::TEquilibriumPricesContext SerializePricesContextToProto() const;

    ui32 CalcEquilibriumPackPrice(const TDuration d, const ui32 distance) const;
};

class TMarketPriceCalculator: public IModelPriceCalculator {
public:
    using TPriceMap = TMap<ui32, ui32>;
    using TPriceMatrix = TMap<std::pair<ui32, ui32>, ui32>;

private:
    R_FIELD(TEntityPriceSettings, AcceptanceCost, TEntityPriceSettings::Zero());
    R_FIELD(TEntityPriceSettings, ParkingPrice);
    R_FIELD(TEntityPriceSettings, RidingPrice);
    R_FIELD(TEntityPriceSettings, KmPrice);
    R_FIELD(TPriceMap, PackDurationPrice);
    R_FIELD(TPriceMap, PackMileagePrice);
    R_FIELD(TPriceMatrix, PackPriceMatrix);
    R_FIELD(TPriceMatrix, PackOverrunPriceMatrix);

public:
    virtual ui32 CalcMinutePriceRiding() const override {
        return RidingPrice.GetCurrentPrice();
    }
    virtual ui32 CalcMinutePriceParking() const override {
        return ParkingPrice.GetCurrentPrice();
    }
    virtual ui32 CalcKmPrice() const override {
        return KmPrice.GetCurrentPrice();
    }

    bool DeserializeFromJson(const NJson::TJsonValue& jsonValue);
    NJson::TJsonValue SerializeToJson() const;

public:
    static NDrive::TScheme GetScheme(const NDrive::IServer* server);
};
