#include "calcer.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/tags/tags_manager.h>

#include <drive/library/cpp/geofeatures/features.h>
#include <drive/library/cpp/user_events_api/user_features.h>
#include <drive/library/cpp/weather/weather_api.h>

#include <util/datetime/systime.h>
#include <util/string/cast.h>

void NDrive::CalcDeviceFeatures(TOfferFeatures& features, const NDrive::IServer& server, const TString& deviceId) {
    const auto& snapshotManager = server.GetSnapshotsManager();
    const auto snapshot = snapshotManager.GetSnapshot(deviceId);
    const auto& tags = snapshot.GetTagsInPoint();
    auto locationOptions = snapshotManager.GetLocationOptions();
    auto location = snapshot.GetLocation(TDuration::Max(), locationOptions);
    if (location) {
        features.Floats[NDriveOfferFactors::FI_LATITUDE] = location->Latitude;
        features.Floats[NDriveOfferFactors::FI_LONGITUDE] = location->Longitude;

        features.Floats2[NDriveOfferFactors2::FI_LATITUDE] = location->Latitude;
        features.Floats2[NDriveOfferFactors2::FI_LONGITUDE] = location->Longitude;
        if (tags.contains(NDrive::MskAreaLocationTag)) {
            auto distanceFromCenter = std::sqrt(
                std::pow((location->Latitude - 55.75) / 0.14, 2) + std::pow((location->Longitude - 37.62) / 0.24, 2)
            );

            features.Floats[NDriveOfferFactors::FI_DISTANCE_FROM_CENTER] = distanceFromCenter;

            features.Floats2[NDriveOfferFactors2::FI_DISTANCE_FROM_CENTER] = distanceFromCenter;
            features.Categories2[NDriveOfferCatFactors2::CITY] = "MSK";
        } else if (tags.contains(NDrive::SpbAreaLocationTag)) {
            features.Categories2[NDriveOfferCatFactors2::CITY] = "SPB";
        } else if (tags.contains(NDrive::KazanAreaLocationTag)) {
            features.Categories2[NDriveOfferCatFactors2::CITY] = "KZN";
        } else if (tags.contains(NDrive::SochiAreaLocationTag)) {
            features.Categories2[NDriveOfferCatFactors2::CITY] = "SCH";
        }
    }
    auto geocoded = snapshot.GetGeocoded(TDuration::Max(), locationOptions);
    if (geocoded) {
        TStringBuf street;
        for (auto&& i : StringSplitter(geocoded->Content).Split(',')) {
            street = i.Token();
            break;
        }
        features.Categories2[NDriveOfferCatFactors2::FI_GEOCODED_STREET] = ToString(street);
    }
    auto fuelLevel = snapshot.GetFuelLevel();
    auto fuelDistance = snapshot.GetFuelDistance();
    if (fuelLevel) {
        features.Floats[NDriveOfferFactors::FI_FUEL_LEVEL_P] = *fuelLevel * 0.01;

        features.Floats2[NDriveOfferFactors2::FI_FUEL_LEVEL_P] = *fuelLevel * 0.01;
    }
    if (fuelDistance) {
        features.Floats[NDriveOfferFactors::FI_NORMALIZED_FUEL_LEVEL_KM] = *fuelDistance * 0.001;
        features.Floats[NDriveOfferFactors::FI_LOG_FUEL_LEVEL_KM] = std::log(1 + *fuelDistance);

        features.Floats2[NDriveOfferFactors2::FI_NORMALIZED_FUEL_LEVEL_KM] = *fuelDistance * 0.001;
        features.Floats2[NDriveOfferFactors2::FI_LOG_FUEL_LEVEL_KM] = std::log(1 + *fuelDistance);
    }
    auto deviceInfos = server.GetDriveAPI()->GetCarsData()->FetchInfo({ deviceId }, TInstant::Zero());
    auto deviceInfo = deviceInfos.GetResultPtr(deviceId);
    if (deviceInfo) {
        features.Categories2[NDriveOfferCatFactors2::FI_MODEL] = deviceInfo->GetModel();
    }

    TVector<TTaggedDevice> devices;
    if (server.GetDriveAPI()->GetTagsManager().GetDeviceTags().GetObjectsFromCache({ deviceId }, {}, devices, TInstant::Zero()) && devices.size() == 1) {
        const auto& tags = devices[0].GetTags();
        for (auto&& tag : tags) {
            const TString& name = tag->GetName();
            if (name == "just_cleaned") {
                features.Floats[NDriveOfferFactors::FI_JUST_CLEANED] = 1;

                features.Floats2[NDriveOfferFactors2::FI_JUST_CLEANED] = 1;
                continue;
            }
            if (name == "full_tank_marker") {
                features.Floats[NDriveOfferFactors::FI_FULL_TANK] = 1;

                features.Floats2[NDriveOfferFactors2::FI_FULL_TANK] = 1;
                continue;
            }
            if (name == "yaauto") {
                features.Floats[NDriveOfferFactors::FI_HAS_HEAD] = 1;

                features.Floats2[NDriveOfferFactors2::FI_HAS_HEAD] = 1;
                continue;
            }
            if (name == "remote_condition") {
                features.Floats[NDriveOfferFactors::FI_HAS_WARMING] = 1;

                features.Floats2[NDriveOfferFactors2::FI_HAS_WARMING] = 1;
                continue;
            }
            if (name == "child_seat") {
                features.Floats[NDriveOfferFactors::FI_HAS_CHILD_SEAT] = 1;

                features.Floats2[NDriveOfferFactors2::FI_HAS_CHILD_SEAT] = 1;
                continue;
            }
        }
    }
}

void NDrive::CalcGeoFeatures(TOfferFeatures& features, const TGeoFeatures* geoFeatures) {
    constexpr float defaultValue = NDrive::GeoFeaturesDefaultValue;
    features.Floats[NDriveOfferFactors::RT_SURGE_PREDICTION_1] = geoFeatures ? geoFeatures->RtSurgePrediction1 : defaultValue;
    features.Floats[NDriveOfferFactors::RT_SURGE_PREDICTION_2] = geoFeatures ? geoFeatures->RtSurgePrediction2 : defaultValue;
    features.Floats[NDriveOfferFactors::RT_SURGE_PREDICTION_3] = geoFeatures ? geoFeatures->RtSurgePrediction3 : defaultValue;
    features.Floats[NDriveOfferFactors::RT_SURGE_PREDICTION_4] = geoFeatures ? geoFeatures->RtSurgePrediction4 : defaultValue;
    features.Floats[NDriveOfferFactors::RT_SURGE_PREDICTION_5] = geoFeatures ? geoFeatures->RtSurgePrediction5 : defaultValue;

    features.Floats2[NDriveOfferFactors2::RT_SURGE_PREDICTION_1] = geoFeatures ? geoFeatures->RtSurgePrediction1 : defaultValue;
    features.Floats2[NDriveOfferFactors2::RT_SURGE_PREDICTION_2] = geoFeatures ? geoFeatures->RtSurgePrediction2 : defaultValue;
    features.Floats2[NDriveOfferFactors2::RT_SURGE_PREDICTION_3] = geoFeatures ? geoFeatures->RtSurgePrediction3 : defaultValue;
    features.Floats2[NDriveOfferFactors2::RT_SURGE_PREDICTION_4] = geoFeatures ? geoFeatures->RtSurgePrediction4 : defaultValue;
    features.Floats2[NDriveOfferFactors2::RT_SURGE_PREDICTION_5] = geoFeatures ? geoFeatures->RtSurgePrediction5 : defaultValue;
    features.Floats2[NDriveOfferFactors2::RT_SURGE_60_MIN] = defaultValue;
    features.Floats2[NDriveOfferFactors2::RT_SESSIONS_30_MIN] = defaultValue;
    features.Floats2[NDriveOfferFactors2::RT_SESSIONS_60_MIN] = defaultValue;
    features.Floats2[NDriveOfferFactors2::RT_SESSIONS_120_MIN] = defaultValue;
    features.Floats2[NDriveOfferFactors2::RT_SESSIONS_60_MIN_EXTENDED] = geoFeatures ? geoFeatures->RtSessions60MinExtended : defaultValue;
    features.Floats2[NDriveOfferFactors2::RT_SESSIONS_120_MIN_EXTENDED] = geoFeatures ? geoFeatures->RtSessions120MinExtended : defaultValue;
    features.Floats2[NDriveOfferFactors2::RT_SESSIONS_180_MIN_EXTENDED] = geoFeatures ? geoFeatures->RtSessions180MinExtended : defaultValue;
    features.Floats2[NDriveOfferFactors2::RT_SURGE_60_MIN_EXTENDED] = geoFeatures ? geoFeatures->RtSurge60MinExtended : defaultValue;
    features.Floats2[NDriveOfferFactors2::RT_SURGE_120_MIN_EXTENDED] = geoFeatures ? geoFeatures->RtSurge120MinExtended : defaultValue;
    features.Floats2[NDriveOfferFactors2::RT_SURGE_180_MIN_EXTENDED] = geoFeatures ? geoFeatures->RtSurge180MinExtended : defaultValue;
    features.Floats2[NDriveOfferFactors2::RT_CARS_180_MIN_EXTENDED] = geoFeatures ? geoFeatures->RtCars180MinExtended : defaultValue;
    features.Floats2[NDriveOfferFactors2::RT_CARS_0005_MIN_WINDOW] = geoFeatures ? geoFeatures->RtCars0005MinWindow : defaultValue;
    features.Floats2[NDriveOfferFactors2::RT_CARS_0510_MIN_WINDOW] = geoFeatures ? geoFeatures->RtCars0510MinWindow : defaultValue;
    features.Floats2[NDriveOfferFactors2::RT_CARS_1015_MIN_WINDOW] = geoFeatures ? geoFeatures->RtCars1015MinWindow : defaultValue;
    features.Floats2[NDriveOfferFactors2::RT_CARS_1520_MIN_WINDOW] = geoFeatures ? geoFeatures->RtCars1520MinWindow : defaultValue;
    features.Floats2[NDriveOfferFactors2::RT_CARS_2025_MIN_WINDOW] = geoFeatures ? geoFeatures->RtCars2025MinWindow : defaultValue;
    features.Floats2[NDriveOfferFactors2::RT_CARS_2530_MIN_WINDOW] = geoFeatures ? geoFeatures->RtCars2530MinWindow : defaultValue;
    features.Floats2[NDriveOfferFactors2::RT_CARS_3035_MIN_WINDOW] = geoFeatures ? geoFeatures->RtCars3035MinWindow : defaultValue;
    features.Floats2[NDriveOfferFactors2::RT_CARS_3540_MIN_WINDOW] = geoFeatures ? geoFeatures->RtCars3540MinWindow : defaultValue;
    features.Floats2[NDriveOfferFactors2::RT_CARS_4045_MIN_WINDOW] = geoFeatures ? geoFeatures->RtCars4045MinWindow : defaultValue;
    features.Floats2[NDriveOfferFactors2::RT_CARS_4550_MIN_WINDOW] = geoFeatures ? geoFeatures->RtCars4550MinWindow : defaultValue;
    features.Floats2[NDriveOfferFactors2::RT_CARS_5055_MIN_WINDOW] = geoFeatures ? geoFeatures->RtCars5055MinWindow : defaultValue;
    features.Floats2[NDriveOfferFactors2::RT_CARS_5560_MIN_WINDOW] = geoFeatures ? geoFeatures->RtCars5560MinWindow : defaultValue;

    features.Floats2[NDriveOfferFactors2::SURGE_LAST_DAY] = geoFeatures ? geoFeatures->SurgeLastDay : defaultValue;
    features.Floats2[NDriveOfferFactors2::SURGE_LAST_2_WEEKS] = geoFeatures ? geoFeatures->SurgeLast2Weeks : defaultValue;
    features.Floats2[NDriveOfferFactors2::SURGE_DAY_WEEK_AGO] = geoFeatures ? geoFeatures->SurgeDayWeekAgo : defaultValue;
    features.Floats2[NDriveOfferFactors2::SESSIONS_LAST_DAY] = geoFeatures ? geoFeatures->SessionsLastDay : defaultValue;
    features.Floats2[NDriveOfferFactors2::SESSIONS_LAST_2_WEEKS] = geoFeatures ? geoFeatures->SessionsLast2Weeks : defaultValue;
    features.Floats2[NDriveOfferFactors2::SESSIONS_DAY_WEEK_AGO] = geoFeatures ? geoFeatures->SessionsDayWeekAgo : defaultValue;
    features.Floats2[NDriveOfferFactors2::CARS_LAST_DAY] = geoFeatures ? geoFeatures->CarsLastDay : defaultValue;
    features.Floats2[NDriveOfferFactors2::CARS_LAST_2_WEEKS] = geoFeatures ? geoFeatures->CarsLast2Weeks : defaultValue;
    features.Floats2[NDriveOfferFactors2::CARS_DAY_WEEK_AGO] = geoFeatures ? geoFeatures->CarsDayWeekAgo : defaultValue;
    features.Floats2[NDriveOfferFactors2::ORDERS_LAST_2_WEEKS] = geoFeatures ? geoFeatures->OrdersLast2Weeks : defaultValue;
}

void NDrive::CalcGeoFeatures(TOfferFeatures& features, ui64 geobaseId, const TGeoFeatures* geobaseFeatures) {
    constexpr float defaultValue = NDrive::GeoFeaturesDefaultValue;
    features.Categories2[NDriveOfferCatFactors2::FI_GEOBASE_ID_A] = geobaseId ? ToString(geobaseId) : TString();

    features.Floats2[NDriveOfferFactors2::SURGE_LAST_DAY_GEOBASE] = geobaseFeatures ? geobaseFeatures->SurgeLastDayGeobase : defaultValue;
    features.Floats2[NDriveOfferFactors2::SURGE_LAST_2_WEEKS_GEOBASE] = geobaseFeatures ? geobaseFeatures->SurgeLast2WeeksGeobase : defaultValue;
    features.Floats2[NDriveOfferFactors2::SURGE_DAY_WEEK_AGO_GEOBASE] = geobaseFeatures ? geobaseFeatures->SurgeDayWeekAgoGeobase : defaultValue;
    features.Floats2[NDriveOfferFactors2::SESSIONS_LAST_DAY_GEOBASE] = geobaseFeatures ? geobaseFeatures->SessionsLastDayGeobase : defaultValue;
    features.Floats2[NDriveOfferFactors2::SESSIONS_LAST_2_WEEKS_GEOBASE] = geobaseFeatures ? geobaseFeatures->SessionsLast2WeeksGeobase : defaultValue;
    features.Floats2[NDriveOfferFactors2::SESSIONS_DAY_WEEK_AGO_GEOBASE] = geobaseFeatures ? geobaseFeatures->SessionsDayWeekAgoGeobase : defaultValue;
    features.Floats2[NDriveOfferFactors2::CARS_LAST_DAY_GEOBASE] = geobaseFeatures ? geobaseFeatures->CarsLastDayGeobase : defaultValue;
    features.Floats2[NDriveOfferFactors2::CARS_LAST_2_WEEKS_GEOBASE] = geobaseFeatures ? geobaseFeatures->CarsLast2WeeksGeobase : defaultValue;
    features.Floats2[NDriveOfferFactors2::CARS_DAY_WEEK_AGO_GEOBASE] = geobaseFeatures ? geobaseFeatures->CarsDayWeekAgoGeobase : defaultValue;
    features.Floats2[NDriveOfferFactors2::ORDERS_LAST_2_WEEKS_GEOBASE] = geobaseFeatures ? geobaseFeatures->OrdersLast2WeeksGeobase : defaultValue;
    features.Floats2[NDriveOfferFactors2::SURGE_LAST_WEEK_GEOBASE] = geobaseFeatures ? geobaseFeatures->SurgeLastWeekGeobase : defaultValue;
    features.Floats2[NDriveOfferFactors2::SESSIONS_LAST_WEEK_GEOBASE] = geobaseFeatures ? geobaseFeatures->SessionsLastWeekGeobase : defaultValue;
    features.Floats2[NDriveOfferFactors2::CARS_LAST_WEEK_GEOBASE] = defaultValue;

    features.Floats2[NDriveOfferFactors2::LONG_TERM_IDLE_LEASING_COST_A] = defaultValue;
    features.Floats2[NDriveOfferFactors2::MID_TERM_IDLE_LEASING_COST_A] = defaultValue;
    features.Floats2[NDriveOfferFactors2::SHORT_TERM_IDLE_LEASING_COST_A] = defaultValue;
}

void NDrive::CalcSourceFeatures(TOfferFeatures& features, const NDrive::TUserGeoFeatures* userGeoFeatures) {
    features.Floats[NDriveOfferFactors::GEO_A_DAY_FREQ] = userGeoFeatures ? userGeoFeatures->A.DayFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats[NDriveOfferFactors::GEO_A_DUR_10_20_FREQ] = userGeoFeatures ? userGeoFeatures->A.Dur1020Freq : NDrive::UserFeaturesDefaultValue;
    features.Floats[NDriveOfferFactors::GEO_A_DUR_10_FREQ] = userGeoFeatures ? userGeoFeatures->A.Dur10Freq : NDrive::UserFeaturesDefaultValue;
    features.Floats[NDriveOfferFactors::GEO_A_DUR_20_40_FREQ] = userGeoFeatures ? userGeoFeatures->A.Dur2040Freq : NDrive::UserFeaturesDefaultValue;
    features.Floats[NDriveOfferFactors::GEO_A_DUR_40_FREQ] = userGeoFeatures ? userGeoFeatures->A.Dur40Freq : NDrive::UserFeaturesDefaultValue;
    features.Floats[NDriveOfferFactors::GEO_A_EVENING_FREQ] = userGeoFeatures ? userGeoFeatures->A.EveningFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats[NDriveOfferFactors::GEO_A_FREQ] = userGeoFeatures ? userGeoFeatures->A.Freq : NDrive::UserFeaturesDefaultValue;
    features.Floats[NDriveOfferFactors::GEO_A_IN_TOP] = userGeoFeatures ? userGeoFeatures->A.InTop : NDrive::UserFeaturesDefaultValue;
    features.Floats[NDriveOfferFactors::GEO_A_MORNING_FREQ] = userGeoFeatures ? userGeoFeatures->A.MorningFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats[NDriveOfferFactors::GEO_A_RATIO] = userGeoFeatures ? userGeoFeatures->A.Ratio : NDrive::UserFeaturesDefaultValue;
    features.Floats[NDriveOfferFactors::GEO_A_TO_TOP1_B_DIST] = userGeoFeatures ? userGeoFeatures->A.Top1Dist : NDrive::UserFeaturesDefaultValue;
    features.Floats[NDriveOfferFactors::GEO_A_TO_TOP1_B_FREQ] = userGeoFeatures ? userGeoFeatures->A.Top1Freq : NDrive::UserFeaturesDefaultValue;
    features.Floats[NDriveOfferFactors::GEO_A_TO_TOP1_B_RATIO] = userGeoFeatures ? userGeoFeatures->A.Top1Ratio : NDrive::UserFeaturesDefaultValue;
    features.Floats[NDriveOfferFactors::GEO_A_TO_TOP2_B_DIST] = userGeoFeatures ? userGeoFeatures->A.Top2Dist : NDrive::UserFeaturesDefaultValue;
    features.Floats[NDriveOfferFactors::GEO_A_TO_TOP2_B_FREQ] = userGeoFeatures ? userGeoFeatures->A.Top2Freq : NDrive::UserFeaturesDefaultValue;
    features.Floats[NDriveOfferFactors::GEO_A_TO_TOP2_B_RATIO] = userGeoFeatures ? userGeoFeatures->A.Top2Ratio : NDrive::UserFeaturesDefaultValue;

    features.Categories2[NDriveOfferCatFactors2::FI_GEO_ID_A] = userGeoFeatures ? userGeoFeatures->A.GeoId : TString();
    features.Categories2[NDriveOfferCatFactors2::GEO_A_TO_TOP1_B] = userGeoFeatures ? userGeoFeatures->A.Top1GeoId : TString();
    features.Categories2[NDriveOfferCatFactors2::GEO_A_TO_TOP2_B] = userGeoFeatures ? userGeoFeatures->A.Top2GeoId : TString();
    features.Floats2[NDriveOfferFactors2::GEO_A_DAY_FREQ] = userGeoFeatures ? userGeoFeatures->A.DayFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_A_DUR_10_20_FREQ] = userGeoFeatures ? userGeoFeatures->A.Dur1020Freq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_A_DUR_10_FREQ] = userGeoFeatures ? userGeoFeatures->A.Dur10Freq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_A_DUR_20_40_FREQ] = userGeoFeatures ? userGeoFeatures->A.Dur2040Freq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_A_DUR_40_FREQ] = userGeoFeatures ? userGeoFeatures->A.Dur40Freq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_A_EVENING_FREQ] = userGeoFeatures ? userGeoFeatures->A.EveningFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_A_FREQ] = userGeoFeatures ? userGeoFeatures->A.Freq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_A_IN_TOP] = userGeoFeatures ? userGeoFeatures->A.InTop : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_A_MORNING_FREQ] = userGeoFeatures ? userGeoFeatures->A.MorningFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_A_RATIO] = userGeoFeatures ? userGeoFeatures->A.Ratio : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_A_TO_TOP1_B_DIST] = userGeoFeatures ? userGeoFeatures->A.Top1Dist : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_A_TO_TOP1_B_FREQ] = userGeoFeatures ? userGeoFeatures->A.Top1Freq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_A_TO_TOP1_B_RATIO] = userGeoFeatures ? userGeoFeatures->A.Top1Ratio : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_A_TO_TOP2_B_DIST] = userGeoFeatures ? userGeoFeatures->A.Top2Dist : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_A_TO_TOP2_B_FREQ] = userGeoFeatures ? userGeoFeatures->A.Top2Freq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_A_TO_TOP2_B_RATIO] = userGeoFeatures ? userGeoFeatures->A.Top2Ratio : NDrive::UserFeaturesDefaultValue;
}

void NDrive::CalcSourceFeatures(TOfferFeatures& features, ui64 geobaseId, const NDrive::TUserGeoFeatures* userGeobaseFeatures) {
    features.Categories2[NDriveOfferCatFactors2::FI_GEOBASE_ID_A] = geobaseId ? ToString(geobaseId) : TString();

    features.Floats2[NDriveOfferFactors2::GEOBASE_A_FREQ] = userGeobaseFeatures ? userGeobaseFeatures->A.GeobaseFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_MORNING_FREQ] = userGeobaseFeatures ? userGeobaseFeatures->A.GeobaseMorningFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_DAY_FREQ] = userGeobaseFeatures ? userGeobaseFeatures->A.GeobaseDayFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_EVENING_FREQ] = userGeobaseFeatures ? userGeobaseFeatures->A.GeobaseEveningFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_WEEKEND_FREQ] = userGeobaseFeatures ? userGeobaseFeatures->A.GeobaseWeekendFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_WEEKDAY_FREQ] = userGeobaseFeatures ? userGeobaseFeatures->A.GeobaseWeekdayFreq : NDrive::UserFeaturesDefaultValue;

    features.Floats2[NDriveOfferFactors2::GEOBASE_A_WEIGHTED_FREQ] = userGeobaseFeatures ? userGeobaseFeatures->A.GeobaseWeightedFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_MORNING_WEIGHTED_FREQ] = userGeobaseFeatures ? userGeobaseFeatures->A.GeobaseMorningWeightedFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_DAY_WEIGHTED_FREQ] = userGeobaseFeatures ? userGeobaseFeatures->A.GeobaseDayWeightedFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_EVENING_WEIGHTED_FREQ] = userGeobaseFeatures ? userGeobaseFeatures->A.GeobaseEveningWeightedFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_WEEKEND_WEIGHTED_FREQ] = userGeobaseFeatures ? userGeobaseFeatures->A.GeobaseWeekendWeightedFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_WEEKDAY_WEIGHTED_FREQ] = userGeobaseFeatures ? userGeobaseFeatures->A.GeobaseWeekdayWeightedFreq : NDrive::UserFeaturesDefaultValue;

    features.Floats2[NDriveOfferFactors2::GEOBASE_A_RATIO] = userGeobaseFeatures ? userGeobaseFeatures->A.GeobaseRatio : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_MORNING_RATIO] = userGeobaseFeatures ? userGeobaseFeatures->A.GeobaseMorningRatio : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_DAY_RATIO] = userGeobaseFeatures ? userGeobaseFeatures->A.GeobaseDayRatio : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_EVENING_RATIO] = userGeobaseFeatures ? userGeobaseFeatures->A.GeobaseEveningRatio : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_WEEKEND_RATIO] = userGeobaseFeatures ? userGeobaseFeatures->A.GeobaseWeekendRatio : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_WEEKDAY_RATIO] = userGeobaseFeatures ? userGeobaseFeatures->A.GeobaseWeekdayRatio : NDrive::UserFeaturesDefaultValue;

    features.Floats2[NDriveOfferFactors2::GEOBASE_A_WEIGHTED_RATIO] = userGeobaseFeatures ? userGeobaseFeatures->A.GeobaseWeightedRatio : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_MORNING_WEIGHTED_RATIO] = userGeobaseFeatures ? userGeobaseFeatures->A.GeobaseMorningWeightedRatio : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_DAY_WEIGHTED_RATIO] = userGeobaseFeatures ? userGeobaseFeatures->A.GeobaseDayWeightedRatio : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_EVENING_WEIGHTED_RATIO] = userGeobaseFeatures ? userGeobaseFeatures->A.GeobaseEveningWeightedRatio : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_WEEKEND_WEIGHTED_RATIO] = userGeobaseFeatures ? userGeobaseFeatures->A.GeobaseWeekendWeightedRatio : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_WEEKDAY_WEIGHTED_RATIO] = userGeobaseFeatures ? userGeobaseFeatures->A.GeobaseWeekdayWeightedRatio : NDrive::UserFeaturesDefaultValue;

    features.Floats2[NDriveOfferFactors2::GEOBASE_A_PCT] = userGeobaseFeatures ? userGeobaseFeatures->A.GeobasePct : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_MORNING_PCT] = userGeobaseFeatures ? userGeobaseFeatures->A.GeobaseMorningPct : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_DAY_PCT] = userGeobaseFeatures ? userGeobaseFeatures->A.GeobaseDayPct : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_EVENING_PCT] = userGeobaseFeatures ? userGeobaseFeatures->A.GeobaseEveningPct : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_WEEKEND_PCT] = userGeobaseFeatures ? userGeobaseFeatures->A.GeobaseWeekendPct : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_WEEKDAY_PCT] = userGeobaseFeatures ? userGeobaseFeatures->A.GeobaseWeekdayPct : NDrive::UserFeaturesDefaultValue;

    features.Floats2[NDriveOfferFactors2::GEO_SAME] = userGeobaseFeatures ? userGeobaseFeatures->A.GeobaseSame : NDrive::UserFeaturesDefaultValue;
}

void NDrive::CalcDestinationFeatures(TOfferFeatures& features, const TGeoCoord& coordinate, const TGeoFeatures* geoFeatures, const NDrive::TUserGeoFeatures* userGeoFeatures, double distance, double duration) {
    features.Floats[NDriveOfferFactors::FI_DESTINATION_DISTANCE] = distance;
    features.Floats[NDriveOfferFactors::FI_DESTINATION_DURATION] = duration;
    features.Floats[NDriveOfferFactors::FI_DESTINATION_LATITUDE] = coordinate.Y;
    features.Floats[NDriveOfferFactors::FI_DESTINATION_LONGITUDE] = coordinate.X;
    features.Floats[NDriveOfferFactors::FI_DESTINATION_RT_SURGE] = geoFeatures ? geoFeatures->RtSurgePrediction1 : NDrive::GeoFeaturesDefaultValue;
    features.Floats[NDriveOfferFactors::GEO_B_DAY_FREQ] = userGeoFeatures ? userGeoFeatures->B.DayFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats[NDriveOfferFactors::GEO_B_DUR_10_20_FREQ] = userGeoFeatures ? userGeoFeatures->B.Dur1020Freq : NDrive::UserFeaturesDefaultValue;
    features.Floats[NDriveOfferFactors::GEO_B_DUR_10_FREQ] = userGeoFeatures ? userGeoFeatures->B.Dur10Freq : NDrive::UserFeaturesDefaultValue;
    features.Floats[NDriveOfferFactors::GEO_B_DUR_20_40_FREQ] = userGeoFeatures ? userGeoFeatures->B.Dur2040Freq : NDrive::UserFeaturesDefaultValue;
    features.Floats[NDriveOfferFactors::GEO_B_DUR_40_FREQ] = userGeoFeatures ? userGeoFeatures->B.Dur40Freq : NDrive::UserFeaturesDefaultValue;
    features.Floats[NDriveOfferFactors::GEO_B_EVENING_FREQ] = userGeoFeatures ? userGeoFeatures->B.EveningFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats[NDriveOfferFactors::GEO_B_FREQ] = userGeoFeatures ? userGeoFeatures->B.Freq : NDrive::UserFeaturesDefaultValue;
    features.Floats[NDriveOfferFactors::GEO_B_IN_TOP] = userGeoFeatures ? userGeoFeatures->B.InTop : NDrive::UserFeaturesDefaultValue;
    features.Floats[NDriveOfferFactors::GEO_B_MORNING_FREQ] = userGeoFeatures ? userGeoFeatures->B.MorningFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats[NDriveOfferFactors::GEO_B_RATIO] = userGeoFeatures ? userGeoFeatures->B.Ratio : NDrive::UserFeaturesDefaultValue;
    features.Floats[NDriveOfferFactors::GEO_TOP1_A_TO_B_DIST] = userGeoFeatures ? userGeoFeatures->B.Top1Dist : NDrive::UserFeaturesDefaultValue;
    features.Floats[NDriveOfferFactors::GEO_TOP1_A_TO_B_FREQ] = userGeoFeatures ? userGeoFeatures->B.Top1Freq : NDrive::UserFeaturesDefaultValue;
    features.Floats[NDriveOfferFactors::GEO_TOP1_A_TO_B_RATIO] = userGeoFeatures ? userGeoFeatures->B.Top1Ratio : NDrive::UserFeaturesDefaultValue;
    features.Floats[NDriveOfferFactors::GEO_TOP2_A_TO_B_DIST] = userGeoFeatures ? userGeoFeatures->B.Top2Dist : NDrive::UserFeaturesDefaultValue;
    features.Floats[NDriveOfferFactors::GEO_TOP2_A_TO_B_FREQ] = userGeoFeatures ? userGeoFeatures->B.Top2Freq : NDrive::UserFeaturesDefaultValue;
    features.Floats[NDriveOfferFactors::GEO_TOP2_A_TO_B_RATIO] = userGeoFeatures ? userGeoFeatures->B.Top2Ratio : NDrive::UserFeaturesDefaultValue;

    features.Categories2[NDriveOfferCatFactors2::FI_GEO_ID_B] = userGeoFeatures ? userGeoFeatures->B.GeoId : TString();
    features.Categories2[NDriveOfferCatFactors2::GEO_TOP1_A_TO_B] = userGeoFeatures ? userGeoFeatures->B.Top1GeoId : TString();
    features.Categories2[NDriveOfferCatFactors2::GEO_TOP2_A_TO_B] = userGeoFeatures ? userGeoFeatures->B.Top2GeoId : TString();
    features.Floats2[NDriveOfferFactors2::FI_DESTINATION_DISTANCE] = distance;
    features.Floats2[NDriveOfferFactors2::FI_DESTINATION_DURATION] = duration;
    features.Floats2[NDriveOfferFactors2::FI_DESTINATION_LATITUDE] = coordinate.Y;
    features.Floats2[NDriveOfferFactors2::FI_DESTINATION_LONGITUDE] = coordinate.X;
    features.Floats2[NDriveOfferFactors2::FI_DESTINATION_RT_SURGE] = geoFeatures ? geoFeatures->RtSurgePrediction1 : NDrive::GeoFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_B_DAY_FREQ] = userGeoFeatures ? userGeoFeatures->B.DayFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_B_DUR_10_20_FREQ] = userGeoFeatures ? userGeoFeatures->B.Dur1020Freq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_B_DUR_10_FREQ] = userGeoFeatures ? userGeoFeatures->B.Dur10Freq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_B_DUR_20_40_FREQ] = userGeoFeatures ? userGeoFeatures->B.Dur2040Freq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_B_DUR_40_FREQ] = userGeoFeatures ? userGeoFeatures->B.Dur40Freq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_B_EVENING_FREQ] = userGeoFeatures ? userGeoFeatures->B.EveningFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_B_FREQ] = userGeoFeatures ? userGeoFeatures->B.Freq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_B_IN_TOP] = userGeoFeatures ? userGeoFeatures->B.InTop : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_B_MORNING_FREQ] = userGeoFeatures ? userGeoFeatures->B.MorningFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_B_RATIO] = userGeoFeatures ? userGeoFeatures->B.Ratio : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_TOP1_A_TO_B_DIST] = userGeoFeatures ? userGeoFeatures->B.Top1Dist : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_TOP1_A_TO_B_FREQ] = userGeoFeatures ? userGeoFeatures->B.Top1Freq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_TOP1_A_TO_B_RATIO] = userGeoFeatures ? userGeoFeatures->B.Top1Ratio : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_TOP2_A_TO_B_DIST] = userGeoFeatures ? userGeoFeatures->B.Top2Dist : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_TOP2_A_TO_B_FREQ] = userGeoFeatures ? userGeoFeatures->B.Top2Freq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_TOP2_A_TO_B_RATIO] = userGeoFeatures ? userGeoFeatures->B.Top2Ratio : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::DESTINATION_SURGE_LAST_DAY] = geoFeatures ? geoFeatures->SurgeLastDay : NDrive::GeoFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::DESTINATION_SURGE_LAST_2_WEEKS] = geoFeatures ? geoFeatures->SurgeLast2Weeks : NDrive::GeoFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::DESTINATION_SURGE_DAY_WEEK_AGO] = geoFeatures ? geoFeatures->SurgeDayWeekAgo : NDrive::GeoFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::DESTINATION_SESSIONS_LAST_DAY] = geoFeatures ? geoFeatures->SessionsLastDay : NDrive::GeoFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::DESTINATION_SESSIONS_LAST_2_WEEKS] = geoFeatures ? geoFeatures->SessionsLast2Weeks : NDrive::GeoFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::DESTINATION_SESSIONS_DAY_WEEK_AGO] = geoFeatures ? geoFeatures->SessionsDayWeekAgo : NDrive::GeoFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::DESTINATION_CARS_LAST_DAY] = geoFeatures ? geoFeatures->CarsLastDay : NDrive::GeoFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::DESTINATION_CARS_LAST_2_WEEKS] = geoFeatures ? geoFeatures->CarsLast2Weeks : NDrive::GeoFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::DESTINATION_CARS_DAY_WEEK_AGO] = geoFeatures ? geoFeatures->CarsDayWeekAgo : NDrive::GeoFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::DESTINATION_ORDERS_LAST_2_WEEKS] = geoFeatures ? geoFeatures->OrdersLast2Weeks : NDrive::GeoFeaturesDefaultValue;
}

void NDrive::CalcDestinationFeatures(TOfferFeatures& features, ui64 geobaseId, const TGeoFeatures* geobaseFeatures, const TUserGeoFeatures* userGeobaseFeatures) {
    features.Categories2[NDriveOfferCatFactors2::FI_GEOBASE_ID_B] = geobaseId ? ToString(geobaseId) : TString();
    features.Floats2[NDriveOfferFactors2::DESTINATION_SURGE_LAST_DAY_GEOBASE] = geobaseFeatures ? geobaseFeatures->SurgeLastDayGeobase : NDrive::GeoFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::DESTINATION_SURGE_LAST_2_WEEKS_GEOBASE] = geobaseFeatures ? geobaseFeatures->SurgeLast2WeeksGeobase : NDrive::GeoFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::DESTINATION_SURGE_DAY_WEEK_AGO_GEOBASE] = geobaseFeatures ? geobaseFeatures->SurgeDayWeekAgoGeobase : NDrive::GeoFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::DESTINATION_SESSIONS_LAST_DAY_GEOBASE] = geobaseFeatures ? geobaseFeatures->SessionsLastDayGeobase : NDrive::GeoFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::DESTINATION_SESSIONS_LAST_2_WEEKS_GEOBASE] = geobaseFeatures ? geobaseFeatures->SessionsLast2WeeksGeobase : NDrive::GeoFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::DESTINATION_SESSIONS_DAY_WEEK_AGO_GEOBASE] = geobaseFeatures ? geobaseFeatures->SessionsDayWeekAgoGeobase : NDrive::GeoFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::DESTINATION_CARS_LAST_DAY_GEOBASE] = geobaseFeatures ? geobaseFeatures->CarsLastDayGeobase : NDrive::GeoFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::DESTINATION_CARS_LAST_2_WEEKS_GEOBASE] = geobaseFeatures ? geobaseFeatures->CarsLast2WeeksGeobase : NDrive::GeoFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::DESTINATION_CARS_DAY_WEEK_AGO_GEOBASE] = geobaseFeatures ? geobaseFeatures->CarsDayWeekAgoGeobase : NDrive::GeoFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::DESTINATION_ORDERS_LAST_2_WEEKS_GEOBASE] = geobaseFeatures ? geobaseFeatures->OrdersLast2WeeksGeobase : NDrive::GeoFeaturesDefaultValue;

    features.Floats2[NDriveOfferFactors2::GEOBASE_B_FREQ] = userGeobaseFeatures ? userGeobaseFeatures->B.GeobaseFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_B_MORNING_FREQ] = userGeobaseFeatures ? userGeobaseFeatures->B.GeobaseMorningFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_B_DAY_FREQ] = userGeobaseFeatures ? userGeobaseFeatures->B.GeobaseDayFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_B_EVENING_FREQ] = userGeobaseFeatures ? userGeobaseFeatures->B.GeobaseEveningFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_B_WEEKEND_FREQ] = userGeobaseFeatures ? userGeobaseFeatures->B.GeobaseWeekendFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_B_WEEKDAY_FREQ] = userGeobaseFeatures ? userGeobaseFeatures->B.GeobaseWeekdayFreq : NDrive::UserFeaturesDefaultValue;

    features.Floats2[NDriveOfferFactors2::GEOBASE_B_WEIGHTED_FREQ] = userGeobaseFeatures ? userGeobaseFeatures->B.GeobaseWeightedFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_B_MORNING_WEIGHTED_FREQ] = userGeobaseFeatures ? userGeobaseFeatures->B.GeobaseMorningWeightedFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_B_DAY_WEIGHTED_FREQ] = userGeobaseFeatures ? userGeobaseFeatures->B.GeobaseDayWeightedFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_B_EVENING_WEIGHTED_FREQ] = userGeobaseFeatures ? userGeobaseFeatures->B.GeobaseEveningWeightedFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_B_WEEKEND_WEIGHTED_FREQ] = userGeobaseFeatures ? userGeobaseFeatures->B.GeobaseWeekendWeightedFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_B_WEEKDAY_WEIGHTED_FREQ] = userGeobaseFeatures ? userGeobaseFeatures->B.GeobaseWeekdayWeightedFreq : NDrive::UserFeaturesDefaultValue;

    features.Floats2[NDriveOfferFactors2::GEOBASE_B_RATIO] = userGeobaseFeatures ? userGeobaseFeatures->B.GeobaseRatio : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_B_MORNING_RATIO] = userGeobaseFeatures ? userGeobaseFeatures->B.GeobaseMorningRatio : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_B_DAY_RATIO] = userGeobaseFeatures ? userGeobaseFeatures->B.GeobaseDayRatio : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_B_EVENING_RATIO] = userGeobaseFeatures ? userGeobaseFeatures->B.GeobaseEveningRatio : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_B_WEEKEND_RATIO] = userGeobaseFeatures ? userGeobaseFeatures->B.GeobaseWeekendRatio : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_B_WEEKDAY_RATIO] = userGeobaseFeatures ? userGeobaseFeatures->B.GeobaseWeekdayRatio : NDrive::UserFeaturesDefaultValue;

    features.Floats2[NDriveOfferFactors2::GEOBASE_B_WEIGHTED_RATIO] = userGeobaseFeatures ? userGeobaseFeatures->B.GeobaseWeightedRatio : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_B_MORNING_WEIGHTED_RATIO] = userGeobaseFeatures ? userGeobaseFeatures->B.GeobaseMorningWeightedRatio : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_B_DAY_WEIGHTED_RATIO] = userGeobaseFeatures ? userGeobaseFeatures->B.GeobaseDayWeightedRatio : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_B_EVENING_WEIGHTED_RATIO] = userGeobaseFeatures ? userGeobaseFeatures->B.GeobaseEveningWeightedRatio : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_B_WEEKEND_WEIGHTED_RATIO] = userGeobaseFeatures ? userGeobaseFeatures->B.GeobaseWeekendWeightedRatio : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_B_WEEKDAY_WEIGHTED_RATIO] = userGeobaseFeatures ? userGeobaseFeatures->B.GeobaseWeekdayWeightedRatio : NDrive::UserFeaturesDefaultValue;

    features.Floats2[NDriveOfferFactors2::GEOBASE_B_PCT] = userGeobaseFeatures ? userGeobaseFeatures->B.GeobasePct : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_B_MORNING_PCT] = userGeobaseFeatures ? userGeobaseFeatures->B.GeobaseMorningPct : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_B_DAY_PCT] = userGeobaseFeatures ? userGeobaseFeatures->B.GeobaseDayPct : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_B_EVENING_PCT] = userGeobaseFeatures ? userGeobaseFeatures->B.GeobaseEveningPct : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_B_WEEKEND_PCT] = userGeobaseFeatures ? userGeobaseFeatures->B.GeobaseWeekendPct : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_B_WEEKDAY_PCT] = userGeobaseFeatures ? userGeobaseFeatures->B.GeobaseWeekdayPct : NDrive::UserFeaturesDefaultValue;

    features.Floats2[NDriveOfferFactors2::DESTINATION_SURGE_LAST_WEEK_GEOBASE] = geobaseFeatures ? geobaseFeatures->SurgeLastWeekGeobase : NDrive::GeoFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::DESTINATION_SESSIONS_LAST_WEEK_GEOBASE] = geobaseFeatures ? geobaseFeatures->SessionsLastWeekGeobase : NDrive::GeoFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::DESTINATION_CARS_LAST_WEEK_GEOBASE] = NDrive::GeoFeaturesDefaultValue;

    features.Floats2[NDriveOfferFactors2::LONG_TERM_IDLE_LEASING_COST_B] = NDrive::GeoFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::MID_TERM_IDLE_LEASING_COST_B] = NDrive::GeoFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::SHORT_TERM_IDLE_LEASING_COST_B] = NDrive::GeoFeaturesDefaultValue;

}

void NDrive::CalcDestinationFeatures(TOfferFeatures& features, const TUserDoubleGeoFeatures* userDoubleGeoFeatures) {
    features.Floats2[NDriveOfferFactors2::GEO_A_TO_B_FREQ] = userDoubleGeoFeatures ? userDoubleGeoFeatures->Freq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_A_TO_B_RATIO] = userDoubleGeoFeatures ? userDoubleGeoFeatures->Ratio : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_A_TO_B_IN_TOP] = userDoubleGeoFeatures ? userDoubleGeoFeatures->InTop : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_A_TO_B_DIST] = userDoubleGeoFeatures ? userDoubleGeoFeatures->Dist : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_A_TO_B_DUR_10_FREQ] = userDoubleGeoFeatures ? userDoubleGeoFeatures->Dur10Freq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_A_TO_B_DUR_10_20_FREQ] = userDoubleGeoFeatures ? userDoubleGeoFeatures->Dur1020Freq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_A_TO_B_DUR_20_40_FREQ] = userDoubleGeoFeatures ? userDoubleGeoFeatures->Dur2040Freq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_A_TO_B_DUR_40_FREQ] = userDoubleGeoFeatures ? userDoubleGeoFeatures->Dur40Freq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_A_TO_B_MORNING_FREQ] = userDoubleGeoFeatures ? userDoubleGeoFeatures->MorningFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_A_TO_B_DAY_FREQ] = userDoubleGeoFeatures ? userDoubleGeoFeatures->DayFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEO_A_TO_B_EVENING_FREQ] = userDoubleGeoFeatures ? userDoubleGeoFeatures->EveningFreq : NDrive::UserFeaturesDefaultValue;
}

void NDrive::CalcDestinationFeatures(TOfferFeatures& features, ui64 geobaseId, const TUserDoubleGeoFeatures* userDoubleGeobaseFeatures) {
    features.Categories2[NDriveOfferCatFactors2::FI_GEOBASE_ID_B] = geobaseId ? ToString(geobaseId) : TString();

    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_FREQ] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_MORNING_FREQ] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseMorningFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_DAY_FREQ] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseDayFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_EVENING_FREQ] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseEveningFreq : NDrive::UserFeaturesDefaultValue;

    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_FREQ_A] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseFreqToA : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_MORNING_FREQ_A] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseMorningFreqToA : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_DAY_FREQ_A] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseDayFreqToA : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_EVENING_FREQ_A] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseEveningFreqToA : NDrive::UserFeaturesDefaultValue;

    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_WEIGHTED_FREQ] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseWeightedFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_MORNING_WEIGHTED_FREQ] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseMorningWeightedFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_DAY_WEIGHTED_FREQ] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseDayWeightedFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_EVENING_WEIGHTED_FREQ] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseEveningWeightedFreq : NDrive::UserFeaturesDefaultValue;

    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_WEIGHTED_FREQ_A] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseWeightedFreqToA : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_MORNING_WEIGHTED_FREQ_A] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseMorningWeightedFreqToA : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_DAY_WEIGHTED_FREQ_A] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseDayWeightedFreqToA : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_EVENING_WEIGHTED_FREQ_A] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseEveningWeightedFreqToA : NDrive::UserFeaturesDefaultValue;

    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_WEEKEND_FREQ] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseWeekendFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_WEEKDAY_FREQ] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseWeekdayFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_WEEKEND_WEIGHTED_FREQ] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseWeekendWeightedFreq : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_WEEKDAY_WEIGHTED_FREQ] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseWeekdayWeightedFreq : NDrive::UserFeaturesDefaultValue;

    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_RATIO] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseRatio : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_MORNING_RATIO] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseMorningRatio : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_DAY_RATIO] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseDayRatio : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_EVENING_RATIO] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseEveningRatio : NDrive::UserFeaturesDefaultValue;

    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_RATIO_A] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseRatioToA : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_MORNING_RATIO_A] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseMorningRatioToA : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_DAY_RATIO_A] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseDayRatioToA : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_EVENING_RATIO_A] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseEveningRatioToA : NDrive::UserFeaturesDefaultValue;

    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_WEIGHTED_RATIO] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseWeightedRatio : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_MORNING_WEIGHTED_RATIO] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseMorningWeightedRatio : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_DAY_WEIGHTED_RATIO] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseDayWeightedRatio : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_EVENING_WEIGHTED_RATIO] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseEveningWeightedRatio : NDrive::UserFeaturesDefaultValue;

    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_WEIGHTED_RATIO_A] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseWeightedRatioToA : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_MORNING_WEIGHTED_RATIO_A] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseMorningWeightedRatioToA : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_DAY_WEIGHTED_RATIO_A] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseDayWeightedRatioToA : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_EVENING_WEIGHTED_RATIO_A] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseEveningWeightedRatioToA : NDrive::UserFeaturesDefaultValue;

    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_WEEKEND_RATIO] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseWeekendRatio : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_WEEKDAY_RATIO] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseWeekdayRatio : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_WEEKEND_WEIGHTED_RATIO] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseWeekendWeightedRatio : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_WEEKDAY_WEIGHTED_RATIO] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseWeekdayWeightedRatio : NDrive::UserFeaturesDefaultValue;

    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_PCT] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobasePct : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_MORNING_PCT] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseMorningPct : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_DAY_PCT] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseDayPct : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_EVENING_PCT] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseEveningPct : NDrive::UserFeaturesDefaultValue;

    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_PCT_A] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobasePctToA : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_MORNING_PCT_A] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseMorningPctToA : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_DAY_PCT_A] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseDayPctToA : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_EVENING_PCT_A] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseEveningPctToA : NDrive::UserFeaturesDefaultValue;

    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_WEEKEND_PCT] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseWeekendPct : NDrive::UserFeaturesDefaultValue;
    features.Floats2[NDriveOfferFactors2::GEOBASE_A_TO_B_WEEKDAY_PCT] = userDoubleGeobaseFeatures ? userDoubleGeobaseFeatures->GeobaseWeekdayPct : NDrive::UserFeaturesDefaultValue;
}

void NDrive::CalcBestDestinationFeatures(TOfferFeatures& features, double score) {
    features.Floats[NDriveOfferFactors::FI_DESTINATION_SCORE] = score;

    features.Floats2[NDriveOfferFactors2::FI_DESTINATION_SCORE] = score;
}

void NDrive::CalcRequestFeatures(TOfferFeatures& features, TInstant timestamp) {
    TInstant shifted = timestamp + TDuration::Hours(3);
    time_t time = shifted.TimeT();
    tm t;
    GmTimeR(&time, &t);
    features.Floats[NDriveOfferFactors::FI_DAY_OF_THE_WEEK] = (t.tm_wday + 6) % 7;
    features.Floats[NDriveOfferFactors::FI_TIME_OF_THE_DAY] = (shifted.Seconds() - 24 * 60 * 60 * shifted.Days()) / 24.0 / 60 / 60;
    features.Floats[NDriveOfferFactors::FI_TIMESTAMP] = timestamp.Seconds();

    features.Floats2[NDriveOfferFactors2::FI_DAY_OF_THE_WEEK] = (t.tm_wday + 6) % 7;
    features.Floats2[NDriveOfferFactors2::FI_TIME_OF_THE_DAY] = (shifted.Seconds() - 24 * 60 * 60 * shifted.Days()) / 24.0 / 60 / 60;
    features.Floats2[NDriveOfferFactors2::FI_HOUR_OF_THE_DAY] = t.tm_hour;
    features.Floats2[NDriveOfferFactors2::FI_TIMESTAMP] = timestamp.Seconds();
}

void NDrive::CalcPriceFeatures(TOfferFeatures& features, float price, float equilibrium) {
    features.Floats[NDriveOfferFactors::FI_PRICE] = price;
    features.Floats[NDriveOfferFactors::FI_NORMALIZED_PRICE] = price / std::max(equilibrium, 1.f);
    features.Floats[NDriveOfferFactors::FI_LOG_PRICE] = std::log(1 + price);

    features.Floats2[NDriveOfferFactors2::FI_PRICE] = price;
    features.Floats2[NDriveOfferFactors2::FI_NORMALIZED_PRICE] = price / std::max(equilibrium, 1.f);
    features.Floats2[NDriveOfferFactors2::FI_LOG_PRICE] = std::log(1 + price);
}

void NDrive::CalcWaitingPriceFeatures(TOfferFeatures& features, float price, float equilibrium) {
    features.Floats[NDriveOfferFactors::FI_WAITING_PRICE] = price;
    features.Floats[NDriveOfferFactors::FI_NORMALIZED_WAITING_PRICE] = price / std::max(equilibrium, 1.f);
    features.Floats[NDriveOfferFactors::FI_LOG_WAITING_PRICE] = std::log(1 + price);

    features.Floats2[NDriveOfferFactors2::FI_WAITING_PRICE] = price;
    features.Floats2[NDriveOfferFactors2::FI_NORMALIZED_WAITING_PRICE] = price / std::max(equilibrium, 1.f);
    features.Floats2[NDriveOfferFactors2::FI_LOG_WAITING_PRICE] = std::log(1 + price);
}

void NDrive::CalcWeatherFeatures(TOfferFeatures& features, const TWeatherInfo& info) {
    features.Floats[NDriveOfferFactors::FI_WEATHER_PRECIPITATION_STRENGTH] = info.GetPrecStrength();
    features.Floats[NDriveOfferFactors::FI_WEATHER_CLOUDNESS] = info.GetCloudness();

    features.Categories2[NDriveOfferCatFactors2::FI_WEATHER_PRECIPITATION_TYPE] = ToString(info.GetPrecType());
    features.Floats2[NDriveOfferFactors2::FI_WEATHER_PRECIPITATION_STRENGTH] = info.GetPrecStrength();
    features.Floats2[NDriveOfferFactors2::FI_WEATHER_CLOUDNESS] = info.GetCloudness();
}

void NDrive::CalcUserFeatures(
    TOfferFeatures& features,
    const TString& userId,
    const TUserFeatures& userFeatures
) {
    features.Floats[NDriveOfferFactors::FI_INV_AVERAGE_SESSION_CLICKS_COUNT] = userFeatures.AverageSessionClicksCount > 0 ? 1 / userFeatures.AverageSessionClicksCount : 0;

    features.Categories2[NDriveOfferCatFactors2::FI_USER_ID] = userId;
    features.Floats2[NDriveOfferFactors2::FI_INV_AVERAGE_SESSION_CLICKS_COUNT] = userFeatures.AverageSessionClicksCount > 0 ? 1 / userFeatures.AverageSessionClicksCount : 0;

    for (size_t i : NDrive::GetExternalUserCatOfferFactors2()) {
        if (i < userFeatures.CatFactors2.size()) {
            features.Categories2[i] = userFeatures.CatFactors2[i].GetOrElse(TString{});
        }
    }

    for (size_t i : NDrive::GetExternalUserOfferFactors()) {
        constexpr float defaultValue = -2;
        auto userFeature = (i < userFeatures.Factors.size()) ? userFeatures.Factors[i] : defaultValue;
        features.Floats[i] = userFeature.GetOrElse(defaultValue);
    }

    for (size_t i : NDrive::GetExternalUserOfferFactors2()) {
        constexpr float defaultValue = -2;
        auto userFeature = (i < userFeatures.Factors2.size()) ? userFeatures.Factors2[i] : defaultValue;
        features.Floats2[i] = userFeature.GetOrElse(defaultValue);
    }
}

void NDrive::CalcUserFeatures(TOfferFeatures& features, const TGeoCoord& coordinate) {
    features.Floats[NDriveOfferFactors::FI_USER_LATITUDE] = coordinate.Y;
    features.Floats[NDriveOfferFactors::FI_USER_LONGITUDE] = coordinate.X;

    features.Floats2[NDriveOfferFactors2::FI_USER_LATITUDE] = coordinate.Y;
    features.Floats2[NDriveOfferFactors2::FI_USER_LONGITUDE] = coordinate.X;
}

void NDrive::CalcUserFeatures(TOfferFeatures& features, TDuration walkingDuration) {
    features.Floats[NDriveOfferFactors::FI_WALKING_DURATION] = walkingDuration.Seconds();

    features.Floats2[NDriveOfferFactors2::FI_WALKING_DURATION] = walkingDuration.Seconds();
}

void NDrive::CalcOfferNameFeatures(TOfferFeatures& features, const TString& name, const TString& groupName) {
    features.Categories2[NDriveOfferCatFactors2::OFFER_NAME] = name;
    features.Categories2[NDriveOfferCatFactors2::OFFER_GROUP_NAME] = groupName;
}

void NDrive::CalcOfferPackPriceFeatures(TOfferFeatures& features, ui32 price, ui32 publicDiscountedPrice) {
    features.Floats2[NDriveOfferFactors2::OFFER_PACK_PRICE] = 0.01 * price;
    features.Floats2[NDriveOfferFactors2::OFFER_PUBLIC_DISCOUNTED_PACK_PRICE] = 0.01 * publicDiscountedPrice;
}

void NDrive::CalcOfferOverrunPriceFeatures(TOfferFeatures& features, ui32 price) {
    features.Floats2[NDriveOfferFactors2::OFFER_OVERRUN_PRICE] = 0.01 * price;
}

void NDrive::CalcOfferDurationFeatures(TOfferFeatures& features, TDuration duration) {
    features.Floats2[NDriveOfferFactors2::OFFER_DURATION] = duration.Seconds();
}

void NDrive::CalcOfferRouteDurationFeatures(TOfferFeatures& features, TDuration duration) {
    features.Floats2[NDriveOfferFactors2::OFFER_ROUTE_DURATION] = duration.Seconds();
}

void NDrive::CalcOfferMileageLimitFeatures(TOfferFeatures& features, float limit) {
    features.Floats2[NDriveOfferFactors2::OFFER_MILEAGE_LIMIT] = limit;
}

void NDrive::CalcOfferAcceptancePriceFeatures(TOfferFeatures& features, ui32 price) {
    features.Floats2[NDriveOfferFactors2::OFFER_ACCEPTANCE_PRICE] = 0.01 * price;
}

void NDrive::CalcOfferDepositAmountFeatures(TOfferFeatures& features, ui32 price) {
    features.Floats2[NDriveOfferFactors2::OFFER_DEPOSIT_AMOUNT] = 0.01 * price;
}

void NDrive::CalcOfferTaxiSurgeFeatures(
    TOfferFeatures& features,
    const NDrive::TTaxiSurgeCalculator::TResult& result
) {
    for (auto&& element : result.Classes) {
        if (element.Name == "econom") {
            features.Floats2[NDriveOfferFactors2::TAXI_OFFER_ECONOM_SURGE] = element.Value;
        } else if (element.Name == "business") {
            features.Floats2[NDriveOfferFactors2::TAXI_OFFER_COMFORT_SURGE] = element.Value;
        } else if (element.Name == "comfortplus") {
            features.Floats2[NDriveOfferFactors2::TAXI_OFFER_COMFORT_PLUS_SURGE] = element.Value;
        } else if (element.Name == "vip") {
            features.Floats2[NDriveOfferFactors2::TAXI_OFFER_BUSINESS_SURGE] = element.Value;
        }
    }
}

void NDrive::CalcOfferTaxiCarSurgeFeatures(
    TOfferFeatures& features,
    const NDrive::TTaxiSurgeCalculator::TResult& result
) {
    for (auto&& element : result.Classes) {
        if (element.Name == "econom") {
            features.Floats2[NDriveOfferFactors2::TAXI_OFFER_ECONOM_CAR_SURGE] = element.Value;
        } else if (element.Name == "business") {
            features.Floats2[NDriveOfferFactors2::TAXI_OFFER_COMFORT_CAR_SURGE] = element.Value;
        } else if (element.Name == "comfortplus") {
            features.Floats2[NDriveOfferFactors2::TAXI_OFFER_COMFORT_PLUS_CAR_SURGE] = element.Value;
        } else if (element.Name == "vip") {
            features.Floats2[NDriveOfferFactors2::TAXI_OFFER_BUSINESS_CAR_SURGE] = element.Value;
        }
    }
}

void NDrive::CalcOfferCashbackPercentFeatures(TOfferFeatures& features, ui32 percent) {
    features.Floats2[NDriveOfferFactors2::OFFER_CASHBACK_PERCENT] = percent;
}

void NDrive::CalcOfferIsPlusUserFeatures(TOfferFeatures& features, bool plus) {
    features.Floats2[NDriveOfferFactors2::OFFER_IS_PLUS_USER] = plus;
}

void NDrive::CalcModelResultFeatures(TOfferFeatures& features, double result) {
    features.Floats2[NDriveOfferFactors2::FI_MATRIXNET] = result;
}

// GetOptionalPriceFeature returns value divided by 100 or -1 if no value presented.
float GetOptionalPriceFeature(const TMaybe<ui32>& price) {
    constexpr float defaultValue = NDrive::GeoFeaturesDefaultValue;
    return price ? 0.01 * (*price) : defaultValue;
}

void NDrive::CalcOfferFixPointAcceptancePriceFeatures(TOfferFeatures& features, const TMaybe<ui32>& price) {
    features.Floats2[NDriveOfferFactors2::OFFER_FIX_POINT_ACCEPTANCE_PRICE] = GetOptionalPriceFeature(price);
}

void NDrive::CalcOfferInsurancePriceFeatures(TOfferFeatures& features, const TMaybe<ui32>& price) {
    features.Floats2[NDriveOfferFactors2::OFFER_INSURANCE_PRICE] = GetOptionalPriceFeature(price);
}

void NDrive::CalcOfferInsurancePackPriceFeatures(TOfferFeatures& features, const TMaybe<ui32>& price) {
    features.Floats2[NDriveOfferFactors2::OFFER_INSURANCE_PACK_PRICE] = GetOptionalPriceFeature(price);
}

namespace {

double GetHashMapValue(const THashMap<TString, double>& map, const TString& key) {
    auto it = map.find(key);
    if (it != map.end()) {
        return it->second;
    }
    return NDrive::GeoFeaturesDefaultValue;
}

}

namespace NDrive {

void CalcOfferRtmrSurgeAreaId(TOfferFeatures& features, const TString& surgeAreaId) {
    features.Categories2[NDriveOfferCatFactors2::RT_SURGE_AREA_ID] = surgeAreaId;
}

void CalcOfferRtmrAreaSurge(TOfferFeatures& features, const NDrive::TRtmrSurge& surge) {
    features.Floats2[NDriveOfferFactors2::RT_SURGE_AREA_PREDICTION_1] = surge.Prediction;
    // TODO(iudovin): Replace with absolute positions.
    features.Floats2[NDriveOfferFactors2::RT_SURGE_AREA_PREDICTION_2] = surge.FloatFeatures.at(surge.FloatFeatures.size() - 3);
    features.Floats2[NDriveOfferFactors2::RT_SURGE_AREA_PREDICTION_3] = surge.FloatFeatures.at(surge.FloatFeatures.size() - 2);
    features.Floats2[NDriveOfferFactors2::RT_SURGE_AREA_PREDICTION_4] = surge.FloatFeatures.at(surge.FloatFeatures.size() - 1);
    features.Floats2[NDriveOfferFactors2::RT_SURGE_AREA_PREDICTION_5] = surge.FloatFeatures.at(4);
}

void CalcOfferRtmrAreaExtendedSurge(TOfferFeatures& features, const NDrive::TRtmrExtendedSurge& surge) {
    features.Floats2[NDriveOfferFactors2::RT_SURGE_AREA_SESSIONS_60_MIN] = GetHashMapValue(surge.Predictions60min, "sessions_prediction");
    features.Floats2[NDriveOfferFactors2::RT_SURGE_AREA_SESSIONS_120_MIN] = GetHashMapValue(surge.Predictions120min, "sessions_prediction");
    features.Floats2[NDriveOfferFactors2::RT_SURGE_AREA_SESSIONS_180_MIN] = GetHashMapValue(surge.Predictions180min, "sessions_prediction");
    // Note: surge = sigmoid(ratio).
    features.Floats2[NDriveOfferFactors2::RT_SURGE_AREA_SURGE_60_MIN] = Sigmoid(GetHashMapValue(surge.Predictions60min, "ratio_prediction"));
    features.Floats2[NDriveOfferFactors2::RT_SURGE_AREA_SURGE_120_MIN] = Sigmoid(GetHashMapValue(surge.Predictions120min, "ratio_prediction"));
    features.Floats2[NDriveOfferFactors2::RT_SURGE_AREA_SURGE_180_MIN] = Sigmoid(GetHashMapValue(surge.Predictions180min, "ratio_prediction"));
    features.Floats2[NDriveOfferFactors2::RT_SURGE_AREA_CARS_180_MIN] = GetHashMapValue(surge.Predictions180min, "cars_prediction");
    // Cars by minutes.
    features.Floats2[NDriveOfferFactors2::RT_SURGE_AREA_CARS_0005_MIN_WINDOW] = surge.FloatFeatures.at(7);
    features.Floats2[NDriveOfferFactors2::RT_SURGE_AREA_CARS_0510_MIN_WINDOW] = surge.FloatFeatures.at(15);
    features.Floats2[NDriveOfferFactors2::RT_SURGE_AREA_CARS_1015_MIN_WINDOW] = surge.FloatFeatures.at(23);
    features.Floats2[NDriveOfferFactors2::RT_SURGE_AREA_CARS_1520_MIN_WINDOW] = surge.FloatFeatures.at(31);
    features.Floats2[NDriveOfferFactors2::RT_SURGE_AREA_CARS_2025_MIN_WINDOW] = surge.FloatFeatures.at(39);
    features.Floats2[NDriveOfferFactors2::RT_SURGE_AREA_CARS_2530_MIN_WINDOW] = surge.FloatFeatures.at(47);
    features.Floats2[NDriveOfferFactors2::RT_SURGE_AREA_CARS_3035_MIN_WINDOW] = surge.FloatFeatures.at(55);
    features.Floats2[NDriveOfferFactors2::RT_SURGE_AREA_CARS_3540_MIN_WINDOW] = surge.FloatFeatures.at(63);
    features.Floats2[NDriveOfferFactors2::RT_SURGE_AREA_CARS_4045_MIN_WINDOW] = surge.FloatFeatures.at(71);
    features.Floats2[NDriveOfferFactors2::RT_SURGE_AREA_CARS_4550_MIN_WINDOW] = surge.FloatFeatures.at(79);
    features.Floats2[NDriveOfferFactors2::RT_SURGE_AREA_CARS_5055_MIN_WINDOW] = surge.FloatFeatures.at(87);
    features.Floats2[NDriveOfferFactors2::RT_SURGE_AREA_CARS_5560_MIN_WINDOW] = surge.FloatFeatures.at(95);
}

void CalcOfferAggressionScoring(TOfferFeatures& features, double value) {
    features.Floats2[NDriveOfferFactors2::USER_AGGRESSIVE_SCORE] = value;
}

void CalcUserLastPricedRideTime(TOfferFeatures& features, TInstant time) {
    features.Floats2[NDriveOfferFactors2::USER_LAST_PRICED_RIDE_TIME] = time.Seconds();
}

void CalcFinishAreaId(TOfferFeatures& features, TString id) {
    features.Categories2[NDriveOfferCatFactors2::FINISH_AREA_ID] = id;
}
}
