#include "context.h"

#include <drive/backend/offers/ranking/calcer.h>
#include <drive/backend/offers/ranking/model.h>

#include <drive/backend/areas/areas.h>
#include <drive/backend/billing/manager.h>
#include <drive/backend/cars/status/state_filters.h>
#include <drive/backend/compiled_riding/manager.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/database/drive/named_filters.h>
#include <drive/backend/models/storage.h>
#include <drive/backend/saas/api.h>

#include <drive/library/cpp/geocoder/formatter/formatter.h>
#include <drive/library/cpp/geofeatures/client.h>
#include <drive/library/cpp/maps_router/router.h>
#include <drive/library/cpp/taxi/request.h>
#include <drive/library/cpp/threading/future_cast.h>
#include <drive/library/cpp/user_events_api/client.h>

#include <library/cpp/geobase/lookup.hpp>
#include <library/cpp/langs/langs.h>

#include <rtline/library/json/adapters.h>
#include <rtline/protos/proto_helper.h>
#include <rtline/util/types/cast.h>

namespace {
    const NUnistat::TIntervals IntervalsWalkingSignals = {0, 60, 120, 180, 240, 300, 360, 420, 480, 540, 600, 660, 720, 780, 840, 900, 960, 1020, 1080, 1140, 1200, 1260, 1320, 1380, 1440, 1500, 1560, 3600};
    class TSignalFeesTime: public TUnistatSignal<double> {
    public:
        TSignalFeesTime()
            : TUnistatSignal<double>({ "fees-time" }, IntervalsWalkingSignals) {
        }
    };

    static TSignalFeesTime SignalFeesTime;
}

TString TUserOfferContext::GetUserId() const {
    return Permissions->GetUserId();
}

TString TUserOfferContext::GetPassportUid() const {
    return Permissions->GetUid();
}

bool TUserOfferContext::FillData(const IRequestProcessor* processor, const NJson::TJsonValue& requestData) {
    return
        ParseAccountId(processor, requestData) &&
        ParseInsuranceInfo(processor, requestData) &&
        ParseDestinations(processor, requestData);
}

bool TUserOfferContext::ParseAccountId(const IRequestProcessor* processor, const NJson::TJsonValue& requestData) {
    const auto& cgi = processor->GetContext()->GetCgiParameters();
    {
        auto account = processor->GetString(cgi, "account_name", false);
        if (account) {
            AccountNames.insert(account);
        }
    }
    {
        auto account = processor->GetString(cgi, "account_id", false);
        if (account) {
            AccountNames.insert(account);
        }
    }
    {
        auto account = processor->GetString(requestData, "account_id", false);
        if (account) {
            AccountNames.insert(account);
        }
    }
    for (const auto& account : requestData["payment_methods"].GetArray()) {
        AccountNames.insert(processor->GetString(account, "account_id", true));
        if (!CreditCard || CreditCard->empty()) {
            ParseCreditCard(processor, account);
        }
        if (!YandexAccountExternalBalance) {
            ParseYandexAccountBalance(processor, account);
        }
    }
    return true;
}

bool TUserOfferContext::ParseCreditCard(const IRequestProcessor* processor, const NJson::TJsonValue& requestData) {
    CreditCard = processor->GetValue<TString>(requestData, "card", false);
    return true;
}

bool TUserOfferContext::ParseYandexAccountBalance(const IRequestProcessor* processor, const NJson::TJsonValue& requestData) {
    YandexAccountExternalBalance = processor->GetValue<i64>(requestData, "yandex_account_balance", false);
    return true;
}

bool TUserOfferContext::PrefetchAccountId() const {
    if (!AccountNames && NeedDefaultAccount) {
        auto eg = BuildEventGuard("last_used_account");
        if (Server->GetDriveAPI()->HasBillingManager()) {
            AccountNames.insert(Server->GetDriveAPI()->GetBillingManager().GetDefaultAccountName(GetUserId()));
        }
    }
    return true;
}

bool TUserOfferContext::PrefetchCreditCard() const {
    if (!CreditCard) {
        auto eg = BuildEventGuard("default_credit_card");
        auto accounts = GetUserAccounts();
        if (Server->GetDriveAPI()->HasBillingManager()) {
            auto card = Server->GetDriveAPI()->GetBillingManager().GetDefaultCreditCard(Server->GetDriveAPI()->GetBillingManager().GetAccountsManager().GetTrustAccount(accounts));
            if (card) {
                CreditCard = *card;
            }
        }
    }
    return true;
}

bool TUserOfferContext::PrefetchUserAccounts() const {
    if (!UserAccounts && Server->GetDriveAPI()->HasBillingManager()) {
        UserAccounts = Server->GetDriveAPI()->GetBillingManager().GetAccountsManager().GetSortedUserAccounts(GetUserId());
    }
    return true;
}

TMaybe<TVector<TPaymentMethod>> TUserOfferContext::GetPaymentMethods() const {
    PrefetchPaymentMethods();
    return PaymentMethods;
}

bool TUserOfferContext::PrefetchPaymentMethods() const {
    auto eg = NDrive::BuildEventGuard("PrefetchPaymentMethods");
    if (!PaymentMethods && NeedYandexPaymentMethod && NeedPaymentMethods) {
        PaymentMethods = Server->GetDriveAPI()->GetUserPaymentMethodsSync(*Permissions, *Server, true);
    }
    YandexPaymentMethod = TBillingManager::GetYandexAccount(PaymentMethods.Get());
    return true;
}

bool TUserOfferContext::PrefetchDistributingBlockEvents() const {
    auto storage = Server->GetDistributingBlockEventsStorage();
    if (NeedDistributingBlockShowsRestriction && storage && !DistributingBlockEvents.Initialized()) {
        DistributingBlockEvents = storage->Retrieve(GetUserId());
    }
    return true;
}

bool TUserOfferContext::PrefetchAggressionScoring() const {
    if (!GetSetting<bool>("offers.fetch_aggression_scoring").GetOrElse(false)) {
        return true;
    }
    auto tagName = GetSetting<TString>("aggression.user_tag").GetOrElse("");
    if (!tagName) {
        return true;
    }
    Y_ENSURE(Server && Server->GetDriveAPI());
    auto& tagsManager = Server->GetDriveAPI()->GetTagsManager().GetUserTags();
    auto tags = tagsManager.GetCachedObject(GetUserId());
    if (!tags) {
        return false;
    }
    for (auto&& tag : tags->GetTags()) {
        if (tag->GetName() == tagName) {
            if (auto scoringTag = tag.GetTagAs<TScoringUserTag>()) {
                AggressionScoring = *scoringTag;
            }
            break;
        }
    }
    return true;
}

bool TUserOfferContext::ParseDestinations(const IRequestProcessor* processor, const NJson::TJsonValue& requestData) {
    TDestinationDescriptions destinations;
    auto genericDestinations = NJson::FromJson<TMaybe<TDestinationDescriptions>>(requestData["destinations"]);
    if (genericDestinations) {
        destinations = std::move(*genericDestinations);
    }

    const auto& cgi = processor->GetContext()->GetCgiParameters();
    auto dst = processor->GetValue<TGeoCoord>(cgi, "dst", /*required=*/false);
    if (!dst) {
        dst = processor->GetValue<TGeoCoord>(cgi, "user_destination", /*required=*/false);
    }
    if (dst && !UserDestination) {
        UserDestination.ConstructInPlace();
        UserDestination->SetCoordinate(*dst);
        UserDestination->SetContextClient(processor->GetString(cgi, "destination_context", /*required=*/false));
        UserDestination->SetName(processor->GetString(cgi, "destination_name", /*required=*/false));
    }

    auto deliveryLocation = NJson::FromJson<TMaybe<TGeoCoord>>(requestData["variables"]["delivery_location"]);
    auto deliveryLocationName = NJson::FromJson<TMaybe<TString>>(requestData["variables"]["delivery_location_name"]);
    if (deliveryLocation && !UserDestination) {
        UserDestination.ConstructInPlace();
        UserDestination->SetCoordinate(*deliveryLocation);
        UserDestination->SetName(deliveryLocationName.GetOrElse({}));
    }

    auto specialDestinations = StringSplitter(Server->GetSettings().GetValueDef<TString>("destination.specials_list", "home, work")).SplitBySet(", ").SkipEmpty().ToList<TString>();
    for (auto&& id : specialDestinations) {
        if (!requestData.Has(id)) {
            continue;
        }
        auto destination = NJson::FromJson<TDestinationDescription>(requestData[id]);
        if (!destination.GetIcon()) {
            destination.SetIcon(Server->GetSettings().GetValueDef<TString>("destination." + id + ".icon", ""));
        }
        if (!destination.GetHintStyle()) {
            destination.SetHintStyle(Server->GetSettings().GetValueDef<TString>("destination." + id + ".hint_style", ""));
        }
        destinations.push_back(std::move(destination));
    }
    ParsedDestinations = std::move(destinations);
    auto suggest = processor->GetValue<bool>(cgi, "suggest", /*required=*/false);
    EnableDestinationSuggest = suggest.GetOrElse(false);
    return true;
}

bool TUserOfferContext::ParseInsuranceInfo(const IRequestProcessor* processor, const NJson::TJsonValue& requestData) {
    if (processor) {
        ComplementaryInsuranceTypes = processor->GetStrings(requestData, "complementary_insurance_types", false);
        InsuranceType = processor->GetValue<TString>(requestData, "insurance_type", false);
        return true;
    } else {
        return
            NJson::ParseField(requestData["complementary_insurance_types"], ComplementaryInsuranceTypes) &&
            NJson::ParseField(requestData["insurance_type"], InsuranceType);
    }
}

const NDrive::TUserFeatures* TUserOfferContext::GetUserFeatures() const {
    auto g = BuildEventGuard("GetUserFeatures");
    PrefetchUserFeatures();
    if (!UserFeatures.Initialized()) {
        return nullptr;
    }
    if (UserFeatures.Wait(GetDeadline()) && UserFeatures.HasValue()) {
        return &UserFeatures.GetValue();
    } else {
        NDrive::TEventLog::Log("GetUserFeaturesError", NJson::TMapBuilder
            ("user_id", GetUserId())
            ("error", NThreading::GetExceptionInfo(UserFeatures))
        );
        return nullptr;
    }
}

TUserOfferContext::TUserOfferContext(const NDrive::IServer* server, TUserPermissions::TConstPtr permissions, IReplyContext::TPtr context)
    : TUserPositionContext(context)
    , Server(server)
    , Permissions(permissions)
{
}

TUserPermissions::TConstPtr TUserOfferContext::GetUserPermissions() const {
    return Permissions;
}

TSet<TString> TUserOfferContext::GetAvailableAccounts(const TSet<TString>& tags, const TSet<TString>& additionalAccounts) const {
    TSet<TString> result;

    const auto& accountDescriptions = GetAccountDescriptions();
    for (auto&& [name, description] : accountDescriptions) {
        if (description.OffersFilter.IsMatching(tags)) {
            result.insert(name);
        }
    }
    result.insert(additionalAccounts.begin(), additionalAccounts.end());
    return result;
}

TMaybe<TSet<TString>> TUserOfferContext::GetAvailableInsuranceTypes() const {
    TMaybe<TSet<TString>> result;

    const auto& requestAccountIds = GetRequestAccountIds();
    const auto& accountDescriptions = GetAccountDescriptions();
    for (auto&& [name, description] : accountDescriptions) {
        if (!requestAccountIds.contains(name)) {
            continue;
        }
        if (description.InsuranceTypes.empty()) {
            continue;
        }
        result = MakeIntersection(std::move(result), description.InsuranceTypes);
    }
    return result;
}

bool TUserOfferContext::CheckRequestAccountId(const TSet<TString>& currentObjectTags, const TSet<TString>& additionalAccounts) const {
    const TSet<TString>& accountIds = GetRequestAccountIds();
    const TSet<TString> chargableAccounts = GetAvailableAccounts(currentObjectTags, additionalAccounts);
    for(const auto& id : accountIds) {
        if (!chargableAccounts.contains(id)) {
            return false;
        }
    }
    return accountIds.size();
}

const TSet<TString>& TUserOfferContext::GetRequestAccountIds() const {
    PrefetchAccountId();
    return AccountNames;
}

const TString& TUserOfferContext::GetRequestCreditCard() const {
    PrefetchCreditCard();
    if (CreditCard) {
        if (auto availableCards = GetPaymentMethods()) {
            for (const auto& card : *availableCards) {
                if (card.Check(*CreditCard)) {
                    return *CreditCard;
                }
            }
            return Default<TString>();
        }
    }
    return CreditCard.GetOrElse(Default<TString>());
}

const TVector<NDrive::NBilling::IBillingAccount::TPtr>& TUserOfferContext::GetUserAccounts() const {
    PrefetchUserAccounts();
    return UserAccounts.GetOrElse(Default<TVector<NDrive::NBilling::IBillingAccount::TPtr>>());
}

TVector<NDrive::NBilling::IBillingAccount::TPtr> TUserOfferContext::GetRequestUserAccounts() const {
    TVector<NDrive::NBilling::IBillingAccount::TPtr> result;
    auto accounts = GetRequestAccountIds();
    for (const auto& account : GetUserAccounts()) {
        if (accounts.contains(account->GetUniqueName())) {
            result.push_back(account);
        }
    }
    return result;
}

TString TUserOfferContext::GetSelectedCharge() const {
    auto accountIds = GetRequestAccountIds();
    if (!accountIds) {
        return "";
    }
    if (accountIds.contains(::ToString(NDrive::NBilling::EAccount::YAccount))) {
        return ::ToString(NDrive::NBilling::EAccount::YAccount);
    }
    return *accountIds.begin();
}

bool IsHomeDestination(const TUserOfferContext::TDestinationDescription& destination) {
    const auto& name = destination.GetName();
    return name == "Дом" || name == "Домой";
}

bool IsWorkDestination(const TUserOfferContext::TDestinationDescription& destination) {
    const auto& name = destination.GetName();
    return name == "Работа" || name == "На работу";
}

const TString USER_DESTINATION_SUGGEST_MODEL = "offers.user_destination_suggest_model";
// USER_DESTINATION_MINIMAL_RADIUS used for points that should be dropped because
// they are near user.
const TString USER_DESTINATION_MINIMAL_RADIUS = "offers.user_destination_minimal_radius";
// USER_DESTINATION_SAME_POINT_RADIUS used for setup radius for points that should be
// treated as the same.
const TString USER_DESTINATION_SAME_POINT_RADIUS = "offers.user_destination_same_point_radius";
// USER_DESTINATION_DROP_POINT_RADIUS used for setup radius of dropping points when
// they are near points with higher score.
const TString USER_DESTINATION_DROP_POINT_RADIUS = "offers.user_destination_drop_point_radius";
// USER_DESTINATION_NAME_LENGTH decribes amount of expected chars in destination title.
const TString USER_DESTINATION_NAME_LENGTH = "offers.user_destination_name_length";
// USER_DESTINATION_NAME_SMART_SHORTEN decribes that smart shorten should be used.
const TString USER_DESTINATION_NAME_SMART_SHORTEN = "offers.user_destination_name_smart_shorten";
// SQ_RANGE_LIMIT used for O(n^2) limitation to O(n*C).
const size_t SQ_RANGE_LIMIT = 20;

const TString GEO_FEATURES_CLIENT_NAME = "geo_features_client_name";

// USER_DESTINATION_COUNT_LIMIT limits amount of hinted destinations.
const TString USER_DESTINATION_COUNT_LIMIT = "offers.user_destination_count_limit";

const TUserOfferContext::TDestinationDescriptions& TUserOfferContext::GetHintedDestinations() const {
    if (HintedDestinations) {
        return *HintedDestinations;
    }
    auto dropRadius = GetSetting<double>(USER_DESTINATION_DROP_POINT_RADIUS).GetOrElse(500);
    auto nameLength = GetSetting<size_t>(USER_DESTINATION_NAME_LENGTH).GetOrElse(20);
    auto nameSmartShorten = GetSetting<bool>(USER_DESTINATION_NAME_SMART_SHORTEN).GetOrElse(true);
    auto language = enum_cast<ELanguage>(Locale);
    if (ParsedDestinations) {
        HintedDestinations.ConstructInPlace();
        HintedDestinations->reserve(ParsedDestinations->size());
        for (auto&& dest : *ParsedDestinations) {
            if (ExternalUserPosition && dest.GetCoordinate().GetLengthTo(*ExternalUserPosition) < dropRadius) {
                continue;
            }
            HintedDestinations->push_back(dest);
        }
        std::stable_sort(
            HintedDestinations->begin(), HintedDestinations->end(),
            [](const TDestinationDescription& lhs, const TDestinationDescription& rhs) {
                return (lhs.GetKind() != "saved") < (rhs.GetKind() != "saved");
            }
        );
        for (auto&& dest : *HintedDestinations) {
            if (nameSmartShorten) {
                dest.SetName(NDrive::SmartShortenAddress(dest.GetName(), nameLength, language));
            } else {
                dest.SetName(NDrive::ShortenAddress(dest.GetName(), nameLength, language));
            }
        }
    }
    if (!HintedDestinations) {
        HintedDestinations.ConstructInPlace();
    }
    if (GetNeedCarSuggest()) {
        // In this case we use destinations only from SAAS.
        ExtractPredictorHintedDestinations();
    } else if (EnableDestinationSuggest && !HasUserDestination()) {
        SetupHomeWorkDestinations();
        ExtractTaxiHintedDestinations();
        ExtractPredictorHintedDestinations();
    }
    auto limit = GetSetting<size_t>(USER_DESTINATION_COUNT_LIMIT).GetOrElse(5);
    if (HintedDestinations->size() > limit) {
        HintedDestinations->erase(HintedDestinations->begin() + limit, HintedDestinations->end());
    }
    for (auto&& i : *HintedDestinations) {
        i.Prefetch(*this);
    }
    return *HintedDestinations;
}

void TUserOfferContext::SetupHomeWorkDestinations() const {
    if (!ParsedDestinations) {
        return;
    }
    for (auto&& destination : *ParsedDestinations) {
        if (IsHomeDestination(destination)) {
            HomeDestination = destination;
        } else if (IsWorkDestination(destination)) {
            WorkDestination = destination;
        }
    }
}

void TUserOfferContext::ExtractTaxiHintedDestinations() const {
    if (!TaxiSuggest.Initialized()) {
        return;
    }
    auto eg = BuildEventGuard("wait_taxi_suggest");
    if (!TaxiSuggest.Wait(GetDeadline())) {
        if (eg) {
            eg->AddEvent("taxi_suggest_timeout");
        }
        NDrive::TEventLog::Log("TaxiSuggestTimeout", NJson::TMapBuilder
            ("deadline", NJson::ToJson(Deadline))
        );
    }
    if (!TaxiSuggest.HasValue()) {
        TString error = NThreading::GetExceptionMessage(TaxiSuggest);
        if (eg) {
            eg->AddEvent(NJson::TMapBuilder
                ("event", "taxi suggest error")
                ("error", error)
            );
        }
        NDrive::TEventLog::Log("TaxiSuggestError", NJson::TMapBuilder
            ("error", std::move(error))
        );
        return;
    }
    const auto& elements = TaxiSuggest.GetValue().Elements;
    const ui32 destCount = std::min<ui32>(DestinationsCountLimit, elements.size());
    if (destCount) {
        HintedDestinations->clear();
    }
    for (size_t i = 0; i < destCount; ++i) {
        const auto& element = elements[i];
        TDestinationDescription destination;
        destination.SetName(element.Title);
        destination.SetCoordinate({element.Longitude, element.Latitude});
        if (IsHomeDestination(destination) && HomeDestination) {
            destination = *HomeDestination;
        } else if (IsWorkDestination(destination) && WorkDestination) {
            destination = *WorkDestination;
        }
        HintedDestinations->emplace_back(std::move(destination));
    }
}

void TUserOfferContext::ExtractPredictorHintedDestinations() const {
    auto g = BuildEventGuard("wait_user_destination_suggest");
    auto modelName = GetSetting<TString>(USER_DESTINATION_SUGGEST_MODEL).GetOrElse("");
    if (!modelName) {
        return;
    }
    auto geocoder = Server->GetDriveAPI()->HasGeocoderClient() ? &Server->GetDriveAPI()->GetGeocoderClient() : nullptr;
    if (!geocoder) {
        if (g) {
            g->AddEvent("GeocoderMissing");
        }
        return;
    }
    auto modelsStorage = Server->GetModelsStorage();
    auto model = modelsStorage ? modelsStorage->GetOfferModel(modelName) : nullptr;
    if (!model) {
        if (g) {
            g->AddEvent(TStringBuilder() << "UserDestinationSuggest model " << modelName << " is missing");
        }
        return;
    }
    auto destinationSuggestPtr = GetUserDestinationSuggest();
    if (!destinationSuggestPtr) {
        if (g) {
            g->AddEvent("UserDestinationSuggest is missing");
        }
        return;
    }
    auto destinationSuggest = *destinationSuggestPtr;
    const auto& userFeatures = *Yensured(GetUserFeatures());
    auto& elements = destinationSuggest.Elements;
    for (auto i = elements.begin(); i != elements.end(); ++i) {
        const auto asyncGeoFeatures = NThreading::Initialize(i->GeoFeatures);
        const auto asyncGeobaseFeatures = NThreading::Initialize(i->GeobaseFeatures);
        const auto asyncUserGeoFeatures = NThreading::Initialize(i->UserGeoFeatures);
        const auto asyncUserGeobaseFeatures = NThreading::Initialize(i->UserGeobaseFeatures);
        {
            auto eg = BuildEventGuard("wait_geo_features");
            if (!asyncGeoFeatures.Wait(Deadline)) {
                if (eg) {
                    eg->AddEvent("GeoFeatures timeout");
                }
            }
        }
        {
            auto eg = BuildEventGuard("wait_geo_features");
            if (!asyncGeobaseFeatures.Wait(Deadline)) {
                if (eg) {
                    eg->AddEvent("GeobaseFeatures timeout");
                }
            }
        }
        {
            auto eg = BuildEventGuard("wait_user_geo_features");
            if (!asyncUserGeoFeatures.Wait(Deadline)) {
                if (eg) {
                    eg->AddEvent("UserGeoFeatures timeout");
                }
            }
        }
        {
            auto eg = BuildEventGuard("wait_user_geobase_features");
            if (!asyncUserGeobaseFeatures.Wait(Deadline)) {
                if (eg) {
                    eg->AddEvent("UserGeobaseFeatures timeout");
                }
            }
        }
        const auto coordinate = TGeoCoord(i->Longitude, i->Latitude);
        const auto geoFeatures = asyncGeoFeatures.HasValue() ? asyncGeoFeatures.GetValue().Get() : nullptr;
        const auto geobaseFeatures = asyncGeobaseFeatures.HasValue() ? asyncGeobaseFeatures.GetValue().Get() : nullptr;
        const auto userGeoFeatures = asyncUserGeoFeatures.HasValue() ? asyncUserGeoFeatures.GetValue().Get() : nullptr;
        const auto userGeobaseFeatures = asyncUserGeobaseFeatures.HasValue() ? asyncUserGeobaseFeatures.GetValue().Get() : nullptr;
        auto geobaseId = i->GeobaseId.GetOrElse(0);
        NDrive::TOfferFeatures features;
        NDrive::CalcUserFeatures(features, GetUserId(), userFeatures);
        NDrive::CalcDestinationFeatures(features, coordinate, geoFeatures, userGeoFeatures, -2, -2);
        NDrive::CalcDestinationFeatures(features, geobaseId, geobaseFeatures, userGeobaseFeatures);
        i->Score = model->Calc(features);
    }
    if (ExternalUserPosition) {
        DropNearDestinationSuggests(
            elements, *ExternalUserPosition,
            GetSetting<double>(USER_DESTINATION_MINIMAL_RADIUS).GetOrElse(1000)
        );
    }
    auto cmp = [](const TUserDestinationSuggestEx::TElement& lhs, const TUserDestinationSuggestEx::TElement& rhs) {
        return *lhs.Score > *rhs.Score;
    };
    std::sort(elements.begin(), elements.end(), cmp);
    auto destCount = std::min<size_t>(DestinationsCountLimit, elements.size());
    if (destCount) {
        HintedDestinations->clear();
    }
    DropNearDestinationSuggests(
        elements,
        GetSetting<double>(USER_DESTINATION_DROP_POINT_RADIUS).GetOrElse(500)
    );
    destCount = std::min(destCount, elements.size());
    for (size_t i = 0; i < destCount; ++i) {
        const auto& element = elements[i];
        TDestinationDescription destination;
        const auto geocoderResp = NThreading::Initialize(element.GeocoderResponse);
        destination.SetName(Sprintf("%.5f %.5f", element.Longitude, element.Latitude));
        destination.SetCoordinate({element.Longitude, element.Latitude});
        auto eg = BuildEventGuard("wait_geocoder_response");
        if (geocoderResp.Wait(Deadline)) {
            if (geocoderResp.HasValue()) {
                destination.SetName(geocoderResp.GetValue().Title);
            } else {
                NDrive::TEventLog::Log("GetDestinationGeocoderResponse", NJson::TMapBuilder
                    ("coordinate", NJson::ToJson(destination.GetCoordinate().ToString()))
                    ("exception", NThreading::GetExceptionInfo(geocoderResp))
                );
                if (eg) {
                    eg->AddEvent(NJson::TMapBuilder
                        ("event", "geocoder_error")
                        ("exception", NThreading::GetExceptionInfo(geocoderResp))
                    );
                }
            }
        } else {
            if (eg) {
                eg->AddEvent("geocoder_timeout");
            }
        }
        HintedDestinations->emplace_back(std::move(destination));
    }
    FixHintedDestinations();
}

void TUserOfferContext::FixHintedDestinations() const {
    if (!HintedDestinations || !ParsedDestinations) {
        return;
    }
    auto language = enum_cast<ELanguage>(Locale);
    size_t nameLength = GetSetting<size_t>(USER_DESTINATION_NAME_LENGTH).GetOrElse(20);
    bool nameSmartShorten = GetSetting<bool>(USER_DESTINATION_NAME_SMART_SHORTEN).GetOrElse(true);
    for (auto& destination : *HintedDestinations) {
        size_t best = ParsedDestinations->size();
        double bestLength = GetSetting<double>(USER_DESTINATION_SAME_POINT_RADIUS).GetOrElse(100);
        for (size_t j = 0; j < std::min(ParsedDestinations->size(), SQ_RANGE_LIMIT); j++) {
            const auto& parsed = (*ParsedDestinations)[j];
            if (parsed.GetKind() != "saved" && !IsHomeDestination(parsed) && !IsWorkDestination(parsed)) {
                continue;
            }
            double length = destination.GetCoordinate().GetLengthTo(parsed.GetCoordinate());
            if (length < bestLength) {
                best = j;
                bestLength = length;
            }
        }
        if (best != ParsedDestinations->size()) {
            destination.SetName((*ParsedDestinations)[best].GetName());
            destination.SetCoordinate((*ParsedDestinations)[best].GetCoordinate());
        }
        if (nameSmartShorten) {
            destination.SetName(NDrive::SmartShortenAddress(destination.GetName(), nameLength, language));
        } else {
            destination.SetName(NDrive::ShortenAddress(destination.GetName(), nameLength, language));
        }
    }
}

void DropNearDestinationSuggests(
    TVector<TUserOfferContext::TUserDestinationSuggestEx::TElement>& elements,
    TGeoCoord point, double radius
) {
    size_t newSize = 0;
    for (size_t i = 0; i < elements.size(); i++) {
        TGeoCoord coord{elements[i].Longitude, elements[i].Latitude};
        if (coord.GetLengthTo(point) < radius) {
            continue;
        }
        elements[newSize] = elements[i];
        newSize++;
    }
    elements.erase(elements.begin() + newSize, elements.end());
}

void DropNearDestinationSuggests(
    TVector<TUserOfferContext::TUserDestinationSuggestEx::TElement>& elements,
    double radius
) {
    if (elements.size() < 2) {
        return;
    }
    size_t newSize = 1;
    for (size_t i = 1; i < elements.size(); i++) {
        TGeoCoord coordI{elements[i].Longitude, elements[i].Latitude};
        bool needDrop = false;
        for (size_t j = std::max(i, SQ_RANGE_LIMIT) - SQ_RANGE_LIMIT; j < i; j++) {
            TGeoCoord coordJ{elements[j].Longitude, elements[j].Latitude};
            if (coordI.GetLengthTo(coordJ) < radius) {
                needDrop = true;
                break;
            }
        }
        if (needDrop) {
            continue;
        }
        elements[newSize] = elements[i];
        newSize++;
    }
    elements.erase(elements.begin() + newSize, elements.end());
}

const TUserOfferContext::TUserDestinationSuggestFuture& TUserOfferContext::GetUserDestinationSuggestFuture() const {
    PrefetchUserDestinationSuggest();
    return UserDestinationSuggest;
}

bool TUserOfferContext::PrefetchHistoryFees() const {
    if (CompiledSessions) {
        return true;
    }
    const auto g = BuildEventGuard("PrefetchHistoryFees");
    const TString keyOffersDeepSettings = (CommonSettingsPrefix ? (CommonSettingsPrefix + ".") : "") + "offers.fees_history_deep";
    const TInstant start = ModelingNow() - Permissions->GetSetting<TDuration>(keyOffersDeepSettings, TDuration::Days(7));
    const auto& compiledSessionsManager = Server->GetDriveAPI()->GetMinimalCompiledRides();
    auto tx = compiledSessionsManager.BuildSession(true);
    CompiledSessions = compiledSessionsManager.GetEvents<TCompiledRiding>({}, start, tx, NSQL::TQueryOptions()
        .AddGenericCondition("history_user_id", GetUserId())
    );
    return true;
}

bool TUserOfferContext::FetchFeesInfo() const {
    if (!GetNeedHistoryFreeTimeFees()) {
        return true;
    }
    if (!CompiledSessions) {
        PrefetchHistoryFees();
    }
    auto g = BuildEventGuard("FetchFeesInfo");
    if (!CompiledSessions) {
        return false;
    }
    const auto& userRides = *CompiledSessions;
    if (g) {
        g->AddEvent(NJson::TMapBuilder
            ("event", "FetchedCompiledSessions")
            ("count", userRides.size())
        );
    }
    LastPricedRideTime = TInstant::Zero();
    {
        std::multimap<TString, TDuration> fees;
        TDuration feesDuration = TDuration::Zero();
        TVector<TAtomicSharedPtr<const ISession>> sessions;
        if (!Server->GetDriveAPI()->GetCurrentUserSessions(GetUserId(), sessions, TInstant::Zero())) {
            return false;
        }
        auto ignoreFeesFromAcceptance = GetSetting<bool>("offers.ignore_fees_from_acceptance").GetOrElse(false);
        for (auto&& s : sessions) {
            TDuration freeDuration;
            TDuration pricedDuration;
            const TBillingSession* bSession = dynamic_cast<const TBillingSession*>(s.Get());
            if (bSession) {
                bSession->GetFreeAndPricedDurations(freeDuration, pricedDuration);
                TMaybe<TBillingCompilation> compilation = bSession->GetCompilationAs<TBillingCompilation>();
                if (compilation) {
                    if (compilation->GetReportSumPrice() > 0) {
                        LastPricedRideTime = std::max(LastPricedRideTime.GetOrElse(TInstant::Zero()), bSession->GetLastTS());
                    }
                    if (ignoreFeesFromAcceptance && compilation->GetHasAcceptance()) {
                        if (!compilation->IsAccepted()) {
                            // In this case we assume that there was a problems with car.
                            continue;
                        }
                        TSnapshotsDiffCompilation snapshotsCompilation;
                        if (bSession->FillCompilation(snapshotsCompilation)) {
                            TMaybe<TSnapshotsDiff> diff = snapshotsCompilation.GetSnapshotsDiff(Server);
                            if (diff && diff->GetMileageDef(0) > 0) {
                                continue;
                            }
                        }
                    }
                }
                auto fee = freeDuration - pricedDuration;
                fees.emplace(bSession->GetSessionId(), fee);
                feesDuration += fee;
            }
        }
        ui32 idx = 0;
        for (auto it = userRides.rbegin(); it != userRides.rend(); ++it) {
            const TCompiledRiding* compiledRide = &*it;
            if (compiledRide->GetSumPrice() > 0) {
                LastPricedRideTime = std::max(LastPricedRideTime.GetOrElse(TInstant::Zero()), compiledRide->GetFinishInstant().Get());
            }
            if (!compiledRide->HasSnapshotsDiff() || !compiledRide->GetSnapshotsDiffUnsafe().HasMileage() || compiledRide->GetSnapshotsDiffUnsafe().GetMileageUnsafe() >= 1) {
                ++idx;
                break;
            }
            if (compiledRide->GetSumPrice() > 100) {
                ++idx;
            }
            if (ignoreFeesFromAcceptance && compiledRide->GetAcceptanceDurationDef(TDuration::Zero())) {
                if (!compiledRide->GetRidingDurationDef(TDuration::Zero())) {
                    // In this case we assume that there was a problems with car.
                    continue;
                }
            }
            auto fee = compiledRide->GetUselessDuration();
            fees.emplace(compiledRide->GetSessionId(), fee);
            feesDuration += fee;
        }
        PricedRidesCount = idx;
        if (feesDuration >= Permissions->GetSetting<TDuration>("offers.min_fees_time_duration", TDuration::Minutes(3))) {
            TDiscount discount;
            discount.SetIdentifier("fee");
            discount.SetVisible(false);
            {
                TDiscount::TDiscountDetails d;
                d.SetAdditionalTime(-1 * feesDuration.Minutes() * 60).SetTagName("old_state_reservation");
                SignalFeesTime.Signal(feesDuration.Minutes() * 60);
                discount.AddDetails(d);
            }
            AdditionalDurationDiscounts.emplace_back(std::move(discount));
            NDrive::TEventLog::Log("FeesInfo", NJson::TMapBuilder
                ("fees", NJson::ToJson(fees))
                ("total", NJson::ToJson(feesDuration))
            );
        }
    }
    return true;
}

bool TUserOfferContext::PrefetchTaxiSuggest() const {
    if (!!TaxiSuggest.Initialized()) {
        return true;
    }
    if (!HasUserPosition()) {
        return false;
    }
    auto taxiSuggestClient = Server->GetTaxiSuggest();
    bool hasCustomDestinations = false;
    if (ParsedDestinations) {
        for (auto&& destination : *ParsedDestinations) {
            if (!IsHomeDestination(destination) && !IsWorkDestination(destination)) {
                hasCustomDestinations = true;
                break;
            }
        }
    }
    if (taxiSuggestClient && !UserDestination && (!hasCustomDestinations || EnableDestinationSuggest)) {
        TaxiSuggest = taxiSuggestClient->GetZeroSuggest(
            GetUserPositionUnsafe().Y, GetUserPositionUnsafe().X,
            Permissions->GetUserFeatures().GetUid(), Permissions->GetUserFeatures().GetTVMTicket(),
            enum_cast<ELanguage>(Locale)
        );
    }
    return true;
}

bool TUserOfferContext::PrefetchUserDestinationSuggest() const {
    if (!UserDestinationSuggest.Initialized()) {
        UserDestinationSuggest = FetchUserDestinationSuggest();
    }
    return true;
}

bool TUserOfferContext::PrefetchUserFeatures() const {
    if (!UserFeatures.Initialized()) {
        UserFeatures = FetchUserFeatures();
    }
    return true;
}

bool TUserOfferContext::PrefetchDestinationsInfo() const {
    if (UserDestination) {
        UserDestination->Prefetch(*this);
    }
    if (GetEnableTaxiSuggest() && !PrefetchTaxiSuggest()) {
        return false;
    }
    GetHintedDestinations();
    return true;
}

void TUserOfferContext::PrefetchTaxiSurge() const {
    if (!TaxiSurge.Initialized()) {
        TaxiSurge = FetchTaxiSurge();
    }
}

const TString FETCH_TAXI_SURGE = "offers.fetch_taxi_surge";
const TString FETCH_TAXI_COMFORT_PRICE = "offers.fetch_taxi_comfort_price";
const TString FETCH_TAXI_COMFORT_PLUS_PRICE = "offers.fetch_taxi_comfort_plus_price";
const TString FETCH_TAXI_BUSINESS_PRICE = "offers.fetch_taxi_business_price";

NThreading::TFuture<NDrive::TTaxiSurgeCalculator::TResult> TUserOfferContext::FetchTaxiSurge() const {
    if (!HasUserPosition() || !GetSetting<bool>(FETCH_TAXI_SURGE).GetOrElse(false)) {
        return {};
    }
    auto client = Server->GetTaxiSurgeCalculator();
    if (!client) {
        return {};
    }
    return client->GetMapSurge(GetUserPositionRef());
}

TSimpleTaxiReply::TPtr TUserOfferContext::FetchTaxiPrice(const TString& taxiClass) const {
    if (!HasUserDestination() || !HasUserPosition()) {
        return nullptr;
    }
    auto taxiRouteInfoClient = Server->GetTaxiRouteInfoClient();
    if (!taxiRouteInfoClient) {
        return nullptr;
    }

    auto deadline = GetDeadline();
    auto timeout = deadline - Now();
    TTaxiRequest request(GetUserPositionRef(), GetUserDestinationRef().GetCoordinate(), taxiClass);
    return taxiRouteInfoClient->SendRequest(request, timeout);
}

void TUserOfferContext::PrefetchTaxiPrices() const {
    if (GetNeedTaxiPrice() && UserDestination) {
        if (!TaxiPrices[TTaxiRequest::ECONOM_CLASS]) {
            TaxiPrices[TTaxiRequest::ECONOM_CLASS] = FetchTaxiPrice(TTaxiRequest::ECONOM_CLASS);
        }
        bool fetchComfort = GetSetting<bool>(FETCH_TAXI_COMFORT_PRICE).GetOrElse(true);
        if (!TaxiPrices[TTaxiRequest::COMFORT_CLASS] && fetchComfort) {
            TaxiPrices[TTaxiRequest::COMFORT_CLASS] = FetchTaxiPrice(TTaxiRequest::COMFORT_CLASS);
        }
        bool fetchComfortPlus = GetSetting<bool>(FETCH_TAXI_COMFORT_PLUS_PRICE).GetOrElse(true);
        if (!TaxiPrices[TTaxiRequest::COMFORT_PLUS_CLASS] && fetchComfortPlus) {
            TaxiPrices[TTaxiRequest::COMFORT_PLUS_CLASS] = FetchTaxiPrice(TTaxiRequest::COMFORT_PLUS_CLASS);
        }
        bool fetchBusiness = GetSetting<bool>(FETCH_TAXI_BUSINESS_PRICE).GetOrElse(true);
        if (!TaxiPrices[TTaxiRequest::BUSINESS_CLASS] && fetchBusiness) {
            TaxiPrices[TTaxiRequest::BUSINESS_CLASS] = FetchTaxiPrice(TTaxiRequest::BUSINESS_CLASS);
        }
    }
}

TOptionalGeobaseId TUserOfferContext::FetchGeobaseId(const TGeoCoord& coordinate) const {
    auto geobase = Server ? Server->GetGeobase() : nullptr;
    if (geobase) {
        return geobase->GetRegionIdByLocation(coordinate.Y, coordinate.X);
    } else {
        return {};
    }
}

template <class T>
TUserOfferContext::TGeoFeaturesFuture TUserOfferContext::FetchGeoFeatures(const T& id) const {
    auto geoFeaturesClientName = GetSetting<TString>(GEO_FEATURES_CLIENT_NAME).GetOrElse("");
    auto geoFeaturesClient = Server->GetGeoFeaturesClient(geoFeaturesClientName);
    if (geoFeaturesClient && GetNeedGeoFeatures()) {
        return geoFeaturesClient->Get(id);
    } else {
        return {};
    }
}

TUserOfferContext::TUserDestinationSuggestFuture TUserOfferContext::FetchUserDestinationSuggest() const {
    auto featuresClient = GetFeaturesClient();
    if (featuresClient && GetNeedUserDestinationSuggest()) {
        auto suggest = featuresClient->GetUserDestinationSuggest(GetUserId());
        if (!suggest.Initialized()) {
            return {};
        }
        auto userId = GetUserId();
        bool hasSuggestModel = !!GetSetting<TString>(USER_DESTINATION_SUGGEST_MODEL).GetOrElse("");
        auto geoFeaturesClientName = GetSetting<TString>(GEO_FEATURES_CLIENT_NAME).GetOrElse("");
        auto applyer = [
            server = Server,
            userId = std::move(userId),
            locale = Locale,
            featuresClient,
            hasSuggestModel,
            geoFeaturesClientName = std::move(geoFeaturesClientName)
        ](const NThreading::TFuture<NDrive::TUserDestinationSuggest>& suggest) {
            const auto& value = suggest.GetValue();
            auto geobase = server ? server->GetGeobase() : nullptr;
            auto geoFeaturesClient = server ? server->GetGeoFeaturesClient(geoFeaturesClientName) : Nothing();
            auto geocoder = server->GetDriveAPI()->HasGeocoderClient() ? &server->GetDriveAPI()->GetGeocoderClient() : nullptr;
            auto language = enum_cast<ELanguage>(locale);
            TUserDestinationSuggestEx result;
            for (auto&& i : value.Elements) {
                TUserDestinationSuggestEx::TElement element = i;
                if (geobase) {
                    element.GeobaseId = geobase->GetRegionIdByLocation(i.Latitude, i.Longitude);
                }
                if (geoFeaturesClient) {
                    element.GeoFeatures = geoFeaturesClient->Get(i.Latitude, i.Longitude);
                }
                if (geoFeaturesClient && element.GeobaseId) {
                    element.GeobaseFeatures = geoFeaturesClient->Get(*element.GeobaseId);
                }
                if (featuresClient && element.GeobaseId) {
                    element.UserGeobaseFeatures = featuresClient->GetUserGeoFeatures(userId, *element.GeobaseId);
                }
                if (geocoder && hasSuggestModel) {
                    element.GeocoderResponse = geocoder->Decode({i.Longitude, i.Latitude}, language);
                }
                result.Elements.push_back(std::move(element));
            }
            return result;
        };
        return suggest.Apply(applyer);
    } else {
        return {};
    }
}

TUserOfferContext::TUserFeaturesFuture TUserOfferContext::FetchUserFeatures() const {
    auto userInfoApi = GetFeaturesClient();
    if (userInfoApi && GetNeedUserFeatures()) {
        return userInfoApi->GetUserFeatures(GetUserId());
    } else {
        return {};
    }
}

template <class T>
TUserOfferContext::TUserGeoFeaturesFuture TUserOfferContext::FetchUserGeoFeatures(const T& id) const {
    auto userInfoApi = GetFeaturesClient();
    if (userInfoApi && GetNeedUserFeatures()) {
        return userInfoApi->GetUserGeoFeatures(GetUserId(), id);
    } else {
        return {};
    }
}

TFinishAreaInfo TUserOfferContext::FetchFinishPointArea(const TGeoCoord& c) const {
    auto observableTags = Permissions ? Permissions->GetTagNamesByActionPtr(TTagAction::ETagAction::Observe) : nullptr;
    auto dDef = GetSetting<TDuration>("offers.fix_point.walking_duration").GetOrElse(TDuration::Minutes(7));
    TFinishAreaInfo preResult;
    preResult.SetWalkingDuration(dDef);
    auto actor = [this, observableTags, &preResult](const TFixPointCorrectionTag* tag) -> bool {
        if (!tag) {
            return true;
        }
        if (tag->GetName() != TFixPointCorrectionTag::TypeName) {
            if (observableTags && !observableTags->contains(tag->GetName()))  {
                return true;
            }
        }
        if (tag->GetLinkArea()) {
            const TMaybe<TArea> area = Server->GetDriveAPI()->GetAreasDB()->GetObject(tag->GetLinkArea());
            const TMaybe<TArea> areaPublic = Server->GetDriveAPI()->GetAreasDB()->GetObject(tag->GetLinkPublicArea());
            if (area) {
                preResult.SetFinishArea(area->GetCoords());
                if (areaPublic) {
                    preResult.SetFinishAreaPublic(areaPublic->GetCoords());
                }
                return true;
            }
        } else if (tag->GetArea().size()) {
            preResult.SetFinishArea(tag->GetArea());
            if (tag->GetPublicArea().size()) {
                preResult.SetFinishAreaPublic(tag->GetPublicArea());
            }
            return true;
        }
        if (tag->HasWalkingDuration()) {
            preResult.SetWalkingDuration(tag->GetWalkingDurationUnsafe());
            return true;
        }
        return false;
    };
    Server->GetDriveAPI()->GetAreasDB()->ProcessHardTagsInPoint<TFixPointCorrectionTag>(c, actor, TInstant::Zero());
    if (preResult.HasFinishArea() && preResult.GetFinishAreaUnsafe().size()) {
        return preResult;
    }

    bool useStubRouter = GetSetting<bool>("stub_router.enabled").GetOrElse(false);
    if (useStubRouter || !NeedFinishArea) {
        TGeoRect finish = { c };
        finish.GrowDistance(dDef.Seconds());
        preResult.SetFinishArea(finish.GetCoords());
        return preResult;
    }
    const TDuration reqTimeout = Server->GetSettings().GetValueDef<TDuration>("offers.fix_point.stop_area_timeout", TDuration::Seconds(5));
    auto area = GetSearchAreaFromRouter(*Server, Permissions, reqTimeout, preResult.GetWalkingDuration(), c);
    if (!area.Initialized()) {
        return preResult;
    }
    preResult.SetDeadline(Now() + reqTimeout);
    preResult.SetFutureArea(std::move(area));
    return preResult;
}

NThreading::TFuture<TVector<TGeoCoord>> TUserOfferContext::GetSearchAreaFromRouter(const NDrive::IServer& server, TUserPermissions::TConstPtr permissions, const TDuration& reqTimeout, const TDuration& walkingDuration, const TGeoCoord& coord) {
    NThreading::TFuture<TVector<TGeoCoord>> result;
    if (Yensured(permissions)->GetSetting<bool>(server.GetSettings(), "pedestrian_router.isochrone_enabled").GetOrElse(false)) {
        auto router = server.GetPedestrianRouter();
        Y_ENSURE(router);
        result = router->GetIsochrone(coord, walkingDuration);
    } else {
        const TRTLineAPI* api = Yensured(GetRTLineAPI(server, "ptPedestrian"));
        auto& sClient = api->GetSearchClient();
        NRTLine::TQuery query;
        TStringBuilder sb;
        sb << "type:routing_area;use_approx:da;";
        sb << "d:" << server.GetSettings().GetValueDef<ui32>("offers.fix_point.finish_precision(m)", 500) << ";";
        sb << "start:" << coord.ToString() << ";";
        sb << "border:" << walkingDuration.Seconds() << ";";
        sb << "perm:ptPedestrian;";
        sb << "build_convex:true;";
        query.SetText(sb);
        query.SetTimeout(reqTimeout);
        query.AddExtraParam("component", "Graph");
        result = sClient.SendAsyncQueryF(query, reqTimeout).Apply(
            [](const NThreading::TFuture<NRTLine::TSearchReply>& r) -> TVector<TGeoCoord> {
                const NRTLine::TSearchReply& reply = r.GetValue();
                Y_ENSURE(reply.GetCode() != HTTP_NOT_FOUND);
                Y_ENSURE(reply.GetCode() == 200);
                Y_ENSURE(reply.GetReport().GroupingSize() >= 1);
                for (auto&& grouping : reply.GetReport().GetGrouping()) {
                    for (auto&& group : grouping.GetGroup()) {
                        for (auto&& d : group.GetDocument()) {
                            if (d.GetDocId().EndsWith("SUMMARY")) {
                                TReadSearchProtoHelper helper(d);
                                TString area;
                                if (!helper.GetProperty("routing_area", area)) {
                                    continue;
                                }
                                TVector<TGeoCoord> coords;
                                TGeoCoord::DeserializeVector(area, coords);
                                return coords;
                            }
                        }
                    }
                }
                return {};
            }
        );
    }
    return result;
}

const TRTLineAPI* TUserOfferContext::GetRTLineAPI(const NDrive::IServer& server, const TString& movingPermissions) {
    auto* rtLineApi = server.GetRTLineAPI(
        server.GetSettings().GetValueDef<TString>("router_api_offer_builder-" + movingPermissions, server.GetSettings().GetValueDef<TString>("router_api_offer_builder", "drive_router"))
    );
    return rtLineApi;
}

TMaybe<EOfferCorrectorResult> TUserOfferContext::GetGeoConditionsCheckResult(TStringBuf cacheId) const {
    auto it = GeoConditionsCheckResults.find(cacheId);
    if (it == GeoConditionsCheckResults.end()) {
        return {};
    } else {
        return it->second;
    }
}

void TUserOfferContext::ClearGeoConditionsCheckResults() const {
    GeoConditionsCheckResults.clear();
}

void TUserOfferContext::SetGeoConditionsCheckResult(const TString& cacheId, EOfferCorrectorResult value) const {
    bool inserted = GeoConditionsCheckResults.emplace(cacheId, value).second;
    Y_ASSERT(inserted);
}

double TUserOfferContext::GetTaxiPrice(const TString& taxiClass /*= TTaxiRequest::ECONOM_CLASS*/) const {
    auto g = BuildEventGuard("GetTaxiPrice");
    PrefetchTaxiPrices();
    auto price = TaxiPrices[taxiClass];
    if (!price) {
        return 0;
    }
    if (price->IsSucceeded()) {
        return price->GetReport().GetPrice();
    } else {
        NDrive::TEventLog::Log("GetTaxiPriceError", NJson::TMapBuilder
            ("user_id", GetUserId())
            ("code", price->GetCode())
            ("taxi_class", taxiClass)
            ("report", NJson::ToJson(NJson::JsonString(price->GetRawReport())))
        );
        return 0;
    }
}

const TUserOfferContext::TUserDestinationSuggestEx* TUserOfferContext::GetUserDestinationSuggest() const {
    auto g = BuildEventGuard("GetUserDestinationSuggest");
    PrefetchUserDestinationSuggest();
    if (!UserDestinationSuggest.Initialized()) {
        return nullptr;
    }
    if (UserDestinationSuggest.Wait(GetDeadline()) && UserDestinationSuggest.HasValue()) {
        return &UserDestinationSuggest.GetValue();
    } else {
        NDrive::TEventLog::Log("GetUserDestinationSuggestError", NJson::TMapBuilder
            ("user_id", GetUserId())
            ("error", NThreading::GetExceptionInfo(UserDestinationSuggest))
        );
        return nullptr;
    }
}

bool TUserOfferContext::Prefetch() {
    if (!PrefetchUserDestinationSuggest()) {
        return false;
    }
    if (!PrefetchUserFeatures()) {
        return false;
    }
    if (!PrefetchDestinationsInfo()) {
        return false;
    }
    PrefetchTaxiPrices();
    PrefetchTaxiSurge();
    if (GetNeedYandexPaymentMethod() && !PrefetchPaymentMethods()) {
        return false;
    }
    if (!PrefetchUserAccounts()) {
        return false;
    }
    if (!PrefetchAccountId()) {
        return false;
    }
    if (!PrefetchCreditCard()) {
        return false;
    }
    if (!PrefetchDistributingBlockEvents()) {
        return false;
    }
    if (!PrefetchAggressionScoring()) {
        return false;
    }
    return true;
}

template <>
NJson::TJsonValue NJson::ToJson(const TUserOfferContext::TSimpleDestination& object) {
    NJson::TJsonValue result;
    result.InsertValue("coord", object.GetCoordinate().ToString());
    result.InsertValue("coordinate", NJson::ToJson(object.GetCoordinate()));
    result.InsertValue("context", object.GetContextClient());
    result.InsertValue("description", NJson::ToJson(NJson::Nullable(object.GetDescription())));
    result.InsertValue("name", object.GetName());
    result.InsertValue("icon", object.GetIcon());
    result.InsertValue("style", object.GetHintStyle());
    return result;
}

template <>
NJson::TJsonValue NJson::ToJson(const TUserOfferContext::TDestinationDescription& object) {
    return NJson::ToJson<TUserOfferContext::TSimpleDestination>(object);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TUserOfferContext::TDestinationDescription& result) {
    return
        NJson::ParseField(value["coordinate"], result.MutableCoordinate(), true) &&
        NJson::ParseField(value["context"], result.MutableContextClient()) &&
        NJson::ParseField(value["description"], result.MutableDescription()) &&
        NJson::ParseField(value["icon"], result.MutableIcon()) &&
        NJson::ParseField(value["name"], result.MutableName(), true) &&
        NJson::ParseField(value["style"], result.MutableHintStyle()) &&
        NJson::ParseField(value["kind"], result.MutableKind());
}

void TFinishAreaInfo::SetFutureArea(NThreading::TFuture<TVector<TGeoCoord>>&& f) {
    FutureArea = std::move(f);
}

bool TFinishAreaInfo::Fetch() {
    if (FinishArea) {
        return true;
    }
    if (!FutureArea.Initialized() || !FutureArea.Wait(Deadline) || !FutureArea.HasValue()) {
        return false;
    }
    TVector<TGeoCoord> coords = FutureArea.GetValue();
    if (!coords) {
        return false;
    }
    FinishArea = std::move(coords);
    return true;
}

const TUserOfferContext::TAccountDescriptions& TUserOfferContext::GetAccountDescriptions() const {
    if (!Server->GetDriveAPI()->HasBillingManager()) {
        return Default<TAccountDescriptions>();
    }
    if (!AccountDescriptions) {
        auto& accountDescriptions = AccountDescriptions.ConstructInPlace();
        TVector<NDrive::NBilling::IBillingAccount::TPtr> accounts = Server->GetDriveAPI()->GetBillingManager().GetAccountsManager().GetUserAccounts(GetUserId(), TInstant::Zero());
        for (auto account : accounts) {
            if (!account) {
                continue;
            }
            if (!account->IsPersonal()) {
                continue;
            }
            if (account->GetOffersFilter()) {
                accountDescriptions[account->GetUniqueName()].OffersFilter = TTagsFilter::BuildFromString(account->GetOffersFilter());
            } else if (account->GetOffersFilterName()) {
                auto filtersDB = Server->GetDriveAPI()->GetNamedFiltersDB();
                if (filtersDB) {
                    auto filterData = filtersDB->GetCustomObject(account->GetOffersFilterName());
                    if (filterData && filterData->GetFilterType() == "tariff") {
                        accountDescriptions[account->GetUniqueName()].OffersFilter = filterData->GetFilter();
                    }
                }
            }
            if (!account->GetInsuranceTypes().empty()) {
                accountDescriptions[account->GetUniqueName()].InsuranceTypes = account->GetInsuranceTypes();
            }
        }
    }
    return GetAccountDescriptionsRef();
}

const NDrive::TUserEventsApi* TUserOfferContext::GetFeaturesClient() const {
    if (!GetSetting<bool>("use_features_client").GetOrElse(false)) {
        return Server->GetUserEventsApi();
    }
    return Server->GetFeaturesClient();
}

TGeobaseId TUserOfferContext::TDestinationDescription::GetGeobaseId() const {
    return GeobaseId.GetOrElse(0);
}

const TFinishAreaInfo* TUserOfferContext::TDestinationDescription::GetFinishArea() const {
    if (!FinishArea) {
        return nullptr;
    }
    if (!FinishArea->Fetch()) {
        return nullptr;
    }
    return FinishArea.Get();
}

const NDrive::TGeoFeatures* TUserOfferContext::TDestinationDescription::GetGeoFeatures() const {
    if (!GeoFeatures.Initialized()) {
        return nullptr;
    }
    if (GeoFeatures.Wait(Deadline) && GeoFeatures.HasValue()) {
        return GeoFeatures.GetValue().Get();
    } else {
        NDrive::TEventLog::Log("GetDestinationGeoFeaturesError", NJson::TMapBuilder
            ("destination", NJson::ToJson<TSimpleDestination>(*this))
            ("exception", NThreading::GetExceptionInfo(GeoFeatures))
        );
        return nullptr;
    }
}

const NDrive::TGeoFeatures* TUserOfferContext::TDestinationDescription::GetGeobaseFeatures() const {
    if (!GeobaseFeatures.Initialized()) {
        return nullptr;
    }
    if (GeobaseFeatures.Wait(Deadline) && GeobaseFeatures.HasValue()) {
        return GeobaseFeatures.GetValue().Get();
    } else {
        NDrive::TEventLog::Log("GetDestinationGeobaseFeaturesError", NJson::TMapBuilder
            ("destination", NJson::ToJson<TSimpleDestination>(*this))
            ("exception", NThreading::GetExceptionInfo(GeobaseFeatures))
        );
        return nullptr;
    }
}

const NDrive::TUserGeoFeatures* TUserOfferContext::TDestinationDescription::GetUserGeoFeatures() const {
    if (!UserGeoFeatures.Initialized()) {
        return nullptr;
    }
    if (UserGeoFeatures.Wait(Deadline) && UserGeoFeatures.HasValue()) {
        return UserGeoFeatures.GetValue().Get();
    } else {
        NDrive::TEventLog::Log("GetDestinationUserGeoFeaturesError", NJson::TMapBuilder
            ("destination", NJson::ToJson<TSimpleDestination>(*this))
            ("exception", NThreading::GetExceptionInfo(UserGeoFeatures))
        );
        return nullptr;
    }
}

const NDrive::TUserGeoFeatures* TUserOfferContext::TDestinationDescription::GetUserGeobaseFeatures() const {
    if (!UserGeobaseFeatures.Initialized()) {
        return nullptr;
    }
    if (UserGeobaseFeatures.Wait(Deadline) && UserGeobaseFeatures.HasValue()) {
        return UserGeobaseFeatures.GetValue().Get();
    } else {
        NDrive::TEventLog::Log("GetDestinationUserGeobaseFeaturesError", NJson::TMapBuilder
            ("destination", NJson::ToJson<TSimpleDestination>(*this))
            ("exception", NThreading::GetExceptionInfo(UserGeobaseFeatures))
        );
        return nullptr;
    }
}

TMaybe<TVector<TDistributingBlockEvent::TPtr>> TUserOfferContext::GetDistributingBlockEvents() const {
    if (!DistributingBlockEvents.Initialized()) {
        return {};
    }
    TUnistatSignalsCache::SignalAdd("distributing_block_events_access", "count", 1);
    if (DistributingBlockEvents.Wait(GetDeadline()) && DistributingBlockEvents.HasValue()) {
        return DistributingBlockEvents.GetValue();
    } else {
        TUnistatSignalsCache::SignalAdd("distributing_block_events_access", "errors", 1);
        NDrive::TEventLog::Log("GetDistributingBlockEventsError", NJson::TMapBuilder
            ("exception", NThreading::GetExceptionInfo(DistributingBlockEvents))
        );
        return {};
    }
}

TMaybe<NDrive::TTaxiSurgeCalculator::TResult> TUserOfferContext::GetTaxiSurge() const {
    auto g = BuildEventGuard("GetTaxiSurge");
    PrefetchTaxiSurge();
    if (!TaxiSurge.Initialized()) {
        return {};
    }
    if (TaxiSurge.Wait(GetDeadline()) && TaxiSurge.HasValue()) {
        return TaxiSurge.GetValue();
    } else {
        NDrive::TEventLog::Log(
            "GetTaxiSurgeError",
            NJson::TMapBuilder("exception", NThreading::GetExceptionInfo(TaxiSurge))
        );
        return {};
    }
}

TMaybe<TScoringUserTag> TUserOfferContext::GetAggressionScoring() const {
    return AggressionScoring;
}

TMaybe<TInstant> TUserOfferContext::GetLastPricedRideTime() const {
    return LastPricedRideTime;
}

bool TUserOfferContext::TSimpleDestination::BuildFromSettings(const ISettings& settings, const TString& destinationKey) {
    {
        TMaybe<TGeoCoord> coordinate = settings.GetValue<TGeoCoord>("destination." + destinationKey + ".coord");
        if (!coordinate) {
            return false;
        }
        Coordinate = *coordinate;
    }
    {
        TMaybe<TString> name = settings.GetValue<TString>("destination." + destinationKey + ".name");
        if (!name) {
            return false;
        }
        Name = *name;
    }
    {
        TMaybe<TString> icon = settings.GetValue<TString>("destination." + destinationKey + ".icon");
        if (!!icon) {
            Icon = *icon;
        }
    }
    {
        TMaybe<TString> context = settings.GetValue<TString>("destination." + destinationKey + ".context");
        if (!!context) {
            ContextClient = *context;
        }
    }
    {
        TMaybe<TString> hintStyle = settings.GetValue<TString>("destination." + destinationKey + ".hint_style");
        if (!!hintStyle) {
            HintStyle = *hintStyle;
        }
    }
    return true;
}

void TUserOfferContext::TDestinationDescription::Prefetch(const TUserOfferContext& context) const {
    Deadline = context.GetDeadline();
    if (!GeobaseId) {
        GeobaseId = context.FetchGeobaseId(GetCoordinate());
    }
    if (!FinishArea) {
        FinishArea = context.FetchFinishPointArea(GetCoordinate());
    }
    if (!GeoFeatures.Initialized()) {
        GeoFeatures = context.FetchGeoFeatures(GetCoordinate());
    }
    if (!GeobaseFeatures.Initialized() && GeobaseId) {
        GeobaseFeatures = context.FetchGeoFeatures(*GeobaseId);
    }
    if (!UserGeoFeatures.Initialized()) {
        UserGeoFeatures = context.FetchUserGeoFeatures(GetCoordinate());
    }
    if (!UserGeobaseFeatures.Initialized() && GeobaseId) {
        UserGeobaseFeatures = context.FetchUserGeoFeatures(*GeobaseId);
    }
}

NJson::TJsonValue TUserOfferContext::TSimpleDestination::SerializeToJson() const {
    return NJson::ToJson<TSimpleDestination>(*this);
}

template <>
NJson::TJsonValue NJson::ToJson(const TUserOfferContext::TUserDestinationSuggestEx::TElement& object) {
    NJson::TJsonValue result = NJson::ToJson<NDrive::TUserDestinationSuggest::TElement>(object);
    result["geobase_id"] = NJson::ToJson(object.GeobaseId);
    result["geobase_features"] = NJson::ToJson(object.GeobaseFeatures);
    result["geo_features"] = NJson::ToJson(object.GeoFeatures);
    result["user_geo_features"] = NJson::ToJson(object.UserGeobaseFeatures);
    return result;
}
