#pragma once

#include "abstract.h"
#include "discount.h"

#include <drive/backend/abstract/user_position_context.h>
#include <drive/backend/billing/interfaces/account.h>
#include <drive/backend/compiled_riding/manager.h>
#include <drive/backend/data/scoring/scoring.h>
#include <drive/backend/distributing_block_storage/common.h>
#include <drive/backend/logging/evlog.h>
#include <drive/backend/processor/processor.h>
#include <drive/backend/roles/permissions.h>

#include <drive/library/cpp/geocoder/api/client.h>
#include <drive/library/cpp/geofeatures/features.h>
#include <drive/library/cpp/taxi/request.h>
#include <drive/library/cpp/taxi/suggest/client.h>
#include <drive/library/cpp/taxi/surge_calculator/client.h>
#include <drive/library/cpp/user_events_api/user_features.h>

#include <rtline/api/graph/router/router.h>
#include <rtline/util/types/accessor.h>

enum class EOfferCorrectorResult;

using TGeobaseId = ui64;
using TOptionalGeobaseId = TMaybe<TGeobaseId>;

class TFinishAreaInfo {
private:
    R_OPTIONAL(TVector<TGeoCoord>, FinishArea);
    R_OPTIONAL(TVector<TGeoCoord>, FinishAreaPublic);
    R_FIELD(TInstant, Deadline, TInstant::Zero());
    R_FIELD(TDuration, WalkingDuration, TDuration::Minutes(7));
    NThreading::TFuture<TVector<TGeoCoord>> FutureArea;

public:
    bool Initialized() const {
        return FutureArea.Initialized();
    }

    void SetFutureArea(NThreading::TFuture<TVector<TGeoCoord>>&& f);

    bool Fetch();
};

class TUserOfferContext: public TUserPositionContext {
public:
    class TSimpleDestination {
    private:
        R_FIELD(TGeoCoord, Coordinate);
        R_FIELD(TString, ContextClient);
        R_FIELD(TString, Description);
        R_FIELD(TString, Name);
        R_FIELD(TString, Icon);
        R_FIELD(TString, HintStyle);
        R_FIELD(TString, Kind);

    public:
        NJson::TJsonValue SerializeToJson() const;
        bool BuildFromSettings(const ISettings& settings, const TString& destinationKey);
    };

    class TDestinationDescription: public TSimpleDestination {
    public:
        TGeobaseId GetGeobaseId() const;
        const TFinishAreaInfo* GetFinishArea() const;
        const NDrive::TGeoFeatures* GetGeoFeatures() const;
        const NDrive::TGeoFeatures* GetGeobaseFeatures() const;
        const NDrive::TUserGeoFeatures* GetUserGeoFeatures() const;
        const NDrive::TUserGeoFeatures* GetUserGeobaseFeatures() const;

        void Prefetch(const TUserOfferContext& context) const;

    private:
        mutable TMaybe<TFinishAreaInfo> FinishArea;
        mutable NThreading::TFuture<NDrive::TOptionalGeoFeatures> GeoFeatures;
        mutable NThreading::TFuture<NDrive::TOptionalGeoFeatures> GeobaseFeatures;
        mutable NThreading::TFuture<NDrive::TOptionalUserGeoFeatures> UserGeoFeatures;
        mutable NThreading::TFuture<NDrive::TOptionalUserGeoFeatures> UserGeobaseFeatures;

        mutable TInstant Deadline;
        mutable TOptionalGeobaseId GeobaseId;
    };

    class TUserDestinationSuggestEx {
    public:
        struct TElement: public NDrive::TUserDestinationSuggest::TElement {
        public:
            using TBase = NDrive::TUserDestinationSuggest::TElement;

        public:
            TElement(const TBase& base)
                : TBase(base)
            {
            }

        public:
            NThreading::TFuture<NDrive::TOptionalGeoFeatures> GeoFeatures;
            NThreading::TFuture<NDrive::TOptionalGeoFeatures> GeobaseFeatures;
            TOptionalGeobaseId GeobaseId;
            NThreading::TFuture<NDrive::TOptionalUserGeoFeatures> UserGeobaseFeatures;
            // GeocoderResponse contains response from geocoder for destination.
            NThreading::TFuture<NDrive::TGeocoder::TResponse> GeocoderResponse;
        };
        using TElements = TVector<TElement>;

    public:
        TElements Elements;
    };

    struct TAccountDescription {
        TSet<TString> InsuranceTypes;
        TTagsFilter OffersFilter;
    };

    using TAccountDescriptions = TMap<TString, TAccountDescription>;
    using TDestinationDescriptions = TVector<TDestinationDescription>;
    using TGeoConditionsCheckResults = TMap<TString, EOfferCorrectorResult>;
    using TGeoFeaturesFuture = NThreading::TFuture<NDrive::TOptionalGeoFeatures>;
    using TUserDestinationSuggestFuture = NThreading::TFuture<TUserDestinationSuggestEx>;
    using TUserFeaturesFuture = NThreading::TFuture<NDrive::TUserFeatures>;
    using TUserGeoFeaturesFuture = NThreading::TFuture<NDrive::TOptionalUserGeoFeatures>;
    using TDistributingBlockEventsFuture = NThreading::TFuture<TVector<TDistributingBlockEvent::TPtr>>;

private:
    R_FIELD(TVector<TDiscount>, AdditionalDurationDiscounts, {}, mutable);
    R_OPTIONAL(ui32, PricedRidesCount, {}, mutable);

    R_OPTIONAL(TAccountDescriptions, AccountDescriptions, {}, mutable);
    R_OPTIONAL(TGeoCoord, ExternalUserPosition);
    R_OPTIONAL(TString, InsuranceType);
    R_FIELD(TVector<TString>, ComplementaryInsuranceTypes);
    R_FIELD(TString, ExternalUserId);
    R_FIELD(TString, Origin);
    R_FIELD(bool, NeedHistoryFreeTimeFees, true, mutable);
    R_FIELD(TString, CommonSettingsPrefix);
    R_FIELD(bool, EnableTaxiSuggest, false);
    R_FIELD(ui32, DestinationsCountLimit, 3);
    R_FIELD(ui32, OfferCountLimit, 1);
    R_FIELD(TInstant, Deadline, Now() + TDuration::Seconds(1));
    R_FIELD(ELocalization, Locale, DefaultLocale);
    R_FIELD(bool, NeedFinishArea, true);
    R_FIELD(bool, NeedGeoFeatures, true);
    R_FIELD(bool, NeedUserDestinationSuggest, true);
    R_FIELD(bool, NeedUserFeatures, true);
    R_READONLY(TString, UID);
    R_OPTIONAL(TDestinationDescription, UserDestination, {}, mutable);
    R_FIELD(bool, NeedTaxiPrice, true);
    R_FIELD(bool, FilterAccounts, true);
    R_FIELD(bool, NeedDefaultAccount, true);
    R_READONLY(TMaybe<TPaymentMethod>, YandexPaymentMethod, {}, mutable);
    R_FIELD(bool, NeedYandexPaymentMethod, false);
    R_FIELD(bool, NeedDistributingBlockShowsRestriction, false);
    R_FIELD(bool, AllowDistributingBlock, true);
    R_FIELD(bool, NeedCarSuggest, false);
    R_OPTIONAL(i64, YandexAccountExternalBalance, {}, mutable);
    R_FIELD(bool, NeedPaymentMethods, true);

private:
    const NDrive::IServer* Server = nullptr;
    TUserPermissions::TConstPtr Permissions;

    mutable TMaybe<TString> CreditCard;
    mutable TMaybe<TVector<TPaymentMethod>> PaymentMethods;
    mutable TMaybe<TVector<NDrive::NBilling::IBillingAccount::TPtr>> UserAccounts;
    TMaybe<TDestinationDescriptions> ParsedDestinations;
    mutable TMaybe<TDestinationDescriptions> HintedDestinations;
    mutable TGeoConditionsCheckResults GeoConditionsCheckResults;
    mutable TUserDestinationSuggestFuture UserDestinationSuggest;
    mutable TUserFeaturesFuture UserFeatures;
    mutable NThreading::TFuture<NDrive::TTaxiSuggest> TaxiSuggest;
    mutable NThreading::TFuture<NDrive::TTaxiSurgeCalculator::TResult> TaxiSurge;
    mutable TMap<TString, TSimpleTaxiReply::TPtr> TaxiPrices;
    mutable TSet<TString> AccountNames;
    mutable TOptionalObjectEvents<TCompiledRiding> CompiledSessions;
    // HomeDestination stores home destination from ParsedDestinations.
    mutable TMaybe<TDestinationDescription> HomeDestination;
    // WorkDestination stores work destination from ParsedDestinations.
    mutable TMaybe<TDestinationDescription> WorkDestination;
    mutable TDistributingBlockEventsFuture DistributingBlockEvents;
    mutable TMaybe<TScoringUserTag> AggressionScoring;
    // LastPricedRideTime contains time of last priced ride.
    //
    // If the user has no paid rides, then it will be TInstant::Zero().
    mutable TMaybe<TInstant> LastPricedRideTime;
    bool EnableDestinationSuggest = false;

private:
    bool ParseAccountId(const IRequestProcessor* processor, const NJson::TJsonValue& requestData);
    bool ParseCreditCard(const IRequestProcessor* processor, const NJson::TJsonValue& requestData);
    bool ParseYandexAccountBalance(const IRequestProcessor* processor, const NJson::TJsonValue& requestData);
    bool ParseDestinations(const IRequestProcessor* processor, const NJson::TJsonValue& requestData);
    bool ParseInsuranceInfo(const IRequestProcessor* processor, const NJson::TJsonValue& requestData);

    bool PrefetchHistoryFees() const;
    bool PrefetchTaxiSuggest() const;
    bool PrefetchUserDestinationSuggest() const;
    bool PrefetchUserFeatures() const;
    bool PrefetchDestinationsInfo() const;
    void PrefetchTaxiSurge() const;
    void PrefetchTaxiPrices() const;
    bool PrefetchAccountId() const;
    bool PrefetchCreditCard() const;
    bool PrefetchUserAccounts() const;
    bool PrefetchPaymentMethods() const;
    bool PrefetchDistributingBlockEvents() const;
    bool PrefetchAggressionScoring() const;

    // SetupHomeWorkDestinations tries to find home/work destinations in ParsedDestinations.
    void SetupHomeWorkDestinations() const;
    // ExtractTaxiHintedDestinations tries to load destinations from Taxi.
    //
    // On success, HintedDestinations will be cleared.
    void ExtractTaxiHintedDestinations() const;
    // ExtractPredictorHintedDestinations tries to load destinations using ML predictions.
    //
    // On success, HintedDestinations will be cleared.
    void ExtractPredictorHintedDestinations() const;
    // FixHintedDestinations finds nearest ParsedDestinations and fixes HintedDestinations.
    void FixHintedDestinations() const;

    TOptionalGeobaseId FetchGeobaseId(const TGeoCoord& coordinate) const;
    TFinishAreaInfo FetchFinishPointArea(const TGeoCoord& coordinate) const;
    TSimpleTaxiReply::TPtr FetchTaxiPrice(const TString& taxiClass) const;
    NThreading::TFuture<NDrive::TTaxiSurgeCalculator::TResult> FetchTaxiSurge() const;
    template <class T>
    TGeoFeaturesFuture FetchGeoFeatures(const T& id) const;
    TUserDestinationSuggestFuture FetchUserDestinationSuggest() const;
    TUserFeaturesFuture FetchUserFeatures() const;
    template <class T>
    TUserGeoFeaturesFuture FetchUserGeoFeatures(const T& id) const;
    const NDrive::TUserEventsApi* GetFeaturesClient() const;

private:
    static const TRTLineAPI* GetRTLineAPI(const NDrive::IServer& server, const TString& movingPermissions);

public:
    TUserOfferContext(const NDrive::IServer* server, TUserPermissions::TConstPtr permissions, IReplyContext::TPtr context = nullptr);

    auto BuildEventGuard(const TString& id) const {
        return NDrive::BuildEventGuard(id);
    }

    template <class T>
    TMaybe<T> GetSetting(const TString& key) const {
        if (!Server) {
            return {};
        }
        return TUserPermissions::GetSetting<T>(key, Server->GetSettings(), Permissions);
    }

    const NDrive::IServer* GetServer() const {
        return Server;
    }

    void SetAccountName(const TString& value) {
        AccountNames.insert(value);
    }

    TMaybe<EOfferCorrectorResult> GetGeoConditionsCheckResult(TStringBuf cacheId) const;
    void ClearGeoConditionsCheckResults() const;
    void SetGeoConditionsCheckResult(const TString& cacheId, EOfferCorrectorResult value) const;

    double GetTaxiPrice(const TString& taxiClass = TTaxiRequest::ECONOM_CLASS) const;

    const TUserDestinationSuggestEx* GetUserDestinationSuggest() const;
    const NDrive::TUserFeatures* GetUserFeatures() const;
    TUserPermissions::TConstPtr GetUserPermissions() const;

    const TAccountDescriptions& GetAccountDescriptions() const;
    TSet<TString> GetAvailableAccounts(const TSet<TString>& tags, const TSet<TString>& additionalAccounts) const;
    TMaybe<TSet<TString>> GetAvailableInsuranceTypes() const;
    bool CheckRequestAccountId(const TSet<TString>& currentObjectTags, const TSet<TString>& additionalAccounts) const;
    const TSet<TString>& GetRequestAccountIds() const;
    const TString& GetRequestCreditCard() const;
    TMaybe<TVector<TPaymentMethod>> GetPaymentMethods() const;
    const TVector<NDrive::NBilling::IBillingAccount::TPtr>& GetUserAccounts() const;
    TVector<NDrive::NBilling::IBillingAccount::TPtr> GetRequestUserAccounts() const;
    // GetSelectedCharge returns selected billing account.
    TString GetSelectedCharge() const;
    const TDestinationDescriptions& GetHintedDestinations() const;
    const TUserDestinationSuggestFuture& GetUserDestinationSuggestFuture() const;
    TMaybe<TVector<TDistributingBlockEvent::TPtr>> GetDistributingBlockEvents() const;
    TMaybe<NDrive::TTaxiSurgeCalculator::TResult> GetTaxiSurge() const;
    TMaybe<TScoringUserTag> GetAggressionScoring() const;
    TMaybe<TInstant> GetLastPricedRideTime() const;

    TString GetUserId() const;
    TString GetPassportUid() const;

    bool FetchFeesInfo() const;
    bool FillData(const IRequestProcessor* processor, const NJson::TJsonValue& requestData);
    bool Prefetch();

public:
    static NThreading::TFuture<TVector<TGeoCoord>> GetSearchAreaFromRouter(const NDrive::IServer& server, TUserPermissions::TConstPtr permissions, const TDuration& reqTimeout, const TDuration& walkingDuration, const TGeoCoord& coord);
};

void DropNearDestinationSuggests(
    TVector<TUserOfferContext::TUserDestinationSuggestEx::TElement>& elements,
    TGeoCoord point, double radius
);

void DropNearDestinationSuggests(
    TVector<TUserOfferContext::TUserDestinationSuggestEx::TElement>& elements,
    double radius
);
