#include "manager.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/saas/api.h>

#include <drive/telematics/api/sensor/interface.h>

#include <drive/library/cpp/parking/inhouse/client.h>

#include <rtline/api/search_client/query.h>
#include <rtline/library/graph/traffic_graph/geo_objects/parking/parking.h>
#include <rtline/protos/proto_helper.h>
#include <rtline/util/types/messages_collector.h>

#include <util/string/vector.h>

using namespace NDrive;

void TParkingZonesManagerConfig::Init(const TYandexConfig::Section* section) {
    RTLineAPIName = section->GetDirectives().Value("RTLineAPIName", RTLineAPIName);
    AssertCorrectConfig(!!RTLineAPIName,
                        "Missing 'RTLineAPIName' field in ParkingZonesManager config");
    ParkingConfig.Init(section);
}

void TParkingZonesManagerConfig::ToString(IOutputStream& os) const {
    os << "RTLineAPIName: " << RTLineAPIName << Endl;
    ParkingConfig.ToString(os);
}

TParkingZonesManager::TParkingZonesManager(const TParkingZonesManagerConfig& config,
                                           const NDrive::IServer& server)
    : Server(server)
    , Config(config)
    , SearchClient(server.GetRTLineAPI(config.GetRTLineAPIName())->GetSearchClient())
{
}

TExpected<TParkingInfo, TString> TParkingZonesManager::IsParkingAllowed(const TString& userId, const IReplyContext& context, TMaybe<double> precisionValue) const {
    const auto& cgi = context.GetCgiParameters();
    const auto& sessionId = cgi.Get("session_id");
    TAtomicSharedPtr<const ISession> userSession;
    if (!Server.GetDriveAPI()->GetUserSession(userId, userSession, sessionId, TInstant::Zero())) {
        return MakeUnexpected<TString>("Server problems");
    }

    if (!userSession || userSession->GetClosed()) {
        return MakeUnexpected<TString>("Inactive session");
    }

    const auto carId = userSession->GetObjectId();
    auto snapshot = Server.GetSnapshotsManager().GetSnapshot(carId);
    const auto* sensorAPI = Server.GetSensorApi();
    if (sensorAPI) {
        auto locationFuture = sensorAPI->GetLocations({ Server.GetDriveAPI()->GetIMEI(carId) }, NDrive::BeaconsLocationName);
        if (locationFuture.Initialized()) {
            try {
                auto locations = locationFuture.GetValue(context.GetRequestDeadline() - Now());
                auto it = locations.find(NDrive::BeaconsLocationName);
                if (it != locations.end()) {
                    snapshot.UpdateBeaconsLocation(it->second, TInstant::Max());
                }
            } catch (const NThreading::TFutureException& ex) {
                NDrive::TEventLog::Log("ParkingZonesManagerBeaconsLocationFetchError", NJson::TMapBuilder("message", ex.what()));
            }
        }
    }
    auto location = snapshot.GetBeaconsOrRawLocation();
    NDrive::TEventLog::Log("ParkingZonesManagerLocationsSnapshot", NJson::TMapBuilder
                                                                        ("simple", NJson::ToJson(snapshot.GetLocations()))
                                                                        ("beacons", NJson::ToJson(snapshot.GetBeaconsLocation()))
                                                                        ("beacons_or_raw", NJson::ToJson(location)));
    if (!location) {
        return MakeUnexpected<TString>("No car location");
    }
    return IsParkingAllowed(location->GetCoord(), precisionValue);
}

TExpected<TParkingInfo, TString> TParkingZonesManager::IsParkingAllowed(const TGeoCoord& position, TMaybe<double> precisionValue) const {
    TParkingInfo res;

    TSet<TString> areaTags = Server.GetDriveAPI()->GetTagsInPoint(position);
    res.SignalIssues = areaTags.contains(NDrive::DefaultSignalIssuesLocationTag);

    if (areaTags.contains(NDrive::DefaultParkingZonesAllowParkingLocationTag)) {
        res.Ability = EParkingAbility::Allow;
        return res;
    }

    if (areaTags.contains(NDrive::DefaultParkingZonesDenyParkingLocationTag)) {
        res.Ability = EParkingAbility::Deny;
        return res;
    }

    const double precision = precisionValue ? *precisionValue : Config.GetMaxPrecision();
    NRTLine::TQuery query = TParkingInfo::CreateQuery(position, precision);
    NRTLine::TSearchReply reply = SearchClient.SendQuery(query);
    if (!reply.IsSucceeded()) {
        return MakeUnexpected<TString>(TStringBuilder() << "Failed search client request: " << reply.GetCode() << ' ' << reply.GetReqId());
    }

    TParkingInfo parkingInfo;
    TMessagesCollector errors;
    if (!res.Parse(reply.GetReport(), Config.GetParkingConfig(), precision, errors)) {
        return MakeUnexpected<TString>(errors.GetStringReport());
    }
    return res;
}


void TParkingAggregatorsManagerConfig::Init(const TYandexConfig::Section* section) {
    KznClientToken = section->GetDirectives().Value("KznClientToken", KznClientToken);
    SpbClientToken = section->GetDirectives().Value("SpbClientToken", SpbClientToken);
    MskClientToken = section->GetDirectives().Value("MskClientToken", MskClientToken);
    {
        const TYandexConfig::TSectionsMap children = section->GetAllChildren();
        auto it = children.find("MskParkingConfig");
        if (it != children.end()) {
            AmppConfig = MakeHolder<TMskParkingPaymentConfig>();
            AmppConfig->Init(it->second);
        }
    }
}

void TParkingAggregatorsManagerConfig::ToString(IOutputStream& os) const {
    if (KznClientToken) {
        os << "HasKznClientToken: true" << Endl;
    }
    if (SpbClientToken) {
        os << "HasSpbClientToken: true" << Endl;
    }
    if (MskClientToken) {
        os << "HasMskClientToken: true" << Endl;
    }
    if (AmppConfig) {
        os << "<MskParkingConfig>" << Endl;
        AmppConfig->ToString(os);
        os << "</MskParkingConfig>" << Endl;
    }
}

const TMskParkingPaymentConfig* TParkingAggregatorsManagerConfig::GetAmppConfig() const {
    return AmppConfig.Get();
}

TParkingAggregatorsManager::TParkingAggregatorsManager(const TParkingAggregatorsManagerConfig& config)
    : Config(config)
{
    if (Config.GetKznClientToken()) {
        KznClient = MakeHolder<TFitDevParkingPaymentClient>(Config.GetKznClientToken());
    }
    if (Config.GetSpbClientToken()) {
        SpbClient = MakeHolder<TSpbParkingPaymentClient>(Config.GetSpbClientToken());
    }
    if (Config.GetMskClientToken()) {
        MskClient = MakeHolder<TMosParkingPaymentClient>(Config.GetMskClientToken());
    }
    if (auto ptr = Config.GetAmppConfig()) {
        AmppClient = MakeHolder<TMskParkingPaymentClient>(*ptr);
    }
}

void TParkingAggregatorsManager::StopParking(const TString& carId, const TString& userId, const TString& carNumber, TSlot& slot) const {
    if (!slot.IsActive(Now()) || !slot.SessionAggregatorId) {
        return;
    }
    NDrive::TEventLog::Log("StoppingParking", NJson::TMapBuilder("id", carId)("slot", slot.ToJson()));
    switch (slot.SessionAggregatorId) {
        case NDrive::TParkingListClient::MskAggregatorId:
            slot.Refund = StopClient3Parking(carId, userId, slot.Token, slot.ParkingSessionId, slot.ParkingVehicleId);
            break;
        case NDrive::TParkingListClient::AmppAggregatorId: {
            auto session = StopAmppParking(carNumber);
            slot.Session = session.ToJson();
            slot.SessionFinish = session.EndTime;
            slot.Refund = NJson::TJsonValue(NJson::JSON_MAP);
            break;
        }
        case NDrive::TParkingListClient::KznAggregatorId:
        case NDrive::TParkingListClient::SpbAggregatorId:
        case NDrive::TParkingListClient::FitDevMskParkingId:
            slot.Refund = StopFitParking(slot.SessionAggregatorId, slot.ParkingSessionId);
            break;
        default:
            throw yexception() << "unknown aggregator";
    }
    NDrive::TEventLog::Log("StoppedParking", NJson::TMapBuilder("id", carId)("slot", slot.ToJson()));
}

NJson::TJsonValue TParkingAggregatorsManager::StopClient3Parking(const TString& carId, const TString& userId, const TString& token, const TString& sessionId, const TString& vehicleId) const {
    TParkingPaymentClient3 client(token);
    auto result = client.StopParkingRobust(sessionId).ToJson();
    if (vehicleId) {
        auto vehicles = client.GetVehicles();
        for (auto&& vehicle : vehicles) {
            if (vehicle.Reference == vehicleId) {
                NDrive::TEventLog::Log("ParkingDeletingVehicle", NJson::TMapBuilder
                    ("id", carId)
                    ("token", token)
                    ("user_id", userId)
                    ("vehicle", vehicle.ToJson())
                );
                client.DeleteVehicleRobust(vehicle);
                NDrive::TEventLog::Log("ParkingDeletedVehicle", NJson::TMapBuilder
                    ("id", carId)
                    ("token", token)
                    ("user_id", userId)
                    ("vehicle", vehicle.ToJson())
                );
            }
        }
    }
    return result;
}

NJson::TJsonValue TParkingAggregatorsManager::StopFitParking(const i64 aggregatorId, const TString& sessionId) const {
    switch (aggregatorId) {
        case TParkingListClient::KznAggregatorId:
            return Yensured(KznClient)->StopParkingRobust(sessionId).ToJson();
        case TParkingListClient::SpbAggregatorId:
            return Yensured(SpbClient)->StopParkingRobust(sessionId).ToJson();
        case TParkingListClient::FitDevMskParkingId:
            return Yensured(MskClient)->StopParkingRobust(sessionId).ToJson();
        default:
            return NJson::JSON_NULL;
    };
}

TMskParkingPaymentClient::TSession TParkingAggregatorsManager::StopAmppParking(const TString& carNumber) const {
    auto futureSession = Yensured(AmppClient)->StopParking(carNumber);
    if (!futureSession.Wait(AmppClient->GetConfig().GetRequestTimeout())) {
        throw yexception() << "stop parking failed by timeout";
    }
    return futureSession.ExtractValue();
}

bool TParkingAggregatorsManager::TSlot::IsActive(TInstant now) const {
    return
        Session.IsDefined() && !Refund.IsDefined() && (SessionFinish > now);
}

NJson::TJsonValue TParkingAggregatorsManager::TSlot::ToJson() const {
    NJson::TJsonValue result;
    result["token"] = Token;
    result["parking_vehicle_id"] = ParkingVehicleId;
    result["parking_session_id"] = ParkingSessionId;
    result["session"] = Session;
    result["refund"] = Refund;
    if (Cost > 0) {
        result["cost"] = Cost;
    }
    if (SessionAggregatorId) {
        result["session_aggregator_id"] = SessionAggregatorId;
    }
    if (SessionFinish) {
        result["session_finish"] = SessionFinish.Seconds();
    }
    return result;
}
