#include "parking_info.h"

#include <library/cpp/string_utils/quote/quote.h>

#include <rtline/library/graph/geometry_graph/common/types.h>
#include <rtline/util/types/messages_collector.h>

void TParkingConfig::Init(const TYandexConfig::Section* section) {
    DistanceAcceptParkingAllow = section->GetDirectives().Value("DistanceAcceptParkingAllow", DistanceAcceptParkingAllow);
    DistanceAcceptParkingDeny = section->GetDirectives().Value("DistanceAcceptParkingDeny", DistanceAcceptParkingDeny);
    BorderRelationStopsAllow = section->GetDirectives().Value("BorderRelationStopsAllow", BorderRelationStopsAllow);
    BorderRelationStopsDeny = section->GetDirectives().Value("BorderRelationStopsDeny", BorderRelationStopsDeny);
    FinalAllowAcceptanceLength = section->GetDirectives().Value("FinalAllowAcceptanceLength", FinalAllowAcceptanceLength);
    FinalDenyAcceptanceLength = section->GetDirectives().Value("FinalDenyAcceptanceLength", FinalDenyAcceptanceLength);
}

void TParkingConfig::ToString(IOutputStream& os) const {
    os << "DistanceAcceptParkingAllow: " << DistanceAcceptParkingAllow << Endl;
    os << "DistanceAcceptParkingDeny: " << DistanceAcceptParkingDeny << Endl;
    os << "BorderRelationStopsAllow: " << BorderRelationStopsAllow << Endl;
    os << "BorderRelationStopsDeny: " << BorderRelationStopsDeny << Endl;
    os << "FinalAllowAcceptanceLength: " << FinalAllowAcceptanceLength << Endl;
    os << "FinalDenyAcceptanceLength: " << FinalDenyAcceptanceLength << Endl;
}

bool TParkingConfig::CheckParkingAbility(const TParking::EParkingType type, const double distance, EParkingAbility& result) const {
    switch (type) {
        case TParking::EParkingType::ptFreeParking:
        case TParking::EParkingType::ptPaidParking:
        case TParking::EParkingType::ptUndefined:
        case TParking::EParkingType::ptRestrictedParking:
            if (distance < DistanceAcceptParkingAllow) {
                result = EParkingAbility::Allow;
                return true;
            }
            break;
        default:
            if (distance < DistanceAcceptParkingDeny) {
                result = EParkingAbility::Deny;
                return true;
            }
            break;
    }
    return false;
}

bool TParkingConfig::CheckParkingAbility(const ui32 constraintsStop, const ui32 passInStop, const ui32 passOutStop, EParkingAbility& result) const {
    if (constraintsStop == 0) {
        if (passOutStop || passInStop) {
            result = EParkingAbility::Deny;
            return true;
        }
    } else {
        const double kff = (double)passOutStop / (double)(constraintsStop + passInStop);
        if (kff < BorderRelationStopsAllow) {
            result = EParkingAbility::Allow;
            return true;
        }
        if (kff < BorderRelationStopsDeny) {
            result = EParkingAbility::Deny;
            return true;
        }
    }
    return false;
}

NRTLine::TQuery TParkingInfo::CreateQuery(const TGeoCoord& c, double precision) {
    NRTLine::TQuery query;
    TString qText = "type:parking;coord:" + c.ToString() + ";d:" + ::ToString(precision);
    query.SetText(qText);
    query.AddExtraParams("&component=Graph");
    return query;
}

bool TParkingInfo::Parse(const NMetaProtocol::TReport& report, const TParkingConfig& config, const double precision, TMessagesCollector& errors) {
    if (report.GroupingSize() != 1) {
        errors.AddMessage("parking", "Incorrect reply (grouping size != 1)");
        return false;
    }

    const ::NMetaProtocol::TGrouping& grouping = report.GetGrouping(0);

    if (grouping.GroupSize() == 0) {
        errors.AddMessage("parking", "Request empty");
        return false;
    }

    EParkingAbility allowExitsLockedCriteria = EParkingAbility::Undefined;

    double distDenyCriteria = Max<ui32>();
    double distAllowCriteria = Max<ui32>();
    for (auto&& group : grouping.GetGroup()) {
        if ("special_tags" == group.GetCategoryName()) {
            CHECK_WITH_LOG(Tags.empty());
            if (group.DocumentSize() != 1) {
                errors.AddMessage("parking", "Incorrect tags reply structure: 'special_tags' group. Documents count is incorrect");
                return false;
            }
            const NMetaProtocol::TDocument& doc = group.GetDocument(0);
            TReadSearchProtoHelper reader(doc);
            TString tags;
            if (!reader.GetProperty("tags", tags)) {
                return false;
            }

            TVector<TString> tagsVector;
            if (!TryParseStringToVector(tags, tagsVector, ',', false)) {
                return false;
            }
            Tags.insert(tagsVector.begin(), tagsVector.end());
        } else if (group.GetCategoryName() == "parking_area") {
            for (auto&& d : group.GetDocument()) {
                TReadSearchProtoHelper helper(d);
                TParking::EParkingType parkingType;
                if (!helper.GetProperty("type", parkingType)) {
                    errors.AddMessage("parking", "Incorrect reply structure: 'parking_area' group. Broken field 'type' in document");
                    return false;
                }
                double distance;
                if (!helper.GetProperty("distance", distance)) {
                    errors.AddMessage("parking", "Incorrect reply structure: 'parking_area' group. Broken field 'distance' in document");
                    return false;
                }

                EParkingAbility parkingAbility = EParkingAbility::Undefined;
                if (precision > distance) {
                    if (config.CheckParkingAbility(parkingType, distance, parkingAbility)) {
                        if (parkingAbility == EParkingAbility::Allow) {
                            distAllowCriteria = Min(distance, distAllowCriteria);
                        }
                        if (parkingAbility == EParkingAbility::Deny) {
                            distDenyCriteria = Min(distance, distDenyCriteria);
                        }
                    }
                }
            }
        } else if (group.GetCategoryName() == "exit_possibility") {
            if (group.DocumentSize() != 1) {
                errors.AddMessage("parking", "Incorrect reply structure: 'exit_possibility' group");
                return false;
            }
            TReadSearchProtoHelper helper(group.GetDocument(0));
            ui32 constraintsStop;
            if (!helper.GetProperty("constraint_stops", constraintsStop)) {
                errors.AddMessage("parking", "'constraint_stops' property broken");
                return false;
            }
            ui32 passInStop;
            if (!helper.GetProperty("pass_in_stops", passInStop)) {
                errors.AddMessage("parking", "'pass_in_stops' property broken");
                return false;
            }
            ui32 passOutStop;
            if (!helper.GetProperty("pass_out_stops", passOutStop)) {
                errors.AddMessage("parking", "'pass_out_stops' property broken");
                return false;
            }
            config.CheckParkingAbility(constraintsStop, passInStop, passOutStop, allowExitsLockedCriteria);
        }
    }
    if (allowExitsLockedCriteria == EParkingAbility::Deny) {
        Ability = EParkingAbility::Deny;
        DenyDetails = EParkingDenyDetails::ClosedZone;
    } else {
        Ability = EParkingAbility::Undefined;
        if (config.GetFinalAllowAcceptanceLength() && config.GetFinalDenyAcceptanceLength()) {
            if (config.GetFinalAllowAcceptanceLength() > distAllowCriteria) {
                Ability = EParkingAbility::Allow;
            } else if (config.GetFinalDenyAcceptanceLength() > distDenyCriteria) {
                Ability = EParkingAbility::Deny;
                DenyDetails = EParkingDenyDetails::Parking;
            }
        } else {
            if (distAllowCriteria < distDenyCriteria) {
                Ability = EParkingAbility::Allow;
            } else if (distDenyCriteria < distAllowCriteria) {
                Ability = EParkingAbility::Deny;
                DenyDetails = EParkingDenyDetails::Parking;
            } else if (distDenyCriteria < Max<ui32>() - 1) {
                Ability = EParkingAbility::Deny;
            }
        }
    }
    return true;
}
