#include "params_processor.h"

#include <drive/backend/common/localization.h>
#include <drive/backend/database/transaction/tx.h>

#include <drive/library/cpp/network/data/data.h>

#include <util/generic/guid.h>

IRequestParamsProcessor::IRequestParamsProcessor(const IServerBase* server, const TString& typeName)
    : ConfigHttpStatus(server->GetHttpStatusManagerConfig())
    , BaseServer(server)
    , TypeName(typeName)
{
}

IRequestParamsProcessor& IRequestParamsProcessor::SetTypeName(const TString& name) {
    TypeName = name;
    return *this;
}

TString IRequestParamsProcessor::GetHandlerLocalization(const TString& resourceId, const TString& defaultResult, ELocalization locale) const {
    if (!BaseServer) {
        return defaultResult;
    }
    const ILocalization* localization = BaseServer->GetLocalization();
    if (!localization) {
        return defaultResult;
    }
    return localization->GetLocalString(locale, "handlers." + TypeName + "." + resourceId,
        localization->GetLocalString(locale, "handlers.default." + resourceId, defaultResult)
    );
}

TVector<TString> IRequestParamsProcessor::GetStrings(const TCgiParameters& cgi, TStringBuf name, bool required /*= true*/) const {
    TVector<TString> result;
    auto range = cgi.equal_range(name);
    for (auto i = range.first; i != range.second; ++i) {
        StringSplitter(i->second).SplitBySet(",").SkipEmpty().AddTo(&result);
    }
    if (required) {
        R_ENSURE(
            !result.empty(),
            ConfigHttpStatus.EmptyRequestStatus,
            "missing " << name << " in Cgi parameters"
        );
    }
    return result;
}

TVector<TString> IRequestParamsProcessor::GetStrings(const NJson::TJsonValue& data, TStringBuf name, bool required /*= true*/) const {
    const NJson::TJsonValue* path = data.GetValueByPath(name);
    TVector<TString> result;
    if (!required) {
        if (!path) {
            return result;
        }
    } else {
        R_ENSURE(
            path,
            ConfigHttpStatus.EmptyRequestStatus,
            "missing " << name << " in JSON: " << data.GetStringRobust()
        );
    }
    const NJson::TJsonValue& section = *Yensured(path);
    if (section.IsArray()) {
        for (auto&& i : section.GetArraySafe()) {
            auto s = i.GetStringRobust();
            for (auto&& v : StringSplitter(s).Split(',')) {
                result.emplace_back(v.Token());
            }
        }
    } else {
        StringSplitter(section.GetStringRobust()).SplitBySet(", ").SkipEmpty().AddTo(&result);
    }
    if (required) {
        R_ENSURE(
            !result.empty(),
            ConfigHttpStatus.EmptyRequestStatus,
            "missing " << name << " in JSON: " << data.GetStringRobust()
        );
    }
    return result;
}

template <class T>
TVector<TString> IRequestParamsProcessor::GetStringsImpl(const T& source, TConstArrayRef<TStringBuf> names, bool merge, bool required) const {
    TVector<TString> result;
    for (auto&& name : names) {
        auto ss = GetStrings(source, name, false);
        if (result.empty()) {
            result = std::move(ss);
        } else {
            result.insert(result.end(), ss.begin(), ss.end());
        }
        if (!result.empty() && !merge) {
            break;
        }
    }
    if (required) {
        R_ENSURE(!result.empty(), ConfigHttpStatus.EmptyRequestStatus, "missing " << JoinSeq(",", names));
    }
    return result;
}

TVector<TString> IRequestParamsProcessor::GetStrings(const TCgiParameters& cgi, TConstArrayRef<TStringBuf> names, bool merge /*= true*/, bool required /*= true*/) const {
    return GetStringsImpl(cgi, names, merge, required);
}

TVector<TString> IRequestParamsProcessor::GetStrings(const NJson::TJsonValue& data, TConstArrayRef<TStringBuf> names, bool merge /*= true*/, bool required /*= true*/) const {
    return GetStringsImpl(data, names, merge, required);
}

TString IRequestParamsProcessor::GetString(const TCgiParameters& cgi, TStringBuf name, bool required /*= true*/) const {
    TVector<TString> result;
    auto range = cgi.equal_range(name);
    for (auto i = range.first; i != range.second; ++i) {
        R_ENSURE(result.empty(), ConfigHttpStatus.SyntaxErrorStatus, "multiple " << name << " are specified");
        result.emplace_back(i->second);
    }
    if (required) {
        R_ENSURE(
            !result.empty(),
            ConfigHttpStatus.EmptyRequestStatus,
            "missing " << name << " in Cgi parameters"
        );
    }

    if (!result.empty()) {
        return result[0];
    } else {
        return {};
    }
}

TString IRequestParamsProcessor::GetString(const NJson::TJsonValue& data, TStringBuf name, bool required /*= true*/) const {
    const NJson::TJsonValue* path = data.GetValueByPath(name);
    const NJson::TJsonValue& parameter = path ? *path : Default<NJson::TJsonValue>();
    R_ENSURE(
        !required || parameter.IsString() || parameter.IsBoolean() || parameter.IsDouble() || parameter.IsInteger() || parameter.IsUInteger(),
        ConfigHttpStatus.SyntaxErrorStatus,
        "required POD type for " << name << " field"
    );
    if (parameter.IsDefined()) {
        return parameter.GetStringRobust();
    } else {
        return {};
    }
}

template <class T>
TString IRequestParamsProcessor::GetStringImpl(const T& source, TConstArrayRef<TStringBuf> names, bool required) const {
    TString result;
    for (size_t i = 0; i < names.size(); ++i) {
        bool req = required ? (i + 1 == names.size()) : false;
        result = GetString(source, names[i], req);
        if (result) {
            break;
        }
    }
    return result;
}

TString IRequestParamsProcessor::GetString(const TCgiParameters& cgi, TConstArrayRef<TStringBuf> names, bool required /*= true*/) const {
    return GetStringImpl(cgi, names, required);
}

TString IRequestParamsProcessor::GetString(const NJson::TJsonValue& data, TConstArrayRef<TStringBuf> names, bool required /*= true*/) const {
    return GetStringImpl(data, names, required);
}

template<>
TString IRequestParamsProcessor::ParseValue(TStringBuf value) const {
    return TString(value.Data(), value.Size());
}

bool IRequestParamsProcessor::IsUUID(const TString& value) const {
    return !GetUuid(value).IsEmpty();
}

void IRequestParamsProcessor::ValidateUUIDs(const TVector<TString>& values) const {
    for (auto&& value : values) {
        ValidateUUID(value);
    }
}

void IRequestParamsProcessor::ValidateUUID(const TString& value) const {
    R_ENSURE(
        IsUUID(value),
        ConfigHttpStatus.SyntaxErrorStatus,
        "ill-formed UUID " << value
    );
}

void IRequestParamsProcessor::ParseDataUUIDField(const NJson::TJsonValue& data, TStringBuf field, TString& result, bool required) const {
    ParseDataField(data, field, result, required);
    if (!!result) {
        R_ENSURE(
            IsUUID(result),
            ConfigHttpStatus.SyntaxErrorStatus,
            "parameter " << field << " is an ill-formed UUID: " << result
        );
    }
}
