#pragma once

#include <drive/backend/abstract/base.h>
#include <drive/backend/database/transaction/assert.h>

#include <drive/library/cpp/searchserver/http_status_config.h>

#include <library/cpp/cgiparam/cgiparam.h>

#include <rtline/library/json/parse.h>

class IRequestParamsProcessor {
public:
    explicit IRequestParamsProcessor(const IServerBase* server, const TString& typeName = "");
    virtual ~IRequestParamsProcessor() = default;

    IRequestParamsProcessor& SetTypeName(const TString& name);

    TVector<TString> GetStrings(const TCgiParameters& cgi, TStringBuf name, bool required = true) const;
    TVector<TString> GetStrings(const NJson::TJsonValue& data, TStringBuf name, bool required = true) const;
    TVector<TString> GetStrings(const TCgiParameters& cgi, TConstArrayRef<TStringBuf> names, bool merge = true, bool required = true) const;
    TVector<TString> GetStrings(const NJson::TJsonValue& data, TConstArrayRef<TStringBuf> names, bool merge = true, bool required = true) const;

    TString GetString(const TCgiParameters& cgi, TStringBuf name, bool required = true) const;
    TString GetString(const NJson::TJsonValue& data, TStringBuf name, bool required = true) const;
    TString GetString(const TCgiParameters& cgi, TConstArrayRef<TStringBuf> names, bool required = true) const;
    TString GetString(const NJson::TJsonValue& data, TConstArrayRef<TStringBuf> names, bool required = true) const;

    template <class T>
    TMaybe<TDuration> GetDuration(const T& request, TStringBuf name, bool required = true) const;
    template <class T>
    TDuration GetDuration(const T& request, TStringBuf name, TDuration fallback) const;

    template <class T>
    TMaybe<TInstant> GetTimestamp(const T& request, TStringBuf name, bool required = true) const;
    template <class T>
    TInstant GetTimestamp(const T& request, TStringBuf name, TInstant fallback) const;

    bool IsUUID(const TString& value) const;

    void ValidateUUIDs(const TVector<TString>& values) const;
    void ValidateUUID(const TString& value) const;

    template <class... TArgs>
    TVector<TString> GetUUIDs(TArgs&&... args) const;

    template <class... TArgs>
    TString GetUUID(TArgs&&... args) const;

    template <class T>
    T ParseValue(TStringBuf value) const;

    template<>
    TString ParseValue(TStringBuf value) const;

    template <class T, class... TArgs>
    TVector<T> GetValuesIgnoreErrors(TArgs&&... args) const;

    template <class T, class... TArgs>
    TVector<T> GetValues(TArgs&&... args) const;

    template <class T, class... TArgs>
    TMaybe<T> GetValue(TArgs&&... args) const;

    template <typename T>
    void ParseDataField(const NJson::TJsonValue& data, TStringBuf field, T&& result, bool required = true) const;

    void ParseDataUUIDField(const NJson::TJsonValue& data, TStringBuf field, TString& result, bool required = true) const;

protected:
    const THttpStatusManagerConfig ConfigHttpStatus;
    const IServerBase* BaseServer;

    TString TypeName;

    TString GetHandlerLocalization(const TString& resourceId, const TString& defaultResult, ELocalization locale) const;

private:
    template <class T>
    TVector<TString> GetStringsImpl(const T& source, TConstArrayRef<TStringBuf> names, bool merge, bool required) const;

    template <class T>
    TString GetStringImpl(const T& source, TConstArrayRef<TStringBuf> names, bool required) const;
};

template <class T>
TMaybe<TDuration> IRequestParamsProcessor::GetDuration(const T& request, TStringBuf name, bool required) const {
    auto value = GetValue<ui64>(request, name, required);
    if (value) {
        return TDuration::MicroSeconds(*value);
    } else {
        return {};
    }
}

template <class T>
TDuration IRequestParamsProcessor::GetDuration(const T& request, TStringBuf name, TDuration fallback) const {
    auto result = GetDuration<T>(request, name, /*required=*/false);
    return result.GetOrElse(fallback);
}

template <class T>
TMaybe<TInstant> IRequestParamsProcessor::GetTimestamp(const T& request, TStringBuf name, bool required) const {
    auto value = GetValue<ui64>(request, name, required);
    if (value) {
        return TInstant::Seconds(*value);
    } else {
        return {};
    }
}

template <class T>
TInstant IRequestParamsProcessor::GetTimestamp(const T& request, TStringBuf name, TInstant fallback) const {
    auto result = GetTimestamp<T>(request, name, /*required=*/false);
    return result.GetOrElse(fallback);
}

template <class... TArgs>
TVector<TString> IRequestParamsProcessor::GetUUIDs(TArgs&&... args) const {
    auto result = GetStrings(std::forward<TArgs>(args)...);
    ValidateUUIDs(result);
    return result;
}

template <class... TArgs>
TString IRequestParamsProcessor::GetUUID(TArgs&&... args) const {
    auto result = GetString(std::forward<TArgs>(args)...);
    if (!!result) {
        ValidateUUID(result);
    }
    return result;
}

template <class T>
T IRequestParamsProcessor::ParseValue(TStringBuf value) const {
    T result;
    R_ENSURE(
        TryFromString(value, result),
        ConfigHttpStatus.UserErrorState,
        "cannot parse " << value << " as " << ::TypeName<T>()
    );
    return result;
}

template <class T, class... TArgs>
TVector<T> IRequestParamsProcessor::GetValuesIgnoreErrors(TArgs&&... args) const {
    TVector<T> result;
    for (auto&& i : GetStrings(std::forward<TArgs>(args)...)) {
        T value;
        if (TryFromString<T>(i, value)) {
            result.emplace_back(std::move(value));
        }
    }
    return result;
}
template <class T, class... TArgs>
TVector<T> IRequestParamsProcessor::GetValues(TArgs&&... args) const {
    TVector<T> result;
    for (auto&& i : GetStrings(std::forward<TArgs>(args)...)) {
        result.push_back(ParseValue<T>(i));
    }
    return result;
}
template <class T, class... TArgs>
TMaybe<T> IRequestParamsProcessor::GetValue(TArgs&&... args) const {
    TString s = GetString(std::forward<TArgs>(args)...);
    if (s) {
        return ParseValue<T>(s);
    } else {
        return {};
    }
}

template <typename T>
void IRequestParamsProcessor::ParseDataField(const NJson::TJsonValue& data, TStringBuf field, T&& result, bool required) const {
    R_ENSURE(
        NJson::ParseField(data, field, std::forward<T>(result), required),
        ConfigHttpStatus.SyntaxErrorStatus,
        "either parameter " << field << " is missing or has invalid format: " << data[field].GetStringRobust()
    );
}
