#pragma once

#include "params_processor.h"
#include "versioned.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/report/json.h>

#include <drive/library/cpp/searchserver/context/replier.h>

#include <library/cpp/object_factory/object_factory.h>

#include <rtline/util/types/accessor.h>

#include <util/generic/ptr.h>
#include <util/thread/pool.h>

class TGeoCoord;
class TRegExMatch;

class IRequestProcessorConfig;

class IRequestProcessor
    : public IRequestParamsProcessor
    , public TAtomicRefCount<IRequestProcessor>
{
protected:
    IReplyContext::TPtr Context;
    TString HandlerName;
    const TRegExMatch* AccessControlAllowOrigin;
    bool DumpEventLog = false;
    bool ReportDebugInfo = true;
    TVersionInfo Version;
    TDuration RateLimitFreshness = TDuration::Seconds(10);
    mutable std::atomic<ui64> RateLimit = 0;
    mutable std::atomic<ui64> RateLimitRefresh = 0;

private:
    using TBase = IRequestParamsProcessor;

private:
    bool CheckRequestsRateLimit() const;

private:
    mutable TMaybe<ELocalization> Locale;
    NDrive::TSettingGetterConstPtr Settings;

protected:
    virtual void DoProcess(TJsonReport::TGuard& g) = 0;

    IServerReportBuilder::TCtx GetReportContext() const;

    // ProcessException builds report for coded exception.
    virtual void ProcessException(TJsonReport::TGuard& g, const TCodedException& exception) const;

public:
    virtual TVector<TString> GetHandlerSettingPathes() const {
        return {"handlers." + TypeName, "handlers.default"};
    }

    enum class EApp {
        Unknown /* "unknown" */,
        AndroidClient /* "android_client" */,
        AndroidService /* "android_service" */,
        iOS /* "ios" */,
        Business /* "business" */,
        WebDesktop /* "web_desktop" */,
        WebMobile /* "web_mobile" */,
    };

    using TPtr = TIntrusivePtr<IRequestProcessor>;
    using TFactory = NObjectFactory::TParametrizedObjectFactory<IRequestProcessor, TString>;

public:
    static bool IsAndroid(EApp app);

public:
    IRequestProcessor(const IRequestProcessorConfig& config, IReplyContext::TPtr context, const IServerBase* server, const TString& typeName = "");

    TPtr Self() {
        return this;
    }

    IRequestProcessor& SetDumpEventLog(bool value) {
        DumpEventLog = value;
        return *this;
    }

    IRequestProcessor& SetReportDebugInfo(bool value) {
        ReportDebugInfo = value;
        return *this;
    }

    IRequestProcessor& SetSettings(NDrive::TSettingGetterConstPtr value) {
        Settings = std::move(value);
        return *this;
    }

    IRequestProcessor& SetVersion(const TVersionInfo& info) {
        Version = info;
        return *this;
    }

    IReplyContext::TPtr GetContext() const;

    void Process();
    void Process(IServerReportBuilder::TPtr report, std::function<void(TJsonReport::TGuard&)>&& f);

    TMaybe<TGeoCoord> GetUserLocation(const TString& cgiUserLocation = "") const;
    EApp GetUserApp() const;
    ui32 GetAppBuild() const;
    ELocalization GetLocale() const;

    template <class T>
    TMaybe<T> GetHandlerSetting(TStringBuf key, TInstant actuality = TInstant::Zero()) const {
        auto local = GetSettings().GetValue<T>(TStringBuilder() << "handlers." << TypeName << "." << key, actuality);
        if (local) {
            return local;
        }
        auto global = GetSettings().GetValue<T>(TStringBuilder() << "handlers.default." << key, actuality);
        if (global) {
            return global;
        }
        return {};
    }

    template <class T>
    T GetHandlerSettingDef(TStringBuf key, const T& defaultValue, TInstant actuality = TInstant::Zero()) const {
        auto mayBeResult = GetHandlerSetting<T>(key, actuality);
        if (!mayBeResult) {
            return defaultValue;
        } else {
            return *mayBeResult;
        }
    }

    const NDrive::ISettingGetter& GetSettings() const;
};

class IRequestProcessorConfig {
public:
    constexpr static TDuration NotInitializedTimeout = TDuration::Max();

private:
    R_READONLY(TString, AdditionalCgi);
    R_READONLY(TString, OverrideCgi);
    R_READONLY(TString, OverrideCgiPart);
    R_READONLY(TString, OverridePost);
    R_READONLY(TDuration, RequestTimeout, NotInitializedTimeout);
    R_READONLY(TDuration, RateLimitFreshness, TDuration::Seconds(10));
    R_READONLY(bool, DumpEventLog, false);
    R_READONLY(bool, ReportDebugInfo, true);

protected:
    const TString Name;
    TString Type;
    TString AliasFor;
    const TVersionedKey Key;

protected:
    virtual void DoInit(const TYandexConfig::Section* /*section*/) {
    }

    virtual void DoToString(IOutputStream& /*os*/) const {
    }

public:
    using TFactory = NObjectFactory::TParametrizedObjectFactory<IRequestProcessorConfig, TString, TString>;
    using TPtr = TAtomicSharedPtr<IRequestProcessorConfig>;

public:
    IRequestProcessorConfig(const TVersionedKey& procKey);
    virtual ~IRequestProcessorConfig();

    const TRegExMatch* GetAccessControlAllowOrigin() const {
        return AccessControlAllowOriginRegEx.Get();
    }

    const TString& GetSpecialType() const {
        return Type;
    }

    const TString& GetName() const {
        return Name;
    }

    const TVersionInfo& GetVersion() const {
        return Key.GetVersion();
    }

    IRequestProcessorConfig& SetAliasFor(const TString& aliasFor) {
        AliasFor = aliasFor;
        return *this;
    }

    virtual const TString& GetHandlerName() const {
        return Default<TString>();
    }

    virtual IRequestProcessor::TPtr ConstructProcessor(IReplyContext::TPtr context, const IServerBase* server) const final;
    virtual void CheckServerForProcessor(const IServerBase* server) const;
    virtual void ReadDefaults(const TYandexConfig::Section* section);

    virtual void Init(const TYandexConfig::Section* section) final;
    virtual void ToString(IOutputStream& os) const;

private:
    virtual IRequestProcessor::TPtr DoConstructProcessor(IReplyContext::TPtr context, const IServerBase* server) const = 0;

private:
    TString AccessControlAllowOrigin;
    THolder<TRegExMatch> AccessControlAllowOriginRegEx;
};
