#include "processor.h"

#include <drive/backend/processors/sessions/base_impl.h>

#include <drive/backend/billing/accounts/bonus.h>
#include <drive/backend/billing/accounts/limited.h>
#include <drive/backend/common/localization.h>
#include <drive/backend/data/billing_tags.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/dedicated_fleet.h>
#include <drive/backend/database/entity/search_request.h>
#include <drive/backend/database/entity/search_index.h>
#include <drive/backend/promo_codes/manager.h>
#include <drive/backend/sessions/manager/dedicated_fleet.h>
#include <drive/backend/compiled_riding/manager.h>

#include <drive/library/cpp/balance/client.h>
#include <drive/library/cpp/yadoc/client.h>
#include <rtline/util/algorithm/type_traits.h>

bool LikePhone(const TStringBuf& text) {
    return text.size() == 12 && text.StartsWith("+7");
}

bool IsPhoneSubstr(const TStringBuf& text) {
    return IsNumber(text) || text.StartsWith("+") && IsNumber(text.SubStr(1));
}

void TListAccountProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    TVector<TString> userIds = GetStrings(Context->GetCgiParameters(), "user_id", false);

    auto accountsFilter = GetAvailableWallets(permissions, TAdministrativeAction::EAction::Observe);
    R_ENSURE(accountsFilter, ConfigHttpStatus.UnknownErrorStatus, "cannot fetch available wallets");
    R_ENSURE(Server->GetDriveAPI()->HasBillingManager(), ConfigHttpStatus.UnknownErrorStatus, "cannot get billing manager");
    const TBillingManager& billingManager = Server->GetDriveAPI()->GetBillingManager();
    const NDrive::NBilling::TAccountsManager& accountsManager = billingManager.GetAccountsManager();
    if (!userIds.empty()) {
        bool standartReport = !(IsTrue(Context->GetCgiParameters().Get("multi_user")));
        NJson::TJsonValue report(NJson::JSON_ARRAY);
        for (auto&& userId : userIds) {
            if (userId && !CheckUserGeneralAccessRights(userId, permissions, NAccessVerification::EAccessVerificationTraits::TagsAccess, true)) {
                continue;
            }

            NJson::TJsonValue reportLocal(NJson::JSON_ARRAY);
            auto userAccounts = accountsManager.GetSortedUserAccounts(userId, Context->GetRequestStartTime());
            for (auto&& account : userAccounts) {
                if (accountsFilter->contains(account->GetUniqueName())) {
                    reportLocal.AppendValue(account->GetReport());
                }
            }

            if (standartReport) {
                report = std::move(reportLocal);
                break;
            } else {
                report[userId] = reportLocal;
            }
        }
        g.MutableReport().AddReportElement("accounts", std::move(report));
        g.SetCode(HTTP_OK);
        return;
    }

    auto debtThreshold = permissions->GetSetting<ui32>("billing.debt_threshold_on_perform").GetOrElse(0);

    bool hasMore = false;
    auto accountIds = MakeSet(GetValues<ui64>(Context->GetCgiParameters(), "account_id", false));
    NJson::TJsonValue report(NJson::JSON_ARRAY);
    if (accountIds) {
        const auto withUsers = GetValue<bool>(Context->GetCgiParameters(), "with_users", false).GetOrElse(false);
        auto tx = BuildTx<NSQL::ReadOnly>();
        auto accounts = accountsManager.GetBillingAccounts(accountIds, tx);
        R_ENSURE(accounts, ConfigHttpStatus.UserErrorState, "cannot fetch accounts", EDriveSessionResult::IncorrectRequest);

        TMap<ui64, TSet<TString>> accountUsers;
        TUsersDB::TFetchResult usersData;
        TMaybe<TMap<TString, TMaybe<ui32>>> usersDebts;
        if (withUsers) {
            TSet<TString> userIds;
            FillAccountUsers(accountIds, accountsManager, tx,
                [&accountUsers, &userIds](const auto& link) {
                    accountUsers[link.GetAccountId()].emplace(link.GetUserId());
                    userIds.insert(link.GetUserId());
            });
            usersData = Server->GetDriveAPI()->GetUsersData()->FetchInfo(userIds, tx);
            usersDebts = billingManager.GetUsersDebts(userIds, tx);
            R_ENSURE(usersDebts, ConfigHttpStatus.UnknownErrorStatus, "cannot get users debts");
        }

        NJson::TJsonValue accountsReport;
        for (const auto& [id, account] : *accounts) {
            R_ENSURE(account, ConfigHttpStatus.PermissionDeniedStatus, "not permitted account " + ToString(id), EDriveSessionResult::IncorrectRequest);
            R_ENSURE(accountsFilter->contains(account->GetUniqueName()), ConfigHttpStatus.PermissionDeniedStatus, "not permitted account " + account->GetUniqueName(), EDriveSessionResult::IncorrectRequest);
            if (withUsers) {
                NJson::TJsonValue accountJson;
                accountJson["data"] = account->GetReport();
                NJson::TJsonValue& usersJson = accountJson.InsertValue("users", NJson::JSON_ARRAY);
                auto usersIt = accountUsers.find(account->GetId());
                if (usersIt != accountUsers.end()) {
                    for (auto&& userId : usersIt->second) {
                        auto gUser = usersData.GetResultPtr(userId);
                        usersJson.AppendValue(GetUserReport(userId, gUser, *usersDebts, debtThreshold));
                    }
                } else if (account->GetRecord() && LikePhone(account->GetRecord()->GetExternalId())) {
                    TDriveUserData externalUser;
                    externalUser.SetPhone(account->GetRecord()->GetExternalId());
                    usersJson.AppendValue(externalUser.GetPublicReport());
                }
                accountsReport["accounts"].AppendValue(accountJson);
            } else if (accountIds.size() == 1) {
                accountsReport = account->GetReport();
            } else {
                accountsReport["accounts"].AppendValue(account->GetReport());
            }
        }
        g.MutableReport().SetExternalReport(std::move(accountsReport));
        g.SetCode(HTTP_OK);
        return;
    } else {
        TVector<TString> accountNames = GetStrings(Context->GetCgiParameters(), "account_name", false);
        TVector<ui64> parentIds = GetValues<ui64>(Context->GetCgiParameters(), "parent_id", false);

        TSet<TString> filtredAccounts;
        if (accountNames) {
            for (const auto& accountName : accountNames) {
                R_ENSURE(accountsFilter->contains(accountName), ConfigHttpStatus.PermissionDeniedStatus, "not permitted account " + accountName, EDriveSessionResult::IncorrectRequest);
                filtredAccounts.insert(accountName);
            }
        } else if (parentIds) {
            auto accounts = DriveApi->GetBillingManager().GetAccountsManager().GetAccountsChildren(MakeSet(parentIds), Context->GetRequestStartTime());
            for (const auto& accountName : accounts) {
                R_ENSURE(accountsFilter->contains(accountName.first), ConfigHttpStatus.PermissionDeniedStatus, "not permitted account " + accountName.first, EDriveSessionResult::IncorrectRequest);
                filtredAccounts.insert(accountName.first);
            }
        } else {
            filtredAccounts = *accountsFilter;
        }

        auto numDoc = GetValue<ui32>(Context->GetCgiParameters(), "numdoc", false);
        auto from = GetValue<ui32>(Context->GetCgiParameters(), "from", false);

        TVector<NDrive::NBilling::IBillingAccount::TPtr> allAccounts;
        for (const auto& accountName : filtredAccounts) {
            auto accounts = accountsManager.GetAccountsByName(accountName, Context->GetRequestStartTime());
            R_ENSURE(accounts, ConfigHttpStatus.UnknownErrorStatus, "cannot fetch accounts");
            allAccounts.insert(allAccounts.end(), accounts->begin(), accounts->end());
        }
        auto cmp = [](const NDrive::NBilling::IBillingAccount::TPtr& l, const NDrive::NBilling::IBillingAccount::TPtr& r) {
            return l->GetId() < r->GetId();
        };
        Sort(allAccounts.begin(), allAccounts.end(), cmp);

        bool standartReport = !(IsTrue(Context->GetCgiParameters().Get("multi_user")));
        auto* usersData = Server->GetDriveAPI()->GetUsersData();

        if (standartReport) {
            auto withUsers = GetValue<bool>(Context->GetCgiParameters(), "with_users", false).GetOrElse(true);
            allAccounts.erase(allAccounts.begin(), (from.GetOrElse(0) > allAccounts.size()) ? allAccounts.end() : allAccounts.begin() + from.GetOrElse(0));
            if (allAccounts.size() > numDoc.GetOrElse(200)) {
                hasMore = true;
            }
            if (numDoc.GetOrElse(200) < allAccounts.size()) {
                allAccounts.erase(allAccounts.begin() + numDoc.GetOrElse(200), allAccounts.end());
            }
            TSet<ui64> accountIds;
            Transform(allAccounts.begin(), allAccounts.end(), std::inserter(accountIds, accountIds.begin()), [](const auto& account) {
                return account->GetId();
            });

            TMap<ui64, TVector<TString>> accountUsers;
            TMaybe<TMap<TString, TMaybe<ui32>>> usersDebts;
            if (withUsers) {
                TSet<TString> userIds;
                auto tx = BuildTx<NSQL::ReadOnly>();
                FillAccountUsers(accountIds, accountsManager, tx,
                    [&accountUsers, &userIds](const auto& account) {
                        userIds.insert(account.GetUserId());
                        return accountUsers[account.GetAccountId()].emplace_back(account.GetUserId());
                });
                usersDebts = billingManager.GetUsersDebts(userIds, tx);
                R_ENSURE(usersDebts, ConfigHttpStatus.UnknownErrorStatus, "cannot get users debts");
            }
            for (const auto& account : allAccounts) {
                NJson::TJsonValue accountJson;
                accountJson.InsertValue("data", account->GetReport());
                if (withUsers) {
                    NJson::TJsonValue& usersJson = accountJson.InsertValue("users", NJson::JSON_ARRAY);
                    auto usersIt = accountUsers.find(account->GetId());
                    if (usersIt != accountUsers.end()) {
                        auto fetchResult = usersData->FetchInfo(usersIt->second);
                        for (auto&& userId : usersIt->second) {
                            auto gUser = fetchResult.GetResultPtr(userId);
                            usersJson.AppendValue(GetUserReport(userId, gUser, *usersDebts, debtThreshold));
                        }
                    } else if (account->GetRecord() && LikePhone(account->GetRecord()->GetExternalId())) {
                        TDriveUserData externalUser;
                        externalUser.SetPhone(account->GetRecord()->GetExternalId());
                        usersJson.AppendValue(externalUser.GetPublicReport());
                    }
                }
                report.AppendValue(std::move(accountJson));
            }
        } else {
            TMap<TString, TVector<NDrive::NBilling::IBillingAccount::TPtr>> userAccountsById;
            TMap<TString, TVector<NDrive::NBilling::IBillingAccount::TPtr>> userAccountsByPhone;

            TSet<ui64> accountIds;
            Transform(allAccounts.begin(), allAccounts.end(), std::inserter(accountIds, accountIds.begin()), [](const auto& account) {
                return account->GetId();
            });
            auto tx = BuildTx<NSQL::ReadOnly>();
            TMap<ui64, TVector<TString>> accountUsers;
            TSet<TString> userIds;
            FillAccountUsers(accountIds, accountsManager, tx,
                [&accountUsers, &userIds](const auto& account) {
                    userIds.insert(account.GetUserId());
                    return accountUsers[account.GetAccountId()].emplace_back(account.GetUserId());
            });
            TMaybe<TMap<TString, TMaybe<ui32>>> usersDebts = billingManager.GetUsersDebts(userIds, tx);
            R_ENSURE(usersDebts, ConfigHttpStatus.UnknownErrorStatus, "cannot get users debts");

            for (auto&& account : allAccounts) {
                auto usersIt = accountUsers.find(account->GetId());
                if (usersIt != accountUsers.end()) {
                    for (auto&& userId : usersIt->second) {
                        userAccountsById[userId].push_back(account);
                    }
                } else if (account->GetRecord() && LikePhone(account->GetRecord()->GetExternalId())) {
                    userAccountsByPhone[account->GetRecord()->GetExternalId()].push_back(account);
                }
            }

            ui32 offset = 0;
            if (from.GetOrElse(0) < userAccountsById.size()) {
                auto userFetchResult = usersData->FetchInfo(NContainer::Keys(userAccountsById));

                for (auto&& [userId, accounts] : userAccountsById) {
                    if (offset++ < from.GetOrElse(0)) {
                        continue;
                    }
                    NJson::TJsonValue userJson;
                    {
                        auto gUser = userFetchResult.GetResultPtr(userId);
                        userJson.InsertValue("user", GetUserReport(userId, gUser, *usersDebts, debtThreshold));
                    }
                    auto& accountsJson = userJson.InsertValue("accounts", NJson::JSON_ARRAY);
                    for (auto&& account : accounts) {
                        accountsJson.AppendValue(account->GetReport());
                    }
                    report.AppendValue(std::move(userJson));
                    if (report.GetArray().size() >= numDoc.GetOrElse(200)) {
                        break;
                    }
                }
            } else {
                offset = userAccountsById.size();
            }
            if (offset < userAccountsById.size() + userAccountsByPhone.size()) {
                hasMore = true;
            }
            if (report.GetArray().size() < numDoc.GetOrElse(200)) {
                for (auto&& [userId, accounts] : userAccountsByPhone) {
                    if (offset++ < from.GetOrElse(0)) {
                        continue;
                    }
                    NJson::TJsonValue userJson;
                    {
                        TDriveUserData externalUser;
                        externalUser.SetPhone(userId);
                        userJson.InsertValue("user", externalUser.GetPublicReport());
                    }
                    auto& accountsJson = userJson.InsertValue("accounts", NJson::JSON_ARRAY);
                    for (auto&& account : accounts) {
                        accountsJson.AppendValue(account->GetReport());
                    }
                    report.AppendValue(std::move(userJson));
                    if (report.GetArray().size() >= numDoc.GetOrElse(200)) {
                        break;
                    }
                }
            }
            if (offset < userAccountsById.size() + userAccountsByPhone.size()) {
                hasMore = true;
            }
        }
    }
    g.MutableReport().AddReportElement("accounts", std::move(report));
    g.MutableReport().AddReportElement("has_more", hasMore);
    g.SetCode(HTTP_OK);
}

template <class Inserter>
void TListAccountProcessor::FillAccountUsers(const TSet<ui64>& accountIds, const NDrive::NBilling::TAccountsManager& accountsManager, NDrive::TEntitySession& session, Inserter inserter) {
    TSet<ui64> querySet;
    ui64 count = 0;
    auto batchSize = GetHandlerSettingDef<ui64>("account_batch_size", DefaultAccountUsersBatchSize);
    for (const auto& id : accountIds) {
        querySet.emplace(id);
        count++;
        if (count % batchSize == 0 || count == accountIds.size()) {
            auto accountLinks = accountsManager.GetLinksManager().GetAccountsUsers(querySet, session);
            R_ENSURE(accountLinks, HTTP_INTERNAL_SERVER_ERROR, "cannot fetch account links", session);
            ForEach(accountLinks->begin(), accountLinks->end(), inserter);
            querySet.clear();
        }
    }
}

NJson::TJsonValue TListAccountProcessor::GetUserReport(const TString& userId, const TDriveUserData* gUser, const TMap<TString, TMaybe<ui32>>& usersDebts, const ui32 debtThreshold) {
    NJson::TJsonValue userData;
    if (!!gUser) {
        userData = gUser->GetPublicReport();
        userData["info"] = gUser->GetFullName();
        if (gUser->GetStatus() == NDrive::UserStatusActive && usersDebts.contains(userId)) {
            if (usersDebts.at(userId).GetOrElse(0) > debtThreshold) {
                userData["status"] = NDrive::UserStatusBlocked;
            }
        }
    } else {
        userData["info"] = "";
    }
    userData["id"] = userId;
    return userData;
}

class TWalletsFilter {
    R_FIELD(TSet<ui64>, ParentIds);
    R_FIELD(TSet<TString>, AccountNames);

public:
    void Parse(const TCgiParameters& cgi) {
        {
            auto range = cgi.equal_range("account_names");
            for (auto i = range.first; i != range.second; ++i) {
                StringSplitter(i->second).SplitBySet(",").SkipEmpty().ParseInto(&AccountNames);
            }
        }
        {
            auto range = cgi.equal_range("parent_ids");
            for (auto i = range.first; i != range.second; ++i) {
                StringSplitter(i->second).SplitBySet(",").SkipEmpty().ParseInto(&ParentIds);
            }
        }
    }

    void Filter(TMaybe<TSet<TString>>& accounts, const TDriveAPI& driveApi) const {
        if (AccountNames) {
            if (accounts) {
                EraseNodesIf(*accounts, [&](const TString& name) { return !AccountNames.contains(name); });
            } else {
                accounts = AccountNames;
            }
        }
        if (accounts && accounts->empty()) {
            return;
        }
        if (ParentIds) {
            auto childAccounts = MakeSet(NContainer::Keys(driveApi.GetBillingManager().GetAccountsManager().GetAccountsChildren(MakeSet(ParentIds), TInstant::Zero())));
            if (accounts) {
                EraseNodesIf(*accounts, [&childAccounts](const TString& name) { return !childAccounts.contains(name); });
            } else {
                accounts = childAccounts;
            }
        }
    }
};

void TUserAccountsProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    TWalletsFilter walletFilter;
    walletFilter.Parse(Context->GetCgiParameters());
    auto availableAccounts = GetAvailableAccountNames(permissions, TAdministrativeAction::EAction::Observe);
    walletFilter.Filter(availableAccounts, *DriveApi);

    TVector<std::pair<TString, TDriveUserData>> fetchedUsers;
    TMap<TString, TVector<NDrive::NBilling::IBillingAccount::TPtr>> userAccounts;
    if (auto searchStr = GetString(Context->GetCgiParameters(), "search_str", false)) {
        R_ENSURE(DriveApi->GetUsersData()->IsSearchEnabled(), ConfigHttpStatus.UnknownErrorStatus, "Search was not enabled in config, therefore no index was built");
        auto searchRequest = TSearchRequest();
        for (auto param : SplitString(searchStr, " ")) {
            searchRequest.AddMatchingCondition(TMatchCondition(param, true));
        }
        const ui32 maxLimit = 100;
        auto limit = GetValue<ui32>(Context->GetCgiParameters(), "limit", false).GetOrElse(maxLimit);
        R_ENSURE(limit <= maxLimit, ConfigHttpStatus.SyntaxErrorStatus, "Limit should be equal or less than " + ToString(maxLimit));
        searchRequest.SetLimit(limit);

        auto tx = BuildTx<NSQL::ReadOnly | NSQL::Deferred>();
        auto entityFilter = [&](const TString& userId) -> bool {
            if (!userId || !CheckUserGeneralAccessRights(userId, permissions, NAccessVerification::EAccessVerificationTraits::TagsAccess, false)) {
                return false;
            }
            if (availableAccounts) {
                auto accounts = DriveApi->GetBillingManager().GetAccountsManager().GetSortedUserAccounts(userId, tx);
                R_ENSURE(accounts, ConfigHttpStatus.UnknownErrorStatus, "cannot fetch user accounts", tx);
                EraseIf(*accounts, [&availableAccounts](const auto& acc) { return !availableAccounts->contains(acc->GetUniqueName()); });
                if (accounts->empty()) {
                    return false;
                }
                userAccounts[userId] = std::move(*accounts);
                return true;
            }
            return true;
        };
        auto fetchedIds = MakeSet(DriveApi->GetUsersData()->GetMatchingIds(searchRequest, entityFilter));

        if (IsPhoneSubstr(searchStr)) {
            NSQL::TQueryOptions additionalAccountsOptions;
            additionalAccountsOptions.SetOrderBy({ "account_id" });
            additionalAccountsOptions.AddCustomCondition("external_id like '%" + searchStr + "%'");

            auto accounts = DriveApi->GetBillingManager().GetAccountsManager().GetBillingAccounts(additionalAccountsOptions, tx);
            R_ENSURE(accounts, ConfigHttpStatus.UnknownErrorStatus, "cannot_fetch_infos", tx);
            if (availableAccounts) {
                EraseIf(*accounts, [&availableAccounts](const auto& acc) { return !availableAccounts->contains(acc->GetUniqueName()); });
            }

            TSet<ui64> accountIds;
            Transform(accounts->begin(), accounts->end(), std::inserter(accountIds, accountIds.begin()), [](const auto& acc){ return acc->GetId(); });
            if (accountIds) {
                auto linked = DriveApi->GetBillingManager().GetAccountsManager().GetLinksManager().GetAccountsUsers(accountIds, tx);
                R_ENSURE(linked, ConfigHttpStatus.UnknownErrorStatus, "cannot fetch account links", tx);
                TMap<ui64, NDrive::NBilling::TAccountLinkRecord> linkedAccounts;
                Transform(linked->begin(), linked->end(), std::inserter(linkedAccounts, linkedAccounts.begin()), [](const auto& link) -> std::pair<ui64, NDrive::NBilling::TAccountLinkRecord> { return {link.GetAccountId(), link }; });
                for (const auto& acc : *accounts) {
                    auto itLink = linkedAccounts.FindPtr(acc->GetId());
                    if (!itLink) {
                        auto exteralId = acc->GetRecord()->GetExternalId();
                        if (LikePhone(exteralId)) {
                            TDriveUserData externalUser;
                            externalUser.SetPhone(exteralId);
                            fetchedUsers.emplace_back(exteralId, std::move(externalUser));
                            userAccounts[exteralId].emplace_back(acc);
                        }
                    } else {
                        if (fetchedIds.emplace(itLink->GetUserId()).second) {
                            userAccounts[itLink->GetUserId()].emplace_back(acc);
                        }
                    }
                }
            }
        }
        auto usersFR = DriveApi->GetUsersData()->FetchInfo(fetchedIds).GetResult();
        fetchedUsers.insert(fetchedUsers.begin(), usersFR.begin(), usersFR.end());
    } else if (auto userIds = GetStrings(Context->GetCgiParameters(), "user_id", false)) {
        TSet<TString> fetchedIds;
        for (auto&& userId : userIds) {
            if (!userId || !CheckUserGeneralAccessRights(userId, permissions, NAccessVerification::EAccessVerificationTraits::TagsAccess, true)) {
                continue;
            }
            fetchedIds.emplace(userId);
        }
        auto users = DriveApi->GetUsersData()->FetchInfo(fetchedIds).GetResult();
        Transform(users.begin(), users.end(), std::back_inserter(fetchedUsers), [](const auto& user) -> std::pair<TString, TDriveUserData> { return user; });
    }

    TSet<TString> batchFetch;
    for (const auto& [userId, _] : fetchedUsers) {
        if (!userAccounts.contains(userId)) {
            batchFetch.emplace(userId);
        }
    }

    if (batchFetch) {
        auto tx = BuildTx<NSQL::ReadOnly | NSQL::Deferred>();
        auto accounts = DriveApi->GetBillingManager().GetAccountsManager().GetUsersAccounts(batchFetch, tx);
        R_ENSURE(accounts, ConfigHttpStatus.UnknownErrorStatus, "cannot fetch user accounts", tx);
        for (const auto& acc : *accounts) {
            if (!availableAccounts || availableAccounts->contains(acc->GetUniqueName())) {
                userAccounts[acc->GetUserId()].emplace_back(acc);
            }
        }
    }

    NJson::TJsonValue report(NJson::JSON_ARRAY);
    for (const auto& [userId, user] : fetchedUsers) {
        NJson::TJsonValue userJson;
        {
            NJson::TJsonValue userData;
            userData = user.GetPublicReport();
            userData["info"] = user.GetFullName();
            userData["id"] = user.GetUserId();
            userJson.InsertValue("user", std::move(userData));
        }
        auto& accountsJson = userJson.InsertValue("accounts", NJson::JSON_ARRAY);
        if (auto it = userAccounts.find(userId); it != userAccounts.end()) {
            for (auto&& account : it->second) {
                accountsJson.AppendValue(account->GetReport());
            }
        }
        report.AppendValue(std::move(userJson));
    }

    g.MutableReport().AddReportElement("accounts", std::move(report));
    g.MutableReport().AddReportElement("has_more", false);
    g.SetCode(HTTP_OK);
}

NDrive::TScheme TUpdateCommonAccountProcessor::GetRequestDataScheme(const IServerBase* server, const TCgiParameters& /* schemeCgi */) {
    return NDrive::NBilling::TAccountDescriptionRecord::GetPublicScheme(server);
}

void TUpdateCommonAccountProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const NDrive::NBilling::TAccountsManager& accountsManager = Server->GetDriveAPI()->GetBillingManager().GetAccountsManager();

    TMessagesCollector errors;
    NDrive::NBilling::TAccountDescriptionRecord description;
    R_ENSURE(description.FromJson(requestData, &errors), ConfigHttpStatus.UserErrorState, "Incorrect input data " + errors.GetStringReport(), EDriveSessionResult::IncorrectRequest);

    auto accountsFilter = GetAvailableWallets(permissions, TAdministrativeAction::EAction::Add);
    R_ENSURE(accountsFilter, ConfigHttpStatus.UnknownErrorStatus, "cannot fetch available wallets");

    if (!accountsFilter->contains(description.GetName())) {
        CheckParentId(permissions, TAdministrativeAction::EAction::Add, description.GetParentId());
    }

    auto session = BuildTx<NSQL::Writable>();
    if (!accountsManager.UpsertAccountDescription(description, permissions->GetUserId(), session) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    g.SetCode(HTTP_OK);
}

void TGetCommonAccountsProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const NDrive::NBilling::TAccountsManager& accountsManager = Server->GetDriveAPI()->GetBillingManager().GetAccountsManager();
    NJson::TJsonValue report(NJson::JSON_ARRAY);

    auto parentId = GetValue<ui64>(Context->GetCgiParameters(), "parent_id", false);
    auto accountNames = MakeSet(GetValues<TString>(Context->GetCgiParameters(), "account_name", false));
    auto accountsFilter = GetAvailableWallets(permissions, TAdministrativeAction::EAction::ObserveStructure);
    R_ENSURE(accountsFilter, ConfigHttpStatus.UnknownErrorStatus, "cannot fetch available wallets");

    TMap<ui64, TVector<NDrive::NBilling::TAccountDescriptionRecord>> filteredAccounts;
    for (const auto& name : *accountsFilter) {
        if (accountNames && !accountNames.contains(name)) {
            continue;
        }

        auto account = accountsManager.GetDescriptionByName(name);
        R_ENSURE(account, ConfigHttpStatus.UnknownErrorStatus, "cannot find valid description");
        if (parentId.Defined() && account->GetParentId() != *parentId) {
            continue;
        }

        filteredAccounts[account->GetParentId()].push_back(*account);
    }

    TMap<ui64, TMap<TString, TVector<TString>>> taggedUsers;
    TSet<TString> taggedUserIds;
    TUsersDB::TFetchResult userFetchResult;
    const auto accessTags = GetStrings(Context->GetCgiParameters(), "access_tags", false);
    if (!accessTags.empty()) {
        auto session = BuildTx<NSQL::ReadOnly>();
        TVector<TDBTag> tags;
        if (!DriveApi->GetTagsManager().GetUserTags().RestoreTags({}, accessTags, tags, session)) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
        for (const auto& tag : tags) {
            if (auto walletTag = tag.GetTagAs<TWalletAccessTag>(); walletTag && filteredAccounts.contains(walletTag->GetParentId())) {
                taggedUsers[walletTag->GetParentId()][tag.GetData()->GetName()].push_back(tag.GetObjectId());
                taggedUserIds.insert(tag.GetObjectId());
            }
        }
        userFetchResult = DriveApi->GetUsersData()->FetchInfo(taggedUserIds, session);
    }

    TMap<TString, TString> accountUsers;
    for (const auto& [_, accounts] : filteredAccounts) {
        for (const auto& account : accounts) {
            NJson::TJsonValue accountReport = account.GetReport();
            if (account.GetParentId() != 0) {
                for (const auto& [tagName, tagObjectIds] : taggedUsers[account.GetParentId()]) {
                    NJson::TJsonValue subReport(NJson::JSON_ARRAY);
                    for (const auto& objectId : tagObjectIds) {
                        auto userData = userFetchResult.GetResultPtr(objectId);
                        if (userData && CheckUserGeneralAccessRights(objectId, permissions, NAccessVerification::EAccessVerificationTraits::TagsAccess, false)) {
                            subReport.AppendValue(userData->GetReport());
                        }
                    }
                    accountReport.SetValueByPath(tagName, std::move(subReport));
                }
            }
            auto accounts = accountsManager.GetAccountsByName(account.GetName(), Context->GetRequestStartTime());
            R_ENSURE(accounts, ConfigHttpStatus.UnknownErrorStatus, "cannot fetch accounts");
            accountReport.InsertValue("accounts_count", accounts->size());
            report.AppendValue(accountReport);
        }
    }

    g.MutableReport().AddReportElement("accounts", std::move(report));
    g.SetCode(HTTP_OK);
}

void TLinkAccountProcessor::LinkSingleAccount(const TString& userId, ui32 accountId, TUserPermissions::TPtr permissions, const TString& promocode, NDrive::TEntitySession& session) {
    const NDrive::NBilling::TAccountsManager& accountsManager = Server->GetDriveAPI()->GetBillingManager().GetAccountsManager();

    auto account = accountsManager.LinkAccount(userId, accountId, permissions->GetUserId(), session);
    R_ENSURE(account, ConfigHttpStatus.UserErrorState, "internal error: cannot link", session);
    R_ENSURE(account->IsPersonal(), ConfigHttpStatus.UserErrorState, "can't link not personal account", EDriveSessionResult::IncorrectRequest);

    if (promocode) {
        TPromocodeAccountLink promoLink;
        promoLink.SetPromocode(promocode).SetAccountId(accountId).SetUserId(userId);
        R_ENSURE(
            Server->GetDriveAPI()->GetBillingManager().GetPromocodeAccountLinks().AddHistory(promoLink, permissions->GetUserId(), EObjectHistoryAction::UpdateData, session, nullptr),
            ConfigHttpStatus.UserErrorState,
            "can't link promocode " + session.GetStringReport(),
            EDriveSessionResult::InconsistencyUser
        );
    }
}

void TLinkAccountProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const NDrive::NBilling::TAccountsManager& accountsManager = Server->GetDriveAPI()->GetBillingManager().GetAccountsManager();
    TString singleUserId = GetString(requestData, "user_id", false);
    auto accountId = GetValue<ui32>(requestData, "account_id", false);
    const TString promocode = GetString(requestData, "promocode", false);
    const TString accountName = GetString(requestData, "name", false);

    TVector<TString> users;
    if (!singleUserId) {
        users = GetUUIDs(requestData, "users", true);
        R_ENSURE(users.size() < Config.GetMaxItems(), ConfigHttpStatus.UserErrorState, "too many items");
    } else {
        users.push_back(singleUserId);
    }

    NJson::TJsonValue accountsJson(NJson::JSON_ARRAY);

    TString action = GetString(requestData, "action", false);
    if (!action || action == "link") {
        auto session = BuildTx<NSQL::Writable>();

        if (accountId) {
            CheckAccountId(permissions, TAdministrativeAction::EAction::Add, *accountId);
            for (auto&& userId : users) {
                accountsJson.AppendValue(*accountId);
                LinkSingleAccount(userId, *accountId, permissions, promocode, session);
            }
        } else {
            auto description = accountsManager.GetDescriptionByName(accountName, TInstant::Zero());
            R_ENSURE(description.Defined(), ConfigHttpStatus.UserErrorState, "unknown type", EDriveSessionResult::IncorrectRequest);
            R_ENSURE(description->GetIsPersonal(), ConfigHttpStatus.UserErrorState, "can't link not personal account", EDriveSessionResult::IncorrectRequest);

            if (description->GetMaxLinks() > 1) {
                R_ENSURE(GetValue<bool>(requestData, "force", false).GetOrElse(false), ConfigHttpStatus.UserErrorState, "can't create new multiuser account", EDriveSessionResult::IncorrectRequest);
            }

            TVector<ui32> accountIds = CreateAccounts(permissions, *description, requestData, session, users.size());
            R_ENSURE(users.size() == accountIds.size(), ConfigHttpStatus.UserErrorState, "can't create accounts");

            for (size_t i = 0; i < users.size(); ++i) {
                accountsJson.AppendValue(accountIds[i]);
                LinkSingleAccount(users[i], accountIds[i], permissions, promocode, session);
            }
        }

        if (!session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    } else if (action == "unlink") {
        R_ENSURE(accountId || accountName, ConfigHttpStatus.UserErrorState, "unknown account_id", EDriveSessionResult::IncorrectRequest);
        TSet<std::pair<TString, ui64>> userAccounts;
        auto session = BuildTx<NSQL::Writable>();
        if (!accountId) {
            auto description = accountsManager.GetDescriptionByName(accountName, TInstant::Zero());
            R_ENSURE(description.Defined(), ConfigHttpStatus.UserErrorState, "unknown type", EDriveSessionResult::IncorrectRequest);
            for (auto&& userId : users) {
                auto accounts = accountsManager.GetLinksManager().GetUserAccounts(userId, session);
                if (!accounts) {
                    session.DoExceptionOnFail(ConfigHttpStatus);
                }
                for (auto&& acc : *accounts) {
                    if (acc.GetTypeId() == description->GetId()) {
                        userAccounts.emplace(userId, acc.GetAccountId());
                        break;
                    }
                }
            }
        } else {
            for (auto&& userId : users) {
                userAccounts.emplace(userId, *accountId);
            }
        }

        for (auto&& [userId, accountId] : userAccounts) {
            CheckAccountId(permissions, TAdministrativeAction::EAction::Modify, accountId);
            accountsJson.AppendValue(accountId);
            R_ENSURE(accountsManager.UnLinkAccount(userId, accountId, permissions->GetUserId(), session), ConfigHttpStatus.UserErrorState, "internal error: cannot unlink", session);
        }

        if (!session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    } else {
        R_ENSURE(false, ConfigHttpStatus.UserErrorState, "unknown action " + action, EDriveSessionResult::IncorrectRequest);
    }
    g.MutableReport().AddReportElement("account_ids", std::move(accountsJson));
    g.SetCode(HTTP_OK);
}

void TSearchAccountProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const NDrive::NBilling::TAccountsManager& accountsManager = Server->GetDriveAPI()->GetBillingManager().GetAccountsManager();
    auto accountId = GetValue<ui32>(requestData, "account_id", false);
    auto externalId = GetValue<TString>(requestData, "external_id", false);
    if (!externalId) {
        externalId = GetValue<TString>(requestData, "inn", false);
    }
    auto spravId = GetValue<TString>(requestData, "sprav_id", false);
    auto limit = GetValue<ui32>(requestData, "limit", false).GetOrElse(50);
    auto cursor = GetValue<ui32>(requestData, "cursor", false).GetOrElse(0);

    R_ENSURE(accountId || externalId || spravId, HTTP_BAD_REQUEST, "no params");

    auto accountsFilter = GetAvailableWallets(permissions, TAdministrativeAction::EAction::Observe);
    R_ENSURE(accountsFilter, ConfigHttpStatus.UnknownErrorStatus, "cannot fetch available wallets");

    auto session = BuildTx<NSQL::ReadOnly>();
    NSQL::TQueryOptions mainOptions(50 * limit);
    mainOptions.SetOrderBy({ "account_id" });
    TStringBuilder customCond;
    {
        NSQL::TQueryOptions accountOptions;
        if (accountId) {
            accountOptions.AddGenericCondition("account_id", ToString(*accountId));
        }
        if (externalId) {
            accountOptions.AddGenericCondition("external_id", *externalId);
        }
        if (!accountOptions.GetGenericConditions().empty() || !accountOptions.GetCustomCondition().empty()) {
            customCond << "(" << accountOptions.PrintConditions(*session) << ")";
        }
    }
    if (spravId) {
        if (!customCond.empty()) {
            customCond << " OR ";
        }
        customCond << "(" << NSQL::TQueryOptions::PrintCondition("sprav_id", TSet<TString>{ *spravId }, *session) << ")";
    }
    mainOptions.AddCustomCondition(cursor ? "(" + customCond + ") AND account_id > " + ToString(cursor) : customCond);

    auto accounts = accountsManager.GetBillingAccounts(mainOptions, session);
    R_ENSURE(accounts, ConfigHttpStatus.UnknownErrorStatus, "cannot_fetch_infos", session);


    TVector<NDrive::NBilling::IBillingAccount::TPtr> filtredAccounts;
    for (auto&& acc : *accounts) {
        if (accountsFilter->contains(acc->GetUniqueName())) {
            filtredAccounts.emplace_back(std::move(acc));
        }
    }
    auto cmp = [](const NDrive::NBilling::IBillingAccount::TPtr& l, const NDrive::NBilling::IBillingAccount::TPtr& r) {
        return l->GetId() < r->GetId();
    };
    Sort(filtredAccounts.begin(), filtredAccounts.end(), cmp);

    NJson::TJsonValue accountsJson(NJson::JSON_ARRAY);
    bool hasMore = (filtredAccounts.size() > limit);
    filtredAccounts.resize(Min<ui32>(filtredAccounts.size(), limit));
    for (auto&& acc : filtredAccounts) {
        accountsJson.AppendValue(acc->GetReport());
    }

    auto spravInviteLink = GetHandlerSettingDef<TString>("sprav_invite_link", "");
    if (spravId && spravInviteLink) {
        SubstGlobal(spravInviteLink, "{{spravId}}", *spravId);
        g.MutableReport().AddReportElement("invite_link", spravInviteLink);
    }
    g.MutableReport().AddReportElement("accounts", std::move(accountsJson));
    g.MutableReport().AddReportElement("has_more", hasMore);
    g.SetCode(HTTP_OK);
}

void TPromocodeLinkAccountListProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const auto& billingManager = Server->GetDriveAPI()->GetBillingManager();
    const auto& accountsManager = billingManager.GetAccountsManager();
    auto since = GetValue<ui64>(Context->GetCgiParameters(), "since_id", true).GetOrElse(0);
    auto session = billingManager.BuildSession(true);
    auto optionalEvents = billingManager.GetPromocodeAccountLinks().GetEvents(since, session);
    R_ENSURE(optionalEvents, {}, "cannot GetEvents since " << since, session);

    NJson::TJsonValue filtredEvents;
    auto accountsFilter = GetAvailableWallets(permissions, TAdministrativeAction::EAction::Observe);
    R_ENSURE(accountsFilter, ConfigHttpStatus.UnknownErrorStatus, "cannot fetch available wallets");
    for (auto&& event : *optionalEvents) {
        auto account = accountsManager.GetAccountById(event.GetAccountId());
        if (!account) {
            continue;
        }
        auto description = accountsManager.GetDescriptionById(account->GetTypeId());
        if (!description) {
            continue;
        }
        if (!accountsFilter->contains(description->GetName())) {
            if (description->GetParentId() != 0) {
                auto parent = accountsManager.GetAccountById(description->GetParentId());
                if (parent && accountsFilter->contains(parent->GetUniqueName())) {
                    filtredEvents.AppendValue(event.BuildReportItem());
                }
            }
        } else {
            filtredEvents.AppendValue(event.BuildReportItem());
        }
    }

    g.AddReportElement("links", std::move(filtredEvents));
    g.SetCode(HTTP_OK);
}

void TGetPromocodeLinkAccountProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const auto& billingManager = Server->GetDriveAPI()->GetBillingManager();
    auto accounts = GetValues<ui32>(requestData, "accounts", true);
    for (auto&& accountId : accounts) {
        CheckAccountId(permissions, TAdministrativeAction::EAction::Observe, accountId);
    }
    auto session = billingManager.BuildSession(true);
    auto optionalEvents = billingManager.GetPromocodeAccountLinks().GetPromocodeByAccountFromDB(accounts, session);
    R_ENSURE(optionalEvents, {}, "cannot GetPromocodeByAccountFromDB", session);

    NJson::TJsonValue accountMap(NJson::JSON_MAP);
    for (const auto& link : *optionalEvents) {
        accountMap[::ToString(link.GetAccountId())] = link.GetPromocode();
    }

    g.AddReportElement("promocodes", std::move(accountMap));
    g.SetCode(HTTP_OK);
}

void TActivateAccountProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const NDrive::NBilling::TAccountsManager& accountsManager = Server->GetDriveAPI()->GetBillingManager().GetAccountsManager();

    auto accountId = GetValue<ui32>(requestData, "account_id", false);
    TVector<ui32> accounts;
    if (accountId) {
        accounts.push_back(*accountId);
    } else {
        accounts = GetValues<ui32>(requestData, "accounts", true);
        R_ENSURE(accounts.size() < Config.GetMaxItems(), ConfigHttpStatus.UserErrorState, "too many items");
    }

    R_ENSURE(accounts.size() > 0, ConfigHttpStatus.SyntaxErrorStatus, "No accounts");
    bool activeFlag = GetValue<bool>(requestData, "active_flag", false).GetOrElse(false);

    for (auto&& accountId : accounts) {
        CheckAccountId(permissions, TAdministrativeAction::EAction::ObserveStructure, accountId);
    }

    auto session = BuildTx<NSQL::Writable>();
    session.SetComment("ActivateAccountProcessor");
    if (!accountsManager.ActivateAccounts(accounts, activeFlag, permissions->GetUserId(), session) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    g.SetCode(HTTP_OK);
}

void TRemoveAccountProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    auto accountId = GetValue<ui32>(requestData, "account_id", true);
    R_ENSURE(accountId.Defined(), ConfigHttpStatus.SyntaxErrorStatus, "Need 'account_id'");
    CheckAccountId(permissions, TAdministrativeAction::EAction::Remove, *accountId);

    const NDrive::NBilling::TAccountsManager& accountsManager = Server->GetDriveAPI()->GetBillingManager().GetAccountsManager();
    auto session = BuildTx<NSQL::Writable>();
    if (!accountsManager.RemoveAccount(*accountId, permissions->GetUserId(), session) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    g.SetCode(HTTP_OK);
}

void TUpdateAccountDataProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    auto accountSpec = GetValue<ui32>(requestData, "account_id", false);
    TVector<ui32> accounts;
    if (accountSpec) {
        accounts.push_back(*accountSpec);
    } else {
        accounts = GetValues<ui32>(requestData, "accounts", true);
        R_ENSURE(accounts.size() < Config.GetMaxItems(), ConfigHttpStatus.UserErrorState, "too many items");
    }

    for (auto&& accountId : accounts) {
        CheckAccountId(permissions, TAdministrativeAction::EAction::Modify, accountId);
    }

    const NDrive::NBilling::TAccountsManager& accountsManager = Server->GetDriveAPI()->GetBillingManager().GetAccountsManager();
    auto session = BuildTx<NSQL::Writable>();
    for (auto&& accountId : accounts) {
        auto account = accountsManager.GetAccountById(accountId);
        R_ENSURE(account,
            ConfigHttpStatus.UserErrorState,
            "can't find account",
            EDriveSessionResult::IncorrectRequest,
            session
        );
        if (!account->PatchAccountData(requestData, permissions->GetUserId(), session)) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }

    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    g.SetCode(HTTP_OK);
}

template<class T>
void CheckBalanceResponse(const TExpected<T, TBalanceClient::TError>& answer, const IServerBase* server) {
    if (!answer) {
        NDrive::TEntitySession tx;
        tx.SetLocalizedMessageKey(server->GetLocalization()->GetLocalString(ELocalization::Rus, "balance_client." + answer.GetError().GetCode(), answer.GetError().GetMessage()));
        R_ENSURE(answer, HTTP_INTERNAL_SERVER_ERROR, answer.GetError().GetFullError(), tx);
    }
}

NDrive::TScheme TCreateAccountProcessor::GetRequestDataScheme(const IServerBase* server, const TCgiParameters& schemeCgi) {
    NDrive::TScheme scheme;
    scheme.Add<TFSBoolean>("active_flag", "Активность").SetDefault(false);
    auto& accountStructure = scheme.Add<TFSStructure>("account_meta", "Данные организации").SetStructure<NDrive::TScheme>();
    accountStructure.Add<TFSString>("company", "Название организации").SetRequired(true);
    auto withBalanceScheme = IsTrue(schemeCgi.Get("balance"));
    auto separateAddress = IsTrue(schemeCgi.Get("separate_address"));
    if (withBalanceScheme && server && server->GetAs<NDrive::IServer>() && server->GetAs<NDrive::IServer>()->GetBalanceClient()) {
        {
            auto descriptionStr = server->GetSettings().GetValue<TString>("scheme.balance_client.person").GetOrElse("");
            NJson::TJsonValue descriptionJson;
            if (!ReadJsonTree(descriptionStr, &descriptionJson)) {
                WARNING_LOG << "Error while parsing gvar value scheme.balance_client.person";
            }

            TBalanceClient::TPerson::AddToScheme(accountStructure, descriptionJson, TBalanceClient::TPerson::EType::Ur, false, separateAddress);
        }
        {
            auto descriptionStr = server->GetSettings().GetValue<TString>("scheme.balance_client.client").GetOrElse("");
            NJson::TJsonValue descriptionJson;
            if (!ReadJsonTree(descriptionStr, &descriptionJson)) {
                WARNING_LOG << "Error while parsing gvar value scheme.balance_client.client";
            }

            TBalanceClient::TClient::AddToScheme(accountStructure, descriptionJson);
        }
    }
    return scheme;
}

void TCreateAccountProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const NDrive::NBilling::TAccountsManager& accountsManager = Server->GetDriveAPI()->GetBillingManager().GetAccountsManager();

    bool uniqueDescription = GetValue<bool>(Context->GetCgiParameters(), "unique_description", false).GetOrElse(false);
    TString accountName = GetString(requestData, "name", false);
    if (!accountName) {
        accountName = GetString(Context->GetCgiParameters(), "name", true);
    }

    TString createAccessTag = GetString(Context->GetCgiParameters(), "create_access_tag", false);
    auto withBalance = IsTrue(GetString(Context->GetCgiParameters(), "balance", false));

    TString metaDescriptionName;
    if (uniqueDescription && requestData.Has("default_description") && requestData["default_description"].Has("name")) {
        metaDescriptionName = requestData["default_description"]["name"].GetString();
    }

    auto lockSession = BuildTx<NSQL::Writable | NSQL::RepeatableRead>();
    TMaybe<bool> locked;
    if (createAccessTag && withBalance) {
        locked = lockSession.TryLock("create_account_lock_" + permissions->GetUserId());
    } else if (uniqueDescription && metaDescriptionName) {
        locked = lockSession.TryLock("create_account_lock_" + metaDescriptionName);
    } else {
        locked = true;
    }

    R_ENSURE(locked, ConfigHttpStatus.UserErrorState, "error while getting lock", EDriveSessionResult::InternalError, lockSession);
    R_ENSURE(*locked, ConfigHttpStatus.TooManyRequestsStatus, "operation in process", EDriveSessionResult::LockedResourcesLimitEnriched, lockSession);

    if (createAccessTag) {
        auto session = BuildTx<NSQL::ReadOnly>();
        TVector<TDBTag> tags;
        if (!Server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreEntityTags(permissions->GetUserId(), { createAccessTag }, tags, session)) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
        if (tags.size()) {
            auto accessTag = std::dynamic_pointer_cast<TWalletAccessTag>(tags.front().GetData());
            R_ENSURE(accessTag, ConfigHttpStatus.SyntaxErrorStatus, "incorrect access tag name", EDriveSessionResult::IncorrectRequest);
            g.MutableReport().AddReportElement("account_id", accessTag->GetParentId());
            g.SetCode(HTTP_OK);
            return;
        }
    }

    auto registeredAccounts = accountsManager.GetRegisteredAccounts(TInstant::Zero());
    TMaybe<NDrive::NBilling::TAccountDescriptionRecord> description;
    TMaybe<NDrive::NBilling::TAccountDescriptionRecord> metaDescription;
    for (auto&& acc : registeredAccounts) {
        if (acc.GetName() == accountName) {
            description = acc;
        }
        if (metaDescriptionName && acc.GetName() == metaDescriptionName) {
            metaDescription = acc;
        }
    }

    R_ENSURE(description, ConfigHttpStatus.UserErrorState, "unknown type '" + accountName + "'", EDriveSessionResult::IncorrectRequest);
    auto accountData = requestData;

    if (uniqueDescription && metaDescriptionName) {
        auto accountsFilter = GetAvailableWallets(permissions, TAdministrativeAction::EAction::ObserveStructure);
        if (accountsFilter->contains(metaDescriptionName)) {
            auto account = accountsManager.GetDescriptionByName(metaDescriptionName);
            R_ENSURE(account, ConfigHttpStatus.UnknownErrorStatus, "cannot find valid description");
            g.MutableReport().AddReportElement("account_id", account->GetParentId());
            g.SetCode(HTTP_OK);
            return;
        }

        R_ENSURE(!(!accountsFilter->contains(metaDescriptionName) && metaDescription), ConfigHttpStatus.PermissionDeniedStatus, "invalid description name");
    }

    if (withBalance && Server->GetBalanceClient()) {
        TBalanceClient::TClient client;
        R_ENSURE(client.FromJson(accountData["account_meta"]), ConfigHttpStatus.UserErrorState, "cannot parse client", EDriveSessionResult::IncorrectRequest);
        TBalanceClient::TPerson person;
        R_ENSURE(person.FromJson(accountData["account_meta"]), ConfigHttpStatus.UserErrorState, "cannot parse partner", EDriveSessionResult::IncorrectRequest);
        TBalanceClient::TContract contract;

        auto clientId = Server->GetBalanceClient()->GetOrCreateContractForUid(permissions->GetUid(), client, person, contract);
        CheckBalanceResponse(clientId, Server);
        NDrive::NBilling::TBalanceClientInfo balanceInfo;
        balanceInfo.SetClientId(clientId->first);
        balanceInfo.SetPersonId(person.GetId() ? person.GetId() : contract.GetPersonId());
        balanceInfo.SetContractId(contract.GetId());
        balanceInfo.SetExternalContractId(contract.GetExternalId());

        accountData["account_meta"]["balance_info"] = balanceInfo.ToJson();
        if (person.HasINN()) {
            accountData["external_id"] = person.GetINNRef();
        }
    }

    auto session = BuildTx<NSQL::Writable>();
    TVector<ui32> accountIds = CreateAccounts(permissions, *description, accountData, session);
    R_ENSURE(accountIds.size(), ConfigHttpStatus.UserErrorState, "cannot create account");

    if (createAccessTag) {
        for (const auto& account : accountIds) {
            auto tag = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(createAccessTag);
            R_ENSURE(tag, ConfigHttpStatus.UnknownErrorStatus, "cannot create access tag");
            auto accessTag = std::dynamic_pointer_cast<TWalletAccessTag>(tag);
            R_ENSURE(accessTag, ConfigHttpStatus.SyntaxErrorStatus, "incorrect access tag name", EDriveSessionResult::IncorrectRequest);
            accessTag->SetParentId(account);
            auto addedTags = Server->GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(tag, permissions->GetUserId(), permissions->GetUserId(), Server, session);
            if (!addedTags) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
            if (addedTags->empty()) {
                TVector<TDBTag> tags;
                if (!Server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreEntityTags(permissions->GetUserId(), { createAccessTag }, tags, session)) {
                    session.DoExceptionOnFail(ConfigHttpStatus);
                }
                R_ENSURE(tags.size(), ConfigHttpStatus.UnknownErrorStatus, "inconsistency tags", EDriveSessionResult::InconsistencyUser);
                auto localAccessTag = std::dynamic_pointer_cast<TWalletAccessTag>(tags.front().GetData());
                R_ENSURE(localAccessTag, ConfigHttpStatus.SyntaxErrorStatus, "incorrect access tag name", EDriveSessionResult::IncorrectRequest);
                g.MutableReport().AddReportElement("account_id", localAccessTag->GetParentId());
                g.SetCode(HTTP_OK);
                return;
            }
        }
    }

    NJson::TJsonValue defaultDescription;
    if (requestData.Has("default_description")) {
        defaultDescription = requestData["default_description"];
    } else {
        TString descriptionStr;
        descriptionStr = GetHandlerSettingDef<TString>("default_description", descriptionStr);
        if (descriptionStr && NJson::ReadJsonFastTree(descriptionStr, &defaultDescription)) {
            defaultDescription.InsertValue("name", ::ToString(accountIds.front()) + "_default");
        }
    }

    if (defaultDescription.IsDefined()) {
        defaultDescription["meta"].InsertValue("parent_id", accountIds.front());
        if (!defaultDescription["meta"].Has("hr_name")) {
            defaultDescription["meta"].InsertValue("hr_name", requestData["account_meta"]["company"]);
        }
        TMessagesCollector errors;
        NDrive::NBilling::TAccountDescriptionRecord descriptionRecord;
        R_ENSURE(descriptionRecord.FromJson(defaultDescription, &errors), ConfigHttpStatus.UserErrorState, "Incorrect input data " + errors.GetStringReport(), EDriveSessionResult::IncorrectRequest);
        if (!accountsManager.UpsertAccountDescription(descriptionRecord, permissions->GetUserId(), session)) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }

    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    g.MutableReport().AddReportElement("account_id", accountIds.front());
    g.SetCode(HTTP_OK);
}

NDrive::TScheme TGetBalancePersonProcessor::GetRequestDataScheme(const IServerBase* server, const TCgiParameters& schemeCgi) {
    NDrive::TScheme scheme;
    auto separateAddress = IsTrue(schemeCgi.Get("separate_address"));
    auto descriptionStr = server->GetSettings().GetValue<TString>("scheme.balance_client.person").GetOrElse("");
    NJson::TJsonValue descriptionJson;
    if (!ReadJsonTree(descriptionStr, &descriptionJson)) {
        WARNING_LOG << "Error while parsing gvar value scheme.balance_client.person";
    }

    TBalanceClient::TPerson::AddToScheme(scheme, descriptionJson, TBalanceClient::TPerson::EType::Ur, true, separateAddress);

    return scheme;
}

void TGetBalancePersonProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    R_ENSURE(Server->GetBalanceClient(), ConfigHttpStatus.UnknownErrorStatus, "no balance client");
    auto accountId = GetValue<ui32>(Context->GetCgiParameters(), "account_id", true);
    CheckAccountId(permissions, TAdministrativeAction::EAction::Observe, *accountId);

    const NDrive::NBilling::TAccountsManager& accountsManager = Server->GetDriveAPI()->GetBillingManager().GetAccountsManager();
    NDrive::NBilling::IBillingAccount::TPtr account = accountsManager.GetAccountById(*accountId);
    R_ENSURE(account, ConfigHttpStatus.SyntaxErrorStatus, "can't find account", EDriveSessionResult::IncorrectRequest);
    auto accountRecord = account->GetRecordAs<NDrive::NBilling::TLimitedAccountRecord>();
    R_ENSURE(accountRecord, ConfigHttpStatus.SyntaxErrorStatus, "incorrect account type", EDriveSessionResult::IncorrectRequest);
    R_ENSURE(accountRecord->HasBalanceInfo(), ConfigHttpStatus.SyntaxErrorStatus, "no balance info", EDriveSessionResult::IncorrectRequest);

    auto person = Server->GetBalanceClient()->GetPerson(accountRecord->GetBalanceInfoRef().GetPersonId());
    CheckBalanceResponse(person, Server);

    NJson::TJsonValue report;
    person->AddToJson(report);
    g.MutableReport().SetExternalReport(std::move(report));
    g.SetCode(HTTP_OK);
}

void TUpdateBalancePersonProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    R_ENSURE(Server->GetBalanceClient(), ConfigHttpStatus.UnknownErrorStatus, "no balance client");
    auto accountId = GetValue<ui32>(Context->GetCgiParameters(), "account_id", true);
    CheckAccountId(permissions, TAdministrativeAction::EAction::Modify, *accountId);

    TBalanceClient::TPerson person;
    R_ENSURE(person.FromJson(requestData), ConfigHttpStatus.SyntaxErrorStatus, "can't parse person", EDriveSessionResult::IncorrectRequest);

    const NDrive::NBilling::TAccountsManager& accountsManager = Server->GetDriveAPI()->GetBillingManager().GetAccountsManager();
    NDrive::NBilling::IBillingAccount::TPtr account = accountsManager.GetAccountById(*accountId);
    R_ENSURE(account, ConfigHttpStatus.SyntaxErrorStatus, "can't find account", EDriveSessionResult::IncorrectRequest);
    auto accountRecord = account->GetRecordAs<NDrive::NBilling::TLimitedAccountRecord>();
    R_ENSURE(accountRecord, ConfigHttpStatus.SyntaxErrorStatus, "incorrect account type", EDriveSessionResult::IncorrectRequest);
    R_ENSURE(accountRecord->HasBalanceInfo() && person.GetId() == accountRecord->GetBalanceInfoRef().GetPersonId() && person.GetClientId() == accountRecord->GetBalanceInfoRef().GetClientId(), ConfigHttpStatus.SyntaxErrorStatus, "no balance info", EDriveSessionResult::IncorrectRequest);

    auto updateResult = Server->GetBalanceClient()->UpdatePerson(permissions->GetUid(), person);
    CheckBalanceResponse(updateResult, Server);

    g.SetCode(HTTP_OK);
}

void TGetBalancePaymentLinkProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    R_ENSURE(Server->GetBalanceClient(), ConfigHttpStatus.UnknownErrorStatus, "no balance client");
    auto accountId = GetValue<ui32>(Context->GetCgiParameters(), "account_id", true);
    auto sum = GetValue<ui32>(Context->GetCgiParameters(), "sum", true).GetOrElse(0);
    R_ENSURE(sum, ConfigHttpStatus.SyntaxErrorStatus, "incorrect sum", EDriveSessionResult::IncorrectRequest);
    auto nds = Server->GetSettings().GetValueDef<ui32>("billing.nds", 0);
    if (nds > 0) {
        sum = sum * (100 + nds) / 100;
    }
    CheckAccountId(permissions, TAdministrativeAction::EAction::Modify, *accountId);
    const NDrive::NBilling::TAccountsManager& accountsManager = Server->GetDriveAPI()->GetBillingManager().GetAccountsManager();
    NDrive::NBilling::IBillingAccount::TPtr account = accountsManager.GetAccountById(*accountId);
    R_ENSURE(account, ConfigHttpStatus.SyntaxErrorStatus, "can't find account", EDriveSessionResult::IncorrectRequest);

    auto accountRecord = account->GetRecordAs<NDrive::NBilling::TLimitedAccountRecord>();
    R_ENSURE(accountRecord, ConfigHttpStatus.SyntaxErrorStatus, "incorrect account type", EDriveSessionResult::IncorrectRequest);
    R_ENSURE(accountRecord->HasBalanceInfo(), ConfigHttpStatus.SyntaxErrorStatus, "no balance info", EDriveSessionResult::IncorrectRequest);

    auto linkResult = Server->GetBalanceClient()->GetPaymentLink(permissions->GetUid(), sum, accountRecord->GetBalanceInfoRef().GetClientId(), accountRecord->GetBalanceInfoRef().GetContractId());
    CheckBalanceResponse(linkResult, Server);
    g.MutableReport().AddReportElement("link", *linkResult);

    auto code = GetString(Context->GetCgiParameters(), "code", false);
    if (code) {
        auto session = BuildTx<NSQL::Writable>();
        auto chatSession = BuildChatSession(false);
        NJson::TJsonValue promoReport;
        IPromoCodesManager::TApplyContext context(code);
        context.PaymentSum = sum;
        if (!Server->GetPromoCodesManager()->ApplyCode(context, ToString(accountId), permissions, promoReport, session, chatSession) || !session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus, Server->GetLocalization());
        }
        if (!chatSession.Commit()) {
            chatSession.DoExceptionOnFail(ConfigHttpStatus, Server->GetLocalization());
        }
        if (promoReport.IsDefined()) {
            g.MutableReport().AddReportElement("promo_report", std::move(promoReport));
        }
    }
    g.SetCode(HTTP_OK);
}

void TUpdateOrganizationBalanceProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    auto accountId = GetValue<ui32>(Context->GetCgiParameters(), "account_id", true);
    auto sum = GetValue<ui32>(Context->GetCgiParameters(), "sum", true).GetOrElse(0);
    R_ENSURE(sum, ConfigHttpStatus.SyntaxErrorStatus, "incorrect sum", EDriveSessionResult::IncorrectRequest);
    auto nds = Server->GetSettings().GetValueDef<ui32>("billing.nds", 0);
    if (nds > 0) {
        sum = sum * (100 + nds) / 100;
    }
    CheckAccountId(permissions, TAdministrativeAction::EAction::Modify, *accountId);

    const NDrive::NBilling::TAccountsManager& accountsManager = Server->GetDriveAPI()->GetBillingManager().GetAccountsManager();
    NDrive::NBilling::IBillingAccount::TPtr account = accountsManager.GetAccountById(*accountId);
    R_ENSURE(account, ConfigHttpStatus.SyntaxErrorStatus, "can't find account", EDriveSessionResult::IncorrectRequest);
    auto accountRecord = account->GetRecordAs<NDrive::NBilling::TLimitedAccountRecord>();
    R_ENSURE(accountRecord, ConfigHttpStatus.SyntaxErrorStatus, "incorrect account type", EDriveSessionResult::IncorrectRequest);
    R_ENSURE(accountRecord->HasBalanceInfo(), ConfigHttpStatus.SyntaxErrorStatus, "no balance info", EDriveSessionResult::IncorrectRequest);

    TBalanceClientConfig testConfig = TBalanceClientConfig::ParseFromString(
        TStringBuilder() << "Host: " << "http://greed-ts.paysys.yandex.ru:30702" << Endl
        << "Uri: xmlrpc" << Endl
        << "<RequestConfig>" << Endl
        << "MaxAttempts: 1" << Endl
        << "TimeoutSendingms: 1000" << Endl
        << "</RequestConfig>" << Endl
        << "MethodPrefix: " << "TestBalance" << Endl
    );
    TBalanceClient balanceClient(testConfig);
    auto result = balanceClient.TopupBalance(sum, accountRecord->GetBalanceInfoRef().GetPersonId());
    CheckBalanceResponse(result, Server);

    auto code = GetString(Context->GetCgiParameters(), "code", false);
    if (code) {
        auto session = BuildTx<NSQL::Writable>();
        auto chatSession = BuildChatSession(false);
        NJson::TJsonValue promoReport;
        IPromoCodesManager::TApplyContext context(code);
        context.PaymentSum = sum;
        if (!Server->GetPromoCodesManager()->ApplyCode(context, ToString(accountId), permissions, promoReport, session, chatSession) || !session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus, Server->GetLocalization());
        }
        if (!chatSession.Commit()) {
            chatSession.DoExceptionOnFail(ConfigHttpStatus, Server->GetLocalization());
        }
        if (promoReport.IsDefined()) {
            g.MutableReport().AddReportElement("promo_report", std::move(promoReport));
        }
    }

    g.SetCode(HTTP_OK);
}

void TSetOrganizationOverdraftProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    auto accountId = GetValue<ui32>(requestData, "account_id", true);
    auto overdraft = GetValue<ui64>(requestData, "overdraft", true).GetOrElse(0);
    CheckAccountId(permissions, TAdministrativeAction::EAction::Modify, *accountId);

    const NDrive::NBilling::TAccountsManager& accountsManager = Server->GetDriveAPI()->GetBillingManager().GetAccountsManager();
    NDrive::NBilling::IBillingAccount::TPtr account = accountsManager.GetAccountById(*accountId);
    R_ENSURE(account, ConfigHttpStatus.SyntaxErrorStatus, "can't find account", EDriveSessionResult::IncorrectRequest);
    auto accountRecord = account->GetRecordAs<NDrive::NBilling::TLimitedAccountRecord>();
    R_ENSURE(accountRecord, ConfigHttpStatus.SyntaxErrorStatus, "incorrect account type", EDriveSessionResult::IncorrectRequest);
    R_ENSURE(accountRecord->HasBalanceInfo(), ConfigHttpStatus.SyntaxErrorStatus, "no balance info", EDriveSessionResult::IncorrectRequest);
    if (overdraft == accountRecord->GetBalanceInfoRef().GetOverdraft()) {
        g.SetCode(HTTP_OK);
        return;
    }

    auto balanceInfo = accountRecord->GetBalanceInfoRef();
    balanceInfo.SetOverdraft(overdraft);
    NJson::TJsonValue accountData;
    accountData["balance_info"] = balanceInfo.ToJson();
    auto session = BuildTx<NSQL::Writable>();
    if (!account->PatchAccountData(accountData, permissions->GetUserId(), session) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    g.SetCode(HTTP_OK);
}

void TGetBalanceDocumentsProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    R_ENSURE(Server->GetYaDocClient(), ConfigHttpStatus.UnknownErrorStatus, "no yadoc client");
    auto accountId = GetValue<ui32>(Context->GetCgiParameters(), "account_id", true).GetOrElse(0);
    CheckAccountId(permissions, TAdministrativeAction::EAction::Observe, accountId);

    const NDrive::NBilling::TAccountsManager& accountsManager = Server->GetDriveAPI()->GetBillingManager().GetAccountsManager();
    NDrive::NBilling::IBillingAccount::TPtr account = accountsManager.GetAccountById(accountId);
    R_ENSURE(account, ConfigHttpStatus.SyntaxErrorStatus, "can't find account", EDriveSessionResult::IncorrectRequest);
    auto accountRecord = account->GetRecordAs<NDrive::NBilling::TLimitedAccountRecord>();
    R_ENSURE(accountRecord, ConfigHttpStatus.SyntaxErrorStatus, "incorrect account type", EDriveSessionResult::IncorrectRequest);
    R_ENSURE(accountRecord->HasBalanceInfo(), ConfigHttpStatus.SyntaxErrorStatus, "no balance info", EDriveSessionResult::IncorrectRequest);

    auto from = GetTimestamp(Context->GetCgiParameters(), "from_date", false);
    if (!from) {
        auto session = BuildTx<NSQL::Writable>();
        NSQL::TQueryOptions options(5);
        options.AddGenericCondition("account_id", ToString(accountId));
        auto accountHistory = accountsManager.GetAccountsHistory().GetEvents(0, session, options);
        if (!accountHistory) {
            session.Check();
        }
        if (accountHistory->empty()) {
            from = Now() - TDuration::Days(366);
        } else {
            from = accountHistory->front().GetHistoryTimestamp();
        }
    }
    auto to = GetTimestamp(Context->GetCgiParameters(), "to_date", false);
    if (!to) {
        to = Now();
    }

    auto types = GetStrings(Context->GetCgiParameters(), "type", false);

    auto documentsInfo = Server->GetYaDocClient()->GetDocuments(*from, *to, ToString(accountRecord->GetBalanceInfoRef().GetContractId()));

    auto eventLogState = NDrive::TEventLog::CaptureState();
    auto report = g.GetReport();
    g.Release();

    documentsInfo.Subscribe([
        eventLogState = std::move(eventLogState),
        report,
        types = MakeSet(types)
    ](const NThreading::TFuture<TYaDocClient::TDocuments>& w) {
        NDrive::TEventLog::TStateGuard stateGuard(eventLogState);
        TJsonReport::TGuard g(report, HTTP_OK);
        if (w.HasValue()) {
            auto docs = w.GetValue();
            if (types) {
                EraseIf(docs, [&types](const TYaDocClient::TDocument& doc){ return !types.contains(doc.Type); });
            }
            g.MutableReport().AddReportElement("documents", NJson::ToJson(docs));
            g.SetCode(HttpCodes::HTTP_OK);
        } else {
            g.MutableReport().AddReportElement("status", "error");
            g.MutableReport().AddReportElement("error", NThreading::GetExceptionInfo(w));
            g.SetCode(HttpCodes::HTTP_INTERNAL_SERVER_ERROR);
        }
    });
}

void TDownloadBalanceDocumentsProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    R_ENSURE(Server->GetYaDocClient(), ConfigHttpStatus.UnknownErrorStatus, "no balance client");
    auto accountId = GetValue<ui32>(Context->GetCgiParameters(), "account_id", true);
    CheckAccountId(permissions, TAdministrativeAction::EAction::Observe, *accountId);

    const NDrive::NBilling::TAccountsManager& accountsManager = Server->GetDriveAPI()->GetBillingManager().GetAccountsManager();
    NDrive::NBilling::IBillingAccount::TPtr account = accountsManager.GetAccountById(*accountId);
    R_ENSURE(account, ConfigHttpStatus.SyntaxErrorStatus, "can't find account", EDriveSessionResult::IncorrectRequest);
    auto accountRecord = account->GetRecordAs<NDrive::NBilling::TLimitedAccountRecord>();
    R_ENSURE(accountRecord, ConfigHttpStatus.SyntaxErrorStatus, "incorrect account type", EDriveSessionResult::IncorrectRequest);
    R_ENSURE(accountRecord->HasBalanceInfo(), ConfigHttpStatus.SyntaxErrorStatus, "no balance info", EDriveSessionResult::IncorrectRequest);

    auto documentIds = GetStrings(Context->GetCgiParameters(), "ids", true);
    auto asFile = GetValue<bool>(Context->GetCgiParameters(), "as_file", false).GetOrElse(documentIds.size() == 1);
    R_ENSURE(!asFile || documentIds.size() == 1, ConfigHttpStatus.SyntaxErrorStatus, "cannot dowload more then one", EDriveSessionResult::IncorrectRequest);

    NThreading::TFuture<TYaDocClient::TRawDocument> result;
    if (asFile) {
        result = Server->GetYaDocClient()->GetDocumentById(documentIds.front());
    } else {
        result = Server->GetYaDocClient()->GetArchive(documentIds);
    }

    auto eventLogState = NDrive::TEventLog::CaptureState();
    auto report = g.GetReport();
    g.Release();

    result.Subscribe([
        eventLogState = std::move(eventLogState),
        report
    ] (const NThreading::TFuture<TYaDocClient::TRawDocument>& w) {
        NDrive::TEventLog::TStateGuard stateGuard(eventLogState);
        try {
            if (!report) {
                throw yexception() << "incorrect report value";
            }
            const auto& reply = w.GetValue();
            TBuffer buf;
            buf.Assign(reply.Buffer.data(), reply.Buffer.size());
            report->Finish(HTTP_OK, reply.MimeType, buf);
        } catch (const yexception& e) {
            TJsonReport::TGuard g(report, HttpCodes::HTTP_INTERNAL_SERVER_ERROR);
            g.MutableReport().AddReportElement("status", "error");
            g.MutableReport().AddReportElement("error", e.what());
        }
    });
}

void TAccountHistoryProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    auto accountIdExp = GetValue<ui32>(Context->GetCgiParameters(), "account_id", true);
    auto since = GetTimestamp(Context->GetCgiParameters(), "since", false);
    auto historyComment = GetValue<TString>(Context->GetCgiParameters(), "history_marker", false);
    auto limit = GetValue<ui64>(Context->GetCgiParameters(), "limit", false).GetOrElse(0);
    auto offset = GetValue<ui64>(Context->GetCgiParameters(), "offset", false).GetOrElse(0);
    R_ENSURE(accountIdExp.Defined(), ConfigHttpStatus.UserErrorState, "can't find account_id", EDriveSessionResult::IncorrectRequest);

    ui32 accountId = accountIdExp.GetRef();
    CheckAccountId(permissions, TAdministrativeAction::EAction::ObserveStructure, accountId);

    const NDrive::NBilling::TAccountsManager& accountsManager = Server->GetDriveAPI()->GetBillingManager().GetAccountsManager();

    auto session = BuildTx<NSQL::ReadOnly>();
    auto accountHistory = accountsManager.GetAccountsHistory().GetEventsByAccountId(*accountIdExp, since, historyComment, session, limit ? limit + 1 : limit, offset);
    if (!accountHistory) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    auto linksHistory = accountsManager.GetLinksManager().GetHistoryManager().GetEventsByAccounts({ accountId }, TInstant::Zero(), session);
    if (!linksHistory) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    TVector<NJson::TJsonValue> accountReport;
    for (size_t i = 0; i < accountHistory->size(); ++i) {
        const auto& accEv = (*accountHistory)[i];
        if (limit && i >= limit) {
            continue;
        }
        accountReport.push_back(accEv.BuildReportItem());
        auto currLimitedRecord = std::dynamic_pointer_cast<NDrive::NBilling::TLimitedAccountRecord>(accEv.GetRecord());
        if (i != accountHistory->size() - 1) {
            auto prevLimitedRecord = std::dynamic_pointer_cast<NDrive::NBilling::TLimitedAccountRecord>((*accountHistory)[i + 1].GetRecord());
            if (currLimitedRecord && prevLimitedRecord) {
                accountReport.back()["account_meta"]["soft_limit_diff"] = (i64)(currLimitedRecord->GetSoftLimit() - prevLimitedRecord->GetSoftLimit());
            }
        } else {
            if (currLimitedRecord) {
                accountReport.back()["account_meta"]["soft_limit_diff"] = currLimitedRecord->GetSoftLimit();
            }
        }
    }

    NJson::TJsonValue linksReport = NJson::JSON_ARRAY;
    for (auto&& linkEv : *linksHistory) {
        NStorage::TTableRecord record = linkEv.SerializeToTableRecord();
        linksReport.AppendValue(record.SerializeToJson());
    }

    g.MutableReport().AddReportElement("operations_history", NJson::RangeToJson(accountReport.begin(), accountReport.end()));
    if (limit) {
        g.MutableReport().AddReportElement("has_more", accountHistory->size() > limit);
    }
    g.MutableReport().AddReportElement("links_history", std::move(linksReport));
    g.SetCode(HTTP_OK);
}

void TAccountDescriptionHistoryProcessor::Parse(const TCgiParameters& cgi) {
    Name = GetValue<TString>(cgi, "name", false);
    Id = GetValue<ui32>(cgi, "id", false);
    Since = GetTimestamp(cgi, "since", false);
    HistoryComment = GetValue<TString>(cgi, "history_marker", false);
    Limit = GetValue<ui64>(cgi, "limit", false).GetOrElse(0);
    Offset = GetValue<ui64>(cgi, "offset", false).GetOrElse(0);
}

void TAccountDescriptionHistoryProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ObserveStructure, TAdministrativeAction::EEntity::Wallet);
    R_ENSURE(Name.Defined() || Id.Defined(), ConfigHttpStatus.UserErrorState, "can't find account description", EDriveSessionResult::IncorrectRequest);

    const auto& accountsManager = Server->GetDriveAPI()->GetBillingManager().GetAccountsManager();
    const auto& historyManager = accountsManager.GetAccountsDescriptionHistory();

    auto session = BuildTx<NSQL::ReadOnly>();
    NDrive::NBilling::TAccountDescriptionHistory::TOptionalEvents history;
    if (Name.Defined()) {
        history = historyManager.GetEventsByDescriptionName(*Name, Since, HistoryComment, session, Limit ? Limit + 1 : Limit, Offset);
    } else {
        history = historyManager.GetEventsByDescriptionId(*Id, Since, HistoryComment, session, Limit ? Limit + 1 : Limit, Offset);
    }

    if (!history) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    TVector<NJson::TJsonValue> descriptionsReport;
    for (size_t i = 0; i < history->size(); ++i) {
        const auto& event = (*history)[i];
        if (Limit && i >= Limit) {
            continue;
        }
        
        descriptionsReport.push_back(event.BuildReportItem());
    }

    g.MutableReport().AddReportElement("operations_history", NJson::RangeToJson(descriptionsReport.begin(), descriptionsReport.end()));
    if (Limit) {
        g.MutableReport().AddReportElement("has_more", history->size() > Limit);
    }

    g.SetCode(HTTP_OK);
}

void TRemoveAccountDescriptionProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    auto descriptionId = GetValue<ui32>(requestData, "type_id", true);
    R_ENSURE(descriptionId, ConfigHttpStatus.SyntaxErrorStatus, "Need 'type_id'");
    const NDrive::NBilling::TAccountsManager& accountsManager = Server->GetDriveAPI()->GetBillingManager().GetAccountsManager();

    auto description = accountsManager.GetDescriptionById(*descriptionId);
    R_ENSURE(description.Defined(), ConfigHttpStatus.SyntaxErrorStatus, "Incorrect 'type_id' " + ::ToString(*descriptionId));
    auto accountsFilter = GetAvailableWallets(permissions, TAdministrativeAction::EAction::Remove);
    R_ENSURE(accountsFilter, ConfigHttpStatus.UnknownErrorStatus, "cannot fetch available wallets");
    R_ENSURE(accountsFilter->contains(description->GetName()), ConfigHttpStatus.PermissionDeniedStatus, "not permitted account", EDriveSessionResult::IncorrectRequest);

    auto session = BuildTx<NSQL::Writable>();
    if (!accountsManager.RemoveAccountDescription(*descriptionId, permissions->GetUserId(), session) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    g.SetCode(HTTP_OK);
}

void TAccountSessionItemProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    Y_UNUSED(requestData);
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const auto sessionId = GetString(cgi, "session_id");

    auto availableWallets = GetAvailableWallets(permissions, TAdministrativeAction::EAction::Observe);
    R_ENSURE(availableWallets, ConfigHttpStatus.UnknownErrorStatus, "cannot fetch available wallets");
    g.AddEvent(NJson::TMapBuilder
        ("event", "AvailableWallets")
        ("wallets", NJson::ToJson(*availableWallets))
    );

    auto tx = BuildTx<NSQL::ReadOnly>();
    auto ydbTx = BuildYdbTx<NSQL::ReadOnly>("account_session_item");

    THistoryRidesContext context(*Server, TInstant::Zero(), true, true);
    {
        auto eg = g.BuildEventGuard("InitializeHistoryRidesContext");
        R_ENSURE(context.InitializeSession(sessionId, tx, ydbTx), ConfigHttpStatus.UnknownErrorStatus, "cannot initialize HistoryRidesContext");
    }

    auto session = context.GetSession(sessionId);
    R_ENSURE(session, ConfigHttpStatus.EmptySetStatus, "cannot find session_id " << sessionId);

    auto sessions = NContainer::Scalar(*session);
    PrefetchTracks(sessions, g);
    PrefetchPayments(sessions, tx, g);
    PrefetchFullCompiledRiding(sessions, g);
    PrefetchGeocodedLocations(sessions, g);
    PrefetchUsers(sessions, tx, g);
    PrefetchAggressiveEvents(sessions, g);
    PrefetchFineAttachments(sessions, tx, g);
    SetReportOffer(true);
    SetTrackFormat("legacy");

    auto offer = session->GetOffer();
    R_ENSURE(offer, ConfigHttpStatus.UnknownErrorStatus, "cannot get offer for session_id " << sessionId);
    R_ENSURE(availableWallets->contains(offer->GetSelectedCharge()), ConfigHttpStatus.PermissionDeniedStatus, "cannot view session_id " << sessionId);

    const auto& fetcher = GetFetcher(sessions, g, permissions);
    {
        NJson::TJsonValue data = GetReport(*session, g, permissions, /*detailed=*/true);
        g.AddReportElement("order", std::move(data));
    }
    fetcher.BuildReportMeta(g.MutableReport());
    g.SetCode(HTTP_OK);
}

void TAccountSessionListProcessor::Parse(const TCgiParameters& cgi) {
    Limit = GetValue<ui64>(cgi, "limit", false).GetOrElse(10);
    Start = Context->GetRequestStartTime();
    Since = GetTimestamp(cgi, "since", Start - TDuration::Days(1));
    Until = GetTimestamp(cgi, "until", Start);
    AddActual = Until >= Start;

    CarIds = MakeSet(GetStrings(cgi, "car_id", false));
    UserIds = MakeSet(GetStrings(cgi, "user_id", false));
    StrSearchParams = GetStrings(cgi, "has_all_of", false);
    AccountNames = GetStrings(Context->GetCgiParameters(), "account_name", false);
    ParentIds = GetValues<ui64>(Context->GetCgiParameters(), "parent_id", false);

    if (CarIds && !UserIds) {
        SessionGetterTraits = EProcessorTraits::GetSessionsByCarIds;
    } else if (StrSearchParams) {
        SessionGetterTraits = EProcessorTraits::FilterByInputString | EProcessorTraits::GetSessionsByUserIds;
    } else {
        SessionGetterTraits = (CarIds && UserIds) ? EProcessorTraits::GetSessionsByCarIds | EProcessorTraits::FilterByUserIds : EProcessorTraits::GetSessionsByUserIds;
    }

    SessionReportTraits = NDriveSession::GetReportTraits(cgi);
    const auto traits = GetValues<NDriveSession::EReportTraits>(cgi, "traits", false);
    for (auto&& trait : traits) {
        SessionReportTraits |= trait;
    }

    const auto reportTraitsCgi = GetValue<TString>(cgi, "report", false).GetOrElse("default");
    DeviceReportTraits = NDeviceReport::GetReportTraits(reportTraitsCgi);

    const TString filter = GetString(cgi, "filter", false);
    const TSet<TString> filterNames = StringSplitter(filter).Split(',').SkipEmpty();
    CommonFilter = ConstructFilter(filterNames);
}

void TAccountSessionListProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    Y_UNUSED(requestData);
    R_ENSURE(!(StrSearchParams && (CarIds || UserIds)), ConfigHttpStatus.SyntaxErrorStatus, "using has_all_of with car_id or user_id is not allowed");

    auto availableWallets = GetAvailableWallets(permissions, TAdministrativeAction::EAction::Observe);
    R_ENSURE(availableWallets, ConfigHttpStatus.UnknownErrorStatus, "cannot fetch available wallets");

    if (AccountNames) {
        for (const auto& accountName : AccountNames) {
            R_ENSURE(availableWallets->contains(accountName), ConfigHttpStatus.PermissionDeniedStatus, "not permitted account " + accountName, EDriveSessionResult::IncorrectRequest);
            ResultContext.FiltredAccounts.insert(accountName);
        }
    } else if (ParentIds) {
        auto accounts = DriveApi->GetBillingManager().GetAccountsManager().GetAccountsChildren(MakeSet(ParentIds), Context->GetRequestStartTime());
        for (const auto& accountName : accounts) {
            R_ENSURE(availableWallets->contains(accountName.first), ConfigHttpStatus.PermissionDeniedStatus, "not permitted account " + accountName.first, EDriveSessionResult::IncorrectRequest);
            ResultContext.FiltredAccounts.insert(accountName.first);
        }
    } else {
        ResultContext.FiltredAccounts = *availableWallets;
    }

    g.AddEvent(NJson::TMapBuilder
        ("event", "AvailableWallets")
        ("wallets", NJson::ToJson(ResultContext.FiltredAccounts))
    );

    THistoryRidesContext context(*Server, Since, SessionReportTraits & NDriveSession::ReportAggression, SessionReportTraits & NDriveSession::ReportFines);
    auto tx = BuildTx<NSQL::ReadOnly>();

    ProcessorPolicy = EResultProcessorPolicy::Dummy;
    if (SessionGetterTraits & EProcessorTraits::FilterByInputString) {
        if (TryUseOptiIndexSearch()) {
            SessionGetterTraits |= EProcessorTraits::GetSessionsByCarIds | EProcessorTraits::FilterWithIndexSearch;
            ProcessorPolicy = EResultProcessorPolicy::Unique;
        } else {
            ProcessorPolicy = EResultProcessorPolicy::SessionsSubStr;
        }
    }

    TVector<THistoryRideObject> sessions;
    TVector<THistoryRideObject> actualSessions;

    if ((SessionGetterTraits & EProcessorTraits::GetSessionsByCarIds)
        && ((SessionGetterTraits & EProcessorTraits::FilterWithIndexSearch) ? !CarIds.empty() : true)) {
        if (GetCarsSessions(g, tx, context)) {
            ResultFilter(ResultContext.ActualSessions, actualSessions);
            ResultFilter(ResultContext.ClosedSessions, sessions);
        }
        ResultContext.ClearCurrentResults();
    }

    if ((SessionGetterTraits & EProcessorTraits::GetSessionsByUserIds)
        && ((SessionGetterTraits & EProcessorTraits::FilterWithIndexSearch) ? !UserIds.empty() : true)) {
        if (AddActual) {
            if (GetActualSessions(g, tx, context)) {
                ResultFilter(ResultContext.ActualSessions, actualSessions);
            }
            ResultContext.ClearCurrentResults();
        }

        TInstant currentUntil = Until;
        while (GetClosedUsersSessions(g, tx, context, currentUntil) && sessions.size() < Limit) {
            currentUntil = TInstant::FromValue(ResultContext.ClosedSessions.back().GetLastTS().GetValue() - 1);

            ResultFilter(ResultContext.ClosedSessions, sessions);
            ResultContext.ClearCurrentResults();

            if (!ResultContext.HasMore) {
                break;
            }
        }
    }

    std::sort(sessions.begin(), sessions.end(),
        [](const THistoryRideObject& left, const THistoryRideObject& right) {
            return left.GetLastTS() > right.GetLastTS();
        });

    ui64 cursorUntil = 0;
    if (ResultContext.HasMore || sessions.size() > Limit) {
        sessions.resize(Min<size_t>(sessions.size(), Limit));
        cursorUntil = sessions.back().GetLastTS().Seconds();
        ResultContext.HasMore = true;
    }

    //TODO actual sessions is not configured to size limits
    sessions.insert(sessions.end(), actualSessions.begin(), actualSessions.end());

    PrefetchFullCompiledRiding(sessions, g);
    PrefetchGeocodedLocations(sessions, g);
    PrefetchUsers(sessions, tx, g);

    if (SessionReportTraits & NDriveSession::ReportAggression) {
        PrefetchAggressiveEvents(sessions, g);
    }

    const NDriveModelReport::TReportTraits modelTraits = SessionReportTraits & NDriveSession::ReportUserModelInfo
        ? NDriveModelReport::UserReport
        : NDriveModelReport::ReportNames;
    const auto& fetcher = GetFetcher(sessions, g, permissions, DeviceReportTraits, modelTraits);
    {
        NJson::TJsonValue orders = NJson::JSON_ARRAY;
        NJson::TJsonValue actualOrders = NJson::JSON_ARRAY;
        for (auto&& session : sessions) {
            if (session.IsActive()) {
                actualOrders.AppendValue(GetReport(session, g, permissions, /* detailed= */ false, SessionReportTraits));
            } else {
                orders.AppendValue(GetReport(session, g, permissions, /* detailed= */ false, SessionReportTraits));
            }
        }
        g.AddReportElement("orders", std::move(orders));
        if (AddActual) {
            g.AddReportElement("actual_orders", std::move(actualOrders));
        }
    }

    if (ResultContext.HasMore) {
        NJson::TJsonValue cursor;
        cursor["since"] = Since.Seconds();
        cursor["until"] = cursorUntil;
        g.AddReportElement("cursor", std::move(cursor));
    }

    fetcher.BuildReportMeta(g.MutableReport());
    g.SetCode(HTTP_OK);
}

bool TAccountSessionListProcessor::GetCarsSessions(TJsonReport::TGuard& g, NDrive::TEntitySession& tx, THistoryRidesContext& context) {
    auto eg = g.BuildEventGuard("InitializeHistoryRidesContext");
    auto ydbTx = BuildYdbTx<NSQL::ReadOnly>("account_session_list");
    R_ENSURE(context.InitializeCars({CarIds.begin(),CarIds.end()}, tx, ydbTx, Until), ConfigHttpStatus.UnknownErrorStatus, "cannot initialize HistoryRidesContext");

    bool hasMore = false;
    ResultContext.ClosedSessions = context.GetSessions(Until, Limit, &hasMore, SessionReportTraits & NDriveSession::ReportEmpty,
        [this](const THistoryRideObject& session) {
            return (!session.IsActive())
                && WalletFilter(ResultContext.FiltredAccounts, session)
                && CommonFilter(session)
                && ((SessionGetterTraits & EProcessorTraits::FilterByUserIds) ? UserIds.contains(session.GetUserId()) : true);
        });

    if (AddActual) {
        ResultContext.ActualSessions = context.GetSessions(TInstant::Max(), Max<ui32>(), nullptr, SessionReportTraits & NDriveSession::ReportEmpty,
            [this](const THistoryRideObject& session) {
                return session.IsActive()
                    && WalletFilter(ResultContext.FiltredAccounts, session)
                    && CommonFilter(session)
                    && ((SessionGetterTraits & EProcessorTraits::FilterByUserIds) ? UserIds.contains(session.GetUserId()) : true);
        });
    }

    ResultContext.HasMore = ResultContext.HasMore || hasMore;

    return !ResultContext.ClosedSessions.empty() || !ResultContext.ActualSessions.empty();
}

bool TAccountSessionListProcessor::GetClosedUsersSessions(TJsonReport::TGuard& g, NDrive::TEntitySession& tx, THistoryRidesContext& context, const TInstant until) {
    auto billingWalletsFilter = GetHandlerSetting<bool>("billing_wallets_filter").GetOrElse(false);
    auto queryOptions = IBaseSequentialTableImpl::TQueryOptions(Limit + 1, true);
    queryOptions.SetOrderBy({"history_event_id"});
    if (billingWalletsFilter && ResultContext.FiltredAccounts.size() < 100) {
        for (auto&& wallet : ResultContext.FiltredAccounts) {
            queryOptions.AddGenericCondition("billing_wallets", wallet);
        }
    } else {
        queryOptions.SetGenericCondition("billing_wallets", NSQL::TNot<NSQL::TEmptyString>());
    }

    if (UserIds) {
        queryOptions.SetGenericCondition("history_user_id", UserIds);
    }

    auto bSessions = Server->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing", TInstant::Zero());
    Y_ENSURE(bSessions);
    TSet<TString> availableSessionIds;
    ui64 maxEventId = Max<ui64>();
    while (availableSessionIds.size() < Limit + 1) {
        auto compiledBills = DriveApi->GetBillingManager().GetCompiledBills().GetEvents({ 0, maxEventId }, { Since, until }, tx, queryOptions);
        if (!compiledBills) {
            return false;
        }

        if (compiledBills->empty()) {
            break;
        }

        for (auto&& compiledBill : *compiledBills) {
            maxEventId = Min(maxEventId, compiledBill.GetHistoryEventId());
            bool matched = false;
            for (auto&& item : compiledBill.GetDetails().GetItems()) {
                if (ResultContext.FiltredAccounts.contains(item.GetUniqueName())) {
                    matched = true;
                    break;
                }
            }

           if (matched) {
                if (auto session = bSessions->GetSession(compiledBill.GetSessionId()); !session || session->GetClosed()) {
                    availableSessionIds.emplace(compiledBill.GetSessionId());
                }
            }
        }
    }

    g.AddEvent(NJson::TMapBuilder
        ("event", "CompiledAvailableSessions")
        ("session_ids", NJson::ToJson(availableSessionIds))
    );

    {
        auto eg = g.BuildEventGuard("InitializeHistoryRidesContext");
        auto ydbTx = BuildYdbTx<NSQL::ReadOnly>("account_session_list");
        R_ENSURE(context.InitializeSessions(MakeVector(availableSessionIds), tx, ydbTx), ConfigHttpStatus.UnknownErrorStatus, "cannot initialize HistoryRidesContext");
    }

    bool hasMore = false;
    ResultContext.ClosedSessions = context.GetSessions(until, Limit, &hasMore, SessionReportTraits & NDriveSession::ReportEmpty,
        [this](const THistoryRideObject& session) {
            return (!session.IsActive())
                && CommonFilter(session)
                && WalletFilter(ResultContext.FiltredAccounts, session);
        });

    ResultContext.HasMore = hasMore;

    return !ResultContext.ClosedSessions.empty();
}

bool TAccountSessionListProcessor::GetActualSessions(TJsonReport::TGuard& g, NDrive::TEntitySession& tx, THistoryRidesContext& context)
{
    auto bSessions = Server->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing", TInstant::Zero());
    Y_ENSURE(bSessions);
    TSet<TString> availableSessionIds;
    TVector<IEventsSession<TCarTagHistoryEvent>::TPtr> sessions;

    if (!UserIds.empty()) {
        for (auto&& userId : UserIds) {
            auto userSessions = bSessions->GetUserSessions(userId);
            sessions.insert(sessions.end(), userSessions.begin(), userSessions.end());
        }
    } else {
        sessions = bSessions->GetSessionsActual();
    }

    for (auto&& session : sessions) {
        auto billingSession = dynamic_cast<const TBillingSession*>(session.Get());
        if (!billingSession) {
            continue;
        }

        auto offer = billingSession->GetCurrentOffer();
        if (!offer) {
            continue;
        }

        if (ResultContext.FiltredAccounts.contains(offer->GetSelectedCharge())) {
            availableSessionIds.emplace(billingSession->GetSessionId());
        }
    }

    {
        auto eg = g.BuildEventGuard("InitializeHistoryRidesContext");
        auto ydbTx = BuildYdbTx<NSQL::ReadOnly>("account_session_list");
        R_ENSURE(context.InitializeSessions(MakeVector(availableSessionIds), tx, ydbTx), ConfigHttpStatus.UnknownErrorStatus, "cannot initialize HistoryRidesContext");
    }

    ResultContext.ActualSessions = context.GetSessions(TInstant::Max(), Max<ui32>(), nullptr, SessionReportTraits & NDriveSession::ReportEmpty,
        [this](const THistoryRideObject& session) {
            return session.IsActive()
                && WalletFilter(ResultContext.FiltredAccounts, session)
                && CommonFilter(session);
        });

    return !ResultContext.ActualSessions.empty();
}

bool TAccountSessionListProcessor::WalletFilter(const TSet<TString>& filtredAccounts, const THistoryRideObject& ride) {
    if (auto offer = ride.GetOffer()) {
        return filtredAccounts.contains(offer->GetSelectedCharge());
    }
    return true;
}

bool TAccountSessionListProcessor::TryUseOptiIndexSearch() {
    const auto indexResultsThreshold = GetHandlerSettingDef<ui32>("use_index_opti_search_threshold", 100);

    auto searchRequest = TSearchRequest();
    for (auto param : StrSearchParams) {
        searchRequest.AddMatchingCondition(TMatchCondition(param, true));
    }
    searchRequest.SetLimit(indexResultsThreshold + 1);

    auto entityFilter = [](const TString&){ return true; };

    const auto& matchedUserIds = DriveApi->GetUsersData()->GetMatchingIds(searchRequest, entityFilter);
    const auto& matchedCarIds = DriveApi->GetCarsData()->GetMatchingIds(searchRequest, entityFilter);

    bool res = (matchedUserIds.size() < indexResultsThreshold)
        && (matchedCarIds.size() < indexResultsThreshold);

    if (res) {
        CarIds = std::move(MakeSet(matchedCarIds));
        UserIds = std::move(MakeSet(matchedUserIds));
    }

    return res;
}

void TAccountSessionListProcessor::ResultFilter(const TVector<THistoryRideObject>& sessions, TVector<THistoryRideObject>& filtredSessions) {
    switch (ProcessorPolicy) {
        case EResultProcessorPolicy::Dummy:
            filtredSessions.insert(filtredSessions.end(), sessions.begin(), sessions.end());
            break;

        case EResultProcessorPolicy::Unique:
            for (auto&& ride : sessions) {
                if (ResultContext.ResultSessionIds.contains(ride.GetSessionId())) {
                    continue;
                }

                ResultContext.ResultSessionIds.insert(ride.GetSessionId());
                filtredSessions.emplace_back(ride);
            }
            break;

        case EResultProcessorPolicy::SessionsSubStr:
            for (auto&& ride : sessions) {
                if (auto carObject = Server->GetDriveDatabase().GetCarManager().GetObject(ride.GetObjectId())) {
                    for (auto&& subStr : StrSearchParams) {
                        if (carObject->GetNumber().find(subStr ) != TString::npos
                            || carObject->GetModel().find(subStr) != TString::npos) {
                            filtredSessions.emplace_back(ride);
                            break;
                        }
                    }
                }

                if (auto userObject = Server->GetDriveDatabase().GetUserManager().GetCachedObject(ride.GetUserId())) {
                    for (auto&& subStr : StrSearchParams) {
                        if (userObject->GetFirstName().find(subStr) != TString::npos
                            || userObject->GetLastName().find(subStr) != TString::npos) {
                            filtredSessions.emplace_back(ride);
                            break;
                        }
                    }
                }

            }
            break;
    }
}

void TAddBalanceInfoProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    R_ENSURE(Server->GetBalanceClient(), ConfigHttpStatus.UnknownErrorStatus, "no balance client");
    auto userId = Coalesce(GetString(Context->GetCgiParameters(), "user_id", false), GetString(requestData, "user_id", false));
    R_ENSURE(userId, ConfigHttpStatus.SyntaxErrorStatus, "incorrect user_id", EDriveSessionResult::IncorrectRequest);
    auto requestAccountId = GetValue<ui32>(requestData, "account_id", false);
    auto ownerPermissions = Server->GetDriveAPI()->GetUserPermissions(userId, TUserPermissionsFeatures());
    R_ENSURE(ownerPermissions, ConfigHttpStatus.UnknownErrorStatus, "no permissions");
    auto organizations = ownerPermissions->GetAdministrativeInstances(TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::B2BOrganization);
    R_ENSURE(organizations && !organizations->empty(), ConfigHttpStatus.SyntaxErrorStatus, "can't find organizations", EDriveSessionResult::IncorrectRequest);
    NJson::TJsonValue updatedAccounts(NJson::JSON_ARRAY);
    for (const auto& accountStr : *organizations) {
        ui32 accountId = 0;
        R_ENSURE(TryFromString(accountStr, accountId), ConfigHttpStatus.SyntaxErrorStatus, "incorrect account", EDriveSessionResult::IncorrectRequest);
        if (requestAccountId && accountId != *requestAccountId) {
            continue;
        }
        CheckAccountId(permissions, TAdministrativeAction::EAction::Modify, accountId);

        const NDrive::NBilling::TAccountsManager& accountsManager = Server->GetDriveAPI()->GetBillingManager().GetAccountsManager();
        NDrive::NBilling::IBillingAccount::TPtr account = accountsManager.GetAccountById(accountId);
        R_ENSURE(account, ConfigHttpStatus.SyntaxErrorStatus, "can't find account", EDriveSessionResult::IncorrectRequest);
        auto accountRecord = account->GetRecordAs<NDrive::NBilling::TLimitedAccountRecord>();
        R_ENSURE(accountRecord, ConfigHttpStatus.SyntaxErrorStatus, "incorrect account type", EDriveSessionResult::IncorrectRequest);
        if (!accountRecord->HasBalanceInfo()) {
            TBalanceClient::TClient client;
            TBalanceClient::TPerson person;
            TBalanceClient::TContract contract;

            auto clientId = Server->GetBalanceClient()->GetOrCreateContractForUid(ownerPermissions->GetUid(), client, person, contract);
            CheckBalanceResponse(clientId, Server);
            NDrive::NBilling::TBalanceClientInfo balanceInfo;
            balanceInfo.SetClientId(clientId->first);
            balanceInfo.SetPersonId(person.GetId() ? person.GetId() : contract.GetPersonId());
            balanceInfo.SetContractId(contract.GetId());
            balanceInfo.SetExternalContractId(contract.GetExternalId());
            NJson::TJsonValue accountData;
            accountData["balance_info"] = balanceInfo.ToJson();
            auto session = BuildTx<NSQL::Writable>();
            if (!account->PatchAccountData(accountData, permissions->GetUserId(), session) || !session.Commit()) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
            updatedAccounts.AppendValue(accountId);
        } else {
            R_ENSURE(false, ConfigHttpStatus.SyntaxErrorStatus, "balance info already exists", EDriveSessionResult::IncorrectRequest);
        }
    }
    g.AddReportElement("account_ids", std::move(updatedAccounts));
    g.SetCode(HTTP_OK);
}

void TCreateOrganizationLeadProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    auto tagName = GetString(Context->GetCgiParameters(), "tag_name", true);

    const auto b2bLeadTagNames = MakeSet(SplitString(GetHandlerSettingDef<TString>("backend_lead_tags", "b2b_application_form"), ","));
    auto organizationTagNames = MakeSet(SplitString(GetHandlerSettingDef<TString>("organization_tags", "organization_creator"), ","));
    organizationTagNames.insert(tagName);
    organizationTagNames.insert(b2bLeadTagNames.begin(), b2bLeadTagNames.end());

    auto session = BuildTx<NSQL::Writable>();
    TVector<TDBTag> organizationTags;
    R_ENSURE(DriveApi->GetTagsManager().GetUserTags().RestoreTagsRobust({ permissions->GetUserId() }, organizationTagNames, organizationTags, session), ConfigHttpStatus.UnknownErrorStatus, "cannot fetch user tags", session);

    TMaybe<ui64> parentId;
    TMaybe<TString> crmLeadTagId;
    TDBTag* b2bLeadTag = nullptr;
    for (auto&& tag : organizationTags) {
        if (auto tagImpl = tag.GetTagAs<TWalletAccessTag>()) {
            parentId = tagImpl->GetParentId();
        } else if (tag->GetName() == tagName) {
            crmLeadTagId = tag.GetTagId();
        } else if (b2bLeadTagNames.contains(tag->GetName())) {
            b2bLeadTag = &tag;
        }
    }

    R_ENSURE(!(parentId && !crmLeadTagId && !b2bLeadTag), ConfigHttpStatus.UnknownErrorStatus, "has organization without same application", NDrive::MakeError("has_existing_organization"), session);

    if (!crmLeadTagId) {
        TMessagesCollector errors;
        NJson::TJsonValue jsonTag = requestData;
        jsonTag["tag"] = tagName;
        auto tagPtr = IJsonSerializableTag::BuildFromJson(DriveApi->GetTagsManager(), jsonTag, &errors);
        R_ENSURE(tagPtr, ConfigHttpStatus.SyntaxErrorStatus, TStringBuilder() << "cannot_build_tag from " << jsonTag.GetStringRobust() << ": " << errors.GetStringReport());
        R_ENSURE(
                permissions->GetTagNamesByAction(TTagAction::ETagAction::Add).contains(tagPtr->GetName()),
                ConfigHttpStatus.PermissionDeniedStatus,
                "no permissions for add " << tagPtr->GetName() << " tag"
            );

        if (!b2bLeadTag) {
            auto optionalAddedTags = DriveApi->GetTagsManager().GetUserTags().AddTag(tagPtr, permissions->GetUserId(), permissions->GetUserId(), Server, session);
            R_ENSURE(optionalAddedTags, ConfigHttpStatus.UnknownErrorStatus, "cannot add tag " + tagName, session);
            R_ENSURE(optionalAddedTags->size() == 1, ConfigHttpStatus.UnknownErrorStatus, "cannot add tag " + tagName, session);
            crmLeadTagId = optionalAddedTags->front().GetTagId();
        } else {
            R_ENSURE(
                DriveApi->GetTagsManager().GetUserTags().EvolveTag(*b2bLeadTag, tagPtr, *permissions, Server, session),
                ConfigHttpStatus.UnknownErrorStatus,
                "evolve from b2b to crm failed",
                NDrive::MakeError("has_existing_organization"),
                session
            );

            crmLeadTagId = b2bLeadTag->GetTagId();
        }
    }

    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    g.AddReportElement("user_id", permissions->GetUserId());
    g.AddReportElement("tag_id", *crmLeadTagId);
    if (parentId) {
        g.AddReportElement("parent_id", *parentId);
    }
    g.SetCode(HTTP_OK);
}

void TDedicatedFleetSessionsProcessor::Parse(const TCgiParameters& cgi) {
    AccountId = GetString(cgi, "account_id", false);
    SessionId = GetString(cgi, "session_id", false);
    //Since = GetTimestamp(cgi, "since").GetRef();
    Until = GetTimestamp(cgi, "until", TInstant::Max());
    ReportActiveSessions = GetValue<bool>(cgi, "report_active_sessions", false).GetOrElse(false);
}

void TDedicatedFleetSessionsProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    R_ENSURE(AccountId || SessionId, ConfigHttpStatus.UnknownErrorStatus, "account_id or session_id must be provided ");

    auto& manager = Server->GetDriveAPI()->GetDedicatedFleetSessionManager();
    auto tx = BuildTx<NSQL::ReadOnly>();
    TCommonTagSessionManager::TConstSessions sessions;
    if (SessionId) {
        auto session = manager.GetTagSession(SessionId, tx);
        R_ENSURE(session, ConfigHttpStatus.UnknownErrorStatus, "cannot get session " << SessionId);
        sessions.push_back(*session);
    } else if (AccountId) {
        auto accountTagsNames = DriveApi->GetTagsManager().GetTagsMeta().GetRegisteredTagNames({TDedicatedFleetOfferHolderTag::Type()});
        auto optSessions = manager.GetAccountSessions(AccountId, tx);
        R_ENSURE(optSessions, ConfigHttpStatus.UnknownErrorStatus, "cannot get sessions for " << AccountId << " by tags " << JoinSeq(", ", accountTagsNames));
        for (auto&& session : *optSessions) {
            sessions.push_back(session);
        }
    }

    NJson::TJsonValue report(NJson::JSON_ARRAY);
    for (auto&& session : sessions) {
        R_ENSURE(session->Compile(), ConfigHttpStatus.UnknownErrorStatus, "cannot compile session " << SessionId);
        CheckAccountId(permissions, TAdministrativeAction::EAction::Observe, FromString<ui64>(session->GetObjectId()));

        NJson::TJsonValue sessionReport(NJson::JSON_MAP);
        sessionReport.InsertValue("events", session->GetEventsReport<TShortSessionReportBuilder>());
        sessionReport.InsertValue("compilation", session->GetReport(GetLocale(), Server, nullptr));
        report.AppendValue(sessionReport);
    }

    g.MutableReport().AddReportElement("sessions", std::move(report));
    g.SetCode(HTTP_OK);
}

void TDedicatedFleetDropSessionProcessor::Parse(const TCgiParameters& cgi) {
    SessionId = GetString(cgi, "session_id", false);
}

void TDedicatedFleetDropSessionProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    R_ENSURE(SessionId, ConfigHttpStatus.UnknownErrorStatus, "session_id must be provided ");

    auto& manager = Server->GetDriveAPI()->GetDedicatedFleetSessionManager();
    const auto& database = Server->GetDriveDatabase();
    const auto& accountTagsManager = database.GetTagsManager().GetAccountTags();

    auto tx = BuildTx<NSQL::Writable>();
    auto session = manager.GetTagSession(SessionId, tx);
    R_ENSURE(session && session->Get(), ConfigHttpStatus.UnknownErrorStatus, "cannot get session " << SessionId);
    R_ENSURE(session->Get()->Compile(), ConfigHttpStatus.UnknownErrorStatus, "cannot compile session " << SessionId);
    CheckAccountId(permissions, TAdministrativeAction::EAction::Control, FromString<ui64>(session->Get()->GetObjectId()));
    R_ENSURE(!session->Get()->GetClosed(), ConfigHttpStatus.UnknownErrorStatus, "session " << SessionId << " already closed");

    auto optTag = accountTagsManager.RestoreTag(SessionId, tx);
    R_ENSURE(optTag, ConfigHttpStatus.UnknownErrorStatus, "cannot restore tag " << SessionId);
    TDedicatedFleetOfferHolderTag::TPtr holderTagImpl = std::dynamic_pointer_cast<TDedicatedFleetOfferHolderTag>(optTag->GetData());
    R_ENSURE(holderTagImpl, ConfigHttpStatus.UnknownErrorStatus, "error while while casting to fleet holder tag implementation");
    R_ENSURE(accountTagsManager.RemoveTag(*optTag, permissions->GetUserId(), Server, tx) && tx.Commit(), ConfigHttpStatus.UnknownErrorStatus, "error while removing session " << SessionId);

    g.SetCode(HTTP_OK);
}
