#pragma once

#include <drive/backend/processors/common_app/processor.h>
#include <drive/backend/processors/sessions/base.h>

#include <drive/backend/common/localization.h>

template <class TImpl, class TConfigImpl = TEmptyConfig>
class TAccountsProcessorsBase : public TAppCommonProcessor<TImpl, TConfigImpl> {
private:
    using TBase = TAppCommonProcessor<TImpl, TConfigImpl>;

public:
    using TBase::TBase;

    TMaybe<TSet<TString>> GetAvailableAccountNames(TUserPermissions::TPtr permissions, TAdministrativeAction::EAction action) const {
        auto eg1 = NDrive::BuildEventGuard("GetAvailableAccountNames");
        Y_ENSURE_BT(permissions);
        auto instances = permissions->GetAdministrativeInstances(action, TAdministrativeAction::EEntity::Wallet, TImpl::TypeName);
        auto organizations = permissions->GetAdministrativeInstances(action, TAdministrativeAction::EEntity::B2BOrganization, TImpl::TypeName);

        if (!instances || !organizations) {
            return {};
        }

        NDrive::NBilling::TAccountParentFilter filter;
        filter.Accounts = *instances;
        for (const auto& organization : *organizations) {
            ui32 id = 0;
            if (TryFromString(organization, id)) {
                filter.ParentIds.emplace(std::move(id));
            }
        }
        filter.WithParent = (action == TAdministrativeAction::EAction::Observe || action == TAdministrativeAction::EAction::ObserveStructure || action == TAdministrativeAction::EAction::Add);
        auto accounts = TImpl::DriveApi->GetBillingManager().GetAccountsManager().GetAccountsChildren(filter);
        R_ENSURE(accounts, TImpl::ConfigHttpStatus.UnknownErrorStatus, "cannot fetch avalilable wallets");
        return MakeSet(NContainer::Keys(*accounts));
    }

    TMaybe<TSet<TString>> GetAvailableWallets(TUserPermissions::TPtr permissions, TAdministrativeAction::EAction action) const {
        auto eventLogger = NDrive::GetThreadEventLogger();
        if (eventLogger) {
            eventLogger->AddEvent(NJson::TMapBuilder
                ("event", "GetAvailableWallets")
                ("action", ToString(action))
            );
        }
        auto accounts = GetAvailableAccountNames(permissions, action);
        if (!accounts) {
            if (eventLogger) {
                eventLogger->AddEvent(NJson::TMapBuilder
                    ("event", "GetRegisteredAccounts")
                );
            }
            TSet<TString> availableWallets;
            for (auto&& description : TImpl::DriveApi->GetBillingManager().GetAccountsManager().GetRegisteredAccounts()) {
                availableWallets.insert(description.GetName());
            }
            return availableWallets;
        }
        return *accounts;
    }

    void CheckParentId(TUserPermissions::TPtr permissions, TAdministrativeAction::EAction action, ui64 parentId) const {
        TImpl::CheckOrganizationsAccess(permissions, action, { parentId });
    }

    void CheckAccountNames(TUserPermissions::TPtr permissions, TAdministrativeAction::EAction action, const TSet<TString>& accountNames) const {
        auto eg1 = NDrive::BuildEventGuard("CheckAccountNames");
        auto accounts = GetAvailableAccountNames(permissions, action);
        if (!accounts) {
            return;
        }

        TSet<TString> difference;
        SetDifference(accountNames.begin(), accountNames.end(), accounts->begin(), accounts->end(), std::inserter(difference, difference.begin()));
        R_ENSURE(
            difference.empty(),
            TImpl::ConfigHttpStatus.PermissionDeniedStatus,
            "no permissions to " << action << " account " << JoinSeq(",", difference),
            EDriveSessionResult::IncorrectRequest
        );
    }

    void CheckAccountId(TUserPermissions::TPtr permissions, TAdministrativeAction::EAction action, ui32 accountId) const {
        if (auto eventLogger = NDrive::GetThreadEventLogger()) {
            eventLogger->AddEvent(NJson::TMapBuilder
                ("event", "GetAccountById")
                ("account_id", accountId)
                ("action", ToString(action))
            );
        }
        auto account = TImpl::DriveApi->GetBillingManager().GetAccountsManager().GetAccountById(accountId);
        R_ENSURE(account, TImpl::ConfigHttpStatus.UserErrorState, TStringBuilder() << "unknown account " << accountId, EDriveSessionResult::IncorrectRequest);

        auto accounts = GetAvailableAccountNames(permissions, action);
        if (!accounts) {
            return;
        }
        if (!accounts->contains(account->GetUniqueName())) {
            CheckParentId(permissions, action, accountId);
        }
    }

    TVector<ui32> CreateAccounts(TUserPermissions::TPtr permissions, NDrive::NBilling::TAccountDescriptionRecord& description, const NJson::TJsonValue& requestData, NDrive::TEntitySession& session, ui32 count = 1) const {
        auto accountsFilter = GetAvailableWallets(permissions, TAdministrativeAction::EAction::Add);
        R_ENSURE(accountsFilter, TImpl::ConfigHttpStatus.UnknownErrorStatus, "cannot fetch avalilable wallets");

        R_ENSURE(accountsFilter->contains(description.GetName()), TImpl::ConfigHttpStatus.PermissionDeniedStatus, "no permissions to add account " + description.GetName(), EDriveSessionResult::IncorrectRequest);

        NDrive::NBilling::TAccountRecord::TPtr accountRecord = NDrive::NBilling::TAccountRecord::TFactory::Construct(description.GetDataType());
        accountRecord->SetTypeId(description.GetId());

        bool activeFlag = TImpl::template GetValue<bool>(requestData, "active_flag", false).GetOrElse(false);
        TString externalId = TBase::GetString(requestData, "external_id", false);
        auto phones = TBase::GetStrings(requestData, "phones", false);
        TString spravId = TBase::GetString(requestData, "sprav_id", false);

        accountRecord->SetActive(activeFlag);
        accountRecord->SetExternalId(externalId);
        accountRecord->SetSpravId(spravId);
        R_ENSURE((accountRecord && accountRecord->PatchMeta(requestData["account_meta"])), TImpl::ConfigHttpStatus.UserErrorState, "can't construct account", EDriveSessionResult::IncorrectRequest);

        TVector<ui32> results;
        const NDrive::NBilling::TAccountsManager& accountsManager = TImpl::DriveApi->GetBillingManager().GetAccountsManager();
        for(size_t i = 0; i < count; ++i) {
            if (!accountRecord->GetExternalId()) {
                if (phones.size()) {
                    if (i < phones.size()) {
                        accountRecord->SetExternalId(phones[i]);
                    } else {
                        accountRecord->SetExternalId({});
                    }
                }
            }
            ui32 accountId = 0;
            R_ENSURE(accountsManager.RegisterAccount(accountRecord, permissions->GetUserId(), session, &accountId) && accountId, TImpl::ConfigHttpStatus.UserErrorState, "internal error: cannot register", session);
            results.emplace_back(std::move(accountId));
        }
        return results;
    }
};

class TListAccountProcessor : public TAccountsProcessorsBase<TListAccountProcessor> {
private:
    using TBase = TAccountsProcessorsBase<TListAccountProcessor>;

private:
    static constexpr ui64 DefaultAccountUsersBatchSize = 300;

public:
    TListAccountProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "list_accounts";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;

private:
    template <class Inserter>
    void FillAccountUsers(const TSet<ui64>& accountIds, const NDrive::NBilling::TAccountsManager& accountsManager, NDrive::TEntitySession& session, Inserter inserter);

    NJson::TJsonValue GetUserReport(const TString& userId, const TDriveUserData* gUser, const TMap<TString, TMaybe<ui32>>& usersDebts, const ui32 debtThreshold);
};

class TUserAccountsProcessor : public TAccountsProcessorsBase<TUserAccountsProcessor> {
private:
    using TBase = TAccountsProcessorsBase<TUserAccountsProcessor>;

public:
    TUserAccountsProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "list_user_accounts";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TUpdateCommonAccountProcessor : public TAccountsProcessorsBase<TUpdateCommonAccountProcessor> {
private:
    using TBase = TAccountsProcessorsBase<TUpdateCommonAccountProcessor>;

public:
    TUpdateCommonAccountProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "update_common_account";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
    static NDrive::TScheme GetRequestDataScheme(const IServerBase* server, const TCgiParameters& schemeCgi = {});
};


class TUpdateAccountDataProcessor : public TAccountsProcessorsBase<TUpdateAccountDataProcessor, TPackProcessorConfig> {
private:
    using TBase = TAccountsProcessorsBase<TUpdateAccountDataProcessor, TPackProcessorConfig>;

public:
    TUpdateAccountDataProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "update_account_data";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};


class TGetCommonAccountsProcessor : public TAccountsProcessorsBase<TGetCommonAccountsProcessor> {
private:
    using TBase = TAccountsProcessorsBase<TGetCommonAccountsProcessor>;

public:
    TGetCommonAccountsProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "get_common_accounts";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TLinkAccountProcessor : public TAccountsProcessorsBase<TLinkAccountProcessor, TPackProcessorConfig> {
private:
    using TBase = TAccountsProcessorsBase<TLinkAccountProcessor, TPackProcessorConfig>;

public:
    TLinkAccountProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "link_account";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;

private:
    void LinkSingleAccount(const TString& userId, ui32 accountId, TUserPermissions::TPtr permissions, const TString& promocode, NDrive::TEntitySession& session);
};

class TSearchAccountProcessor : public TAccountsProcessorsBase<TSearchAccountProcessor, TEmptyConfig> {
private:
    using TBase = TAccountsProcessorsBase<TSearchAccountProcessor, TEmptyConfig>;

public:
    TSearchAccountProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "search_account";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TPromocodeLinkAccountListProcessor : public TAccountsProcessorsBase<TPromocodeLinkAccountListProcessor, TEmptyConfig> {
private:
    using TBase = TAccountsProcessorsBase<TPromocodeLinkAccountListProcessor, TEmptyConfig>;

public:
    TPromocodeLinkAccountListProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "promocode_link_account_list";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TGetPromocodeLinkAccountProcessor : public TAccountsProcessorsBase<TGetPromocodeLinkAccountProcessor, TEmptyConfig> {
private:
    using TBase = TAccountsProcessorsBase<TGetPromocodeLinkAccountProcessor, TEmptyConfig>;

public:
    TGetPromocodeLinkAccountProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "promocode_link_account";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TRemoveAccountProcessor : public TAccountsProcessorsBase<TRemoveAccountProcessor> {
private:
    using TBase = TAccountsProcessorsBase<TRemoveAccountProcessor>;

public:
    TRemoveAccountProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "remove_account";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};


class TRemoveAccountDescriptionProcessor : public TAccountsProcessorsBase<TRemoveAccountDescriptionProcessor> {
private:
    using TBase = TAccountsProcessorsBase<TRemoveAccountDescriptionProcessor>;

public:
    TRemoveAccountDescriptionProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "remove_account_description";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TActivateAccountProcessor : public TAccountsProcessorsBase<TActivateAccountProcessor, TPackProcessorConfig> {
private:
    using TBase = TAccountsProcessorsBase<TActivateAccountProcessor, TPackProcessorConfig>;

public:
    TActivateAccountProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "activate_account";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TCreateAccountProcessor : public TAccountsProcessorsBase<TCreateAccountProcessor> {
private:
    using TBase = TAccountsProcessorsBase<TCreateAccountProcessor>;

public:
    TCreateAccountProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "create_account";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
    static NDrive::TScheme GetRequestDataScheme(const IServerBase* server, const TCgiParameters& schemeCgi = {});
};

class TGetBalancePersonProcessor : public TAccountsProcessorsBase<TGetBalancePersonProcessor> {
private:
    using TBase = TAccountsProcessorsBase<TGetBalancePersonProcessor>;

public:
    TGetBalancePersonProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "get_balance_person";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
    static NDrive::TScheme GetRequestDataScheme(const IServerBase* server, const TCgiParameters& schemeCgi = {});
};

class TUpdateBalancePersonProcessor : public TAccountsProcessorsBase<TUpdateBalancePersonProcessor> {
private:
    using TBase = TAccountsProcessorsBase<TUpdateBalancePersonProcessor>;

public:
    TUpdateBalancePersonProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "update_balance_person";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TGetBalancePaymentLinkProcessor : public TAccountsProcessorsBase<TGetBalancePaymentLinkProcessor> {
private:
    using TBase = TAccountsProcessorsBase<TGetBalancePaymentLinkProcessor>;

public:
    TGetBalancePaymentLinkProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "get_balance_payment_link";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TUpdateOrganizationBalanceProcessor : public TAccountsProcessorsBase<TUpdateOrganizationBalanceProcessor> {
private:
    using TBase = TAccountsProcessorsBase<TUpdateOrganizationBalanceProcessor>;

public:
    TUpdateOrganizationBalanceProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "update_organization_balance";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TSetOrganizationOverdraftProcessor : public TAccountsProcessorsBase<TSetOrganizationOverdraftProcessor> {
private:
    using TBase = TAccountsProcessorsBase<TSetOrganizationOverdraftProcessor>;

public:
    TSetOrganizationOverdraftProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "set_organization_overdraft";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TGetBalanceDocumentsProcessor : public TAccountsProcessorsBase<TGetBalanceDocumentsProcessor> {
private:
    using TBase = TAccountsProcessorsBase<TGetBalanceDocumentsProcessor>;

public:
    TGetBalanceDocumentsProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "get_balance_documents";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TDownloadBalanceDocumentsProcessor : public TAccountsProcessorsBase<TDownloadBalanceDocumentsProcessor> {
private:
    using TBase = TAccountsProcessorsBase<TDownloadBalanceDocumentsProcessor>;

public:
    TDownloadBalanceDocumentsProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "download_balance_documents";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TAccountHistoryProcessor : public TAccountsProcessorsBase<TAccountHistoryProcessor> {
private:
    using TBase = TAccountsProcessorsBase<TAccountHistoryProcessor>;

public:
    TAccountHistoryProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "account_history";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TAccountDescriptionHistoryProcessor : public TAccountsProcessorsBase<TAccountDescriptionHistoryProcessor> {
private:
    using TBase = TAccountsProcessorsBase<TAccountDescriptionHistoryProcessor>;

public:
    TAccountDescriptionHistoryProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "account_description_history";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;

protected:
    void Parse(const TCgiParameters& cgi) override;

private:
    R_OPTIONAL(TString, Name);
    R_OPTIONAL(ui64, Id);
    R_OPTIONAL(TTimestamp, Since);
    R_OPTIONAL(TString, HistoryComment);
    R_READONLY(ui64, Limit, 0);
    R_READONLY(ui64, Offset, 0);
};

class TAccountSessionItemProcessor
    : public TAccountsProcessorsBase<TAccountSessionItemProcessor>
    , public TSessionProcessorTraits<TAccountSessionItemProcessor>
{
private:
    using TBase = TAccountsProcessorsBase<TAccountSessionItemProcessor>;

public:
    TAccountSessionItemProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server)
        , TSessionProcessorTraits<TAccountSessionItemProcessor>(context, server)
    {
    }

    static TString GetTypeName() {
        return "account_session_item";
    }

protected:
    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TAccountSessionListProcessor
    : public TAccountsProcessorsBase<TAccountSessionListProcessor>
    , public TSessionProcessorTraits<TAccountSessionListProcessor>
{
private:
    using TBase = TAccountsProcessorsBase<TAccountSessionListProcessor>;

    using TProcessorTraits = ui32;
    enum EProcessorTraits: TProcessorTraits {
        GetSessionsByCarIds = 1 << 0,
        GetSessionsByUserIds = 1 << 1,
        FilterByInputString = 1 << 2,
        FilterByUserIds = 1 << 3,
        FilterWithIndexSearch = 1 << 4,
    };

    enum class EResultProcessorPolicy {
        Dummy = 0,
        Unique = 1,
        SessionsSubStr = 2,
    };

    class SessionsResultContext {
    public:
        TSet<TString> ResultSessionIds;
        TVector<THistoryRideObject> ActualSessions;
        TVector<THistoryRideObject> ClosedSessions;
        TSet<TString> FiltredAccounts;

        bool HasMore = false;

        void ClearCurrentResults() {
            ActualSessions.clear();
            ClosedSessions.clear();
        }
    };

public:
    TAccountSessionListProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server)
        , TSessionProcessorTraits<TAccountSessionListProcessor>(context, server)
    {
    }

    static TString GetTypeName() {
        return "account_session_list";
    }

protected:
    void Parse(const TCgiParameters& cgi) override;
    void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;

    bool GetCarsSessions(TJsonReport::TGuard& g, NDrive::TEntitySession& tx, THistoryRidesContext& context);
    bool GetClosedUsersSessions(TJsonReport::TGuard& g, NDrive::TEntitySession& tx, THistoryRidesContext& context, const TInstant until);
    bool GetActualSessions(TJsonReport::TGuard& g, NDrive::TEntitySession& tx, THistoryRidesContext& context);

    bool WalletFilter(const TSet<TString>& filtredAccounts, const THistoryRideObject& ride);
    bool TryUseOptiIndexSearch();

    void ResultFilter(const TVector<THistoryRideObject>& sessions, TVector<THistoryRideObject>& filtredSessions);

protected:
    SessionsResultContext ResultContext;
    std::function<bool(const THistoryRideObject& ride)> CommonFilter;

private:
    R_READONLY(EResultProcessorPolicy, ProcessorPolicy, EResultProcessorPolicy::Dummy);
    R_READONLY(TProcessorTraits, SessionGetterTraits, 0);
    R_READONLY(NDriveSession::TReportTraits, SessionReportTraits, 0);
    R_READONLY(NDeviceReport::TReportTraits, DeviceReportTraits, 0);
    R_READONLY(ui64, Limit, 15);
    R_READONLY(TInstant, Start);
    R_READONLY(TInstant, Since);
    R_READONLY(TInstant, Until);
    R_READONLY(bool, AddActual, false);
    R_READONLY(TSet<TString>, CarIds);
    R_READONLY(TSet<TString>, UserIds);
    R_READONLY(TVector<TString>, StrSearchParams);
    R_READONLY(TVector<TString>, AccountNames);
    R_READONLY(TVector<ui64>, ParentIds);
};

class TAddBalanceInfoProcessor : public TAccountsProcessorsBase<TAddBalanceInfoProcessor> {
private:
    using TBase = TAccountsProcessorsBase<TAddBalanceInfoProcessor>;

public:
    TAddBalanceInfoProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "add_exists_balance_info";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TCreateOrganizationLeadProcessor : public TAppCommonProcessor<TCreateOrganizationLeadProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TCreateOrganizationLeadProcessor, TEmptyConfig>;

public:
    TCreateOrganizationLeadProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "create_lead";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TDedicatedFleetSessionsProcessor : public TAccountsProcessorsBase<TDedicatedFleetSessionsProcessor> {
private:
    using TBase = TAccountsProcessorsBase<TDedicatedFleetSessionsProcessor>;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "dedicated_fleet_sessions";
    }

protected:
    virtual void Parse(const TCgiParameters& cgi) override;
    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;

private:
    R_READONLY(TString, AccountId);
    R_READONLY(TString, SessionId);
    R_READONLY(TInstant, Since);
    R_READONLY(TInstant, Until, TInstant::Max());
    R_READONLY(bool, ReportActiveSessions, false);
};

class TDedicatedFleetDropSessionProcessor : public TAccountsProcessorsBase<TDedicatedFleetDropSessionProcessor> {
private:
    using TBase = TAccountsProcessorsBase<TDedicatedFleetDropSessionProcessor>;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "dedicated_fleet_drop_session";
    }

protected:
    virtual void Parse(const TCgiParameters& cgi) override;
    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;

private:
    R_READONLY(TString, SessionId);
};
