#include "processor.h"

#include <drive/backend/data/area_tags.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/model.h>
#include <drive/backend/data/operation_area.h>
#include <drive/backend/data/leasing/acl/acl.h>
#include <drive/backend/data/leasing/company.h>
#include <drive/backend/database/config.h>
#include <drive/backend/drivematics/zone/config/zone_config.h>
#include <drive/backend/drivematics/zone/zone.h>
#include <drive/backend/offers/manager.h>
#include <drive/backend/offers/actions/abstract.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/sessions/manager/billing.h>

#include <drive/library/cpp/geocoder/api/client.h>
#include <drive/library/cpp/geojson/value.h>

#include <rtline/library/json/replace.h>
#include <rtline/util/algorithm/container.h>

#include <util/string/split.h>

template <class T>
NJson::TJsonValue TAreaReportCommonProcessor<T>::GetLocalized(NJson::TJsonValue&& value) const {
    auto server = TBase::Server;
    auto localization = server ? server->GetLocalization() : nullptr;
    if (localization) {
        auto locale = TBase::GetLocale();
        localization->ApplyResourcesForJson(value, locale);
    }
    return value;
}

template <class T>
NJson::TJsonValue TAreaReportCommonProcessor<T>::GetLocalizedJsonSetting(TStringBuf key) const {
    return GetLocalized(TBase::GetSettings().GetJsonValue(key));
}

template <class T>
NJson::TJsonValue TAreaReportCommonProcessor<T>::GetLocalizedJsonSetting(TConstArrayRef<TString> keys) const {
    for (auto&& key : keys) {
        auto result = GetLocalizedJsonSetting(key);
        if (result.IsDefined()) {
            return result;
        }
    }
    return {};
}

template <class T>
NJson::TJsonValue TAreaReportCommonProcessor<T>::GetLocalizedJsonSetting(TConstArrayRef<TString> pathes, const TString& key, TString* resultPath) const {
    return GetLocalized(TBase::GetSettings().GetJsonValue(pathes, key, resultPath));
}

template <class T>
NJson::TJsonValue TAreaReportCommonProcessor<T>::GetLocalizedStyleReport(TAreasDB::EAreaStyle style, TStringBuf suffix, TString* resultPath) const {
    const TVector<TString> pathes = {
        TStringBuilder() << "client.styles." << suffix << ".areas",
        "client.styles.areas",
    };
    return GetLocalized(TBase::GetSettings().GetJsonValue(pathes, ToString(style), resultPath));
}

template <class T>
NJson::TJsonValue TAreaReportCommonProcessor<T>::GetLocalizedStyleLegendReport(TStringBuf suffix) const {
    NJson::TJsonValue result(NJson::JSON_MAP);
    for (auto&& style : GetEnumAllValues<TAreasDB::EAreaStyle>()) {
        auto styleJson = GetLocalizedStyleReport(style, suffix);
        if (styleJson.IsDefined()) {
            result.InsertValue(ToString(style), std::move(styleJson));
        }
    }
    return result;
}

void TAreasRemoveProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Remove, TAdministrativeAction::EEntity::Areas);
    R_ENSURE(requestData["areas"].IsArray(), ConfigHttpStatus.SyntaxErrorStatus, "'areas' is not array");

    TSet<TString> areas;
    for (auto&& i : requestData["areas"].GetArraySafe()) {
        TArea area;
        R_ENSURE(i["area_id"].IsString(), ConfigHttpStatus.SyntaxErrorStatus, "incorrect area description");
        areas.emplace(i["area_id"].GetString());
    }
    NDrive::TEntitySession session = BuildTx<NSQL::Writable>();
    if (!Server->GetDriveAPI()->GetAreasDB()->RemoveObject(areas, permissions->GetUserId(), session)) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    g.SetCode(HTTP_OK);
}

void TAreasUpsertProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Areas);

    TVector<TArea> areas;
    NGeoJson::TFeatureCollection featureCollection;
    if (requestData["geo_areas"].IsArray()) {
        const TString& uniqueId = Context->GetCgiParameters().Get("unique_id");
        const TSet<TString> areaTags = MakeSet(StringSplitter(Context->GetCgiParameters().Get("area_tag")).SplitBySet(", ").SkipEmpty().ToList<TString>());
        const TSet<TString> holeTags = MakeSet(StringSplitter(Context->GetCgiParameters().Get("hole_tag")).SplitBySet(", ").SkipEmpty().ToList<TString>());
        R_ENSURE(uniqueId && !areaTags.empty() && !holeTags.empty(), ConfigHttpStatus.SyntaxErrorStatus, "cgi-s 'area_tag', 'hole_tag', 'unique_id' have to been");
        ui32 idxMulti = 0;
        for (auto&& multiPolygon : requestData["geo_areas"].GetArraySafe()) {
            ++idxMulti;
            ui32 idxLine = 0;
            R_ENSURE(multiPolygon.IsArray(), ConfigHttpStatus.SyntaxErrorStatus, "multi polyline is incorrect array");
            for (auto&& polygon : multiPolygon.GetArraySafe()) {
                TArea area(uniqueId + "/" + ToString(idxMulti) + "/" + ToString(++idxLine));
                area.SetTags((idxLine == 1) ? areaTags : holeTags);
                TVector<TGeoCoord> coords;
                R_ENSURE(polygon.IsArray(), ConfigHttpStatus.SyntaxErrorStatus, "polyline is incorrect array");
                for (auto&& coord : polygon.GetArraySafe()) {
                    R_ENSURE(coord.IsArray() && coord.GetArraySafe().size() == 2, ConfigHttpStatus.SyntaxErrorStatus, "polyline coord is incorrect array");
                    coords.emplace_back(TGeoCoord(coord.GetArraySafe()[0].GetDouble(), coord.GetArraySafe()[1].GetDouble()));
                }
                area.SetCoords(std::move(coords));
                if (coords.size()) {
                    areas.emplace_back(std::move(area));
                }
            }
        }

    } else if (requestData["areas"].IsArray()) {
        for (auto&& i : requestData["areas"].GetArraySafe()) {
            TArea area;
            R_ENSURE(area.DeserializeFromJson(i), ConfigHttpStatus.SyntaxErrorStatus, "incorrect area record");
            areas.emplace_back(area);
        }
    } else if (NJson::TryFromJson(requestData, featureCollection)) {
        TString basename;
        if (!basename) {
            basename = featureCollection.GetMetadata()["name"].GetString();
        }
        if (!basename) {
            basename = "geojson";
        }
        for (auto&& feature : featureCollection.GetFeatures()) {
            auto geometry = feature.GetGeometry();
            R_ENSURE(geometry, ConfigHttpStatus.SyntaxErrorStatus, "feature " << feature.GetId() << " must have geometry");
            R_ENSURE(geometry->GetType() == NGeoJson::TGeometry::Polygon, ConfigHttpStatus.SyntaxErrorStatus, "feature " << feature.GetId() << " must be a polygon");

            TArea area;
            area.SetIdentifier(basename + '_' + ToString(feature.GetId()));
            area.SetCoords(std::move(geometry->GetCoordinates()));
            area.SetTitle(feature.GetProperties()["description"].GetString());
            areas.push_back(std::move(area));
        }
    } else {
        R_ENSURE(false, ConfigHttpStatus.SyntaxErrorStatus, "no arrays with areas");
    }

    NDrive::TEntitySession session = BuildTx<NSQL::Writable>();
    for (auto&& i : areas) {
        if (!Server->GetDriveAPI()->GetAreasDB()->UpsertObject(i, permissions->GetUserId(), session)) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }
    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    g.SetCode(HTTP_OK);
}

void TAreasHistoryProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Areas);

    TVector<TAtomicSharedPtr<TObjectEvent<TArea>>> result;
    R_ENSURE(Server->GetDriveAPI()->GetAreasDB()->GetHistory(TInstant::Zero(), result, Context->GetRequestStartTime()), ConfigHttpStatus.UnknownErrorStatus, "cannot_fetch_history");
    NJson::TJsonValue areasHistory(NJson::JSON_ARRAY);
    for (auto it = result.rbegin(); it != result.rend(); ++it) {
        areasHistory.AppendValue((*it)->SerializeToJson());
    }
    g.MutableReport().AddReportElement("areas_history", std::move(areasHistory));

    g.SetCode(HTTP_OK);
}

void TAreasInfoProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Areas);

    const TSet<TString> ids = MakeSet(GetStrings(Context->GetCgiParameters(), "ids", false));

    auto session = BuildTx<NSQL::ReadOnly>();

    TConstAreasSnapshot::TPtr areas = Server->GetDriveAPI()->GetAreasDB()->GetSnapshot(Context->GetRequestStartTime());

    auto optionalAreaPropositions = DriveApi->GetAreasDB()->GetPropositions()->Get(session);
    R_ENSURE(optionalAreaPropositions, {}, "cannot get propositions", session);
    auto areaPropositions = *optionalAreaPropositions;

    NJson::TJsonValue areasJson(NJson::JSON_ARRAY);
    bool checkVisibility = GetHandlerSettingDef<bool>("check_visibility", false);
    for (auto&& fullAreaInfo : Reversed(*areas)) {
        if (ids.size() && !ids.contains(fullAreaInfo.GetArea().GetIdentifier())) {
            continue;
        }
        if (!checkVisibility
            || permissions->GetVisibility(fullAreaInfo, NEntityTagsManager::EEntityType::Area) == TUserPermissions::EVisibility::Visible) {
            areasJson.AppendValue(fullAreaInfo.SerializeToReport());
        }
    }
    NJson::TJsonValue propositionsJson = NJson::JSON_ARRAY;
    TSet<TString> users;
    for (auto&& i : areaPropositions) {
        if (ids.size() && !ids.contains(i.second.GetIdentifier())) {
            continue;
        }
        propositionsJson.AppendValue(i.second.BuildJsonReport());
        users.emplace(i.second.GetPropositionAuthor());
    }

    auto userData = DriveApi->GetUsersData()->FetchInfo(users);
    NJson::TJsonValue usersJson = NJson::JSON_MAP;
    for (auto&& [_, i] : userData) {
        usersJson.InsertValue(i.GetUserId(), i.GetReport(permissions->GetUserReportTraits()));
    }
    g.MutableReport().AddReportElement("areas", std::move(areasJson));
    g.MutableReport().AddReportElement("propositions", std::move(propositionsJson));
    g.MutableReport().AddReportElement("users", std::move(usersJson));

    g.SetCode(HTTP_OK);
}

void TParkingAreasInfoProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    auto coordinate = GetValue<TGeoCoord>(cgi, "coordinate", false);
    auto requestedStyle = GetValue<TAreasDB::EAreaStyle>(cgi, "style", false);
    auto useStyles = GetValue<bool>(cgi, "use_styles", false).GetOrElse(false);

    const auto& areaDb = DriveApi->GetAreasDB();
    auto enableSessionLookup = GetHandlerSetting<bool>("enable_session_lookup").GetOrElse(false);
    auto localization = Server->GetLocalization();
    auto locale = GetLocale();
    auto language = enum_cast<ELanguage>(locale);
    auto geocoded = coordinate && DriveApi->HasGeocoderClient() ? DriveApi->GetGeocoderClient().Decode(*coordinate, language) : NThreading::Uninitialized;
    auto snapshot = coordinate ? Yensured(areaDb)->GetSnapshot(*coordinate) : Yensured(areaDb)->GetSnapshot();

    auto tx = BuildTx<NSQL::ReadOnly | NSQL::Deferred>();

    if (DriveApi->GetConfig().GetZoneConfig()->GetEnable()) {
        auto optionalObject = DriveApi->GetTagsManager().GetUserTags().GetCachedObject(permissions->GetUserId());
        if (optionalObject && static_cast<bool>(optionalObject->GetFirstTagByClass<NDrivematics::TUserOrganizationAffiliationTag>())) {
            const auto& zoneDb = DriveApi->GetZoneDB();

            auto acl = NDrivematics::TACLTag::GetACLTag(permissions, tx, *Server);
            auto aclCompany = acl->GetCompanyTagDescription(permissions->GetUserId(), *Server);

            TZoneStorage::TOptions options;
            options.SetForMobile(true);
            NDrive::TAreaIds visibleAreaIds;
            R_ENSURE(zoneDb->GetAreasIds(aclCompany->GetEntityObjects(TAdministrativeAction::EEntity::Zone, permissions), visibleAreaIds, &options), HTTP_INTERNAL_SERVER_ERROR, "can't get areas", tx);
            TFullAreaInfos areas;
            auto action = [&](const TFullAreaInfo& area) -> bool {
                areas.push_back(area);
                return true;
            };
            R_ENSURE(areaDb->ProcessAreaIds(visibleAreaIds, action), HTTP_INTERNAL_SERVER_ERROR, "can't filtered areas", tx);
            auto newSnapshot = MakeAtomicShared<TConstAreasSnapshot>(std::move(areas));
            std::swap(snapshot, newSnapshot);
        }
    }

    ICommonOffer::TPtr commonOffer;
    TString offerId = GetString(cgi, "offer_id", false);
    TString objectId;
    if (offerId && !commonOffer && enableSessionLookup) {
        TEventsGuard eg(g.MutableReport(), "session_lookup");
        auto sessionId = offerId;
        auto optionalSession = DriveApi->GetSessionManager().GetSession(sessionId, tx);
        R_ENSURE(optionalSession, {}, "cannot GetSession " << sessionId, tx);
        auto session = *optionalSession;
        auto billingSession = std::dynamic_pointer_cast<const TBillingSession>(session);
        commonOffer = billingSession ? billingSession->GetCurrentOffer() : nullptr;
    }
    if (offerId && !commonOffer) {
        TEventsGuard eg(g.MutableReport(), "restore_offer");
        auto asyncOffer = Server->GetOffersStorage()->RestoreOffer(offerId, tx);
        R_ENSURE(asyncOffer.Initialized(), {}, "cannot RestoreOffer " << offerId, tx);
        R_ENSURE(asyncOffer.Wait(Context->GetRequestDeadline()), ConfigHttpStatus.TimeoutStatus, "RestoreOffer timeout");
        commonOffer = asyncOffer.GetValue();
        R_ENSURE(commonOffer, ConfigHttpStatus.EmptySetStatus, "cannot find offer " << offerId);
    }
    if (!commonOffer) {
        TEventsGuard eg(g.MutableReport(), "restore_session");
        auto current = GetCurrentUserSession(permissions);
        auto bSession = dynamic_cast<const TBillingSession*>(current.Get());
        if (bSession) {
            commonOffer = bSession->GetCurrentOffer();
        }
        if (current && !current->GetClosed()) {
            objectId = current->GetObjectId();
        }
    }

    ReqCheckCondition(!commonOffer || commonOffer->GetUserId() == permissions->GetUserId() || CheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::User, commonOffer->GetUserId()), ConfigHttpStatus.PermissionDeniedStatus, EDriveLocalizationCodes::NoPermissions);

    if (!objectId) {
        objectId = GetString(cgi, "car_id", false);
        auto carNumber = GetString(cgi, "car_number", false);
        if (carNumber) {
            objectId = DriveApi->GetCarIdByNumber(carNumber);
        }
    }
    IOffer::TPtr offer = std::dynamic_pointer_cast<IOffer>(commonOffer);
    if (!objectId && !!offer) {
        objectId = offer->GetObjectId();
    }
    g.AddEvent(NJson::TMapBuilder
        ("event", "input_finalized")
        ("coordinate", NJson::ToJson(coordinate))
        ("object_id", objectId)
        ("offer_id", offerId)
        ("offer", NJson::ToJson(commonOffer))
    );

    const auto action = commonOffer ? Server->GetDriveAPI()->GetRolesManager()->GetActionsDB().GetObject(commonOffer->GetBehaviourConstructorId()) : Nothing();
    const auto constructor = action ? action->GetAs<IOfferBuilderAction>() : nullptr;
    const auto deliveryAreaConstructor = action ? action->GetAs<NDrive::IDeliveryAreaFilter>() : nullptr;
    const auto deliveryAreaFilter = deliveryAreaConstructor ? deliveryAreaConstructor->GetDeliveryAreaFilter() : Nothing();

    TSet<EDropAbility> abilities;
    TMaybe<EDropOfferPolicy> feePolicy;
    TMaybe<ui32> fee;
    bool allowsRiding = false;

    NJson::TJsonValue allowAreas(NJson::JSON_ARRAY);
    NJson::TJsonValue deliveryAreas(NJson::JSON_ARRAY);
    NJson::TJsonValue denyAreas(NJson::JSON_ARRAY);
    NJson::TJsonValue ridingAreas(NJson::JSON_ARRAY);
    NJson::TJsonValue areas(NJson::JSON_ARRAY);
    TString visualType;
    if (constructor) {
        const auto& visual = constructor->GetVisual();
        if (visual) {
            visualType = visual->GetOfferVisualType();
        }
    }
    for (auto&& i : Reversed(*snapshot)) {
        TEventsGuard eg(g.MutableReport(), "build_report_" + i.GetId());
        TCarLocationContext context = TCarLocationContext::BuildByArea(objectId, i.GetArea(), *Server);
        TCarLocationFeatures features = context.GetCarAreaFeatures(false, offer.Get(), Server);

        NJson::TJsonValue* reportArea = nullptr;
        const bool allowDelivery = deliveryAreaFilter ? deliveryAreaFilter->FilterWeak(i.GetArea().GetTags()) : false;
        const EDropAbility dropAbility = features.GetAllowDropDef(EDropAbility::NotAllow);
        const bool hasRealFees = features.HasFeeInfo() && features.GetFeeInfoUnsafe().GetPolicy() != EDropOfferPolicy::FeesMinute && features.GetFeeInfoUnsafe().GetFee();
        const TAreasDB::EAreaStyle style = TAreasDB::GetStyle(dropAbility, features.GetAllowRiding(), allowDelivery, hasRealFees);
        if (requestedStyle && requestedStyle != style) {
            continue;
        }
        abilities.insert(dropAbility);
        allowsRiding |= features.GetAllowRiding();
        switch (dropAbility) {
            case EDropAbility::Allow:
                reportArea = &allowAreas;
                break;
            case EDropAbility::Deny:
            case EDropAbility::DenyIncorrectData:
                reportArea = &denyAreas;
                break;
            default:
                break;
        }
        if (!reportArea) {
            if (features.GetAllowRiding()) {
                reportArea = &ridingAreas;
            } else if (allowDelivery) {
                reportArea = &deliveryAreas;
            } else {
                continue;
            }
        }
        if (useStyles) {
            reportArea = &areas;
        }
        NJson::TJsonValue& areaDropInfo = reportArea->AppendValue(i.GetArea().SerializeToPublicReport());
        if (dropAbility == EDropAbility::Allow && hasRealFees) {
            fee = ICommonOffer::RoundPrice(features.GetFeeInfoRef().GetFee(), 100);
            feePolicy = features.GetFeeInfoRef().GetPolicy();
            areaDropInfo.InsertValue("fee", *fee);
            areaDropInfo.InsertValue("fee_policy", ::ToString(*feePolicy));
        }
        areaDropInfo.InsertValue("area_style", GetLocalizedStyleReport(style, visualType));
    }
    if (coordinate) {
        auto eg = g.BuildEventGuard("poi");
        TVector<TArea> areas;
        for (auto&& i : *snapshot) {
            areas.push_back(i.GetArea());
        }
        g.AddReportElement("poi", NDrive::GetPoiReport(areas, NDrive::DefaultPoiLocationTags, {}, locale, *Server));
    }
    if (coordinate) {
        NJson::TJsonValue ablties = NJson::JSON_ARRAY;
        if (abilities.contains(EDropAbility::Allow) && !abilities.contains(EDropAbility::Deny)) {
            ablties.AppendValue(GetLocalizedJsonSetting("client.abilities.allow_drop"));
        } else if (allowsRiding) {
            ablties.AppendValue(GetLocalizedJsonSetting("client.abilities.allow_parking"));
            ablties.AppendValue(GetLocalizedJsonSetting("client.abilities.deny_drop"));
        } else {
            ablties.AppendValue(GetLocalizedJsonSetting("client.abilities.deny_riding"));
        }
        NJson::TJsonValue notifications = NJson::JSON_ARRAY;
        if (fee) {
            auto feeString = localization ? localization->FormatPrice(locale, *fee) : ToString(fee);
            auto feePolicyString = ToString(feePolicy.GetOrElse(EDropOfferPolicy::FeesMax));
            auto notificationKey = TStringBuilder() << "client.notification." << feePolicyString << "_drop";
            auto notification = GetLocalizedJsonSetting(notificationKey);
            NJson::ReplaceAll(notification, "_FeeValue_", feeString);
            notifications.AppendValue(std::move(notification));
        }
        g.AddReportElement("abilities", std::move(ablties));
        g.AddReportElement("notifications", std::move(notifications));
    } else {
        auto eg = g.BuildEventGuard("build_report_meta");
        g.AddReportElement("legend", GetLocalizedStyleLegendReport(visualType));
        if (useStyles) {
            g.AddReportElement("areas", std::move(areas));
        } else {
            g.AddReportElement("allow_areas", std::move(allowAreas));
            g.AddReportElement("delivery_areas", std::move(deliveryAreas));
            g.AddReportElement("deny_areas", std::move(denyAreas));
            g.AddReportElement("riding_areas", std::move(ridingAreas));
        }
    }
    if (geocoded.Initialized()) {
        auto eg = g.BuildEventGuard("wait_geocoded_location");
        if (!geocoded.Wait(Context->GetRequestDeadline())) {
            g.AddEvent("wait_timeout");
        }
        if (geocoded.HasValue()) {
            g.AddReportElement("geocoded_location", geocoded.GetValue().Title);
        } else {
            g.AddEvent(NJson::TMapBuilder
                ("event", "geocoded_location_exception")
                ("exception", NThreading::GetExceptionInfo(geocoded))
            );
        }
    }

    g.SetCode(HTTP_OK);
}

void TAreasOperationProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    Y_UNUSED(requestData);
    const TCgiParameters& cgi = Context->GetCgiParameters();
    auto coordinate = GetValue<TGeoCoord>(cgi, "coordinate", false);
    auto isCompactReport = GetValue<bool>(cgi, "is_compact_report", false).GetOrElse(false);
    auto useStyles = GetValue<bool>(cgi, "use_styles", false).GetOrElse(false);

    const auto& areaDb = DriveApi->GetAreasDB();
    auto localization = Server->GetLocalization();
    auto locale = GetLocale();
    auto language = enum_cast<ELanguage>(locale);
    auto geocoded = coordinate && DriveApi->HasGeocoderClient() ? DriveApi->GetGeocoderClient().Decode(*coordinate, language) : NThreading::Uninitialized;
    auto snapshot = coordinate ? Yensured(areaDb)->GetSnapshot(*coordinate) : Yensured(areaDb)->GetSnapshot();
    auto observableTags = permissions->GetTagNamesByAction(TTagAction::ETagAction::Observe);
    auto tags = DriveApi->GetTagsManager().GetTagsMeta().GetTagsByType(TOperationAreaTag::Type());

    auto defaultOfferTagsString = GetHandlerSetting<TString>("default_offer_tags");
    auto defaultOfferTags = defaultOfferTagsString ? MakeSet<TString>(StringSplitter(*defaultOfferTagsString).Split(',').SkipEmpty()) : TSet<TString>{
        "standart_offer"
    };

    TVector<const TOperationAreaTag::TDescription*> descriptions;
    for (auto&& description : tags) {
        if (!description) {
            continue;
        }
        if (!observableTags.contains(description->GetName())) {
            continue;
        }
        auto operationAreaTagDescription = dynamic_cast<const TOperationAreaTag::TDescription*>(description.Get());
        if (!operationAreaTagDescription) {
            continue;
        }
        descriptions.push_back(operationAreaTagDescription);
    }
    std::sort(descriptions.begin(), descriptions.end(), [](const TOperationAreaTag::TDescription* left, const TOperationAreaTag::TDescription* right) {
        return (left ? left->GetOrder() : 0) < (right ? right->GetOrder() : 0);
    });

    TMap<TString, ui32> idxByAreas;
    TMap<TString, ui32> idxByStyle;
    NJson::TJsonValue records(NJson::JSON_ARRAY);
    NJson::TJsonValue areaDetails(NJson::JSON_ARRAY);
    NJson::TJsonValue styleDetails(NJson::JSON_ARRAY);
    for (auto&& description : descriptions) {
        R_ENSURE(description, ConfigHttpStatus.UnknownErrorStatus, "null action encountered");

        TSet<EDropAbility> abilities;
        TMaybe<EDropOfferPolicy> feePolicy;
        TMaybe<ui32> fee;
        bool globalAllowRiding = false;

        NJson::TJsonValue allowAreas(NJson::JSON_ARRAY);
        NJson::TJsonValue denyAreas(NJson::JSON_ARRAY);
        NJson::TJsonValue ridingAreas(NJson::JSON_ARRAY);
        NJson::TJsonValue areas(NJson::JSON_ARRAY);
        for (auto&& i : *snapshot) {
            TCarLocationContext context = TCarLocationContext::BuildByArea("", i.GetArea(), *Server);
            context.SetSpecialAreasInfo(*description);
            context.SetTakeFeesFromOffer(false);
            TCarLocationFeatures features = context.GetCarAreaFeatures(false, defaultOfferTags, Server);

            NJson::TJsonValue* areasArray = nullptr;
            EDropAbility dropAbility = features.GetAllowDropDef(EDropAbility::NotAllow);
            const bool allowDelivery = false;
            const bool allowRiding = features.GetAllowRiding();
            const bool hasFees = features.HasFeeInfo() && features.GetFeeInfoUnsafe().GetPolicy() != EDropOfferPolicy::FeesMinute;
            const auto style = TAreasDB::GetStyle(dropAbility, allowRiding, allowDelivery, hasFees);
            abilities.insert(dropAbility);
            globalAllowRiding |= allowRiding;
            if (dropAbility == EDropAbility::Allow) {
                areasArray = &allowAreas;
            } else if (dropAbility == EDropAbility::Deny) {
                areasArray = &denyAreas;
            } else if (allowRiding) {
                areasArray = &ridingAreas;
            } else {
                continue;
            }
            if (useStyles) {
                areasArray = &areas;
            }
            if (!isCompactReport) {
                NJson::TJsonValue& areaInfo = areasArray->AppendValue(i.GetArea().SerializeToPublicReport());
                areaInfo.InsertValue("area_style", GetLocalizedStyleReport(style, TypeName));
                if (dropAbility == EDropAbility::Allow && hasFees) {
                    feePolicy = features.GetFeeInfoRef().GetPolicy();
                    fee = features.GetFeeInfoRef().GetFee();
                }
            } else {
                NJson::TJsonValue areaIdxJson = NJson::JSON_MAP;
                {
                    auto idxIt = idxByAreas.find(i.GetArea().GetIdentifier());
                    if (idxIt == idxByAreas.end()) {
                        areaDetails.AppendValue(i.GetArea().SerializeToPublicReport());
                        idxIt = idxByAreas.emplace(i.GetArea().GetIdentifier(), idxByAreas.size()).first;
                    }
                    areaIdxJson.InsertValue("area_idx", idxIt->second);
                }
                NJson::TJsonValue& areaInfo = areasArray->AppendValue(std::move(areaIdxJson));

                TString resultKey;
                NJson::TJsonValue jsonStyle = GetLocalizedStyleReport(style, TypeName, &resultKey);
                {
                    auto idxIt = idxByStyle.find(resultKey);
                    if (idxIt == idxByStyle.end()) {
                        styleDetails.AppendValue(std::move(jsonStyle));
                        idxIt = idxByStyle.emplace(resultKey, idxByStyle.size()).first;
                    }
                    areaInfo.InsertValue("style_idx", idxIt->second);
                }
            }
        }

        NJson::TJsonValue record = description->GetReport();
        if (coordinate) {
            NJson::TJsonValue& ablties = record.InsertValue("abilities", NJson::JSON_ARRAY);
            if (abilities.contains(EDropAbility::Allow) && !abilities.contains(EDropAbility::Deny)) {
                ablties.AppendValue(GetLocalizedJsonSetting({
                    "client." + description->GetName() + ".abilities.allow_drop",
                    "client.abilities.allow_drop",
                }));
            } else if (globalAllowRiding) {
                ablties.AppendValue(GetLocalizedJsonSetting({
                    "client." + description->GetName() + ".abilities.allow_parking",
                    "client.abilities.allow_parking",
                }));
                ablties.AppendValue(GetLocalizedJsonSetting({
                    "client." + description->GetName() + ".abilities.deny_drop",
                    "client.abilities.deny_drop",
                }));
            } else {
                ablties.AppendValue(GetLocalizedJsonSetting({
                    "client." + description->GetName() + ".abilities.deny_riding",
                    "client.abilities.deny_riding",
                }));
            }
            NJson::TJsonValue& notifications = record.InsertValue("notifications", NJson::JSON_ARRAY);
            if (feePolicy) {
                auto feePolicyString = ToString(feePolicy.GetOrElse(EDropOfferPolicy::FeesMax));
                auto notificationKey = TStringBuilder() << "client.notification." << feePolicyString << "_drop_general";
                if (fee && fee > 0) {
                    notificationKey << "_with_value";
                }
                auto notification = GetLocalizedJsonSetting(notificationKey);
                if (fee && fee > 0) {
                    auto feeString = localization ? localization->FormatPrice(locale, *fee) : ToString(fee);
                    NJson::ReplaceAll(notification, "_FeeValue_", feeString);
                }
                notifications.AppendValue(std::move(notification));
            }
        } else if (useStyles) {
            record["areas"] = std::move(areas);
        } else {
            record["allow_areas"] = std::move(allowAreas);
            record["deny_areas"] = std::move(denyAreas);
            record["riding_areas"] = std::move(ridingAreas);
        }
        records.AppendValue(std::move(record));
    }
    if (coordinate) {
        auto eg = g.BuildEventGuard("poi");
        TVector<TArea> areas;
        for (auto&& i : *snapshot) {
            areas.push_back(i.GetArea());
        }
        g.AddReportElement("poi", NDrive::GetPoiReport(areas, NDrive::DefaultPoiLocationTags, {}, locale, *Server));
    }
    if (geocoded.Initialized()) {
        auto eg = g.BuildEventGuard("wait_geocoded_location");
        if (!geocoded.Wait(Context->GetRequestDeadline())) {
            g.AddEvent("wait_timeout");
        }
        if (geocoded.HasValue()) {
            g.AddReportElement("geocoded_location", geocoded.GetValue().Title);
        } else {
            g.AddEvent(NJson::TMapBuilder
                ("event", "geocoded_location_exception")
                ("exception", NThreading::GetExceptionInfo(geocoded))
            );
        }
    }
    g.MutableReport().AddReportElement("legend", GetLocalizedStyleLegendReport(TypeName));
    g.MutableReport().AddReportElement("records", std::move(records));
    if (isCompactReport) {
        g.MutableReport().AddReportElement("areas", std::move(areaDetails));
        g.MutableReport().AddReportElement("styles", std::move(styleDetails));
    }
    g.SetCode(HTTP_OK);
}

namespace {
    IRequestProcessorConfig::TFactory::TRegistrator<TAreasProposeProcessor::THandlerConfig> Registrator1(TAreasProposeProcessor::GetTypeName() + "~");
    IRequestProcessorConfig::TFactory::TRegistrator<TAreasConfirmProcessor::THandlerConfig> Registrator2(TAreasConfirmProcessor::GetTypeName() + "~");
    IRequestProcessorConfig::TFactory::TRegistrator<TAreasRejectPropositionProcessor::THandlerConfig> Registrator3(TAreasRejectPropositionProcessor::GetTypeName() + "~");
}
